/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \file ValuePlotMethod.h
    \brief Definition of the Template class ValuePlotMethod.
    
    Magics Team - ECMWF 2004
    
    Started: Thu 26-Aug-2004
    
    Changes:
    
*/

#ifndef ValuePlotMethod_H
#define ValuePlotMethod_H

#include "magics.h"

#include "ValuePlotMethodAttributes.h"
#include "MatrixHandler.h"
#include "PointsHandler.h"
#include "Text.h"
#include "Transformation.h"


namespace magics {

template <class P>
class ValuePlotMethod: public ValuePlotMethodAttributes, public vector<BasicGraphicsObject*> {

public:
	ValuePlotMethod() {}
	virtual ~ValuePlotMethod() {}
    virtual void set(const map<string, string>& map ) 
        { ValuePlotMethodAttributes::set(map); }
    virtual void set(const XmlNode& node ) 
        { ValuePlotMethodAttributes::set(node); }
    
    virtual ValuePlotMethod<P>* clone() const {
    	ValuePlotMethod<P>* object = new ValuePlotMethod<P>();
    	object->copy(*this);
    	return object;
    }
    virtual void operator()(MatrixHandler<P>& data, const Transformation& transformation) {      
           int rows = data.rows();
           int columns = data.columns();
           for (int j = 0; j < rows ; j += lat_frequency_ ) {
            for (int i = 0; i <columns; i += lon_frequency_) {
                 double val =  data(j,i);
                 if ( min_ <= val && val <= max_ && val != data.missing() ) {
                	P point(data.column(j, i), data.row(j, i), data(j, i));
                	PaperPoint xy = transformation(point);
                 	if ( transformation.in(xy) ) 
                    	add(xy);
                 }
            }
           }
        }
      virtual void operator()(PointsHandler<P>& data, const Transformation& transformation)
      {
    	 data.setToFirst();
    	 
    	 while ( data.more() ) {
    		 P point =  data.current();
    		 double val = point.value();
    		 if ( min_ <= val && val <= max_ )
    		 {
    			 PaperPoint xy = transformation(point);
    		 	add(xy);
    	 	}
    		 data.advance();
    	 }  
    }


protected:
     //! Method to print string about this class on to a stream of type ostream (virtual).
	 virtual void print(ostream& out) const {
         out << "ValuePlotMethod[";
         ValuePlotMethodAttributes::print(out);
         out << "]";
	 }
     
     virtual void add(const PaperPoint& xy) {
     	 ostringstream nice;
    	 nice << MagicsFormat(format_, xy.value()); 
         Text* text = new Text(); 
         text->addText(nice.str(), *colour_, height_);
         text->setJustification(MCENTRE);
         text->push_back(xy);
         push_back(text);
    }
     
     string label(double val) {
    	 ostringstream nice;
    	 nice << MagicsFormat(format_, val);
    	 return nice.str();
     }

private:
    //! Copy constructor - No copy allowed
	ValuePlotMethod(const ValuePlotMethod&);
    //! Overloaded << operator to copy - No copy allowed
	ValuePlotMethod& operator=(const ValuePlotMethod&);
    
// -- Friends
    //! Overloaded << operator to call print().
	friend ostream& operator<<(ostream& s,const ValuePlotMethod<P>& p)
		{ p.print(s); return s; }

};

template <class P>
class MagTranslator<string, ValuePlotMethod<P> > { 
public:
	ValuePlotMethod<P>* operator()(const string& val ) {
		 return SimpleObjectMaker<ValuePlotMethod<P> >::create(val);
	}     
    ValuePlotMethod<P>* magics(const string& param)
    {
        ValuePlotMethod<P>* object=0;
		ParameterManager::update(param, object);
		return object;
    }
};

} // namespace magics


#endif
