/**
 * GUI Commands
 * Copyright 2005 Andrew Pietsch
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * $Id: DelegateManager.java,v 1.12 2007/01/11 08:28:38 pietschy Exp $
 */
package org.pietschy.command.delegate;

import javax.swing.SwingUtilities;
import java.util.HashMap;
import java.awt.Window;
import java.awt.Component;


/**
 * DelegateManager class acts as a singleton factory for all {@link DelegateMediator} instances.
 * {@link DelegateMediator}s are created using the currently installed {@link DelegateMediatorFactory}.
 *
 * @author andrewp
 * @version $Revision: 1.12 $
 * @see #getMediatorFor(java.awt.Window)
 * @see DelegatingCommand#trackDelegateIn(String, java.awt.Window)
 */
public class
DelegateManager
{
   // Constants and variables
   // -------------------------------------------------------------------------
   private static final String _ID_ = "$Id: DelegateManager.java,v 1.12 2007/01/11 08:28:38 pietschy Exp $";

   private static HashMap delegateMediators = new HashMap();

   private static DelegateMediatorFactory delegateMediatorFactory = new DefaultDelegateMediatorFactory();

   /**
    * Gets the DelegateMediator for the specified window.  If the mediator doesn't exist it will
    * be created using the current {@link DelegateMediatorFactory}.
    *
    * @param window the window of interest.
    * @return the DelegateMediator for the specified window.
    * @see #getMediatorFor(java.awt.Component)
    * @see #setDelegateMediatorFactory(DelegateMediatorFactory)
    */
   public static DelegateMediator
   getMediatorFor(Window window)
   {
      DelegateMediator mediator = (DelegateMediator) delegateMediators.get(window);
      if (mediator == null)
      {
         mediator = delegateMediatorFactory.createDelegateTracker(window);
         delegateMediators.put(window, mediator);
      }

      return mediator;
   }

   /**
    * This is a convenience method that gets the {@link DelegateMediator} based on the specified
    * components window ancestor.
    *
    * @param component
    * @return the {@link DelegateMediator} for the specified component.
    * @throws IllegalStateException if the component is not contained within a window ancestor.
    */
   public static DelegateMediator
   getMediatorFor(Component component)
   {
      Window windowAncestor = SwingUtilities.getWindowAncestor(component);
      if (windowAncestor == null)
      {
         throw new IllegalStateException("Component not contained within a window.");
      }

      return getMediatorFor(windowAncestor);
   }

   /**
    * Sets the factory to use to create {@link DelegateMediator} instances.
    *
    * @param delegateMediatorFactory the new factory to use.
    */
   public static void setDelegateMediatorFactory(DelegateMediatorFactory delegateMediatorFactory)
   {
      DelegateManager.delegateMediatorFactory = delegateMediatorFactory;
   }

}
