// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#include <iostream>
using std::cout;
using std::endl;
#include <set>
using std::set;

#include "Puma/CTree.h"
#include "Puma/CObjectInfo.h"
#include "Puma/CFileInfo.h"
#include "Puma/CEnumInfo.h"
#include "Puma/ErrorStream.h"
#include "Puma/TokenStream.h"
#include "Puma/TokenProvider.h"
#include "Puma/CCSyntax.h"
#include "Puma/CCParser.h"
#include "Puma/PreprocessorParser.h"
#include "Puma/PreMacroManager.h"
#include "Puma/PreFileIncluder.h"
#include "Puma/CPrintVisitor.h"

#include "Introducer.h"
#include "ACUnit.h"
#include "Plan.h"
#include "CodeWeaver.h"
#include "LineDirectiveMgr.h"
#include "IncludeGraph.h"
#include "Naming.h"
#include "AspectInfo.h"
#include "OrderInfo.h"
#include "IntroductionInfo.h"
#include "ModelBuilder.h"
#ifdef ACMODEL
#include "Utils.h"
#else
#include "JoinPointLoc.h"
#endif
#include "PointCutContext.h"
#include "PointCutExpr.h"
#include "IntroductionUnit.h"
#include "TransformInfo.h"

// Destructor: release all allocated resources
Introducer::~Introducer () {
  for (list<TokenProvider*>::iterator i = _token_providers.begin ();
    i != _token_providers.end (); ++i)
    delete *i;
  for (list<TokenStream*>::iterator i = _token_streams.begin ();
    i != _token_streams.end (); ++i)
    delete *i;
}
  
// called at the beginning of the parse process
void Introducer::trans_unit_begin () {
  _jpm.set_db (_parser->semantic().getFile()->SemDB());
}

// called when a new class/union/struct/aspect is created, current scope
// is the global scope
void Introducer::class_before (CT_ClassDef *cd) {
  enter ();
  assert (cd);
  
  CClassInfo *ci = (CClassInfo*)cd->Object ()->DefObject ();
  CProject &project = *ci->SemDB ()->Project ();
  ErrorStream &err = project.err ();
  const Unit *primary = ci->SourceInfo ()->FileInfo ()->Primary ();

  // create the weaving plan for this class
  JPP_Class *jp_plan = create_plan (ci);
  if (!jp_plan) {
    // this class is not the target of an introduction
    leave ();
    return;
  }
  
  // determine the units that should be included in front of the class
  set<const Unit*> units;
  for (int i = 0; i < jp_plan->otherIntros (); i++) {
#ifdef ACMODEL
    ACM_ClassSlice *cs = get_slice (*jp_plan->otherIntro (i));
    if (cs->get_kind () == CST_NORMAL ||
        (cs->get_kind () == CST_DEP_NORMAL &&
         cs->get_prot () != SP_UNKNOWN)) {
      units.insert (&TI_ClassSlice::of (*cs)->slice_unit ());
    }
#else
    JPL_ClassSlice *cs   = jp_plan->otherIntro (i)->introduced ();
    if (cs->slice_type () == JPL_ClassSlice::CS_NORMAL ||
        (cs->slice_type () == JPL_ClassSlice::CS_OLD_OTHER &&
         cs->prot () != CProtection::PROT_NONE)) {
      units.insert (&TI_ClassSlice::of (*cs)->slice_unit ());
    }
#endif
  }
  for (int i = 0; i < jp_plan->baseIntros (); i++) {
#ifdef ACMODEL
    ACM_ClassSlice *cs = get_slice (*jp_plan->baseIntro (i));
    units.insert (&TI_ClassSlice::of (*cs)->slice_unit ());
#else
    JPL_Introduction *ii = jp_plan->baseIntro (i);
    units.insert (&TI_ClassSlice::of (*ii->introduced ())->slice_unit ());
#endif
  }
  const Unit *this_unit = (Unit*)cd->token ()->belonging_to ();
  // TODO: this_unit might be a macro unit!

  // handle introductions into introduced classes (nested introductions)
  const IntroductionUnit *intro_unit = IntroductionUnit::cast (this_unit);
  if (intro_unit) this_unit = intro_unit->final_target_unit ();

//  cout << "included units for " << ci->QualName () << " in "
//       << this_unit->name () << " :" << endl;
  for (set<const Unit*>::iterator iter = units.begin ();
    iter != units.end (); ++iter) {
    const Unit *slice_unit = *iter;
    if (slice_unit == primary) {
      if (this_unit != primary)
        err << sev_error << cd->token ()->location () 
          << "affected by aspect in '" << slice_unit->name ()
          << "'. Move the aspect into a separate aspect header." << endMessage;
    }
    else if (_ig.includes (slice_unit, this_unit)) {
      err << sev_warning << cd->token ()->location () 
          << "can't include '" << slice_unit->name ()
          << "' to avoid include cycle" << endMessage;
    }
    else {
//      cout << "new edge from " << this_unit->name () << " to "
//           << slice_unit->name () << endl;
      _ig.add_edge (this_unit, slice_unit);

      // handling of nested classes -> search the outermost class
      CClassInfo *inscls = ci;
      while (inscls->Parent ()->ClassInfo ())
        inscls = inscls->Parent ()->ClassInfo ();
        
      // namespace should be closed and re-opened
      ostringstream includes;
      _code_weaver.close_namespace (includes, inscls);
            
      // inspos should be infront of the class
      Token *inspos = (Token*)inscls->Tree ()->token ();
    
      Filename incname = project.getInclString (slice_unit->name ());
      includes << endl << "#ifndef ";
      Naming::guard (includes, (FileUnit*)slice_unit);
      includes << endl << "#define ";
      Naming::guard (includes, (FileUnit*)slice_unit);
      includes << endl;
      includes << "#include \"" << incname << "\"" << endl;
      includes << "#endif" << endl;
      
      // re-open the namespace
      _code_weaver.open_namespace (includes, inscls);

      string inc (includes.str ());
      _code_weaver.paste (_code_weaver.weave_pos (inspos, WeavePos::WP_BEFORE),
        inc);

      if (_included_aspect_headers.find (slice_unit) ==
          _included_aspect_headers.end ()) {
        _included_aspect_headers.insert (slice_unit);

		CCParser *prev_parser = _parser;
        _parser = new CCParser;
        _parser->introducer (this);
        _parser->configure (project.config ());
        // setup the preprocessor
        TokenStream stream;
        stream.push ((Unit*)slice_unit);
        PreprocessorParser *prev_cpp = _cpp;
        _cpp = new PreprocessorParser (&project.err (),
          &project.unitManager (), prev_cpp->locals());
        _cpp->cscanner ().configure (project.config ());
        // cpp.macroManager ()->init (unit->name ());
        PreMacroManager *old_mm = _cpp->macroManager ();
        PreFileIncluder *old_fi = _cpp->fileIncluder ();
        PrePredicateManager *old_pm = _cpp->predicateManager ();
        _cpp->macroManager (prev_cpp->macroManager ());
        _cpp->fileIncluder (prev_cpp->fileIncluder ());
        _cpp->predicateManager (prev_cpp->predicateManager ());
        _cpp->fileIncluder ()->preprocessor (_cpp);
        _cpp->stream (&stream);
//        _cpp->configure (project.config (), false); // do not process --include option

        // initialize semantic analyzer
        _parser->semantic ().init (*ci->SemDB (),
                            *ci->SourceInfo ()->FileInfo ()->Primary ());
        ((ErrorCollector&)_parser->builder ().err ()).index (0);
        _parser->semantic ().error_sink (_parser->builder ().err ());

        TokenProvider provider (*_cpp);        // look ahead token buffer
		CTree *tree = _parser->syntax ().run (provider, &CCSyntax::trans_unit);
        if (!tree)
        	;
//          err << sev_error << cd->token ()->location ()
//              << "Parsing auto-included header failed." << endMessage;
        else
          // TODO: tree should later be freed!
          _ah_trees.push_back (tree);
        
        // print all error messages
        _parser->builder ().errors (err);
//        cout << "--------" << endl;
//        CPrintVisitor printer;
//        cout << "Printing syntax tree..." << endl;
//        printer.print (tree, cout);
        
        _cpp->macroManager (old_mm);
        _cpp->fileIncluder (old_fi);
        _cpp->predicateManager (old_pm);
        delete _cpp;
        _cpp = prev_cpp;
        _cpp->fileIncluder ()->preprocessor (_cpp);
        delete _parser;
        _parser = prev_parser;
      }
//      else
//        cout << "'" << slice_unit->name () << "' not included again!" << endl;
    }
  }
  
//  cd->add_intro (tree);
  
  leave ();
}

// called when a new class/union/struct/aspect is created
void Introducer::class_begin (CT_ClassDef *cd) {
  enter ();
  assert (cd);
  leave ();
}

// parse code that shall be introduced
CTree *Introducer::parse (list<Unit*> units, bool (CCSyntax::*rule)(),
  const char *expected_id, ErrorStream &err) {
    
  // create a token stream and push all units onto it
  TokenStream *stream = new TokenStream;
  _token_streams.push_back (stream);
  for (list<Unit*>::reverse_iterator ui = units.rbegin ();
    ui != units.rend (); ++ui)
    stream->push (*ui);

  // parse it
  TokenProvider *provider = new TokenProvider (*stream); // look ahead buffer
  _token_providers.push_back (provider);
  CTree *tree = _parser->syntax ().run (*provider, rule);
  
  // print all error messages => does that really work?
  _parser->builder ().errors (err);
  if (tree && tree->NodeName () != expected_id)
    tree = 0; // TODO: tree should be deleted!
  return tree;
}

// called when a new class/union/struct/aspect definition ends
// (still in the class scope)
void Introducer::class_end (CT_ClassDef *cd) {
  enter ();
  assert (cd);

  CClassInfo *ci = (CClassInfo*)cd->Object ()->DefObject ();
  ErrorStream &err = ci->SemDB ()->Project ()->err ();
  // manipulate the code (will be effective after commit),
  // paste declaration before "}" of the class definition
  Token *inspos = cd->Members ()->end_token ();
  const WeavePos &pos = _code_weaver.weave_pos (inspos, WeavePos::WP_BEFORE);

  // first check with the plan if there are intros for this class
  JPP_Class *jp_plan = plan_lookup (ci);
  if (!jp_plan) {
  	insert_introspection_code (cd);
    leave ();
    return;
  }
  
  // collect the intros
  list<Unit*> units;
  LineDirectiveMgr &lmgr = _code_weaver.line_directive_mgr ();
  jp_plan->gen_intros (units, err, ci, lmgr, _conf.introduction_depth ());
  if (units.size () > 0) {
    // parse the intros
    CTree *tree = parse (units, &CCSyntax::member_spec, CT_MembList::NodeId (), err);
    if (tree) {
	  // move the introduced members into the normal member list
	  CT_MembList *memb_list = (CT_MembList*)tree;
	  for (int e = 0; e < memb_list->Entries (); e++) {
	    CTree *entry = memb_list->Entry (e);
	    cd->Members ()->InsertSon (cd->Members ()->Sons () - 1, entry);
  	  }
	  delete memb_list;

	  // paste the generated code and add the units to the manipulator's unit mgr.
	  for (list<Unit*>::iterator ui = units.begin ();
	    ui != units.end (); ++ui)
	    _code_weaver.paste (pos, *ui);
    }
    else {
	  err << sev_error << inspos->location()
		  << "class member introduction into '" << ci->QualName() << "' failed"
		  << endMessage;

  	  // delete all intro units
	  for (list<Unit*>::iterator ui = units.begin (); ui != units.end (); ++ui)
	    delete *ui;
    }
  }
  
  insert_introspection_code (cd);
  leave ();
}


// insert introspection code
//  * at the end of class definitions, after AspectC++ introductions
void Introducer::insert_introspection_code (CT_ClassDef *cd) {
  if (!_conf.introspection())
    return;
  
  CClassInfo *ci = (CClassInfo*)cd->Object ()->DefObject ();
  ErrorStream &err = ci->SemDB ()->Project ()->err ();

  // return if this class is not an introduction target
  if (!_jpm.is_valid_model_class (ci) || !_jpm.is_intro_target (ci))
    return;

  // manipulate the code (will be effective after commit),
  // paste declaration before "}" of the class definition
  Token *inspos = cd->Members ()->end_token ();
  const WeavePos &pos = _code_weaver.weave_pos (inspos, WeavePos::WP_BEFORE);

  list<Unit*> units;
  ACUnit *unit = new ACUnit (err);
  unit->name ("<typeinfo>");
  
  *unit << "public:" << endl;
  // generate __AttrTypes => a list with all attribute types
  *unit << "  typedef ";
  unsigned e = 0;
  for (unsigned a = 0; a < ci->Attributes (); a++) {
    if (!is_attribute (ci->Attribute (a)))
      continue;
    *unit << "AC::TL< ";
    // TODO: Better check if there is really an ambiguity - not only for "stat"
//    if (ci->Attribute (a)->TypeInfo ()->Record () &&
//        strcmp (ci->Attribute (a)->TypeInfo ()->Record ()->Name (), "stat") == 0)
//      *unit << "struct ";
    ci->Attribute (a)->TypeInfo ()->TypeText (*unit, "", true, true, true);
    *unit << ",";
    e++;
  }
  *unit << "AC::TLE ";
  while (e > 0) {
  	*unit << " > ";
  	e--;
  }
  *unit << "__AttrTypes;" << endl;

  // generate __attr_name => the name of attribute i
  *unit << "  const char *__attr_name (unsigned i) const {";
  if (ci->Attributes () == 0) {
  	*unit << " return 0; }";
  }
  else {
    *unit << endl << "    static const char *names[] = { ";
    bool first = true;
    for (unsigned a = 0; a < ci->Attributes (); a++) {
      if (!is_attribute (ci->Attribute (a)))
        continue;
  	  if (!first) *unit << ", ";
  	  *unit << "\"" << ci->Attribute (a)->Name () << "\"";
      first = false;
	  }
	  *unit << " }; return names[i];" << endl
          << "  }";
  }
  *unit << endl;
  
  // generate __attr => the untyped pointer to attribute i
  *unit << "  void *__attr (unsigned __i) const {";
  if (ci->Attributes () == 0) {
  	*unit << " return 0; }";
  }
  else {
  	*unit << endl
  	      << "    switch (__i) { ";
    for (unsigned a = 0; a < ci->Attributes (); a++) {
      if (!is_attribute (ci->Attribute (a)))
        continue;
  	  *unit << "case " << a << ": return (void*)&" << ci->Attribute (a)->Name () << "; ";
	  }
	  *unit << "default: return 0; }" << endl
	        << "  }";
  }
  *unit << endu;
  
//  if (ci->Name()) cout << "Class: " << ci->Name() << endl;
//  cout << "Unit:" << endl << *unit << endl;
  units.push_back (unit);
  CTree *tree = parse (units, &CCSyntax::member_spec, CT_MembList::NodeId (), err);
  if (tree) {
    // move the introduced members into the normal member list
    CT_MembList *memb_list = (CT_MembList*)tree;
    for (int e = 0; e < memb_list->Entries (); e++) {
      CTree *entry = memb_list->Entry (e);
      cd->Members ()->InsertSon (cd->Members ()->Sons () - 1, entry);
    }
    _code_weaver.paste (pos, unit);
    delete memb_list;
  }
  else {
    err << sev_error << inspos->location()
        << "parsing introspection code for class '" << ci->QualName()
        << "' failed" << endMessage;
  	delete unit;
  }
  // paste a #line directive
  LineDirectiveMgr &lmgr = _code_weaver.line_directive_mgr ();
  ACUnit *dunit = new ACUnit (err);
  lmgr.directive (*dunit, (Unit*)inspos->belonging_to (), inspos);
  *dunit << endu;
  if (dunit->empty ()) delete dunit; else _code_weaver.paste (pos, dunit);
}


// checks if an attribute that us returned by the parser is an attribute
// in the sense of the AspectC++ introspection mechnism
bool Introducer::is_attribute (CAttributeInfo *obj) {
  if (obj->isAnonymous () ||
      obj->isStatic () ||
      obj->EnumeratorInfo () ||
      // TODO: temporary hack - attributes that have an anonymous type
      //       shall no be ignored in the type list!
      (obj->TypeInfo () &&
       (obj->TypeInfo ()->isBitField() ||
        (obj->TypeInfo ()->Record () && obj->TypeInfo ()->Record ()->isAnonymous ()) ||
        (obj->TypeInfo ()->EnumInfo () && obj->TypeInfo ()->EnumInfo ()->isAnonymous ()))))
    return false;
  return true;
}


// called after the parser tried to parse a base clause
void Introducer::base_clause_end (CT_ClassDef *cd, Token *open) {
  enter ();
  assert (cd);
  CClassInfo *ci = (CClassInfo*)cd->Object ()->DefObject ();
  ErrorStream &err = ci->SemDB ()->Project ()->err ();
  Token *inspos = open;

  // first check with the plan if there are intros for this class
  JPP_Class *jp_plan = plan_lookup (ci);
  if (!jp_plan) {
    leave ();
    return;
  }
  
  // create a unit with the code that shall be introduced
  LineDirectiveMgr &lmgr = _code_weaver.line_directive_mgr ();

  // collect the intros
  list<Unit*> units;
  jp_plan->gen_base_intros (units, err, ci, lmgr);

  if (units.size () > 0) {
    // parse the introduced code as a base clause
    CTree *tree = parse (units, &CCSyntax::base_clause,
    		             CT_BaseSpecList::NodeId (), err);
    if (tree) {
      cd->BaseIntros (tree);
      // manipulate the code (will be effective after commit),
      // paste declaration before "{" of the class definition
      const WeavePos &pos = _code_weaver.weave_pos (inspos, WeavePos::WP_BEFORE);
      // paste the generated code and add the unit to the manipulator's unit mgr.
      if (cd->BaseClasses () != 0) {
        // get the first introduced base class unit
        Unit *first_unit = units.front ();
        // delete the ":" token at the beginning of the unit
        first_unit->remove ((ListElement*)first_unit->first ());
        // insert "," instead
        ACUnit comma (err);
        comma << "," << endu;
        first_unit->move_before ((ListElement*)first_unit->first (), comma);
      }
      // now paste all units
      for (list<Unit*>::iterator ui = units.begin ();
        ui != units.end (); ++ui)
        _code_weaver.paste (pos, *ui);
      // paste a #line directive
      ACUnit *dunit = new ACUnit (err);
      lmgr.directive (*dunit, (Unit*)inspos->belonging_to (), inspos);
      *dunit << endu;
      if (dunit->empty ()) delete dunit; else _code_weaver.paste (pos, dunit);
    }
    else {
      err << sev_error << inspos->location()
          << "base class introduction into '" << ci->QualName() << "' failed"
          << endMessage;
      // delete all base into units
      for (list<Unit*>::iterator ui = units.begin (); ui != units.end (); ++ui)
        delete *ui;
    }
  }
  leave ();
}


// called after the program has been parsed completely
void Introducer::trans_unit_end (CT_Program *pr) {
  // ignore this translation unit, if it is a nested parser run
  if (_intro_level != 0)
    return;
  enter ();
    
  CFileInfo *fi = pr->Scope ()->FileInfo ();
  assert (fi);
  CSemDatabase &db = *fi->SemDB ();
  CProject &project = *db.Project ();
  ErrorStream &err = project.err ();
  const Unit *primary = fi->Primary ();
  LineDirectiveMgr &lmgr = _code_weaver.line_directive_mgr ();
  
  // ... some unit for formatting
  ACUnit ws (err); ws << " " << endu;

  // loop until there are no more entries in the target map
  while (!_targets.empty ()) {
    
    // get the information from the first entry and delete it
    TargetMap::iterator i = _targets.begin ();
    CClassInfo *ci     = i->first;
#ifdef ACMODEL
    ACM_Class &jp_loc  = *i->second;
#else
    JPL_Class &jp_loc  = *i->second;
#endif
    JPP_Class &jp_plan = *(JPP_Class*)jp_loc.plan();
    _targets.erase (i);
    
    // create units with the code that shall be introduced
    list<Unit*> intros;
    jp_plan.gen_intros (intros, err, ci, lmgr, _conf.introduction_depth (), true);
    // ignore this class if there are no non-inline intros for it
    if (intros.size () == 0)
      continue;
    
    CSourceInfo *si = ci->SourceInfo ();
    bool in_header = (strcmp (si->FileName (),
                              si->FileInfo ()->Primary ()->name ()) != 0);
                              
    if (in_header) {
      // check if there is a link-once code element in the class
      CObjectInfo *loo = link_once_object (ci);
      
      // TODO: for now loo has to != 0, later we can exploit the proj. repo
      if (!loo) {
        err << sev_warning << TransformInfo::location (jp_loc)
            << "cannot introduce non-inline function or static attribute"
#ifdef ACMODEL
            << " into \"class " << signature (jp_loc).c_str()
#else
            << " into \"class " << jp_loc.signature ()
#endif
            << "\". It has to contain link-once code." << endMessage;
        for (list<Unit*>::iterator i = intros.begin (); i != intros.end (); ++i)
          delete *i;
        continue;
      }
      
      // continue silently if this is only a declaration
      if (loo->Scope () == ci) {
        for (list<Unit*>::iterator i = intros.begin (); i != intros.end (); ++i)
          delete *i;
        continue;
      }
    }
      
    // parse the introduced code
    CTree *tree = parse (intros, &CCSyntax::decl_seq,
                         Builder::Container::NodeId (), err);
    if (!tree) {
      err << sev_error << "parsing non-inline introduction code for class '"
          << ci->QualName() << "' failed" << endMessage;
      for (list<Unit*>::iterator i = intros.begin (); i != intros.end (); ++i)
        delete *i;
      leave ();
      return;
    }
    
    // move the introduced members into the normal program node
    Builder::Container *decls = (Builder::Container*)tree;
    for (int e = 0; e < decls->Entries (); e++) {
      CTree *entry = decls->Entry (e);
      pr->AddSon (entry);
    }
    delete decls;
    
    // paste the generated code and add the unit to the manipulator's unit mgr.
    for (list<Unit*>::iterator i = intros.begin (); i != intros.end (); ++i)
      _code_weaver.paste_end (*i);

    // determine the units that should be included in front of the intros
    set<const Unit*> units;
    for (int i = 0; i < jp_plan.otherIntros (); i++) {
#ifdef ACMODEL
      ACM_ClassSlice *cs = get_slice (*jp_plan.otherIntro (i));
      if (cs->get_kind () == CST_NORMAL) {
#else
        JPL_ClassSlice *cs   = jp_plan.otherIntro (i)->introduced ();
        if (cs->slice_type () == JPL_ClassSlice::CS_NORMAL) {
#endif
          CObjectInfo *obj = TI_ClassSlice::of (*cs)->assoc_obj ();
          assert (obj && obj->Tree ());
        assert (obj->Tree ()->NodeName () == CT_ClassSliceDecl::NodeId ());
        ACSliceInfo *acsi = obj->SemDB ()->SliceInfo (obj)->definition ();
        for (int m = 0; m < acsi->members (); m++)
          units.insert (acsi->member_unit (m));
      }
#ifdef ACMODEL
      else if (cs->get_kind () == CST_DEP_NORMAL &&
               cs->get_prot () == SP_UNKNOWN) {
#else
      else if (cs->slice_type () == JPL_ClassSlice::CS_OLD_OTHER &&
               cs->prot () == CProtection::PROT_NONE) {
#endif
          units.insert (&TI_ClassSlice::of (*cs)->slice_unit ());
      }
    }
    
    // parse the aspect headers that are needed by this intro
    for (set<const Unit*>::iterator iter = units.begin ();
      iter != units.end (); ++iter) {
      const Unit *slice_unit = *iter;
      if (slice_unit != primary) {
//        cout << "new edge from " << primary->name () << " to "
//             << slice_unit->name () << endl;
        _ig.add_edge (primary, slice_unit);
        // generate a unit with the include
        ostringstream includes;
        Filename incname = project.getInclString (slice_unit->name ());
        includes << endl << "#ifndef ";
        Naming::guard (includes, (FileUnit*)slice_unit);
        includes << endl << "#define ";
        Naming::guard (includes, (FileUnit*)slice_unit);
        includes << endl;
        includes << "#include \"" << incname << "\"" << endl;
        includes << "#endif" << endl;
        string inc (includes.str ());
        _code_weaver.paste_end (inc);

        if (_included_aspect_headers.find (slice_unit) ==
            _included_aspect_headers.end ()) {
          _included_aspect_headers.insert (slice_unit);
          // prepare a new C preprocessor
          TokenStream stream;           // linearize tokens from several files
          stream.push ((Unit*)slice_unit);
          PreprocessorParser cpp (&project.err (), &project.unitManager (),
            _cpp->locals ());
          PreMacroManager *old_mm = cpp.macroManager ();
          cpp.macroManager (_cpp->macroManager ());
          cpp.stream (&stream);
          cpp.configure (project.config ());
      
          TokenProvider provider (cpp);        // look ahead token buffer
          CTree *tree = _parser->syntax ().run (provider, &CCSyntax::trans_unit);
          // TODO: tree should later be freed!
          _ah_trees.push_back (tree);
        
          // print all error messages
          _parser->builder ().errors (err);
          cpp.macroManager (old_mm);
        }
//        else
//          cout << "'" << slice_unit->name () << "' not included again!" << endl;
      }
    }
  }
  leave ();
}


// manage the intro nesting level and the _cpp pointer
void Introducer::enter () {
  if (_intro_level == 0)
    _cpp = &(PreprocessorParser&)_parser->syntax ().provider ()->source ();
  _intro_level++;
}
void Introducer::leave () {
  _intro_level--;
  if (_intro_level == 0)
    _cpp = 0;
}

// create the weaving plan for a given class
JPP_Class *Introducer::create_plan (CClassInfo *ci) {
  
  // return if this class is not an introduction target
  if (!_jpm.is_intro_target (ci->DefObject ()))
    return 0;
    
  // try to register this class (might be a newly introduced class)
#ifdef ACMODEL
  ACM_Class *jpl = 0;
#else
  JPL_Class *jpl = 0;
#endif
  ACAspectInfo *ai = ci->SemDB()->AspectInfo (ci->DefObject ());
  if (ai)
    jpl = _jpm.register_aspect (ai);
  else
    jpl = _jpm.register_class (ci);
  
  // return if this is either not a valid model class or no valid intro target
#ifdef ACMODEL
  if (!jpl || !jpl->get_intro_target ())
#else
  if (!jpl || !jpl->intro_target ())
#endif
    return 0;
  
  // Plan object pointer
  JPP_Class *jpp = 0;

  // iterate through all introduction advice in the plan
  PointCutContext context (_jpm);
  const list<IntroductionInfo*> &intros = _plan.introduction_infos ();
  for (list<IntroductionInfo*>::const_iterator i = intros.begin ();
       i != intros.end (); ++i) {
    IntroductionInfo *intro = *i;
    // TODO: consider stand-alone advice here as well in the future (C-mode)
    // something like ... if (!intro->is_activated ()) continue;
    context.concrete_aspect (intro->aspect ());
    Binding binding;     // binding and condition not used for intros
    Condition condition;
    if (intro->expr ()->evaluate (*jpl, context, binding, condition)) {
      jpp = _plan.consider (jpl, &intro->intro ());
    }
  }
  
  if (jpp) {
    // ordering
    _plan.order (jpl);

    // check the plan
    _plan.check (jpl);
    
    // remember the class info and join point location
    _targets.insert (TargetMap::value_type (ci, jpl));
  }
  return jpp;
}
  

JPP_Class *Introducer::plan_lookup (CClassInfo *ci) {
  TargetMap::iterator i = _targets.find (ci);
  if (i != _targets.end ())
    return (JPP_Class*)i->second->plan ();
  return 0;
}


CObjectInfo *Introducer::link_once_object (CClassInfo *ci) {
  for (unsigned i = 0; i < ci->Functions (); i++) {
    CFunctionInfo *fi = ci->Function (i)->DefObject ();
    // skip template functions and built-in functions
    // they don't need link-once code
    if (fi->isBuiltin () || fi->isTemplate () || is_intro (fi)) {
      continue;
    }
    // if a member function is undefined it is link-once code!
    if (!fi->isDefined ()) {
      return fi;
    }
    // if the function is defined, outside the class scope, and is not inline,
    // we found the implementation
    if (fi->Scope () != ci && !fi->isInline ()) {
      return fi;
    }
  }
  for (unsigned i = 0; i < ci->Attributes (); i++) {
    CAttributeInfo *ai = ci->Attribute (i)->DefObject ();
    // ignore introduced attributes
    if (is_intro (ai))
      continue;
    // if the scope is outside the class, we or definition
    if (ai->Scope () != ci) {
      return ai;
    }
    // initialized, we can us this object
    if (ai->isStatic () && !ai->Init ()) {
      return ai;
    }
  }
  return 0;
}

bool Introducer::is_intro (CObjectInfo *obj) {
  Unit *unit = obj->SourceInfo ()->SrcUnit ();
  return (IntroductionUnit::cast (unit) != 0);
}

