/* hashmap.vala
 *
 * Copyright (C) 1995-1997  Peter Mattis, Spencer Kimball and Josh MacDonald
 * Copyright (C) 1997-2000  GLib Team and others
 * Copyright (C) 2007-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>


#define GEE_TYPE_COLLECTION_OBJECT (gee_collection_object_get_type ())
#define GEE_COLLECTION_OBJECT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_COLLECTION_OBJECT, GeeCollectionObject))
#define GEE_COLLECTION_OBJECT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_TYPE_COLLECTION_OBJECT, GeeCollectionObjectClass))
#define GEE_IS_COLLECTION_OBJECT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_COLLECTION_OBJECT))
#define GEE_IS_COLLECTION_OBJECT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_TYPE_COLLECTION_OBJECT))
#define GEE_COLLECTION_OBJECT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_TYPE_COLLECTION_OBJECT, GeeCollectionObjectClass))

typedef struct _GeeCollectionObject GeeCollectionObject;
typedef struct _GeeCollectionObjectClass GeeCollectionObjectClass;
typedef struct _GeeCollectionObjectPrivate GeeCollectionObjectPrivate;

#define GEE_TYPE_MAP (gee_map_get_type ())
#define GEE_MAP(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_MAP, GeeMap))
#define GEE_IS_MAP(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_MAP))
#define GEE_MAP_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_MAP, GeeMapIface))

typedef struct _GeeMap GeeMap;
typedef struct _GeeMapIface GeeMapIface;

#define GEE_TYPE_ITERABLE (gee_iterable_get_type ())
#define GEE_ITERABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_ITERABLE, GeeIterable))
#define GEE_IS_ITERABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_ITERABLE))
#define GEE_ITERABLE_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_ITERABLE, GeeIterableIface))

typedef struct _GeeIterable GeeIterable;
typedef struct _GeeIterableIface GeeIterableIface;

#define GEE_TYPE_ITERATOR (gee_iterator_get_type ())
#define GEE_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_ITERATOR, GeeIterator))
#define GEE_IS_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_ITERATOR))
#define GEE_ITERATOR_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_ITERATOR, GeeIteratorIface))

typedef struct _GeeIterator GeeIterator;
typedef struct _GeeIteratorIface GeeIteratorIface;

#define GEE_TYPE_COLLECTION (gee_collection_get_type ())
#define GEE_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_COLLECTION, GeeCollection))
#define GEE_IS_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_COLLECTION))
#define GEE_COLLECTION_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_COLLECTION, GeeCollectionIface))

typedef struct _GeeCollection GeeCollection;
typedef struct _GeeCollectionIface GeeCollectionIface;

#define GEE_TYPE_SET (gee_set_get_type ())
#define GEE_SET(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_SET, GeeSet))
#define GEE_IS_SET(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_SET))
#define GEE_SET_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), GEE_TYPE_SET, GeeSetIface))

typedef struct _GeeSet GeeSet;
typedef struct _GeeSetIface GeeSetIface;

#define GEE_TYPE_HASH_MAP (gee_hash_map_get_type ())
#define GEE_HASH_MAP(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_TYPE_HASH_MAP, GeeHashMap))
#define GEE_HASH_MAP_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_TYPE_HASH_MAP, GeeHashMapClass))
#define GEE_IS_HASH_MAP(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_TYPE_HASH_MAP))
#define GEE_IS_HASH_MAP_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_TYPE_HASH_MAP))
#define GEE_HASH_MAP_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_TYPE_HASH_MAP, GeeHashMapClass))

typedef struct _GeeHashMap GeeHashMap;
typedef struct _GeeHashMapClass GeeHashMapClass;
typedef struct _GeeHashMapPrivate GeeHashMapPrivate;
typedef struct _GeeHashMapNode GeeHashMapNode;

#define GEE_HASH_MAP_TYPE_KEY_SET (gee_hash_map_key_set_get_type ())
#define GEE_HASH_MAP_KEY_SET(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_HASH_MAP_TYPE_KEY_SET, GeeHashMapKeySet))
#define GEE_HASH_MAP_KEY_SET_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_HASH_MAP_TYPE_KEY_SET, GeeHashMapKeySetClass))
#define GEE_HASH_MAP_IS_KEY_SET(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_HASH_MAP_TYPE_KEY_SET))
#define GEE_HASH_MAP_IS_KEY_SET_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_HASH_MAP_TYPE_KEY_SET))
#define GEE_HASH_MAP_KEY_SET_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_HASH_MAP_TYPE_KEY_SET, GeeHashMapKeySetClass))

typedef struct _GeeHashMapKeySet GeeHashMapKeySet;
typedef struct _GeeHashMapKeySetClass GeeHashMapKeySetClass;

#define GEE_HASH_MAP_TYPE_VALUE_COLLECTION (gee_hash_map_value_collection_get_type ())
#define GEE_HASH_MAP_VALUE_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_HASH_MAP_TYPE_VALUE_COLLECTION, GeeHashMapValueCollection))
#define GEE_HASH_MAP_VALUE_COLLECTION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_HASH_MAP_TYPE_VALUE_COLLECTION, GeeHashMapValueCollectionClass))
#define GEE_HASH_MAP_IS_VALUE_COLLECTION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_HASH_MAP_TYPE_VALUE_COLLECTION))
#define GEE_HASH_MAP_IS_VALUE_COLLECTION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_HASH_MAP_TYPE_VALUE_COLLECTION))
#define GEE_HASH_MAP_VALUE_COLLECTION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_HASH_MAP_TYPE_VALUE_COLLECTION, GeeHashMapValueCollectionClass))

typedef struct _GeeHashMapValueCollection GeeHashMapValueCollection;
typedef struct _GeeHashMapValueCollectionClass GeeHashMapValueCollectionClass;
#define _gee_hash_map_node_free0(var) ((var == NULL) ? NULL : (var = (gee_hash_map_node_free (var), NULL)))
typedef struct _GeeHashMapKeySetPrivate GeeHashMapKeySetPrivate;
#define _gee_collection_object_unref0(var) ((var == NULL) ? NULL : (var = (gee_collection_object_unref (var), NULL)))

#define GEE_HASH_MAP_TYPE_KEY_ITERATOR (gee_hash_map_key_iterator_get_type ())
#define GEE_HASH_MAP_KEY_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_HASH_MAP_TYPE_KEY_ITERATOR, GeeHashMapKeyIterator))
#define GEE_HASH_MAP_KEY_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_HASH_MAP_TYPE_KEY_ITERATOR, GeeHashMapKeyIteratorClass))
#define GEE_HASH_MAP_IS_KEY_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_HASH_MAP_TYPE_KEY_ITERATOR))
#define GEE_HASH_MAP_IS_KEY_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_HASH_MAP_TYPE_KEY_ITERATOR))
#define GEE_HASH_MAP_KEY_ITERATOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_HASH_MAP_TYPE_KEY_ITERATOR, GeeHashMapKeyIteratorClass))

typedef struct _GeeHashMapKeyIterator GeeHashMapKeyIterator;
typedef struct _GeeHashMapKeyIteratorClass GeeHashMapKeyIteratorClass;
typedef struct _GeeHashMapKeyIteratorPrivate GeeHashMapKeyIteratorPrivate;
typedef struct _GeeHashMapValueCollectionPrivate GeeHashMapValueCollectionPrivate;

#define GEE_HASH_MAP_TYPE_VALUE_ITERATOR (gee_hash_map_value_iterator_get_type ())
#define GEE_HASH_MAP_VALUE_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_HASH_MAP_TYPE_VALUE_ITERATOR, GeeHashMapValueIterator))
#define GEE_HASH_MAP_VALUE_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_HASH_MAP_TYPE_VALUE_ITERATOR, GeeHashMapValueIteratorClass))
#define GEE_HASH_MAP_IS_VALUE_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_HASH_MAP_TYPE_VALUE_ITERATOR))
#define GEE_HASH_MAP_IS_VALUE_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_HASH_MAP_TYPE_VALUE_ITERATOR))
#define GEE_HASH_MAP_VALUE_ITERATOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_HASH_MAP_TYPE_VALUE_ITERATOR, GeeHashMapValueIteratorClass))

typedef struct _GeeHashMapValueIterator GeeHashMapValueIterator;
typedef struct _GeeHashMapValueIteratorClass GeeHashMapValueIteratorClass;
typedef struct _GeeHashMapValueIteratorPrivate GeeHashMapValueIteratorPrivate;

struct _GeeCollectionObject {
	GTypeInstance parent_instance;
	volatile int ref_count;
	GeeCollectionObjectPrivate * priv;
};

struct _GeeCollectionObjectClass {
	GTypeClass parent_class;
	void (*finalize) (GeeCollectionObject *self);
};

struct _GeeIteratorIface {
	GTypeInterface parent_iface;
	gboolean (*next) (GeeIterator* self);
	gpointer (*get) (GeeIterator* self);
};

struct _GeeIterableIface {
	GTypeInterface parent_iface;
	GType (*get_element_type) (GeeIterable* self);
	GeeIterator* (*iterator) (GeeIterable* self);
};

struct _GeeCollectionIface {
	GTypeInterface parent_iface;
	gboolean (*contains) (GeeCollection* self, gconstpointer item);
	gboolean (*add) (GeeCollection* self, gconstpointer item);
	gboolean (*remove) (GeeCollection* self, gconstpointer item);
	void (*clear) (GeeCollection* self);
	gint (*get_size) (GeeCollection* self);
};

struct _GeeSetIface {
	GTypeInterface parent_iface;
};

struct _GeeMapIface {
	GTypeInterface parent_iface;
	GeeSet* (*get_keys) (GeeMap* self);
	GeeCollection* (*get_values) (GeeMap* self);
	gboolean (*contains) (GeeMap* self, gconstpointer key);
	gpointer (*get) (GeeMap* self, gconstpointer key);
	void (*set) (GeeMap* self, gconstpointer key, gconstpointer value);
	gboolean (*remove) (GeeMap* self, gconstpointer key);
	void (*clear) (GeeMap* self);
	gint (*get_size) (GeeMap* self);
};

struct _GeeHashMap {
	GeeCollectionObject parent_instance;
	GeeHashMapPrivate * priv;
};

struct _GeeHashMapClass {
	GeeCollectionObjectClass parent_class;
};

struct _GeeHashMapPrivate {
	GType k_type;
	GBoxedCopyFunc k_dup_func;
	GDestroyNotify k_destroy_func;
	GType v_type;
	GBoxedCopyFunc v_dup_func;
	GDestroyNotify v_destroy_func;
	gint _array_size;
	gint _nnodes;
	GeeHashMapNode** _nodes;
	gint _nodes_length1;
	gint _nodes_size;
	gint _stamp;
	GHashFunc _key_hash_func;
	GEqualFunc _key_equal_func;
	GEqualFunc _value_equal_func;
};

struct _GeeHashMapNode {
	gpointer key;
	gpointer value;
	GeeHashMapNode* next;
	guint key_hash;
};

struct _GeeHashMapKeySet {
	GeeCollectionObject parent_instance;
	GeeHashMapKeySetPrivate * priv;
};

struct _GeeHashMapKeySetClass {
	GeeCollectionObjectClass parent_class;
};

struct _GeeHashMapKeySetPrivate {
	GType k_type;
	GBoxedCopyFunc k_dup_func;
	GDestroyNotify k_destroy_func;
	GType v_type;
	GBoxedCopyFunc v_dup_func;
	GDestroyNotify v_destroy_func;
	GeeHashMap* _map;
};

struct _GeeHashMapKeyIterator {
	GeeCollectionObject parent_instance;
	GeeHashMapKeyIteratorPrivate * priv;
};

struct _GeeHashMapKeyIteratorClass {
	GeeCollectionObjectClass parent_class;
};

struct _GeeHashMapKeyIteratorPrivate {
	GType k_type;
	GBoxedCopyFunc k_dup_func;
	GDestroyNotify k_destroy_func;
	GType v_type;
	GBoxedCopyFunc v_dup_func;
	GDestroyNotify v_destroy_func;
	GeeHashMap* _map;
	gint _index;
	GeeHashMapNode* _node;
	gint _stamp;
};

struct _GeeHashMapValueCollection {
	GeeCollectionObject parent_instance;
	GeeHashMapValueCollectionPrivate * priv;
};

struct _GeeHashMapValueCollectionClass {
	GeeCollectionObjectClass parent_class;
};

struct _GeeHashMapValueCollectionPrivate {
	GType k_type;
	GBoxedCopyFunc k_dup_func;
	GDestroyNotify k_destroy_func;
	GType v_type;
	GBoxedCopyFunc v_dup_func;
	GDestroyNotify v_destroy_func;
	GeeHashMap* _map;
};

struct _GeeHashMapValueIterator {
	GeeCollectionObject parent_instance;
	GeeHashMapValueIteratorPrivate * priv;
};

struct _GeeHashMapValueIteratorClass {
	GeeCollectionObjectClass parent_class;
};

struct _GeeHashMapValueIteratorPrivate {
	GType k_type;
	GBoxedCopyFunc k_dup_func;
	GDestroyNotify k_destroy_func;
	GType v_type;
	GBoxedCopyFunc v_dup_func;
	GDestroyNotify v_destroy_func;
	GeeHashMap* _map;
	gint _index;
	GeeHashMapNode* _node;
	gint _stamp;
};


static gpointer gee_hash_map_key_set_parent_class = NULL;
static GeeIterableIface* gee_hash_map_key_set_gee_iterable_parent_iface = NULL;
static GeeCollectionIface* gee_hash_map_key_set_gee_collection_parent_iface = NULL;
static GeeSetIface* gee_hash_map_key_set_gee_set_parent_iface = NULL;
static gpointer gee_hash_map_key_iterator_parent_class = NULL;
static GeeIteratorIface* gee_hash_map_key_iterator_gee_iterator_parent_iface = NULL;
static gpointer gee_hash_map_value_collection_parent_class = NULL;
static GeeIterableIface* gee_hash_map_value_collection_gee_iterable_parent_iface = NULL;
static GeeCollectionIface* gee_hash_map_value_collection_gee_collection_parent_iface = NULL;
static gpointer gee_hash_map_value_iterator_parent_class = NULL;
static GeeIteratorIface* gee_hash_map_value_iterator_gee_iterator_parent_iface = NULL;
static gpointer gee_hash_map_parent_class = NULL;
static GeeMapIface* gee_hash_map_gee_map_parent_iface = NULL;

gpointer gee_collection_object_ref (gpointer instance);
void gee_collection_object_unref (gpointer instance);
GParamSpec* gee_param_spec_collection_object (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void gee_value_set_collection_object (GValue* value, gpointer v_object);
gpointer gee_value_get_collection_object (const GValue* value);
GType gee_collection_object_get_type (void);
GType gee_iterator_get_type (void);
GType gee_iterable_get_type (void);
GType gee_collection_get_type (void);
GType gee_set_get_type (void);
GType gee_map_get_type (void);
GType gee_hash_map_get_type (void);
#define GEE_HASH_MAP_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_TYPE_HASH_MAP, GeeHashMapPrivate))
enum  {
	GEE_HASH_MAP_DUMMY_PROPERTY
};
#define GEE_HASH_MAP_MIN_SIZE 11
#define GEE_HASH_MAP_MAX_SIZE 13845163
GeeCollectionObject* gee_collection_object_new (void);
GeeCollectionObject* gee_collection_object_construct (GType object_type);
void gee_hash_map_set_key_hash_func (GeeHashMap* self, GHashFunc value);
void gee_hash_map_set_key_equal_func (GeeHashMap* self, GEqualFunc value);
void gee_hash_map_set_value_equal_func (GeeHashMap* self, GEqualFunc value);
GeeHashMap* gee_hash_map_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GHashFunc key_hash_func, GEqualFunc key_equal_func, GEqualFunc value_equal_func);
GeeHashMap* gee_hash_map_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GHashFunc key_hash_func, GEqualFunc key_equal_func, GEqualFunc value_equal_func);
static GeeHashMapKeySet* gee_hash_map_key_set_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map);
static GeeHashMapKeySet* gee_hash_map_key_set_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map);
static GType gee_hash_map_key_set_get_type (void);
static GeeSet* gee_hash_map_real_get_keys (GeeMap* base);
static GeeHashMapValueCollection* gee_hash_map_value_collection_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map);
static GeeHashMapValueCollection* gee_hash_map_value_collection_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map);
static GType gee_hash_map_value_collection_get_type (void);
static GeeCollection* gee_hash_map_real_get_values (GeeMap* base);
static GeeHashMapNode** gee_hash_map_lookup_node (GeeHashMap* self, gconstpointer key);
static gboolean gee_hash_map_real_contains (GeeMap* base, gconstpointer key);
static gpointer gee_hash_map_real_get (GeeMap* base, gconstpointer key);
static GeeHashMapNode* gee_hash_map_node_new (gpointer k, gpointer v, guint hash);
static GeeHashMapNode* gee_hash_map_node_new (gpointer k, gpointer v, guint hash);
static void gee_hash_map_resize (GeeHashMap* self);
static void gee_hash_map_real_set (GeeMap* base, gconstpointer key, gconstpointer value);
static gboolean gee_hash_map_real_remove (GeeMap* base, gconstpointer key);
static void gee_hash_map_real_clear (GeeMap* base);
void gee_map_clear (GeeMap* self);
static void gee_hash_map_node_free (GeeHashMapNode* self);
static void gee_hash_map_node_instance_init (GeeHashMapNode * self);
#define GEE_HASH_MAP_KEY_SET_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_HASH_MAP_TYPE_KEY_SET, GeeHashMapKeySetPrivate))
enum  {
	GEE_HASH_MAP_KEY_SET_DUMMY_PROPERTY
};
static void gee_hash_map_key_set_set_map (GeeHashMapKeySet* self, GeeHashMap* value);
static GType gee_hash_map_key_set_real_get_element_type (GeeIterable* base);
static GeeHashMapKeyIterator* gee_hash_map_key_iterator_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map);
static GeeHashMapKeyIterator* gee_hash_map_key_iterator_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map);
static GType gee_hash_map_key_iterator_get_type (void);
static GeeIterator* gee_hash_map_key_set_real_iterator (GeeIterable* base);
static gboolean gee_hash_map_key_set_real_add (GeeCollection* base, gconstpointer key);
static void gee_hash_map_key_set_real_clear (GeeCollection* base);
static gboolean gee_hash_map_key_set_real_remove (GeeCollection* base, gconstpointer key);
gboolean gee_map_contains (GeeMap* self, gconstpointer key);
static gboolean gee_hash_map_key_set_real_contains (GeeCollection* base, gconstpointer key);
gint gee_map_get_size (GeeMap* self);
static void gee_hash_map_key_set_finalize (GeeCollectionObject* obj);
#define GEE_HASH_MAP_KEY_ITERATOR_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_HASH_MAP_TYPE_KEY_ITERATOR, GeeHashMapKeyIteratorPrivate))
enum  {
	GEE_HASH_MAP_KEY_ITERATOR_DUMMY_PROPERTY
};
static void gee_hash_map_key_iterator_set_map (GeeHashMapKeyIterator* self, GeeHashMap* value);
static gboolean gee_hash_map_key_iterator_real_next (GeeIterator* base);
static gpointer gee_hash_map_key_iterator_real_get (GeeIterator* base);
static void gee_hash_map_key_iterator_finalize (GeeCollectionObject* obj);
#define GEE_HASH_MAP_VALUE_COLLECTION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_HASH_MAP_TYPE_VALUE_COLLECTION, GeeHashMapValueCollectionPrivate))
enum  {
	GEE_HASH_MAP_VALUE_COLLECTION_DUMMY_PROPERTY
};
static void gee_hash_map_value_collection_set_map (GeeHashMapValueCollection* self, GeeHashMap* value);
static GType gee_hash_map_value_collection_real_get_element_type (GeeIterable* base);
static GeeHashMapValueIterator* gee_hash_map_value_iterator_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map);
static GeeHashMapValueIterator* gee_hash_map_value_iterator_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map);
static GType gee_hash_map_value_iterator_get_type (void);
static GeeIterator* gee_hash_map_value_collection_real_iterator (GeeIterable* base);
static gboolean gee_hash_map_value_collection_real_add (GeeCollection* base, gconstpointer value);
static void gee_hash_map_value_collection_real_clear (GeeCollection* base);
static gboolean gee_hash_map_value_collection_real_remove (GeeCollection* base, gconstpointer value);
GeeIterator* gee_iterable_iterator (GeeIterable* self);
gboolean gee_iterator_next (GeeIterator* self);
gpointer gee_iterator_get (GeeIterator* self);
static gboolean gee_hash_map_value_collection_real_contains (GeeCollection* base, gconstpointer value);
static void gee_hash_map_value_collection_finalize (GeeCollectionObject* obj);
#define GEE_HASH_MAP_VALUE_ITERATOR_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_HASH_MAP_TYPE_VALUE_ITERATOR, GeeHashMapValueIteratorPrivate))
enum  {
	GEE_HASH_MAP_VALUE_ITERATOR_DUMMY_PROPERTY
};
static void gee_hash_map_value_iterator_set_map (GeeHashMapValueIterator* self, GeeHashMap* value);
static gboolean gee_hash_map_value_iterator_real_next (GeeIterator* base);
static gpointer gee_hash_map_value_iterator_real_get (GeeIterator* base);
static void gee_hash_map_value_iterator_finalize (GeeCollectionObject* obj);
static void gee_hash_map_finalize (GeeCollectionObject* obj);
static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);



GeeHashMap* gee_hash_map_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GHashFunc key_hash_func, GEqualFunc key_equal_func, GEqualFunc value_equal_func) {
	GeeHashMap* self;
	GeeHashMapNode** _tmp0_;
	self = (GeeHashMap*) gee_collection_object_construct (object_type);
	self->priv->k_type = k_type;
	self->priv->k_dup_func = k_dup_func;
	self->priv->k_destroy_func = k_destroy_func;
	self->priv->v_type = v_type;
	self->priv->v_dup_func = v_dup_func;
	self->priv->v_destroy_func = v_destroy_func;
	gee_hash_map_set_key_hash_func (self, key_hash_func);
	gee_hash_map_set_key_equal_func (self, key_equal_func);
	gee_hash_map_set_value_equal_func (self, value_equal_func);
	self->priv->_array_size = GEE_HASH_MAP_MIN_SIZE;
	self->priv->_nodes = (_tmp0_ = g_new0 (GeeHashMapNode*, self->priv->_array_size + 1), self->priv->_nodes = (_vala_array_free (self->priv->_nodes, self->priv->_nodes_length1, (GDestroyNotify) gee_hash_map_node_free), NULL), self->priv->_nodes_length1 = self->priv->_array_size, self->priv->_nodes_size = self->priv->_nodes_length1, _tmp0_);
	return self;
}


GeeHashMap* gee_hash_map_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GHashFunc key_hash_func, GEqualFunc key_equal_func, GEqualFunc value_equal_func) {
	return gee_hash_map_construct (GEE_TYPE_HASH_MAP, k_type, k_dup_func, k_destroy_func, v_type, v_dup_func, v_destroy_func, key_hash_func, key_equal_func, value_equal_func);
}


static GeeSet* gee_hash_map_real_get_keys (GeeMap* base) {
	GeeHashMap * self;
	GeeSet* result;
	self = (GeeHashMap*) base;
	result = (GeeSet*) gee_hash_map_key_set_new (self->priv->k_type, (GBoxedCopyFunc) self->priv->k_dup_func, self->priv->k_destroy_func, self->priv->v_type, (GBoxedCopyFunc) self->priv->v_dup_func, self->priv->v_destroy_func, self);
	return result;
}


static GeeCollection* gee_hash_map_real_get_values (GeeMap* base) {
	GeeHashMap * self;
	GeeCollection* result;
	self = (GeeHashMap*) base;
	result = (GeeCollection*) gee_hash_map_value_collection_new (self->priv->k_type, (GBoxedCopyFunc) self->priv->k_dup_func, self->priv->k_destroy_func, self->priv->v_type, (GBoxedCopyFunc) self->priv->v_dup_func, self->priv->v_destroy_func, self);
	return result;
}


static GeeHashMapNode** gee_hash_map_lookup_node (GeeHashMap* self, gconstpointer key) {
	GeeHashMapNode** result;
	guint hash_value;
	GeeHashMapNode** node;
	g_return_val_if_fail (self != NULL, NULL);
	hash_value = self->priv->_key_hash_func (key);
	node = &self->priv->_nodes[hash_value % self->priv->_array_size];
	while (TRUE) {
		gboolean _tmp0_;
		_tmp0_ = FALSE;
		if ((*node) != NULL) {
			gboolean _tmp1_;
			_tmp1_ = FALSE;
			if (hash_value != (*node)->key_hash) {
				_tmp1_ = TRUE;
			} else {
				_tmp1_ = !self->priv->_key_equal_func ((*node)->key, key);
			}
			_tmp0_ = _tmp1_;
		} else {
			_tmp0_ = FALSE;
		}
		if (!_tmp0_) {
			break;
		}
		node = &(*node)->next;
	}
	result = node;
	return result;
}


static gboolean gee_hash_map_real_contains (GeeMap* base, gconstpointer key) {
	GeeHashMap * self;
	gboolean result;
	GeeHashMapNode** node;
	self = (GeeHashMap*) base;
	node = gee_hash_map_lookup_node (self, key);
	result = (*node) != NULL;
	return result;
}


static gpointer gee_hash_map_real_get (GeeMap* base, gconstpointer key) {
	GeeHashMap * self;
	gpointer result;
	GeeHashMapNode* node;
	self = (GeeHashMap*) base;
	node = *gee_hash_map_lookup_node (self, key);
	if (node != NULL) {
		gconstpointer _tmp0_;
		result = (_tmp0_ = node->value, ((_tmp0_ == NULL) || (self->priv->v_dup_func == NULL)) ? ((gpointer) _tmp0_) : self->priv->v_dup_func ((gpointer) _tmp0_));
		return result;
	} else {
		result = NULL;
		return result;
	}
}


static void gee_hash_map_real_set (GeeMap* base, gconstpointer key, gconstpointer value) {
	GeeHashMap * self;
	GeeHashMapNode** node;
	self = (GeeHashMap*) base;
	node = gee_hash_map_lookup_node (self, key);
	if ((*node) != NULL) {
		gpointer _tmp1_;
		gconstpointer _tmp0_;
		(*node)->value = (_tmp1_ = (_tmp0_ = value, ((_tmp0_ == NULL) || (self->priv->v_dup_func == NULL)) ? ((gpointer) _tmp0_) : self->priv->v_dup_func ((gpointer) _tmp0_)), (((*node)->value == NULL) || (self->priv->v_destroy_func == NULL)) ? NULL : ((*node)->value = (self->priv->v_destroy_func ((*node)->value), NULL)), _tmp1_);
	} else {
		guint hash_value;
		gconstpointer _tmp3_;
		gconstpointer _tmp2_;
		hash_value = self->priv->_key_hash_func (key);
		*node = gee_hash_map_node_new ((_tmp2_ = key, ((_tmp2_ == NULL) || (self->priv->k_dup_func == NULL)) ? ((gpointer) _tmp2_) : self->priv->k_dup_func ((gpointer) _tmp2_)), (_tmp3_ = value, ((_tmp3_ == NULL) || (self->priv->v_dup_func == NULL)) ? ((gpointer) _tmp3_) : self->priv->v_dup_func ((gpointer) _tmp3_)), hash_value);
		self->priv->_nnodes++;
		gee_hash_map_resize (self);
	}
	self->priv->_stamp++;
}


static gboolean gee_hash_map_real_remove (GeeMap* base, gconstpointer key) {
	GeeHashMap * self;
	gboolean result;
	GeeHashMapNode** node;
	self = (GeeHashMap*) base;
	node = gee_hash_map_lookup_node (self, key);
	if ((*node) != NULL) {
		GeeHashMapNode* _tmp0_;
		GeeHashMapNode* next;
		gpointer _tmp1_;
		gpointer _tmp2_;
		GeeHashMapNode* _tmp3_;
		next = (_tmp0_ = (*node)->next, (*node)->next = NULL, _tmp0_);
		(*node)->key = (_tmp1_ = NULL, (((*node)->key == NULL) || (self->priv->k_destroy_func == NULL)) ? NULL : ((*node)->key = (self->priv->k_destroy_func ((*node)->key), NULL)), _tmp1_);
		(*node)->value = (_tmp2_ = NULL, (((*node)->value == NULL) || (self->priv->v_destroy_func == NULL)) ? NULL : ((*node)->value = (self->priv->v_destroy_func ((*node)->value), NULL)), _tmp2_);
		gee_hash_map_node_free (*node);
		*node = (_tmp3_ = next, next = NULL, _tmp3_);
		self->priv->_nnodes--;
		gee_hash_map_resize (self);
		self->priv->_stamp++;
		result = TRUE;
		_gee_hash_map_node_free0 (next);
		return result;
	}
	result = FALSE;
	return result;
}


static void gee_hash_map_real_clear (GeeMap* base) {
	GeeHashMap * self;
	self = (GeeHashMap*) base;
	{
		gint i;
		i = 0;
		{
			gboolean _tmp0_;
			_tmp0_ = TRUE;
			while (TRUE) {
				GeeHashMapNode* _tmp1_;
				GeeHashMapNode* node;
				if (!_tmp0_) {
					i++;
				}
				_tmp0_ = FALSE;
				if (!(i < self->priv->_array_size)) {
					break;
				}
				node = (_tmp1_ = self->priv->_nodes[i], self->priv->_nodes[i] = NULL, _tmp1_);
				while (TRUE) {
					GeeHashMapNode* _tmp2_;
					GeeHashMapNode* next;
					gpointer _tmp3_;
					gpointer _tmp4_;
					GeeHashMapNode* _tmp6_;
					GeeHashMapNode* _tmp5_;
					if (!(node != NULL)) {
						break;
					}
					next = (_tmp2_ = node->next, node->next = NULL, _tmp2_);
					node->key = (_tmp3_ = NULL, ((node->key == NULL) || (self->priv->k_destroy_func == NULL)) ? NULL : (node->key = (self->priv->k_destroy_func (node->key), NULL)), _tmp3_);
					node->value = (_tmp4_ = NULL, ((node->value == NULL) || (self->priv->v_destroy_func == NULL)) ? NULL : (node->value = (self->priv->v_destroy_func (node->value), NULL)), _tmp4_);
					node = (_tmp6_ = (_tmp5_ = next, next = NULL, _tmp5_), _gee_hash_map_node_free0 (node), _tmp6_);
					_gee_hash_map_node_free0 (next);
				}
				_gee_hash_map_node_free0 (node);
			}
		}
	}
	self->priv->_nnodes = 0;
	gee_hash_map_resize (self);
}


static void gee_hash_map_resize (GeeHashMap* self) {
	gboolean _tmp0_;
	gboolean _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp0_ = FALSE;
	_tmp1_ = FALSE;
	if (self->priv->_array_size >= (3 * self->priv->_nnodes)) {
		_tmp1_ = self->priv->_array_size >= GEE_HASH_MAP_MIN_SIZE;
	} else {
		_tmp1_ = FALSE;
	}
	if (_tmp1_) {
		_tmp0_ = TRUE;
	} else {
		gboolean _tmp2_;
		_tmp2_ = FALSE;
		if ((3 * self->priv->_array_size) <= self->priv->_nnodes) {
			_tmp2_ = self->priv->_array_size < GEE_HASH_MAP_MAX_SIZE;
		} else {
			_tmp2_ = FALSE;
		}
		_tmp0_ = _tmp2_;
	}
	if (_tmp0_) {
		gint new_array_size;
		GeeHashMapNode** _tmp3_;
		gint new_nodes_size;
		gint new_nodes_length1;
		GeeHashMapNode** new_nodes;
		GeeHashMapNode** _tmp17_;
		GeeHashMapNode** _tmp16_;
		new_array_size = (gint) g_spaced_primes_closest ((guint) self->priv->_nnodes);
		new_array_size = CLAMP (new_array_size, GEE_HASH_MAP_MIN_SIZE, GEE_HASH_MAP_MAX_SIZE);
		new_nodes = (_tmp3_ = g_new0 (GeeHashMapNode*, new_array_size + 1), new_nodes_length1 = new_array_size, new_nodes_size = new_nodes_length1, _tmp3_);
		{
			gint i;
			i = 0;
			{
				gboolean _tmp4_;
				_tmp4_ = TRUE;
				while (TRUE) {
					GeeHashMapNode* node;
					GeeHashMapNode* next;
					if (!_tmp4_) {
						i++;
					}
					_tmp4_ = FALSE;
					if (!(i < self->priv->_array_size)) {
						break;
					}
					node = NULL;
					next = NULL;
					{
						GeeHashMapNode* _tmp6_;
						GeeHashMapNode* _tmp5_;
						gboolean _tmp7_;
						node = (_tmp6_ = (_tmp5_ = self->priv->_nodes[i], self->priv->_nodes[i] = NULL, _tmp5_), _gee_hash_map_node_free0 (node), _tmp6_);
						_tmp7_ = TRUE;
						while (TRUE) {
							GeeHashMapNode* _tmp11_;
							GeeHashMapNode* _tmp10_;
							guint hash_val;
							GeeHashMapNode* _tmp13_;
							GeeHashMapNode* _tmp12_;
							GeeHashMapNode* _tmp15_;
							GeeHashMapNode* _tmp14_;
							if (!_tmp7_) {
								GeeHashMapNode* _tmp9_;
								GeeHashMapNode* _tmp8_;
								node = (_tmp9_ = (_tmp8_ = next, next = NULL, _tmp8_), _gee_hash_map_node_free0 (node), _tmp9_);
							}
							_tmp7_ = FALSE;
							if (!(node != NULL)) {
								break;
							}
							next = (_tmp11_ = (_tmp10_ = node->next, node->next = NULL, _tmp10_), _gee_hash_map_node_free0 (next), _tmp11_);
							hash_val = node->key_hash % new_array_size;
							node->next = (_tmp13_ = (_tmp12_ = new_nodes[hash_val], new_nodes[hash_val] = NULL, _tmp12_), _gee_hash_map_node_free0 (node->next), _tmp13_);
							new_nodes[hash_val] = (_tmp15_ = (_tmp14_ = node, node = NULL, _tmp14_), _gee_hash_map_node_free0 (new_nodes[hash_val]), _tmp15_);
						}
					}
					_gee_hash_map_node_free0 (node);
					_gee_hash_map_node_free0 (next);
				}
			}
		}
		self->priv->_nodes = (_tmp17_ = (_tmp16_ = new_nodes, new_nodes = NULL, _tmp16_), self->priv->_nodes = (_vala_array_free (self->priv->_nodes, self->priv->_nodes_length1, (GDestroyNotify) gee_hash_map_node_free), NULL), self->priv->_nodes_length1 = new_nodes_length1, self->priv->_nodes_size = self->priv->_nodes_length1, _tmp17_);
		self->priv->_array_size = new_array_size;
		new_nodes = (_vala_array_free (new_nodes, new_nodes_length1, (GDestroyNotify) gee_hash_map_node_free), NULL);
	}
}


static gint gee_hash_map_real_get_size (GeeMap* base) {
	gint result;
	GeeHashMap* self;
	self = (GeeHashMap*) base;
	result = self->priv->_nnodes;
	return result;
}


void gee_hash_map_set_key_hash_func (GeeHashMap* self, GHashFunc value) {
	g_return_if_fail (self != NULL);
	self->priv->_key_hash_func = value;
}


void gee_hash_map_set_key_equal_func (GeeHashMap* self, GEqualFunc value) {
	g_return_if_fail (self != NULL);
	self->priv->_key_equal_func = value;
}


void gee_hash_map_set_value_equal_func (GeeHashMap* self, GEqualFunc value) {
	g_return_if_fail (self != NULL);
	self->priv->_value_equal_func = value;
}


static GeeHashMapNode* gee_hash_map_node_new (gpointer k, gpointer v, guint hash) {
	GeeHashMapNode* self;
	gpointer _tmp0_;
	gpointer _tmp1_;
	self = g_slice_new0 (GeeHashMapNode);
	gee_hash_map_node_instance_init (self);
	self->key = (_tmp0_ = k, k = NULL, _tmp0_);
	self->value = (_tmp1_ = v, v = NULL, _tmp1_);
	self->key_hash = hash;
	return self;
}


static void gee_hash_map_node_instance_init (GeeHashMapNode * self) {
}


static void gee_hash_map_node_free (GeeHashMapNode* self) {
	_gee_hash_map_node_free0 (self->next);
	g_slice_free (GeeHashMapNode, self);
}


static GeeHashMapKeySet* gee_hash_map_key_set_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map) {
	GeeHashMapKeySet* self;
	g_return_val_if_fail (map != NULL, NULL);
	self = (GeeHashMapKeySet*) gee_collection_object_construct (object_type);
	self->priv->k_type = k_type;
	self->priv->k_dup_func = k_dup_func;
	self->priv->k_destroy_func = k_destroy_func;
	self->priv->v_type = v_type;
	self->priv->v_dup_func = v_dup_func;
	self->priv->v_destroy_func = v_destroy_func;
	gee_hash_map_key_set_set_map (self, map);
	return self;
}


static GeeHashMapKeySet* gee_hash_map_key_set_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map) {
	return gee_hash_map_key_set_construct (GEE_HASH_MAP_TYPE_KEY_SET, k_type, k_dup_func, k_destroy_func, v_type, v_dup_func, v_destroy_func, map);
}


static GType gee_hash_map_key_set_real_get_element_type (GeeIterable* base) {
	GeeHashMapKeySet * self;
	GType result;
	self = (GeeHashMapKeySet*) base;
	result = self->priv->k_type;
	return result;
}


static GeeIterator* gee_hash_map_key_set_real_iterator (GeeIterable* base) {
	GeeHashMapKeySet * self;
	GeeIterator* result;
	self = (GeeHashMapKeySet*) base;
	result = (GeeIterator*) gee_hash_map_key_iterator_new (self->priv->k_type, (GBoxedCopyFunc) self->priv->k_dup_func, self->priv->k_destroy_func, self->priv->v_type, (GBoxedCopyFunc) self->priv->v_dup_func, self->priv->v_destroy_func, self->priv->_map);
	return result;
}


static gboolean gee_hash_map_key_set_real_add (GeeCollection* base, gconstpointer key) {
	GeeHashMapKeySet * self;
	self = (GeeHashMapKeySet*) base;
	g_assert_not_reached ();
}


static void gee_hash_map_key_set_real_clear (GeeCollection* base) {
	GeeHashMapKeySet * self;
	self = (GeeHashMapKeySet*) base;
	g_assert_not_reached ();
}


static gboolean gee_hash_map_key_set_real_remove (GeeCollection* base, gconstpointer key) {
	GeeHashMapKeySet * self;
	self = (GeeHashMapKeySet*) base;
	g_assert_not_reached ();
}


static gboolean gee_hash_map_key_set_real_contains (GeeCollection* base, gconstpointer key) {
	GeeHashMapKeySet * self;
	gboolean result;
	self = (GeeHashMapKeySet*) base;
	result = gee_map_contains ((GeeMap*) self->priv->_map, key);
	return result;
}


static gpointer _gee_collection_object_ref0 (gpointer self) {
	return self ? gee_collection_object_ref (self) : NULL;
}


static void gee_hash_map_key_set_set_map (GeeHashMapKeySet* self, GeeHashMap* value) {
	GeeHashMap* _tmp0_;
	g_return_if_fail (self != NULL);
	self->priv->_map = (_tmp0_ = _gee_collection_object_ref0 (value), _gee_collection_object_unref0 (self->priv->_map), _tmp0_);
}


static gint gee_hash_map_key_set_real_get_size (GeeCollection* base) {
	gint result;
	GeeHashMapKeySet* self;
	self = (GeeHashMapKeySet*) base;
	result = gee_map_get_size ((GeeMap*) self->priv->_map);
	return result;
}


static void gee_hash_map_key_set_class_init (GeeHashMapKeySetClass * klass) {
	gee_hash_map_key_set_parent_class = g_type_class_peek_parent (klass);
	GEE_COLLECTION_OBJECT_CLASS (klass)->finalize = gee_hash_map_key_set_finalize;
	g_type_class_add_private (klass, sizeof (GeeHashMapKeySetPrivate));
}


static void gee_hash_map_key_set_gee_iterable_interface_init (GeeIterableIface * iface) {
	gee_hash_map_key_set_gee_iterable_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_element_type = gee_hash_map_key_set_real_get_element_type;
	iface->iterator = gee_hash_map_key_set_real_iterator;
}


static void gee_hash_map_key_set_gee_collection_interface_init (GeeCollectionIface * iface) {
	gee_hash_map_key_set_gee_collection_parent_iface = g_type_interface_peek_parent (iface);
	iface->add = gee_hash_map_key_set_real_add;
	iface->clear = gee_hash_map_key_set_real_clear;
	iface->remove = gee_hash_map_key_set_real_remove;
	iface->contains = gee_hash_map_key_set_real_contains;
	iface->get_size = gee_hash_map_key_set_real_get_size;
}


static void gee_hash_map_key_set_gee_set_interface_init (GeeSetIface * iface) {
	gee_hash_map_key_set_gee_set_parent_iface = g_type_interface_peek_parent (iface);
}


static void gee_hash_map_key_set_instance_init (GeeHashMapKeySet * self) {
	self->priv = GEE_HASH_MAP_KEY_SET_GET_PRIVATE (self);
}


static void gee_hash_map_key_set_finalize (GeeCollectionObject* obj) {
	GeeHashMapKeySet * self;
	self = GEE_HASH_MAP_KEY_SET (obj);
	_gee_collection_object_unref0 (self->priv->_map);
	GEE_COLLECTION_OBJECT_CLASS (gee_hash_map_key_set_parent_class)->finalize (obj);
}


static GType gee_hash_map_key_set_get_type (void) {
	static GType gee_hash_map_key_set_type_id = 0;
	if (gee_hash_map_key_set_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeHashMapKeySetClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_hash_map_key_set_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeHashMapKeySet), 0, (GInstanceInitFunc) gee_hash_map_key_set_instance_init, NULL };
		static const GInterfaceInfo gee_iterable_info = { (GInterfaceInitFunc) gee_hash_map_key_set_gee_iterable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		static const GInterfaceInfo gee_collection_info = { (GInterfaceInitFunc) gee_hash_map_key_set_gee_collection_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		static const GInterfaceInfo gee_set_info = { (GInterfaceInitFunc) gee_hash_map_key_set_gee_set_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_hash_map_key_set_type_id = g_type_register_static (GEE_TYPE_COLLECTION_OBJECT, "GeeHashMapKeySet", &g_define_type_info, 0);
		g_type_add_interface_static (gee_hash_map_key_set_type_id, GEE_TYPE_ITERABLE, &gee_iterable_info);
		g_type_add_interface_static (gee_hash_map_key_set_type_id, GEE_TYPE_COLLECTION, &gee_collection_info);
		g_type_add_interface_static (gee_hash_map_key_set_type_id, GEE_TYPE_SET, &gee_set_info);
	}
	return gee_hash_map_key_set_type_id;
}


static GeeHashMapKeyIterator* gee_hash_map_key_iterator_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map) {
	GeeHashMapKeyIterator* self;
	g_return_val_if_fail (map != NULL, NULL);
	self = (GeeHashMapKeyIterator*) gee_collection_object_construct (object_type);
	self->priv->k_type = k_type;
	self->priv->k_dup_func = k_dup_func;
	self->priv->k_destroy_func = k_destroy_func;
	self->priv->v_type = v_type;
	self->priv->v_dup_func = v_dup_func;
	self->priv->v_destroy_func = v_destroy_func;
	gee_hash_map_key_iterator_set_map (self, map);
	return self;
}


static GeeHashMapKeyIterator* gee_hash_map_key_iterator_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map) {
	return gee_hash_map_key_iterator_construct (GEE_HASH_MAP_TYPE_KEY_ITERATOR, k_type, k_dup_func, k_destroy_func, v_type, v_dup_func, v_destroy_func, map);
}


static gboolean gee_hash_map_key_iterator_real_next (GeeIterator* base) {
	GeeHashMapKeyIterator * self;
	gboolean result;
	self = (GeeHashMapKeyIterator*) base;
	if (self->priv->_node != NULL) {
		self->priv->_node = self->priv->_node->next;
	}
	while (TRUE) {
		gboolean _tmp0_;
		_tmp0_ = FALSE;
		if (self->priv->_node == NULL) {
			_tmp0_ = (self->priv->_index + 1) < self->priv->_map->priv->_array_size;
		} else {
			_tmp0_ = FALSE;
		}
		if (!_tmp0_) {
			break;
		}
		self->priv->_index++;
		self->priv->_node = self->priv->_map->priv->_nodes[self->priv->_index];
	}
	result = self->priv->_node != NULL;
	return result;
}


static gpointer gee_hash_map_key_iterator_real_get (GeeIterator* base) {
	GeeHashMapKeyIterator * self;
	gpointer result;
	gconstpointer _tmp0_;
	self = (GeeHashMapKeyIterator*) base;
	g_assert (self->priv->_stamp == self->priv->_map->priv->_stamp);
	g_assert (self->priv->_node != NULL);
	result = (_tmp0_ = self->priv->_node->key, ((_tmp0_ == NULL) || (self->priv->k_dup_func == NULL)) ? ((gpointer) _tmp0_) : self->priv->k_dup_func ((gpointer) _tmp0_));
	return result;
}


static void gee_hash_map_key_iterator_set_map (GeeHashMapKeyIterator* self, GeeHashMap* value) {
	GeeHashMap* _tmp0_;
	g_return_if_fail (self != NULL);
	self->priv->_map = (_tmp0_ = _gee_collection_object_ref0 (value), _gee_collection_object_unref0 (self->priv->_map), _tmp0_);
	self->priv->_stamp = self->priv->_map->priv->_stamp;
}


static void gee_hash_map_key_iterator_class_init (GeeHashMapKeyIteratorClass * klass) {
	gee_hash_map_key_iterator_parent_class = g_type_class_peek_parent (klass);
	GEE_COLLECTION_OBJECT_CLASS (klass)->finalize = gee_hash_map_key_iterator_finalize;
	g_type_class_add_private (klass, sizeof (GeeHashMapKeyIteratorPrivate));
}


static void gee_hash_map_key_iterator_gee_iterator_interface_init (GeeIteratorIface * iface) {
	gee_hash_map_key_iterator_gee_iterator_parent_iface = g_type_interface_peek_parent (iface);
	iface->next = gee_hash_map_key_iterator_real_next;
	iface->get = gee_hash_map_key_iterator_real_get;
}


static void gee_hash_map_key_iterator_instance_init (GeeHashMapKeyIterator * self) {
	self->priv = GEE_HASH_MAP_KEY_ITERATOR_GET_PRIVATE (self);
	self->priv->_index = -1;
}


static void gee_hash_map_key_iterator_finalize (GeeCollectionObject* obj) {
	GeeHashMapKeyIterator * self;
	self = GEE_HASH_MAP_KEY_ITERATOR (obj);
	_gee_collection_object_unref0 (self->priv->_map);
	GEE_COLLECTION_OBJECT_CLASS (gee_hash_map_key_iterator_parent_class)->finalize (obj);
}


static GType gee_hash_map_key_iterator_get_type (void) {
	static GType gee_hash_map_key_iterator_type_id = 0;
	if (gee_hash_map_key_iterator_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeHashMapKeyIteratorClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_hash_map_key_iterator_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeHashMapKeyIterator), 0, (GInstanceInitFunc) gee_hash_map_key_iterator_instance_init, NULL };
		static const GInterfaceInfo gee_iterator_info = { (GInterfaceInitFunc) gee_hash_map_key_iterator_gee_iterator_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_hash_map_key_iterator_type_id = g_type_register_static (GEE_TYPE_COLLECTION_OBJECT, "GeeHashMapKeyIterator", &g_define_type_info, 0);
		g_type_add_interface_static (gee_hash_map_key_iterator_type_id, GEE_TYPE_ITERATOR, &gee_iterator_info);
	}
	return gee_hash_map_key_iterator_type_id;
}


static GeeHashMapValueCollection* gee_hash_map_value_collection_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map) {
	GeeHashMapValueCollection* self;
	g_return_val_if_fail (map != NULL, NULL);
	self = (GeeHashMapValueCollection*) gee_collection_object_construct (object_type);
	self->priv->k_type = k_type;
	self->priv->k_dup_func = k_dup_func;
	self->priv->k_destroy_func = k_destroy_func;
	self->priv->v_type = v_type;
	self->priv->v_dup_func = v_dup_func;
	self->priv->v_destroy_func = v_destroy_func;
	gee_hash_map_value_collection_set_map (self, map);
	return self;
}


static GeeHashMapValueCollection* gee_hash_map_value_collection_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map) {
	return gee_hash_map_value_collection_construct (GEE_HASH_MAP_TYPE_VALUE_COLLECTION, k_type, k_dup_func, k_destroy_func, v_type, v_dup_func, v_destroy_func, map);
}


static GType gee_hash_map_value_collection_real_get_element_type (GeeIterable* base) {
	GeeHashMapValueCollection * self;
	GType result;
	self = (GeeHashMapValueCollection*) base;
	result = self->priv->v_type;
	return result;
}


static GeeIterator* gee_hash_map_value_collection_real_iterator (GeeIterable* base) {
	GeeHashMapValueCollection * self;
	GeeIterator* result;
	self = (GeeHashMapValueCollection*) base;
	result = (GeeIterator*) gee_hash_map_value_iterator_new (self->priv->k_type, (GBoxedCopyFunc) self->priv->k_dup_func, self->priv->k_destroy_func, self->priv->v_type, (GBoxedCopyFunc) self->priv->v_dup_func, self->priv->v_destroy_func, self->priv->_map);
	return result;
}


static gboolean gee_hash_map_value_collection_real_add (GeeCollection* base, gconstpointer value) {
	GeeHashMapValueCollection * self;
	self = (GeeHashMapValueCollection*) base;
	g_assert_not_reached ();
}


static void gee_hash_map_value_collection_real_clear (GeeCollection* base) {
	GeeHashMapValueCollection * self;
	self = (GeeHashMapValueCollection*) base;
	g_assert_not_reached ();
}


static gboolean gee_hash_map_value_collection_real_remove (GeeCollection* base, gconstpointer value) {
	GeeHashMapValueCollection * self;
	self = (GeeHashMapValueCollection*) base;
	g_assert_not_reached ();
}


static gboolean gee_hash_map_value_collection_real_contains (GeeCollection* base, gconstpointer value) {
	GeeHashMapValueCollection * self;
	gboolean result;
	GeeIterator* it;
	self = (GeeHashMapValueCollection*) base;
	it = gee_iterable_iterator ((GeeIterable*) self);
	while (TRUE) {
		gpointer _tmp0_;
		gboolean _tmp1_;
		if (!gee_iterator_next (it)) {
			break;
		}
		if ((_tmp1_ = self->priv->_map->priv->_value_equal_func (_tmp0_ = gee_iterator_get (it), value), ((_tmp0_ == NULL) || (self->priv->v_destroy_func == NULL)) ? NULL : (_tmp0_ = (self->priv->v_destroy_func (_tmp0_), NULL)), _tmp1_)) {
			result = TRUE;
			_gee_collection_object_unref0 (it);
			return result;
		}
	}
	result = FALSE;
	_gee_collection_object_unref0 (it);
	return result;
}


static void gee_hash_map_value_collection_set_map (GeeHashMapValueCollection* self, GeeHashMap* value) {
	GeeHashMap* _tmp0_;
	g_return_if_fail (self != NULL);
	self->priv->_map = (_tmp0_ = _gee_collection_object_ref0 (value), _gee_collection_object_unref0 (self->priv->_map), _tmp0_);
}


static gint gee_hash_map_value_collection_real_get_size (GeeCollection* base) {
	gint result;
	GeeHashMapValueCollection* self;
	self = (GeeHashMapValueCollection*) base;
	result = gee_map_get_size ((GeeMap*) self->priv->_map);
	return result;
}


static void gee_hash_map_value_collection_class_init (GeeHashMapValueCollectionClass * klass) {
	gee_hash_map_value_collection_parent_class = g_type_class_peek_parent (klass);
	GEE_COLLECTION_OBJECT_CLASS (klass)->finalize = gee_hash_map_value_collection_finalize;
	g_type_class_add_private (klass, sizeof (GeeHashMapValueCollectionPrivate));
}


static void gee_hash_map_value_collection_gee_iterable_interface_init (GeeIterableIface * iface) {
	gee_hash_map_value_collection_gee_iterable_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_element_type = gee_hash_map_value_collection_real_get_element_type;
	iface->iterator = gee_hash_map_value_collection_real_iterator;
}


static void gee_hash_map_value_collection_gee_collection_interface_init (GeeCollectionIface * iface) {
	gee_hash_map_value_collection_gee_collection_parent_iface = g_type_interface_peek_parent (iface);
	iface->add = gee_hash_map_value_collection_real_add;
	iface->clear = gee_hash_map_value_collection_real_clear;
	iface->remove = gee_hash_map_value_collection_real_remove;
	iface->contains = gee_hash_map_value_collection_real_contains;
	iface->get_size = gee_hash_map_value_collection_real_get_size;
}


static void gee_hash_map_value_collection_instance_init (GeeHashMapValueCollection * self) {
	self->priv = GEE_HASH_MAP_VALUE_COLLECTION_GET_PRIVATE (self);
}


static void gee_hash_map_value_collection_finalize (GeeCollectionObject* obj) {
	GeeHashMapValueCollection * self;
	self = GEE_HASH_MAP_VALUE_COLLECTION (obj);
	_gee_collection_object_unref0 (self->priv->_map);
	GEE_COLLECTION_OBJECT_CLASS (gee_hash_map_value_collection_parent_class)->finalize (obj);
}


static GType gee_hash_map_value_collection_get_type (void) {
	static GType gee_hash_map_value_collection_type_id = 0;
	if (gee_hash_map_value_collection_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeHashMapValueCollectionClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_hash_map_value_collection_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeHashMapValueCollection), 0, (GInstanceInitFunc) gee_hash_map_value_collection_instance_init, NULL };
		static const GInterfaceInfo gee_iterable_info = { (GInterfaceInitFunc) gee_hash_map_value_collection_gee_iterable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		static const GInterfaceInfo gee_collection_info = { (GInterfaceInitFunc) gee_hash_map_value_collection_gee_collection_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_hash_map_value_collection_type_id = g_type_register_static (GEE_TYPE_COLLECTION_OBJECT, "GeeHashMapValueCollection", &g_define_type_info, 0);
		g_type_add_interface_static (gee_hash_map_value_collection_type_id, GEE_TYPE_ITERABLE, &gee_iterable_info);
		g_type_add_interface_static (gee_hash_map_value_collection_type_id, GEE_TYPE_COLLECTION, &gee_collection_info);
	}
	return gee_hash_map_value_collection_type_id;
}


static GeeHashMapValueIterator* gee_hash_map_value_iterator_construct (GType object_type, GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map) {
	GeeHashMapValueIterator* self;
	g_return_val_if_fail (map != NULL, NULL);
	self = (GeeHashMapValueIterator*) gee_collection_object_construct (object_type);
	self->priv->k_type = k_type;
	self->priv->k_dup_func = k_dup_func;
	self->priv->k_destroy_func = k_destroy_func;
	self->priv->v_type = v_type;
	self->priv->v_dup_func = v_dup_func;
	self->priv->v_destroy_func = v_destroy_func;
	gee_hash_map_value_iterator_set_map (self, map);
	return self;
}


static GeeHashMapValueIterator* gee_hash_map_value_iterator_new (GType k_type, GBoxedCopyFunc k_dup_func, GDestroyNotify k_destroy_func, GType v_type, GBoxedCopyFunc v_dup_func, GDestroyNotify v_destroy_func, GeeHashMap* map) {
	return gee_hash_map_value_iterator_construct (GEE_HASH_MAP_TYPE_VALUE_ITERATOR, k_type, k_dup_func, k_destroy_func, v_type, v_dup_func, v_destroy_func, map);
}


static gboolean gee_hash_map_value_iterator_real_next (GeeIterator* base) {
	GeeHashMapValueIterator * self;
	gboolean result;
	self = (GeeHashMapValueIterator*) base;
	if (self->priv->_node != NULL) {
		self->priv->_node = self->priv->_node->next;
	}
	while (TRUE) {
		gboolean _tmp0_;
		_tmp0_ = FALSE;
		if (self->priv->_node == NULL) {
			_tmp0_ = (self->priv->_index + 1) < self->priv->_map->priv->_array_size;
		} else {
			_tmp0_ = FALSE;
		}
		if (!_tmp0_) {
			break;
		}
		self->priv->_index++;
		self->priv->_node = self->priv->_map->priv->_nodes[self->priv->_index];
	}
	result = self->priv->_node != NULL;
	return result;
}


static gpointer gee_hash_map_value_iterator_real_get (GeeIterator* base) {
	GeeHashMapValueIterator * self;
	gpointer result;
	gconstpointer _tmp0_;
	self = (GeeHashMapValueIterator*) base;
	g_assert (self->priv->_stamp == self->priv->_map->priv->_stamp);
	g_assert (self->priv->_node != NULL);
	result = (_tmp0_ = self->priv->_node->value, ((_tmp0_ == NULL) || (self->priv->v_dup_func == NULL)) ? ((gpointer) _tmp0_) : self->priv->v_dup_func ((gpointer) _tmp0_));
	return result;
}


static void gee_hash_map_value_iterator_set_map (GeeHashMapValueIterator* self, GeeHashMap* value) {
	GeeHashMap* _tmp0_;
	g_return_if_fail (self != NULL);
	self->priv->_map = (_tmp0_ = _gee_collection_object_ref0 (value), _gee_collection_object_unref0 (self->priv->_map), _tmp0_);
	self->priv->_stamp = self->priv->_map->priv->_stamp;
}


static void gee_hash_map_value_iterator_class_init (GeeHashMapValueIteratorClass * klass) {
	gee_hash_map_value_iterator_parent_class = g_type_class_peek_parent (klass);
	GEE_COLLECTION_OBJECT_CLASS (klass)->finalize = gee_hash_map_value_iterator_finalize;
	g_type_class_add_private (klass, sizeof (GeeHashMapValueIteratorPrivate));
}


static void gee_hash_map_value_iterator_gee_iterator_interface_init (GeeIteratorIface * iface) {
	gee_hash_map_value_iterator_gee_iterator_parent_iface = g_type_interface_peek_parent (iface);
	iface->next = gee_hash_map_value_iterator_real_next;
	iface->get = gee_hash_map_value_iterator_real_get;
}


static void gee_hash_map_value_iterator_instance_init (GeeHashMapValueIterator * self) {
	self->priv = GEE_HASH_MAP_VALUE_ITERATOR_GET_PRIVATE (self);
	self->priv->_index = -1;
}


static void gee_hash_map_value_iterator_finalize (GeeCollectionObject* obj) {
	GeeHashMapValueIterator * self;
	self = GEE_HASH_MAP_VALUE_ITERATOR (obj);
	_gee_collection_object_unref0 (self->priv->_map);
	GEE_COLLECTION_OBJECT_CLASS (gee_hash_map_value_iterator_parent_class)->finalize (obj);
}


static GType gee_hash_map_value_iterator_get_type (void) {
	static GType gee_hash_map_value_iterator_type_id = 0;
	if (gee_hash_map_value_iterator_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeHashMapValueIteratorClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_hash_map_value_iterator_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeHashMapValueIterator), 0, (GInstanceInitFunc) gee_hash_map_value_iterator_instance_init, NULL };
		static const GInterfaceInfo gee_iterator_info = { (GInterfaceInitFunc) gee_hash_map_value_iterator_gee_iterator_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_hash_map_value_iterator_type_id = g_type_register_static (GEE_TYPE_COLLECTION_OBJECT, "GeeHashMapValueIterator", &g_define_type_info, 0);
		g_type_add_interface_static (gee_hash_map_value_iterator_type_id, GEE_TYPE_ITERATOR, &gee_iterator_info);
	}
	return gee_hash_map_value_iterator_type_id;
}


static void gee_hash_map_class_init (GeeHashMapClass * klass) {
	gee_hash_map_parent_class = g_type_class_peek_parent (klass);
	GEE_COLLECTION_OBJECT_CLASS (klass)->finalize = gee_hash_map_finalize;
	g_type_class_add_private (klass, sizeof (GeeHashMapPrivate));
}


static void gee_hash_map_gee_map_interface_init (GeeMapIface * iface) {
	gee_hash_map_gee_map_parent_iface = g_type_interface_peek_parent (iface);
	iface->get_keys = gee_hash_map_real_get_keys;
	iface->get_values = gee_hash_map_real_get_values;
	iface->contains = gee_hash_map_real_contains;
	iface->get = gee_hash_map_real_get;
	iface->set = gee_hash_map_real_set;
	iface->remove = gee_hash_map_real_remove;
	iface->clear = gee_hash_map_real_clear;
	iface->get_size = gee_hash_map_real_get_size;
}


static void gee_hash_map_instance_init (GeeHashMap * self) {
	self->priv = GEE_HASH_MAP_GET_PRIVATE (self);
	self->priv->_stamp = 0;
}


static void gee_hash_map_finalize (GeeCollectionObject* obj) {
	GeeHashMap * self;
	self = GEE_HASH_MAP (obj);
	{
		gee_map_clear ((GeeMap*) self);
	}
	self->priv->_nodes = (_vala_array_free (self->priv->_nodes, self->priv->_nodes_length1, (GDestroyNotify) gee_hash_map_node_free), NULL);
	GEE_COLLECTION_OBJECT_CLASS (gee_hash_map_parent_class)->finalize (obj);
}


GType gee_hash_map_get_type (void) {
	static GType gee_hash_map_type_id = 0;
	if (gee_hash_map_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeHashMapClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_hash_map_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeHashMap), 0, (GInstanceInitFunc) gee_hash_map_instance_init, NULL };
		static const GInterfaceInfo gee_map_info = { (GInterfaceInitFunc) gee_hash_map_gee_map_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_hash_map_type_id = g_type_register_static (GEE_TYPE_COLLECTION_OBJECT, "GeeHashMap", &g_define_type_info, 0);
		g_type_add_interface_static (gee_hash_map_type_id, GEE_TYPE_MAP, &gee_map_info);
	}
	return gee_hash_map_type_id;
}


static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	_vala_array_destroy (array, array_length, destroy_func);
	g_free (array);
}




