/* valareport.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>


#define VALA_TYPE_REPORT (vala_report_get_type ())
#define VALA_REPORT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_REPORT, ValaReport))
#define VALA_REPORT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_REPORT, ValaReportClass))
#define VALA_IS_REPORT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_REPORT))
#define VALA_IS_REPORT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_REPORT))
#define VALA_REPORT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_REPORT, ValaReportClass))

typedef struct _ValaReport ValaReport;
typedef struct _ValaReportClass ValaReportClass;
typedef struct _ValaReportPrivate ValaReportPrivate;

#define VALA_TYPE_SOURCE_REFERENCE (vala_source_reference_get_type ())
#define VALA_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReference))
#define VALA_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))
#define VALA_IS_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_IS_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_SOURCE_REFERENCE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))

typedef struct _ValaSourceReference ValaSourceReference;
typedef struct _ValaSourceReferenceClass ValaSourceReferenceClass;

#define VALA_TYPE_SOURCE_FILE (vala_source_file_get_type ())
#define VALA_SOURCE_FILE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_FILE, ValaSourceFile))
#define VALA_SOURCE_FILE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_FILE, ValaSourceFileClass))
#define VALA_IS_SOURCE_FILE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_FILE))
#define VALA_IS_SOURCE_FILE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_FILE))
#define VALA_SOURCE_FILE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_FILE, ValaSourceFileClass))

typedef struct _ValaSourceFile ValaSourceFile;
typedef struct _ValaSourceFileClass ValaSourceFileClass;

#define VALA_TYPE_CODE_CONTEXT (vala_code_context_get_type ())
#define VALA_CODE_CONTEXT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_CONTEXT, ValaCodeContext))
#define VALA_CODE_CONTEXT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_CONTEXT, ValaCodeContextClass))
#define VALA_IS_CODE_CONTEXT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_CONTEXT))
#define VALA_IS_CODE_CONTEXT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_CONTEXT))
#define VALA_CODE_CONTEXT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_CONTEXT, ValaCodeContextClass))

typedef struct _ValaCodeContext ValaCodeContext;
typedef struct _ValaCodeContextClass ValaCodeContextClass;

/**
 * Namespace to centralize reporting warnings and errors.
 */
struct _ValaReport {
	GObject parent_instance;
	ValaReportPrivate * priv;
};

struct _ValaReportClass {
	GObjectClass parent_class;
	void (*warn) (ValaReport* self, ValaSourceReference* source, const char* message);
	void (*err) (ValaReport* self, ValaSourceReference* source, const char* message);
};

struct _ValaReportPrivate {
	gint warnings;
	gint errors;
	gboolean verbose_errors;
};


static gpointer vala_report_parent_class = NULL;

GType vala_report_get_type (void);
gpointer vala_source_reference_ref (gpointer instance);
void vala_source_reference_unref (gpointer instance);
GParamSpec* vala_param_spec_source_reference (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_reference (GValue* value, gpointer v_object);
gpointer vala_value_get_source_reference (const GValue* value);
GType vala_source_reference_get_type (void);
#define VALA_REPORT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_REPORT, ValaReportPrivate))
enum  {
	VALA_REPORT_DUMMY_PROPERTY
};
void vala_report_set_verbose_errors (ValaReport* self, gboolean verbose);
gint vala_report_get_warnings (ValaReport* self);
gint vala_report_get_errors (ValaReport* self);
gint vala_source_reference_get_first_line (ValaSourceReference* self);
gint vala_source_reference_get_last_line (ValaSourceReference* self);
gpointer vala_source_file_ref (gpointer instance);
void vala_source_file_unref (gpointer instance);
GParamSpec* vala_param_spec_source_file (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_file (GValue* value, gpointer v_object);
gpointer vala_value_get_source_file (const GValue* value);
GType vala_source_file_get_type (void);
ValaSourceFile* vala_source_reference_get_file (ValaSourceReference* self);
char* vala_source_file_get_source_line (ValaSourceFile* self, gint lineno);
gint vala_source_reference_get_first_column (ValaSourceReference* self);
gint vala_source_reference_get_last_column (ValaSourceReference* self);
static void vala_report_report_source (ValaSourceReference* source);
char* vala_source_reference_to_string (ValaSourceReference* self);
void vala_report_warn (ValaReport* self, ValaSourceReference* source, const char* message);
static void vala_report_real_warn (ValaReport* self, ValaSourceReference* source, const char* message);
void vala_report_err (ValaReport* self, ValaSourceReference* source, const char* message);
static void vala_report_real_err (ValaReport* self, ValaSourceReference* source, const char* message);
gpointer vala_code_context_ref (gpointer instance);
void vala_code_context_unref (gpointer instance);
GParamSpec* vala_param_spec_code_context (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_context (GValue* value, gpointer v_object);
gpointer vala_value_get_code_context (const GValue* value);
GType vala_code_context_get_type (void);
ValaCodeContext* vala_code_context_get (void);
ValaReport* vala_code_context_get_report (ValaCodeContext* self);
void vala_report_warning (ValaSourceReference* source, const char* message);
void vala_report_error (ValaSourceReference* source, const char* message);
ValaReport* vala_report_new (void);
ValaReport* vala_report_construct (GType object_type);
static void vala_report_finalize (GObject* obj);



/**
 * Set the error verbosity.
 */
void vala_report_set_verbose_errors (ValaReport* self, gboolean verbose) {
	g_return_if_fail (self != NULL);
	self->priv->verbose_errors = verbose;
}


/**
 * Returns the total number of warnings reported.
 */
gint vala_report_get_warnings (ValaReport* self) {
	gint result;
	g_return_val_if_fail (self != NULL, 0);
	result = self->priv->warnings;
	return result;
}


/**
 * Returns the total number of errors reported.
 */
gint vala_report_get_errors (ValaReport* self) {
	gint result;
	g_return_val_if_fail (self != NULL, 0);
	result = self->priv->errors;
	return result;
}


/**
 * Pretty-print the actual line of offending code if possible.
 */
static void vala_report_report_source (ValaSourceReference* source) {
	char* offending_line;
	g_return_if_fail (source != NULL);
	if (vala_source_reference_get_first_line (source) != vala_source_reference_get_last_line (source)) {
		/* FIXME Cannot report multi-line issues currently*/
		return;
	}
	offending_line = vala_source_file_get_source_line (vala_source_reference_get_file (source), vala_source_reference_get_first_line (source));
	if (offending_line != NULL) {
		gint idx;
		fprintf (stderr, "%s\n", offending_line);
		idx = 0;
		/* We loop in this manner so that we don't fall over on differing
		 * tab widths. This means we get the ^s in the right places.
		 */
		{
			gboolean _tmp0_;
			idx = 1;
			/* We loop in this manner so that we don't fall over on differing
			 * tab widths. This means we get the ^s in the right places.
			 */
			_tmp0_ = TRUE;
			/* We loop in this manner so that we don't fall over on differing
			 * tab widths. This means we get the ^s in the right places.
			 */
			while (TRUE) {
				/* We loop in this manner so that we don't fall over on differing
				 * tab widths. This means we get the ^s in the right places.
				 */
				if (!_tmp0_) {
					idx = idx + 1;
				}
				/* We loop in this manner so that we don't fall over on differing
				 * tab widths. This means we get the ^s in the right places.
				 */
				_tmp0_ = FALSE;
				if (!(idx < vala_source_reference_get_first_column (source))) {
					break;
				}
				if (g_utf8_get_char (g_utf8_offset_to_pointer (offending_line, idx - 1)) == '\t') {
					fprintf (stderr, "\t");
				} else {
					fprintf (stderr, " ");
				}
			}
		}
		{
			gboolean _tmp1_;
			idx = vala_source_reference_get_first_column (source);
			_tmp1_ = TRUE;
			while (TRUE) {
				if (!_tmp1_) {
					idx = idx + 1;
				}
				_tmp1_ = FALSE;
				if (!(idx <= vala_source_reference_get_last_column (source))) {
					break;
				}
				if (g_utf8_get_char (g_utf8_offset_to_pointer (offending_line, idx - 1)) == '\t') {
					fprintf (stderr, "\t");
				} else {
					fprintf (stderr, "^");
				}
			}
		}
		fprintf (stderr, "\n");
	}
	offending_line = (g_free (offending_line), NULL);
}


/**
 * Reports the specified message as warning.
 *
 * @param source  reference to source code
 * @param message warning message
 */
static void vala_report_real_warn (ValaReport* self, ValaSourceReference* source, const char* message) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (message != NULL);
	self->priv->warnings++;
	if (source == NULL) {
		fprintf (stderr, "warning: %s\n", message);
	} else {
		char* _tmp0_;
		_tmp0_ = NULL;
		fprintf (stderr, "%s: warning: %s\n", _tmp0_ = vala_source_reference_to_string (source), message);
		_tmp0_ = (g_free (_tmp0_), NULL);
		if (self->priv->verbose_errors) {
			vala_report_report_source (source);
		}
	}
}


void vala_report_warn (ValaReport* self, ValaSourceReference* source, const char* message) {
	VALA_REPORT_GET_CLASS (self)->warn (self, source, message);
}


/**
 * Reports the specified message as error.
 *
 * @param source  reference to source code
 * @param message error message
 */
static void vala_report_real_err (ValaReport* self, ValaSourceReference* source, const char* message) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (message != NULL);
	self->priv->errors++;
	if (source == NULL) {
		fprintf (stderr, "error: %s\n", message);
	} else {
		char* _tmp0_;
		_tmp0_ = NULL;
		fprintf (stderr, "%s: error: %s\n", _tmp0_ = vala_source_reference_to_string (source), message);
		_tmp0_ = (g_free (_tmp0_), NULL);
		if (self->priv->verbose_errors) {
			vala_report_report_source (source);
		}
	}
}


void vala_report_err (ValaReport* self, ValaSourceReference* source, const char* message) {
	VALA_REPORT_GET_CLASS (self)->err (self, source, message);
}


/* Convenience methods calling warn and err on correct instance */
void vala_report_warning (ValaSourceReference* source, const char* message) {
	ValaCodeContext* _tmp0_;
	g_return_if_fail (message != NULL);
	_tmp0_ = NULL;
	vala_report_warn (vala_code_context_get_report (_tmp0_ = vala_code_context_get ()), source, message);
	(_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_code_context_unref (_tmp0_), NULL));
}


void vala_report_error (ValaSourceReference* source, const char* message) {
	ValaCodeContext* _tmp0_;
	g_return_if_fail (message != NULL);
	_tmp0_ = NULL;
	vala_report_err (vala_code_context_get_report (_tmp0_ = vala_code_context_get ()), source, message);
	(_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_code_context_unref (_tmp0_), NULL));
}


/**
 * Namespace to centralize reporting warnings and errors.
 */
ValaReport* vala_report_construct (GType object_type) {
	ValaReport * self;
	self = g_object_newv (object_type, 0, NULL);
	return self;
}


ValaReport* vala_report_new (void) {
	return vala_report_construct (VALA_TYPE_REPORT);
}


static void vala_report_class_init (ValaReportClass * klass) {
	vala_report_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaReportPrivate));
	VALA_REPORT_CLASS (klass)->warn = vala_report_real_warn;
	VALA_REPORT_CLASS (klass)->err = vala_report_real_err;
	G_OBJECT_CLASS (klass)->finalize = vala_report_finalize;
}


static void vala_report_instance_init (ValaReport * self) {
	self->priv = VALA_REPORT_GET_PRIVATE (self);
}


static void vala_report_finalize (GObject* obj) {
	ValaReport * self;
	self = VALA_REPORT (obj);
	G_OBJECT_CLASS (vala_report_parent_class)->finalize (obj);
}


GType vala_report_get_type (void) {
	static GType vala_report_type_id = 0;
	if (vala_report_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaReportClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_report_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaReport), 0, (GInstanceInitFunc) vala_report_instance_init, NULL };
		vala_report_type_id = g_type_register_static (G_TYPE_OBJECT, "ValaReport", &g_define_type_info, 0);
	}
	return vala_report_type_id;
}




