/* valacompiler.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 * Copyright (C) 1996-2002, 2004, 2005, 2006 Free Software Foundation, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <vala.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <glib/gstdio.h>
#include <valacodegen.h>
#include <config.h>
#include <gobject/gvaluecollector.h>


#define VALA_TYPE_COMPILER (vala_compiler_get_type ())
#define VALA_COMPILER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_COMPILER, ValaCompiler))
#define VALA_COMPILER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_COMPILER, ValaCompilerClass))
#define VALA_IS_COMPILER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_COMPILER))
#define VALA_IS_COMPILER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_COMPILER))
#define VALA_COMPILER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_COMPILER, ValaCompilerClass))

typedef struct _ValaCompiler ValaCompiler;
typedef struct _ValaCompilerClass ValaCompilerClass;
typedef struct _ValaCompilerPrivate ValaCompilerPrivate;
typedef struct _ValaParamSpecCompiler ValaParamSpecCompiler;

struct _ValaCompiler {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCompilerPrivate * priv;
};

struct _ValaCompilerClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCompiler *self);
};

struct _ValaCompilerPrivate {
	ValaCodeContext* context;
};

struct _ValaParamSpecCompiler {
	GParamSpec parent_instance;
};


static char* vala_compiler_basedir;
static char* vala_compiler_basedir = NULL;
static char* vala_compiler_directory;
static char* vala_compiler_directory = NULL;
static gboolean vala_compiler_version;
static gboolean vala_compiler_version = FALSE;
static char** vala_compiler_sources;
static char** vala_compiler_sources = NULL;
static char** vala_compiler_vapi_directories;
static char** vala_compiler_vapi_directories = NULL;
static char* vala_compiler_vapi_filename;
static char* vala_compiler_vapi_filename = NULL;
static char* vala_compiler_library;
static char* vala_compiler_library = NULL;
static char* vala_compiler_gir;
static char* vala_compiler_gir = NULL;
static char** vala_compiler_packages;
static char** vala_compiler_packages = NULL;
static char* vala_compiler_target_glib;
static char* vala_compiler_target_glib = NULL;
static gboolean vala_compiler_ccode_only;
static gboolean vala_compiler_ccode_only = FALSE;
static char* vala_compiler_header_filename;
static char* vala_compiler_header_filename = NULL;
static char* vala_compiler_internal_header_filename;
static char* vala_compiler_internal_header_filename = NULL;
static char* vala_compiler_internal_vapi_filename;
static char* vala_compiler_internal_vapi_filename = NULL;
static gboolean vala_compiler_compile_only;
static gboolean vala_compiler_compile_only = FALSE;
static char* vala_compiler_output;
static char* vala_compiler_output = NULL;
static gboolean vala_compiler_debug;
static gboolean vala_compiler_debug = FALSE;
static gboolean vala_compiler_thread;
static gboolean vala_compiler_thread = FALSE;
static gboolean vala_compiler_disable_assert;
static gboolean vala_compiler_disable_assert = FALSE;
static gboolean vala_compiler_enable_checking;
static gboolean vala_compiler_enable_checking = FALSE;
static gboolean vala_compiler_deprecated;
static gboolean vala_compiler_deprecated = FALSE;
static gboolean vala_compiler_experimental;
static gboolean vala_compiler_experimental = FALSE;
static gboolean vala_compiler_non_null_experimental;
static gboolean vala_compiler_non_null_experimental = FALSE;
static gboolean vala_compiler_disable_dbus_transformation;
static gboolean vala_compiler_disable_dbus_transformation = FALSE;
static char* vala_compiler_cc_command;
static char* vala_compiler_cc_command = NULL;
static char** vala_compiler_cc_options;
static char** vala_compiler_cc_options = NULL;
static char* vala_compiler_dump_tree;
static char* vala_compiler_dump_tree = NULL;
static gboolean vala_compiler_save_temps;
static gboolean vala_compiler_save_temps = FALSE;
static char** vala_compiler_defines;
static char** vala_compiler_defines = NULL;
static gboolean vala_compiler_quiet_mode;
static gboolean vala_compiler_quiet_mode = FALSE;
static gboolean vala_compiler_verbose_mode;
static gboolean vala_compiler_verbose_mode = FALSE;
static char* vala_compiler_profile;
static char* vala_compiler_profile = NULL;
static gpointer vala_compiler_parent_class = NULL;

gpointer vala_compiler_ref (gpointer instance);
void vala_compiler_unref (gpointer instance);
GParamSpec* vala_param_spec_compiler (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_compiler (GValue* value, gpointer v_object);
gpointer vala_value_get_compiler (const GValue* value);
GType vala_compiler_get_type (void);
#define VALA_COMPILER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_COMPILER, ValaCompilerPrivate))
enum  {
	VALA_COMPILER_DUMMY_PROPERTY
};
static gint vala_compiler_quit (ValaCompiler* self);
static gboolean vala_compiler_add_package (ValaCompiler* self, ValaCodeContext* context, const char* pkg);
static char* vala_compiler_realpath (const char* name);
static gint vala_compiler_run (ValaCompiler* self);
static gboolean vala_compiler_ends_with_dir_separator (const char* s);
ValaCompiler* vala_compiler_new (void);
ValaCompiler* vala_compiler_construct (GType object_type);
static gint vala_compiler_main (char** args, int args_length1);
ValaCompiler* vala_compiler_new (void);
static void vala_compiler_finalize (ValaCompiler* obj);
static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static gint _vala_array_length (gpointer array);
static int _vala_strcmp0 (const char * str1, const char * str2);

static const GOptionEntry VALA_COMPILER_options[] = {{"vapidir", (gchar) 0, 0, G_OPTION_ARG_FILENAME_ARRAY, &vala_compiler_vapi_directories, "Look for package bindings in DIRECTORY", "DIRECTORY..."}, {"pkg", (gchar) 0, 0, G_OPTION_ARG_STRING_ARRAY, &vala_compiler_packages, "Include binding for PACKAGE", "PACKAGE..."}, {"vapi", (gchar) 0, 0, G_OPTION_ARG_FILENAME, &vala_compiler_vapi_filename, "Output VAPI file name", "FILE"}, {"library", (gchar) 0, 0, G_OPTION_ARG_STRING, &vala_compiler_library, "Library name", "NAME"}, {"gir", (gchar) 0, 0, G_OPTION_ARG_STRING, &vala_compiler_gir, "GObject-Introspection repository file name", "NAME-VERSION.gir"}, {"basedir", 'b', 0, G_OPTION_ARG_FILENAME, &vala_compiler_basedir, "Base source directory", "DIRECTORY"}, {"directory", 'd', 0, G_OPTION_ARG_FILENAME, &vala_compiler_directory, "Output directory", "DIRECTORY"}, {"version", (gchar) 0, 0, G_OPTION_ARG_NONE, &vala_compiler_version, "Display version number", NULL}, {"ccode", 'C', 0, G_OPTION_ARG_NONE, &vala_compiler_ccode_only, "Output C code", NULL}, {"header", 'H', 0, G_OPTION_ARG_FILENAME, &vala_compiler_header_filename, "Output C header file", "FILE"}, {"internal-header", 'h', 0, G_OPTION_ARG_FILENAME, &vala_compiler_internal_header_filename, "Output internal C header file", "FILE"}, {"internal-vapi", (gchar) 0, 0, G_OPTION_ARG_FILENAME, &vala_compiler_internal_vapi_filename, "Output vapi with internal api", "FILE"}, {"compile", 'c', 0, G_OPTION_ARG_NONE, &vala_compiler_compile_only, "Compile but do not link", NULL}, {"output", 'o', 0, G_OPTION_ARG_FILENAME, &vala_compiler_output, "Place output in file FILE", "FILE"}, {"debug", 'g', 0, G_OPTION_ARG_NONE, &vala_compiler_debug, "Produce debug information", NULL}, {"thread", (gchar) 0, 0, G_OPTION_ARG_NONE, &vala_compiler_thread, "Enable multithreading support", NULL}, {"define", 'D', 0, G_OPTION_ARG_STRING_ARRAY, &vala_compiler_defines, "Define SYMBOL", "SYMBOL..."}, {"disable-assert", (gchar) 0, 0, G_OPTION_ARG_NONE, &vala_compiler_disable_assert, "Disable assertions", NULL}, {"enable-checking", (gchar) 0, 0, G_OPTION_ARG_NONE, &vala_compiler_enable_checking, "Enable additional run-time checks", NULL}, {"enable-deprecated", (gchar) 0, 0, G_OPTION_ARG_NONE, &vala_compiler_deprecated, "Enable deprecated features", NULL}, {"enable-experimental", (gchar) 0, 0, G_OPTION_ARG_NONE, &vala_compiler_experimental, "Enable experimental features", NULL}, {"enable-non-null-experimental", (gchar) 0, 0, G_OPTION_ARG_NONE, &vala_compiler_non_null_experimental, "Enable experimental enhancements for non-null types", NULL}, {"disable-dbus-transformation", (gchar) 0, 0, G_OPTION_ARG_NONE, &vala_compiler_disable_dbus_transformation, "Disable transformation of D-Bus member names", NULL}, {"cc", (gchar) 0, 0, G_OPTION_ARG_STRING, &vala_compiler_cc_command, "Use COMMAND as C compiler command", "COMMAND"}, {"Xcc", 'X', 0, G_OPTION_ARG_STRING_ARRAY, &vala_compiler_cc_options, "Pass OPTION to the C compiler", "OPTION..."}, {"dump-tree", (gchar) 0, 0, G_OPTION_ARG_FILENAME, &vala_compiler_dump_tree, "Write code tree to FILE", "FILE"}, {"save-temps", (gchar) 0, 0, G_OPTION_ARG_NONE, &vala_compiler_save_temps, "Keep temporary files", NULL}, {"profile", (gchar) 0, 0, G_OPTION_ARG_STRING, &vala_compiler_profile, "Use the given profile instead of the default", "PROFILE"}, {"quiet", 'q', 0, G_OPTION_ARG_NONE, &vala_compiler_quiet_mode, "Do not print messages to the console", NULL}, {"verbose", 'v', 0, G_OPTION_ARG_NONE, &vala_compiler_verbose_mode, "Print additional messages to the console", NULL}, {"target-glib", (gchar) 0, 0, G_OPTION_ARG_STRING, &vala_compiler_target_glib, "Target version of glib for code generation", "MAJOR.MINOR"}, {"", (gchar) 0, 0, G_OPTION_ARG_FILENAME_ARRAY, &vala_compiler_sources, NULL, "FILE..."}, {NULL}};


static gint vala_compiler_quit (ValaCompiler* self) {
	gint result;
	gboolean _tmp0_;
	g_return_val_if_fail (self != NULL, 0);
	_tmp0_ = FALSE;
	if (vala_report_get_errors (vala_code_context_get_report (self->priv->context)) == 0) {
		_tmp0_ = vala_report_get_warnings (vala_code_context_get_report (self->priv->context)) == 0;
	} else {
		_tmp0_ = FALSE;
	}
	if (_tmp0_) {
		result = 0;
		return result;
	}
	if (vala_report_get_errors (vala_code_context_get_report (self->priv->context)) == 0) {
		if (!vala_compiler_quiet_mode) {
			fprintf (stdout, "Compilation succeeded - %d warning(s)\n", vala_report_get_warnings (vala_code_context_get_report (self->priv->context)));
		}
		result = 0;
		return result;
	} else {
		if (!vala_compiler_quiet_mode) {
			fprintf (stdout, "Compilation failed: %d error(s), %d warning(s)\n", vala_report_get_errors (vala_code_context_get_report (self->priv->context)), vala_report_get_warnings (vala_code_context_get_report (self->priv->context)));
		}
		result = 1;
		return result;
	}
}


static gboolean vala_compiler_add_package (ValaCompiler* self, ValaCodeContext* context, const char* pkg) {
	gboolean result;
	GError * _inner_error_;
	char* package_path;
	ValaSourceFile* _tmp0_;
	char* _tmp2_;
	char* _tmp1_;
	char* _tmp3_;
	char* deps_filename;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (context != NULL, FALSE);
	g_return_val_if_fail (pkg != NULL, FALSE);
	_inner_error_ = NULL;
	if (vala_code_context_has_package (context, pkg)) {
		/* ignore multiple occurences of the same package*/
		result = TRUE;
		return result;
	}
	package_path = vala_code_context_get_package_path (context, pkg, vala_compiler_vapi_directories, _vala_array_length (vala_compiler_vapi_directories));
	if (package_path == NULL) {
		result = FALSE;
		package_path = (g_free (package_path), NULL);
		return result;
	}
	vala_code_context_add_package (context, pkg);
	_tmp0_ = NULL;
	vala_code_context_add_source_file (context, _tmp0_ = vala_source_file_new (context, package_path, TRUE, NULL));
	(_tmp0_ == NULL) ? NULL : (_tmp0_ = (vala_source_file_unref (_tmp0_), NULL));
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	_tmp3_ = NULL;
	deps_filename = (_tmp3_ = g_build_filename (_tmp1_ = g_path_get_dirname (package_path), _tmp2_ = g_strdup_printf ("%s.deps", pkg), NULL), _tmp2_ = (g_free (_tmp2_), NULL), _tmp1_ = (g_free (_tmp1_), NULL), _tmp3_);
	if (g_file_test (deps_filename, G_FILE_TEST_EXISTS)) {
		{
			char* deps_content;
			gulong deps_len;
			char* _tmp6_;
			gboolean _tmp5_;
			char* _tmp4_;
			deps_content = NULL;
			deps_len = 0UL;
			_tmp6_ = NULL;
			_tmp4_ = NULL;
			_tmp5_ = g_file_get_contents (deps_filename, &_tmp4_, &deps_len, &_inner_error_);
			deps_content = (_tmp6_ = _tmp4_, deps_content = (g_free (deps_content), NULL), _tmp6_);
			_tmp5_;
			if (_inner_error_ != NULL) {
				deps_content = (g_free (deps_content), NULL);
				if (_inner_error_->domain == G_FILE_ERROR) {
					goto __catch0_g_file_error;
				}
				goto __finally0;
			}
			{
				char** _tmp7_;
				char** dep_collection;
				int dep_collection_length1;
				int dep_it;
				_tmp7_ = NULL;
				dep_collection = _tmp7_ = g_strsplit (deps_content, "\n", 0);
				dep_collection_length1 = _vala_array_length (_tmp7_);
				for (dep_it = 0; dep_it < _vala_array_length (_tmp7_); dep_it = dep_it + 1) {
					const char* _tmp9_;
					char* dep;
					_tmp9_ = NULL;
					dep = (_tmp9_ = dep_collection[dep_it], (_tmp9_ == NULL) ? NULL : g_strdup (_tmp9_));
					{
						g_strstrip (dep);
						if (_vala_strcmp0 (dep, "") != 0) {
							if (!vala_compiler_add_package (self, context, dep)) {
								char* _tmp8_;
								_tmp8_ = NULL;
								vala_report_error (NULL, _tmp8_ = g_strdup_printf ("%s, dependency of %s, not found in specified Vala API directories", dep, pkg));
								_tmp8_ = (g_free (_tmp8_), NULL);
							}
						}
						dep = (g_free (dep), NULL);
					}
				}
				dep_collection = (_vala_array_free (dep_collection, dep_collection_length1, (GDestroyNotify) g_free), NULL);
			}
			deps_content = (g_free (deps_content), NULL);
		}
		goto __finally0;
		__catch0_g_file_error:
		{
			GError * e;
			e = _inner_error_;
			_inner_error_ = NULL;
			{
				char* _tmp10_;
				_tmp10_ = NULL;
				vala_report_error (NULL, _tmp10_ = g_strdup_printf ("Unable to read dependency file: %s", e->message));
				_tmp10_ = (g_free (_tmp10_), NULL);
				(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			}
		}
		__finally0:
		if (_inner_error_ != NULL) {
			package_path = (g_free (package_path), NULL);
			deps_filename = (g_free (deps_filename), NULL);
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
			g_clear_error (&_inner_error_);
			return FALSE;
		}
	}
	result = TRUE;
	package_path = (g_free (package_path), NULL);
	deps_filename = (g_free (deps_filename), NULL);
	return result;
}


static char* string_substring (const char* self, glong offset, glong len) {
	char* result;
	glong string_length;
	const char* start;
	g_return_val_if_fail (self != NULL, NULL);
	string_length = g_utf8_strlen (self, -1);
	if (offset < 0) {
		offset = string_length + offset;
		g_return_val_if_fail (offset >= 0, NULL);
	} else {
		g_return_val_if_fail (offset <= string_length, NULL);
	}
	if (len < 0) {
		len = string_length - offset;
	}
	g_return_val_if_fail ((offset + len) <= string_length, NULL);
	start = g_utf8_offset_to_pointer (self, offset);
	result = g_strndup (start, ((gchar*) g_utf8_offset_to_pointer (start, len)) - ((gchar*) start));
	return result;
}


static gboolean string_contains (const char* self, const char* needle) {
	gboolean result;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (needle != NULL, FALSE);
	result = strstr (self, needle) != NULL;
	return result;
}


static gint vala_compiler_run (ValaCompiler* self) {
	gint result;
	ValaCodeContext* _tmp0_;
	gboolean _tmp1_;
	gboolean _tmp2_;
	ValaCCodeGenerator* _tmp13_;
	char** _tmp26_;
	ValaParser* parser;
	ValaGenieParser* genie_parser;
	ValaSymbolResolver* resolver;
	ValaSemanticAnalyzer* analyzer;
	ValaFlowAnalyzer* flow_analyzer;
	gboolean _tmp27_;
	g_return_val_if_fail (self != NULL, 0);
	_tmp0_ = NULL;
	self->priv->context = (_tmp0_ = vala_code_context_new (), (self->priv->context == NULL) ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL)), _tmp0_);
	vala_code_context_push (self->priv->context);
	_tmp1_ = FALSE;
	_tmp2_ = FALSE;
	if (!vala_compiler_ccode_only) {
		_tmp2_ = !vala_compiler_compile_only;
	} else {
		_tmp2_ = FALSE;
	}
	if (_tmp2_) {
		_tmp1_ = vala_compiler_output == NULL;
	} else {
		_tmp1_ = FALSE;
	}
	/* default to build executable*/
	if (_tmp1_) {
		/* strip extension if there is one
		 else we use the default output file of the C compiler*/
		if (g_utf8_strrchr (vala_compiler_sources[0], (glong) (-1), (gunichar) '.') != NULL) {
			glong dot;
			char* _tmp4_;
			char* _tmp3_;
			dot = g_utf8_pointer_to_offset (vala_compiler_sources[0], g_utf8_strrchr (vala_compiler_sources[0], (glong) (-1), (gunichar) '.'));
			_tmp4_ = NULL;
			_tmp3_ = NULL;
			vala_compiler_output = (_tmp4_ = g_path_get_basename (_tmp3_ = string_substring (vala_compiler_sources[0], (glong) 0, dot)), vala_compiler_output = (g_free (vala_compiler_output), NULL), _tmp4_);
			_tmp3_ = (g_free (_tmp3_), NULL);
		}
	}
	vala_code_context_set_assert (self->priv->context, !vala_compiler_disable_assert);
	vala_code_context_set_checking (self->priv->context, vala_compiler_enable_checking);
	vala_code_context_set_deprecated (self->priv->context, vala_compiler_deprecated);
	vala_code_context_set_experimental (self->priv->context, vala_compiler_experimental);
	vala_code_context_set_non_null_experimental (self->priv->context, vala_compiler_non_null_experimental);
	vala_code_context_set_dbus_transformation (self->priv->context, !vala_compiler_disable_dbus_transformation);
	vala_report_set_verbose_errors (vala_code_context_get_report (self->priv->context), !vala_compiler_quiet_mode);
	vala_code_context_set_verbose_mode (self->priv->context, vala_compiler_verbose_mode);
	vala_code_context_set_ccode_only (self->priv->context, vala_compiler_ccode_only);
	vala_code_context_set_compile_only (self->priv->context, vala_compiler_compile_only);
	vala_code_context_set_header_filename (self->priv->context, vala_compiler_header_filename);
	vala_code_context_set_internal_header_filename (self->priv->context, vala_compiler_internal_header_filename);
	vala_code_context_set_output (self->priv->context, vala_compiler_output);
	if (vala_compiler_basedir == NULL) {
		char* _tmp5_;
		_tmp5_ = NULL;
		vala_code_context_set_basedir (self->priv->context, _tmp5_ = vala_compiler_realpath ("."));
		_tmp5_ = (g_free (_tmp5_), NULL);
	} else {
		char* _tmp6_;
		_tmp6_ = NULL;
		vala_code_context_set_basedir (self->priv->context, _tmp6_ = vala_compiler_realpath (vala_compiler_basedir));
		_tmp6_ = (g_free (_tmp6_), NULL);
	}
	if (vala_compiler_directory != NULL) {
		char* _tmp7_;
		_tmp7_ = NULL;
		vala_code_context_set_directory (self->priv->context, _tmp7_ = vala_compiler_realpath (vala_compiler_directory));
		_tmp7_ = (g_free (_tmp7_), NULL);
	} else {
		vala_code_context_set_directory (self->priv->context, vala_code_context_get_basedir (self->priv->context));
	}
	vala_code_context_set_debug (self->priv->context, vala_compiler_debug);
	vala_code_context_set_thread (self->priv->context, vala_compiler_thread);
	vala_code_context_set_save_temps (self->priv->context, vala_compiler_save_temps);
	if (_vala_strcmp0 (vala_compiler_profile, "posix") == 0) {
		vala_code_context_set_profile (self->priv->context, VALA_PROFILE_POSIX);
		vala_code_context_add_define (self->priv->context, "POSIX");
	} else {
		gboolean _tmp8_;
		gboolean _tmp9_;
		_tmp8_ = FALSE;
		_tmp9_ = FALSE;
		if (_vala_strcmp0 (vala_compiler_profile, "gobject-2.0") == 0) {
			_tmp9_ = TRUE;
		} else {
			_tmp9_ = _vala_strcmp0 (vala_compiler_profile, "gobject") == 0;
		}
		if (_tmp9_) {
			_tmp8_ = TRUE;
		} else {
			_tmp8_ = vala_compiler_profile == NULL;
		}
		if (_tmp8_) {
			/* default profile*/
			vala_code_context_set_profile (self->priv->context, VALA_PROFILE_GOBJECT);
			vala_code_context_add_define (self->priv->context, "GOBJECT");
		} else {
			char* _tmp10_;
			_tmp10_ = NULL;
			vala_report_error (NULL, _tmp10_ = g_strdup_printf ("Unknown profile %s", vala_compiler_profile));
			_tmp10_ = (g_free (_tmp10_), NULL);
		}
	}
	if (vala_compiler_defines != NULL) {
		{
			char** define_collection;
			int define_collection_length1;
			int define_it;
			define_collection = vala_compiler_defines;
			define_collection_length1 = _vala_array_length (vala_compiler_defines);
			for (define_it = 0; define_it < _vala_array_length (vala_compiler_defines); define_it = define_it + 1) {
				const char* _tmp11_;
				char* define;
				_tmp11_ = NULL;
				define = (_tmp11_ = define_collection[define_it], (_tmp11_ == NULL) ? NULL : g_strdup (_tmp11_));
				{
					vala_code_context_add_define (self->priv->context, define);
					define = (g_free (define), NULL);
				}
			}
		}
	}
	if (vala_code_context_get_profile (self->priv->context) == VALA_PROFILE_POSIX) {
		/* default package */
		if (!vala_compiler_add_package (self, self->priv->context, "posix")) {
			vala_report_error (NULL, "posix not found in specified Vala API directories");
		}
	} else {
		if (vala_code_context_get_profile (self->priv->context) == VALA_PROFILE_GOBJECT) {
			gint glib_major;
			gint glib_minor;
			gboolean _tmp12_;
			glib_major = 2;
			glib_minor = 12;
			_tmp12_ = FALSE;
			if (vala_compiler_target_glib != NULL) {
				_tmp12_ = sscanf (vala_compiler_target_glib, "%d.%d", &glib_major, &glib_minor, NULL) != 2;
			} else {
				_tmp12_ = FALSE;
			}
			if (_tmp12_) {
				vala_report_error (NULL, "Invalid format for --target-glib");
			}
			vala_code_context_set_target_glib_major (self->priv->context, glib_major);
			vala_code_context_set_target_glib_minor (self->priv->context, glib_minor);
			if (vala_code_context_get_target_glib_major (self->priv->context) != 2) {
				vala_report_error (NULL, "This version of valac only supports GLib 2");
			}
			/* default packages */
			if (!vala_compiler_add_package (self, self->priv->context, "glib-2.0")) {
				vala_report_error (NULL, "glib-2.0 not found in specified Vala API directories");
			}
			if (!vala_compiler_add_package (self, self->priv->context, "gobject-2.0")) {
				vala_report_error (NULL, "gobject-2.0 not found in specified Vala API directories");
			}
		}
	}
	_tmp13_ = NULL;
	vala_code_context_set_codegen (self->priv->context, (ValaCodeGenerator*) (_tmp13_ = vala_ccode_generator_new ()));
	(_tmp13_ == NULL) ? NULL : (_tmp13_ = (vala_code_visitor_unref (_tmp13_), NULL));
	if (vala_compiler_packages != NULL) {
		char** _tmp16_;
		{
			char** package_collection;
			int package_collection_length1;
			int package_it;
			package_collection = vala_compiler_packages;
			package_collection_length1 = _vala_array_length (vala_compiler_packages);
			for (package_it = 0; package_it < _vala_array_length (vala_compiler_packages); package_it = package_it + 1) {
				const char* _tmp15_;
				char* package;
				_tmp15_ = NULL;
				package = (_tmp15_ = package_collection[package_it], (_tmp15_ == NULL) ? NULL : g_strdup (_tmp15_));
				{
					if (!vala_compiler_add_package (self, self->priv->context, package)) {
						char* _tmp14_;
						_tmp14_ = NULL;
						vala_report_error (NULL, _tmp14_ = g_strdup_printf ("%s not found in specified Vala API directories", package));
						_tmp14_ = (g_free (_tmp14_), NULL);
					}
					package = (g_free (package), NULL);
				}
			}
		}
		_tmp16_ = NULL;
		vala_compiler_packages = (_tmp16_ = NULL, vala_compiler_packages = (_vala_array_free (vala_compiler_packages, _vala_array_length (vala_compiler_packages), (GDestroyNotify) g_free), NULL), _tmp16_);
	}
	if (vala_report_get_errors (vala_code_context_get_report (self->priv->context)) > 0) {
		result = vala_compiler_quit (self);
		return result;
	}
	{
		char** source_collection;
		int source_collection_length1;
		int source_it;
		source_collection = vala_compiler_sources;
		source_collection_length1 = _vala_array_length (vala_compiler_sources);
		for (source_it = 0; source_it < _vala_array_length (vala_compiler_sources); source_it = source_it + 1) {
			const char* _tmp25_;
			char* source;
			_tmp25_ = NULL;
			source = (_tmp25_ = source_collection[source_it], (_tmp25_ == NULL) ? NULL : g_strdup (_tmp25_));
			{
				if (g_file_test (source, G_FILE_TEST_EXISTS)) {
					char* rpath;
					gboolean _tmp17_;
					rpath = vala_compiler_realpath (source);
					_tmp17_ = FALSE;
					if (g_str_has_suffix (source, ".vala")) {
						_tmp17_ = TRUE;
					} else {
						_tmp17_ = g_str_has_suffix (source, ".gs");
					}
					if (_tmp17_) {
						ValaSourceFile* source_file;
						source_file = vala_source_file_new (self->priv->context, rpath, FALSE, NULL);
						if (vala_code_context_get_profile (self->priv->context) == VALA_PROFILE_POSIX) {
							ValaUsingDirective* _tmp19_;
							ValaUnresolvedSymbol* _tmp18_;
							/* import the Posix namespace by default (namespace of backend-specific standard library)*/
							_tmp19_ = NULL;
							_tmp18_ = NULL;
							vala_source_file_add_using_directive (source_file, _tmp19_ = vala_using_directive_new ((ValaSymbol*) (_tmp18_ = vala_unresolved_symbol_new (NULL, "Posix", NULL)), NULL));
							(_tmp19_ == NULL) ? NULL : (_tmp19_ = (vala_code_node_unref (_tmp19_), NULL));
							(_tmp18_ == NULL) ? NULL : (_tmp18_ = (vala_code_node_unref (_tmp18_), NULL));
						} else {
							if (vala_code_context_get_profile (self->priv->context) == VALA_PROFILE_GOBJECT) {
								ValaUsingDirective* _tmp21_;
								ValaUnresolvedSymbol* _tmp20_;
								/* import the GLib namespace by default (namespace of backend-specific standard library)*/
								_tmp21_ = NULL;
								_tmp20_ = NULL;
								vala_source_file_add_using_directive (source_file, _tmp21_ = vala_using_directive_new ((ValaSymbol*) (_tmp20_ = vala_unresolved_symbol_new (NULL, "GLib", NULL)), NULL));
								(_tmp21_ == NULL) ? NULL : (_tmp21_ = (vala_code_node_unref (_tmp21_), NULL));
								(_tmp20_ == NULL) ? NULL : (_tmp20_ = (vala_code_node_unref (_tmp20_), NULL));
							}
						}
						vala_code_context_add_source_file (self->priv->context, source_file);
						(source_file == NULL) ? NULL : (source_file = (vala_source_file_unref (source_file), NULL));
					} else {
						if (g_str_has_suffix (source, ".vapi")) {
							ValaSourceFile* _tmp22_;
							_tmp22_ = NULL;
							vala_code_context_add_source_file (self->priv->context, _tmp22_ = vala_source_file_new (self->priv->context, rpath, TRUE, NULL));
							(_tmp22_ == NULL) ? NULL : (_tmp22_ = (vala_source_file_unref (_tmp22_), NULL));
						} else {
							if (g_str_has_suffix (source, ".c")) {
								vala_code_context_add_c_source_file (self->priv->context, rpath);
							} else {
								char* _tmp23_;
								_tmp23_ = NULL;
								vala_report_error (NULL, _tmp23_ = g_strdup_printf ("%s is not a supported source file type. Only .vala, .vapi, .gs, and .c files are supported.", source));
								_tmp23_ = (g_free (_tmp23_), NULL);
							}
						}
					}
					rpath = (g_free (rpath), NULL);
				} else {
					char* _tmp24_;
					_tmp24_ = NULL;
					vala_report_error (NULL, _tmp24_ = g_strdup_printf ("%s not found", source));
					_tmp24_ = (g_free (_tmp24_), NULL);
				}
				source = (g_free (source), NULL);
			}
		}
	}
	_tmp26_ = NULL;
	vala_compiler_sources = (_tmp26_ = NULL, vala_compiler_sources = (_vala_array_free (vala_compiler_sources, _vala_array_length (vala_compiler_sources), (GDestroyNotify) g_free), NULL), _tmp26_);
	if (vala_report_get_errors (vala_code_context_get_report (self->priv->context)) > 0) {
		result = vala_compiler_quit (self);
		return result;
	}
	parser = vala_parser_new ();
	vala_parser_parse (parser, self->priv->context);
	genie_parser = vala_genie_parser_new ();
	vala_genie_parser_parse (genie_parser, self->priv->context);
	if (vala_report_get_errors (vala_code_context_get_report (self->priv->context)) > 0) {
		result = vala_compiler_quit (self);
		(parser == NULL) ? NULL : (parser = (vala_code_visitor_unref (parser), NULL));
		(genie_parser == NULL) ? NULL : (genie_parser = (vala_code_visitor_unref (genie_parser), NULL));
		return result;
	}
	resolver = vala_symbol_resolver_new ();
	vala_symbol_resolver_resolve (resolver, self->priv->context);
	if (vala_report_get_errors (vala_code_context_get_report (self->priv->context)) > 0) {
		result = vala_compiler_quit (self);
		(parser == NULL) ? NULL : (parser = (vala_code_visitor_unref (parser), NULL));
		(genie_parser == NULL) ? NULL : (genie_parser = (vala_code_visitor_unref (genie_parser), NULL));
		(resolver == NULL) ? NULL : (resolver = (vala_code_visitor_unref (resolver), NULL));
		return result;
	}
	analyzer = vala_semantic_analyzer_new ();
	vala_semantic_analyzer_analyze (analyzer, self->priv->context);
	if (vala_compiler_dump_tree != NULL) {
		ValaCodeWriter* code_writer;
		code_writer = vala_code_writer_new (TRUE, FALSE);
		vala_code_writer_write_file (code_writer, self->priv->context, vala_compiler_dump_tree);
		(code_writer == NULL) ? NULL : (code_writer = (vala_code_visitor_unref (code_writer), NULL));
	}
	if (vala_report_get_errors (vala_code_context_get_report (self->priv->context)) > 0) {
		result = vala_compiler_quit (self);
		(parser == NULL) ? NULL : (parser = (vala_code_visitor_unref (parser), NULL));
		(genie_parser == NULL) ? NULL : (genie_parser = (vala_code_visitor_unref (genie_parser), NULL));
		(resolver == NULL) ? NULL : (resolver = (vala_code_visitor_unref (resolver), NULL));
		(analyzer == NULL) ? NULL : (analyzer = (vala_code_visitor_unref (analyzer), NULL));
		return result;
	}
	flow_analyzer = vala_flow_analyzer_new ();
	vala_flow_analyzer_analyze (flow_analyzer, self->priv->context);
	if (vala_report_get_errors (vala_code_context_get_report (self->priv->context)) > 0) {
		result = vala_compiler_quit (self);
		(parser == NULL) ? NULL : (parser = (vala_code_visitor_unref (parser), NULL));
		(genie_parser == NULL) ? NULL : (genie_parser = (vala_code_visitor_unref (genie_parser), NULL));
		(resolver == NULL) ? NULL : (resolver = (vala_code_visitor_unref (resolver), NULL));
		(analyzer == NULL) ? NULL : (analyzer = (vala_code_visitor_unref (analyzer), NULL));
		(flow_analyzer == NULL) ? NULL : (flow_analyzer = (vala_code_visitor_unref (flow_analyzer), NULL));
		return result;
	}
	if (vala_code_context_get_non_null_experimental (self->priv->context)) {
		ValaNullChecker* null_checker;
		null_checker = vala_null_checker_new ();
		vala_null_checker_check (null_checker, self->priv->context);
		if (vala_report_get_errors (vala_code_context_get_report (self->priv->context)) > 0) {
			result = vala_compiler_quit (self);
			(null_checker == NULL) ? NULL : (null_checker = (vala_code_visitor_unref (null_checker), NULL));
			(parser == NULL) ? NULL : (parser = (vala_code_visitor_unref (parser), NULL));
			(genie_parser == NULL) ? NULL : (genie_parser = (vala_code_visitor_unref (genie_parser), NULL));
			(resolver == NULL) ? NULL : (resolver = (vala_code_visitor_unref (resolver), NULL));
			(analyzer == NULL) ? NULL : (analyzer = (vala_code_visitor_unref (analyzer), NULL));
			(flow_analyzer == NULL) ? NULL : (flow_analyzer = (vala_code_visitor_unref (flow_analyzer), NULL));
			return result;
		}
		(null_checker == NULL) ? NULL : (null_checker = (vala_code_visitor_unref (null_checker), NULL));
	}
	vala_code_generator_emit (vala_code_context_get_codegen (self->priv->context), self->priv->context);
	if (vala_report_get_errors (vala_code_context_get_report (self->priv->context)) > 0) {
		result = vala_compiler_quit (self);
		(parser == NULL) ? NULL : (parser = (vala_code_visitor_unref (parser), NULL));
		(genie_parser == NULL) ? NULL : (genie_parser = (vala_code_visitor_unref (genie_parser), NULL));
		(resolver == NULL) ? NULL : (resolver = (vala_code_visitor_unref (resolver), NULL));
		(analyzer == NULL) ? NULL : (analyzer = (vala_code_visitor_unref (analyzer), NULL));
		(flow_analyzer == NULL) ? NULL : (flow_analyzer = (vala_code_visitor_unref (flow_analyzer), NULL));
		return result;
	}
	_tmp27_ = FALSE;
	if (vala_compiler_vapi_filename == NULL) {
		_tmp27_ = vala_compiler_library != NULL;
	} else {
		_tmp27_ = FALSE;
	}
	if (_tmp27_) {
		char* _tmp28_;
		/* keep backward compatibility with --library option*/
		_tmp28_ = NULL;
		vala_compiler_vapi_filename = (_tmp28_ = g_strdup_printf ("%s.vapi", vala_compiler_library), vala_compiler_vapi_filename = (g_free (vala_compiler_vapi_filename), NULL), _tmp28_);
	}
	if (vala_compiler_vapi_filename != NULL) {
		ValaCodeWriter* interface_writer;
		gboolean _tmp29_;
		interface_writer = vala_code_writer_new (FALSE, FALSE);
		_tmp29_ = FALSE;
		if (vala_compiler_directory != NULL) {
			_tmp29_ = !g_path_is_absolute (vala_compiler_vapi_filename);
		} else {
			_tmp29_ = FALSE;
		}
		/* put .vapi file in current directory unless -d has been explicitly specified*/
		if (_tmp29_) {
			char* _tmp30_;
			_tmp30_ = NULL;
			vala_compiler_vapi_filename = (_tmp30_ = g_strdup_printf ("%s%c%s", vala_code_context_get_directory (self->priv->context), (gint) G_DIR_SEPARATOR, vala_compiler_vapi_filename), vala_compiler_vapi_filename = (g_free (vala_compiler_vapi_filename), NULL), _tmp30_);
		}
		vala_code_writer_write_file (interface_writer, self->priv->context, vala_compiler_vapi_filename);
		(interface_writer == NULL) ? NULL : (interface_writer = (vala_code_visitor_unref (interface_writer), NULL));
	}
	if (vala_compiler_library != NULL) {
		char* _tmp40_;
		if (vala_compiler_gir != NULL) {
			char* _tmp39_;
			if (vala_code_context_get_profile (self->priv->context) == VALA_PROFILE_GOBJECT) {
				glong gir_len;
				const char* last_hyphen;
				gboolean _tmp31_;
				gir_len = g_utf8_strlen (vala_compiler_gir, -1);
				last_hyphen = g_utf8_strrchr (vala_compiler_gir, gir_len, (gunichar) '-');
				_tmp31_ = FALSE;
				if (last_hyphen == NULL) {
					_tmp31_ = TRUE;
				} else {
					_tmp31_ = !g_str_has_suffix (vala_compiler_gir, ".gir");
				}
				if (_tmp31_) {
					char* _tmp32_;
					_tmp32_ = NULL;
					vala_report_error (NULL, _tmp32_ = g_strdup_printf ("GIR file name `%s' is not well-formed, expected NAME-VERSION.gir", vala_compiler_gir));
					_tmp32_ = (g_free (_tmp32_), NULL);
				} else {
					glong offset;
					char* gir_namespace;
					char* gir_version;
					gboolean _tmp33_;
					gboolean _tmp34_;
					gboolean _tmp35_;
					offset = g_utf8_pointer_to_offset (vala_compiler_gir, last_hyphen);
					gir_namespace = string_substring (vala_compiler_gir, (glong) 0, offset);
					gir_version = string_substring (vala_compiler_gir, offset + 1, (gir_len - offset) - 5);
					g_strcanon (gir_version, "0123456789.", '?');
					_tmp33_ = FALSE;
					_tmp34_ = FALSE;
					_tmp35_ = FALSE;
					if (_vala_strcmp0 (gir_namespace, "") == 0) {
						_tmp35_ = TRUE;
					} else {
						_tmp35_ = _vala_strcmp0 (gir_version, "") == 0;
					}
					if (_tmp35_) {
						_tmp34_ = TRUE;
					} else {
						_tmp34_ = !g_unichar_isdigit (g_utf8_get_char (g_utf8_offset_to_pointer (gir_version, 0)));
					}
					if (_tmp34_) {
						_tmp33_ = TRUE;
					} else {
						_tmp33_ = string_contains (gir_version, "?");
					}
					if (_tmp33_) {
						char* _tmp36_;
						_tmp36_ = NULL;
						vala_report_error (NULL, _tmp36_ = g_strdup_printf ("GIR file name `%s' is not well-formed, expected NAME-VERSION.gir", vala_compiler_gir));
						_tmp36_ = (g_free (_tmp36_), NULL);
					} else {
						ValaGIRWriter* gir_writer;
						char* gir_directory;
						gir_writer = vala_gir_writer_new ();
						/* put .gir file in current directory unless -d has been explicitly specified*/
						gir_directory = g_strdup (".");
						if (vala_compiler_directory != NULL) {
							char* _tmp38_;
							const char* _tmp37_;
							_tmp38_ = NULL;
							_tmp37_ = NULL;
							gir_directory = (_tmp38_ = (_tmp37_ = vala_code_context_get_directory (self->priv->context), (_tmp37_ == NULL) ? NULL : g_strdup (_tmp37_)), gir_directory = (g_free (gir_directory), NULL), _tmp38_);
						}
						vala_gir_writer_write_file (gir_writer, self->priv->context, gir_directory, gir_namespace, gir_version, vala_compiler_library);
						(gir_writer == NULL) ? NULL : (gir_writer = (vala_code_visitor_unref (gir_writer), NULL));
						gir_directory = (g_free (gir_directory), NULL);
					}
					gir_namespace = (g_free (gir_namespace), NULL);
					gir_version = (g_free (gir_version), NULL);
				}
			}
			_tmp39_ = NULL;
			vala_compiler_gir = (_tmp39_ = NULL, vala_compiler_gir = (g_free (vala_compiler_gir), NULL), _tmp39_);
		}
		_tmp40_ = NULL;
		vala_compiler_library = (_tmp40_ = NULL, vala_compiler_library = (g_free (vala_compiler_library), NULL), _tmp40_);
	}
	if (vala_compiler_internal_vapi_filename != NULL) {
		gboolean _tmp41_;
		ValaCodeWriter* interface_writer;
		const char* _tmp42_;
		char* vapi_filename;
		gboolean _tmp43_;
		char* _tmp45_;
		_tmp41_ = FALSE;
		if (vala_compiler_internal_header_filename == NULL) {
			_tmp41_ = TRUE;
		} else {
			_tmp41_ = vala_compiler_header_filename == NULL;
		}
		if (_tmp41_) {
			vala_report_error (NULL, "--internal-vapi may only be used in combination with --header and --internal-header");
			result = vala_compiler_quit (self);
			(parser == NULL) ? NULL : (parser = (vala_code_visitor_unref (parser), NULL));
			(genie_parser == NULL) ? NULL : (genie_parser = (vala_code_visitor_unref (genie_parser), NULL));
			(resolver == NULL) ? NULL : (resolver = (vala_code_visitor_unref (resolver), NULL));
			(analyzer == NULL) ? NULL : (analyzer = (vala_code_visitor_unref (analyzer), NULL));
			(flow_analyzer == NULL) ? NULL : (flow_analyzer = (vala_code_visitor_unref (flow_analyzer), NULL));
			return result;
		}
		interface_writer = vala_code_writer_new (FALSE, TRUE);
		vala_code_writer_set_cheader_override (interface_writer, vala_compiler_header_filename, vala_compiler_internal_header_filename);
		_tmp42_ = NULL;
		vapi_filename = (_tmp42_ = vala_compiler_internal_vapi_filename, (_tmp42_ == NULL) ? NULL : g_strdup (_tmp42_));
		_tmp43_ = FALSE;
		if (vala_compiler_directory != NULL) {
			_tmp43_ = !g_path_is_absolute (vapi_filename);
		} else {
			_tmp43_ = FALSE;
		}
		/* put .vapi file in current directory unless -d has been explicitly specified*/
		if (_tmp43_) {
			char* _tmp44_;
			_tmp44_ = NULL;
			vapi_filename = (_tmp44_ = g_strdup_printf ("%s%c%s", vala_code_context_get_directory (self->priv->context), (gint) G_DIR_SEPARATOR, vapi_filename), vapi_filename = (g_free (vapi_filename), NULL), _tmp44_);
		}
		vala_code_writer_write_file (interface_writer, self->priv->context, vapi_filename);
		_tmp45_ = NULL;
		vala_compiler_internal_vapi_filename = (_tmp45_ = NULL, vala_compiler_internal_vapi_filename = (g_free (vala_compiler_internal_vapi_filename), NULL), _tmp45_);
		(interface_writer == NULL) ? NULL : (interface_writer = (vala_code_visitor_unref (interface_writer), NULL));
		vapi_filename = (g_free (vapi_filename), NULL);
	}
	if (!vala_compiler_ccode_only) {
		ValaCCodeCompiler* ccompiler;
		gboolean _tmp46_;
		ccompiler = vala_ccode_compiler_new ();
		_tmp46_ = FALSE;
		if (vala_compiler_cc_command == NULL) {
			_tmp46_ = g_getenv ("CC") != NULL;
		} else {
			_tmp46_ = FALSE;
		}
		if (_tmp46_) {
			char* _tmp48_;
			const char* _tmp47_;
			_tmp48_ = NULL;
			_tmp47_ = NULL;
			vala_compiler_cc_command = (_tmp48_ = (_tmp47_ = g_getenv ("CC"), (_tmp47_ == NULL) ? NULL : g_strdup (_tmp47_)), vala_compiler_cc_command = (g_free (vala_compiler_cc_command), NULL), _tmp48_);
		}
		if (vala_compiler_cc_options == NULL) {
			gint _tmp50__length1;
			char** _tmp50_;
			char** _tmp49_;
			_tmp50_ = NULL;
			_tmp49_ = NULL;
			vala_ccode_compiler_compile (ccompiler, self->priv->context, vala_compiler_cc_command, (_tmp50_ = (_tmp49_ = g_new0 (char*, 0 + 1), _tmp49_), _tmp50__length1 = 0, _tmp50_), 0);
			_tmp50_ = (_vala_array_free (_tmp50_, _tmp50__length1, (GDestroyNotify) g_free), NULL);
		} else {
			vala_ccode_compiler_compile (ccompiler, self->priv->context, vala_compiler_cc_command, vala_compiler_cc_options, _vala_array_length (vala_compiler_cc_options));
		}
		(ccompiler == NULL) ? NULL : (ccompiler = (vala_ccode_compiler_unref (ccompiler), NULL));
	}
	result = vala_compiler_quit (self);
	(parser == NULL) ? NULL : (parser = (vala_code_visitor_unref (parser), NULL));
	(genie_parser == NULL) ? NULL : (genie_parser = (vala_code_visitor_unref (genie_parser), NULL));
	(resolver == NULL) ? NULL : (resolver = (vala_code_visitor_unref (resolver), NULL));
	(analyzer == NULL) ? NULL : (analyzer = (vala_code_visitor_unref (analyzer), NULL));
	(flow_analyzer == NULL) ? NULL : (flow_analyzer = (vala_code_visitor_unref (flow_analyzer), NULL));
	return result;
}


static gboolean vala_compiler_ends_with_dir_separator (const char* s) {
	gboolean result;
	g_return_val_if_fail (s != NULL, FALSE);
	result = G_IS_DIR_SEPARATOR (g_utf8_get_char (g_utf8_offset_to_pointer (s, g_utf8_strlen (s, -1) - 1)));
	return result;
}


/* ported from glibc */
static char* vala_compiler_realpath (const char* name) {
	char* result;
	char* rpath;
	const char* start;
	const char* end;
	glong root_len;
	gboolean _tmp12_;
	g_return_val_if_fail (name != NULL, NULL);
	rpath = NULL;
	/* start of path component*/
	start = NULL;
	/* end of path component*/
	end = NULL;
	if (!g_path_is_absolute (name)) {
		char* _tmp0_;
		/* relative path*/
		_tmp0_ = NULL;
		rpath = (_tmp0_ = g_get_current_dir (), rpath = (g_free (rpath), NULL), _tmp0_);
		start = end = name;
	} else {
		char* _tmp1_;
		/* set start after root*/
		start = end = g_path_skip_root (name);
		/* extract root*/
		_tmp1_ = NULL;
		rpath = (_tmp1_ = string_substring (name, (glong) 0, g_utf8_pointer_to_offset (name, start)), rpath = (g_free (rpath), NULL), _tmp1_);
	}
	root_len = g_utf8_pointer_to_offset (rpath, g_path_skip_root (rpath));
	{
		gboolean _tmp2_;
		_tmp2_ = TRUE;
		while (TRUE) {
			glong len;
			if (!_tmp2_) {
				start = end;
			}
			_tmp2_ = FALSE;
			if (!(g_utf8_get_char (start) != 0)) {
				break;
			}
			while (TRUE) {
				if (!G_IS_DIR_SEPARATOR (g_utf8_get_char (start))) {
					break;
				}
				start = g_utf8_next_char (start);
			}
			/* find end of path component*/
			len = (glong) 0;
			{
				gboolean _tmp3_;
				end = start;
				_tmp3_ = TRUE;
				while (TRUE) {
					gboolean _tmp4_;
					if (!_tmp3_) {
						end = g_utf8_next_char (end);
					}
					_tmp3_ = FALSE;
					_tmp4_ = FALSE;
					if (g_utf8_get_char (end) != 0) {
						_tmp4_ = !G_IS_DIR_SEPARATOR (g_utf8_get_char (end));
					} else {
						_tmp4_ = FALSE;
					}
					if (!_tmp4_) {
						break;
					}
					len++;
				}
			}
			if (len == 0) {
				break;
			} else {
				gboolean _tmp5_;
				_tmp5_ = FALSE;
				if (len == 1) {
					_tmp5_ = g_utf8_get_char (start) == '.';
				} else {
					_tmp5_ = FALSE;
				}
				if (_tmp5_) {
				} else {
					gboolean _tmp6_;
					_tmp6_ = FALSE;
					if (len == 2) {
						_tmp6_ = g_str_has_prefix (start, "..");
					} else {
						_tmp6_ = FALSE;
					}
					if (_tmp6_) {
						/* back up to previous component, ignore if at root already*/
						if (g_utf8_strlen (rpath, -1) > root_len) {
							{
								gboolean _tmp7_;
								_tmp7_ = TRUE;
								while (TRUE) {
									char* _tmp8_;
									if (!_tmp7_) {
										if (!(!vala_compiler_ends_with_dir_separator (rpath))) {
											break;
										}
									}
									_tmp7_ = FALSE;
									_tmp8_ = NULL;
									rpath = (_tmp8_ = string_substring (rpath, (glong) 0, g_utf8_strlen (rpath, -1) - 1), rpath = (g_free (rpath), NULL), _tmp8_);
								}
							}
						}
					} else {
						char* _tmp11_;
						char* _tmp10_;
						if (!vala_compiler_ends_with_dir_separator (rpath)) {
							char* _tmp9_;
							_tmp9_ = NULL;
							rpath = (_tmp9_ = g_strconcat (rpath, G_DIR_SEPARATOR_S, NULL), rpath = (g_free (rpath), NULL), _tmp9_);
						}
						_tmp11_ = NULL;
						_tmp10_ = NULL;
						rpath = (_tmp11_ = g_strconcat (rpath, _tmp10_ = string_substring (start, (glong) 0, len), NULL), rpath = (g_free (rpath), NULL), _tmp11_);
						_tmp10_ = (g_free (_tmp10_), NULL);
					}
				}
			}
		}
	}
	_tmp12_ = FALSE;
	if (g_utf8_strlen (rpath, -1) > root_len) {
		_tmp12_ = vala_compiler_ends_with_dir_separator (rpath);
	} else {
		_tmp12_ = FALSE;
	}
	if (_tmp12_) {
		char* _tmp13_;
		_tmp13_ = NULL;
		rpath = (_tmp13_ = string_substring (rpath, (glong) 0, g_utf8_strlen (rpath, -1) - 1), rpath = (g_free (rpath), NULL), _tmp13_);
	}
	if (G_DIR_SEPARATOR != '/') {
		char** _tmp15_;
		gint components_size;
		gint components_length1;
		char** _tmp14_;
		char** components;
		char* _tmp16_;
		/* don't use backslashes internally,
		 to avoid problems in #include directives*/
		_tmp15_ = NULL;
		_tmp14_ = NULL;
		components = (_tmp15_ = _tmp14_ = g_strsplit (rpath, "\\", 0), components_length1 = _vala_array_length (_tmp14_), components_size = components_length1, _tmp15_);
		_tmp16_ = NULL;
		rpath = (_tmp16_ = g_strjoinv ("/", components), rpath = (g_free (rpath), NULL), _tmp16_);
		components = (_vala_array_free (components, components_length1, (GDestroyNotify) g_free), NULL);
	}
	result = rpath;
	return result;
}


static gint vala_compiler_main (char** args, int args_length1) {
	gint result;
	GError * _inner_error_;
	ValaCompiler* compiler;
	_inner_error_ = NULL;
	{
		GOptionContext* opt_context;
		opt_context = g_option_context_new ("- Vala Compiler");
		g_option_context_set_help_enabled (opt_context, TRUE);
		g_option_context_add_main_entries (opt_context, VALA_COMPILER_options, NULL);
		g_option_context_parse (opt_context, &args_length1, &args, &_inner_error_);
		if (_inner_error_ != NULL) {
			(opt_context == NULL) ? NULL : (opt_context = (g_option_context_free (opt_context), NULL));
			if (_inner_error_->domain == G_OPTION_ERROR) {
				goto __catch1_g_option_error;
			}
			goto __finally1;
		}
		(opt_context == NULL) ? NULL : (opt_context = (g_option_context_free (opt_context), NULL));
	}
	goto __finally1;
	__catch1_g_option_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			fprintf (stdout, "%s\n", e->message);
			fprintf (stdout, "Run '%s --help' to see a full list of available command line options.\n", args[0]);
			result = 1;
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			return result;
		}
	}
	__finally1:
	if (_inner_error_ != NULL) {
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return 0;
	}
	if (vala_compiler_version) {
		fprintf (stdout, "Vala %s\n", PACKAGE_VERSION);
		result = 0;
		return result;
	}
	if (vala_compiler_sources == NULL) {
		fprintf (stderr, "No source file specified.\n");
		result = 1;
		return result;
	}
	compiler = vala_compiler_new ();
	result = vala_compiler_run (compiler);
	(compiler == NULL) ? NULL : (compiler = (vala_compiler_unref (compiler), NULL));
	return result;
}


int main (int argc, char ** argv) {
	g_type_init ();
	return vala_compiler_main (argv, argc);
}


ValaCompiler* vala_compiler_construct (GType object_type) {
	ValaCompiler* self;
	self = (ValaCompiler*) g_type_create_instance (object_type);
	return self;
}


ValaCompiler* vala_compiler_new (void) {
	return vala_compiler_construct (VALA_TYPE_COMPILER);
}


static void vala_value_compiler_init (GValue* value) {
	value->data[0].v_pointer = NULL;
}


static void vala_value_compiler_free_value (GValue* value) {
	if (value->data[0].v_pointer) {
		vala_compiler_unref (value->data[0].v_pointer);
	}
}


static void vala_value_compiler_copy_value (const GValue* src_value, GValue* dest_value) {
	if (src_value->data[0].v_pointer) {
		dest_value->data[0].v_pointer = vala_compiler_ref (src_value->data[0].v_pointer);
	} else {
		dest_value->data[0].v_pointer = NULL;
	}
}


static gpointer vala_value_compiler_peek_pointer (const GValue* value) {
	return value->data[0].v_pointer;
}


static gchar* vala_value_compiler_collect_value (GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	if (collect_values[0].v_pointer) {
		ValaCompiler* object;
		object = collect_values[0].v_pointer;
		if (object->parent_instance.g_class == NULL) {
			return g_strconcat ("invalid unclassed object pointer for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		} else if (!g_value_type_compatible (G_TYPE_FROM_INSTANCE (object), G_VALUE_TYPE (value))) {
			return g_strconcat ("invalid object type `", g_type_name (G_TYPE_FROM_INSTANCE (object)), "' for value type `", G_VALUE_TYPE_NAME (value), "'", NULL);
		}
		value->data[0].v_pointer = vala_compiler_ref (object);
	} else {
		value->data[0].v_pointer = NULL;
	}
	return NULL;
}


static gchar* vala_value_compiler_lcopy_value (const GValue* value, guint n_collect_values, GTypeCValue* collect_values, guint collect_flags) {
	ValaCompiler** object_p;
	object_p = collect_values[0].v_pointer;
	if (!object_p) {
		return g_strdup_printf ("value location for `%s' passed as NULL", G_VALUE_TYPE_NAME (value));
	}
	if (!value->data[0].v_pointer) {
		*object_p = NULL;
	} else if (collect_flags && G_VALUE_NOCOPY_CONTENTS) {
		*object_p = value->data[0].v_pointer;
	} else {
		*object_p = vala_compiler_ref (value->data[0].v_pointer);
	}
	return NULL;
}


GParamSpec* vala_param_spec_compiler (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags) {
	ValaParamSpecCompiler* spec;
	g_return_val_if_fail (g_type_is_a (object_type, VALA_TYPE_COMPILER), NULL);
	spec = g_param_spec_internal (G_TYPE_PARAM_OBJECT, name, nick, blurb, flags);
	G_PARAM_SPEC (spec)->value_type = object_type;
	return G_PARAM_SPEC (spec);
}


gpointer vala_value_get_compiler (const GValue* value) {
	g_return_val_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_TYPE_COMPILER), NULL);
	return value->data[0].v_pointer;
}


void vala_value_set_compiler (GValue* value, gpointer v_object) {
	ValaCompiler* old;
	g_return_if_fail (G_TYPE_CHECK_VALUE_TYPE (value, VALA_TYPE_COMPILER));
	old = value->data[0].v_pointer;
	if (v_object) {
		g_return_if_fail (G_TYPE_CHECK_INSTANCE_TYPE (v_object, VALA_TYPE_COMPILER));
		g_return_if_fail (g_value_type_compatible (G_TYPE_FROM_INSTANCE (v_object), G_VALUE_TYPE (value)));
		value->data[0].v_pointer = v_object;
		vala_compiler_ref (value->data[0].v_pointer);
	} else {
		value->data[0].v_pointer = NULL;
	}
	if (old) {
		vala_compiler_unref (old);
	}
}


static void vala_compiler_class_init (ValaCompilerClass * klass) {
	vala_compiler_parent_class = g_type_class_peek_parent (klass);
	VALA_COMPILER_CLASS (klass)->finalize = vala_compiler_finalize;
	g_type_class_add_private (klass, sizeof (ValaCompilerPrivate));
}


static void vala_compiler_instance_init (ValaCompiler * self) {
	self->priv = VALA_COMPILER_GET_PRIVATE (self);
	self->ref_count = 1;
}


static void vala_compiler_finalize (ValaCompiler* obj) {
	ValaCompiler * self;
	self = VALA_COMPILER (obj);
	(self->priv->context == NULL) ? NULL : (self->priv->context = (vala_code_context_unref (self->priv->context), NULL));
}


GType vala_compiler_get_type (void) {
	static GType vala_compiler_type_id = 0;
	if (vala_compiler_type_id == 0) {
		static const GTypeValueTable g_define_type_value_table = { vala_value_compiler_init, vala_value_compiler_free_value, vala_value_compiler_copy_value, vala_value_compiler_peek_pointer, "p", vala_value_compiler_collect_value, "p", vala_value_compiler_lcopy_value };
		static const GTypeInfo g_define_type_info = { sizeof (ValaCompilerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_compiler_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCompiler), 0, (GInstanceInitFunc) vala_compiler_instance_init, &g_define_type_value_table };
		static const GTypeFundamentalInfo g_define_type_fundamental_info = { (G_TYPE_FLAG_CLASSED | G_TYPE_FLAG_INSTANTIATABLE | G_TYPE_FLAG_DERIVABLE | G_TYPE_FLAG_DEEP_DERIVABLE) };
		vala_compiler_type_id = g_type_register_fundamental (g_type_fundamental_next (), "ValaCompiler", &g_define_type_info, &g_define_type_fundamental_info, 0);
	}
	return vala_compiler_type_id;
}


gpointer vala_compiler_ref (gpointer instance) {
	ValaCompiler* self;
	self = instance;
	g_atomic_int_inc (&self->ref_count);
	return instance;
}


void vala_compiler_unref (gpointer instance) {
	ValaCompiler* self;
	self = instance;
	if (g_atomic_int_dec_and_test (&self->ref_count)) {
		VALA_COMPILER_GET_CLASS (self)->finalize (self);
		g_type_free_instance ((GTypeInstance *) self);
	}
}


static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	_vala_array_destroy (array, array_length, destroy_func);
	g_free (array);
}


static gint _vala_array_length (gpointer array) {
	int length;
	length = 0;
	if (array) {
		while (((gpointer*) array)[length]) {
			length++;
		}
	}
	return length;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




