/* valatrystatement.vala
 *
 * Copyright (C) 2007-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>


#define VALA_TYPE_CODE_NODE (vala_code_node_get_type ())
#define VALA_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_NODE, ValaCodeNode))
#define VALA_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))
#define VALA_IS_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_NODE))
#define VALA_IS_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_NODE))
#define VALA_CODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))

typedef struct _ValaCodeNode ValaCodeNode;
typedef struct _ValaCodeNodeClass ValaCodeNodeClass;
typedef struct _ValaCodeNodePrivate ValaCodeNodePrivate;

#define VALA_TYPE_CODE_VISITOR (vala_code_visitor_get_type ())
#define VALA_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitor))
#define VALA_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))
#define VALA_IS_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_VISITOR))
#define VALA_IS_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_VISITOR))
#define VALA_CODE_VISITOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))

typedef struct _ValaCodeVisitor ValaCodeVisitor;
typedef struct _ValaCodeVisitorClass ValaCodeVisitorClass;

#define VALA_TYPE_SEMANTIC_ANALYZER (vala_semantic_analyzer_get_type ())
#define VALA_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzer))
#define VALA_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))
#define VALA_IS_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_IS_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_SEMANTIC_ANALYZER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))

typedef struct _ValaSemanticAnalyzer ValaSemanticAnalyzer;
typedef struct _ValaSemanticAnalyzerClass ValaSemanticAnalyzerClass;

#define VALA_TYPE_DATA_TYPE (vala_data_type_get_type ())
#define VALA_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DATA_TYPE, ValaDataType))
#define VALA_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))
#define VALA_IS_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DATA_TYPE))
#define VALA_IS_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DATA_TYPE))
#define VALA_DATA_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))

typedef struct _ValaDataType ValaDataType;
typedef struct _ValaDataTypeClass ValaDataTypeClass;

#define VALA_TYPE_EXPRESSION (vala_expression_get_type ())
#define VALA_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_EXPRESSION, ValaExpression))
#define VALA_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_EXPRESSION, ValaExpressionClass))
#define VALA_IS_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_EXPRESSION))
#define VALA_IS_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_EXPRESSION))
#define VALA_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_EXPRESSION, ValaExpressionClass))

typedef struct _ValaExpression ValaExpression;
typedef struct _ValaExpressionClass ValaExpressionClass;

#define VALA_TYPE_SYMBOL (vala_symbol_get_type ())
#define VALA_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SYMBOL, ValaSymbol))
#define VALA_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SYMBOL, ValaSymbolClass))
#define VALA_IS_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SYMBOL))
#define VALA_IS_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SYMBOL))
#define VALA_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SYMBOL, ValaSymbolClass))

typedef struct _ValaSymbol ValaSymbol;
typedef struct _ValaSymbolClass ValaSymbolClass;

#define VALA_TYPE_LOCAL_VARIABLE (vala_local_variable_get_type ())
#define VALA_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariable))
#define VALA_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))
#define VALA_IS_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_IS_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_LOCAL_VARIABLE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))

typedef struct _ValaLocalVariable ValaLocalVariable;
typedef struct _ValaLocalVariableClass ValaLocalVariableClass;

#define VALA_TYPE_ATTRIBUTE (vala_attribute_get_type ())
#define VALA_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ATTRIBUTE, ValaAttribute))
#define VALA_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))
#define VALA_IS_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ATTRIBUTE))
#define VALA_IS_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ATTRIBUTE))
#define VALA_ATTRIBUTE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))

typedef struct _ValaAttribute ValaAttribute;
typedef struct _ValaAttributeClass ValaAttributeClass;

#define VALA_TYPE_STATEMENT (vala_statement_get_type ())
#define VALA_STATEMENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_STATEMENT, ValaStatement))
#define VALA_IS_STATEMENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_STATEMENT))
#define VALA_STATEMENT_GET_INTERFACE(obj) (G_TYPE_INSTANCE_GET_INTERFACE ((obj), VALA_TYPE_STATEMENT, ValaStatementIface))

typedef struct _ValaStatement ValaStatement;
typedef struct _ValaStatementIface ValaStatementIface;

#define VALA_TYPE_TRY_STATEMENT (vala_try_statement_get_type ())
#define VALA_TRY_STATEMENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_TRY_STATEMENT, ValaTryStatement))
#define VALA_TRY_STATEMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_TRY_STATEMENT, ValaTryStatementClass))
#define VALA_IS_TRY_STATEMENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_TRY_STATEMENT))
#define VALA_IS_TRY_STATEMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_TRY_STATEMENT))
#define VALA_TRY_STATEMENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_TRY_STATEMENT, ValaTryStatementClass))

typedef struct _ValaTryStatement ValaTryStatement;
typedef struct _ValaTryStatementClass ValaTryStatementClass;
typedef struct _ValaTryStatementPrivate ValaTryStatementPrivate;

#define VALA_TYPE_BLOCK (vala_block_get_type ())
#define VALA_BLOCK(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_BLOCK, ValaBlock))
#define VALA_BLOCK_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_BLOCK, ValaBlockClass))
#define VALA_IS_BLOCK(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_BLOCK))
#define VALA_IS_BLOCK_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_BLOCK))
#define VALA_BLOCK_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_BLOCK, ValaBlockClass))

typedef struct _ValaBlock ValaBlock;
typedef struct _ValaBlockClass ValaBlockClass;

#define VALA_TYPE_CATCH_CLAUSE (vala_catch_clause_get_type ())
#define VALA_CATCH_CLAUSE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CATCH_CLAUSE, ValaCatchClause))
#define VALA_CATCH_CLAUSE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CATCH_CLAUSE, ValaCatchClauseClass))
#define VALA_IS_CATCH_CLAUSE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CATCH_CLAUSE))
#define VALA_IS_CATCH_CLAUSE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CATCH_CLAUSE))
#define VALA_CATCH_CLAUSE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CATCH_CLAUSE, ValaCatchClauseClass))

typedef struct _ValaCatchClause ValaCatchClause;
typedef struct _ValaCatchClauseClass ValaCatchClauseClass;

#define VALA_TYPE_SOURCE_REFERENCE (vala_source_reference_get_type ())
#define VALA_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReference))
#define VALA_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))
#define VALA_IS_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_IS_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_SOURCE_REFERENCE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))

typedef struct _ValaSourceReference ValaSourceReference;
typedef struct _ValaSourceReferenceClass ValaSourceReferenceClass;

/**
 * Represents a part of the parsed source code.
 *
 * Code nodes get created by the parser and are used throughout the whole
 * compilation process.
 */
struct _ValaCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCodeNodePrivate * priv;
	GList* attributes;
};

struct _ValaCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCodeNode *self);
	void (*accept) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	void (*accept_children) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	gboolean (*check) (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
	void (*replace_type) (ValaCodeNode* self, ValaDataType* old_type, ValaDataType* new_type);
	void (*replace_expression) (ValaCodeNode* self, ValaExpression* old_node, ValaExpression* new_node);
	char* (*to_string) (ValaCodeNode* self);
	void (*get_defined_variables) (ValaCodeNode* self, GeeCollection* collection);
	void (*get_used_variables) (ValaCodeNode* self, GeeCollection* collection);
};

/**
 * Interface for all statement types.
 */
struct _ValaStatementIface {
	GTypeInterface parent_iface;
};

/**
 * Represents a try statement in the source code.
 */
struct _ValaTryStatement {
	ValaCodeNode parent_instance;
	ValaTryStatementPrivate * priv;
};

struct _ValaTryStatementClass {
	ValaCodeNodeClass parent_class;
};

struct _ValaTryStatementPrivate {
	ValaBlock* _body;
	ValaBlock* _finally_body;
	GeeList* catch_clauses;
};



gpointer vala_code_node_ref (gpointer instance);
void vala_code_node_unref (gpointer instance);
GParamSpec* vala_param_spec_code_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_node (GValue* value, gpointer v_object);
gpointer vala_value_get_code_node (const GValue* value);
GType vala_code_node_get_type (void);
gpointer vala_code_visitor_ref (gpointer instance);
void vala_code_visitor_unref (gpointer instance);
GParamSpec* vala_param_spec_code_visitor (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_visitor (GValue* value, gpointer v_object);
gpointer vala_value_get_code_visitor (const GValue* value);
GType vala_code_visitor_get_type (void);
GType vala_semantic_analyzer_get_type (void);
GType vala_data_type_get_type (void);
GType vala_expression_get_type (void);
GType vala_symbol_get_type (void);
GType vala_local_variable_get_type (void);
GType vala_attribute_get_type (void);
GType vala_statement_get_type (void);
GType vala_try_statement_get_type (void);
GType vala_block_get_type (void);
GType vala_catch_clause_get_type (void);
#define VALA_TRY_STATEMENT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_TRY_STATEMENT, ValaTryStatementPrivate))
enum  {
	VALA_TRY_STATEMENT_DUMMY_PROPERTY
};
void vala_try_statement_set_body (ValaTryStatement* self, ValaBlock* value);
void vala_try_statement_set_finally_body (ValaTryStatement* self, ValaBlock* value);
gpointer vala_source_reference_ref (gpointer instance);
void vala_source_reference_unref (gpointer instance);
GParamSpec* vala_param_spec_source_reference (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_reference (GValue* value, gpointer v_object);
gpointer vala_value_get_source_reference (const GValue* value);
GType vala_source_reference_get_type (void);
void vala_code_node_set_source_reference (ValaCodeNode* self, ValaSourceReference* value);
ValaTryStatement* vala_try_statement_new (ValaBlock* body, ValaBlock* finally_body, ValaSourceReference* source_reference);
ValaTryStatement* vala_try_statement_construct (GType object_type, ValaBlock* body, ValaBlock* finally_body, ValaSourceReference* source_reference);
ValaTryStatement* vala_try_statement_new (ValaBlock* body, ValaBlock* finally_body, ValaSourceReference* source_reference);
void vala_try_statement_add_catch_clause (ValaTryStatement* self, ValaCatchClause* clause);
GeeList* vala_try_statement_get_catch_clauses (ValaTryStatement* self);
void vala_code_visitor_visit_try_statement (ValaCodeVisitor* self, ValaTryStatement* stmt);
static void vala_try_statement_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
ValaBlock* vala_try_statement_get_body (ValaTryStatement* self);
void vala_code_node_accept (ValaCodeNode* self, ValaCodeVisitor* visitor);
ValaBlock* vala_try_statement_get_finally_body (ValaTryStatement* self);
static void vala_try_statement_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
gboolean vala_code_node_get_checked (ValaCodeNode* self);
gboolean vala_code_node_get_error (ValaCodeNode* self);
void vala_code_node_set_checked (ValaCodeNode* self, gboolean value);
gboolean vala_code_node_check (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
static gboolean vala_try_statement_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static gpointer vala_try_statement_parent_class = NULL;
static ValaStatementIface* vala_try_statement_vala_statement_parent_iface = NULL;
static void vala_try_statement_finalize (ValaCodeNode* obj);



/**
 * Creates a new try statement.
 *
 * @param body             body of the try statement
 * @param finally_body     body of the optional finally clause
 * @param source_reference reference to source code
 * @return                 newly created try statement
 */
ValaTryStatement* vala_try_statement_construct (GType object_type, ValaBlock* body, ValaBlock* finally_body, ValaSourceReference* source_reference) {
	ValaTryStatement* self;
	g_return_val_if_fail (body != NULL, NULL);
	self = (ValaTryStatement*) g_type_create_instance (object_type);
	vala_try_statement_set_body (self, body);
	vala_try_statement_set_finally_body (self, finally_body);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source_reference);
	return self;
}


ValaTryStatement* vala_try_statement_new (ValaBlock* body, ValaBlock* finally_body, ValaSourceReference* source_reference) {
	return vala_try_statement_construct (VALA_TYPE_TRY_STATEMENT, body, finally_body, source_reference);
}


/**
 * Appends the specified clause to the list of catch clauses.
 *
 * @param clause a catch clause
 */
void vala_try_statement_add_catch_clause (ValaTryStatement* self, ValaCatchClause* clause) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (clause != NULL);
	gee_collection_add ((GeeCollection*) self->priv->catch_clauses, clause);
}


/**
 * Returns a copy of the list of catch clauses.
 *
 * @return list of catch clauses
 */
GeeList* vala_try_statement_get_catch_clauses (ValaTryStatement* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_CATCH_CLAUSE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->catch_clauses);
}


static void vala_try_statement_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaTryStatement * self;
	self = (ValaTryStatement*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_try_statement (visitor, self);
}


static void vala_try_statement_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaTryStatement * self;
	self = (ValaTryStatement*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_node_accept ((ValaCodeNode*) self->priv->_body, visitor);
	{
		GeeIterator* _clause_it;
		_clause_it = gee_iterable_iterator ((GeeIterable*) self->priv->catch_clauses);
		while (gee_iterator_next (_clause_it)) {
			ValaCatchClause* clause;
			clause = (ValaCatchClause*) gee_iterator_get (_clause_it);
			vala_code_node_accept ((ValaCodeNode*) clause, visitor);
			(clause == NULL) ? NULL : (clause = (vala_code_node_unref (clause), NULL));
		}
		(_clause_it == NULL) ? NULL : (_clause_it = (gee_collection_object_unref (_clause_it), NULL));
	}
	if (self->priv->_finally_body != NULL) {
		vala_code_node_accept ((ValaCodeNode*) self->priv->_finally_body, visitor);
	}
}


static gboolean vala_try_statement_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaTryStatement * self;
	self = (ValaTryStatement*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	vala_code_node_check ((ValaCodeNode*) self->priv->_body, analyzer);
	{
		GeeIterator* _clause_it;
		_clause_it = gee_iterable_iterator ((GeeIterable*) self->priv->catch_clauses);
		while (gee_iterator_next (_clause_it)) {
			ValaCatchClause* clause;
			clause = (ValaCatchClause*) gee_iterator_get (_clause_it);
			vala_code_node_check ((ValaCodeNode*) clause, analyzer);
			(clause == NULL) ? NULL : (clause = (vala_code_node_unref (clause), NULL));
		}
		(_clause_it == NULL) ? NULL : (_clause_it = (gee_collection_object_unref (_clause_it), NULL));
	}
	if (self->priv->_finally_body != NULL) {
		vala_code_node_check ((ValaCodeNode*) self->priv->_finally_body, analyzer);
	}
	return !vala_code_node_get_error ((ValaCodeNode*) self);
}


ValaBlock* vala_try_statement_get_body (ValaTryStatement* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_body;
}


void vala_try_statement_set_body (ValaTryStatement* self, ValaBlock* value) {
	ValaBlock* _tmp2_;
	ValaBlock* _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_body = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_)), (self->priv->_body == NULL) ? NULL : (self->priv->_body = (vala_code_node_unref (self->priv->_body), NULL)), _tmp2_);
}


ValaBlock* vala_try_statement_get_finally_body (ValaTryStatement* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_finally_body;
}


void vala_try_statement_set_finally_body (ValaTryStatement* self, ValaBlock* value) {
	ValaBlock* _tmp2_;
	ValaBlock* _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_finally_body = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_)), (self->priv->_finally_body == NULL) ? NULL : (self->priv->_finally_body = (vala_code_node_unref (self->priv->_finally_body), NULL)), _tmp2_);
}


static void vala_try_statement_class_init (ValaTryStatementClass * klass) {
	vala_try_statement_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_try_statement_finalize;
	g_type_class_add_private (klass, sizeof (ValaTryStatementPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_try_statement_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_try_statement_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->check = vala_try_statement_real_check;
}


static void vala_try_statement_vala_statement_interface_init (ValaStatementIface * iface) {
	vala_try_statement_vala_statement_parent_iface = g_type_interface_peek_parent (iface);
}


static void vala_try_statement_instance_init (ValaTryStatement * self) {
	self->priv = VALA_TRY_STATEMENT_GET_PRIVATE (self);
	self->priv->catch_clauses = (GeeList*) gee_array_list_new (VALA_TYPE_CATCH_CLAUSE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
}


static void vala_try_statement_finalize (ValaCodeNode* obj) {
	ValaTryStatement * self;
	self = VALA_TRY_STATEMENT (obj);
	(self->priv->_body == NULL) ? NULL : (self->priv->_body = (vala_code_node_unref (self->priv->_body), NULL));
	(self->priv->_finally_body == NULL) ? NULL : (self->priv->_finally_body = (vala_code_node_unref (self->priv->_finally_body), NULL));
	(self->priv->catch_clauses == NULL) ? NULL : (self->priv->catch_clauses = (gee_collection_object_unref (self->priv->catch_clauses), NULL));
	VALA_CODE_NODE_CLASS (vala_try_statement_parent_class)->finalize (obj);
}


GType vala_try_statement_get_type (void) {
	static GType vala_try_statement_type_id = 0;
	if (vala_try_statement_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaTryStatementClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_try_statement_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaTryStatement), 0, (GInstanceInitFunc) vala_try_statement_instance_init, NULL };
		static const GInterfaceInfo vala_statement_info = { (GInterfaceInitFunc) vala_try_statement_vala_statement_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		vala_try_statement_type_id = g_type_register_static (VALA_TYPE_CODE_NODE, "ValaTryStatement", &g_define_type_info, 0);
		g_type_add_interface_static (vala_try_statement_type_id, VALA_TYPE_STATEMENT, &vala_statement_info);
	}
	return vala_try_statement_type_id;
}




