/* valaenumvalue.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>
#include <valaccode.h>


#define VALA_TYPE_CODE_NODE (vala_code_node_get_type ())
#define VALA_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_NODE, ValaCodeNode))
#define VALA_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))
#define VALA_IS_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_NODE))
#define VALA_IS_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_NODE))
#define VALA_CODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))

typedef struct _ValaCodeNode ValaCodeNode;
typedef struct _ValaCodeNodeClass ValaCodeNodeClass;
typedef struct _ValaCodeNodePrivate ValaCodeNodePrivate;

#define VALA_TYPE_CODE_VISITOR (vala_code_visitor_get_type ())
#define VALA_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitor))
#define VALA_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))
#define VALA_IS_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_VISITOR))
#define VALA_IS_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_VISITOR))
#define VALA_CODE_VISITOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))

typedef struct _ValaCodeVisitor ValaCodeVisitor;
typedef struct _ValaCodeVisitorClass ValaCodeVisitorClass;

#define VALA_TYPE_SEMANTIC_ANALYZER (vala_semantic_analyzer_get_type ())
#define VALA_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzer))
#define VALA_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))
#define VALA_IS_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_IS_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_SEMANTIC_ANALYZER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))

typedef struct _ValaSemanticAnalyzer ValaSemanticAnalyzer;
typedef struct _ValaSemanticAnalyzerClass ValaSemanticAnalyzerClass;

#define VALA_TYPE_DATA_TYPE (vala_data_type_get_type ())
#define VALA_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DATA_TYPE, ValaDataType))
#define VALA_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))
#define VALA_IS_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DATA_TYPE))
#define VALA_IS_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DATA_TYPE))
#define VALA_DATA_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))

typedef struct _ValaDataType ValaDataType;
typedef struct _ValaDataTypeClass ValaDataTypeClass;

#define VALA_TYPE_EXPRESSION (vala_expression_get_type ())
#define VALA_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_EXPRESSION, ValaExpression))
#define VALA_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_EXPRESSION, ValaExpressionClass))
#define VALA_IS_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_EXPRESSION))
#define VALA_IS_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_EXPRESSION))
#define VALA_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_EXPRESSION, ValaExpressionClass))

typedef struct _ValaExpression ValaExpression;
typedef struct _ValaExpressionClass ValaExpressionClass;

#define VALA_TYPE_SYMBOL (vala_symbol_get_type ())
#define VALA_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SYMBOL, ValaSymbol))
#define VALA_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SYMBOL, ValaSymbolClass))
#define VALA_IS_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SYMBOL))
#define VALA_IS_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SYMBOL))
#define VALA_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SYMBOL, ValaSymbolClass))

typedef struct _ValaSymbol ValaSymbol;
typedef struct _ValaSymbolClass ValaSymbolClass;

#define VALA_TYPE_LOCAL_VARIABLE (vala_local_variable_get_type ())
#define VALA_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariable))
#define VALA_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))
#define VALA_IS_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_IS_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_LOCAL_VARIABLE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))

typedef struct _ValaLocalVariable ValaLocalVariable;
typedef struct _ValaLocalVariableClass ValaLocalVariableClass;

#define VALA_TYPE_ATTRIBUTE (vala_attribute_get_type ())
#define VALA_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ATTRIBUTE, ValaAttribute))
#define VALA_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))
#define VALA_IS_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ATTRIBUTE))
#define VALA_IS_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ATTRIBUTE))
#define VALA_ATTRIBUTE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))

typedef struct _ValaAttribute ValaAttribute;
typedef struct _ValaAttributeClass ValaAttributeClass;
typedef struct _ValaSymbolPrivate ValaSymbolPrivate;

#define VALA_TYPE_ENUM_VALUE (vala_enum_value_get_type ())
#define VALA_ENUM_VALUE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ENUM_VALUE, ValaEnumValue))
#define VALA_ENUM_VALUE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ENUM_VALUE, ValaEnumValueClass))
#define VALA_IS_ENUM_VALUE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ENUM_VALUE))
#define VALA_IS_ENUM_VALUE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ENUM_VALUE))
#define VALA_ENUM_VALUE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ENUM_VALUE, ValaEnumValueClass))

typedef struct _ValaEnumValue ValaEnumValue;
typedef struct _ValaEnumValueClass ValaEnumValueClass;
typedef struct _ValaEnumValuePrivate ValaEnumValuePrivate;

#define VALA_TYPE_SOURCE_REFERENCE (vala_source_reference_get_type ())
#define VALA_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReference))
#define VALA_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))
#define VALA_IS_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_IS_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_SOURCE_REFERENCE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))

typedef struct _ValaSourceReference ValaSourceReference;
typedef struct _ValaSourceReferenceClass ValaSourceReferenceClass;

#define VALA_TYPE_TYPESYMBOL (vala_typesymbol_get_type ())
#define VALA_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbol))
#define VALA_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))
#define VALA_IS_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_TYPESYMBOL))
#define VALA_IS_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_TYPESYMBOL))
#define VALA_TYPESYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))

typedef struct _ValaTypeSymbol ValaTypeSymbol;
typedef struct _ValaTypeSymbolClass ValaTypeSymbolClass;

#define VALA_TYPE_ENUM (vala_enum_get_type ())
#define VALA_ENUM(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ENUM, ValaEnum))
#define VALA_ENUM_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ENUM, ValaEnumClass))
#define VALA_IS_ENUM(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ENUM))
#define VALA_IS_ENUM_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ENUM))
#define VALA_ENUM_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ENUM, ValaEnumClass))

typedef struct _ValaEnum ValaEnum;
typedef struct _ValaEnumClass ValaEnumClass;

/**
 * Represents a part of the parsed source code.
 *
 * Code nodes get created by the parser and are used throughout the whole
 * compilation process.
 */
struct _ValaCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCodeNodePrivate * priv;
	GList* attributes;
};

struct _ValaCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCodeNode *self);
	void (*accept) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	void (*accept_children) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	gboolean (*check) (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
	void (*replace_type) (ValaCodeNode* self, ValaDataType* old_type, ValaDataType* new_type);
	void (*replace_expression) (ValaCodeNode* self, ValaExpression* old_node, ValaExpression* new_node);
	char* (*to_string) (ValaCodeNode* self);
	void (*get_defined_variables) (ValaCodeNode* self, GeeCollection* collection);
	void (*get_used_variables) (ValaCodeNode* self, GeeCollection* collection);
};

/**
 * Represents a node in the symbol tree.
 */
struct _ValaSymbol {
	ValaCodeNode parent_instance;
	ValaSymbolPrivate * priv;
};

struct _ValaSymbolClass {
	ValaCodeNodeClass parent_class;
	char* (*get_cprefix) (ValaSymbol* self);
	char* (*get_lower_case_cname) (ValaSymbol* self, const char* infix);
	char* (*get_lower_case_cprefix) (ValaSymbol* self);
	GeeList* (*get_cheader_filenames) (ValaSymbol* self);
};

/**
 * Represents an enum member in the source code.
 */
struct _ValaEnumValue {
	ValaSymbol parent_instance;
	ValaEnumValuePrivate * priv;
};

struct _ValaEnumValueClass {
	ValaSymbolClass parent_class;
};

struct _ValaEnumValuePrivate {
	ValaExpression* _value;
	char* cname;
};



gpointer vala_code_node_ref (gpointer instance);
void vala_code_node_unref (gpointer instance);
GParamSpec* vala_param_spec_code_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_node (GValue* value, gpointer v_object);
gpointer vala_value_get_code_node (const GValue* value);
GType vala_code_node_get_type (void);
gpointer vala_code_visitor_ref (gpointer instance);
void vala_code_visitor_unref (gpointer instance);
GParamSpec* vala_param_spec_code_visitor (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_visitor (GValue* value, gpointer v_object);
gpointer vala_value_get_code_visitor (const GValue* value);
GType vala_code_visitor_get_type (void);
GType vala_semantic_analyzer_get_type (void);
GType vala_data_type_get_type (void);
GType vala_expression_get_type (void);
GType vala_symbol_get_type (void);
GType vala_local_variable_get_type (void);
GType vala_attribute_get_type (void);
GType vala_enum_value_get_type (void);
#define VALA_ENUM_VALUE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_ENUM_VALUE, ValaEnumValuePrivate))
enum  {
	VALA_ENUM_VALUE_DUMMY_PROPERTY
};
gpointer vala_source_reference_ref (gpointer instance);
void vala_source_reference_unref (gpointer instance);
GParamSpec* vala_param_spec_source_reference (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_reference (GValue* value, gpointer v_object);
gpointer vala_value_get_source_reference (const GValue* value);
GType vala_source_reference_get_type (void);
ValaSymbol* vala_symbol_new (const char* name, ValaSourceReference* source_reference);
ValaSymbol* vala_symbol_construct (GType object_type, const char* name, ValaSourceReference* source_reference);
ValaEnumValue* vala_enum_value_new (const char* name, ValaSourceReference* source_reference);
ValaEnumValue* vala_enum_value_construct (GType object_type, const char* name, ValaSourceReference* source_reference);
ValaEnumValue* vala_enum_value_new (const char* name, ValaSourceReference* source_reference);
void vala_enum_value_set_value (ValaEnumValue* self, ValaExpression* value);
ValaEnumValue* vala_enum_value_new_with_value (const char* name, ValaExpression* value, ValaSourceReference* source_reference);
ValaEnumValue* vala_enum_value_construct_with_value (GType object_type, const char* name, ValaExpression* value, ValaSourceReference* source_reference);
ValaEnumValue* vala_enum_value_new_with_value (const char* name, ValaExpression* value, ValaSourceReference* source_reference);
const char* vala_symbol_get_name (ValaSymbol* self);
ValaCCodeConstant* vala_enum_value_get_canonical_cconstant (ValaEnumValue* self);
void vala_code_visitor_visit_enum_value (ValaCodeVisitor* self, ValaEnumValue* ev);
static void vala_enum_value_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
ValaExpression* vala_enum_value_get_value (ValaEnumValue* self);
void vala_code_node_accept (ValaCodeNode* self, ValaCodeVisitor* visitor);
static void vala_enum_value_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
const char* vala_attribute_get_name (ValaAttribute* self);
gboolean vala_attribute_has_argument (ValaAttribute* self, const char* name);
char* vala_attribute_get_string (ValaAttribute* self, const char* name);
void vala_enum_value_process_attributes (ValaEnumValue* self);
char* vala_enum_value_get_default_cname (ValaEnumValue* self);
char* vala_enum_value_get_cname (ValaEnumValue* self);
ValaSymbol* vala_symbol_get_parent_symbol (ValaSymbol* self);
GType vala_typesymbol_get_type (void);
GType vala_enum_get_type (void);
char* vala_symbol_get_cprefix (ValaSymbol* self);
void vala_enum_value_set_cname (ValaEnumValue* self, const char* cname);
gboolean vala_code_node_get_checked (ValaCodeNode* self);
gboolean vala_code_node_get_error (ValaCodeNode* self);
void vala_code_node_set_checked (ValaCodeNode* self, gboolean value);
gboolean vala_code_node_check (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
static gboolean vala_enum_value_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static gpointer vala_enum_value_parent_class = NULL;
static void vala_enum_value_finalize (ValaCodeNode* obj);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Creates a new enum value.
 *
 * @param name enum value name
 * @return     newly created enum value
 */
ValaEnumValue* vala_enum_value_construct (GType object_type, const char* name, ValaSourceReference* source_reference) {
	ValaEnumValue* self;
	g_return_val_if_fail (name != NULL, NULL);
	self = (ValaEnumValue*) vala_symbol_construct (object_type, name, source_reference);
	return self;
}


ValaEnumValue* vala_enum_value_new (const char* name, ValaSourceReference* source_reference) {
	return vala_enum_value_construct (VALA_TYPE_ENUM_VALUE, name, source_reference);
}


/**
 * Creates a new enum value with the specified numerical representation.
 *
 * @param name  enum value name
 * @param value numerical representation
 * @return      newly created enum value
 */
ValaEnumValue* vala_enum_value_construct_with_value (GType object_type, const char* name, ValaExpression* value, ValaSourceReference* source_reference) {
	ValaEnumValue* self;
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (value != NULL, NULL);
	self = (ValaEnumValue*) vala_enum_value_construct (object_type, name, source_reference);
	vala_enum_value_set_value (self, value);
	return self;
}


ValaEnumValue* vala_enum_value_new_with_value (const char* name, ValaExpression* value, ValaSourceReference* source_reference) {
	return vala_enum_value_construct_with_value (VALA_TYPE_ENUM_VALUE, name, value, source_reference);
}


/**
 * Returns the string literal of this signal to be used in C code.
 * (FIXME: from vlaasignal.vala)
 *
 * @return string literal to be used in C code
 */
ValaCCodeConstant* vala_enum_value_get_canonical_cconstant (ValaEnumValue* self) {
	GString* str;
	const char* _tmp0_;
	char* i;
	ValaCCodeConstant* _tmp3_;
	g_return_val_if_fail (self != NULL, NULL);
	str = g_string_new ("\"");
	_tmp0_ = NULL;
	i = (_tmp0_ = vala_symbol_get_name ((ValaSymbol*) self), (_tmp0_ == NULL) ? NULL : g_strdup (_tmp0_));
	while (g_utf8_strlen (i, -1) > 0) {
		gunichar c;
		char* _tmp2_;
		const char* _tmp1_;
		c = g_utf8_get_char (i);
		if (c == '_') {
			g_string_append_c (str, '-');
		} else {
			g_string_append_unichar (str, g_unichar_tolower (c));
		}
		_tmp2_ = NULL;
		_tmp1_ = NULL;
		i = (_tmp2_ = (_tmp1_ = g_utf8_next_char (i), (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_)), i = (g_free (i), NULL), _tmp2_);
	}
	g_string_append_c (str, '"');
	_tmp3_ = NULL;
	return (_tmp3_ = vala_ccode_constant_new (str->str), (str == NULL) ? NULL : (str = (g_string_free (str, TRUE), NULL)), i = (g_free (i), NULL), _tmp3_);
}


static void vala_enum_value_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaEnumValue * self;
	self = (ValaEnumValue*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_enum_value (visitor, self);
}


static void vala_enum_value_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaEnumValue * self;
	self = (ValaEnumValue*) base;
	g_return_if_fail (visitor != NULL);
	if (self->priv->_value != NULL) {
		vala_code_node_accept ((ValaCodeNode*) self->priv->_value, visitor);
	}
}


/**
 * Process all associated attributes.
 */
void vala_enum_value_process_attributes (ValaEnumValue* self) {
	g_return_if_fail (self != NULL);
	{
		GList* a_collection;
		GList* a_it;
		a_collection = ((ValaCodeNode*) self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp2_;
			ValaAttribute* a;
			_tmp2_ = NULL;
			a = (_tmp2_ = (ValaAttribute*) a_it->data, (_tmp2_ == NULL) ? NULL : vala_code_node_ref (_tmp2_));
			{
				gboolean _tmp0_;
				_tmp0_ = FALSE;
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0) {
					_tmp0_ = vala_attribute_has_argument (a, "cname");
				} else {
					_tmp0_ = FALSE;
				}
				if (_tmp0_) {
					char* _tmp1_;
					_tmp1_ = NULL;
					self->priv->cname = (_tmp1_ = vala_attribute_get_string (a, "cname"), self->priv->cname = (g_free (self->priv->cname), NULL), _tmp1_);
				}
				(a == NULL) ? NULL : (a = (vala_code_node_unref (a), NULL));
			}
		}
	}
}


/**
 * Returns the name of this enum value as it is used in C code.
 *
 * @return the name to be used in C code
 */
char* vala_enum_value_get_cname (ValaEnumValue* self) {
	const char* _tmp1_;
	g_return_val_if_fail (self != NULL, NULL);
	if (self->priv->cname == NULL) {
		char* _tmp0_;
		_tmp0_ = NULL;
		self->priv->cname = (_tmp0_ = vala_enum_value_get_default_cname (self), self->priv->cname = (g_free (self->priv->cname), NULL), _tmp0_);
	}
	_tmp1_ = NULL;
	return (_tmp1_ = self->priv->cname, (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_));
}


char* vala_enum_value_get_default_cname (ValaEnumValue* self) {
	ValaEnum* _tmp0_;
	ValaEnum* en;
	char* _tmp1_;
	char* _tmp2_;
	char* _tmp3_;
	g_return_val_if_fail (self != NULL, NULL);
	_tmp0_ = NULL;
	en = (_tmp0_ = VALA_ENUM (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), (_tmp0_ == NULL) ? NULL : vala_code_node_ref (_tmp0_));
	_tmp1_ = NULL;
	_tmp2_ = NULL;
	_tmp3_ = NULL;
	return (_tmp3_ = (_tmp2_ = g_strdup_printf ("%s%s", _tmp1_ = vala_symbol_get_cprefix ((ValaSymbol*) en), vala_symbol_get_name ((ValaSymbol*) self)), _tmp1_ = (g_free (_tmp1_), NULL), _tmp2_), (en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL)), _tmp3_);
}


/**
 * Sets the name of this enum value to be used in C code.
 *
 * @param cname the name to be used in C code
 */
void vala_enum_value_set_cname (ValaEnumValue* self, const char* cname) {
	char* _tmp1_;
	const char* _tmp0_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (cname != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->cname = (_tmp1_ = (_tmp0_ = cname, (_tmp0_ == NULL) ? NULL : g_strdup (_tmp0_)), self->priv->cname = (g_free (self->priv->cname), NULL), _tmp1_);
}


static gboolean vala_enum_value_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaEnumValue * self;
	self = (ValaEnumValue*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	vala_enum_value_process_attributes (self);
	if (self->priv->_value != NULL) {
		vala_code_node_check ((ValaCodeNode*) self->priv->_value, analyzer);
	}
	return !vala_code_node_get_error ((ValaCodeNode*) self);
}


ValaExpression* vala_enum_value_get_value (ValaEnumValue* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_value;
}


void vala_enum_value_set_value (ValaEnumValue* self, ValaExpression* value) {
	ValaExpression* _tmp2_;
	ValaExpression* _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_value = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : vala_code_node_ref (_tmp1_)), (self->priv->_value == NULL) ? NULL : (self->priv->_value = (vala_code_node_unref (self->priv->_value), NULL)), _tmp2_);
}


static void vala_enum_value_class_init (ValaEnumValueClass * klass) {
	vala_enum_value_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_enum_value_finalize;
	g_type_class_add_private (klass, sizeof (ValaEnumValuePrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_enum_value_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_enum_value_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->check = vala_enum_value_real_check;
}


static void vala_enum_value_instance_init (ValaEnumValue * self) {
	self->priv = VALA_ENUM_VALUE_GET_PRIVATE (self);
}


static void vala_enum_value_finalize (ValaCodeNode* obj) {
	ValaEnumValue * self;
	self = VALA_ENUM_VALUE (obj);
	(self->priv->_value == NULL) ? NULL : (self->priv->_value = (vala_code_node_unref (self->priv->_value), NULL));
	self->priv->cname = (g_free (self->priv->cname), NULL);
	VALA_CODE_NODE_CLASS (vala_enum_value_parent_class)->finalize (obj);
}


GType vala_enum_value_get_type (void) {
	static GType vala_enum_value_type_id = 0;
	if (vala_enum_value_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaEnumValueClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_enum_value_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaEnumValue), 0, (GInstanceInitFunc) vala_enum_value_instance_init, NULL };
		vala_enum_value_type_id = g_type_register_static (VALA_TYPE_SYMBOL, "ValaEnumValue", &g_define_type_info, 0);
	}
	return vala_enum_value_type_id;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




