/* valaccodeforstatement.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <gee.h>
#include <stdlib.h>
#include <string.h>


#define VALA_TYPE_CCODE_NODE (vala_ccode_node_get_type ())
#define VALA_CCODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_NODE, ValaCCodeNode))
#define VALA_CCODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_NODE, ValaCCodeNodeClass))
#define VALA_IS_CCODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_NODE))
#define VALA_IS_CCODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_NODE))
#define VALA_CCODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_NODE, ValaCCodeNodeClass))

typedef struct _ValaCCodeNode ValaCCodeNode;
typedef struct _ValaCCodeNodeClass ValaCCodeNodeClass;
typedef struct _ValaCCodeNodePrivate ValaCCodeNodePrivate;

#define VALA_TYPE_CCODE_WRITER (vala_ccode_writer_get_type ())
#define VALA_CCODE_WRITER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_WRITER, ValaCCodeWriter))
#define VALA_CCODE_WRITER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_WRITER, ValaCCodeWriterClass))
#define VALA_IS_CCODE_WRITER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_WRITER))
#define VALA_IS_CCODE_WRITER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_WRITER))
#define VALA_CCODE_WRITER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_WRITER, ValaCCodeWriterClass))

typedef struct _ValaCCodeWriter ValaCCodeWriter;
typedef struct _ValaCCodeWriterClass ValaCCodeWriterClass;

#define VALA_TYPE_CCODE_STATEMENT (vala_ccode_statement_get_type ())
#define VALA_CCODE_STATEMENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_STATEMENT, ValaCCodeStatement))
#define VALA_CCODE_STATEMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_STATEMENT, ValaCCodeStatementClass))
#define VALA_IS_CCODE_STATEMENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_STATEMENT))
#define VALA_IS_CCODE_STATEMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_STATEMENT))
#define VALA_CCODE_STATEMENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_STATEMENT, ValaCCodeStatementClass))

typedef struct _ValaCCodeStatement ValaCCodeStatement;
typedef struct _ValaCCodeStatementClass ValaCCodeStatementClass;
typedef struct _ValaCCodeStatementPrivate ValaCCodeStatementPrivate;

#define VALA_TYPE_CCODE_FOR_STATEMENT (vala_ccode_for_statement_get_type ())
#define VALA_CCODE_FOR_STATEMENT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_FOR_STATEMENT, ValaCCodeForStatement))
#define VALA_CCODE_FOR_STATEMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_FOR_STATEMENT, ValaCCodeForStatementClass))
#define VALA_IS_CCODE_FOR_STATEMENT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_FOR_STATEMENT))
#define VALA_IS_CCODE_FOR_STATEMENT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_FOR_STATEMENT))
#define VALA_CCODE_FOR_STATEMENT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_FOR_STATEMENT, ValaCCodeForStatementClass))

typedef struct _ValaCCodeForStatement ValaCCodeForStatement;
typedef struct _ValaCCodeForStatementClass ValaCCodeForStatementClass;
typedef struct _ValaCCodeForStatementPrivate ValaCCodeForStatementPrivate;

#define VALA_TYPE_CCODE_EXPRESSION (vala_ccode_expression_get_type ())
#define VALA_CCODE_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_EXPRESSION, ValaCCodeExpression))
#define VALA_CCODE_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_EXPRESSION, ValaCCodeExpressionClass))
#define VALA_IS_CCODE_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_EXPRESSION))
#define VALA_IS_CCODE_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_EXPRESSION))
#define VALA_CCODE_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_EXPRESSION, ValaCCodeExpressionClass))

typedef struct _ValaCCodeExpression ValaCCodeExpression;
typedef struct _ValaCCodeExpressionClass ValaCCodeExpressionClass;

#define VALA_TYPE_CCODE_LINE_DIRECTIVE (vala_ccode_line_directive_get_type ())
#define VALA_CCODE_LINE_DIRECTIVE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CCODE_LINE_DIRECTIVE, ValaCCodeLineDirective))
#define VALA_CCODE_LINE_DIRECTIVE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CCODE_LINE_DIRECTIVE, ValaCCodeLineDirectiveClass))
#define VALA_IS_CCODE_LINE_DIRECTIVE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CCODE_LINE_DIRECTIVE))
#define VALA_IS_CCODE_LINE_DIRECTIVE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CCODE_LINE_DIRECTIVE))
#define VALA_CCODE_LINE_DIRECTIVE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CCODE_LINE_DIRECTIVE, ValaCCodeLineDirectiveClass))

typedef struct _ValaCCodeLineDirective ValaCCodeLineDirective;
typedef struct _ValaCCodeLineDirectiveClass ValaCCodeLineDirectiveClass;

/**
 * Represents a node in the C code tree.
 */
struct _ValaCCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCCodeNodePrivate * priv;
};

struct _ValaCCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCCodeNode *self);
	void (*write) (ValaCCodeNode* self, ValaCCodeWriter* writer);
	void (*write_declaration) (ValaCCodeNode* self, ValaCCodeWriter* writer);
	void (*write_combined) (ValaCCodeNode* self, ValaCCodeWriter* writer);
};

/**
 * Represents a statement in the C code.
 */
struct _ValaCCodeStatement {
	ValaCCodeNode parent_instance;
	ValaCCodeStatementPrivate * priv;
};

struct _ValaCCodeStatementClass {
	ValaCCodeNodeClass parent_class;
};

/**
 * Represents a for iteration statement in the C code.
 */
struct _ValaCCodeForStatement {
	ValaCCodeStatement parent_instance;
	ValaCCodeForStatementPrivate * priv;
};

struct _ValaCCodeForStatementClass {
	ValaCCodeStatementClass parent_class;
};

struct _ValaCCodeForStatementPrivate {
	ValaCCodeExpression* _condition;
	ValaCCodeStatement* _body;
	GeeList* initializer;
	GeeList* iterator;
};



gpointer vala_ccode_node_ref (gpointer instance);
void vala_ccode_node_unref (gpointer instance);
GParamSpec* vala_param_spec_ccode_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_ccode_node (GValue* value, gpointer v_object);
gpointer vala_value_get_ccode_node (const GValue* value);
GType vala_ccode_node_get_type (void);
gpointer vala_ccode_writer_ref (gpointer instance);
void vala_ccode_writer_unref (gpointer instance);
GParamSpec* vala_param_spec_ccode_writer (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_ccode_writer (GValue* value, gpointer v_object);
gpointer vala_value_get_ccode_writer (const GValue* value);
GType vala_ccode_writer_get_type (void);
GType vala_ccode_statement_get_type (void);
GType vala_ccode_for_statement_get_type (void);
GType vala_ccode_expression_get_type (void);
#define VALA_CCODE_FOR_STATEMENT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CCODE_FOR_STATEMENT, ValaCCodeForStatementPrivate))
enum  {
	VALA_CCODE_FOR_STATEMENT_DUMMY_PROPERTY
};
void vala_ccode_for_statement_set_condition (ValaCCodeForStatement* self, ValaCCodeExpression* value);
void vala_ccode_for_statement_set_body (ValaCCodeForStatement* self, ValaCCodeStatement* value);
ValaCCodeForStatement* vala_ccode_for_statement_new (ValaCCodeExpression* condition, ValaCCodeStatement* body);
ValaCCodeForStatement* vala_ccode_for_statement_construct (GType object_type, ValaCCodeExpression* condition, ValaCCodeStatement* body);
ValaCCodeForStatement* vala_ccode_for_statement_new (ValaCCodeExpression* condition, ValaCCodeStatement* body);
void vala_ccode_for_statement_add_initializer (ValaCCodeForStatement* self, ValaCCodeExpression* expr);
void vala_ccode_for_statement_add_iterator (ValaCCodeForStatement* self, ValaCCodeExpression* expr);
GType vala_ccode_line_directive_get_type (void);
void vala_ccode_writer_write_indent (ValaCCodeWriter* self, ValaCCodeLineDirective* line);
ValaCCodeLineDirective* vala_ccode_node_get_line (ValaCCodeNode* self);
void vala_ccode_writer_write_string (ValaCCodeWriter* self, const char* s);
void vala_ccode_node_write (ValaCCodeNode* self, ValaCCodeWriter* writer);
ValaCCodeExpression* vala_ccode_for_statement_get_condition (ValaCCodeForStatement* self);
ValaCCodeStatement* vala_ccode_for_statement_get_body (ValaCCodeForStatement* self);
static void vala_ccode_for_statement_real_write (ValaCCodeNode* base, ValaCCodeWriter* writer);
static gpointer vala_ccode_for_statement_parent_class = NULL;
static void vala_ccode_for_statement_finalize (ValaCCodeNode* obj);



ValaCCodeForStatement* vala_ccode_for_statement_construct (GType object_type, ValaCCodeExpression* condition, ValaCCodeStatement* body) {
	ValaCCodeForStatement* self;
	self = (ValaCCodeForStatement*) g_type_create_instance (object_type);
	vala_ccode_for_statement_set_condition (self, condition);
	vala_ccode_for_statement_set_body (self, body);
	return self;
}


ValaCCodeForStatement* vala_ccode_for_statement_new (ValaCCodeExpression* condition, ValaCCodeStatement* body) {
	return vala_ccode_for_statement_construct (VALA_TYPE_CCODE_FOR_STATEMENT, condition, body);
}


/**
 * Appends the specified expression to the list of initializers.
 *
 * @param expr an initializer expression
 */
void vala_ccode_for_statement_add_initializer (ValaCCodeForStatement* self, ValaCCodeExpression* expr) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (expr != NULL);
	gee_collection_add ((GeeCollection*) self->priv->initializer, expr);
}


/**
 * Appends the specified expression to the iterator.
 *
 * @param expr an iterator expression
 */
void vala_ccode_for_statement_add_iterator (ValaCCodeForStatement* self, ValaCCodeExpression* expr) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (expr != NULL);
	gee_collection_add ((GeeCollection*) self->priv->iterator, expr);
}


static void vala_ccode_for_statement_real_write (ValaCCodeNode* base, ValaCCodeWriter* writer) {
	ValaCCodeForStatement * self;
	gboolean first;
	self = (ValaCCodeForStatement*) base;
	g_return_if_fail (writer != NULL);
	first = FALSE;
	vala_ccode_writer_write_indent (writer, vala_ccode_node_get_line ((ValaCCodeNode*) self));
	vala_ccode_writer_write_string (writer, "for (");
	first = TRUE;
	{
		GeeIterator* _init_expr_it;
		_init_expr_it = gee_iterable_iterator ((GeeIterable*) self->priv->initializer);
		while (gee_iterator_next (_init_expr_it)) {
			ValaCCodeExpression* init_expr;
			init_expr = (ValaCCodeExpression*) gee_iterator_get (_init_expr_it);
			if (!first) {
				vala_ccode_writer_write_string (writer, ", ");
			} else {
				first = FALSE;
			}
			if (init_expr != NULL) {
				vala_ccode_node_write ((ValaCCodeNode*) init_expr, writer);
			}
			(init_expr == NULL) ? NULL : (init_expr = (vala_ccode_node_unref (init_expr), NULL));
		}
		(_init_expr_it == NULL) ? NULL : (_init_expr_it = (gee_collection_object_unref (_init_expr_it), NULL));
	}
	vala_ccode_writer_write_string (writer, "; ");
	if (self->priv->_condition != NULL) {
		vala_ccode_node_write ((ValaCCodeNode*) self->priv->_condition, writer);
	}
	vala_ccode_writer_write_string (writer, "; ");
	first = TRUE;
	{
		GeeIterator* _it_expr_it;
		_it_expr_it = gee_iterable_iterator ((GeeIterable*) self->priv->iterator);
		while (gee_iterator_next (_it_expr_it)) {
			ValaCCodeExpression* it_expr;
			it_expr = (ValaCCodeExpression*) gee_iterator_get (_it_expr_it);
			if (!first) {
				vala_ccode_writer_write_string (writer, ", ");
			} else {
				first = FALSE;
			}
			if (it_expr != NULL) {
				vala_ccode_node_write ((ValaCCodeNode*) it_expr, writer);
			}
			(it_expr == NULL) ? NULL : (it_expr = (vala_ccode_node_unref (it_expr), NULL));
		}
		(_it_expr_it == NULL) ? NULL : (_it_expr_it = (gee_collection_object_unref (_it_expr_it), NULL));
	}
	vala_ccode_writer_write_string (writer, ")");
	vala_ccode_node_write ((ValaCCodeNode*) self->priv->_body, writer);
}


ValaCCodeExpression* vala_ccode_for_statement_get_condition (ValaCCodeForStatement* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_condition;
}


void vala_ccode_for_statement_set_condition (ValaCCodeForStatement* self, ValaCCodeExpression* value) {
	ValaCCodeExpression* _tmp2_;
	ValaCCodeExpression* _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_condition = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : vala_ccode_node_ref (_tmp1_)), (self->priv->_condition == NULL) ? NULL : (self->priv->_condition = (vala_ccode_node_unref (self->priv->_condition), NULL)), _tmp2_);
}


ValaCCodeStatement* vala_ccode_for_statement_get_body (ValaCCodeForStatement* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_body;
}


void vala_ccode_for_statement_set_body (ValaCCodeForStatement* self, ValaCCodeStatement* value) {
	ValaCCodeStatement* _tmp2_;
	ValaCCodeStatement* _tmp1_;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_body = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : vala_ccode_node_ref (_tmp1_)), (self->priv->_body == NULL) ? NULL : (self->priv->_body = (vala_ccode_node_unref (self->priv->_body), NULL)), _tmp2_);
}


static void vala_ccode_for_statement_class_init (ValaCCodeForStatementClass * klass) {
	vala_ccode_for_statement_parent_class = g_type_class_peek_parent (klass);
	VALA_CCODE_NODE_CLASS (klass)->finalize = vala_ccode_for_statement_finalize;
	g_type_class_add_private (klass, sizeof (ValaCCodeForStatementPrivate));
	VALA_CCODE_NODE_CLASS (klass)->write = vala_ccode_for_statement_real_write;
}


static void vala_ccode_for_statement_instance_init (ValaCCodeForStatement * self) {
	self->priv = VALA_CCODE_FOR_STATEMENT_GET_PRIVATE (self);
	self->priv->initializer = (GeeList*) gee_array_list_new (VALA_TYPE_CCODE_EXPRESSION, (GBoxedCopyFunc) vala_ccode_node_ref, vala_ccode_node_unref, g_direct_equal);
	self->priv->iterator = (GeeList*) gee_array_list_new (VALA_TYPE_CCODE_EXPRESSION, (GBoxedCopyFunc) vala_ccode_node_ref, vala_ccode_node_unref, g_direct_equal);
}


static void vala_ccode_for_statement_finalize (ValaCCodeNode* obj) {
	ValaCCodeForStatement * self;
	self = VALA_CCODE_FOR_STATEMENT (obj);
	(self->priv->_condition == NULL) ? NULL : (self->priv->_condition = (vala_ccode_node_unref (self->priv->_condition), NULL));
	(self->priv->_body == NULL) ? NULL : (self->priv->_body = (vala_ccode_node_unref (self->priv->_body), NULL));
	(self->priv->initializer == NULL) ? NULL : (self->priv->initializer = (gee_collection_object_unref (self->priv->initializer), NULL));
	(self->priv->iterator == NULL) ? NULL : (self->priv->iterator = (gee_collection_object_unref (self->priv->iterator), NULL));
	VALA_CCODE_NODE_CLASS (vala_ccode_for_statement_parent_class)->finalize (obj);
}


GType vala_ccode_for_statement_get_type (void) {
	static GType vala_ccode_for_statement_type_id = 0;
	if (vala_ccode_for_statement_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCCodeForStatementClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_ccode_for_statement_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCCodeForStatement), 0, (GInstanceInitFunc) vala_ccode_for_statement_instance_init, NULL };
		vala_ccode_for_statement_type_id = g_type_register_static (VALA_TYPE_CCODE_STATEMENT, "ValaCCodeForStatement", &g_define_type_info, 0);
	}
	return vala_ccode_for_statement_type_id;
}




