/* valanamespace.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <glib.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>
#include <gee.h>


#define VALA_TYPE_CODE_NODE (vala_code_node_get_type ())
#define VALA_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_NODE, ValaCodeNode))
#define VALA_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))
#define VALA_IS_CODE_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_NODE))
#define VALA_IS_CODE_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_NODE))
#define VALA_CODE_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_NODE, ValaCodeNodeClass))

typedef struct _ValaCodeNode ValaCodeNode;
typedef struct _ValaCodeNodeClass ValaCodeNodeClass;
typedef struct _ValaCodeNodePrivate ValaCodeNodePrivate;

#define VALA_TYPE_CODE_VISITOR (vala_code_visitor_get_type ())
#define VALA_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitor))
#define VALA_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))
#define VALA_IS_CODE_VISITOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CODE_VISITOR))
#define VALA_IS_CODE_VISITOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CODE_VISITOR))
#define VALA_CODE_VISITOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CODE_VISITOR, ValaCodeVisitorClass))

typedef struct _ValaCodeVisitor ValaCodeVisitor;
typedef struct _ValaCodeVisitorClass ValaCodeVisitorClass;

#define VALA_TYPE_SEMANTIC_ANALYZER (vala_semantic_analyzer_get_type ())
#define VALA_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzer))
#define VALA_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))
#define VALA_IS_SEMANTIC_ANALYZER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_IS_SEMANTIC_ANALYZER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SEMANTIC_ANALYZER))
#define VALA_SEMANTIC_ANALYZER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerClass))

typedef struct _ValaSemanticAnalyzer ValaSemanticAnalyzer;
typedef struct _ValaSemanticAnalyzerClass ValaSemanticAnalyzerClass;

#define VALA_TYPE_DATA_TYPE (vala_data_type_get_type ())
#define VALA_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DATA_TYPE, ValaDataType))
#define VALA_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))
#define VALA_IS_DATA_TYPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DATA_TYPE))
#define VALA_IS_DATA_TYPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DATA_TYPE))
#define VALA_DATA_TYPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DATA_TYPE, ValaDataTypeClass))

typedef struct _ValaDataType ValaDataType;
typedef struct _ValaDataTypeClass ValaDataTypeClass;

#define VALA_TYPE_EXPRESSION (vala_expression_get_type ())
#define VALA_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_EXPRESSION, ValaExpression))
#define VALA_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_EXPRESSION, ValaExpressionClass))
#define VALA_IS_EXPRESSION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_EXPRESSION))
#define VALA_IS_EXPRESSION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_EXPRESSION))
#define VALA_EXPRESSION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_EXPRESSION, ValaExpressionClass))

typedef struct _ValaExpression ValaExpression;
typedef struct _ValaExpressionClass ValaExpressionClass;

#define VALA_TYPE_SYMBOL (vala_symbol_get_type ())
#define VALA_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SYMBOL, ValaSymbol))
#define VALA_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SYMBOL, ValaSymbolClass))
#define VALA_IS_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SYMBOL))
#define VALA_IS_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SYMBOL))
#define VALA_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SYMBOL, ValaSymbolClass))

typedef struct _ValaSymbol ValaSymbol;
typedef struct _ValaSymbolClass ValaSymbolClass;

#define VALA_TYPE_LOCAL_VARIABLE (vala_local_variable_get_type ())
#define VALA_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariable))
#define VALA_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))
#define VALA_IS_LOCAL_VARIABLE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_IS_LOCAL_VARIABLE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_LOCAL_VARIABLE))
#define VALA_LOCAL_VARIABLE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_LOCAL_VARIABLE, ValaLocalVariableClass))

typedef struct _ValaLocalVariable ValaLocalVariable;
typedef struct _ValaLocalVariableClass ValaLocalVariableClass;

#define VALA_TYPE_ATTRIBUTE (vala_attribute_get_type ())
#define VALA_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ATTRIBUTE, ValaAttribute))
#define VALA_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))
#define VALA_IS_ATTRIBUTE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ATTRIBUTE))
#define VALA_IS_ATTRIBUTE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ATTRIBUTE))
#define VALA_ATTRIBUTE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ATTRIBUTE, ValaAttributeClass))

typedef struct _ValaAttribute ValaAttribute;
typedef struct _ValaAttributeClass ValaAttributeClass;
typedef struct _ValaSymbolPrivate ValaSymbolPrivate;

#define VALA_TYPE_NAMESPACE (vala_namespace_get_type ())
#define VALA_NAMESPACE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_NAMESPACE, ValaNamespace))
#define VALA_NAMESPACE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_NAMESPACE, ValaNamespaceClass))
#define VALA_IS_NAMESPACE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_NAMESPACE))
#define VALA_IS_NAMESPACE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_NAMESPACE))
#define VALA_NAMESPACE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_NAMESPACE, ValaNamespaceClass))

typedef struct _ValaNamespace ValaNamespace;
typedef struct _ValaNamespaceClass ValaNamespaceClass;
typedef struct _ValaNamespacePrivate ValaNamespacePrivate;

#define VALA_TYPE_TYPESYMBOL (vala_typesymbol_get_type ())
#define VALA_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbol))
#define VALA_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))
#define VALA_IS_TYPESYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_TYPESYMBOL))
#define VALA_IS_TYPESYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_TYPESYMBOL))
#define VALA_TYPESYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_TYPESYMBOL, ValaTypeSymbolClass))

typedef struct _ValaTypeSymbol ValaTypeSymbol;
typedef struct _ValaTypeSymbolClass ValaTypeSymbolClass;

#define VALA_TYPE_OBJECT_TYPE_SYMBOL (vala_object_type_symbol_get_type ())
#define VALA_OBJECT_TYPE_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_OBJECT_TYPE_SYMBOL, ValaObjectTypeSymbol))
#define VALA_OBJECT_TYPE_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_OBJECT_TYPE_SYMBOL, ValaObjectTypeSymbolClass))
#define VALA_IS_OBJECT_TYPE_SYMBOL(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_OBJECT_TYPE_SYMBOL))
#define VALA_IS_OBJECT_TYPE_SYMBOL_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_OBJECT_TYPE_SYMBOL))
#define VALA_OBJECT_TYPE_SYMBOL_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_OBJECT_TYPE_SYMBOL, ValaObjectTypeSymbolClass))

typedef struct _ValaObjectTypeSymbol ValaObjectTypeSymbol;
typedef struct _ValaObjectTypeSymbolClass ValaObjectTypeSymbolClass;

#define VALA_TYPE_CLASS (vala_class_get_type ())
#define VALA_CLASS(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CLASS, ValaClass))
#define VALA_CLASS_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CLASS, ValaClassClass))
#define VALA_IS_CLASS(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CLASS))
#define VALA_IS_CLASS_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CLASS))
#define VALA_CLASS_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CLASS, ValaClassClass))

typedef struct _ValaClass ValaClass;
typedef struct _ValaClassClass ValaClassClass;

#define VALA_TYPE_INTERFACE (vala_interface_get_type ())
#define VALA_INTERFACE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_INTERFACE, ValaInterface))
#define VALA_INTERFACE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_INTERFACE, ValaInterfaceClass))
#define VALA_IS_INTERFACE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_INTERFACE))
#define VALA_IS_INTERFACE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_INTERFACE))
#define VALA_INTERFACE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_INTERFACE, ValaInterfaceClass))

typedef struct _ValaInterface ValaInterface;
typedef struct _ValaInterfaceClass ValaInterfaceClass;

#define VALA_TYPE_STRUCT (vala_struct_get_type ())
#define VALA_STRUCT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_STRUCT, ValaStruct))
#define VALA_STRUCT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_STRUCT, ValaStructClass))
#define VALA_IS_STRUCT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_STRUCT))
#define VALA_IS_STRUCT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_STRUCT))
#define VALA_STRUCT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_STRUCT, ValaStructClass))

typedef struct _ValaStruct ValaStruct;
typedef struct _ValaStructClass ValaStructClass;

#define VALA_TYPE_ENUM (vala_enum_get_type ())
#define VALA_ENUM(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ENUM, ValaEnum))
#define VALA_ENUM_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ENUM, ValaEnumClass))
#define VALA_IS_ENUM(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ENUM))
#define VALA_IS_ENUM_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ENUM))
#define VALA_ENUM_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ENUM, ValaEnumClass))

typedef struct _ValaEnum ValaEnum;
typedef struct _ValaEnumClass ValaEnumClass;

#define VALA_TYPE_ERROR_DOMAIN (vala_error_domain_get_type ())
#define VALA_ERROR_DOMAIN(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_ERROR_DOMAIN, ValaErrorDomain))
#define VALA_ERROR_DOMAIN_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_ERROR_DOMAIN, ValaErrorDomainClass))
#define VALA_IS_ERROR_DOMAIN(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_ERROR_DOMAIN))
#define VALA_IS_ERROR_DOMAIN_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_ERROR_DOMAIN))
#define VALA_ERROR_DOMAIN_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_ERROR_DOMAIN, ValaErrorDomainClass))

typedef struct _ValaErrorDomain ValaErrorDomain;
typedef struct _ValaErrorDomainClass ValaErrorDomainClass;

#define VALA_TYPE_DELEGATE (vala_delegate_get_type ())
#define VALA_DELEGATE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_DELEGATE, ValaDelegate))
#define VALA_DELEGATE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_DELEGATE, ValaDelegateClass))
#define VALA_IS_DELEGATE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_DELEGATE))
#define VALA_IS_DELEGATE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_DELEGATE))
#define VALA_DELEGATE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_DELEGATE, ValaDelegateClass))

typedef struct _ValaDelegate ValaDelegate;
typedef struct _ValaDelegateClass ValaDelegateClass;

#define VALA_TYPE_MEMBER (vala_member_get_type ())
#define VALA_MEMBER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_MEMBER, ValaMember))
#define VALA_MEMBER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_MEMBER, ValaMemberClass))
#define VALA_IS_MEMBER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_MEMBER))
#define VALA_IS_MEMBER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_MEMBER))
#define VALA_MEMBER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_MEMBER, ValaMemberClass))

typedef struct _ValaMember ValaMember;
typedef struct _ValaMemberClass ValaMemberClass;

#define VALA_TYPE_CONSTANT (vala_constant_get_type ())
#define VALA_CONSTANT(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CONSTANT, ValaConstant))
#define VALA_CONSTANT_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CONSTANT, ValaConstantClass))
#define VALA_IS_CONSTANT(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CONSTANT))
#define VALA_IS_CONSTANT_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CONSTANT))
#define VALA_CONSTANT_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CONSTANT, ValaConstantClass))

typedef struct _ValaConstant ValaConstant;
typedef struct _ValaConstantClass ValaConstantClass;

#define VALA_TYPE_FIELD (vala_field_get_type ())
#define VALA_FIELD(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_FIELD, ValaField))
#define VALA_FIELD_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_FIELD, ValaFieldClass))
#define VALA_IS_FIELD(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_FIELD))
#define VALA_IS_FIELD_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_FIELD))
#define VALA_FIELD_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_FIELD, ValaFieldClass))

typedef struct _ValaField ValaField;
typedef struct _ValaFieldClass ValaFieldClass;

#define VALA_TYPE_METHOD (vala_method_get_type ())
#define VALA_METHOD(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_METHOD, ValaMethod))
#define VALA_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_METHOD, ValaMethodClass))
#define VALA_IS_METHOD(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_METHOD))
#define VALA_IS_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_METHOD))
#define VALA_METHOD_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_METHOD, ValaMethodClass))

typedef struct _ValaMethod ValaMethod;
typedef struct _ValaMethodClass ValaMethodClass;

#define VALA_TYPE_SOURCE_REFERENCE (vala_source_reference_get_type ())
#define VALA_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReference))
#define VALA_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))
#define VALA_IS_SOURCE_REFERENCE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_IS_SOURCE_REFERENCE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SOURCE_REFERENCE))
#define VALA_SOURCE_REFERENCE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SOURCE_REFERENCE, ValaSourceReferenceClass))

typedef struct _ValaSourceReference ValaSourceReference;
typedef struct _ValaSourceReferenceClass ValaSourceReferenceClass;

#define VALA_TYPE_SYMBOL_ACCESSIBILITY (vala_symbol_accessibility_get_type ())

#define VALA_TYPE_SCOPE (vala_scope_get_type ())
#define VALA_SCOPE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_SCOPE, ValaScope))
#define VALA_SCOPE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_SCOPE, ValaScopeClass))
#define VALA_IS_SCOPE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_SCOPE))
#define VALA_IS_SCOPE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_SCOPE))
#define VALA_SCOPE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_SCOPE, ValaScopeClass))

typedef struct _ValaScope ValaScope;
typedef struct _ValaScopeClass ValaScopeClass;

#define TYPE_MEMBER_BINDING (member_binding_get_type ())

#define VALA_TYPE_CREATION_METHOD (vala_creation_method_get_type ())
#define VALA_CREATION_METHOD(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), VALA_TYPE_CREATION_METHOD, ValaCreationMethod))
#define VALA_CREATION_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), VALA_TYPE_CREATION_METHOD, ValaCreationMethodClass))
#define VALA_IS_CREATION_METHOD(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), VALA_TYPE_CREATION_METHOD))
#define VALA_IS_CREATION_METHOD_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), VALA_TYPE_CREATION_METHOD))
#define VALA_CREATION_METHOD_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), VALA_TYPE_CREATION_METHOD, ValaCreationMethodClass))

typedef struct _ValaCreationMethod ValaCreationMethod;
typedef struct _ValaCreationMethodClass ValaCreationMethodClass;

/**
 * Represents a part of the parsed source code.
 *
 * Code nodes get created by the parser and are used throughout the whole
 * compilation process.
 */
struct _ValaCodeNode {
	GTypeInstance parent_instance;
	volatile int ref_count;
	ValaCodeNodePrivate * priv;
	GList* attributes;
};

struct _ValaCodeNodeClass {
	GTypeClass parent_class;
	void (*finalize) (ValaCodeNode *self);
	void (*accept) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	void (*accept_children) (ValaCodeNode* self, ValaCodeVisitor* visitor);
	gboolean (*check) (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
	void (*replace_type) (ValaCodeNode* self, ValaDataType* old_type, ValaDataType* new_type);
	void (*replace_expression) (ValaCodeNode* self, ValaExpression* old_node, ValaExpression* new_node);
	char* (*to_string) (ValaCodeNode* self);
	void (*get_defined_variables) (ValaCodeNode* self, GeeCollection* collection);
	void (*get_used_variables) (ValaCodeNode* self, GeeCollection* collection);
};

/**
 * Represents a node in the symbol tree.
 */
struct _ValaSymbol {
	ValaCodeNode parent_instance;
	ValaSymbolPrivate * priv;
};

struct _ValaSymbolClass {
	ValaCodeNodeClass parent_class;
	char* (*get_cprefix) (ValaSymbol* self);
	char* (*get_lower_case_cname) (ValaSymbol* self, const char* infix);
	char* (*get_lower_case_cprefix) (ValaSymbol* self);
	GeeList* (*get_cheader_filenames) (ValaSymbol* self);
};

/**
 * Represents a namespace declaration in the source code.
 */
struct _ValaNamespace {
	ValaSymbol parent_instance;
	ValaNamespacePrivate * priv;
};

struct _ValaNamespaceClass {
	ValaSymbolClass parent_class;
};

struct _ValaNamespacePrivate {
	GeeList* classes;
	GeeList* interfaces;
	GeeList* structs;
	GeeList* enums;
	GeeList* error_domains;
	GeeList* delegates;
	GeeList* constants;
	GeeList* fields;
	GeeList* methods;
	GeeList* cprefixes;
	char* lower_case_cprefix;
	GeeList* cheader_filenames;
	GeeList* namespaces;
};

typedef enum  {
	VALA_SYMBOL_ACCESSIBILITY_PRIVATE,
	VALA_SYMBOL_ACCESSIBILITY_INTERNAL,
	VALA_SYMBOL_ACCESSIBILITY_PROTECTED,
	VALA_SYMBOL_ACCESSIBILITY_PUBLIC
} ValaSymbolAccessibility;

typedef enum  {
	MEMBER_BINDING_INSTANCE,
	MEMBER_BINDING_CLASS,
	MEMBER_BINDING_STATIC
} MemberBinding;



gpointer vala_code_node_ref (gpointer instance);
void vala_code_node_unref (gpointer instance);
GParamSpec* vala_param_spec_code_node (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_node (GValue* value, gpointer v_object);
gpointer vala_value_get_code_node (const GValue* value);
GType vala_code_node_get_type (void);
gpointer vala_code_visitor_ref (gpointer instance);
void vala_code_visitor_unref (gpointer instance);
GParamSpec* vala_param_spec_code_visitor (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_code_visitor (GValue* value, gpointer v_object);
gpointer vala_value_get_code_visitor (const GValue* value);
GType vala_code_visitor_get_type (void);
GType vala_semantic_analyzer_get_type (void);
GType vala_data_type_get_type (void);
GType vala_expression_get_type (void);
GType vala_symbol_get_type (void);
GType vala_local_variable_get_type (void);
GType vala_attribute_get_type (void);
GType vala_namespace_get_type (void);
GType vala_typesymbol_get_type (void);
GType vala_object_type_symbol_get_type (void);
GType vala_class_get_type (void);
GType vala_interface_get_type (void);
GType vala_struct_get_type (void);
GType vala_enum_get_type (void);
GType vala_error_domain_get_type (void);
GType vala_delegate_get_type (void);
GType vala_member_get_type (void);
GType vala_constant_get_type (void);
GType vala_field_get_type (void);
GType vala_method_get_type (void);
#define VALA_NAMESPACE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_NAMESPACE, ValaNamespacePrivate))
enum  {
	VALA_NAMESPACE_DUMMY_PROPERTY
};
gpointer vala_source_reference_ref (gpointer instance);
void vala_source_reference_unref (gpointer instance);
GParamSpec* vala_param_spec_source_reference (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_source_reference (GValue* value, gpointer v_object);
gpointer vala_value_get_source_reference (const GValue* value);
GType vala_source_reference_get_type (void);
ValaSymbol* vala_symbol_new (const char* name, ValaSourceReference* source_reference);
ValaSymbol* vala_symbol_construct (GType object_type, const char* name, ValaSourceReference* source_reference);
GType vala_symbol_accessibility_get_type (void);
void vala_symbol_set_access (ValaSymbol* self, ValaSymbolAccessibility value);
ValaNamespace* vala_namespace_new (const char* name, ValaSourceReference* source_reference);
ValaNamespace* vala_namespace_construct (GType object_type, const char* name, ValaSourceReference* source_reference);
ValaNamespace* vala_namespace_new (const char* name, ValaSourceReference* source_reference);
gpointer vala_scope_ref (gpointer instance);
void vala_scope_unref (gpointer instance);
GParamSpec* vala_param_spec_scope (const gchar* name, const gchar* nick, const gchar* blurb, GType object_type, GParamFlags flags);
void vala_value_set_scope (GValue* value, gpointer v_object);
gpointer vala_value_get_scope (const GValue* value);
GType vala_scope_get_type (void);
ValaScope* vala_symbol_get_scope (ValaSymbol* self);
ValaSymbol* vala_scope_lookup (ValaScope* self, const char* name);
const char* vala_symbol_get_name (ValaSymbol* self);
gboolean vala_symbol_get_external_package (ValaSymbol* self);
ValaSourceReference* vala_code_node_get_source_reference (ValaCodeNode* self);
void vala_code_node_set_source_reference (ValaCodeNode* self, ValaSourceReference* value);
GeeList* vala_namespace_get_namespaces (ValaNamespace* self);
void vala_namespace_add_namespace (ValaNamespace* self, ValaNamespace* ns);
GeeList* vala_namespace_get_classes (ValaNamespace* self);
void vala_namespace_add_class (ValaNamespace* self, ValaClass* cl);
GeeList* vala_namespace_get_structs (ValaNamespace* self);
void vala_namespace_add_struct (ValaNamespace* self, ValaStruct* st);
GeeList* vala_namespace_get_interfaces (ValaNamespace* self);
void vala_namespace_add_interface (ValaNamespace* self, ValaInterface* iface);
GeeList* vala_namespace_get_delegates (ValaNamespace* self);
void vala_namespace_add_delegate (ValaNamespace* self, ValaDelegate* d);
GeeList* vala_namespace_get_enums (ValaNamespace* self);
void vala_namespace_add_enum (ValaNamespace* self, ValaEnum* en);
GeeList* vala_namespace_get_error_domains (ValaNamespace* self);
void vala_namespace_add_error_domain (ValaNamespace* self, ValaErrorDomain* edomain);
GeeList* vala_namespace_get_constants (ValaNamespace* self);
void vala_namespace_add_constant (ValaNamespace* self, ValaConstant* constant);
GeeList* vala_namespace_get_fields (ValaNamespace* self);
void vala_namespace_add_field (ValaNamespace* self, ValaField* f);
GeeList* vala_namespace_get_methods (ValaNamespace* self);
void vala_namespace_add_method (ValaNamespace* self, ValaMethod* m);
void vala_scope_add (ValaScope* self, const char* name, ValaSymbol* sym);
ValaSymbolAccessibility vala_symbol_get_access (ValaSymbol* self);
void vala_scope_remove (ValaScope* self, const char* name);
void vala_namespace_remove_struct (ValaNamespace* self, ValaStruct* st);
GType member_binding_get_type (void);
MemberBinding vala_field_get_binding (ValaField* self);
void vala_report_error (ValaSourceReference* source, const char* message);
void vala_code_node_set_error (ValaCodeNode* self, gboolean value);
GType vala_creation_method_get_type (void);
MemberBinding vala_method_get_binding (ValaMethod* self);
void vala_code_visitor_visit_namespace (ValaCodeVisitor* self, ValaNamespace* ns);
static void vala_namespace_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
void vala_code_node_accept (ValaCodeNode* self, ValaCodeVisitor* visitor);
static void vala_namespace_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
ValaSymbol* vala_symbol_get_parent_symbol (ValaSymbol* self);
char* vala_symbol_get_cprefix (ValaSymbol* self);
static char* vala_namespace_real_get_cprefix (ValaSymbol* base);
GeeList* vala_namespace_get_cprefixes (ValaNamespace* self);
void vala_namespace_add_cprefix (ValaNamespace* self, const char* cprefix);
char* vala_symbol_get_lower_case_cprefix (ValaSymbol* self);
char* vala_symbol_camel_case_to_lower_case (const char* camel_case);
static char* vala_namespace_real_get_lower_case_cprefix (ValaSymbol* base);
void vala_namespace_set_lower_case_cprefix (ValaNamespace* self, const char* cprefix);
static GeeList* vala_namespace_real_get_cheader_filenames (ValaSymbol* base);
GeeList* vala_symbol_get_cheader_filenames (ValaSymbol* self);
char* vala_namespace_get_cheader_filename (ValaNamespace* self);
void vala_namespace_set_cheader_filename (ValaNamespace* self, const char* cheader_filename);
void vala_namespace_add_cheader_filename (ValaNamespace* self, const char* cheader_filename);
gboolean vala_attribute_has_argument (ValaAttribute* self, const char* name);
char* vala_attribute_get_string (ValaAttribute* self, const char* name);
static void vala_namespace_process_ccode_attribute (ValaNamespace* self, ValaAttribute* a);
const char* vala_attribute_get_name (ValaAttribute* self);
void vala_namespace_process_attributes (ValaNamespace* self);
gboolean vala_code_node_get_checked (ValaCodeNode* self);
gboolean vala_code_node_get_error (ValaCodeNode* self);
void vala_code_node_set_checked (ValaCodeNode* self, gboolean value);
gboolean vala_code_node_check (ValaCodeNode* self, ValaSemanticAnalyzer* analyzer);
static gboolean vala_namespace_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static gpointer vala_namespace_parent_class = NULL;
static void vala_namespace_finalize (ValaCodeNode* obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static gint _vala_array_length (gpointer array);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Creates a new namespace.
 *
 * @param name             namespace name
 * @param source_reference reference to source code
 * @return                 newly created namespace
 */
ValaNamespace* vala_namespace_construct (GType object_type, const char* name, ValaSourceReference* source_reference) {
	ValaNamespace* self;
	self = (ValaNamespace*) vala_symbol_construct (object_type, name, source_reference);
	vala_symbol_set_access ((ValaSymbol*) self, VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	return self;
}


ValaNamespace* vala_namespace_new (const char* name, ValaSourceReference* source_reference) {
	return vala_namespace_construct (VALA_TYPE_NAMESPACE, name, source_reference);
}


/**
 * Adds the specified namespace to this source file.
 *
 * @param ns a namespace
 */
void vala_namespace_add_namespace (ValaNamespace* self, ValaNamespace* ns) {
	ValaSymbol* _tmp0;
	gboolean _tmp1;
	g_return_if_fail (self != NULL);
	g_return_if_fail (ns != NULL);
	_tmp0 = NULL;
	if ((_tmp1 = VALA_IS_NAMESPACE (_tmp0 = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) ns))), (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL)), _tmp1)) {
		ValaNamespace* old_ns;
		gboolean _tmp2;
		/* merge if namespace already exists*/
		old_ns = VALA_NAMESPACE (vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) ns)));
		_tmp2 = FALSE;
		if (vala_symbol_get_external_package ((ValaSymbol*) old_ns)) {
			_tmp2 = !vala_symbol_get_external_package ((ValaSymbol*) ns);
		} else {
			_tmp2 = FALSE;
		}
		if (_tmp2) {
			vala_code_node_set_source_reference ((ValaCodeNode*) old_ns, vala_code_node_get_source_reference ((ValaCodeNode*) ns));
		}
		{
			GeeList* _tmp3;
			GeeIterator* _tmp4;
			GeeIterator* _sub_ns_it;
			_tmp3 = NULL;
			_tmp4 = NULL;
			_sub_ns_it = (_tmp4 = gee_iterable_iterator ((GeeIterable*) (_tmp3 = vala_namespace_get_namespaces (ns))), (_tmp3 == NULL) ? NULL : (_tmp3 = (gee_collection_object_unref (_tmp3), NULL)), _tmp4);
			while (gee_iterator_next (_sub_ns_it)) {
				ValaNamespace* sub_ns;
				sub_ns = (ValaNamespace*) gee_iterator_get (_sub_ns_it);
				vala_namespace_add_namespace (old_ns, sub_ns);
				(sub_ns == NULL) ? NULL : (sub_ns = (vala_code_node_unref (sub_ns), NULL));
			}
			(_sub_ns_it == NULL) ? NULL : (_sub_ns_it = (gee_collection_object_unref (_sub_ns_it), NULL));
		}
		{
			GeeList* _tmp5;
			GeeIterator* _tmp6;
			GeeIterator* _cl_it;
			_tmp5 = NULL;
			_tmp6 = NULL;
			_cl_it = (_tmp6 = gee_iterable_iterator ((GeeIterable*) (_tmp5 = vala_namespace_get_classes (ns))), (_tmp5 == NULL) ? NULL : (_tmp5 = (gee_collection_object_unref (_tmp5), NULL)), _tmp6);
			while (gee_iterator_next (_cl_it)) {
				ValaClass* cl;
				cl = (ValaClass*) gee_iterator_get (_cl_it);
				vala_namespace_add_class (old_ns, cl);
				(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
			}
			(_cl_it == NULL) ? NULL : (_cl_it = (gee_collection_object_unref (_cl_it), NULL));
		}
		{
			GeeList* _tmp7;
			GeeIterator* _tmp8;
			GeeIterator* _st_it;
			_tmp7 = NULL;
			_tmp8 = NULL;
			_st_it = (_tmp8 = gee_iterable_iterator ((GeeIterable*) (_tmp7 = vala_namespace_get_structs (ns))), (_tmp7 == NULL) ? NULL : (_tmp7 = (gee_collection_object_unref (_tmp7), NULL)), _tmp8);
			while (gee_iterator_next (_st_it)) {
				ValaStruct* st;
				st = (ValaStruct*) gee_iterator_get (_st_it);
				vala_namespace_add_struct (old_ns, st);
				(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
			}
			(_st_it == NULL) ? NULL : (_st_it = (gee_collection_object_unref (_st_it), NULL));
		}
		{
			GeeList* _tmp9;
			GeeIterator* _tmp10;
			GeeIterator* _iface_it;
			_tmp9 = NULL;
			_tmp10 = NULL;
			_iface_it = (_tmp10 = gee_iterable_iterator ((GeeIterable*) (_tmp9 = vala_namespace_get_interfaces (ns))), (_tmp9 == NULL) ? NULL : (_tmp9 = (gee_collection_object_unref (_tmp9), NULL)), _tmp10);
			while (gee_iterator_next (_iface_it)) {
				ValaInterface* iface;
				iface = (ValaInterface*) gee_iterator_get (_iface_it);
				vala_namespace_add_interface (old_ns, iface);
				(iface == NULL) ? NULL : (iface = (vala_code_node_unref (iface), NULL));
			}
			(_iface_it == NULL) ? NULL : (_iface_it = (gee_collection_object_unref (_iface_it), NULL));
		}
		{
			GeeList* _tmp11;
			GeeIterator* _tmp12;
			GeeIterator* _d_it;
			_tmp11 = NULL;
			_tmp12 = NULL;
			_d_it = (_tmp12 = gee_iterable_iterator ((GeeIterable*) (_tmp11 = vala_namespace_get_delegates (ns))), (_tmp11 == NULL) ? NULL : (_tmp11 = (gee_collection_object_unref (_tmp11), NULL)), _tmp12);
			while (gee_iterator_next (_d_it)) {
				ValaDelegate* d;
				d = (ValaDelegate*) gee_iterator_get (_d_it);
				vala_namespace_add_delegate (old_ns, d);
				(d == NULL) ? NULL : (d = (vala_code_node_unref (d), NULL));
			}
			(_d_it == NULL) ? NULL : (_d_it = (gee_collection_object_unref (_d_it), NULL));
		}
		{
			GeeList* _tmp13;
			GeeIterator* _tmp14;
			GeeIterator* _en_it;
			_tmp13 = NULL;
			_tmp14 = NULL;
			_en_it = (_tmp14 = gee_iterable_iterator ((GeeIterable*) (_tmp13 = vala_namespace_get_enums (ns))), (_tmp13 == NULL) ? NULL : (_tmp13 = (gee_collection_object_unref (_tmp13), NULL)), _tmp14);
			while (gee_iterator_next (_en_it)) {
				ValaEnum* en;
				en = (ValaEnum*) gee_iterator_get (_en_it);
				vala_namespace_add_enum (old_ns, en);
				(en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL));
			}
			(_en_it == NULL) ? NULL : (_en_it = (gee_collection_object_unref (_en_it), NULL));
		}
		{
			GeeList* _tmp15;
			GeeIterator* _tmp16;
			GeeIterator* _ed_it;
			_tmp15 = NULL;
			_tmp16 = NULL;
			_ed_it = (_tmp16 = gee_iterable_iterator ((GeeIterable*) (_tmp15 = vala_namespace_get_error_domains (ns))), (_tmp15 == NULL) ? NULL : (_tmp15 = (gee_collection_object_unref (_tmp15), NULL)), _tmp16);
			while (gee_iterator_next (_ed_it)) {
				ValaErrorDomain* ed;
				ed = (ValaErrorDomain*) gee_iterator_get (_ed_it);
				vala_namespace_add_error_domain (old_ns, ed);
				(ed == NULL) ? NULL : (ed = (vala_code_node_unref (ed), NULL));
			}
			(_ed_it == NULL) ? NULL : (_ed_it = (gee_collection_object_unref (_ed_it), NULL));
		}
		{
			GeeList* _tmp17;
			GeeIterator* _tmp18;
			GeeIterator* _c_it;
			_tmp17 = NULL;
			_tmp18 = NULL;
			_c_it = (_tmp18 = gee_iterable_iterator ((GeeIterable*) (_tmp17 = vala_namespace_get_constants (ns))), (_tmp17 == NULL) ? NULL : (_tmp17 = (gee_collection_object_unref (_tmp17), NULL)), _tmp18);
			while (gee_iterator_next (_c_it)) {
				ValaConstant* c;
				c = (ValaConstant*) gee_iterator_get (_c_it);
				vala_namespace_add_constant (old_ns, c);
				(c == NULL) ? NULL : (c = (vala_code_node_unref (c), NULL));
			}
			(_c_it == NULL) ? NULL : (_c_it = (gee_collection_object_unref (_c_it), NULL));
		}
		{
			GeeList* _tmp19;
			GeeIterator* _tmp20;
			GeeIterator* _f_it;
			_tmp19 = NULL;
			_tmp20 = NULL;
			_f_it = (_tmp20 = gee_iterable_iterator ((GeeIterable*) (_tmp19 = vala_namespace_get_fields (ns))), (_tmp19 == NULL) ? NULL : (_tmp19 = (gee_collection_object_unref (_tmp19), NULL)), _tmp20);
			while (gee_iterator_next (_f_it)) {
				ValaField* f;
				f = (ValaField*) gee_iterator_get (_f_it);
				vala_namespace_add_field (old_ns, f);
				(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
			}
			(_f_it == NULL) ? NULL : (_f_it = (gee_collection_object_unref (_f_it), NULL));
		}
		{
			GeeList* _tmp21;
			GeeIterator* _tmp22;
			GeeIterator* _m_it;
			_tmp21 = NULL;
			_tmp22 = NULL;
			_m_it = (_tmp22 = gee_iterable_iterator ((GeeIterable*) (_tmp21 = vala_namespace_get_methods (ns))), (_tmp21 == NULL) ? NULL : (_tmp21 = (gee_collection_object_unref (_tmp21), NULL)), _tmp22);
			while (gee_iterator_next (_m_it)) {
				ValaMethod* m;
				m = (ValaMethod*) gee_iterator_get (_m_it);
				vala_namespace_add_method (old_ns, m);
				(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
			}
			(_m_it == NULL) ? NULL : (_m_it = (gee_collection_object_unref (_m_it), NULL));
		}
		(old_ns == NULL) ? NULL : (old_ns = (vala_code_node_unref (old_ns), NULL));
	} else {
		gee_collection_add ((GeeCollection*) self->priv->namespaces, ns);
		vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) ns), (ValaSymbol*) ns);
	}
}


/**
 * Returns a copy of the list of namespaces.
 *
 * @return namespace list
 */
GeeList* vala_namespace_get_namespaces (ValaNamespace* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_NAMESPACE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->namespaces);
}


/**
 * Adds the specified class to this namespace.
 *
 * @param cl a class
 */
void vala_namespace_add_class (ValaNamespace* self, ValaClass* cl) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (cl != NULL);
	/* namespaces do not support private memebers*/
	if (vala_symbol_get_access ((ValaSymbol*) cl) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
		vala_symbol_set_access ((ValaSymbol*) cl, VALA_SYMBOL_ACCESSIBILITY_INTERNAL);
	}
	gee_collection_add ((GeeCollection*) self->priv->classes, cl);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) cl), (ValaSymbol*) cl);
}


/**
 * Adds the specified interface to this namespace.
 *
 * @param iface an interface
 */
void vala_namespace_add_interface (ValaNamespace* self, ValaInterface* iface) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (iface != NULL);
	/* namespaces do not support private memebers*/
	if (vala_symbol_get_access ((ValaSymbol*) iface) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
		vala_symbol_set_access ((ValaSymbol*) iface, VALA_SYMBOL_ACCESSIBILITY_INTERNAL);
	}
	gee_collection_add ((GeeCollection*) self->priv->interfaces, iface);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) iface), (ValaSymbol*) iface);
}


/**
 * Adds the specified struct to this namespace.
 *
 * @param st a struct
 */
void vala_namespace_add_struct (ValaNamespace* self, ValaStruct* st) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (st != NULL);
	/* namespaces do not support private memebers*/
	if (vala_symbol_get_access ((ValaSymbol*) st) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
		vala_symbol_set_access ((ValaSymbol*) st, VALA_SYMBOL_ACCESSIBILITY_INTERNAL);
	}
	gee_collection_add ((GeeCollection*) self->priv->structs, st);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) st), (ValaSymbol*) st);
}


/**
 * Removes the specified struct from this namespace.
 *
 * @param st a struct
 */
void vala_namespace_remove_struct (ValaNamespace* self, ValaStruct* st) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (st != NULL);
	gee_collection_remove ((GeeCollection*) self->priv->structs, st);
	vala_scope_remove (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) st));
}


/**
 * Adds the specified enum to this namespace.
 *
 * @param en an enum
 */
void vala_namespace_add_enum (ValaNamespace* self, ValaEnum* en) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (en != NULL);
	/* namespaces do not support private memebers*/
	if (vala_symbol_get_access ((ValaSymbol*) en) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
		vala_symbol_set_access ((ValaSymbol*) en, VALA_SYMBOL_ACCESSIBILITY_INTERNAL);
	}
	gee_collection_add ((GeeCollection*) self->priv->enums, en);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) en), (ValaSymbol*) en);
}


/**
 * Adds the specified error domain to this namespace.
 *
 * @param edomain an error domain
 */
void vala_namespace_add_error_domain (ValaNamespace* self, ValaErrorDomain* edomain) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (edomain != NULL);
	/* namespaces do not support private memebers*/
	if (vala_symbol_get_access ((ValaSymbol*) edomain) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
		vala_symbol_set_access ((ValaSymbol*) edomain, VALA_SYMBOL_ACCESSIBILITY_INTERNAL);
	}
	gee_collection_add ((GeeCollection*) self->priv->error_domains, edomain);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) edomain), (ValaSymbol*) edomain);
}


/**
 * Adds the specified delegate to this namespace.
 *
 * @param d a delegate
 */
void vala_namespace_add_delegate (ValaNamespace* self, ValaDelegate* d) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (d != NULL);
	/* namespaces do not support private memebers*/
	if (vala_symbol_get_access ((ValaSymbol*) d) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
		vala_symbol_set_access ((ValaSymbol*) d, VALA_SYMBOL_ACCESSIBILITY_INTERNAL);
	}
	gee_collection_add ((GeeCollection*) self->priv->delegates, d);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) d), (ValaSymbol*) d);
}


/**
 * Returns a copy of the list of structs.
 *
 * @return struct list
 */
GeeList* vala_namespace_get_structs (ValaNamespace* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_STRUCT, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->structs);
}


/**
 * Returns a copy of the list of classes.
 *
 * @return class list
 */
GeeList* vala_namespace_get_classes (ValaNamespace* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_CLASS, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->classes);
}


/**
 * Returns a copy of the list of interfaces.
 *
 * @return interface list
 */
GeeList* vala_namespace_get_interfaces (ValaNamespace* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_INTERFACE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->interfaces);
}


/**
 * Returns a copy of the list of enums.
 *
 * @return enum list
 */
GeeList* vala_namespace_get_enums (ValaNamespace* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_ENUM, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->enums);
}


/**
 * Returns a copy of the list of error domains.
 *
 * @return error domain list
 */
GeeList* vala_namespace_get_error_domains (ValaNamespace* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_ERROR_DOMAIN, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->error_domains);
}


/**
 * Returns a copy of the list of fields.
 *
 * @return field list
 */
GeeList* vala_namespace_get_fields (ValaNamespace* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_FIELD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->fields);
}


/**
 * Returns a copy of the list of constants.
 *
 * @return constant list
 */
GeeList* vala_namespace_get_constants (ValaNamespace* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_CONSTANT, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->constants);
}


/**
 * Returns a copy of the list of delegates.
 *
 * @return delegate list
 */
GeeList* vala_namespace_get_delegates (ValaNamespace* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_DELEGATE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->delegates);
}


/**
 * Returns a copy of the list of methods.
 *
 * @return method list
 */
GeeList* vala_namespace_get_methods (ValaNamespace* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_METHOD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->methods);
}


/**
 * Adds the specified constant to this namespace.
 *
 * @param constant a constant
 */
void vala_namespace_add_constant (ValaNamespace* self, ValaConstant* constant) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (constant != NULL);
	/* namespaces do not support private memebers*/
	if (vala_symbol_get_access ((ValaSymbol*) constant) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
		vala_symbol_set_access ((ValaSymbol*) constant, VALA_SYMBOL_ACCESSIBILITY_INTERNAL);
	}
	gee_collection_add ((GeeCollection*) self->priv->constants, constant);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) constant), (ValaSymbol*) constant);
}


/**
 * Adds the specified field to this namespace.
 *
 * @param f a field
 */
void vala_namespace_add_field (ValaNamespace* self, ValaField* f) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (f != NULL);
	/* namespaces do not support private memebers*/
	if (vala_symbol_get_access ((ValaSymbol*) f) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
		vala_symbol_set_access ((ValaSymbol*) f, VALA_SYMBOL_ACCESSIBILITY_INTERNAL);
	}
	if (vala_field_get_binding (f) == MEMBER_BINDING_INSTANCE) {
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) f), "instance members are not allowed outside of data types");
		vala_code_node_set_error ((ValaCodeNode*) f, TRUE);
		return;
	} else {
		if (vala_field_get_binding (f) == MEMBER_BINDING_CLASS) {
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) f), "class members are not allowed outside of classes");
			vala_code_node_set_error ((ValaCodeNode*) f, TRUE);
			return;
		}
	}
	gee_collection_add ((GeeCollection*) self->priv->fields, f);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) f), (ValaSymbol*) f);
}


/**
 * Adds the specified method to this namespace.
 *
 * @param m a method
 */
void vala_namespace_add_method (ValaNamespace* self, ValaMethod* m) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (m != NULL);
	/* namespaces do not support private memebers*/
	if (vala_symbol_get_access ((ValaSymbol*) m) == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
		vala_symbol_set_access ((ValaSymbol*) m, VALA_SYMBOL_ACCESSIBILITY_INTERNAL);
	}
	if (VALA_IS_CREATION_METHOD (m)) {
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) m), "construction methods may only be declared within classes and structs");
		vala_code_node_set_error ((ValaCodeNode*) m, TRUE);
		return;
	}
	if (vala_method_get_binding (m) == MEMBER_BINDING_INSTANCE) {
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) m), "instance members are not allowed outside of data types");
		vala_code_node_set_error ((ValaCodeNode*) m, TRUE);
		return;
	} else {
		if (vala_method_get_binding (m) == MEMBER_BINDING_CLASS) {
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) m), "class members are not allowed outside of classes");
			vala_code_node_set_error ((ValaCodeNode*) m, TRUE);
			return;
		}
	}
	gee_collection_add ((GeeCollection*) self->priv->methods, m);
	vala_scope_add (vala_symbol_get_scope ((ValaSymbol*) self), vala_symbol_get_name ((ValaSymbol*) m), (ValaSymbol*) m);
}


static void vala_namespace_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaNamespace * self;
	self = (ValaNamespace*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_namespace (visitor, self);
}


static void vala_namespace_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaNamespace * self;
	self = (ValaNamespace*) base;
	g_return_if_fail (visitor != NULL);
	{
		GeeIterator* _ns_it;
		_ns_it = gee_iterable_iterator ((GeeIterable*) self->priv->namespaces);
		while (gee_iterator_next (_ns_it)) {
			ValaNamespace* ns;
			ns = (ValaNamespace*) gee_iterator_get (_ns_it);
			vala_code_node_accept ((ValaCodeNode*) ns, visitor);
			(ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL));
		}
		(_ns_it == NULL) ? NULL : (_ns_it = (gee_collection_object_unref (_ns_it), NULL));
	}
	/* process enums first to avoid order problems in C code */
	{
		GeeIterator* _en_it;
		/* process enums first to avoid order problems in C code */
		_en_it = gee_iterable_iterator ((GeeIterable*) self->priv->enums);
		/* process enums first to avoid order problems in C code */
		while (gee_iterator_next (_en_it)) {
			ValaEnum* en;
			/* process enums first to avoid order problems in C code */
			en = (ValaEnum*) gee_iterator_get (_en_it);
			vala_code_node_accept ((ValaCodeNode*) en, visitor);
			(en == NULL) ? NULL : (en = (vala_code_node_unref (en), NULL));
		}
		(_en_it == NULL) ? NULL : (_en_it = (gee_collection_object_unref (_en_it), NULL));
	}
	{
		GeeIterator* _edomain_it;
		_edomain_it = gee_iterable_iterator ((GeeIterable*) self->priv->error_domains);
		while (gee_iterator_next (_edomain_it)) {
			ValaErrorDomain* edomain;
			edomain = (ValaErrorDomain*) gee_iterator_get (_edomain_it);
			vala_code_node_accept ((ValaCodeNode*) edomain, visitor);
			(edomain == NULL) ? NULL : (edomain = (vala_code_node_unref (edomain), NULL));
		}
		(_edomain_it == NULL) ? NULL : (_edomain_it = (gee_collection_object_unref (_edomain_it), NULL));
	}
	{
		GeeIterator* _cl_it;
		_cl_it = gee_iterable_iterator ((GeeIterable*) self->priv->classes);
		while (gee_iterator_next (_cl_it)) {
			ValaClass* cl;
			cl = (ValaClass*) gee_iterator_get (_cl_it);
			vala_code_node_accept ((ValaCodeNode*) cl, visitor);
			(cl == NULL) ? NULL : (cl = (vala_code_node_unref (cl), NULL));
		}
		(_cl_it == NULL) ? NULL : (_cl_it = (gee_collection_object_unref (_cl_it), NULL));
	}
	{
		GeeIterator* _iface_it;
		_iface_it = gee_iterable_iterator ((GeeIterable*) self->priv->interfaces);
		while (gee_iterator_next (_iface_it)) {
			ValaInterface* iface;
			iface = (ValaInterface*) gee_iterator_get (_iface_it);
			vala_code_node_accept ((ValaCodeNode*) iface, visitor);
			(iface == NULL) ? NULL : (iface = (vala_code_node_unref (iface), NULL));
		}
		(_iface_it == NULL) ? NULL : (_iface_it = (gee_collection_object_unref (_iface_it), NULL));
	}
	{
		GeeIterator* _st_it;
		_st_it = gee_iterable_iterator ((GeeIterable*) self->priv->structs);
		while (gee_iterator_next (_st_it)) {
			ValaStruct* st;
			st = (ValaStruct*) gee_iterator_get (_st_it);
			vala_code_node_accept ((ValaCodeNode*) st, visitor);
			(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
		}
		(_st_it == NULL) ? NULL : (_st_it = (gee_collection_object_unref (_st_it), NULL));
	}
	{
		GeeIterator* _d_it;
		_d_it = gee_iterable_iterator ((GeeIterable*) self->priv->delegates);
		while (gee_iterator_next (_d_it)) {
			ValaDelegate* d;
			d = (ValaDelegate*) gee_iterator_get (_d_it);
			vala_code_node_accept ((ValaCodeNode*) d, visitor);
			(d == NULL) ? NULL : (d = (vala_code_node_unref (d), NULL));
		}
		(_d_it == NULL) ? NULL : (_d_it = (gee_collection_object_unref (_d_it), NULL));
	}
	{
		GeeIterator* _c_it;
		_c_it = gee_iterable_iterator ((GeeIterable*) self->priv->constants);
		while (gee_iterator_next (_c_it)) {
			ValaConstant* c;
			c = (ValaConstant*) gee_iterator_get (_c_it);
			vala_code_node_accept ((ValaCodeNode*) c, visitor);
			(c == NULL) ? NULL : (c = (vala_code_node_unref (c), NULL));
		}
		(_c_it == NULL) ? NULL : (_c_it = (gee_collection_object_unref (_c_it), NULL));
	}
	{
		GeeIterator* _f_it;
		_f_it = gee_iterable_iterator ((GeeIterable*) self->priv->fields);
		while (gee_iterator_next (_f_it)) {
			ValaField* f;
			f = (ValaField*) gee_iterator_get (_f_it);
			vala_code_node_accept ((ValaCodeNode*) f, visitor);
			(f == NULL) ? NULL : (f = (vala_code_node_unref (f), NULL));
		}
		(_f_it == NULL) ? NULL : (_f_it = (gee_collection_object_unref (_f_it), NULL));
	}
	{
		GeeIterator* _m_it;
		_m_it = gee_iterable_iterator ((GeeIterable*) self->priv->methods);
		while (gee_iterator_next (_m_it)) {
			ValaMethod* m;
			m = (ValaMethod*) gee_iterator_get (_m_it);
			vala_code_node_accept ((ValaCodeNode*) m, visitor);
			(m == NULL) ? NULL : (m = (vala_code_node_unref (m), NULL));
		}
		(_m_it == NULL) ? NULL : (_m_it = (gee_collection_object_unref (_m_it), NULL));
	}
}


static char* vala_namespace_real_get_cprefix (ValaSymbol* base) {
	ValaNamespace * self;
	self = (ValaNamespace*) base;
	if (gee_collection_get_size ((GeeCollection*) self->priv->cprefixes) > 0) {
		return (char*) gee_list_get (self->priv->cprefixes, 0);
	} else {
		if (NULL != vala_symbol_get_name ((ValaSymbol*) self)) {
			char* parent_prefix;
			char* _tmp3;
			parent_prefix = NULL;
			if (vala_symbol_get_parent_symbol ((ValaSymbol*) self) == NULL) {
				char* _tmp1;
				_tmp1 = NULL;
				parent_prefix = (_tmp1 = g_strdup (""), parent_prefix = (g_free (parent_prefix), NULL), _tmp1);
			} else {
				char* _tmp2;
				_tmp2 = NULL;
				parent_prefix = (_tmp2 = vala_symbol_get_cprefix (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), parent_prefix = (g_free (parent_prefix), NULL), _tmp2);
			}
			_tmp3 = NULL;
			return (_tmp3 = g_strconcat (parent_prefix, vala_symbol_get_name ((ValaSymbol*) self), NULL), parent_prefix = (g_free (parent_prefix), NULL), _tmp3);
		} else {
			return g_strdup ("");
		}
	}
}


GeeList* vala_namespace_get_cprefixes (ValaNamespace* self) {
	gboolean _tmp0;
	GeeList* _tmp1;
	g_return_val_if_fail (self != NULL, NULL);
	_tmp0 = FALSE;
	if (0 == gee_collection_get_size ((GeeCollection*) self->priv->cprefixes)) {
		_tmp0 = NULL != vala_symbol_get_name ((ValaSymbol*) self);
	} else {
		_tmp0 = FALSE;
	}
	if (_tmp0) {
		gee_collection_add ((GeeCollection*) self->priv->cprefixes, vala_symbol_get_name ((ValaSymbol*) self));
	}
	_tmp1 = NULL;
	return (_tmp1 = self->priv->cprefixes, (_tmp1 == NULL) ? NULL : gee_collection_object_ref (_tmp1));
}


/**
 * Adds a camel case string to be prepended to the name of members of
 * this namespace when used in C code.
 *
 * @param cprefixes the camel case prefixes used in C code
 */
void vala_namespace_add_cprefix (ValaNamespace* self, const char* cprefix) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (cprefix != NULL);
	g_return_if_fail (g_utf8_strlen (cprefix, -1) >= 1);
	gee_collection_add ((GeeCollection*) self->priv->cprefixes, cprefix);
}


/**
 * Returns the lower case string to be prepended to the name of members
 * of this namespace when used in C code.
 *
 * @return the lower case prefix to be used in C code
 */
static char* vala_namespace_real_get_lower_case_cprefix (ValaSymbol* base) {
	ValaNamespace * self;
	const char* _tmp5;
	self = (ValaNamespace*) base;
	if (self->priv->lower_case_cprefix == NULL) {
		if (vala_symbol_get_name ((ValaSymbol*) self) == NULL) {
			char* _tmp0;
			_tmp0 = NULL;
			self->priv->lower_case_cprefix = (_tmp0 = g_strdup (""), self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL), _tmp0);
		} else {
			char* parent_prefix;
			char* _tmp4;
			char* _tmp3;
			parent_prefix = NULL;
			if (vala_symbol_get_parent_symbol ((ValaSymbol*) self) == NULL) {
				char* _tmp1;
				_tmp1 = NULL;
				parent_prefix = (_tmp1 = g_strdup (""), parent_prefix = (g_free (parent_prefix), NULL), _tmp1);
			} else {
				char* _tmp2;
				_tmp2 = NULL;
				parent_prefix = (_tmp2 = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol ((ValaSymbol*) self)), parent_prefix = (g_free (parent_prefix), NULL), _tmp2);
			}
			_tmp4 = NULL;
			_tmp3 = NULL;
			self->priv->lower_case_cprefix = (_tmp4 = g_strdup_printf ("%s%s_", parent_prefix, _tmp3 = vala_symbol_camel_case_to_lower_case (vala_symbol_get_name ((ValaSymbol*) self))), self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL), _tmp4);
			_tmp3 = (g_free (_tmp3), NULL);
			parent_prefix = (g_free (parent_prefix), NULL);
		}
	}
	_tmp5 = NULL;
	return (_tmp5 = self->priv->lower_case_cprefix, (_tmp5 == NULL) ? NULL : g_strdup (_tmp5));
}


/**
 * Sets the lower case string to be prepended to the name of members of
 * this namespace when used in C code.
 *
 * @param cprefix the lower case prefix to be used in C code
 */
void vala_namespace_set_lower_case_cprefix (ValaNamespace* self, const char* cprefix) {
	char* _tmp1;
	const char* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (cprefix != NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->lower_case_cprefix = (_tmp1 = (_tmp0 = cprefix, (_tmp0 == NULL) ? NULL : g_strdup (_tmp0)), self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL), _tmp1);
}


static GeeList* vala_namespace_real_get_cheader_filenames (ValaSymbol* base) {
	ValaNamespace * self;
	self = (ValaNamespace*) base;
	return (GeeList*) gee_read_only_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, self->priv->cheader_filenames);
}


/**
 * Returns the C header filename of this namespace.
 *
 * @return header filename
 */
char* vala_namespace_get_cheader_filename (ValaNamespace* self) {
	GString* s;
	gboolean first;
	const char* _tmp2;
	char* _tmp3;
	g_return_val_if_fail (self != NULL, NULL);
	s = g_string_new ("");
	first = TRUE;
	{
		GeeList* _tmp0;
		GeeIterator* _tmp1;
		GeeIterator* _cheader_filename_it;
		_tmp0 = NULL;
		_tmp1 = NULL;
		_cheader_filename_it = (_tmp1 = gee_iterable_iterator ((GeeIterable*) (_tmp0 = vala_symbol_get_cheader_filenames ((ValaSymbol*) self))), (_tmp0 == NULL) ? NULL : (_tmp0 = (gee_collection_object_unref (_tmp0), NULL)), _tmp1);
		while (gee_iterator_next (_cheader_filename_it)) {
			char* cheader_filename;
			cheader_filename = (char*) gee_iterator_get (_cheader_filename_it);
			if (first) {
				first = FALSE;
			} else {
				g_string_append_c (s, ',');
			}
			g_string_append (s, cheader_filename);
			cheader_filename = (g_free (cheader_filename), NULL);
		}
		(_cheader_filename_it == NULL) ? NULL : (_cheader_filename_it = (gee_collection_object_unref (_cheader_filename_it), NULL));
	}
	_tmp2 = NULL;
	_tmp3 = NULL;
	return (_tmp3 = (_tmp2 = s->str, (_tmp2 == NULL) ? NULL : g_strdup (_tmp2)), (s == NULL) ? NULL : (s = (g_string_free (s, TRUE), NULL)), _tmp3);
}


/**
 * Sets the C header filename of this namespace to the specified
 * filename.
 *
 * @param cheader_filename header filename
 */
void vala_namespace_set_cheader_filename (ValaNamespace* self, const char* cheader_filename) {
	GeeList* _tmp0;
	g_return_if_fail (self != NULL);
	g_return_if_fail (cheader_filename != NULL);
	_tmp0 = NULL;
	self->priv->cheader_filenames = (_tmp0 = (GeeList*) gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_direct_equal), (self->priv->cheader_filenames == NULL) ? NULL : (self->priv->cheader_filenames = (gee_collection_object_unref (self->priv->cheader_filenames), NULL)), _tmp0);
	gee_collection_add ((GeeCollection*) self->priv->cheader_filenames, cheader_filename);
}


/**
 * Adds a C header filename of this namespace.
 *
 * @param cheader_filename header filename
 */
void vala_namespace_add_cheader_filename (ValaNamespace* self, const char* cheader_filename) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (cheader_filename != NULL);
	gee_collection_add ((GeeCollection*) self->priv->cheader_filenames, cheader_filename);
}


static void vala_namespace_process_ccode_attribute (ValaNamespace* self, ValaAttribute* a) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (a != NULL);
	if (vala_attribute_has_argument (a, "cprefix")) {
		{
			char** _tmp1;
			char* _tmp0;
			char** _tmp2;
			char** name_collection;
			int name_collection_length1;
			int name_it;
			_tmp1 = NULL;
			_tmp0 = NULL;
			_tmp2 = NULL;
			name_collection = (_tmp2 = _tmp1 = g_strsplit (_tmp0 = vala_attribute_get_string (a, "cprefix"), ",", 0), _tmp0 = (g_free (_tmp0), NULL), _tmp2);
			name_collection_length1 = _vala_array_length (_tmp1);
			for (name_it = 0; name_it < _vala_array_length (_tmp1); name_it = name_it + 1) {
				const char* _tmp3;
				char* name;
				_tmp3 = NULL;
				name = (_tmp3 = name_collection[name_it], (_tmp3 == NULL) ? NULL : g_strdup (_tmp3));
				{
					vala_namespace_add_cprefix (self, name);
					name = (g_free (name), NULL);
				}
			}
			name_collection = (_vala_array_free (name_collection, name_collection_length1, (GDestroyNotify) g_free), NULL);
		}
	}
	if (vala_attribute_has_argument (a, "lower_case_cprefix")) {
		char* _tmp4;
		_tmp4 = NULL;
		vala_namespace_set_lower_case_cprefix (self, _tmp4 = vala_attribute_get_string (a, "lower_case_cprefix"));
		_tmp4 = (g_free (_tmp4), NULL);
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** _tmp5;
			char** filename_collection;
			int filename_collection_length1;
			int filename_it;
			_tmp5 = NULL;
			filename_collection = _tmp5 = g_strsplit (val, ",", 0);
			filename_collection_length1 = _vala_array_length (_tmp5);
			for (filename_it = 0; filename_it < _vala_array_length (_tmp5); filename_it = filename_it + 1) {
				const char* _tmp6;
				char* filename;
				_tmp6 = NULL;
				filename = (_tmp6 = filename_collection[filename_it], (_tmp6 == NULL) ? NULL : g_strdup (_tmp6));
				{
					gee_collection_add ((GeeCollection*) self->priv->cheader_filenames, filename);
					filename = (g_free (filename), NULL);
				}
			}
			filename_collection = (_vala_array_free (filename_collection, filename_collection_length1, (GDestroyNotify) g_free), NULL);
		}
		val = (g_free (val), NULL);
	}
}


/**
 * Process all associated attributes.
 */
void vala_namespace_process_attributes (ValaNamespace* self) {
	g_return_if_fail (self != NULL);
	{
		GList* a_collection;
		GList* a_it;
		a_collection = ((ValaCodeNode*) self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* _tmp0;
			ValaAttribute* a;
			_tmp0 = NULL;
			a = (_tmp0 = (ValaAttribute*) a_it->data, (_tmp0 == NULL) ? NULL : vala_code_node_ref (_tmp0));
			{
				if (_vala_strcmp0 (vala_attribute_get_name (a), "CCode") == 0) {
					vala_namespace_process_ccode_attribute (self, a);
				}
				(a == NULL) ? NULL : (a = (vala_code_node_unref (a), NULL));
			}
		}
	}
}


static gboolean vala_namespace_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaNamespace * self;
	self = (ValaNamespace*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	vala_namespace_process_attributes (self);
	{
		GeeIterator* _ns_it;
		_ns_it = gee_iterable_iterator ((GeeIterable*) self->priv->namespaces);
		while (gee_iterator_next (_ns_it)) {
			ValaNamespace* ns;
			ns = (ValaNamespace*) gee_iterator_get (_ns_it);
			vala_code_node_check ((ValaCodeNode*) ns, analyzer);
			(ns == NULL) ? NULL : (ns = (vala_code_node_unref (ns), NULL));
		}
		(_ns_it == NULL) ? NULL : (_ns_it = (gee_collection_object_unref (_ns_it), NULL));
	}
	return !vala_code_node_get_error ((ValaCodeNode*) self);
}


static void vala_namespace_class_init (ValaNamespaceClass * klass) {
	vala_namespace_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_namespace_finalize;
	g_type_class_add_private (klass, sizeof (ValaNamespacePrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_namespace_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_namespace_real_accept_children;
	VALA_SYMBOL_CLASS (klass)->get_cprefix = vala_namespace_real_get_cprefix;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cprefix = vala_namespace_real_get_lower_case_cprefix;
	VALA_SYMBOL_CLASS (klass)->get_cheader_filenames = vala_namespace_real_get_cheader_filenames;
	VALA_CODE_NODE_CLASS (klass)->check = vala_namespace_real_check;
}


static void vala_namespace_instance_init (ValaNamespace * self) {
	self->priv = VALA_NAMESPACE_GET_PRIVATE (self);
	self->priv->classes = (GeeList*) gee_array_list_new (VALA_TYPE_CLASS, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->interfaces = (GeeList*) gee_array_list_new (VALA_TYPE_INTERFACE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->structs = (GeeList*) gee_array_list_new (VALA_TYPE_STRUCT, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->enums = (GeeList*) gee_array_list_new (VALA_TYPE_ENUM, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->error_domains = (GeeList*) gee_array_list_new (VALA_TYPE_ERROR_DOMAIN, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->delegates = (GeeList*) gee_array_list_new (VALA_TYPE_DELEGATE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->constants = (GeeList*) gee_array_list_new (VALA_TYPE_CONSTANT, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->fields = (GeeList*) gee_array_list_new (VALA_TYPE_FIELD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->methods = (GeeList*) gee_array_list_new (VALA_TYPE_METHOD, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
	self->priv->cprefixes = (GeeList*) gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_direct_equal);
	self->priv->cheader_filenames = (GeeList*) gee_array_list_new (G_TYPE_STRING, (GBoxedCopyFunc) g_strdup, g_free, g_direct_equal);
	self->priv->namespaces = (GeeList*) gee_array_list_new (VALA_TYPE_NAMESPACE, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
}


static void vala_namespace_finalize (ValaCodeNode* obj) {
	ValaNamespace * self;
	self = VALA_NAMESPACE (obj);
	(self->priv->classes == NULL) ? NULL : (self->priv->classes = (gee_collection_object_unref (self->priv->classes), NULL));
	(self->priv->interfaces == NULL) ? NULL : (self->priv->interfaces = (gee_collection_object_unref (self->priv->interfaces), NULL));
	(self->priv->structs == NULL) ? NULL : (self->priv->structs = (gee_collection_object_unref (self->priv->structs), NULL));
	(self->priv->enums == NULL) ? NULL : (self->priv->enums = (gee_collection_object_unref (self->priv->enums), NULL));
	(self->priv->error_domains == NULL) ? NULL : (self->priv->error_domains = (gee_collection_object_unref (self->priv->error_domains), NULL));
	(self->priv->delegates == NULL) ? NULL : (self->priv->delegates = (gee_collection_object_unref (self->priv->delegates), NULL));
	(self->priv->constants == NULL) ? NULL : (self->priv->constants = (gee_collection_object_unref (self->priv->constants), NULL));
	(self->priv->fields == NULL) ? NULL : (self->priv->fields = (gee_collection_object_unref (self->priv->fields), NULL));
	(self->priv->methods == NULL) ? NULL : (self->priv->methods = (gee_collection_object_unref (self->priv->methods), NULL));
	(self->priv->cprefixes == NULL) ? NULL : (self->priv->cprefixes = (gee_collection_object_unref (self->priv->cprefixes), NULL));
	self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL);
	(self->priv->cheader_filenames == NULL) ? NULL : (self->priv->cheader_filenames = (gee_collection_object_unref (self->priv->cheader_filenames), NULL));
	(self->priv->namespaces == NULL) ? NULL : (self->priv->namespaces = (gee_collection_object_unref (self->priv->namespaces), NULL));
	VALA_CODE_NODE_CLASS (vala_namespace_parent_class)->finalize (obj);
}


GType vala_namespace_get_type (void) {
	static GType vala_namespace_type_id = 0;
	if (vala_namespace_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaNamespaceClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_namespace_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaNamespace), 0, (GInstanceInitFunc) vala_namespace_instance_init, NULL };
		vala_namespace_type_id = g_type_register_static (VALA_TYPE_SYMBOL, "ValaNamespace", &g_define_type_info, 0);
	}
	return vala_namespace_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
	g_free (array);
}


static gint _vala_array_length (gpointer array) {
	int length;
	length = 0;
	if (array) {
		while (((gpointer*) array)[length]) {
			length++;
		}
	}
	return length;
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




