/* valainitializerlist.vala
 *
 * Copyright (C) 2006-2009  Jürg Billeter
 * Copyright (C) 2006-2008  Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 * 	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <vala/valainitializerlist.h>
#include <gee/arraylist.h>
#include <gee/collection.h>
#include <gee/readonlylist.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <stdlib.h>
#include <string.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valareport.h>
#include <vala/valaarraytype.h>
#include <vala/valaarraycreationexpression.h>
#include <vala/valaconstant.h>
#include <vala/valadatatype.h>
#include <vala/valastruct.h>
#include <vala/valafield.h>
#include <vala/valamember.h>
#include <vala/valaunaryexpression.h>




struct _ValaInitializerListPrivate {
	GeeList* initializers;
};

#define VALA_INITIALIZER_LIST_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_INITIALIZER_LIST, ValaInitializerListPrivate))
enum  {
	VALA_INITIALIZER_LIST_DUMMY_PROPERTY
};
static void vala_initializer_list_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_initializer_list_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gboolean vala_initializer_list_real_is_pure (ValaExpression* base);
static void vala_initializer_list_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static gboolean vala_initializer_list_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static gpointer vala_initializer_list_parent_class = NULL;
static void vala_initializer_list_finalize (ValaCodeNode* obj);



/**
 * Appends the specified expression to this initializer 
 *
 * @param expr an expression
 */
void vala_initializer_list_append (ValaInitializerList* self, ValaExpression* expr) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (expr != NULL);
	gee_collection_add ((GeeCollection*) self->priv->initializers, expr);
	vala_code_node_set_parent_node ((ValaCodeNode*) expr, (ValaCodeNode*) self);
}


/**
 * Returns a copy of the expression 
 *
 * @return expression list
 */
GeeList* vala_initializer_list_get_initializers (ValaInitializerList* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_EXPRESSION, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->initializers);
}


/**
 * Creates a new initializer 
 *
 * @param source_reference reference to source code
 * @return                 newly created initializer list
 */
ValaInitializerList* vala_initializer_list_construct (GType object_type, ValaSourceReference* source_reference) {
	ValaInitializerList* self;
	g_return_val_if_fail (source_reference != NULL, NULL);
	self = (ValaInitializerList*) g_type_create_instance (object_type);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source_reference);
	return self;
}


ValaInitializerList* vala_initializer_list_new (ValaSourceReference* source_reference) {
	return vala_initializer_list_construct (VALA_TYPE_INITIALIZER_LIST, source_reference);
}


static void vala_initializer_list_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaInitializerList * self;
	self = (ValaInitializerList*) base;
	g_return_if_fail (visitor != NULL);
	{
		GeeIterator* _expr_it;
		_expr_it = gee_iterable_iterator ((GeeIterable*) self->priv->initializers);
		while (gee_iterator_next (_expr_it)) {
			ValaExpression* expr;
			expr = (ValaExpression*) gee_iterator_get (_expr_it);
			vala_code_node_accept ((ValaCodeNode*) expr, visitor);
			(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
		}
		(_expr_it == NULL) ? NULL : (_expr_it = (gee_collection_object_unref (_expr_it), NULL));
	}
}


static void vala_initializer_list_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaInitializerList * self;
	self = (ValaInitializerList*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_initializer_list (visitor, self);
}


static gboolean vala_initializer_list_real_is_pure (ValaExpression* base) {
	ValaInitializerList * self;
	self = (ValaInitializerList*) base;
	{
		GeeIterator* _initializer_it;
		_initializer_it = gee_iterable_iterator ((GeeIterable*) self->priv->initializers);
		while (gee_iterator_next (_initializer_it)) {
			ValaExpression* initializer;
			initializer = (ValaExpression*) gee_iterator_get (_initializer_it);
			if (!vala_expression_is_pure (initializer)) {
				gboolean _tmp0;
				return (_tmp0 = FALSE, (initializer == NULL) ? NULL : (initializer = (vala_code_node_unref (initializer), NULL)), (_initializer_it == NULL) ? NULL : (_initializer_it = (gee_collection_object_unref (_initializer_it), NULL)), _tmp0);
			}
			(initializer == NULL) ? NULL : (initializer = (vala_code_node_unref (initializer), NULL));
		}
		(_initializer_it == NULL) ? NULL : (_initializer_it = (gee_collection_object_unref (_initializer_it), NULL));
	}
	return TRUE;
}


static void vala_initializer_list_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaInitializerList * self;
	self = (ValaInitializerList*) base;
	g_return_if_fail (old_node != NULL);
	g_return_if_fail (new_node != NULL);
	{
		gint i;
		i = 0;
		for (; i < gee_collection_get_size ((GeeCollection*) self->priv->initializers); i++) {
			ValaExpression* _tmp0;
			gboolean _tmp1;
			_tmp0 = NULL;
			if ((_tmp1 = (_tmp0 = (ValaExpression*) gee_list_get ((GeeList*) self->priv->initializers, i)) == old_node, (_tmp0 == NULL) ? NULL : (_tmp0 = (vala_code_node_unref (_tmp0), NULL)), _tmp1)) {
				gee_list_set ((GeeList*) self->priv->initializers, i, new_node);
			}
		}
	}
}


static gboolean vala_initializer_list_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaInitializerList * self;
	gboolean error;
	self = (ValaInitializerList*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	if (vala_expression_get_target_type ((ValaExpression*) self) == NULL) {
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "initializer list used for unknown type");
		return FALSE;
	} else {
		if (VALA_IS_ARRAY_TYPE (vala_expression_get_target_type ((ValaExpression*) self))) {
			ValaArrayType* _tmp2;
			ValaArrayType* array_type;
			gboolean _tmp3;
			gboolean _tmp4;
			ValaDataType* inner_target_type;
			/* initializer is used as array initializer */
			_tmp2 = NULL;
			array_type = (_tmp2 = VALA_ARRAY_TYPE (vala_expression_get_target_type ((ValaExpression*) self)), (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
			_tmp3 = FALSE;
			_tmp4 = FALSE;
			if (!(VALA_IS_ARRAY_CREATION_EXPRESSION (vala_code_node_get_parent_node ((ValaCodeNode*) self)))) {
				_tmp4 = !(VALA_IS_CONSTANT (vala_code_node_get_parent_node ((ValaCodeNode*) self)));
			} else {
				_tmp4 = FALSE;
			}
			if (_tmp4) {
				_tmp3 = !(VALA_IS_INITIALIZER_LIST (vala_code_node_get_parent_node ((ValaCodeNode*) self)));
			} else {
				_tmp3 = FALSE;
			}
			if (_tmp3) {
				ValaCodeNode* _tmp5;
				ValaCodeNode* old_parent_node;
				ValaDataType* _tmp6;
				ValaArrayCreationExpression* _tmp7;
				ValaArrayCreationExpression* array_creation;
				gboolean _tmp8;
				/* transform shorthand form
				     int[] array = { 42 };
				 into
				     int[] array = new int[] { 42 };*/
				_tmp5 = NULL;
				old_parent_node = (_tmp5 = vala_code_node_get_parent_node ((ValaCodeNode*) self), (_tmp5 == NULL) ? NULL : vala_code_node_ref (_tmp5));
				_tmp6 = NULL;
				_tmp7 = NULL;
				array_creation = (_tmp7 = vala_array_creation_expression_new (_tmp6 = vala_data_type_copy (vala_array_type_get_element_type (array_type)), vala_array_type_get_rank (array_type), self, vala_code_node_get_source_reference ((ValaCodeNode*) self)), (_tmp6 == NULL) ? NULL : (_tmp6 = (vala_code_node_unref (_tmp6), NULL)), _tmp7);
				vala_expression_set_target_type ((ValaExpression*) array_creation, vala_expression_get_target_type ((ValaExpression*) self));
				vala_code_node_replace_expression (old_parent_node, (ValaExpression*) self, (ValaExpression*) array_creation);
				vala_code_node_set_checked ((ValaCodeNode*) self, FALSE);
				return (_tmp8 = vala_code_node_check ((ValaCodeNode*) array_creation, analyzer), (old_parent_node == NULL) ? NULL : (old_parent_node = (vala_code_node_unref (old_parent_node), NULL)), (array_creation == NULL) ? NULL : (array_creation = (vala_code_node_unref (array_creation), NULL)), (array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL)), _tmp8);
			}
			inner_target_type = NULL;
			if (vala_array_type_get_rank (array_type) > 1) {
				ValaArrayType* inner_array_type;
				gint _tmp9;
				ValaDataType* _tmp11;
				ValaDataType* _tmp10;
				/* allow initialization of multi-dimensional arrays*/
				inner_array_type = VALA_ARRAY_TYPE (vala_data_type_copy ((ValaDataType*) array_type));
				_tmp9 = vala_array_type_get_rank (inner_array_type);
				vala_array_type_set_rank (inner_array_type, _tmp9 - 1);
				_tmp9;
				_tmp11 = NULL;
				_tmp10 = NULL;
				inner_target_type = (_tmp11 = (_tmp10 = (ValaDataType*) inner_array_type, (_tmp10 == NULL) ? NULL : vala_code_node_ref (_tmp10)), (inner_target_type == NULL) ? NULL : (inner_target_type = (vala_code_node_unref (inner_target_type), NULL)), _tmp11);
				(inner_array_type == NULL) ? NULL : (inner_array_type = (vala_code_node_unref (inner_array_type), NULL));
			} else {
				ValaDataType* _tmp12;
				_tmp12 = NULL;
				inner_target_type = (_tmp12 = vala_data_type_copy (vala_array_type_get_element_type (array_type)), (inner_target_type == NULL) ? NULL : (inner_target_type = (vala_code_node_unref (inner_target_type), NULL)), _tmp12);
			}
			{
				GeeList* _tmp13;
				GeeIterator* _tmp14;
				GeeIterator* _e_it;
				_tmp13 = NULL;
				_tmp14 = NULL;
				_e_it = (_tmp14 = gee_iterable_iterator ((GeeIterable*) (_tmp13 = vala_initializer_list_get_initializers (self))), (_tmp13 == NULL) ? NULL : (_tmp13 = (gee_collection_object_unref (_tmp13), NULL)), _tmp14);
				while (gee_iterator_next (_e_it)) {
					ValaExpression* e;
					e = (ValaExpression*) gee_iterator_get (_e_it);
					vala_expression_set_target_type (e, inner_target_type);
					(e == NULL) ? NULL : (e = (vala_code_node_unref (e), NULL));
				}
				(_e_it == NULL) ? NULL : (_e_it = (gee_collection_object_unref (_e_it), NULL));
			}
			(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
			(inner_target_type == NULL) ? NULL : (inner_target_type = (vala_code_node_unref (inner_target_type), NULL));
		} else {
			if (VALA_IS_STRUCT (vala_data_type_get_data_type (vala_expression_get_target_type ((ValaExpression*) self)))) {
				ValaStruct* _tmp15;
				ValaStruct* st;
				GeeList* _tmp16;
				GeeIterator* _tmp17;
				GeeIterator* field_it;
				/* initializer is used as struct initializer */
				_tmp15 = NULL;
				st = (_tmp15 = VALA_STRUCT (vala_data_type_get_data_type (vala_expression_get_target_type ((ValaExpression*) self))), (_tmp15 == NULL) ? NULL : vala_code_node_ref (_tmp15));
				_tmp16 = NULL;
				_tmp17 = NULL;
				field_it = (_tmp17 = gee_iterable_iterator ((GeeIterable*) (_tmp16 = vala_struct_get_fields (st))), (_tmp16 == NULL) ? NULL : (_tmp16 = (gee_collection_object_unref (_tmp16), NULL)), _tmp17);
				{
					GeeList* _tmp18;
					GeeIterator* _tmp19;
					GeeIterator* _e_it;
					_tmp18 = NULL;
					_tmp19 = NULL;
					_e_it = (_tmp19 = gee_iterable_iterator ((GeeIterable*) (_tmp18 = vala_initializer_list_get_initializers (self))), (_tmp18 == NULL) ? NULL : (_tmp18 = (gee_collection_object_unref (_tmp18), NULL)), _tmp19);
					while (gee_iterator_next (_e_it)) {
						ValaExpression* e;
						ValaField* field;
						ValaDataType* _tmp25;
						e = (ValaExpression*) gee_iterator_get (_e_it);
						field = NULL;
						while (field == NULL) {
							ValaField* _tmp23;
							if (!gee_iterator_next (field_it)) {
								char* _tmp21;
								char* _tmp20;
								gboolean _tmp22;
								vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
								_tmp21 = NULL;
								_tmp20 = NULL;
								vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) e), _tmp21 = g_strdup_printf ("too many expressions in initializer list for `%s'", _tmp20 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_target_type ((ValaExpression*) self))));
								_tmp21 = (g_free (_tmp21), NULL);
								_tmp20 = (g_free (_tmp20), NULL);
								return (_tmp22 = FALSE, (e == NULL) ? NULL : (e = (vala_code_node_unref (e), NULL)), (field == NULL) ? NULL : (field = (vala_code_node_unref (field), NULL)), (_e_it == NULL) ? NULL : (_e_it = (gee_collection_object_unref (_e_it), NULL)), (st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL)), (field_it == NULL) ? NULL : (field_it = (gee_collection_object_unref (field_it), NULL)), _tmp22);
							}
							_tmp23 = NULL;
							field = (_tmp23 = (ValaField*) gee_iterator_get (field_it), (field == NULL) ? NULL : (field = (vala_code_node_unref (field), NULL)), _tmp23);
							if (vala_field_get_binding (field) != MEMBER_BINDING_INSTANCE) {
								ValaField* _tmp24;
								/* we only initialize instance fields*/
								_tmp24 = NULL;
								field = (_tmp24 = NULL, (field == NULL) ? NULL : (field = (vala_code_node_unref (field), NULL)), _tmp24);
							}
						}
						_tmp25 = NULL;
						vala_expression_set_target_type (e, _tmp25 = vala_data_type_copy (vala_field_get_field_type (field)));
						(_tmp25 == NULL) ? NULL : (_tmp25 = (vala_code_node_unref (_tmp25), NULL));
						if (!vala_data_type_get_value_owned (vala_expression_get_target_type ((ValaExpression*) self))) {
							vala_data_type_set_value_owned (vala_expression_get_target_type (e), FALSE);
						}
						(e == NULL) ? NULL : (e = (vala_code_node_unref (e), NULL));
						(field == NULL) ? NULL : (field = (vala_code_node_unref (field), NULL));
					}
					(_e_it == NULL) ? NULL : (_e_it = (gee_collection_object_unref (_e_it), NULL));
				}
				(st == NULL) ? NULL : (st = (vala_code_node_unref (st), NULL));
				(field_it == NULL) ? NULL : (field_it = (gee_collection_object_unref (field_it), NULL));
			} else {
				char* _tmp27;
				char* _tmp26;
				vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
				_tmp27 = NULL;
				_tmp26 = NULL;
				vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp27 = g_strdup_printf ("initializer list used for `%s', which is neither array nor struct", _tmp26 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_target_type ((ValaExpression*) self))));
				_tmp27 = (g_free (_tmp27), NULL);
				_tmp26 = (g_free (_tmp26), NULL);
				return FALSE;
			}
		}
	}
	{
		GeeIterator* _expr_it;
		_expr_it = gee_iterable_iterator ((GeeIterable*) self->priv->initializers);
		while (gee_iterator_next (_expr_it)) {
			ValaExpression* expr;
			expr = (ValaExpression*) gee_iterator_get (_expr_it);
			vala_code_node_check ((ValaCodeNode*) expr, analyzer);
			(expr == NULL) ? NULL : (expr = (vala_code_node_unref (expr), NULL));
		}
		(_expr_it == NULL) ? NULL : (_expr_it = (gee_collection_object_unref (_expr_it), NULL));
	}
	error = FALSE;
	{
		GeeList* _tmp29;
		GeeIterator* _tmp30;
		GeeIterator* _e_it;
		_tmp29 = NULL;
		_tmp30 = NULL;
		_e_it = (_tmp30 = gee_iterable_iterator ((GeeIterable*) (_tmp29 = vala_initializer_list_get_initializers (self))), (_tmp29 == NULL) ? NULL : (_tmp29 = (gee_collection_object_unref (_tmp29), NULL)), _tmp30);
		while (gee_iterator_next (_e_it)) {
			ValaExpression* e;
			ValaUnaryExpression* _tmp32;
			ValaExpression* _tmp31;
			ValaUnaryExpression* unary;
			gboolean _tmp33;
			e = (ValaExpression*) gee_iterator_get (_e_it);
			if (vala_expression_get_value_type (e) == NULL) {
				error = TRUE;
				(e == NULL) ? NULL : (e = (vala_code_node_unref (e), NULL));
				continue;
			}
			_tmp32 = NULL;
			_tmp31 = NULL;
			unary = (_tmp32 = (_tmp31 = e, VALA_IS_UNARY_EXPRESSION (_tmp31) ? ((ValaUnaryExpression*) _tmp31) : NULL), (_tmp32 == NULL) ? NULL : vala_code_node_ref (_tmp32));
			_tmp33 = FALSE;
			if (unary != NULL) {
				gboolean _tmp34;
				_tmp34 = FALSE;
				if (vala_unary_expression_get_operator (unary) == VALA_UNARY_OPERATOR_REF) {
					_tmp34 = TRUE;
				} else {
					_tmp34 = vala_unary_expression_get_operator (unary) == VALA_UNARY_OPERATOR_OUT;
				}
				_tmp33 = (_tmp34);
			} else {
				_tmp33 = FALSE;
			}
			if (_tmp33) {
			} else {
				if (!vala_data_type_compatible (vala_expression_get_value_type (e), vala_expression_get_target_type (e))) {
					char* _tmp37;
					char* _tmp36;
					char* _tmp35;
					error = TRUE;
					vala_code_node_set_error ((ValaCodeNode*) e, TRUE);
					_tmp37 = NULL;
					_tmp36 = NULL;
					_tmp35 = NULL;
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) e), _tmp37 = g_strdup_printf ("Expected initializer of type `%s' but got `%s'", _tmp35 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_target_type (e)), _tmp36 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (e))));
					_tmp37 = (g_free (_tmp37), NULL);
					_tmp36 = (g_free (_tmp36), NULL);
					_tmp35 = (g_free (_tmp35), NULL);
				}
			}
			(e == NULL) ? NULL : (e = (vala_code_node_unref (e), NULL));
			(unary == NULL) ? NULL : (unary = (vala_code_node_unref (unary), NULL));
		}
		(_e_it == NULL) ? NULL : (_e_it = (gee_collection_object_unref (_e_it), NULL));
	}
	if (!error) {
		/* everything seems to be correct */
		vala_expression_set_value_type ((ValaExpression*) self, vala_expression_get_target_type ((ValaExpression*) self));
	}
	return !error;
}


gint vala_initializer_list_get_size (ValaInitializerList* self) {
	g_return_val_if_fail (self != NULL, 0);
	return gee_collection_get_size ((GeeCollection*) self->priv->initializers);
}


static void vala_initializer_list_class_init (ValaInitializerListClass * klass) {
	vala_initializer_list_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_initializer_list_finalize;
	g_type_class_add_private (klass, sizeof (ValaInitializerListPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_initializer_list_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_initializer_list_real_accept;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_initializer_list_real_is_pure;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_initializer_list_real_replace_expression;
	VALA_CODE_NODE_CLASS (klass)->check = vala_initializer_list_real_check;
}


static void vala_initializer_list_instance_init (ValaInitializerList * self) {
	self->priv = VALA_INITIALIZER_LIST_GET_PRIVATE (self);
	self->priv->initializers = (GeeList*) gee_array_list_new (VALA_TYPE_EXPRESSION, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
}


static void vala_initializer_list_finalize (ValaCodeNode* obj) {
	ValaInitializerList * self;
	self = VALA_INITIALIZER_LIST (obj);
	(self->priv->initializers == NULL) ? NULL : (self->priv->initializers = (gee_collection_object_unref (self->priv->initializers), NULL));
	VALA_CODE_NODE_CLASS (vala_initializer_list_parent_class)->finalize (obj);
}


GType vala_initializer_list_get_type (void) {
	static GType vala_initializer_list_type_id = 0;
	if (vala_initializer_list_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaInitializerListClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_initializer_list_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaInitializerList), 0, (GInstanceInitFunc) vala_initializer_list_instance_init, NULL };
		vala_initializer_list_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaInitializerList", &g_define_type_info, 0);
	}
	return vala_initializer_list_type_id;
}




