/* valaelementaccess.vala
 *
 * Copyright (C) 2006-2008  Raffaele Sandrini, Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Raffaele Sandrini <raffaele@sandrini.ch>
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaelementaccess.h>
#include <gee/arraylist.h>
#include <gee/readonlylist.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <stdlib.h>
#include <string.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valadatatype.h>
#include <vala/valatypesymbol.h>
#include <vala/valamemberaccess.h>
#include <vala/valasignal.h>
#include <vala/valareport.h>
#include <vala/valapointertype.h>
#include <vala/valaarraytype.h>
#include <vala/valasymbol.h>
#include <vala/valascope.h>
#include <vala/valamethod.h>
#include <vala/valaformalparameter.h>
#include <vala/valagenerictype.h>




struct _ValaElementAccessPrivate {
	GeeList* indices;
	ValaExpression* _container;
};

#define VALA_ELEMENT_ACCESS_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_ELEMENT_ACCESS, ValaElementAccessPrivate))
enum  {
	VALA_ELEMENT_ACCESS_DUMMY_PROPERTY
};
static void vala_element_access_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_element_access_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_element_access_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static gboolean vala_element_access_real_is_pure (ValaExpression* base);
static gboolean vala_element_access_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer);
static void vala_element_access_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection);
static void vala_element_access_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection);
static gpointer vala_element_access_parent_class = NULL;
static void vala_element_access_finalize (ValaCodeNode* obj);



void vala_element_access_append_index (ValaElementAccess* self, ValaExpression* index) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (index != NULL);
	gee_collection_add ((GeeCollection*) self->priv->indices, index);
	vala_code_node_set_parent_node ((ValaCodeNode*) index, (ValaCodeNode*) self);
}


GeeList* vala_element_access_get_indices (ValaElementAccess* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return (GeeList*) gee_read_only_list_new (VALA_TYPE_EXPRESSION, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, self->priv->indices);
}


ValaElementAccess* vala_element_access_construct (GType object_type, ValaExpression* container, ValaSourceReference* source_reference) {
	ValaElementAccess* self;
	g_return_val_if_fail (container != NULL, NULL);
	g_return_val_if_fail (source_reference != NULL, NULL);
	self = (ValaElementAccess*) g_type_create_instance (object_type);
	vala_code_node_set_source_reference ((ValaCodeNode*) self, source_reference);
	vala_element_access_set_container (self, container);
	return self;
}


ValaElementAccess* vala_element_access_new (ValaExpression* container, ValaSourceReference* source_reference) {
	return vala_element_access_construct (VALA_TYPE_ELEMENT_ACCESS, container, source_reference);
}


static void vala_element_access_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaElementAccess * self;
	self = (ValaElementAccess*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_visitor_visit_element_access (visitor, self);
	vala_code_visitor_visit_expression (visitor, (ValaExpression*) self);
}


static void vala_element_access_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaElementAccess * self;
	self = (ValaElementAccess*) base;
	g_return_if_fail (visitor != NULL);
	vala_code_node_accept ((ValaCodeNode*) vala_element_access_get_container (self), visitor);
	{
		GeeIterator* _e_it;
		_e_it = gee_iterable_iterator ((GeeIterable*) self->priv->indices);
		while (gee_iterator_next (_e_it)) {
			ValaExpression* e;
			e = (ValaExpression*) gee_iterator_get (_e_it);
			vala_code_node_accept ((ValaCodeNode*) e, visitor);
			(e == NULL) ? NULL : (e = (vala_code_node_unref (e), NULL));
		}
		(_e_it == NULL) ? NULL : (_e_it = (gee_collection_object_unref (_e_it), NULL));
	}
}


static void vala_element_access_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaElementAccess * self;
	gint index;
	gboolean _tmp0;
	self = (ValaElementAccess*) base;
	g_return_if_fail (old_node != NULL);
	g_return_if_fail (new_node != NULL);
	if (vala_element_access_get_container (self) == old_node) {
		vala_element_access_set_container (self, new_node);
	}
	index = gee_list_index_of (self->priv->indices, old_node);
	_tmp0 = FALSE;
	if (index >= 0) {
		_tmp0 = vala_code_node_get_parent_node ((ValaCodeNode*) new_node) == NULL;
	} else {
		_tmp0 = FALSE;
	}
	if (_tmp0) {
		gee_list_set ((GeeList*) self->priv->indices, index, new_node);
		vala_code_node_set_parent_node ((ValaCodeNode*) new_node, (ValaCodeNode*) self);
	}
}


static gboolean vala_element_access_real_is_pure (ValaExpression* base) {
	ValaElementAccess * self;
	self = (ValaElementAccess*) base;
	{
		GeeIterator* _index_it;
		_index_it = gee_iterable_iterator ((GeeIterable*) self->priv->indices);
		while (gee_iterator_next (_index_it)) {
			ValaExpression* index;
			index = (ValaExpression*) gee_iterator_get (_index_it);
			if (!vala_expression_is_pure (index)) {
				gboolean _tmp0;
				return (_tmp0 = FALSE, (index == NULL) ? NULL : (index = (vala_code_node_unref (index), NULL)), (_index_it == NULL) ? NULL : (_index_it = (gee_collection_object_unref (_index_it), NULL)), _tmp0);
			}
			(index == NULL) ? NULL : (index = (vala_code_node_unref (index), NULL));
		}
		(_index_it == NULL) ? NULL : (_index_it = (gee_collection_object_unref (_index_it), NULL));
	}
	return vala_expression_is_pure (vala_element_access_get_container (self));
}


static gboolean vala_element_access_real_check (ValaCodeNode* base, ValaSemanticAnalyzer* analyzer) {
	ValaElementAccess * self;
	ValaTypeSymbol* _tmp2;
	ValaTypeSymbol* container_type;
	gboolean _tmp3;
	gboolean index_int_type_check;
	ValaPointerType* _tmp13;
	ValaDataType* _tmp12;
	ValaPointerType* pointer_type;
	gboolean _tmp43;
	self = (ValaElementAccess*) base;
	g_return_val_if_fail (analyzer != NULL, FALSE);
	if (vala_code_node_get_checked ((ValaCodeNode*) self)) {
		return !vala_code_node_get_error ((ValaCodeNode*) self);
	}
	vala_code_node_set_checked ((ValaCodeNode*) self, TRUE);
	vala_code_node_check ((ValaCodeNode*) vala_element_access_get_container (self), analyzer);
	if (vala_expression_get_value_type (vala_element_access_get_container (self)) == NULL) {
		/* don't proceed if a child expression failed */
		vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
		return FALSE;
	}
	_tmp2 = NULL;
	container_type = (_tmp2 = vala_data_type_get_data_type (vala_expression_get_value_type (vala_element_access_get_container (self))), (_tmp2 == NULL) ? NULL : vala_code_node_ref (_tmp2));
	_tmp3 = FALSE;
	if (VALA_IS_MEMBER_ACCESS (vala_element_access_get_container (self))) {
		_tmp3 = VALA_IS_SIGNAL (vala_expression_get_symbol_reference (vala_element_access_get_container (self)));
	} else {
		_tmp3 = FALSE;
	}
	if (_tmp3) {
		GeeList* _tmp4;
		gboolean _tmp5;
		ValaExpression* _tmp9;
		GeeList* _tmp8;
		ValaDataType* _tmp7;
		/* signal detail access*/
		_tmp4 = NULL;
		if ((_tmp5 = gee_collection_get_size ((GeeCollection*) (_tmp4 = vala_element_access_get_indices (self))) != 1, (_tmp4 == NULL) ? NULL : (_tmp4 = (gee_collection_object_unref (_tmp4), NULL)), _tmp5)) {
			gboolean _tmp6;
			vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
			vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Element access with more than one dimension is not supported for signals");
			return (_tmp6 = FALSE, (container_type == NULL) ? NULL : (container_type = (vala_code_node_unref (container_type), NULL)), _tmp6);
		}
		_tmp9 = NULL;
		_tmp8 = NULL;
		_tmp7 = NULL;
		vala_expression_set_target_type (_tmp9 = (ValaExpression*) gee_list_get (_tmp8 = vala_element_access_get_indices (self), 0), _tmp7 = vala_data_type_copy (analyzer->string_type));
		(_tmp9 == NULL) ? NULL : (_tmp9 = (vala_code_node_unref (_tmp9), NULL));
		(_tmp8 == NULL) ? NULL : (_tmp8 = (gee_collection_object_unref (_tmp8), NULL));
		(_tmp7 == NULL) ? NULL : (_tmp7 = (vala_code_node_unref (_tmp7), NULL));
	}
	{
		GeeList* _tmp10;
		GeeIterator* _tmp11;
		GeeIterator* _index_it;
		_tmp10 = NULL;
		_tmp11 = NULL;
		_index_it = (_tmp11 = gee_iterable_iterator ((GeeIterable*) (_tmp10 = vala_element_access_get_indices (self))), (_tmp10 == NULL) ? NULL : (_tmp10 = (gee_collection_object_unref (_tmp10), NULL)), _tmp11);
		while (gee_iterator_next (_index_it)) {
			ValaExpression* index;
			index = (ValaExpression*) gee_iterator_get (_index_it);
			vala_code_node_check ((ValaCodeNode*) index, analyzer);
			(index == NULL) ? NULL : (index = (vala_code_node_unref (index), NULL));
		}
		(_index_it == NULL) ? NULL : (_index_it = (gee_collection_object_unref (_index_it), NULL));
	}
	index_int_type_check = TRUE;
	_tmp13 = NULL;
	_tmp12 = NULL;
	pointer_type = (_tmp13 = (_tmp12 = vala_expression_get_value_type (vala_element_access_get_container (self)), VALA_IS_POINTER_TYPE (_tmp12) ? ((ValaPointerType*) _tmp12) : NULL), (_tmp13 == NULL) ? NULL : vala_code_node_ref (_tmp13));
	/* assign a value_type when possible */
	if (VALA_IS_ARRAY_TYPE (vala_expression_get_value_type (vala_element_access_get_container (self)))) {
		ValaArrayType* _tmp14;
		ValaArrayType* array_type;
		ValaDataType* _tmp15;
		_tmp14 = NULL;
		array_type = (_tmp14 = VALA_ARRAY_TYPE (vala_expression_get_value_type (vala_element_access_get_container (self))), (_tmp14 == NULL) ? NULL : vala_code_node_ref (_tmp14));
		_tmp15 = NULL;
		vala_expression_set_value_type ((ValaExpression*) self, _tmp15 = vala_data_type_copy (vala_array_type_get_element_type (array_type)));
		(_tmp15 == NULL) ? NULL : (_tmp15 = (vala_code_node_unref (_tmp15), NULL));
		if (!vala_expression_get_lvalue ((ValaExpression*) self)) {
			vala_data_type_set_value_owned (vala_expression_get_value_type ((ValaExpression*) self), FALSE);
		}
		(array_type == NULL) ? NULL : (array_type = (vala_code_node_unref (array_type), NULL));
	} else {
		gboolean _tmp16;
		_tmp16 = FALSE;
		if (pointer_type != NULL) {
			_tmp16 = !vala_data_type_is_reference_type_or_type_parameter (vala_pointer_type_get_base_type (pointer_type));
		} else {
			_tmp16 = FALSE;
		}
		if (_tmp16) {
			ValaDataType* _tmp17;
			_tmp17 = NULL;
			vala_expression_set_value_type ((ValaExpression*) self, _tmp17 = vala_data_type_copy (vala_pointer_type_get_base_type (pointer_type)));
			(_tmp17 == NULL) ? NULL : (_tmp17 = (vala_code_node_unref (_tmp17), NULL));
		} else {
			if (container_type == vala_data_type_get_data_type (analyzer->string_type)) {
				GeeList* _tmp18;
				gboolean _tmp19;
				_tmp18 = NULL;
				if ((_tmp19 = gee_collection_get_size ((GeeCollection*) (_tmp18 = vala_element_access_get_indices (self))) != 1, (_tmp18 == NULL) ? NULL : (_tmp18 = (gee_collection_object_unref (_tmp18), NULL)), _tmp19)) {
					gboolean _tmp20;
					vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Element access with more than one dimension is not supported for strings");
					return (_tmp20 = FALSE, (container_type == NULL) ? NULL : (container_type = (vala_code_node_unref (container_type), NULL)), (pointer_type == NULL) ? NULL : (pointer_type = (vala_code_node_unref (pointer_type), NULL)), _tmp20);
				}
				vala_expression_set_value_type ((ValaExpression*) self, analyzer->unichar_type);
			} else {
				gboolean _tmp21;
				gboolean _tmp22;
				gboolean _tmp23;
				_tmp21 = FALSE;
				_tmp22 = FALSE;
				_tmp23 = FALSE;
				if (container_type != NULL) {
					_tmp23 = analyzer->list_type != NULL;
				} else {
					_tmp23 = FALSE;
				}
				if (_tmp23) {
					_tmp22 = analyzer->map_type != NULL;
				} else {
					_tmp22 = FALSE;
				}
				if (_tmp22) {
					gboolean _tmp24;
					_tmp24 = FALSE;
					if (vala_typesymbol_is_subtype_of (container_type, (ValaTypeSymbol*) analyzer->list_type)) {
						_tmp24 = TRUE;
					} else {
						_tmp24 = vala_typesymbol_is_subtype_of (container_type, (ValaTypeSymbol*) analyzer->map_type);
					}
					_tmp21 = (_tmp24);
				} else {
					_tmp21 = FALSE;
				}
				if (_tmp21) {
					GeeList* indices;
					GeeIterator* indices_it;
					ValaExpression* index;
					ValaSymbol* get_sym;
					ValaMethod* _tmp28;
					ValaMethod* get_method;
					GeeList* get_params;
					GeeIterator* get_params_it;
					ValaFormalParameter* get_param;
					ValaDataType* _tmp29;
					ValaDataType* index_type;
					ValaDataType* _tmp36;
					ValaDataType* _tmp35;
					indices = vala_element_access_get_indices (self);
					if (gee_collection_get_size ((GeeCollection*) indices) != 1) {
						gboolean _tmp25;
						vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
						vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), "Element access with more than one dimension is not supported for the specified type");
						return (_tmp25 = FALSE, (indices == NULL) ? NULL : (indices = (gee_collection_object_unref (indices), NULL)), (container_type == NULL) ? NULL : (container_type = (vala_code_node_unref (container_type), NULL)), (pointer_type == NULL) ? NULL : (pointer_type = (vala_code_node_unref (pointer_type), NULL)), _tmp25);
					}
					indices_it = gee_iterable_iterator ((GeeIterable*) indices);
					gee_iterator_next (indices_it);
					index = (ValaExpression*) gee_iterator_get (indices_it);
					index_int_type_check = FALSE;
					/* lookup symbol in interface instead of class as implemented interface methods are not in VAPI files*/
					get_sym = NULL;
					if (vala_typesymbol_is_subtype_of (container_type, (ValaTypeSymbol*) analyzer->list_type)) {
						ValaSymbol* _tmp26;
						_tmp26 = NULL;
						get_sym = (_tmp26 = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) analyzer->list_type), "get"), (get_sym == NULL) ? NULL : (get_sym = (vala_code_node_unref (get_sym), NULL)), _tmp26);
					} else {
						if (vala_typesymbol_is_subtype_of (container_type, (ValaTypeSymbol*) analyzer->map_type)) {
							ValaSymbol* _tmp27;
							_tmp27 = NULL;
							get_sym = (_tmp27 = vala_scope_lookup (vala_symbol_get_scope ((ValaSymbol*) analyzer->map_type), "get"), (get_sym == NULL) ? NULL : (get_sym = (vala_code_node_unref (get_sym), NULL)), _tmp27);
						}
					}
					_tmp28 = NULL;
					get_method = (_tmp28 = VALA_METHOD (get_sym), (_tmp28 == NULL) ? NULL : vala_code_node_ref (_tmp28));
					get_params = vala_method_get_parameters (get_method);
					get_params_it = gee_iterable_iterator ((GeeIterable*) get_params);
					gee_iterator_next (get_params_it);
					get_param = (ValaFormalParameter*) gee_iterator_get (get_params_it);
					_tmp29 = NULL;
					index_type = (_tmp29 = vala_formal_parameter_get_parameter_type (get_param), (_tmp29 == NULL) ? NULL : vala_code_node_ref (_tmp29));
					if (VALA_IS_GENERIC_TYPE (index_type)) {
						ValaDataType* _tmp30;
						_tmp30 = NULL;
						index_type = (_tmp30 = vala_semantic_analyzer_get_actual_type (vala_expression_get_value_type (vala_element_access_get_container (self)), VALA_GENERIC_TYPE (index_type), (ValaCodeNode*) self), (index_type == NULL) ? NULL : (index_type = (vala_code_node_unref (index_type), NULL)), _tmp30);
					}
					if (!vala_data_type_compatible (vala_expression_get_value_type (index), index_type)) {
						char* _tmp33;
						char* _tmp32;
						char* _tmp31;
						gboolean _tmp34;
						vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
						_tmp33 = NULL;
						_tmp32 = NULL;
						_tmp31 = NULL;
						vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp33 = g_strdup_printf ("index expression: Cannot convert from `%s' to `%s'", _tmp31 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (index)), _tmp32 = vala_code_node_to_string ((ValaCodeNode*) index_type)));
						_tmp33 = (g_free (_tmp33), NULL);
						_tmp32 = (g_free (_tmp32), NULL);
						_tmp31 = (g_free (_tmp31), NULL);
						return (_tmp34 = FALSE, (indices == NULL) ? NULL : (indices = (gee_collection_object_unref (indices), NULL)), (indices_it == NULL) ? NULL : (indices_it = (gee_collection_object_unref (indices_it), NULL)), (index == NULL) ? NULL : (index = (vala_code_node_unref (index), NULL)), (get_sym == NULL) ? NULL : (get_sym = (vala_code_node_unref (get_sym), NULL)), (get_method == NULL) ? NULL : (get_method = (vala_code_node_unref (get_method), NULL)), (get_params == NULL) ? NULL : (get_params = (gee_collection_object_unref (get_params), NULL)), (get_params_it == NULL) ? NULL : (get_params_it = (gee_collection_object_unref (get_params_it), NULL)), (get_param == NULL) ? NULL : (get_param = (vala_code_node_unref (get_param), NULL)), (index_type == NULL) ? NULL : (index_type = (vala_code_node_unref (index_type), NULL)), (container_type == NULL) ? NULL : (container_type = (vala_code_node_unref (container_type), NULL)), (pointer_type == NULL) ? NULL : (pointer_type = (vala_code_node_unref (pointer_type), NULL)), _tmp34);
					}
					_tmp36 = NULL;
					_tmp35 = NULL;
					vala_expression_set_value_type ((ValaExpression*) self, _tmp36 = vala_data_type_copy (_tmp35 = vala_semantic_analyzer_get_actual_type (vala_expression_get_value_type (vala_element_access_get_container (self)), VALA_GENERIC_TYPE (vala_method_get_return_type (get_method)), (ValaCodeNode*) self)));
					(_tmp36 == NULL) ? NULL : (_tmp36 = (vala_code_node_unref (_tmp36), NULL));
					(_tmp35 == NULL) ? NULL : (_tmp35 = (vala_code_node_unref (_tmp35), NULL));
					if (vala_expression_get_lvalue ((ValaExpression*) self)) {
						/* get () returns owned value, set () accepts unowned value*/
						vala_data_type_set_value_owned (vala_expression_get_value_type ((ValaExpression*) self), FALSE);
					}
					(indices == NULL) ? NULL : (indices = (gee_collection_object_unref (indices), NULL));
					(indices_it == NULL) ? NULL : (indices_it = (gee_collection_object_unref (indices_it), NULL));
					(index == NULL) ? NULL : (index = (vala_code_node_unref (index), NULL));
					(get_sym == NULL) ? NULL : (get_sym = (vala_code_node_unref (get_sym), NULL));
					(get_method == NULL) ? NULL : (get_method = (vala_code_node_unref (get_method), NULL));
					(get_params == NULL) ? NULL : (get_params = (gee_collection_object_unref (get_params), NULL));
					(get_params_it == NULL) ? NULL : (get_params_it = (gee_collection_object_unref (get_params_it), NULL));
					(get_param == NULL) ? NULL : (get_param = (vala_code_node_unref (get_param), NULL));
					(index_type == NULL) ? NULL : (index_type = (vala_code_node_unref (index_type), NULL));
				} else {
					gboolean _tmp37;
					_tmp37 = FALSE;
					if (VALA_IS_MEMBER_ACCESS (vala_element_access_get_container (self))) {
						_tmp37 = VALA_IS_SIGNAL (vala_expression_get_symbol_reference (vala_element_access_get_container (self)));
					} else {
						_tmp37 = FALSE;
					}
					if (_tmp37) {
						index_int_type_check = FALSE;
						vala_expression_set_symbol_reference ((ValaExpression*) self, vala_expression_get_symbol_reference (vala_element_access_get_container (self)));
						vala_expression_set_value_type ((ValaExpression*) self, vala_expression_get_value_type (vala_element_access_get_container (self)));
					} else {
						char* _tmp39;
						char* _tmp38;
						vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
						_tmp39 = NULL;
						_tmp38 = NULL;
						vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) self), _tmp39 = g_strdup_printf ("The expression `%s' does not denote an Array", _tmp38 = vala_code_node_to_string ((ValaCodeNode*) vala_expression_get_value_type (vala_element_access_get_container (self)))));
						_tmp39 = (g_free (_tmp39), NULL);
						_tmp38 = (g_free (_tmp38), NULL);
					}
				}
			}
		}
	}
	if (index_int_type_check) {
		/* check if the index is of type integer */
		{
			GeeList* _tmp40;
			GeeIterator* _tmp41;
			GeeIterator* _e_it;
			/* check if the index is of type integer */
			_tmp40 = NULL;
			_tmp41 = NULL;
			_e_it = (_tmp41 = gee_iterable_iterator ((GeeIterable*) (_tmp40 = vala_element_access_get_indices (self))), (_tmp40 == NULL) ? NULL : (_tmp40 = (gee_collection_object_unref (_tmp40), NULL)), _tmp41);
			/* check if the index is of type integer */
			while (gee_iterator_next (_e_it)) {
				ValaExpression* e;
				/* check if the index is of type integer */
				e = (ValaExpression*) gee_iterator_get (_e_it);
				/* don't proceed if a child expression failed */
				if (vala_expression_get_value_type (e) == NULL) {
					gboolean _tmp42;
					return (_tmp42 = FALSE, (e == NULL) ? NULL : (e = (vala_code_node_unref (e), NULL)), (_e_it == NULL) ? NULL : (_e_it = (gee_collection_object_unref (_e_it), NULL)), (container_type == NULL) ? NULL : (container_type = (vala_code_node_unref (container_type), NULL)), (pointer_type == NULL) ? NULL : (pointer_type = (vala_code_node_unref (pointer_type), NULL)), _tmp42);
				}
				/* check if the index is of type integer */
				if (!vala_data_type_compatible (vala_expression_get_value_type (e), analyzer->uint64_type)) {
					vala_code_node_set_error ((ValaCodeNode*) self, TRUE);
					vala_report_error (vala_code_node_get_source_reference ((ValaCodeNode*) e), "Expression of integer type expected");
				}
				(e == NULL) ? NULL : (e = (vala_code_node_unref (e), NULL));
			}
			(_e_it == NULL) ? NULL : (_e_it = (gee_collection_object_unref (_e_it), NULL));
		}
	}
	return (_tmp43 = !vala_code_node_get_error ((ValaCodeNode*) self), (container_type == NULL) ? NULL : (container_type = (vala_code_node_unref (container_type), NULL)), (pointer_type == NULL) ? NULL : (pointer_type = (vala_code_node_unref (pointer_type), NULL)), _tmp43);
}


static void vala_element_access_real_get_defined_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaElementAccess * self;
	self = (ValaElementAccess*) base;
	g_return_if_fail (collection != NULL);
	vala_code_node_get_defined_variables ((ValaCodeNode*) vala_element_access_get_container (self), collection);
	{
		GeeIterator* _index_it;
		_index_it = gee_iterable_iterator ((GeeIterable*) self->priv->indices);
		while (gee_iterator_next (_index_it)) {
			ValaExpression* index;
			index = (ValaExpression*) gee_iterator_get (_index_it);
			vala_code_node_get_defined_variables ((ValaCodeNode*) index, collection);
			(index == NULL) ? NULL : (index = (vala_code_node_unref (index), NULL));
		}
		(_index_it == NULL) ? NULL : (_index_it = (gee_collection_object_unref (_index_it), NULL));
	}
}


static void vala_element_access_real_get_used_variables (ValaCodeNode* base, GeeCollection* collection) {
	ValaElementAccess * self;
	self = (ValaElementAccess*) base;
	g_return_if_fail (collection != NULL);
	vala_code_node_get_used_variables ((ValaCodeNode*) vala_element_access_get_container (self), collection);
	{
		GeeIterator* _index_it;
		_index_it = gee_iterable_iterator ((GeeIterable*) self->priv->indices);
		while (gee_iterator_next (_index_it)) {
			ValaExpression* index;
			index = (ValaExpression*) gee_iterator_get (_index_it);
			vala_code_node_get_used_variables ((ValaCodeNode*) index, collection);
			(index == NULL) ? NULL : (index = (vala_code_node_unref (index), NULL));
		}
		(_index_it == NULL) ? NULL : (_index_it = (gee_collection_object_unref (_index_it), NULL));
	}
}


ValaExpression* vala_element_access_get_container (ValaElementAccess* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_container;
}


void vala_element_access_set_container (ValaElementAccess* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (self != NULL);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_container = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL) ? NULL : vala_code_node_ref (_tmp1)), (self->priv->_container == NULL) ? NULL : (self->priv->_container = (vala_code_node_unref (self->priv->_container), NULL)), _tmp2);
	vala_code_node_set_parent_node ((ValaCodeNode*) self->priv->_container, (ValaCodeNode*) self);
}


static void vala_element_access_class_init (ValaElementAccessClass * klass) {
	vala_element_access_parent_class = g_type_class_peek_parent (klass);
	VALA_CODE_NODE_CLASS (klass)->finalize = vala_element_access_finalize;
	g_type_class_add_private (klass, sizeof (ValaElementAccessPrivate));
	VALA_CODE_NODE_CLASS (klass)->accept = vala_element_access_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_element_access_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_element_access_real_replace_expression;
	VALA_EXPRESSION_CLASS (klass)->is_pure = vala_element_access_real_is_pure;
	VALA_CODE_NODE_CLASS (klass)->check = vala_element_access_real_check;
	VALA_CODE_NODE_CLASS (klass)->get_defined_variables = vala_element_access_real_get_defined_variables;
	VALA_CODE_NODE_CLASS (klass)->get_used_variables = vala_element_access_real_get_used_variables;
}


static void vala_element_access_instance_init (ValaElementAccess * self) {
	self->priv = VALA_ELEMENT_ACCESS_GET_PRIVATE (self);
	self->priv->indices = (GeeList*) gee_array_list_new (VALA_TYPE_EXPRESSION, (GBoxedCopyFunc) vala_code_node_ref, vala_code_node_unref, g_direct_equal);
}


static void vala_element_access_finalize (ValaCodeNode* obj) {
	ValaElementAccess * self;
	self = VALA_ELEMENT_ACCESS (obj);
	(self->priv->indices == NULL) ? NULL : (self->priv->indices = (gee_collection_object_unref (self->priv->indices), NULL));
	(self->priv->_container == NULL) ? NULL : (self->priv->_container = (vala_code_node_unref (self->priv->_container), NULL));
	VALA_CODE_NODE_CLASS (vala_element_access_parent_class)->finalize (obj);
}


GType vala_element_access_get_type (void) {
	static GType vala_element_access_type_id = 0;
	if (vala_element_access_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaElementAccessClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_element_access_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaElementAccess), 0, (GInstanceInitFunc) vala_element_access_instance_init, NULL };
		vala_element_access_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaElementAccess", &g_define_type_info, 0);
	}
	return vala_element_access_type_id;
}




