/* valagidlparser.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <vapigen/valagidlparser.h>
#include <vala/valasourcereference.h>
#include <vala/valatypesymbol.h>
#include <gee/map.h>
#include <gee/set.h>
#include <gee/hashmap.h>
#include <stdlib.h>
#include <string.h>
#include <vala/valacodenode.h>
#include <glib/gstdio.h>
#include <vala/valareport.h>
#include <gidlparser.h>
#include <gidlmodule.h>
#include <stdio.h>
#include <vala/valasymbol.h>
#include <vala/valascope.h>
#include <gidlnode.h>
#include <vala/valaerrordomain.h>
#include <vala/valaconstant.h>
#include <vala/valamethod.h>
#include <vala/valadatatype.h>
#include <vala/valaformalparameter.h>
#include <vala/valafield.h>
#include <vala/valaunresolvedtype.h>
#include <vala/valaunresolvedsymbol.h>
#include <vala/valaenumvalue.h>
#include <vala/valaerrorcode.h>
#include <gee/hashset.h>
#include <vala/valaproperty.h>
#include <vala/valasignal.h>
#include <gee/collection.h>
#include <gmetadata.h>
#include <vala/valacreationmethod.h>
#include <vala/valavoidtype.h>
#include <float.h>
#include <math.h>
#include <vala/valaattribute.h>
#include <vala/valapropertyaccessor.h>
#include <vala/valablock.h>
#include <vala/valaexpression.h>

struct _ValaGIdlParserPrivate {
	ValaCodeContext* context;
	ValaSourceFile* current_source_file;
	ValaSourceReference* current_source_reference;
	ValaNamespace* current_namespace;
	ValaTypesymbol* current_data_type;
	GeeMap* codenode_attributes_map;
	GeeMap* codenode_attributes_patterns;
	GeeSet* current_type_symbol_set;
	GeeMap* cname_type_map;
};
#define VALA_GIDL_PARSER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_GIDL_PARSER, ValaGIdlParserPrivate))
enum  {
	VALA_GIDL_PARSER_DUMMY_PROPERTY
};
static void vala_gidl_parser_real_visit_namespace (ValaCodeVisitor* base, ValaNamespace* ns);
static void vala_gidl_parser_real_visit_class (ValaCodeVisitor* base, ValaClass* cl);
static void vala_gidl_parser_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st);
static void vala_gidl_parser_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface);
static void vala_gidl_parser_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en);
static void vala_gidl_parser_real_visit_delegate (ValaCodeVisitor* base, ValaDelegate* d);
static void vala_gidl_parser_visit_type (ValaGIdlParser* self, ValaTypesymbol* t);
static void vala_gidl_parser_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file);
static void vala_gidl_parser_parse_file (ValaGIdlParser* self, ValaSourceFile* source_file);
static char* vala_gidl_parser_fix_type_name (ValaGIdlParser* self, const char* type_name, ValaNamespace* ns);
static char* vala_gidl_parser_fix_const_name (ValaGIdlParser* self, const char* const_name, ValaNamespace* ns);
static ValaNamespace* vala_gidl_parser_parse_module (ValaGIdlParser* self, GIdlModule* module);
static ValaDelegate* vala_gidl_parser_parse_delegate (ValaGIdlParser* self, GIdlNodeFunction* f_node);
static gboolean vala_gidl_parser_is_reference_type (ValaGIdlParser* self, const char* cname);
static void vala_gidl_parser_parse_struct (ValaGIdlParser* self, GIdlNodeStruct* st_node, ValaNamespace* ns, GIdlModule* module);
static void vala_gidl_parser_parse_union (ValaGIdlParser* self, GIdlNodeUnion* un_node, ValaNamespace* ns, GIdlModule* module);
static void vala_gidl_parser_parse_boxed (ValaGIdlParser* self, GIdlNodeBoxed* boxed_node, ValaNamespace* ns, GIdlModule* module);
static ValaTypesymbol* vala_gidl_parser_parse_enum (ValaGIdlParser* self, GIdlNodeEnum* en_node);
static void vala_gidl_parser_parse_object (ValaGIdlParser* self, GIdlNodeInterface* node, ValaNamespace* ns, GIdlModule* module);
static void vala_gidl_parser_parse_interface (ValaGIdlParser* self, GIdlNodeInterface* node, ValaNamespace* ns, GIdlModule* module);
static ValaUnresolvedType* vala_gidl_parser_parse_type (ValaGIdlParser* self, GIdlNodeType* type_node);
static gboolean vala_gidl_parser_is_simple_type (ValaGIdlParser* self, const char* type_name);
static void vala_gidl_parser_parse_type_string (ValaGIdlParser* self, ValaUnresolvedType* type, const char* n);
static ValaUnresolvedType* vala_gidl_parser_parse_param (ValaGIdlParser* self, GIdlNodeParam* param);
static ValaMethod* vala_gidl_parser_create_method (ValaGIdlParser* self, const char* name, const char* symbol, GIdlNodeParam* res, GList* parameters, gboolean is_constructor, gboolean is_interface);
static gboolean vala_gidl_parser_param_is_exception (ValaGIdlParser* self, GIdlNodeParam* param);
static ValaMethod* vala_gidl_parser_parse_function (ValaGIdlParser* self, GIdlNodeFunction* f, gboolean is_interface);
static ValaMethod* vala_gidl_parser_parse_virtual (ValaGIdlParser* self, GIdlNodeVFunc* v, GIdlNodeFunction* func, gboolean is_interface);
static char* vala_gidl_parser_fix_prop_name (ValaGIdlParser* self, const char* name);
static ValaProperty* vala_gidl_parser_parse_property (ValaGIdlParser* self, GIdlNodeProperty* prop_node);
static ValaConstant* vala_gidl_parser_parse_constant (ValaGIdlParser* self, GIdlNodeConstant* const_node);
static ValaField* vala_gidl_parser_parse_field (ValaGIdlParser* self, GIdlNodeField* field_node);
static char** vala_gidl_parser_get_attributes (ValaGIdlParser* self, const char* codenode);
static char* vala_gidl_parser_eval (ValaGIdlParser* self, const char* s);
static ValaSignal* vala_gidl_parser_parse_signal (ValaGIdlParser* self, GIdlNodeSignal* sig_node);
static gpointer vala_gidl_parser_parent_class = NULL;
static void vala_gidl_parser_dispose (GObject * obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static int _vala_strcmp0 (const char * str1, const char * str2);


/**
 * Parse all source files in the specified code context and build a
 * code tree.
 *
 * @param context a code context
 */
void vala_gidl_parser_parse (ValaGIdlParser* self, ValaCodeContext* context) {
	GeeMap* _tmp0;
	ValaCodeContext* _tmp2;
	ValaCodeContext* _tmp1;
	GeeMap* _tmp3;
	g_return_if_fail (VALA_IS_GIDL_PARSER (self));
	g_return_if_fail (context == NULL || VALA_IS_CODE_CONTEXT (context));
	_tmp0 = NULL;
	self->priv->cname_type_map = (_tmp0 = GEE_MAP (gee_hash_map_new (G_TYPE_STRING, ((GBoxedCopyFunc) g_strdup), g_free, VALA_TYPE_TYPESYMBOL, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_str_hash, g_str_equal, g_direct_equal)), (self->priv->cname_type_map == NULL ? NULL : (self->priv->cname_type_map = (g_object_unref (self->priv->cname_type_map), NULL))), _tmp0);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->context = (_tmp2 = (_tmp1 = context, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->context == NULL ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL))), _tmp2);
	vala_code_context_accept (context, VALA_CODE_VISITOR (self));
	_tmp3 = NULL;
	self->priv->cname_type_map = (_tmp3 = NULL, (self->priv->cname_type_map == NULL ? NULL : (self->priv->cname_type_map = (g_object_unref (self->priv->cname_type_map), NULL))), _tmp3);
}


static void vala_gidl_parser_real_visit_namespace (ValaCodeVisitor* base, ValaNamespace* ns) {
	ValaGIdlParser * self;
	self = VALA_GIDL_PARSER (base);
	g_return_if_fail (ns == NULL || VALA_IS_NAMESPACE (ns));
	vala_code_node_accept_children (VALA_CODE_NODE (ns), VALA_CODE_VISITOR (self));
}


static void vala_gidl_parser_real_visit_class (ValaCodeVisitor* base, ValaClass* cl) {
	ValaGIdlParser * self;
	self = VALA_GIDL_PARSER (base);
	g_return_if_fail (cl == NULL || VALA_IS_CLASS (cl));
	vala_gidl_parser_visit_type (self, VALA_TYPESYMBOL (cl));
}


static void vala_gidl_parser_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st) {
	ValaGIdlParser * self;
	self = VALA_GIDL_PARSER (base);
	g_return_if_fail (st == NULL || VALA_IS_STRUCT (st));
	vala_gidl_parser_visit_type (self, VALA_TYPESYMBOL (st));
}


static void vala_gidl_parser_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface) {
	ValaGIdlParser * self;
	self = VALA_GIDL_PARSER (base);
	g_return_if_fail (iface == NULL || VALA_IS_INTERFACE (iface));
	vala_gidl_parser_visit_type (self, VALA_TYPESYMBOL (iface));
}


static void vala_gidl_parser_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en) {
	ValaGIdlParser * self;
	self = VALA_GIDL_PARSER (base);
	g_return_if_fail (en == NULL || VALA_IS_ENUM (en));
	vala_gidl_parser_visit_type (self, VALA_TYPESYMBOL (en));
}


static void vala_gidl_parser_real_visit_delegate (ValaCodeVisitor* base, ValaDelegate* d) {
	ValaGIdlParser * self;
	self = VALA_GIDL_PARSER (base);
	g_return_if_fail (d == NULL || VALA_IS_DELEGATE (d));
	vala_gidl_parser_visit_type (self, VALA_TYPESYMBOL (d));
}


static void vala_gidl_parser_visit_type (ValaGIdlParser* self, ValaTypesymbol* t) {
	char* _tmp0;
	gboolean _tmp1;
	g_return_if_fail (VALA_IS_GIDL_PARSER (self));
	g_return_if_fail (t == NULL || VALA_IS_TYPESYMBOL (t));
	_tmp0 = NULL;
	if ((_tmp1 = !gee_map_contains (self->priv->cname_type_map, (_tmp0 = vala_typesymbol_get_cname (t, FALSE))), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1)) {
		gee_map_set (((GeeMap*) self->priv->cname_type_map), vala_typesymbol_get_cname (t, FALSE), t);
	}
}


static void vala_gidl_parser_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file) {
	ValaGIdlParser * self;
	self = VALA_GIDL_PARSER (base);
	g_return_if_fail (source_file == NULL || VALA_IS_SOURCE_FILE (source_file));
	if (g_str_has_suffix (vala_source_file_get_filename (source_file), ".gi")) {
		vala_gidl_parser_parse_file (self, source_file);
	}
}


static void vala_gidl_parser_parse_file (ValaGIdlParser* self, ValaSourceFile* source_file) {
	GError * inner_error;
	char* _tmp0;
	char* _tmp1;
	char* metadata_filename;
	ValaSourceFile* _tmp3;
	ValaSourceFile* _tmp2;
	GeeMap* _tmp4;
	GeeMap* _tmp5;
	g_return_if_fail (VALA_IS_GIDL_PARSER (self));
	g_return_if_fail (source_file == NULL || VALA_IS_SOURCE_FILE (source_file));
	inner_error = NULL;
	_tmp0 = NULL;
	_tmp1 = NULL;
	metadata_filename = (_tmp1 = g_strdup_printf ("%s.metadata", (_tmp0 = g_strndup (vala_source_file_get_filename (source_file), ((gulong) strlen (vala_source_file_get_filename (source_file)) - strlen (".gi"))))), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->current_source_file = (_tmp3 = (_tmp2 = source_file, (_tmp2 == NULL ? NULL : g_object_ref (_tmp2))), (self->priv->current_source_file == NULL ? NULL : (self->priv->current_source_file = (g_object_unref (self->priv->current_source_file), NULL))), _tmp3);
	_tmp4 = NULL;
	self->priv->codenode_attributes_map = (_tmp4 = GEE_MAP (gee_hash_map_new (G_TYPE_STRING, ((GBoxedCopyFunc) g_strdup), g_free, G_TYPE_STRING, ((GBoxedCopyFunc) g_strdup), g_free, g_str_hash, g_str_equal, g_direct_equal)), (self->priv->codenode_attributes_map == NULL ? NULL : (self->priv->codenode_attributes_map = (g_object_unref (self->priv->codenode_attributes_map), NULL))), _tmp4);
	_tmp5 = NULL;
	self->priv->codenode_attributes_patterns = (_tmp5 = GEE_MAP (gee_hash_map_new (G_TYPE_POINTER, NULL, NULL, G_TYPE_STRING, ((GBoxedCopyFunc) g_strdup), g_free, g_direct_hash, g_pattern_spec_equal, g_direct_equal)), (self->priv->codenode_attributes_patterns == NULL ? NULL : (self->priv->codenode_attributes_patterns = (g_object_unref (self->priv->codenode_attributes_patterns), NULL))), _tmp5);
	if (g_file_test (metadata_filename, G_FILE_TEST_EXISTS)) {
		{
			char* metadata;
			gulong metadata_len;
			gboolean _tmp7;
			char* _tmp6;
			metadata = NULL;
			metadata_len = 0UL;
			_tmp6 = NULL;
			_tmp7 = g_file_get_contents (metadata_filename, &_tmp6, &metadata_len, &inner_error);
			(metadata = (g_free (metadata), NULL));
			metadata = _tmp6;
			_tmp7;
			if (inner_error != NULL) {
				if (inner_error->domain == G_FILE_ERROR) {
					goto __catch0_g_file_error;
				}
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
			}
			{
				char** line_collection;
				char** line_it;
				int line_collection_length1;
				line_collection = g_strsplit (metadata, "\n", 0);
				line_collection_length1 = -1;
				for (line_it = line_collection; *line_it != NULL; line_it = line_it + 1) {
					const char* _tmp10;
					char* line;
					_tmp10 = NULL;
					line = (_tmp10 = *line_it, (_tmp10 == NULL ? NULL : g_strdup (_tmp10)));
					{
						char** _tmp8;
						gint tokens_length1;
						char** tokens;
						if (g_str_has_prefix (line, "#")) {
							line = (g_free (line), NULL);
							continue;
						}
						/* ignore comment lines*/
						_tmp8 = NULL;
						tokens = (_tmp8 = g_strsplit (line, " ", 2), tokens_length1 = -1, _tmp8);
						if (NULL == tokens[0]) {
							line = (g_free (line), NULL);
							tokens = (_vala_array_free (tokens, tokens_length1, ((GDestroyNotify) g_free)), NULL);
							continue;
						}
						if (NULL != g_utf8_strchr (tokens[0], ((glong) -1), ((gunichar) '*'))) {
							GPatternSpec* pattern;
							GPatternSpec* _tmp9;
							pattern = g_pattern_spec_new (tokens[0]);
							_tmp9 = NULL;
							gee_map_set (((GeeMap*) self->priv->codenode_attributes_patterns), (_tmp9 = pattern, pattern = NULL, _tmp9), tokens[0]);
							(pattern == NULL ? NULL : (pattern = (g_pattern_spec_free (pattern), NULL)));
						}
						gee_map_set (((GeeMap*) self->priv->codenode_attributes_map), tokens[0], tokens[1]);
						line = (g_free (line), NULL);
						tokens = (_vala_array_free (tokens, tokens_length1, ((GDestroyNotify) g_free)), NULL);
					}
				}
				line_collection = (_vala_array_free (line_collection, line_collection_length1, ((GDestroyNotify) g_free)), NULL);
			}
			metadata = (g_free (metadata), NULL);
		}
		goto __finally0;
		__catch0_g_file_error:
		{
			GError * e;
			e = inner_error;
			inner_error = NULL;
			{
				char* _tmp11;
				_tmp11 = NULL;
				vala_report_error (NULL, (_tmp11 = g_strdup_printf ("Unable to read metadata file: %s", e->message)));
				_tmp11 = (g_free (_tmp11), NULL);
			}
		}
		__finally0:
		;
	}
	{
		GList* modules;
		ValaSourceReference* _tmp12;
		modules = g_idl_parse_file (vala_source_file_get_filename (source_file), &inner_error);
		if (inner_error != NULL) {
			if (inner_error->domain == G_MARKUP_ERROR) {
				goto __catch1_g_markup_error;
			}
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
		}
		_tmp12 = NULL;
		self->priv->current_source_reference = (_tmp12 = vala_source_reference_new (source_file, 0, 0, 0, 0), (self->priv->current_source_reference == NULL ? NULL : (self->priv->current_source_reference = (g_object_unref (self->priv->current_source_reference), NULL))), _tmp12);
		{
			GList* module_collection;
			GList* module_it;
			module_collection = modules;
			for (module_it = module_collection; module_it != NULL; module_it = module_it->next) {
				GIdlModule* module;
				module = ((GIdlModule*) module_it->data);
				{
					ValaNamespace* ns;
					ns = vala_gidl_parser_parse_module (self, module);
					if (ns != NULL) {
						vala_namespace_add_namespace (vala_code_context_get_root (self->priv->context), ns);
					}
					(ns == NULL ? NULL : (ns = (g_object_unref (ns), NULL)));
				}
			}
		}
		(modules == NULL ? NULL : (modules = (g_list_foreach (modules, ((GFunc) g_idl_module_free), NULL), g_list_free (modules), NULL)));
	}
	goto __finally1;
	__catch1_g_markup_error:
	{
		GError * e;
		e = inner_error;
		inner_error = NULL;
		{
			fprintf (stdout, "error parsing GIDL file: %s\n", e->message);
		}
	}
	__finally1:
	;
	metadata_filename = (g_free (metadata_filename), NULL);
}


static char* vala_gidl_parser_fix_type_name (ValaGIdlParser* self, const char* type_name, ValaNamespace* ns) {
	char** _tmp0;
	gint attributes_length1;
	char** attributes;
	const char* _tmp12;
	char* _tmp13;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (ns == NULL || VALA_IS_NAMESPACE (ns), NULL);
	_tmp0 = NULL;
	attributes = (_tmp0 = vala_gidl_parser_get_attributes (self, type_name), attributes_length1 = -1, _tmp0);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_it;
			attr_collection = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
				const char* _tmp3;
				char* attr;
				_tmp3 = NULL;
				attr = (_tmp3 = attr_collection[attr_it], (_tmp3 == NULL ? NULL : g_strdup (_tmp3)));
				{
					char** _tmp1;
					gint nv_length1;
					char** nv;
					_tmp1 = NULL;
					nv = (_tmp1 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp1);
					if (_vala_strcmp0 (nv[0], "name") == 0) {
						char* _tmp2;
						_tmp2 = NULL;
						return (_tmp2 = vala_gidl_parser_eval (self, nv[1]), (attr = (g_free (attr), NULL)), (nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL)), (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp2);
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL);
				}
			}
		}
	}
	if (g_str_has_prefix (type_name, vala_symbol_get_name (VALA_SYMBOL (ns)))) {
		const char* _tmp4;
		char* _tmp5;
		_tmp4 = NULL;
		_tmp5 = NULL;
		return (_tmp5 = (_tmp4 = g_utf8_offset_to_pointer (type_name, g_utf8_strlen (vala_symbol_get_name (VALA_SYMBOL (ns)), -1)), (_tmp4 == NULL ? NULL : g_strdup (_tmp4))), (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp5);
	} else {
		if (_vala_strcmp0 (vala_symbol_get_name (VALA_SYMBOL (ns)), "GLib") == 0 && g_str_has_prefix (type_name, "G")) {
			const char* _tmp6;
			char* _tmp7;
			_tmp6 = NULL;
			_tmp7 = NULL;
			return (_tmp7 = (_tmp6 = g_utf8_offset_to_pointer (type_name, ((glong) 1)), (_tmp6 == NULL ? NULL : g_strdup (_tmp6))), (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp7);
		} else {
			char* best_match;
			best_match = NULL;
			{
				GeeList* cprefix_collection;
				GeeIterator* cprefix_it;
				cprefix_collection = vala_namespace_get_cprefixes (ns);
				cprefix_it = gee_iterable_iterator (GEE_ITERABLE (cprefix_collection));
				while (gee_iterator_next (cprefix_it)) {
					char* cprefix;
					cprefix = ((char*) gee_iterator_get (cprefix_it));
					{
						if (g_str_has_prefix (type_name, cprefix)) {
							if (best_match == NULL || g_utf8_strlen (cprefix, -1) > g_utf8_strlen (best_match, -1)) {
								char* _tmp9;
								const char* _tmp8;
								_tmp9 = NULL;
								_tmp8 = NULL;
								best_match = (_tmp9 = (_tmp8 = cprefix, (_tmp8 == NULL ? NULL : g_strdup (_tmp8))), (best_match = (g_free (best_match), NULL)), _tmp9);
							}
						}
						cprefix = (g_free (cprefix), NULL);
					}
				}
				(cprefix_collection == NULL ? NULL : (cprefix_collection = (g_object_unref (cprefix_collection), NULL)));
				(cprefix_it == NULL ? NULL : (cprefix_it = (g_object_unref (cprefix_it), NULL)));
			}
			if (best_match != NULL) {
				const char* _tmp10;
				char* _tmp11;
				_tmp10 = NULL;
				_tmp11 = NULL;
				return (_tmp11 = (_tmp10 = g_utf8_offset_to_pointer (type_name, g_utf8_strlen (best_match, -1)), (_tmp10 == NULL ? NULL : g_strdup (_tmp10))), (best_match = (g_free (best_match), NULL)), (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp11);
			}
			best_match = (g_free (best_match), NULL);
		}
	}
	_tmp12 = NULL;
	_tmp13 = NULL;
	return (_tmp13 = (_tmp12 = type_name, (_tmp12 == NULL ? NULL : g_strdup (_tmp12))), (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp13);
}


static char* vala_gidl_parser_fix_const_name (ValaGIdlParser* self, const char* const_name, ValaNamespace* ns) {
	char* _tmp1;
	char* _tmp0;
	gboolean _tmp2;
	const char* _tmp7;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	g_return_val_if_fail (ns == NULL || VALA_IS_NAMESPACE (ns), NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	if ((_tmp2 = g_str_has_prefix (const_name, (_tmp1 = g_strconcat ((_tmp0 = g_utf8_strup (vala_symbol_get_name (VALA_SYMBOL (ns)), -1)), "_", NULL))), (_tmp1 = (g_free (_tmp1), NULL)), (_tmp0 = (g_free (_tmp0), NULL)), _tmp2)) {
		const char* _tmp3;
		_tmp3 = NULL;
		return (_tmp3 = g_utf8_offset_to_pointer (const_name, g_utf8_strlen (vala_symbol_get_name (VALA_SYMBOL (ns)), -1) + 1), (_tmp3 == NULL ? NULL : g_strdup (_tmp3)));
	} else {
		if (_vala_strcmp0 (vala_symbol_get_name (VALA_SYMBOL (ns)), "GLib") == 0 && g_str_has_prefix (const_name, "G_")) {
			const char* _tmp5;
			_tmp5 = NULL;
			return (_tmp5 = g_utf8_offset_to_pointer (const_name, ((glong) 2)), (_tmp5 == NULL ? NULL : g_strdup (_tmp5)));
		}
	}
	_tmp7 = NULL;
	return (_tmp7 = const_name, (_tmp7 == NULL ? NULL : g_strdup (_tmp7)));
}


static ValaNamespace* vala_gidl_parser_parse_module (ValaGIdlParser* self, GIdlModule* module) {
	ValaSymbol* sym;
	ValaNamespace* ns;
	ValaNamespace* _tmp4;
	ValaNamespace* _tmp3;
	char** _tmp5;
	gint attributes_length1;
	char** attributes;
	ValaNamespace* _tmp21;
	ValaNamespace* _tmp23;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	sym = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (vala_code_context_get_root (self->priv->context))), module->name);
	ns = NULL;
	if (VALA_IS_NAMESPACE (sym)) {
		ValaNamespace* _tmp1;
		ValaNamespace* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		ns = (_tmp1 = (_tmp0 = VALA_NAMESPACE (sym), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0))), (ns == NULL ? NULL : (ns = (g_object_unref (ns), NULL))), _tmp1);
		vala_namespace_set_pkg (ns, FALSE);
	} else {
		ValaNamespace* _tmp2;
		_tmp2 = NULL;
		ns = (_tmp2 = vala_namespace_new (module->name, self->priv->current_source_reference), (ns == NULL ? NULL : (ns = (g_object_unref (ns), NULL))), _tmp2);
	}
	_tmp4 = NULL;
	_tmp3 = NULL;
	self->priv->current_namespace = (_tmp4 = (_tmp3 = ns, (_tmp3 == NULL ? NULL : g_object_ref (_tmp3))), (self->priv->current_namespace == NULL ? NULL : (self->priv->current_namespace = (g_object_unref (self->priv->current_namespace), NULL))), _tmp4);
	_tmp5 = NULL;
	attributes = (_tmp5 = vala_gidl_parser_get_attributes (self, vala_symbol_get_name (VALA_SYMBOL (ns))), attributes_length1 = -1, _tmp5);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_it;
			attr_collection = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
				const char* _tmp13;
				char* attr;
				_tmp13 = NULL;
				attr = (_tmp13 = attr_collection[attr_it], (_tmp13 == NULL ? NULL : g_strdup (_tmp13)));
				{
					char** _tmp6;
					gint nv_length1;
					char** nv;
					_tmp6 = NULL;
					nv = (_tmp6 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp6);
					if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
						char* _tmp7;
						_tmp7 = NULL;
						vala_namespace_set_cheader_filename (ns, (_tmp7 = vala_gidl_parser_eval (self, nv[1])));
						_tmp7 = (g_free (_tmp7), NULL);
					} else {
						if (_vala_strcmp0 (nv[0], "cprefix") == 0) {
							char** _tmp10;
							gint cprefixes_length1;
							char* _tmp8;
							char** _tmp9;
							char** cprefixes;
							_tmp10 = NULL;
							_tmp8 = NULL;
							_tmp9 = NULL;
							cprefixes = (_tmp10 = (_tmp9 = g_strsplit ((_tmp8 = vala_gidl_parser_eval (self, nv[1])), ",", 0), (_tmp8 = (g_free (_tmp8), NULL)), _tmp9), cprefixes_length1 = -1, _tmp10);
							{
								char** name_collection;
								int name_it;
								name_collection = cprefixes;
								for (name_it = 0; (cprefixes_length1 != -1 && name_it < cprefixes_length1) || (cprefixes_length1 == -1 && name_collection[name_it] != NULL); name_it = name_it + 1) {
									const char* _tmp11;
									char* name;
									_tmp11 = NULL;
									name = (_tmp11 = name_collection[name_it], (_tmp11 == NULL ? NULL : g_strdup (_tmp11)));
									{
										vala_namespace_add_cprefix (ns, name);
										name = (g_free (name), NULL);
									}
								}
							}
							cprefixes = (_vala_array_free (cprefixes, cprefixes_length1, ((GDestroyNotify) g_free)), NULL);
						} else {
							if (_vala_strcmp0 (nv[0], "lower_case_cprefix") == 0) {
								char* _tmp12;
								_tmp12 = NULL;
								vala_namespace_set_lower_case_cprefix (ns, (_tmp12 = vala_gidl_parser_eval (self, nv[1])));
								_tmp12 = (g_free (_tmp12), NULL);
							}
						}
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL);
				}
			}
		}
	}
	{
		GList* node_collection;
		GList* node_it;
		node_collection = module->entries;
		for (node_it = node_collection; node_it != NULL; node_it = node_it->next) {
			GIdlNode* node;
			node = ((GIdlNode*) node_it->data);
			{
				if (node->type == G_IDL_NODE_CALLBACK) {
					ValaDelegate* cb;
					char* _tmp14;
					cb = vala_gidl_parser_parse_delegate (self, ((GIdlNodeFunction*) node));
					if (cb == NULL) {
						(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
						continue;
					}
					_tmp14 = NULL;
					vala_symbol_set_name (VALA_SYMBOL (cb), (_tmp14 = vala_gidl_parser_fix_type_name (self, vala_symbol_get_name (VALA_SYMBOL (cb)), ns)));
					_tmp14 = (g_free (_tmp14), NULL);
					vala_namespace_add_delegate (ns, cb);
					vala_source_file_add_node (self->priv->current_source_file, VALA_CODE_NODE (cb));
					(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
				} else {
					if (node->type == G_IDL_NODE_STRUCT) {
						vala_gidl_parser_parse_struct (self, ((GIdlNodeStruct*) node), ns, module);
					} else {
						if (node->type == G_IDL_NODE_UNION) {
							vala_gidl_parser_parse_union (self, ((GIdlNodeUnion*) node), ns, module);
						} else {
							if (node->type == G_IDL_NODE_BOXED) {
								vala_gidl_parser_parse_boxed (self, ((GIdlNodeBoxed*) node), ns, module);
							} else {
								if (node->type == G_IDL_NODE_ENUM) {
									ValaTypesymbol* en;
									char* _tmp15;
									en = vala_gidl_parser_parse_enum (self, ((GIdlNodeEnum*) node));
									if (en == NULL) {
										(en == NULL ? NULL : (en = (g_object_unref (en), NULL)));
										continue;
									}
									_tmp15 = NULL;
									vala_symbol_set_name (VALA_SYMBOL (en), (_tmp15 = vala_gidl_parser_fix_type_name (self, vala_symbol_get_name (VALA_SYMBOL (en)), ns)));
									_tmp15 = (g_free (_tmp15), NULL);
									if (VALA_IS_ERROR_DOMAIN (en)) {
										ValaTypesymbol* _tmp16;
										_tmp16 = NULL;
										vala_namespace_add_error_domain (ns, (_tmp16 = en, (VALA_IS_ERROR_DOMAIN (_tmp16) ? ((ValaErrorDomain*) _tmp16) : NULL)));
									} else {
										ValaTypesymbol* _tmp17;
										_tmp17 = NULL;
										vala_namespace_add_enum (ns, (_tmp17 = en, (VALA_IS_ENUM (_tmp17) ? ((ValaEnum*) _tmp17) : NULL)));
									}
									vala_source_file_add_node (self->priv->current_source_file, VALA_CODE_NODE (en));
									(en == NULL ? NULL : (en = (g_object_unref (en), NULL)));
								} else {
									if (node->type == G_IDL_NODE_FLAGS) {
										ValaTypesymbol* _tmp18;
										ValaEnum* en;
										char* _tmp19;
										_tmp18 = NULL;
										en = (_tmp18 = vala_gidl_parser_parse_enum (self, ((GIdlNodeEnum*) node)), (VALA_IS_ENUM (_tmp18) ? ((ValaEnum*) _tmp18) : NULL));
										if (en == NULL) {
											(en == NULL ? NULL : (en = (g_object_unref (en), NULL)));
											continue;
										}
										_tmp19 = NULL;
										vala_symbol_set_name (VALA_SYMBOL (en), (_tmp19 = vala_gidl_parser_fix_type_name (self, vala_symbol_get_name (VALA_SYMBOL (en)), ns)));
										_tmp19 = (g_free (_tmp19), NULL);
										vala_enum_set_is_flags (en, TRUE);
										vala_namespace_add_enum (ns, en);
										vala_source_file_add_node (self->priv->current_source_file, VALA_CODE_NODE (en));
										(en == NULL ? NULL : (en = (g_object_unref (en), NULL)));
									} else {
										if (node->type == G_IDL_NODE_OBJECT) {
											vala_gidl_parser_parse_object (self, ((GIdlNodeInterface*) node), ns, module);
										} else {
											if (node->type == G_IDL_NODE_INTERFACE) {
												vala_gidl_parser_parse_interface (self, ((GIdlNodeInterface*) node), ns, module);
											} else {
												if (node->type == G_IDL_NODE_CONSTANT) {
													ValaConstant* c;
													char* _tmp20;
													c = vala_gidl_parser_parse_constant (self, ((GIdlNodeConstant*) node));
													_tmp20 = NULL;
													vala_symbol_set_name (VALA_SYMBOL (c), (_tmp20 = vala_gidl_parser_fix_const_name (self, vala_symbol_get_name (VALA_SYMBOL (c)), ns)));
													_tmp20 = (g_free (_tmp20), NULL);
													vala_namespace_add_constant (ns, c);
													vala_source_file_add_node (self->priv->current_source_file, VALA_CODE_NODE (c));
													(c == NULL ? NULL : (c = (g_object_unref (c), NULL)));
												} else {
													if (node->type == G_IDL_NODE_FUNCTION) {
														ValaMethod* m;
														m = vala_gidl_parser_parse_function (self, ((GIdlNodeFunction*) node), FALSE);
														if (m != NULL) {
															vala_method_set_instance (m, FALSE);
															vala_namespace_add_method (ns, m);
															vala_source_file_add_node (self->priv->current_source_file, VALA_CODE_NODE (m));
														}
														(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
													}
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}
	_tmp21 = NULL;
	self->priv->current_namespace = (_tmp21 = NULL, (self->priv->current_namespace == NULL ? NULL : (self->priv->current_namespace = (g_object_unref (self->priv->current_namespace), NULL))), _tmp21);
	if (VALA_IS_NAMESPACE (sym)) {
		gpointer _tmp22;
		return (_tmp22 = NULL, (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), (ns == NULL ? NULL : (ns = (g_object_unref (ns), NULL))), (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp22);
	}
	_tmp23 = NULL;
	return (_tmp23 = ns, (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp23);
}


static ValaDelegate* vala_gidl_parser_parse_delegate (ValaGIdlParser* self, GIdlNodeFunction* f_node) {
	GIdlNode* node;
	ValaUnresolvedType* _tmp0;
	ValaDelegate* _tmp1;
	ValaDelegate* cb;
	char** _tmp2;
	gint attributes_length1;
	char** attributes;
	ValaDelegate* _tmp13;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	node = ((GIdlNode*) f_node);
	_tmp0 = NULL;
	_tmp1 = NULL;
	cb = (_tmp1 = vala_delegate_new (node->name, VALA_DATA_TYPE ((_tmp0 = vala_gidl_parser_parse_param (self, f_node->result))), self->priv->current_source_reference), (_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL))), _tmp1);
	vala_symbol_set_access (VALA_SYMBOL (cb), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	_tmp2 = NULL;
	attributes = (_tmp2 = vala_gidl_parser_get_attributes (self, node->name), attributes_length1 = -1, _tmp2);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_it;
			attr_collection = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
				const char* _tmp8;
				char* attr;
				_tmp8 = NULL;
				attr = (_tmp8 = attr_collection[attr_it], (_tmp8 == NULL ? NULL : g_strdup (_tmp8)));
				{
					char** _tmp3;
					gint nv_length1;
					char** nv;
					_tmp3 = NULL;
					nv = (_tmp3 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp3);
					if (_vala_strcmp0 (nv[0], "hidden") == 0) {
						char* _tmp4;
						gboolean _tmp5;
						_tmp4 = NULL;
						if ((_tmp5 = _vala_strcmp0 ((_tmp4 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp4 = (g_free (_tmp4), NULL)), _tmp5)) {
							gpointer _tmp6;
							return (_tmp6 = NULL, (attr = (g_free (attr), NULL)), (nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL)), (cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL))), (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp6);
						}
					} else {
						if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
							char* _tmp7;
							_tmp7 = NULL;
							vala_typesymbol_add_cheader_filename (VALA_TYPESYMBOL (cb), (_tmp7 = vala_gidl_parser_eval (self, nv[1])));
							_tmp7 = (g_free (_tmp7), NULL);
						}
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL);
				}
			}
		}
	}
	{
		GList* param_collection;
		GList* param_it;
		param_collection = f_node->parameters;
		for (param_it = param_collection; param_it != NULL; param_it = param_it->next) {
			GIdlNodeParam* param;
			param = ((GIdlNodeParam*) param_it->data);
			{
				GIdlNode* param_node;
				param_node = ((GIdlNode*) param);
				if (_vala_strcmp0 (param_node->name, "user_data") == 0) {
					/* hide user_data parameter for instance delegates*/
					vala_delegate_set_instance (cb, TRUE);
				} else {
					const char* _tmp9;
					char* param_name;
					ValaUnresolvedType* _tmp11;
					ValaFormalParameter* _tmp12;
					ValaFormalParameter* p;
					_tmp9 = NULL;
					param_name = (_tmp9 = param_node->name, (_tmp9 == NULL ? NULL : g_strdup (_tmp9)));
					if (_vala_strcmp0 (param_name, "string") == 0) {
						char* _tmp10;
						/* avoid conflict with string type*/
						_tmp10 = NULL;
						param_name = (_tmp10 = g_strdup ("str"), (param_name = (g_free (param_name), NULL)), _tmp10);
					}
					_tmp11 = NULL;
					_tmp12 = NULL;
					p = (_tmp12 = vala_formal_parameter_new (param_name, VALA_DATA_TYPE ((_tmp11 = vala_gidl_parser_parse_param (self, param))), NULL), (_tmp11 == NULL ? NULL : (_tmp11 = (g_object_unref (_tmp11), NULL))), _tmp12);
					vala_delegate_add_parameter (cb, p);
					param_name = (g_free (param_name), NULL);
					(p == NULL ? NULL : (p = (g_object_unref (p), NULL)));
				}
			}
		}
	}
	_tmp13 = NULL;
	return (_tmp13 = cb, (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp13);
}


static gboolean vala_gidl_parser_is_reference_type (ValaGIdlParser* self, const char* cname) {
	char** _tmp0;
	gint st_attributes_length1;
	char** st_attributes;
	gboolean _tmp6;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), FALSE);
	_tmp0 = NULL;
	st_attributes = (_tmp0 = vala_gidl_parser_get_attributes (self, cname), st_attributes_length1 = -1, _tmp0);
	if (st_attributes != NULL) {
		{
			char** attr_collection;
			int attr_it;
			attr_collection = st_attributes;
			for (attr_it = 0; (st_attributes_length1 != -1 && attr_it < st_attributes_length1) || (st_attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
				const char* _tmp5;
				char* attr;
				_tmp5 = NULL;
				attr = (_tmp5 = attr_collection[attr_it], (_tmp5 == NULL ? NULL : g_strdup (_tmp5)));
				{
					char** _tmp1;
					gint nv_length1;
					char** nv;
					char* _tmp2;
					gboolean _tmp3;
					_tmp1 = NULL;
					nv = (_tmp1 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp1);
					_tmp2 = NULL;
					if ((_tmp3 = _vala_strcmp0 (nv[0], "is_value_type") == 0 && _vala_strcmp0 ((_tmp2 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp2 = (g_free (_tmp2), NULL)), _tmp3)) {
						gboolean _tmp4;
						return (_tmp4 = FALSE, (attr = (g_free (attr), NULL)), (nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL)), (st_attributes = (_vala_array_free (st_attributes, st_attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp4);
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL);
				}
			}
		}
	}
	return (_tmp6 = TRUE, (st_attributes = (_vala_array_free (st_attributes, st_attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp6);
}


static void vala_gidl_parser_parse_struct (ValaGIdlParser* self, GIdlNodeStruct* st_node, ValaNamespace* ns, GIdlModule* module) {
	GIdlNode* node;
	char* name;
	g_return_if_fail (VALA_IS_GIDL_PARSER (self));
	g_return_if_fail (ns == NULL || VALA_IS_NAMESPACE (ns));
	node = ((GIdlNode*) st_node);
	if (st_node->deprecated) {
		return;
	}
	name = vala_gidl_parser_fix_type_name (self, node->name, ns);
	if (!vala_gidl_parser_is_reference_type (self, node->name)) {
		ValaSymbol* _tmp0;
		ValaStruct* st;
		ValaTypesymbol* _tmp11;
		ValaStruct* _tmp10;
		ValaTypesymbol* _tmp12;
		_tmp0 = NULL;
		st = (_tmp0 = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (ns)), name), (VALA_IS_STRUCT (_tmp0) ? ((ValaStruct*) _tmp0) : NULL));
		if (st == NULL) {
			ValaStruct* _tmp1;
			char** _tmp2;
			gint st_attributes_length1;
			char** st_attributes;
			_tmp1 = NULL;
			st = (_tmp1 = vala_struct_new (name, self->priv->current_source_reference), (st == NULL ? NULL : (st = (g_object_unref (st), NULL))), _tmp1);
			vala_symbol_set_access (VALA_SYMBOL (st), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
			_tmp2 = NULL;
			st_attributes = (_tmp2 = vala_gidl_parser_get_attributes (self, node->name), st_attributes_length1 = -1, _tmp2);
			if (st_attributes != NULL) {
				{
					char** attr_collection;
					int attr_it;
					attr_collection = st_attributes;
					for (attr_it = 0; (st_attributes_length1 != -1 && attr_it < st_attributes_length1) || (st_attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
						const char* _tmp9;
						char* attr;
						_tmp9 = NULL;
						attr = (_tmp9 = attr_collection[attr_it], (_tmp9 == NULL ? NULL : g_strdup (_tmp9)));
						{
							char** _tmp3;
							gint nv_length1;
							char** nv;
							_tmp3 = NULL;
							nv = (_tmp3 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp3);
							if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
								char* _tmp4;
								_tmp4 = NULL;
								vala_typesymbol_add_cheader_filename (VALA_TYPESYMBOL (st), (_tmp4 = vala_gidl_parser_eval (self, nv[1])));
								_tmp4 = (g_free (_tmp4), NULL);
							} else {
								if (_vala_strcmp0 (nv[0], "hidden") == 0) {
									char* _tmp5;
									gboolean _tmp6;
									_tmp5 = NULL;
									if ((_tmp6 = _vala_strcmp0 ((_tmp5 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp5 = (g_free (_tmp5), NULL)), _tmp6)) {
										attr = (g_free (attr), NULL);
										nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL);
										st_attributes = (_vala_array_free (st_attributes, st_attributes_length1, ((GDestroyNotify) g_free)), NULL);
										(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
										name = (g_free (name), NULL);
										return;
									}
								} else {
									if (_vala_strcmp0 (nv[0], "simple_type") == 0) {
										char* _tmp7;
										gboolean _tmp8;
										_tmp7 = NULL;
										if ((_tmp8 = _vala_strcmp0 ((_tmp7 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp7 = (g_free (_tmp7), NULL)), _tmp8)) {
											vala_struct_set_simple_type (st, TRUE);
										}
									}
								}
							}
							attr = (g_free (attr), NULL);
							nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL);
						}
					}
				}
			}
			vala_namespace_add_struct (ns, st);
			vala_source_file_add_node (self->priv->current_source_file, VALA_CODE_NODE (st));
			st_attributes = (_vala_array_free (st_attributes, st_attributes_length1, ((GDestroyNotify) g_free)), NULL);
		}
		_tmp11 = NULL;
		_tmp10 = NULL;
		self->priv->current_data_type = (_tmp11 = VALA_TYPESYMBOL ((_tmp10 = st, (_tmp10 == NULL ? NULL : g_object_ref (_tmp10)))), (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), _tmp11);
		{
			GList* member_collection;
			GList* member_it;
			member_collection = st_node->members;
			for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
				GIdlNode* member;
				member = ((GIdlNode*) member_it->data);
				{
					if (member->type == G_IDL_NODE_FUNCTION) {
						ValaMethod* m;
						m = vala_gidl_parser_parse_function (self, ((GIdlNodeFunction*) member), FALSE);
						if (m != NULL) {
							vala_struct_add_method (st, m);
						}
						(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
					} else {
						if (member->type == G_IDL_NODE_FIELD) {
							ValaField* f;
							f = vala_gidl_parser_parse_field (self, ((GIdlNodeField*) member));
							if (f != NULL) {
								vala_struct_add_field (st, f);
							}
							(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
						}
					}
				}
			}
		}
		_tmp12 = NULL;
		self->priv->current_data_type = (_tmp12 = NULL, (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), _tmp12);
		(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
	} else {
		ValaSymbol* _tmp13;
		ValaClass* cl;
		ValaTypesymbol* _tmp23;
		ValaClass* _tmp22;
		char* ref_function;
		char* unref_function;
		char* copy_function;
		char* free_function;
		ValaTypesymbol* _tmp32;
		_tmp13 = NULL;
		cl = (_tmp13 = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (ns)), name), (VALA_IS_CLASS (_tmp13) ? ((ValaClass*) _tmp13) : NULL));
		if (cl == NULL) {
			char* base_class;
			ValaClass* _tmp14;
			char** _tmp15;
			gint cl_attributes_length1;
			char** cl_attributes;
			base_class = NULL;
			_tmp14 = NULL;
			cl = (_tmp14 = vala_class_new (name, self->priv->current_source_reference), (cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL))), _tmp14);
			vala_symbol_set_access (VALA_SYMBOL (cl), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
			_tmp15 = NULL;
			cl_attributes = (_tmp15 = vala_gidl_parser_get_attributes (self, node->name), cl_attributes_length1 = -1, _tmp15);
			if (cl_attributes != NULL) {
				{
					char** attr_collection;
					int attr_it;
					attr_collection = cl_attributes;
					for (attr_it = 0; (cl_attributes_length1 != -1 && attr_it < cl_attributes_length1) || (cl_attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
						const char* _tmp21;
						char* attr;
						_tmp21 = NULL;
						attr = (_tmp21 = attr_collection[attr_it], (_tmp21 == NULL ? NULL : g_strdup (_tmp21)));
						{
							char** _tmp16;
							gint nv_length1;
							char** nv;
							_tmp16 = NULL;
							nv = (_tmp16 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp16);
							if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
								char* _tmp17;
								_tmp17 = NULL;
								vala_typesymbol_add_cheader_filename (VALA_TYPESYMBOL (cl), (_tmp17 = vala_gidl_parser_eval (self, nv[1])));
								_tmp17 = (g_free (_tmp17), NULL);
							} else {
								if (_vala_strcmp0 (nv[0], "base_class") == 0) {
									char* _tmp18;
									_tmp18 = NULL;
									base_class = (_tmp18 = vala_gidl_parser_eval (self, nv[1]), (base_class = (g_free (base_class), NULL)), _tmp18);
								} else {
									if (_vala_strcmp0 (nv[0], "hidden") == 0) {
										char* _tmp19;
										gboolean _tmp20;
										_tmp19 = NULL;
										if ((_tmp20 = _vala_strcmp0 ((_tmp19 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp19 = (g_free (_tmp19), NULL)), _tmp20)) {
											attr = (g_free (attr), NULL);
											nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL);
											base_class = (g_free (base_class), NULL);
											cl_attributes = (_vala_array_free (cl_attributes, cl_attributes_length1, ((GDestroyNotify) g_free)), NULL);
											(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
											name = (g_free (name), NULL);
											return;
										}
									}
								}
							}
							attr = (g_free (attr), NULL);
							nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL);
						}
					}
				}
			}
			vala_namespace_add_class (ns, cl);
			vala_source_file_add_node (self->priv->current_source_file, VALA_CODE_NODE (cl));
			if (base_class != NULL) {
				ValaUnresolvedType* parent;
				parent = vala_unresolved_type_new ();
				vala_gidl_parser_parse_type_string (self, parent, base_class);
				vala_class_add_base_type (cl, VALA_DATA_TYPE (parent));
				(parent == NULL ? NULL : (parent = (g_object_unref (parent), NULL)));
			}
			base_class = (g_free (base_class), NULL);
			cl_attributes = (_vala_array_free (cl_attributes, cl_attributes_length1, ((GDestroyNotify) g_free)), NULL);
		}
		_tmp23 = NULL;
		_tmp22 = NULL;
		self->priv->current_data_type = (_tmp23 = VALA_TYPESYMBOL ((_tmp22 = cl, (_tmp22 == NULL ? NULL : g_object_ref (_tmp22)))), (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), _tmp23);
		ref_function = NULL;
		unref_function = NULL;
		copy_function = NULL;
		free_function = NULL;
		{
			GList* member_collection;
			GList* member_it;
			member_collection = st_node->members;
			for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
				GIdlNode* member;
				member = ((GIdlNode*) member_it->data);
				{
					if (member->type == G_IDL_NODE_FUNCTION) {
						if (_vala_strcmp0 (member->name, "ref") == 0) {
							char* _tmp25;
							const char* _tmp24;
							_tmp25 = NULL;
							_tmp24 = NULL;
							ref_function = (_tmp25 = (_tmp24 = (((GIdlNodeFunction*) member))->symbol, (_tmp24 == NULL ? NULL : g_strdup (_tmp24))), (ref_function = (g_free (ref_function), NULL)), _tmp25);
						} else {
							if (_vala_strcmp0 (member->name, "unref") == 0) {
								char* _tmp27;
								const char* _tmp26;
								_tmp27 = NULL;
								_tmp26 = NULL;
								unref_function = (_tmp27 = (_tmp26 = (((GIdlNodeFunction*) member))->symbol, (_tmp26 == NULL ? NULL : g_strdup (_tmp26))), (unref_function = (g_free (unref_function), NULL)), _tmp27);
							} else {
								if (_vala_strcmp0 (member->name, "free") == 0 || _vala_strcmp0 (member->name, "destroy") == 0) {
									char* _tmp29;
									const char* _tmp28;
									_tmp29 = NULL;
									_tmp28 = NULL;
									free_function = (_tmp29 = (_tmp28 = (((GIdlNodeFunction*) member))->symbol, (_tmp28 == NULL ? NULL : g_strdup (_tmp28))), (free_function = (g_free (free_function), NULL)), _tmp29);
								} else {
									ValaMethod* m;
									if (_vala_strcmp0 (member->name, "copy") == 0) {
										char* _tmp31;
										const char* _tmp30;
										_tmp31 = NULL;
										_tmp30 = NULL;
										copy_function = (_tmp31 = (_tmp30 = (((GIdlNodeFunction*) member))->symbol, (_tmp30 == NULL ? NULL : g_strdup (_tmp30))), (copy_function = (g_free (copy_function), NULL)), _tmp31);
									}
									m = vala_gidl_parser_parse_function (self, ((GIdlNodeFunction*) member), FALSE);
									if (m != NULL) {
										vala_class_add_method (cl, m);
									}
									(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
								}
							}
						}
					} else {
						if (member->type == G_IDL_NODE_FIELD) {
							ValaField* f;
							f = vala_gidl_parser_parse_field (self, ((GIdlNodeField*) member));
							if (f != NULL) {
								vala_class_add_field (cl, f);
							}
							(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
						}
					}
				}
			}
		}
		if (ref_function != NULL) {
			vala_class_set_ref_function (cl, ref_function);
		}
		if (copy_function != NULL) {
			vala_class_set_dup_function (cl, copy_function);
		}
		if (unref_function != NULL) {
			vala_class_set_unref_function (cl, unref_function);
		} else {
			if (free_function != NULL) {
				vala_class_set_free_function (cl, free_function);
			}
		}
		_tmp32 = NULL;
		self->priv->current_data_type = (_tmp32 = NULL, (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), _tmp32);
		(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
		ref_function = (g_free (ref_function), NULL);
		unref_function = (g_free (unref_function), NULL);
		copy_function = (g_free (copy_function), NULL);
		free_function = (g_free (free_function), NULL);
	}
	name = (g_free (name), NULL);
}


static void vala_gidl_parser_parse_union (ValaGIdlParser* self, GIdlNodeUnion* un_node, ValaNamespace* ns, GIdlModule* module) {
	GIdlNode* node;
	char* name;
	g_return_if_fail (VALA_IS_GIDL_PARSER (self));
	g_return_if_fail (ns == NULL || VALA_IS_NAMESPACE (ns));
	node = ((GIdlNode*) un_node);
	if (un_node->deprecated) {
		return;
	}
	name = vala_gidl_parser_fix_type_name (self, node->name, ns);
	if (!vala_gidl_parser_is_reference_type (self, node->name)) {
		ValaSymbol* _tmp0;
		ValaStruct* st;
		ValaTypesymbol* _tmp9;
		ValaStruct* _tmp8;
		ValaTypesymbol* _tmp10;
		_tmp0 = NULL;
		st = (_tmp0 = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (ns)), name), (VALA_IS_STRUCT (_tmp0) ? ((ValaStruct*) _tmp0) : NULL));
		if (st == NULL) {
			ValaStruct* _tmp1;
			char** _tmp2;
			gint st_attributes_length1;
			char** st_attributes;
			_tmp1 = NULL;
			st = (_tmp1 = vala_struct_new (name, self->priv->current_source_reference), (st == NULL ? NULL : (st = (g_object_unref (st), NULL))), _tmp1);
			vala_symbol_set_access (VALA_SYMBOL (st), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
			_tmp2 = NULL;
			st_attributes = (_tmp2 = vala_gidl_parser_get_attributes (self, node->name), st_attributes_length1 = -1, _tmp2);
			if (st_attributes != NULL) {
				{
					char** attr_collection;
					int attr_it;
					attr_collection = st_attributes;
					for (attr_it = 0; (st_attributes_length1 != -1 && attr_it < st_attributes_length1) || (st_attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
						const char* _tmp7;
						char* attr;
						_tmp7 = NULL;
						attr = (_tmp7 = attr_collection[attr_it], (_tmp7 == NULL ? NULL : g_strdup (_tmp7)));
						{
							char** _tmp3;
							gint nv_length1;
							char** nv;
							_tmp3 = NULL;
							nv = (_tmp3 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp3);
							if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
								char* _tmp4;
								_tmp4 = NULL;
								vala_typesymbol_add_cheader_filename (VALA_TYPESYMBOL (st), (_tmp4 = vala_gidl_parser_eval (self, nv[1])));
								_tmp4 = (g_free (_tmp4), NULL);
							} else {
								if (_vala_strcmp0 (nv[0], "hidden") == 0) {
									char* _tmp5;
									gboolean _tmp6;
									_tmp5 = NULL;
									if ((_tmp6 = _vala_strcmp0 ((_tmp5 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp5 = (g_free (_tmp5), NULL)), _tmp6)) {
										attr = (g_free (attr), NULL);
										nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL);
										st_attributes = (_vala_array_free (st_attributes, st_attributes_length1, ((GDestroyNotify) g_free)), NULL);
										(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
										name = (g_free (name), NULL);
										return;
									}
								}
							}
							attr = (g_free (attr), NULL);
							nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL);
						}
					}
				}
			}
			vala_namespace_add_struct (ns, st);
			vala_source_file_add_node (self->priv->current_source_file, VALA_CODE_NODE (st));
			st_attributes = (_vala_array_free (st_attributes, st_attributes_length1, ((GDestroyNotify) g_free)), NULL);
		}
		_tmp9 = NULL;
		_tmp8 = NULL;
		self->priv->current_data_type = (_tmp9 = VALA_TYPESYMBOL ((_tmp8 = st, (_tmp8 == NULL ? NULL : g_object_ref (_tmp8)))), (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), _tmp9);
		{
			GList* member_collection;
			GList* member_it;
			member_collection = un_node->members;
			for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
				GIdlNode* member;
				member = ((GIdlNode*) member_it->data);
				{
					if (member->type == G_IDL_NODE_FUNCTION) {
						ValaMethod* m;
						m = vala_gidl_parser_parse_function (self, ((GIdlNodeFunction*) member), FALSE);
						if (m != NULL) {
							vala_struct_add_method (st, m);
						}
						(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
					} else {
						if (member->type == G_IDL_NODE_FIELD) {
							ValaField* f;
							f = vala_gidl_parser_parse_field (self, ((GIdlNodeField*) member));
							if (f != NULL) {
								vala_struct_add_field (st, f);
							}
							(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
						}
					}
				}
			}
		}
		_tmp10 = NULL;
		self->priv->current_data_type = (_tmp10 = NULL, (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), _tmp10);
		(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
	} else {
		ValaSymbol* _tmp11;
		ValaClass* cl;
		ValaTypesymbol* _tmp20;
		ValaClass* _tmp19;
		char* ref_function;
		char* unref_function;
		char* copy_function;
		char* free_function;
		ValaTypesymbol* _tmp29;
		_tmp11 = NULL;
		cl = (_tmp11 = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (ns)), name), (VALA_IS_CLASS (_tmp11) ? ((ValaClass*) _tmp11) : NULL));
		if (cl == NULL) {
			ValaClass* _tmp12;
			char** _tmp13;
			gint cl_attributes_length1;
			char** cl_attributes;
			_tmp12 = NULL;
			cl = (_tmp12 = vala_class_new (name, self->priv->current_source_reference), (cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL))), _tmp12);
			vala_symbol_set_access (VALA_SYMBOL (cl), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
			_tmp13 = NULL;
			cl_attributes = (_tmp13 = vala_gidl_parser_get_attributes (self, node->name), cl_attributes_length1 = -1, _tmp13);
			if (cl_attributes != NULL) {
				{
					char** attr_collection;
					int attr_it;
					attr_collection = cl_attributes;
					for (attr_it = 0; (cl_attributes_length1 != -1 && attr_it < cl_attributes_length1) || (cl_attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
						const char* _tmp18;
						char* attr;
						_tmp18 = NULL;
						attr = (_tmp18 = attr_collection[attr_it], (_tmp18 == NULL ? NULL : g_strdup (_tmp18)));
						{
							char** _tmp14;
							gint nv_length1;
							char** nv;
							_tmp14 = NULL;
							nv = (_tmp14 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp14);
							if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
								char* _tmp15;
								_tmp15 = NULL;
								vala_typesymbol_add_cheader_filename (VALA_TYPESYMBOL (cl), (_tmp15 = vala_gidl_parser_eval (self, nv[1])));
								_tmp15 = (g_free (_tmp15), NULL);
							} else {
								if (_vala_strcmp0 (nv[0], "hidden") == 0) {
									char* _tmp16;
									gboolean _tmp17;
									_tmp16 = NULL;
									if ((_tmp17 = _vala_strcmp0 ((_tmp16 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp16 = (g_free (_tmp16), NULL)), _tmp17)) {
										attr = (g_free (attr), NULL);
										nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL);
										cl_attributes = (_vala_array_free (cl_attributes, cl_attributes_length1, ((GDestroyNotify) g_free)), NULL);
										(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
										name = (g_free (name), NULL);
										return;
									}
								}
							}
							attr = (g_free (attr), NULL);
							nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL);
						}
					}
				}
			}
			vala_namespace_add_class (ns, cl);
			vala_source_file_add_node (self->priv->current_source_file, VALA_CODE_NODE (cl));
			cl_attributes = (_vala_array_free (cl_attributes, cl_attributes_length1, ((GDestroyNotify) g_free)), NULL);
		}
		_tmp20 = NULL;
		_tmp19 = NULL;
		self->priv->current_data_type = (_tmp20 = VALA_TYPESYMBOL ((_tmp19 = cl, (_tmp19 == NULL ? NULL : g_object_ref (_tmp19)))), (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), _tmp20);
		ref_function = NULL;
		unref_function = NULL;
		copy_function = NULL;
		free_function = NULL;
		{
			GList* member_collection;
			GList* member_it;
			member_collection = un_node->members;
			for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
				GIdlNode* member;
				member = ((GIdlNode*) member_it->data);
				{
					if (member->type == G_IDL_NODE_FUNCTION) {
						if (_vala_strcmp0 (member->name, "ref") == 0) {
							char* _tmp22;
							const char* _tmp21;
							_tmp22 = NULL;
							_tmp21 = NULL;
							ref_function = (_tmp22 = (_tmp21 = (((GIdlNodeFunction*) member))->symbol, (_tmp21 == NULL ? NULL : g_strdup (_tmp21))), (ref_function = (g_free (ref_function), NULL)), _tmp22);
						} else {
							if (_vala_strcmp0 (member->name, "unref") == 0) {
								char* _tmp24;
								const char* _tmp23;
								_tmp24 = NULL;
								_tmp23 = NULL;
								unref_function = (_tmp24 = (_tmp23 = (((GIdlNodeFunction*) member))->symbol, (_tmp23 == NULL ? NULL : g_strdup (_tmp23))), (unref_function = (g_free (unref_function), NULL)), _tmp24);
							} else {
								if (_vala_strcmp0 (member->name, "free") == 0 || _vala_strcmp0 (member->name, "destroy") == 0) {
									char* _tmp26;
									const char* _tmp25;
									_tmp26 = NULL;
									_tmp25 = NULL;
									free_function = (_tmp26 = (_tmp25 = (((GIdlNodeFunction*) member))->symbol, (_tmp25 == NULL ? NULL : g_strdup (_tmp25))), (free_function = (g_free (free_function), NULL)), _tmp26);
								} else {
									ValaMethod* m;
									if (_vala_strcmp0 (member->name, "copy") == 0) {
										char* _tmp28;
										const char* _tmp27;
										_tmp28 = NULL;
										_tmp27 = NULL;
										copy_function = (_tmp28 = (_tmp27 = (((GIdlNodeFunction*) member))->symbol, (_tmp27 == NULL ? NULL : g_strdup (_tmp27))), (copy_function = (g_free (copy_function), NULL)), _tmp28);
									}
									m = vala_gidl_parser_parse_function (self, ((GIdlNodeFunction*) member), FALSE);
									if (m != NULL) {
										vala_class_add_method (cl, m);
									}
									(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
								}
							}
						}
					} else {
						if (member->type == G_IDL_NODE_FIELD) {
							ValaField* f;
							f = vala_gidl_parser_parse_field (self, ((GIdlNodeField*) member));
							if (f != NULL) {
								vala_class_add_field (cl, f);
							}
							(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
						}
					}
				}
			}
		}
		if (ref_function != NULL) {
			vala_class_set_ref_function (cl, ref_function);
		}
		if (copy_function != NULL) {
			vala_class_set_dup_function (cl, copy_function);
		}
		if (unref_function != NULL) {
			vala_class_set_unref_function (cl, unref_function);
		} else {
			if (free_function != NULL) {
				vala_class_set_free_function (cl, free_function);
			}
		}
		_tmp29 = NULL;
		self->priv->current_data_type = (_tmp29 = NULL, (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), _tmp29);
		(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
		ref_function = (g_free (ref_function), NULL);
		unref_function = (g_free (unref_function), NULL);
		copy_function = (g_free (copy_function), NULL);
		free_function = (g_free (free_function), NULL);
	}
	name = (g_free (name), NULL);
}


static void vala_gidl_parser_parse_boxed (ValaGIdlParser* self, GIdlNodeBoxed* boxed_node, ValaNamespace* ns, GIdlModule* module) {
	GIdlNode* node;
	char* name;
	g_return_if_fail (VALA_IS_GIDL_PARSER (self));
	g_return_if_fail (ns == NULL || VALA_IS_NAMESPACE (ns));
	node = ((GIdlNode*) boxed_node);
	name = vala_gidl_parser_fix_type_name (self, node->name, ns);
	if (!vala_gidl_parser_is_reference_type (self, node->name)) {
		ValaSymbol* _tmp0;
		ValaStruct* st;
		ValaTypesymbol* _tmp8;
		ValaStruct* _tmp7;
		ValaTypesymbol* _tmp9;
		_tmp0 = NULL;
		st = (_tmp0 = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (ns)), name), (VALA_IS_STRUCT (_tmp0) ? ((ValaStruct*) _tmp0) : NULL));
		if (st == NULL) {
			ValaStruct* _tmp1;
			char** _tmp2;
			gint st_attributes_length1;
			char** st_attributes;
			char* _tmp6;
			_tmp1 = NULL;
			st = (_tmp1 = vala_struct_new (name, self->priv->current_source_reference), (st == NULL ? NULL : (st = (g_object_unref (st), NULL))), _tmp1);
			vala_symbol_set_access (VALA_SYMBOL (st), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
			_tmp2 = NULL;
			st_attributes = (_tmp2 = vala_gidl_parser_get_attributes (self, node->name), st_attributes_length1 = -1, _tmp2);
			if (st_attributes != NULL) {
				{
					char** attr_collection;
					int attr_it;
					attr_collection = st_attributes;
					for (attr_it = 0; (st_attributes_length1 != -1 && attr_it < st_attributes_length1) || (st_attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
						const char* _tmp5;
						char* attr;
						_tmp5 = NULL;
						attr = (_tmp5 = attr_collection[attr_it], (_tmp5 == NULL ? NULL : g_strdup (_tmp5)));
						{
							char** _tmp3;
							gint nv_length1;
							char** nv;
							_tmp3 = NULL;
							nv = (_tmp3 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp3);
							if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
								char* _tmp4;
								_tmp4 = NULL;
								vala_typesymbol_add_cheader_filename (VALA_TYPESYMBOL (st), (_tmp4 = vala_gidl_parser_eval (self, nv[1])));
								_tmp4 = (g_free (_tmp4), NULL);
							}
							attr = (g_free (attr), NULL);
							nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL);
						}
					}
				}
			}
			vala_namespace_add_struct (ns, st);
			_tmp6 = NULL;
			vala_struct_set_type_id (st, (_tmp6 = vala_typesymbol_get_upper_case_cname (VALA_TYPESYMBOL (st), "TYPE_")));
			_tmp6 = (g_free (_tmp6), NULL);
			vala_source_file_add_node (self->priv->current_source_file, VALA_CODE_NODE (st));
			st_attributes = (_vala_array_free (st_attributes, st_attributes_length1, ((GDestroyNotify) g_free)), NULL);
		}
		_tmp8 = NULL;
		_tmp7 = NULL;
		self->priv->current_data_type = (_tmp8 = VALA_TYPESYMBOL ((_tmp7 = st, (_tmp7 == NULL ? NULL : g_object_ref (_tmp7)))), (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), _tmp8);
		{
			GList* member_collection;
			GList* member_it;
			member_collection = boxed_node->members;
			for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
				GIdlNode* member;
				member = ((GIdlNode*) member_it->data);
				{
					if (member->type == G_IDL_NODE_FUNCTION) {
						ValaMethod* m;
						m = vala_gidl_parser_parse_function (self, ((GIdlNodeFunction*) member), FALSE);
						if (m != NULL) {
							vala_struct_add_method (st, m);
						}
						(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
					} else {
						if (member->type == G_IDL_NODE_FIELD) {
							ValaField* f;
							f = vala_gidl_parser_parse_field (self, ((GIdlNodeField*) member));
							if (f != NULL) {
								vala_struct_add_field (st, f);
							}
							(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
						}
					}
				}
			}
		}
		_tmp9 = NULL;
		self->priv->current_data_type = (_tmp9 = NULL, (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), _tmp9);
		(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
	} else {
		ValaSymbol* _tmp10;
		ValaClass* cl;
		ValaUnresolvedSymbol* _tmp17;
		ValaUnresolvedSymbol* _tmp18;
		ValaUnresolvedSymbol* gboxed_symbol;
		ValaUnresolvedType* _tmp19;
		ValaTypesymbol* _tmp21;
		ValaClass* _tmp20;
		char* ref_function;
		char* unref_function;
		char* copy_function;
		char* free_function;
		ValaTypesymbol* _tmp30;
		_tmp10 = NULL;
		cl = (_tmp10 = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (ns)), name), (VALA_IS_CLASS (_tmp10) ? ((ValaClass*) _tmp10) : NULL));
		if (cl == NULL) {
			ValaClass* _tmp11;
			char** _tmp12;
			gint cl_attributes_length1;
			char** cl_attributes;
			char* _tmp16;
			_tmp11 = NULL;
			cl = (_tmp11 = vala_class_new (name, self->priv->current_source_reference), (cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL))), _tmp11);
			vala_symbol_set_access (VALA_SYMBOL (cl), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
			_tmp12 = NULL;
			cl_attributes = (_tmp12 = vala_gidl_parser_get_attributes (self, node->name), cl_attributes_length1 = -1, _tmp12);
			if (cl_attributes != NULL) {
				{
					char** attr_collection;
					int attr_it;
					attr_collection = cl_attributes;
					for (attr_it = 0; (cl_attributes_length1 != -1 && attr_it < cl_attributes_length1) || (cl_attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
						const char* _tmp15;
						char* attr;
						_tmp15 = NULL;
						attr = (_tmp15 = attr_collection[attr_it], (_tmp15 == NULL ? NULL : g_strdup (_tmp15)));
						{
							char** _tmp13;
							gint nv_length1;
							char** nv;
							_tmp13 = NULL;
							nv = (_tmp13 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp13);
							if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
								char* _tmp14;
								_tmp14 = NULL;
								vala_typesymbol_add_cheader_filename (VALA_TYPESYMBOL (cl), (_tmp14 = vala_gidl_parser_eval (self, nv[1])));
								_tmp14 = (g_free (_tmp14), NULL);
							}
							attr = (g_free (attr), NULL);
							nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL);
						}
					}
				}
			}
			vala_namespace_add_class (ns, cl);
			_tmp16 = NULL;
			vala_class_set_type_id (cl, (_tmp16 = vala_typesymbol_get_upper_case_cname (VALA_TYPESYMBOL (cl), "TYPE_")));
			_tmp16 = (g_free (_tmp16), NULL);
			vala_source_file_add_node (self->priv->current_source_file, VALA_CODE_NODE (cl));
			cl_attributes = (_vala_array_free (cl_attributes, cl_attributes_length1, ((GDestroyNotify) g_free)), NULL);
		}
		_tmp17 = NULL;
		_tmp18 = NULL;
		gboxed_symbol = (_tmp18 = vala_unresolved_symbol_new ((_tmp17 = vala_unresolved_symbol_new (NULL, "GLib", NULL)), "Boxed", NULL), (_tmp17 == NULL ? NULL : (_tmp17 = (g_object_unref (_tmp17), NULL))), _tmp18);
		_tmp19 = NULL;
		vala_class_add_base_type (cl, VALA_DATA_TYPE ((_tmp19 = vala_unresolved_type_new_from_symbol (gboxed_symbol, NULL))));
		(_tmp19 == NULL ? NULL : (_tmp19 = (g_object_unref (_tmp19), NULL)));
		_tmp21 = NULL;
		_tmp20 = NULL;
		self->priv->current_data_type = (_tmp21 = VALA_TYPESYMBOL ((_tmp20 = cl, (_tmp20 == NULL ? NULL : g_object_ref (_tmp20)))), (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), _tmp21);
		ref_function = NULL;
		unref_function = NULL;
		copy_function = NULL;
		free_function = NULL;
		{
			GList* member_collection;
			GList* member_it;
			member_collection = boxed_node->members;
			for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
				GIdlNode* member;
				member = ((GIdlNode*) member_it->data);
				{
					if (member->type == G_IDL_NODE_FUNCTION) {
						if (_vala_strcmp0 (member->name, "ref") == 0) {
							char* _tmp23;
							const char* _tmp22;
							_tmp23 = NULL;
							_tmp22 = NULL;
							ref_function = (_tmp23 = (_tmp22 = (((GIdlNodeFunction*) member))->symbol, (_tmp22 == NULL ? NULL : g_strdup (_tmp22))), (ref_function = (g_free (ref_function), NULL)), _tmp23);
						} else {
							if (_vala_strcmp0 (member->name, "unref") == 0) {
								char* _tmp25;
								const char* _tmp24;
								_tmp25 = NULL;
								_tmp24 = NULL;
								unref_function = (_tmp25 = (_tmp24 = (((GIdlNodeFunction*) member))->symbol, (_tmp24 == NULL ? NULL : g_strdup (_tmp24))), (unref_function = (g_free (unref_function), NULL)), _tmp25);
							} else {
								if (_vala_strcmp0 (member->name, "free") == 0 || _vala_strcmp0 (member->name, "destroy") == 0) {
									char* _tmp27;
									const char* _tmp26;
									_tmp27 = NULL;
									_tmp26 = NULL;
									free_function = (_tmp27 = (_tmp26 = (((GIdlNodeFunction*) member))->symbol, (_tmp26 == NULL ? NULL : g_strdup (_tmp26))), (free_function = (g_free (free_function), NULL)), _tmp27);
								} else {
									ValaMethod* m;
									if (_vala_strcmp0 (member->name, "copy") == 0) {
										char* _tmp29;
										const char* _tmp28;
										_tmp29 = NULL;
										_tmp28 = NULL;
										copy_function = (_tmp29 = (_tmp28 = (((GIdlNodeFunction*) member))->symbol, (_tmp28 == NULL ? NULL : g_strdup (_tmp28))), (copy_function = (g_free (copy_function), NULL)), _tmp29);
									}
									m = vala_gidl_parser_parse_function (self, ((GIdlNodeFunction*) member), FALSE);
									if (m != NULL) {
										vala_class_add_method (cl, m);
									}
									(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
								}
							}
						}
					} else {
						if (member->type == G_IDL_NODE_FIELD) {
							ValaField* f;
							f = vala_gidl_parser_parse_field (self, ((GIdlNodeField*) member));
							if (f != NULL) {
								vala_class_add_field (cl, f);
							}
							(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
						}
					}
				}
			}
		}
		if (ref_function != NULL) {
			vala_class_set_ref_function (cl, ref_function);
		}
		if (copy_function != NULL) {
			vala_class_set_dup_function (cl, copy_function);
		}
		if (unref_function != NULL) {
			vala_class_set_unref_function (cl, unref_function);
		} else {
			if (free_function != NULL) {
				vala_class_set_free_function (cl, free_function);
			}
		}
		_tmp30 = NULL;
		self->priv->current_data_type = (_tmp30 = NULL, (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), _tmp30);
		(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
		(gboxed_symbol == NULL ? NULL : (gboxed_symbol = (g_object_unref (gboxed_symbol), NULL)));
		ref_function = (g_free (ref_function), NULL);
		unref_function = (g_free (unref_function), NULL);
		copy_function = (g_free (copy_function), NULL);
		free_function = (g_free (free_function), NULL);
	}
	name = (g_free (name), NULL);
}


static ValaTypesymbol* vala_gidl_parser_parse_enum (ValaGIdlParser* self, GIdlNodeEnum* en_node) {
	GIdlNode* node;
	ValaEnum* en;
	char* common_prefix;
	gboolean is_errordomain;
	char** _tmp5;
	gint en_attributes_length1;
	char** en_attributes;
	ValaEnum* _tmp18;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	node = ((GIdlNode*) en_node);
	en = vala_enum_new (node->name, self->priv->current_source_reference);
	vala_symbol_set_access (VALA_SYMBOL (en), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	common_prefix = NULL;
	{
		GList* value_collection;
		GList* value_it;
		value_collection = en_node->values;
		for (value_it = value_collection; value_it != NULL; value_it = value_it->next) {
			GIdlNode* value;
			value = ((GIdlNode*) value_it->data);
			{
				if (common_prefix == NULL) {
					char* _tmp1;
					const char* _tmp0;
					_tmp1 = NULL;
					_tmp0 = NULL;
					common_prefix = (_tmp1 = (_tmp0 = value->name, (_tmp0 == NULL ? NULL : g_strdup (_tmp0))), (common_prefix = (g_free (common_prefix), NULL)), _tmp1);
					while (g_utf8_strlen (common_prefix, -1) > 0 && !g_str_has_suffix (common_prefix, "_")) {
						char* _tmp2;
						/* FIXME: could easily be made faster*/
						_tmp2 = NULL;
						common_prefix = (_tmp2 = g_strndup (common_prefix, ((gulong) strlen (common_prefix) - 1)), (common_prefix = (g_free (common_prefix), NULL)), _tmp2);
					}
				} else {
					while (!g_str_has_prefix (value->name, common_prefix)) {
						char* _tmp3;
						_tmp3 = NULL;
						common_prefix = (_tmp3 = g_strndup (common_prefix, ((gulong) strlen (common_prefix) - 1)), (common_prefix = (g_free (common_prefix), NULL)), _tmp3);
					}
				}
				while (g_utf8_strlen (common_prefix, -1) > 0 && (!g_str_has_suffix (common_prefix, "_") || (g_unichar_isdigit (g_utf8_get_char (g_utf8_offset_to_pointer (value->name, strlen (common_prefix))))) && (g_utf8_strlen (value->name, -1) - g_utf8_strlen (common_prefix, -1)) <= 1)) {
					char* _tmp4;
					/* enum values may not consist solely of digits*/
					_tmp4 = NULL;
					common_prefix = (_tmp4 = g_strndup (common_prefix, ((gulong) strlen (common_prefix) - 1)), (common_prefix = (g_free (common_prefix), NULL)), _tmp4);
				}
			}
		}
	}
	is_errordomain = FALSE;
	_tmp5 = NULL;
	en_attributes = (_tmp5 = vala_gidl_parser_get_attributes (self, node->name), en_attributes_length1 = -1, _tmp5);
	if (en_attributes != NULL) {
		{
			char** attr_collection;
			int attr_it;
			attr_collection = en_attributes;
			for (attr_it = 0; (en_attributes_length1 != -1 && attr_it < en_attributes_length1) || (en_attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
				const char* _tmp15;
				char* attr;
				_tmp15 = NULL;
				attr = (_tmp15 = attr_collection[attr_it], (_tmp15 == NULL ? NULL : g_strdup (_tmp15)));
				{
					char** _tmp6;
					gint nv_length1;
					char** nv;
					_tmp6 = NULL;
					nv = (_tmp6 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp6);
					if (_vala_strcmp0 (nv[0], "common_prefix") == 0) {
						char* _tmp7;
						_tmp7 = NULL;
						common_prefix = (_tmp7 = vala_gidl_parser_eval (self, nv[1]), (common_prefix = (g_free (common_prefix), NULL)), _tmp7);
					} else {
						if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
							char* _tmp8;
							_tmp8 = NULL;
							vala_typesymbol_add_cheader_filename (VALA_TYPESYMBOL (en), (_tmp8 = vala_gidl_parser_eval (self, nv[1])));
							_tmp8 = (g_free (_tmp8), NULL);
						} else {
							if (_vala_strcmp0 (nv[0], "hidden") == 0) {
								char* _tmp9;
								gboolean _tmp10;
								_tmp9 = NULL;
								if ((_tmp10 = _vala_strcmp0 ((_tmp9 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp9 = (g_free (_tmp9), NULL)), _tmp10)) {
									gpointer _tmp11;
									return (_tmp11 = NULL, (attr = (g_free (attr), NULL)), (nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL)), (en == NULL ? NULL : (en = (g_object_unref (en), NULL))), (common_prefix = (g_free (common_prefix), NULL)), (en_attributes = (_vala_array_free (en_attributes, en_attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp11);
								}
							} else {
								if (_vala_strcmp0 (nv[0], "rename_to") == 0) {
									char* _tmp12;
									_tmp12 = NULL;
									vala_symbol_set_name (VALA_SYMBOL (en), (_tmp12 = vala_gidl_parser_eval (self, nv[1])));
									_tmp12 = (g_free (_tmp12), NULL);
								} else {
									if (_vala_strcmp0 (nv[0], "errordomain") == 0) {
										char* _tmp13;
										gboolean _tmp14;
										_tmp13 = NULL;
										if ((_tmp14 = _vala_strcmp0 ((_tmp13 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp13 = (g_free (_tmp13), NULL)), _tmp14)) {
											is_errordomain = TRUE;
										}
									}
								}
							}
						}
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL);
				}
			}
		}
	}
	vala_enum_set_cprefix (en, common_prefix);
	{
		GList* value2_collection;
		GList* value2_it;
		value2_collection = en_node->values;
		for (value2_it = value2_collection; value2_it != NULL; value2_it = value2_it->next) {
			GIdlNode* value2;
			value2 = ((GIdlNode*) value2_it->data);
			{
				ValaEnumValue* ev;
				ev = vala_enum_value_new (g_utf8_offset_to_pointer (value2->name, g_utf8_strlen (common_prefix, -1)), NULL);
				vala_enum_add_value (en, ev);
				(ev == NULL ? NULL : (ev = (g_object_unref (ev), NULL)));
			}
		}
	}
	if (is_errordomain) {
		ValaErrorDomain* ed;
		ValaErrorDomain* _tmp17;
		ed = vala_error_domain_new (vala_symbol_get_name (VALA_SYMBOL (en)), NULL);
		vala_symbol_set_access (VALA_SYMBOL (ed), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
		vala_error_domain_set_cprefix (ed, common_prefix);
		{
			GeeCollection* ev_collection;
			GeeIterator* ev_it;
			ev_collection = vala_enum_get_values (en);
			ev_it = gee_iterable_iterator (GEE_ITERABLE (ev_collection));
			while (gee_iterator_next (ev_it)) {
				ValaEnumValue* ev;
				ev = ((ValaEnumValue*) gee_iterator_get (ev_it));
				{
					ValaErrorCode* _tmp16;
					_tmp16 = NULL;
					vala_error_domain_add_code (ed, (_tmp16 = vala_error_code_new (vala_symbol_get_name (VALA_SYMBOL (ev)))));
					(_tmp16 == NULL ? NULL : (_tmp16 = (g_object_unref (_tmp16), NULL)));
					(ev == NULL ? NULL : (ev = (g_object_unref (ev), NULL)));
				}
			}
			(ev_collection == NULL ? NULL : (ev_collection = (g_object_unref (ev_collection), NULL)));
			(ev_it == NULL ? NULL : (ev_it = (g_object_unref (ev_it), NULL)));
		}
		_tmp17 = NULL;
		return VALA_TYPESYMBOL ((_tmp17 = ed, (en == NULL ? NULL : (en = (g_object_unref (en), NULL))), (common_prefix = (g_free (common_prefix), NULL)), (en_attributes = (_vala_array_free (en_attributes, en_attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp17));
	}
	_tmp18 = NULL;
	return VALA_TYPESYMBOL ((_tmp18 = en, (common_prefix = (g_free (common_prefix), NULL)), (en_attributes = (_vala_array_free (en_attributes, en_attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp18));
}


static void vala_gidl_parser_parse_object (ValaGIdlParser* self, GIdlNodeInterface* node, ValaNamespace* ns, GIdlModule* module) {
	char* name;
	char* base_class;
	ValaSymbol* _tmp0;
	ValaClass* cl;
	ValaTypesymbol* _tmp14;
	ValaClass* _tmp13;
	GeeSet* _tmp15;
	GeeHashMap* current_type_func_map;
	GeeHashMap* current_type_vfunc_map;
	ValaTypesymbol* _tmp16;
	GeeSet* _tmp17;
	g_return_if_fail (VALA_IS_GIDL_PARSER (self));
	g_return_if_fail (ns == NULL || VALA_IS_NAMESPACE (ns));
	name = vala_gidl_parser_fix_type_name (self, (((GIdlNode*) node))->name, ns);
	base_class = NULL;
	_tmp0 = NULL;
	cl = (_tmp0 = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (ns)), name), (VALA_IS_CLASS (_tmp0) ? ((ValaClass*) _tmp0) : NULL));
	if (cl == NULL) {
		ValaClass* _tmp1;
		char** _tmp2;
		gint attributes_length1;
		char** attributes;
		_tmp1 = NULL;
		cl = (_tmp1 = vala_class_new (name, self->priv->current_source_reference), (cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL))), _tmp1);
		vala_symbol_set_access (VALA_SYMBOL (cl), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
		_tmp2 = NULL;
		attributes = (_tmp2 = vala_gidl_parser_get_attributes (self, node->gtype_name), attributes_length1 = -1, _tmp2);
		if (attributes != NULL) {
			{
				char** attr_collection;
				int attr_it;
				attr_collection = attributes;
				for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
					const char* _tmp8;
					char* attr;
					_tmp8 = NULL;
					attr = (_tmp8 = attr_collection[attr_it], (_tmp8 == NULL ? NULL : g_strdup (_tmp8)));
					{
						char** _tmp3;
						gint nv_length1;
						char** nv;
						_tmp3 = NULL;
						nv = (_tmp3 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp3);
						if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
							char* _tmp4;
							_tmp4 = NULL;
							vala_typesymbol_add_cheader_filename (VALA_TYPESYMBOL (cl), (_tmp4 = vala_gidl_parser_eval (self, nv[1])));
							_tmp4 = (g_free (_tmp4), NULL);
						} else {
							if (_vala_strcmp0 (nv[0], "base_class") == 0) {
								char* _tmp5;
								_tmp5 = NULL;
								base_class = (_tmp5 = vala_gidl_parser_eval (self, nv[1]), (base_class = (g_free (base_class), NULL)), _tmp5);
							} else {
								if (_vala_strcmp0 (nv[0], "hidden") == 0) {
									char* _tmp6;
									gboolean _tmp7;
									_tmp6 = NULL;
									if ((_tmp7 = _vala_strcmp0 ((_tmp6 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp6 = (g_free (_tmp6), NULL)), _tmp7)) {
										attr = (g_free (attr), NULL);
										nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL);
										attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL);
										name = (g_free (name), NULL);
										base_class = (g_free (base_class), NULL);
										(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
										return;
									}
								}
							}
						}
						attr = (g_free (attr), NULL);
						nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL);
					}
				}
			}
		}
		vala_namespace_add_class (ns, cl);
		vala_source_file_add_node (self->priv->current_source_file, VALA_CODE_NODE (cl));
		attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL);
	}
	if (base_class != NULL) {
		ValaUnresolvedType* parent;
		parent = vala_unresolved_type_new ();
		vala_gidl_parser_parse_type_string (self, parent, base_class);
		vala_class_add_base_type (cl, VALA_DATA_TYPE (parent));
		(parent == NULL ? NULL : (parent = (g_object_unref (parent), NULL)));
	} else {
		if (node->parent != NULL) {
			ValaUnresolvedType* parent;
			parent = vala_unresolved_type_new ();
			vala_gidl_parser_parse_type_string (self, parent, node->parent);
			vala_class_add_base_type (cl, VALA_DATA_TYPE (parent));
			(parent == NULL ? NULL : (parent = (g_object_unref (parent), NULL)));
		} else {
			ValaUnresolvedSymbol* _tmp9;
			ValaUnresolvedSymbol* _tmp10;
			ValaUnresolvedSymbol* gobject_symbol;
			ValaUnresolvedType* _tmp11;
			_tmp9 = NULL;
			_tmp10 = NULL;
			gobject_symbol = (_tmp10 = vala_unresolved_symbol_new ((_tmp9 = vala_unresolved_symbol_new (NULL, "GLib", NULL)), "Object", NULL), (_tmp9 == NULL ? NULL : (_tmp9 = (g_object_unref (_tmp9), NULL))), _tmp10);
			_tmp11 = NULL;
			vala_class_add_base_type (cl, VALA_DATA_TYPE ((_tmp11 = vala_unresolved_type_new_from_symbol (gobject_symbol, NULL))));
			(_tmp11 == NULL ? NULL : (_tmp11 = (g_object_unref (_tmp11), NULL)));
			(gobject_symbol == NULL ? NULL : (gobject_symbol = (g_object_unref (gobject_symbol), NULL)));
		}
	}
	{
		GList* iface_name_collection;
		GList* iface_name_it;
		iface_name_collection = node->interfaces;
		for (iface_name_it = iface_name_collection; iface_name_it != NULL; iface_name_it = iface_name_it->next) {
			const char* _tmp12;
			char* iface_name;
			_tmp12 = NULL;
			iface_name = (_tmp12 = ((char*) iface_name_it->data), (_tmp12 == NULL ? NULL : g_strdup (_tmp12)));
			{
				ValaUnresolvedType* iface;
				iface = vala_unresolved_type_new ();
				vala_gidl_parser_parse_type_string (self, iface, iface_name);
				vala_class_add_base_type (cl, VALA_DATA_TYPE (iface));
				iface_name = (g_free (iface_name), NULL);
				(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
			}
		}
	}
	_tmp14 = NULL;
	_tmp13 = NULL;
	self->priv->current_data_type = (_tmp14 = VALA_TYPESYMBOL ((_tmp13 = cl, (_tmp13 == NULL ? NULL : g_object_ref (_tmp13)))), (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), _tmp14);
	_tmp15 = NULL;
	self->priv->current_type_symbol_set = (_tmp15 = GEE_SET (gee_hash_set_new (G_TYPE_STRING, ((GBoxedCopyFunc) g_strdup), g_free, g_str_hash, g_str_equal)), (self->priv->current_type_symbol_set == NULL ? NULL : (self->priv->current_type_symbol_set = (g_object_unref (self->priv->current_type_symbol_set), NULL))), _tmp15);
	current_type_func_map = gee_hash_map_new (G_TYPE_STRING, ((GBoxedCopyFunc) g_strdup), g_free, G_TYPE_POINTER, NULL, NULL, g_str_hash, g_str_equal, g_direct_equal);
	current_type_vfunc_map = gee_hash_map_new (G_TYPE_STRING, ((GBoxedCopyFunc) g_strdup), g_free, G_TYPE_STRING, ((GBoxedCopyFunc) g_strdup), g_free, g_str_hash, g_str_equal, g_direct_equal);
	{
		GList* member_collection;
		GList* member_it;
		member_collection = node->members;
		for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = ((GIdlNode*) member_it->data);
			{
				if (member->type == G_IDL_NODE_FUNCTION) {
					gee_map_set (GEE_MAP (current_type_func_map), member->name, ((GIdlNodeFunction*) member));
				}
				if (member->type == G_IDL_NODE_VFUNC) {
					gee_map_set (GEE_MAP (current_type_vfunc_map), member->name, "1");
				}
			}
		}
	}
	{
		GList* member_collection;
		GList* member_it;
		member_collection = node->members;
		for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = ((GIdlNode*) member_it->data);
			{
				if (member->type == G_IDL_NODE_FUNCTION) {
					/* Ignore if vfunc (handled below) */
					if (!gee_map_contains (GEE_MAP (current_type_vfunc_map), member->name)) {
						ValaMethod* m;
						m = vala_gidl_parser_parse_function (self, ((GIdlNodeFunction*) member), FALSE);
						if (m != NULL) {
							vala_class_add_method (cl, m);
						}
						(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
					}
				} else {
					if (member->type == G_IDL_NODE_VFUNC) {
						ValaMethod* m;
						m = vala_gidl_parser_parse_virtual (self, ((GIdlNodeVFunc*) member), ((GIdlNodeFunction*) gee_map_get (GEE_MAP (current_type_func_map), member->name)), FALSE);
						if (m != NULL) {
							vala_class_add_method (cl, m);
						}
						(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
					} else {
						if (member->type == G_IDL_NODE_PROPERTY) {
							ValaProperty* prop;
							prop = vala_gidl_parser_parse_property (self, ((GIdlNodeProperty*) member));
							if (prop != NULL) {
								vala_class_add_property (cl, prop, TRUE);
							}
							(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
						} else {
							if (member->type == G_IDL_NODE_SIGNAL) {
								ValaSignal* sig;
								sig = vala_gidl_parser_parse_signal (self, ((GIdlNodeSignal*) member));
								if (sig != NULL) {
									vala_class_add_signal (cl, sig);
								}
								(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
							}
						}
					}
				}
			}
		}
	}
	{
		GList* member_collection;
		GList* member_it;
		member_collection = node->members;
		for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = ((GIdlNode*) member_it->data);
			{
				if (member->type == G_IDL_NODE_FIELD) {
					if (!gee_collection_contains (GEE_COLLECTION (self->priv->current_type_symbol_set), member->name)) {
						ValaField* f;
						f = vala_gidl_parser_parse_field (self, ((GIdlNodeField*) member));
						if (f != NULL) {
							vala_class_add_field (cl, f);
						}
						(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
					}
				}
			}
		}
	}
	{
		GeeCollection* prop_collection;
		GeeIterator* prop_it;
		prop_collection = vala_class_get_properties (cl);
		prop_it = gee_iterable_iterator (GEE_ITERABLE (prop_collection));
		while (gee_iterator_next (prop_it)) {
			ValaProperty* prop;
			prop = ((ValaProperty*) gee_iterator_get (prop_it));
			{
				char* getter;
				char* setter;
				getter = g_strdup_printf ("get_%s", vala_symbol_get_name (VALA_SYMBOL (prop)));
				if (vala_property_get_get_accessor (prop) != NULL && !gee_collection_contains (GEE_COLLECTION (self->priv->current_type_symbol_set), getter)) {
					vala_property_set_no_accessor_method (prop, TRUE);
				}
				setter = g_strdup_printf ("set_%s", vala_symbol_get_name (VALA_SYMBOL (prop)));
				if (vala_property_get_set_accessor (prop) != NULL && !gee_collection_contains (GEE_COLLECTION (self->priv->current_type_symbol_set), setter)) {
					vala_property_set_no_accessor_method (prop, TRUE);
				}
				(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
				getter = (g_free (getter), NULL);
				setter = (g_free (setter), NULL);
			}
		}
		(prop_collection == NULL ? NULL : (prop_collection = (g_object_unref (prop_collection), NULL)));
		(prop_it == NULL ? NULL : (prop_it = (g_object_unref (prop_it), NULL)));
	}
	_tmp16 = NULL;
	self->priv->current_data_type = (_tmp16 = NULL, (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), _tmp16);
	_tmp17 = NULL;
	self->priv->current_type_symbol_set = (_tmp17 = NULL, (self->priv->current_type_symbol_set == NULL ? NULL : (self->priv->current_type_symbol_set = (g_object_unref (self->priv->current_type_symbol_set), NULL))), _tmp17);
	name = (g_free (name), NULL);
	base_class = (g_free (base_class), NULL);
	(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
	(current_type_func_map == NULL ? NULL : (current_type_func_map = (g_object_unref (current_type_func_map), NULL)));
	(current_type_vfunc_map == NULL ? NULL : (current_type_vfunc_map = (g_object_unref (current_type_vfunc_map), NULL)));
}


static void vala_gidl_parser_parse_interface (ValaGIdlParser* self, GIdlNodeInterface* node, ValaNamespace* ns, GIdlModule* module) {
	char* name;
	ValaSymbol* _tmp0;
	ValaInterface* iface;
	ValaTypesymbol* _tmp8;
	ValaInterface* _tmp7;
	GeeHashMap* current_type_func_map;
	GeeHashMap* current_type_vfunc_map;
	ValaTypesymbol* _tmp9;
	g_return_if_fail (VALA_IS_GIDL_PARSER (self));
	g_return_if_fail (ns == NULL || VALA_IS_NAMESPACE (ns));
	name = vala_gidl_parser_fix_type_name (self, node->gtype_name, ns);
	_tmp0 = NULL;
	iface = (_tmp0 = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (ns)), name), (VALA_IS_INTERFACE (_tmp0) ? ((ValaInterface*) _tmp0) : NULL));
	if (iface == NULL) {
		ValaInterface* _tmp1;
		char** _tmp2;
		gint attributes_length1;
		char** attributes;
		_tmp1 = NULL;
		iface = (_tmp1 = vala_interface_new (name, self->priv->current_source_reference), (iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL))), _tmp1);
		vala_symbol_set_access (VALA_SYMBOL (iface), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
		_tmp2 = NULL;
		attributes = (_tmp2 = vala_gidl_parser_get_attributes (self, node->gtype_name), attributes_length1 = -1, _tmp2);
		if (attributes != NULL) {
			{
				char** attr_collection;
				int attr_it;
				attr_collection = attributes;
				for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
					const char* _tmp5;
					char* attr;
					_tmp5 = NULL;
					attr = (_tmp5 = attr_collection[attr_it], (_tmp5 == NULL ? NULL : g_strdup (_tmp5)));
					{
						char** _tmp3;
						gint nv_length1;
						char** nv;
						_tmp3 = NULL;
						nv = (_tmp3 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp3);
						if (_vala_strcmp0 (nv[0], "cheader_filename") == 0) {
							char* _tmp4;
							_tmp4 = NULL;
							vala_typesymbol_add_cheader_filename (VALA_TYPESYMBOL (iface), (_tmp4 = vala_gidl_parser_eval (self, nv[1])));
							_tmp4 = (g_free (_tmp4), NULL);
						}
						attr = (g_free (attr), NULL);
						nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL);
					}
				}
			}
		}
		{
			GList* prereq_name_collection;
			GList* prereq_name_it;
			prereq_name_collection = node->prerequisites;
			for (prereq_name_it = prereq_name_collection; prereq_name_it != NULL; prereq_name_it = prereq_name_it->next) {
				const char* _tmp6;
				char* prereq_name;
				_tmp6 = NULL;
				prereq_name = (_tmp6 = ((char*) prereq_name_it->data), (_tmp6 == NULL ? NULL : g_strdup (_tmp6)));
				{
					ValaUnresolvedType* prereq;
					prereq = vala_unresolved_type_new ();
					vala_gidl_parser_parse_type_string (self, prereq, prereq_name);
					vala_interface_add_prerequisite (iface, VALA_DATA_TYPE (prereq));
					prereq_name = (g_free (prereq_name), NULL);
					(prereq == NULL ? NULL : (prereq = (g_object_unref (prereq), NULL)));
				}
			}
		}
		vala_namespace_add_interface (ns, iface);
		vala_source_file_add_node (self->priv->current_source_file, VALA_CODE_NODE (iface));
		attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL);
	}
	_tmp8 = NULL;
	_tmp7 = NULL;
	self->priv->current_data_type = (_tmp8 = VALA_TYPESYMBOL ((_tmp7 = iface, (_tmp7 == NULL ? NULL : g_object_ref (_tmp7)))), (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), _tmp8);
	current_type_func_map = gee_hash_map_new (G_TYPE_STRING, ((GBoxedCopyFunc) g_strdup), g_free, G_TYPE_POINTER, NULL, NULL, g_str_hash, g_str_equal, g_direct_equal);
	current_type_vfunc_map = gee_hash_map_new (G_TYPE_STRING, ((GBoxedCopyFunc) g_strdup), g_free, G_TYPE_STRING, ((GBoxedCopyFunc) g_strdup), g_free, g_str_hash, g_str_equal, g_direct_equal);
	{
		GList* member_collection;
		GList* member_it;
		member_collection = node->members;
		for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = ((GIdlNode*) member_it->data);
			{
				if (member->type == G_IDL_NODE_FUNCTION) {
					gee_map_set (GEE_MAP (current_type_func_map), member->name, ((GIdlNodeFunction*) member));
				}
				if (member->type == G_IDL_NODE_VFUNC) {
					gee_map_set (GEE_MAP (current_type_vfunc_map), member->name, "1");
				}
			}
		}
	}
	{
		GList* member_collection;
		GList* member_it;
		member_collection = node->members;
		for (member_it = member_collection; member_it != NULL; member_it = member_it->next) {
			GIdlNode* member;
			member = ((GIdlNode*) member_it->data);
			{
				if (member->type == G_IDL_NODE_FUNCTION) {
					/* Ignore if vfunc (handled below) */
					if (!gee_map_contains (GEE_MAP (current_type_vfunc_map), member->name)) {
						ValaMethod* m;
						m = vala_gidl_parser_parse_function (self, ((GIdlNodeFunction*) member), TRUE);
						if (m != NULL) {
							vala_interface_add_method (iface, m);
						}
						(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
					}
				} else {
					if (member->type == G_IDL_NODE_VFUNC) {
						ValaMethod* m;
						m = vala_gidl_parser_parse_virtual (self, ((GIdlNodeVFunc*) member), ((GIdlNodeFunction*) gee_map_get (GEE_MAP (current_type_func_map), member->name)), TRUE);
						if (m != NULL) {
							vala_interface_add_method (iface, m);
						}
						(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
					} else {
						if (member->type == G_IDL_NODE_PROPERTY) {
							ValaProperty* prop;
							prop = vala_gidl_parser_parse_property (self, ((GIdlNodeProperty*) member));
							if (prop != NULL) {
								vala_interface_add_property (iface, prop);
							}
							(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
						} else {
							if (member->type == G_IDL_NODE_SIGNAL) {
								ValaSignal* sig;
								sig = vala_gidl_parser_parse_signal (self, ((GIdlNodeSignal*) member));
								if (sig != NULL) {
									vala_interface_add_signal (iface, sig);
								}
								(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
							}
						}
					}
				}
			}
		}
	}
	_tmp9 = NULL;
	self->priv->current_data_type = (_tmp9 = NULL, (self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL))), _tmp9);
	name = (g_free (name), NULL);
	(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
	(current_type_func_map == NULL ? NULL : (current_type_func_map = (g_object_unref (current_type_func_map), NULL)));
	(current_type_vfunc_map == NULL ? NULL : (current_type_vfunc_map = (g_object_unref (current_type_vfunc_map), NULL)));
}


static ValaUnresolvedType* vala_gidl_parser_parse_type (ValaGIdlParser* self, GIdlNodeType* type_node) {
	ValaUnresolvedType* type;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	type = vala_unresolved_type_new ();
	if (type_node->tag == TYPE_TAG_VOID) {
		if (type_node->is_pointer) {
			ValaUnresolvedSymbol* _tmp0;
			_tmp0 = NULL;
			vala_unresolved_type_set_unresolved_symbol (type, (_tmp0 = vala_unresolved_symbol_new (NULL, "pointer", NULL)));
			(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
		} else {
			ValaUnresolvedSymbol* _tmp1;
			_tmp1 = NULL;
			vala_unresolved_type_set_unresolved_symbol (type, (_tmp1 = vala_unresolved_symbol_new (NULL, "void", NULL)));
			(_tmp1 == NULL ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL)));
		}
	} else {
		if (type_node->tag == TYPE_TAG_BOOLEAN) {
			ValaUnresolvedSymbol* _tmp2;
			_tmp2 = NULL;
			vala_unresolved_type_set_unresolved_symbol (type, (_tmp2 = vala_unresolved_symbol_new (NULL, "bool", NULL)));
			(_tmp2 == NULL ? NULL : (_tmp2 = (g_object_unref (_tmp2), NULL)));
		} else {
			if (type_node->tag == TYPE_TAG_INT8) {
				ValaUnresolvedSymbol* _tmp3;
				_tmp3 = NULL;
				vala_unresolved_type_set_unresolved_symbol (type, (_tmp3 = vala_unresolved_symbol_new (NULL, "char", NULL)));
				(_tmp3 == NULL ? NULL : (_tmp3 = (g_object_unref (_tmp3), NULL)));
			} else {
				if (type_node->tag == TYPE_TAG_UINT8) {
					ValaUnresolvedSymbol* _tmp4;
					_tmp4 = NULL;
					vala_unresolved_type_set_unresolved_symbol (type, (_tmp4 = vala_unresolved_symbol_new (NULL, "uchar", NULL)));
					(_tmp4 == NULL ? NULL : (_tmp4 = (g_object_unref (_tmp4), NULL)));
				} else {
					if (type_node->tag == TYPE_TAG_INT16) {
						ValaUnresolvedSymbol* _tmp5;
						_tmp5 = NULL;
						vala_unresolved_type_set_unresolved_symbol (type, (_tmp5 = vala_unresolved_symbol_new (NULL, "short", NULL)));
						(_tmp5 == NULL ? NULL : (_tmp5 = (g_object_unref (_tmp5), NULL)));
					} else {
						if (type_node->tag == TYPE_TAG_UINT16) {
							ValaUnresolvedSymbol* _tmp6;
							_tmp6 = NULL;
							vala_unresolved_type_set_unresolved_symbol (type, (_tmp6 = vala_unresolved_symbol_new (NULL, "ushort", NULL)));
							(_tmp6 == NULL ? NULL : (_tmp6 = (g_object_unref (_tmp6), NULL)));
						} else {
							if (type_node->tag == TYPE_TAG_INT32) {
								ValaUnresolvedSymbol* _tmp7;
								_tmp7 = NULL;
								vala_unresolved_type_set_unresolved_symbol (type, (_tmp7 = vala_unresolved_symbol_new (NULL, "int", NULL)));
								(_tmp7 == NULL ? NULL : (_tmp7 = (g_object_unref (_tmp7), NULL)));
							} else {
								if (type_node->tag == TYPE_TAG_UINT32) {
									ValaUnresolvedSymbol* _tmp8;
									_tmp8 = NULL;
									vala_unresolved_type_set_unresolved_symbol (type, (_tmp8 = vala_unresolved_symbol_new (NULL, "uint", NULL)));
									(_tmp8 == NULL ? NULL : (_tmp8 = (g_object_unref (_tmp8), NULL)));
								} else {
									if (type_node->tag == TYPE_TAG_INT64) {
										ValaUnresolvedSymbol* _tmp9;
										_tmp9 = NULL;
										vala_unresolved_type_set_unresolved_symbol (type, (_tmp9 = vala_unresolved_symbol_new (NULL, "int64", NULL)));
										(_tmp9 == NULL ? NULL : (_tmp9 = (g_object_unref (_tmp9), NULL)));
									} else {
										if (type_node->tag == TYPE_TAG_UINT64) {
											ValaUnresolvedSymbol* _tmp10;
											_tmp10 = NULL;
											vala_unresolved_type_set_unresolved_symbol (type, (_tmp10 = vala_unresolved_symbol_new (NULL, "uint64", NULL)));
											(_tmp10 == NULL ? NULL : (_tmp10 = (g_object_unref (_tmp10), NULL)));
										} else {
											if (type_node->tag == TYPE_TAG_INT) {
												ValaUnresolvedSymbol* _tmp11;
												_tmp11 = NULL;
												vala_unresolved_type_set_unresolved_symbol (type, (_tmp11 = vala_unresolved_symbol_new (NULL, "int", NULL)));
												(_tmp11 == NULL ? NULL : (_tmp11 = (g_object_unref (_tmp11), NULL)));
											} else {
												if (type_node->tag == TYPE_TAG_UINT) {
													ValaUnresolvedSymbol* _tmp12;
													_tmp12 = NULL;
													vala_unresolved_type_set_unresolved_symbol (type, (_tmp12 = vala_unresolved_symbol_new (NULL, "uint", NULL)));
													(_tmp12 == NULL ? NULL : (_tmp12 = (g_object_unref (_tmp12), NULL)));
												} else {
													if (type_node->tag == TYPE_TAG_LONG) {
														ValaUnresolvedSymbol* _tmp13;
														_tmp13 = NULL;
														vala_unresolved_type_set_unresolved_symbol (type, (_tmp13 = vala_unresolved_symbol_new (NULL, "long", NULL)));
														(_tmp13 == NULL ? NULL : (_tmp13 = (g_object_unref (_tmp13), NULL)));
													} else {
														if (type_node->tag == TYPE_TAG_ULONG) {
															ValaUnresolvedSymbol* _tmp14;
															_tmp14 = NULL;
															vala_unresolved_type_set_unresolved_symbol (type, (_tmp14 = vala_unresolved_symbol_new (NULL, "ulong", NULL)));
															(_tmp14 == NULL ? NULL : (_tmp14 = (g_object_unref (_tmp14), NULL)));
														} else {
															if (type_node->tag == TYPE_TAG_SSIZE) {
																ValaUnresolvedSymbol* _tmp15;
																_tmp15 = NULL;
																vala_unresolved_type_set_unresolved_symbol (type, (_tmp15 = vala_unresolved_symbol_new (NULL, "long", NULL)));
																(_tmp15 == NULL ? NULL : (_tmp15 = (g_object_unref (_tmp15), NULL)));
															} else {
																if (type_node->tag == TYPE_TAG_SIZE) {
																	ValaUnresolvedSymbol* _tmp16;
																	_tmp16 = NULL;
																	vala_unresolved_type_set_unresolved_symbol (type, (_tmp16 = vala_unresolved_symbol_new (NULL, "ulong", NULL)));
																	(_tmp16 == NULL ? NULL : (_tmp16 = (g_object_unref (_tmp16), NULL)));
																} else {
																	if (type_node->tag == TYPE_TAG_FLOAT) {
																		ValaUnresolvedSymbol* _tmp17;
																		_tmp17 = NULL;
																		vala_unresolved_type_set_unresolved_symbol (type, (_tmp17 = vala_unresolved_symbol_new (NULL, "float", NULL)));
																		(_tmp17 == NULL ? NULL : (_tmp17 = (g_object_unref (_tmp17), NULL)));
																	} else {
																		if (type_node->tag == TYPE_TAG_DOUBLE) {
																			ValaUnresolvedSymbol* _tmp18;
																			_tmp18 = NULL;
																			vala_unresolved_type_set_unresolved_symbol (type, (_tmp18 = vala_unresolved_symbol_new (NULL, "double", NULL)));
																			(_tmp18 == NULL ? NULL : (_tmp18 = (g_object_unref (_tmp18), NULL)));
																		} else {
																			if (type_node->tag == TYPE_TAG_UTF8) {
																				ValaUnresolvedSymbol* _tmp19;
																				_tmp19 = NULL;
																				vala_unresolved_type_set_unresolved_symbol (type, (_tmp19 = vala_unresolved_symbol_new (NULL, "string", NULL)));
																				(_tmp19 == NULL ? NULL : (_tmp19 = (g_object_unref (_tmp19), NULL)));
																			} else {
																				if (type_node->tag == TYPE_TAG_FILENAME) {
																					ValaUnresolvedSymbol* _tmp20;
																					_tmp20 = NULL;
																					vala_unresolved_type_set_unresolved_symbol (type, (_tmp20 = vala_unresolved_symbol_new (NULL, "string", NULL)));
																					(_tmp20 == NULL ? NULL : (_tmp20 = (g_object_unref (_tmp20), NULL)));
																				} else {
																					if (type_node->tag == TYPE_TAG_ARRAY) {
																						ValaUnresolvedType* _tmp21;
																						_tmp21 = NULL;
																						type = (_tmp21 = vala_gidl_parser_parse_type (self, type_node->parameter_type1), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), _tmp21);
																						vala_unresolved_type_set_array_rank (type, 1);
																					} else {
																						if (type_node->tag == TYPE_TAG_LIST) {
																							ValaUnresolvedSymbol* _tmp23;
																							ValaUnresolvedSymbol* _tmp22;
																							_tmp23 = NULL;
																							_tmp22 = NULL;
																							vala_unresolved_type_set_unresolved_symbol (type, (_tmp23 = vala_unresolved_symbol_new ((_tmp22 = vala_unresolved_symbol_new (NULL, "GLib", NULL)), "List", NULL)));
																							(_tmp23 == NULL ? NULL : (_tmp23 = (g_object_unref (_tmp23), NULL)));
																							(_tmp22 == NULL ? NULL : (_tmp22 = (g_object_unref (_tmp22), NULL)));
																						} else {
																							if (type_node->tag == TYPE_TAG_SLIST) {
																								ValaUnresolvedSymbol* _tmp25;
																								ValaUnresolvedSymbol* _tmp24;
																								_tmp25 = NULL;
																								_tmp24 = NULL;
																								vala_unresolved_type_set_unresolved_symbol (type, (_tmp25 = vala_unresolved_symbol_new ((_tmp24 = vala_unresolved_symbol_new (NULL, "GLib", NULL)), "SList", NULL)));
																								(_tmp25 == NULL ? NULL : (_tmp25 = (g_object_unref (_tmp25), NULL)));
																								(_tmp24 == NULL ? NULL : (_tmp24 = (g_object_unref (_tmp24), NULL)));
																							} else {
																								if (type_node->tag == TYPE_TAG_HASH) {
																									ValaUnresolvedSymbol* _tmp27;
																									ValaUnresolvedSymbol* _tmp26;
																									_tmp27 = NULL;
																									_tmp26 = NULL;
																									vala_unresolved_type_set_unresolved_symbol (type, (_tmp27 = vala_unresolved_symbol_new ((_tmp26 = vala_unresolved_symbol_new (NULL, "GLib", NULL)), "HashTable", NULL)));
																									(_tmp27 == NULL ? NULL : (_tmp27 = (g_object_unref (_tmp27), NULL)));
																									(_tmp26 == NULL ? NULL : (_tmp26 = (g_object_unref (_tmp26), NULL)));
																								} else {
																									if (type_node->tag == TYPE_TAG_ERROR) {
																										ValaUnresolvedSymbol* _tmp29;
																										ValaUnresolvedSymbol* _tmp28;
																										_tmp29 = NULL;
																										_tmp28 = NULL;
																										vala_unresolved_type_set_unresolved_symbol (type, (_tmp29 = vala_unresolved_symbol_new ((_tmp28 = vala_unresolved_symbol_new (NULL, "GLib", NULL)), "Error", NULL)));
																										(_tmp29 == NULL ? NULL : (_tmp29 = (g_object_unref (_tmp29), NULL)));
																										(_tmp28 == NULL ? NULL : (_tmp28 = (g_object_unref (_tmp28), NULL)));
																									} else {
																										if (type_node->is_interface) {
																											const char* _tmp30;
																											char* n;
																											_tmp30 = NULL;
																											n = (_tmp30 = type_node->interface, (_tmp30 == NULL ? NULL : g_strdup (_tmp30)));
																											if (_vala_strcmp0 (n, "") == 0) {
																												gpointer _tmp31;
																												return (_tmp31 = NULL, (n = (g_free (n), NULL)), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), _tmp31);
																											}
																											if (g_str_has_prefix (n, "const-")) {
																												char* _tmp33;
																												const char* _tmp32;
																												_tmp33 = NULL;
																												_tmp32 = NULL;
																												n = (_tmp33 = (_tmp32 = g_utf8_offset_to_pointer (n, g_utf8_strlen ("const-", -1)), (_tmp32 == NULL ? NULL : g_strdup (_tmp32))), (n = (g_free (n), NULL)), _tmp33);
																											}
																											if (type_node->is_pointer && (_vala_strcmp0 (n, "gchar") == 0 || _vala_strcmp0 (n, "char") == 0)) {
																												ValaUnresolvedSymbol* _tmp34;
																												_tmp34 = NULL;
																												vala_unresolved_type_set_unresolved_symbol (type, (_tmp34 = vala_unresolved_symbol_new (NULL, "string", NULL)));
																												(_tmp34 == NULL ? NULL : (_tmp34 = (g_object_unref (_tmp34), NULL)));
																												if (g_str_has_suffix (type_node->unparsed, "**")) {
																													vala_data_type_set_is_out (VALA_DATA_TYPE (type), TRUE);
																												}
																											} else {
																												if (_vala_strcmp0 (n, "gunichar") == 0) {
																													ValaUnresolvedSymbol* _tmp35;
																													_tmp35 = NULL;
																													vala_unresolved_type_set_unresolved_symbol (type, (_tmp35 = vala_unresolved_symbol_new (NULL, "unichar", NULL)));
																													(_tmp35 == NULL ? NULL : (_tmp35 = (g_object_unref (_tmp35), NULL)));
																												} else {
																													if (_vala_strcmp0 (n, "gchar") == 0) {
																														ValaUnresolvedSymbol* _tmp36;
																														_tmp36 = NULL;
																														vala_unresolved_type_set_unresolved_symbol (type, (_tmp36 = vala_unresolved_symbol_new (NULL, "char", NULL)));
																														(_tmp36 == NULL ? NULL : (_tmp36 = (g_object_unref (_tmp36), NULL)));
																													} else {
																														if (_vala_strcmp0 (n, "guchar") == 0 || _vala_strcmp0 (n, "guint8") == 0) {
																															ValaUnresolvedSymbol* _tmp37;
																															_tmp37 = NULL;
																															vala_unresolved_type_set_unresolved_symbol (type, (_tmp37 = vala_unresolved_symbol_new (NULL, "uchar", NULL)));
																															(_tmp37 == NULL ? NULL : (_tmp37 = (g_object_unref (_tmp37), NULL)));
																															if (type_node->is_pointer) {
																																vala_unresolved_type_set_array_rank (type, 1);
																															}
																														} else {
																															if (_vala_strcmp0 (n, "gushort") == 0) {
																																ValaUnresolvedSymbol* _tmp38;
																																_tmp38 = NULL;
																																vala_unresolved_type_set_unresolved_symbol (type, (_tmp38 = vala_unresolved_symbol_new (NULL, "ushort", NULL)));
																																(_tmp38 == NULL ? NULL : (_tmp38 = (g_object_unref (_tmp38), NULL)));
																															} else {
																																if (_vala_strcmp0 (n, "gshort") == 0) {
																																	ValaUnresolvedSymbol* _tmp39;
																																	_tmp39 = NULL;
																																	vala_unresolved_type_set_unresolved_symbol (type, (_tmp39 = vala_unresolved_symbol_new (NULL, "short", NULL)));
																																	(_tmp39 == NULL ? NULL : (_tmp39 = (g_object_unref (_tmp39), NULL)));
																																} else {
																																	if (_vala_strcmp0 (n, "gconstpointer") == 0 || _vala_strcmp0 (n, "void") == 0) {
																																		ValaUnresolvedSymbol* _tmp40;
																																		_tmp40 = NULL;
																																		vala_unresolved_type_set_unresolved_symbol (type, (_tmp40 = vala_unresolved_symbol_new (NULL, "pointer", NULL)));
																																		(_tmp40 == NULL ? NULL : (_tmp40 = (g_object_unref (_tmp40), NULL)));
																																	} else {
																																		if (_vala_strcmp0 (n, "goffset") == 0 || _vala_strcmp0 (n, "off_t") == 0) {
																																			ValaUnresolvedSymbol* _tmp41;
																																			_tmp41 = NULL;
																																			vala_unresolved_type_set_unresolved_symbol (type, (_tmp41 = vala_unresolved_symbol_new (NULL, "int64", NULL)));
																																			(_tmp41 == NULL ? NULL : (_tmp41 = (g_object_unref (_tmp41), NULL)));
																																		} else {
																																			if (_vala_strcmp0 (n, "value_array") == 0) {
																																				ValaUnresolvedSymbol* _tmp43;
																																				ValaUnresolvedSymbol* _tmp42;
																																				_tmp43 = NULL;
																																				_tmp42 = NULL;
																																				vala_unresolved_type_set_unresolved_symbol (type, (_tmp43 = vala_unresolved_symbol_new ((_tmp42 = vala_unresolved_symbol_new (NULL, "GLib", NULL)), "ValueArray", NULL)));
																																				(_tmp43 == NULL ? NULL : (_tmp43 = (g_object_unref (_tmp43), NULL)));
																																				(_tmp42 == NULL ? NULL : (_tmp42 = (g_object_unref (_tmp42), NULL)));
																																			} else {
																																				if (_vala_strcmp0 (n, "time_t") == 0) {
																																					ValaUnresolvedSymbol* _tmp44;
																																					_tmp44 = NULL;
																																					vala_unresolved_type_set_unresolved_symbol (type, (_tmp44 = vala_unresolved_symbol_new (NULL, "ulong", NULL)));
																																					(_tmp44 == NULL ? NULL : (_tmp44 = (g_object_unref (_tmp44), NULL)));
																																				} else {
																																					if (_vala_strcmp0 (n, "socklen_t") == 0) {
																																						ValaUnresolvedSymbol* _tmp45;
																																						_tmp45 = NULL;
																																						vala_unresolved_type_set_unresolved_symbol (type, (_tmp45 = vala_unresolved_symbol_new (NULL, "uint32", NULL)));
																																						(_tmp45 == NULL ? NULL : (_tmp45 = (g_object_unref (_tmp45), NULL)));
																																					} else {
																																						if (_vala_strcmp0 (n, "mode_t") == 0) {
																																							ValaUnresolvedSymbol* _tmp46;
																																							_tmp46 = NULL;
																																							vala_unresolved_type_set_unresolved_symbol (type, (_tmp46 = vala_unresolved_symbol_new (NULL, "uint", NULL)));
																																							(_tmp46 == NULL ? NULL : (_tmp46 = (g_object_unref (_tmp46), NULL)));
																																						} else {
																																							if (_vala_strcmp0 (n, "gint") == 0 || _vala_strcmp0 (n, "pid_t") == 0) {
																																								ValaUnresolvedSymbol* _tmp47;
																																								_tmp47 = NULL;
																																								vala_unresolved_type_set_unresolved_symbol (type, (_tmp47 = vala_unresolved_symbol_new (NULL, "int", NULL)));
																																								(_tmp47 == NULL ? NULL : (_tmp47 = (g_object_unref (_tmp47), NULL)));
																																							} else {
																																								if (_vala_strcmp0 (n, "unsigned") == 0 || _vala_strcmp0 (n, "unsigned-int") == 0) {
																																									ValaUnresolvedSymbol* _tmp48;
																																									_tmp48 = NULL;
																																									vala_unresolved_type_set_unresolved_symbol (type, (_tmp48 = vala_unresolved_symbol_new (NULL, "uint", NULL)));
																																									(_tmp48 == NULL ? NULL : (_tmp48 = (g_object_unref (_tmp48), NULL)));
																																								} else {
																																									if (_vala_strcmp0 (n, "FILE") == 0) {
																																										ValaUnresolvedSymbol* _tmp50;
																																										ValaUnresolvedSymbol* _tmp49;
																																										_tmp50 = NULL;
																																										_tmp49 = NULL;
																																										vala_unresolved_type_set_unresolved_symbol (type, (_tmp50 = vala_unresolved_symbol_new ((_tmp49 = vala_unresolved_symbol_new (NULL, "GLib", NULL)), "FileStream", NULL)));
																																										(_tmp50 == NULL ? NULL : (_tmp50 = (g_object_unref (_tmp50), NULL)));
																																										(_tmp49 == NULL ? NULL : (_tmp49 = (g_object_unref (_tmp49), NULL)));
																																									} else {
																																										if (_vala_strcmp0 (n, "struct") == 0) {
																																											ValaUnresolvedSymbol* _tmp51;
																																											_tmp51 = NULL;
																																											vala_unresolved_type_set_unresolved_symbol (type, (_tmp51 = vala_unresolved_symbol_new (NULL, "pointer", NULL)));
																																											(_tmp51 == NULL ? NULL : (_tmp51 = (g_object_unref (_tmp51), NULL)));
																																										} else {
																																											if (_vala_strcmp0 (n, "iconv_t") == 0) {
																																												ValaUnresolvedSymbol* _tmp52;
																																												_tmp52 = NULL;
																																												vala_unresolved_type_set_unresolved_symbol (type, (_tmp52 = vala_unresolved_symbol_new (NULL, "pointer", NULL)));
																																												(_tmp52 == NULL ? NULL : (_tmp52 = (g_object_unref (_tmp52), NULL)));
																																											} else {
																																												if (_vala_strcmp0 (n, "GType") == 0) {
																																													ValaUnresolvedSymbol* _tmp54;
																																													ValaUnresolvedSymbol* _tmp53;
																																													_tmp54 = NULL;
																																													_tmp53 = NULL;
																																													vala_unresolved_type_set_unresolved_symbol (type, (_tmp54 = vala_unresolved_symbol_new ((_tmp53 = vala_unresolved_symbol_new (NULL, "GLib", NULL)), "Type", NULL)));
																																													(_tmp54 == NULL ? NULL : (_tmp54 = (g_object_unref (_tmp54), NULL)));
																																													(_tmp53 == NULL ? NULL : (_tmp53 = (g_object_unref (_tmp53), NULL)));
																																													if (type_node->is_pointer) {
																																														vala_unresolved_type_set_array_rank (type, 1);
																																													}
																																												} else {
																																													vala_gidl_parser_parse_type_string (self, type, n);
																																													if (vala_gidl_parser_is_simple_type (self, n)) {
																																														if (type_node->is_pointer) {
																																															vala_data_type_set_is_out (VALA_DATA_TYPE (type), TRUE);
																																														}
																																													} else {
																																														if (g_str_has_suffix (type_node->unparsed, "**")) {
																																															vala_data_type_set_is_out (VALA_DATA_TYPE (type), TRUE);
																																														}
																																													}
																																												}
																																											}
																																										}
																																									}
																																								}
																																							}
																																						}
																																					}
																																				}
																																			}
																																		}
																																	}
																																}
																															}
																														}
																													}
																												}
																											}
																											n = (g_free (n), NULL);
																										} else {
																											fprintf (stdout, "%d\n", type_node->tag);
																										}
																									}
																								}
																							}
																						}
																					}
																				}
																			}
																		}
																	}
																}
															}
														}
													}
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}
	return type;
}


static gboolean vala_gidl_parser_is_simple_type (ValaGIdlParser* self, const char* type_name) {
	ValaTypesymbol* _tmp0;
	ValaStruct* st;
	gboolean _tmp2;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), FALSE);
	_tmp0 = NULL;
	st = (_tmp0 = ((ValaTypesymbol*) gee_map_get (((GeeMap*) self->priv->cname_type_map), type_name)), (VALA_IS_STRUCT (_tmp0) ? ((ValaStruct*) _tmp0) : NULL));
	if (st != NULL && vala_struct_is_simple_type (st)) {
		gboolean _tmp1;
		return (_tmp1 = TRUE, (st == NULL ? NULL : (st = (g_object_unref (st), NULL))), _tmp1);
	}
	return (_tmp2 = FALSE, (st == NULL ? NULL : (st = (g_object_unref (st), NULL))), _tmp2);
}


static void vala_gidl_parser_parse_type_string (ValaGIdlParser* self, ValaUnresolvedType* type, const char* n) {
	ValaTypesymbol* dt;
	char** _tmp2;
	gint type_attributes_length1;
	char** type_attributes;
	char* ns_name;
	g_return_if_fail (VALA_IS_GIDL_PARSER (self));
	g_return_if_fail (type == NULL || VALA_IS_UNRESOLVED_TYPE (type));
	dt = ((ValaTypesymbol*) gee_map_get (((GeeMap*) self->priv->cname_type_map), n));
	if (dt != NULL) {
		ValaUnresolvedSymbol* parent_symbol;
		ValaUnresolvedSymbol* _tmp1;
		parent_symbol = NULL;
		if (vala_symbol_get_name (vala_symbol_get_parent_symbol (VALA_SYMBOL (dt))) != NULL) {
			ValaUnresolvedSymbol* _tmp0;
			_tmp0 = NULL;
			parent_symbol = (_tmp0 = vala_unresolved_symbol_new (NULL, vala_symbol_get_name (vala_symbol_get_parent_symbol (VALA_SYMBOL (dt))), NULL), (parent_symbol == NULL ? NULL : (parent_symbol = (g_object_unref (parent_symbol), NULL))), _tmp0);
		}
		_tmp1 = NULL;
		vala_unresolved_type_set_unresolved_symbol (type, (_tmp1 = vala_unresolved_symbol_new (parent_symbol, vala_symbol_get_name (VALA_SYMBOL (dt)), NULL)));
		(_tmp1 == NULL ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL)));
		(parent_symbol == NULL ? NULL : (parent_symbol = (g_object_unref (parent_symbol), NULL)));
		(dt == NULL ? NULL : (dt = (g_object_unref (dt), NULL)));
		return;
	}
	_tmp2 = NULL;
	type_attributes = (_tmp2 = vala_gidl_parser_get_attributes (self, n), type_attributes_length1 = -1, _tmp2);
	ns_name = NULL;
	if (NULL != type_attributes) {
		{
			char** attr_collection;
			int attr_it;
			attr_collection = type_attributes;
			for (attr_it = 0; (type_attributes_length1 != -1 && attr_it < type_attributes_length1) || (type_attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
				const char* _tmp11;
				char* attr;
				_tmp11 = NULL;
				attr = (_tmp11 = attr_collection[attr_it], (_tmp11 == NULL ? NULL : g_strdup (_tmp11)));
				{
					char** _tmp3;
					gint nv_length1;
					char** nv;
					_tmp3 = NULL;
					nv = (_tmp3 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp3);
					if (_vala_strcmp0 (nv[0], "cprefix") == 0) {
						ValaUnresolvedSymbol* _tmp5;
						char* _tmp4;
						_tmp5 = NULL;
						_tmp4 = NULL;
						vala_unresolved_type_set_unresolved_symbol (type, (_tmp5 = vala_unresolved_symbol_new (NULL, g_utf8_offset_to_pointer (n, g_utf8_strlen ((_tmp4 = vala_gidl_parser_eval (self, nv[1])), -1)), NULL)));
						(_tmp5 == NULL ? NULL : (_tmp5 = (g_object_unref (_tmp5), NULL)));
						_tmp4 = (g_free (_tmp4), NULL);
					} else {
						if (_vala_strcmp0 (nv[0], "name") == 0) {
							ValaUnresolvedSymbol* _tmp7;
							char* _tmp6;
							_tmp7 = NULL;
							_tmp6 = NULL;
							vala_unresolved_type_set_unresolved_symbol (type, (_tmp7 = vala_unresolved_symbol_new (NULL, (_tmp6 = vala_gidl_parser_eval (self, nv[1])), NULL)));
							(_tmp7 == NULL ? NULL : (_tmp7 = (g_object_unref (_tmp7), NULL)));
							_tmp6 = (g_free (_tmp6), NULL);
						} else {
							if (_vala_strcmp0 (nv[0], "namespace") == 0) {
								char* _tmp8;
								_tmp8 = NULL;
								ns_name = (_tmp8 = vala_gidl_parser_eval (self, nv[1]), (ns_name = (g_free (ns_name), NULL)), _tmp8);
							} else {
								if (_vala_strcmp0 (nv[0], "rename_to") == 0) {
									ValaUnresolvedSymbol* _tmp10;
									char* _tmp9;
									_tmp10 = NULL;
									_tmp9 = NULL;
									vala_unresolved_type_set_unresolved_symbol (type, (_tmp10 = vala_unresolved_symbol_new (NULL, (_tmp9 = vala_gidl_parser_eval (self, nv[1])), NULL)));
									(_tmp10 == NULL ? NULL : (_tmp10 = (g_object_unref (_tmp10), NULL)));
									_tmp9 = (g_free (_tmp9), NULL);
								}
							}
						}
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL);
				}
			}
		}
	}
	if (vala_unresolved_type_get_unresolved_symbol (type) != NULL) {
		if (ns_name != NULL) {
			ValaUnresolvedSymbol* _tmp12;
			_tmp12 = NULL;
			vala_unresolved_symbol_set_inner (vala_unresolved_type_get_unresolved_symbol (type), (_tmp12 = vala_unresolved_symbol_new (NULL, ns_name, NULL)));
			(_tmp12 == NULL ? NULL : (_tmp12 = (g_object_unref (_tmp12), NULL)));
		}
		(dt == NULL ? NULL : (dt = (g_object_unref (dt), NULL)));
		type_attributes = (_vala_array_free (type_attributes, type_attributes_length1, ((GDestroyNotify) g_free)), NULL);
		ns_name = (g_free (ns_name), NULL);
		return;
	}
	if (_vala_strcmp0 (n, "va_list") == 0) {
		ValaUnresolvedSymbol* _tmp13;
		/* unsupported*/
		_tmp13 = NULL;
		vala_unresolved_type_set_unresolved_symbol (type, (_tmp13 = vala_unresolved_symbol_new (NULL, "pointer", NULL)));
		(_tmp13 == NULL ? NULL : (_tmp13 = (g_object_unref (_tmp13), NULL)));
	} else {
		if (g_str_has_prefix (n, vala_symbol_get_name (VALA_SYMBOL (self->priv->current_namespace)))) {
			ValaUnresolvedSymbol* _tmp15;
			ValaUnresolvedSymbol* _tmp14;
			_tmp15 = NULL;
			_tmp14 = NULL;
			vala_unresolved_type_set_unresolved_symbol (type, (_tmp15 = vala_unresolved_symbol_new ((_tmp14 = vala_unresolved_symbol_new (NULL, vala_symbol_get_name (VALA_SYMBOL (self->priv->current_namespace)), NULL)), g_utf8_offset_to_pointer (n, g_utf8_strlen (vala_symbol_get_name (VALA_SYMBOL (self->priv->current_namespace)), -1)), NULL)));
			(_tmp15 == NULL ? NULL : (_tmp15 = (g_object_unref (_tmp15), NULL)));
			(_tmp14 == NULL ? NULL : (_tmp14 = (g_object_unref (_tmp14), NULL)));
		} else {
			if (g_str_has_prefix (n, "G")) {
				ValaUnresolvedSymbol* _tmp17;
				ValaUnresolvedSymbol* _tmp16;
				_tmp17 = NULL;
				_tmp16 = NULL;
				vala_unresolved_type_set_unresolved_symbol (type, (_tmp17 = vala_unresolved_symbol_new ((_tmp16 = vala_unresolved_symbol_new (NULL, "GLib", NULL)), g_utf8_offset_to_pointer (n, ((glong) 1)), NULL)));
				(_tmp17 == NULL ? NULL : (_tmp17 = (g_object_unref (_tmp17), NULL)));
				(_tmp16 == NULL ? NULL : (_tmp16 = (g_object_unref (_tmp16), NULL)));
			} else {
				char** _tmp18;
				gint name_parts_length1;
				char** name_parts;
				_tmp18 = NULL;
				name_parts = (_tmp18 = g_strsplit (n, ".", 2), name_parts_length1 = -1, _tmp18);
				if (name_parts[1] == NULL) {
					ValaUnresolvedSymbol* _tmp19;
					_tmp19 = NULL;
					vala_unresolved_type_set_unresolved_symbol (type, (_tmp19 = vala_unresolved_symbol_new (NULL, name_parts[0], NULL)));
					(_tmp19 == NULL ? NULL : (_tmp19 = (g_object_unref (_tmp19), NULL)));
				} else {
					ValaUnresolvedSymbol* _tmp21;
					ValaUnresolvedSymbol* _tmp20;
					_tmp21 = NULL;
					_tmp20 = NULL;
					vala_unresolved_type_set_unresolved_symbol (type, (_tmp21 = vala_unresolved_symbol_new ((_tmp20 = vala_unresolved_symbol_new (NULL, name_parts[0], NULL)), name_parts[1], NULL)));
					(_tmp21 == NULL ? NULL : (_tmp21 = (g_object_unref (_tmp21), NULL)));
					(_tmp20 == NULL ? NULL : (_tmp20 = (g_object_unref (_tmp20), NULL)));
				}
				name_parts = (_vala_array_free (name_parts, name_parts_length1, ((GDestroyNotify) g_free)), NULL);
			}
		}
	}
	(dt == NULL ? NULL : (dt = (g_object_unref (dt), NULL)));
	type_attributes = (_vala_array_free (type_attributes, type_attributes_length1, ((GDestroyNotify) g_free)), NULL);
	ns_name = (g_free (ns_name), NULL);
}


static ValaUnresolvedType* vala_gidl_parser_parse_param (ValaGIdlParser* self, GIdlNodeParam* param) {
	ValaUnresolvedType* type;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	type = vala_gidl_parser_parse_type (self, param->type);
	return type;
}


/* disable for now as null_ok not yet correctly set
 type.non_null = !param.null_ok;*/
static ValaMethod* vala_gidl_parser_create_method (ValaGIdlParser* self, const char* name, const char* symbol, GIdlNodeParam* res, GList* parameters, gboolean is_constructor, gboolean is_interface) {
	ValaUnresolvedType* return_type;
	ValaMethod* m;
	gboolean add_ellipsis;
	gboolean suppress_throws;
	char** _tmp14;
	gint attributes_length1;
	char** attributes;
	gboolean first;
	ValaFormalParameter* last_param;
	ValaUnresolvedType* last_param_type;
	ValaMethod* _tmp63;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	return_type = NULL;
	if (res != NULL) {
		ValaUnresolvedType* _tmp0;
		_tmp0 = NULL;
		return_type = (_tmp0 = vala_gidl_parser_parse_param (self, res), (return_type == NULL ? NULL : (return_type = (g_object_unref (return_type), NULL))), _tmp0);
	}
	m = NULL;
	if (!is_interface && (is_constructor || g_str_has_prefix (name, "new"))) {
		ValaMethod* _tmp1;
		_tmp1 = NULL;
		m = (_tmp1 = VALA_METHOD (vala_creation_method_new (NULL, name, self->priv->current_source_reference)), (m == NULL ? NULL : (m = (g_object_unref (m), NULL))), _tmp1);
		if (_vala_strcmp0 (vala_symbol_get_name (VALA_SYMBOL (m)), "new") == 0) {
			vala_symbol_set_name (VALA_SYMBOL (m), NULL);
		} else {
			if (g_str_has_prefix (vala_symbol_get_name (VALA_SYMBOL (m)), "new_")) {
				vala_symbol_set_name (VALA_SYMBOL (m), g_utf8_offset_to_pointer (vala_symbol_get_name (VALA_SYMBOL (m)), g_utf8_strlen ("new_", -1)));
			}
		}
	} else {
		if (_vala_strcmp0 (vala_unresolved_symbol_get_name (vala_unresolved_type_get_unresolved_symbol (return_type)), "void") == 0) {
			ValaMethod* _tmp3;
			ValaVoidType* _tmp2;
			_tmp3 = NULL;
			_tmp2 = NULL;
			m = (_tmp3 = vala_method_new (name, VALA_DATA_TYPE ((_tmp2 = vala_void_type_new ())), self->priv->current_source_reference), (m == NULL ? NULL : (m = (g_object_unref (m), NULL))), _tmp3);
			(_tmp2 == NULL ? NULL : (_tmp2 = (g_object_unref (_tmp2), NULL)));
		} else {
			ValaMethod* _tmp4;
			_tmp4 = NULL;
			m = (_tmp4 = vala_method_new (name, VALA_DATA_TYPE (return_type), self->priv->current_source_reference), (m == NULL ? NULL : (m = (g_object_unref (m), NULL))), _tmp4);
		}
	}
	vala_symbol_set_access (VALA_SYMBOL (m), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	if (self->priv->current_type_symbol_set != NULL) {
		gee_collection_add (GEE_COLLECTION (self->priv->current_type_symbol_set), name);
	}
	if (self->priv->current_data_type != NULL) {
		char** _tmp8;
		gint sig_attributes_length1;
		char* _tmp6;
		char* _tmp5;
		char** _tmp7;
		char** sig_attributes;
		_tmp8 = NULL;
		_tmp6 = NULL;
		_tmp5 = NULL;
		_tmp7 = NULL;
		sig_attributes = (_tmp8 = (_tmp7 = vala_gidl_parser_get_attributes (self, (_tmp6 = g_strdup_printf ("%s::%s", (_tmp5 = vala_typesymbol_get_cname (self->priv->current_data_type, FALSE)), name))), (_tmp6 = (g_free (_tmp6), NULL)), (_tmp5 = (g_free (_tmp5), NULL)), _tmp7), sig_attributes_length1 = -1, _tmp8);
		if (sig_attributes != NULL) {
			{
				char** attr_collection;
				int attr_it;
				attr_collection = sig_attributes;
				for (attr_it = 0; (sig_attributes_length1 != -1 && attr_it < sig_attributes_length1) || (sig_attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
					const char* _tmp13;
					char* attr;
					_tmp13 = NULL;
					attr = (_tmp13 = attr_collection[attr_it], (_tmp13 == NULL ? NULL : g_strdup (_tmp13)));
					{
						char** _tmp9;
						gint nv_length1;
						char** nv;
						char* _tmp10;
						gboolean _tmp11;
						_tmp9 = NULL;
						nv = (_tmp9 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp9);
						_tmp10 = NULL;
						if ((_tmp11 = _vala_strcmp0 (nv[0], "has_emitter") == 0 && _vala_strcmp0 ((_tmp10 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp10 = (g_free (_tmp10), NULL)), _tmp11)) {
							gpointer _tmp12;
							return (_tmp12 = NULL, (attr = (g_free (attr), NULL)), (nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL)), (sig_attributes = (_vala_array_free (sig_attributes, sig_attributes_length1, ((GDestroyNotify) g_free)), NULL)), (return_type == NULL ? NULL : (return_type = (g_object_unref (return_type), NULL))), (m == NULL ? NULL : (m = (g_object_unref (m), NULL))), _tmp12);
						}
						attr = (g_free (attr), NULL);
						nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL);
					}
				}
			}
		}
		sig_attributes = (_vala_array_free (sig_attributes, sig_attributes_length1, ((GDestroyNotify) g_free)), NULL);
	}
	add_ellipsis = FALSE;
	suppress_throws = FALSE;
	_tmp14 = NULL;
	attributes = (_tmp14 = vala_gidl_parser_get_attributes (self, symbol), attributes_length1 = -1, _tmp14);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_it;
			attr_collection = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
				const char* _tmp31;
				char* attr;
				_tmp31 = NULL;
				attr = (_tmp31 = attr_collection[attr_it], (_tmp31 == NULL ? NULL : g_strdup (_tmp31)));
				{
					char** _tmp15;
					gint nv_length1;
					char** nv;
					_tmp15 = NULL;
					nv = (_tmp15 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp15);
					if (_vala_strcmp0 (nv[0], "name") == 0) {
						char* _tmp16;
						vala_method_set_cname (m, vala_symbol_get_name (VALA_SYMBOL (m)));
						_tmp16 = NULL;
						vala_symbol_set_name (VALA_SYMBOL (m), (_tmp16 = vala_gidl_parser_eval (self, nv[1])));
						_tmp16 = (g_free (_tmp16), NULL);
					} else {
						if (_vala_strcmp0 (nv[0], "hidden") == 0) {
							char* _tmp17;
							gboolean _tmp18;
							_tmp17 = NULL;
							if ((_tmp18 = _vala_strcmp0 ((_tmp17 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp17 = (g_free (_tmp17), NULL)), _tmp18)) {
								gpointer _tmp19;
								return (_tmp19 = NULL, (attr = (g_free (attr), NULL)), (nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL)), (return_type == NULL ? NULL : (return_type = (g_object_unref (return_type), NULL))), (m == NULL ? NULL : (m = (g_object_unref (m), NULL))), (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp19);
							}
						} else {
							if (_vala_strcmp0 (nv[0], "ellipsis") == 0) {
								char* _tmp20;
								gboolean _tmp21;
								_tmp20 = NULL;
								if ((_tmp21 = _vala_strcmp0 ((_tmp20 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp20 = (g_free (_tmp20), NULL)), _tmp21)) {
									add_ellipsis = TRUE;
								}
							} else {
								if (_vala_strcmp0 (nv[0], "transfer_ownership") == 0) {
									char* _tmp22;
									gboolean _tmp23;
									_tmp22 = NULL;
									if ((_tmp23 = _vala_strcmp0 ((_tmp22 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp22 = (g_free (_tmp22), NULL)), _tmp23)) {
										vala_data_type_set_transfers_ownership (VALA_DATA_TYPE (return_type), TRUE);
									}
								} else {
									if (_vala_strcmp0 (nv[0], "sentinel") == 0) {
										char* _tmp24;
										_tmp24 = NULL;
										vala_method_set_sentinel (m, (_tmp24 = vala_gidl_parser_eval (self, nv[1])));
										_tmp24 = (g_free (_tmp24), NULL);
									} else {
										if (_vala_strcmp0 (nv[0], "is_array") == 0) {
											char* _tmp25;
											gboolean _tmp26;
											_tmp25 = NULL;
											if ((_tmp26 = _vala_strcmp0 ((_tmp25 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp25 = (g_free (_tmp25), NULL)), _tmp26)) {
												vala_unresolved_type_set_array_rank (return_type, 1);
												vala_data_type_set_is_out (VALA_DATA_TYPE (return_type), FALSE);
											}
										} else {
											if (_vala_strcmp0 (nv[0], "throws") == 0) {
												char* _tmp27;
												gboolean _tmp28;
												_tmp27 = NULL;
												if ((_tmp28 = _vala_strcmp0 ((_tmp27 = vala_gidl_parser_eval (self, nv[1])), "0") == 0, (_tmp27 = (g_free (_tmp27), NULL)), _tmp28)) {
													suppress_throws = TRUE;
												}
											} else {
												if (_vala_strcmp0 (nv[0], "no_array_length") == 0) {
													char* _tmp29;
													gboolean _tmp30;
													_tmp29 = NULL;
													if ((_tmp30 = _vala_strcmp0 ((_tmp29 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp29 = (g_free (_tmp29), NULL)), _tmp30)) {
														vala_method_set_no_array_length (m, TRUE);
													}
												}
											}
										}
									}
								}
							}
						}
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL);
				}
			}
		}
	}
	vala_method_set_cname (m, symbol);
	first = TRUE;
	last_param = NULL;
	last_param_type = NULL;
	{
		GList* param_collection;
		GList* param_it;
		param_collection = parameters;
		for (param_it = param_collection; param_it != NULL; param_it = param_it->next) {
			GIdlNodeParam* param;
			param = ((GIdlNodeParam*) param_it->data);
			{
				GIdlNode* param_node;
				const char* _tmp35;
				char* param_name;
				ValaUnresolvedType* param_type;
				ValaFormalParameter* p;
				gboolean hide_param;
				gboolean set_array_length_pos;
				double array_length_pos;
				char** _tmp40;
				gint attributes_length1;
				char* _tmp38;
				char** _tmp39;
				char** attributes;
				char* _tmp56;
				gboolean _tmp57;
				ValaFormalParameter* _tmp59;
				ValaFormalParameter* _tmp58;
				ValaUnresolvedType* _tmp61;
				ValaUnresolvedType* _tmp60;
				param_node = ((GIdlNode*) param);
				if (first) {
					char* _tmp32;
					gboolean _tmp33;
					first = FALSE;
					_tmp32 = NULL;
					if ((_tmp33 = !(VALA_IS_CREATION_METHOD (m)) && self->priv->current_data_type != NULL && param->type->is_interface && (_vala_strcmp0 (param_node->name, "self") == 0 || g_str_has_suffix (param->type->interface, (_tmp32 = vala_typesymbol_get_cname (self->priv->current_data_type, FALSE)))), (_tmp32 = (g_free (_tmp32), NULL)), _tmp33)) {
						continue;
					} else {
						/* instance method
						 static method*/
						vala_method_set_instance (m, FALSE);
					}
				}
				if (suppress_throws == FALSE && vala_gidl_parser_param_is_exception (self, param)) {
					ValaUnresolvedType* _tmp34;
					_tmp34 = NULL;
					vala_method_add_error_domain (m, VALA_DATA_TYPE ((_tmp34 = vala_gidl_parser_parse_type (self, param->type))));
					(_tmp34 == NULL ? NULL : (_tmp34 = (g_object_unref (_tmp34), NULL)));
					continue;
				}
				_tmp35 = NULL;
				param_name = (_tmp35 = param_node->name, (_tmp35 == NULL ? NULL : g_strdup (_tmp35)));
				if (_vala_strcmp0 (param_name, "result") == 0) {
					char* _tmp36;
					/* avoid conflict with generated result variable*/
					_tmp36 = NULL;
					param_name = (_tmp36 = g_strdup ("_result"), (param_name = (g_free (param_name), NULL)), _tmp36);
				} else {
					if (_vala_strcmp0 (param_name, "string") == 0) {
						char* _tmp37;
						/* avoid conflict with string type*/
						_tmp37 = NULL;
						param_name = (_tmp37 = g_strdup ("str"), (param_name = (g_free (param_name), NULL)), _tmp37);
					}
				}
				param_type = vala_gidl_parser_parse_param (self, param);
				p = vala_formal_parameter_new (param_name, VALA_DATA_TYPE (param_type), NULL);
				hide_param = FALSE;
				set_array_length_pos = FALSE;
				array_length_pos = ((double) 0);
				_tmp40 = NULL;
				_tmp38 = NULL;
				_tmp39 = NULL;
				attributes = (_tmp40 = (_tmp39 = vala_gidl_parser_get_attributes (self, (_tmp38 = g_strdup_printf ("%s.%s", symbol, param_node->name))), (_tmp38 = (g_free (_tmp38), NULL)), _tmp39), attributes_length1 = -1, _tmp40);
				if (attributes != NULL) {
					{
						char** attr_collection;
						int attr_it;
						attr_collection = attributes;
						for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
							const char* _tmp55;
							char* attr;
							_tmp55 = NULL;
							attr = (_tmp55 = attr_collection[attr_it], (_tmp55 == NULL ? NULL : g_strdup (_tmp55)));
							{
								char** _tmp41;
								gint nv_length1;
								char** nv;
								_tmp41 = NULL;
								nv = (_tmp41 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp41);
								if (_vala_strcmp0 (nv[0], "is_array") == 0) {
									char* _tmp42;
									gboolean _tmp43;
									_tmp42 = NULL;
									if ((_tmp43 = _vala_strcmp0 ((_tmp42 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp42 = (g_free (_tmp42), NULL)), _tmp43)) {
										vala_unresolved_type_set_array_rank (param_type, 1);
										vala_data_type_set_is_out (VALA_DATA_TYPE (param_type), FALSE);
									}
								} else {
									if (_vala_strcmp0 (nv[0], "is_out") == 0) {
										char* _tmp44;
										gboolean _tmp45;
										_tmp44 = NULL;
										if ((_tmp45 = _vala_strcmp0 ((_tmp44 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp44 = (g_free (_tmp44), NULL)), _tmp45)) {
											vala_data_type_set_is_out (VALA_DATA_TYPE (param_type), TRUE);
										}
									} else {
										if (_vala_strcmp0 (nv[0], "is_ref") == 0) {
											char* _tmp46;
											gboolean _tmp47;
											_tmp46 = NULL;
											if ((_tmp47 = _vala_strcmp0 ((_tmp46 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp46 = (g_free (_tmp46), NULL)), _tmp47)) {
												vala_data_type_set_is_ref (VALA_DATA_TYPE (param_type), TRUE);
											}
										} else {
											if (_vala_strcmp0 (nv[0], "nullable") == 0) {
												char* _tmp48;
												gboolean _tmp49;
												_tmp48 = NULL;
												if ((_tmp49 = _vala_strcmp0 ((_tmp48 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp48 = (g_free (_tmp48), NULL)), _tmp49)) {
													vala_data_type_set_nullable (VALA_DATA_TYPE (param_type), TRUE);
													vala_data_type_set_requires_null_check (VALA_DATA_TYPE (param_type), TRUE);
												}
											} else {
												if (_vala_strcmp0 (nv[0], "transfer_ownership") == 0) {
													char* _tmp50;
													gboolean _tmp51;
													_tmp50 = NULL;
													if ((_tmp51 = _vala_strcmp0 ((_tmp50 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp50 = (g_free (_tmp50), NULL)), _tmp51)) {
														vala_data_type_set_transfers_ownership (VALA_DATA_TYPE (param_type), TRUE);
													}
												} else {
													if (_vala_strcmp0 (nv[0], "hidden") == 0) {
														char* _tmp52;
														gboolean _tmp53;
														_tmp52 = NULL;
														if ((_tmp53 = _vala_strcmp0 ((_tmp52 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp52 = (g_free (_tmp52), NULL)), _tmp53)) {
															hide_param = TRUE;
														}
													} else {
														if (_vala_strcmp0 (nv[0], "array_length_pos") == 0) {
															char* _tmp54;
															set_array_length_pos = TRUE;
															_tmp54 = NULL;
															array_length_pos = strtod ((_tmp54 = vala_gidl_parser_eval (self, nv[1])), NULL);
															_tmp54 = (g_free (_tmp54), NULL);
														}
													}
												}
											}
										}
									}
								}
								attr = (g_free (attr), NULL);
								nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL);
							}
						}
					}
				}
				_tmp56 = NULL;
				if ((_tmp57 = last_param != NULL && _vala_strcmp0 (vala_symbol_get_name (VALA_SYMBOL (p)), (_tmp56 = g_strconcat ("n_", vala_symbol_get_name (VALA_SYMBOL (last_param)), NULL))) == 0, (_tmp56 = (g_free (_tmp56), NULL)), _tmp57)) {
					/* last_param is array, p is array length*/
					vala_unresolved_type_set_array_rank (last_param_type, 1);
					vala_data_type_set_is_out (VALA_DATA_TYPE (last_param_type), FALSE);
					/* hide array length param*/
					hide_param = TRUE;
				} else {
					if (last_param != NULL && _vala_strcmp0 (vala_symbol_get_name (VALA_SYMBOL (p)), "user_data") == 0) {
						/* last_param is delegate
						 hide deleate target param*/
						hide_param = TRUE;
					}
				}
				if (!hide_param) {
					vala_method_add_parameter (m, p);
					if (set_array_length_pos) {
						vala_formal_parameter_set_carray_length_parameter_position (p, array_length_pos);
					}
				}
				_tmp59 = NULL;
				_tmp58 = NULL;
				last_param = (_tmp59 = (_tmp58 = p, (_tmp58 == NULL ? NULL : g_object_ref (_tmp58))), (last_param == NULL ? NULL : (last_param = (g_object_unref (last_param), NULL))), _tmp59);
				_tmp61 = NULL;
				_tmp60 = NULL;
				last_param_type = (_tmp61 = (_tmp60 = param_type, (_tmp60 == NULL ? NULL : g_object_ref (_tmp60))), (last_param_type == NULL ? NULL : (last_param_type = (g_object_unref (last_param_type), NULL))), _tmp61);
				param_name = (g_free (param_name), NULL);
				(param_type == NULL ? NULL : (param_type = (g_object_unref (param_type), NULL)));
				(p == NULL ? NULL : (p = (g_object_unref (p), NULL)));
				attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL);
			}
		}
	}
	if (first) {
		/* no parameters => static method*/
		vala_method_set_instance (m, FALSE);
	}
	if (last_param != NULL && g_str_has_prefix (vala_symbol_get_name (VALA_SYMBOL (last_param)), "first_")) {
		vala_formal_parameter_set_ellipsis (last_param, TRUE);
	} else {
		if (add_ellipsis) {
			ValaFormalParameter* _tmp62;
			_tmp62 = NULL;
			vala_method_add_parameter (m, (_tmp62 = vala_formal_parameter_new_with_ellipsis (NULL)));
			(_tmp62 == NULL ? NULL : (_tmp62 = (g_object_unref (_tmp62), NULL)));
		}
	}
	_tmp63 = NULL;
	return (_tmp63 = m, (return_type == NULL ? NULL : (return_type = (g_object_unref (return_type), NULL))), (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), (last_param == NULL ? NULL : (last_param = (g_object_unref (last_param), NULL))), (last_param_type == NULL ? NULL : (last_param_type = (g_object_unref (last_param_type), NULL))), _tmp63);
}


static gboolean vala_gidl_parser_param_is_exception (ValaGIdlParser* self, GIdlNodeParam* param) {
	const char* _tmp1;
	char* s;
	gboolean _tmp3;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), FALSE);
	if (!param->type->is_error) {
		return FALSE;
	}
	_tmp1 = NULL;
	s = (_tmp1 = g_strchomp (param->type->unparsed), (_tmp1 == NULL ? NULL : g_strdup (_tmp1)));
	if (g_str_has_suffix (s, "**")) {
		gboolean _tmp2;
		return (_tmp2 = TRUE, (s = (g_free (s), NULL)), _tmp2);
	}
	return (_tmp3 = FALSE, (s = (g_free (s), NULL)), _tmp3);
}


static ValaMethod* vala_gidl_parser_parse_function (ValaGIdlParser* self, GIdlNodeFunction* f, gboolean is_interface) {
	GIdlNode* node;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	node = ((GIdlNode*) f);
	if (f->deprecated) {
		return NULL;
	}
	return vala_gidl_parser_create_method (self, node->name, f->symbol, f->result, f->parameters, f->is_constructor, is_interface);
}


static ValaMethod* vala_gidl_parser_parse_virtual (ValaGIdlParser* self, GIdlNodeVFunc* v, GIdlNodeFunction* func, gboolean is_interface) {
	GIdlNode* node;
	char* _tmp0;
	char* _tmp1;
	char* symbol;
	ValaMethod* m;
	ValaMethod* _tmp4;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	node = ((GIdlNode*) v);
	_tmp0 = NULL;
	_tmp1 = NULL;
	symbol = (_tmp1 = g_strdup_printf ("%s%s", (_tmp0 = vala_symbol_get_lower_case_cprefix (VALA_SYMBOL (self->priv->current_data_type))), node->name), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
	if (func != NULL) {
		char* _tmp3;
		const char* _tmp2;
		_tmp3 = NULL;
		_tmp2 = NULL;
		symbol = (_tmp3 = (_tmp2 = func->symbol, (_tmp2 == NULL ? NULL : g_strdup (_tmp2))), (symbol = (g_free (symbol), NULL)), _tmp3);
	}
	m = vala_gidl_parser_create_method (self, node->name, symbol, v->result, (func != NULL ? func->parameters : v->parameters), FALSE, is_interface);
	if (m != NULL) {
		vala_method_set_instance (m, TRUE);
		vala_method_set_is_virtual (m, !is_interface);
		vala_method_set_is_abstract (m, is_interface);
		if (func == NULL) {
			VALA_CODE_NODE (m)->attributes = g_list_append (VALA_CODE_NODE (m)->attributes, vala_attribute_new ("NoWrapper", NULL));
		}
	}
	_tmp4 = NULL;
	return (_tmp4 = m, (symbol = (g_free (symbol), NULL)), _tmp4);
}


static char* vala_gidl_parser_fix_prop_name (ValaGIdlParser* self, const char* name) {
	GString* str;
	const char* _tmp0;
	char* i;
	const char* _tmp3;
	char* _tmp4;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	str = g_string_new ("");
	_tmp0 = NULL;
	i = (_tmp0 = name, (_tmp0 == NULL ? NULL : g_strdup (_tmp0)));
	while (g_utf8_strlen (i, -1) > 0) {
		gunichar c;
		char* _tmp2;
		const char* _tmp1;
		c = g_utf8_get_char (i);
		if (c == '-') {
			g_string_append_c (str, '_');
		} else {
			g_string_append_unichar (str, c);
		}
		_tmp2 = NULL;
		_tmp1 = NULL;
		i = (_tmp2 = (_tmp1 = g_utf8_next_char (i), (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (i = (g_free (i), NULL)), _tmp2);
	}
	_tmp3 = NULL;
	_tmp4 = NULL;
	return (_tmp4 = (_tmp3 = str->str, (_tmp3 == NULL ? NULL : g_strdup (_tmp3))), (str == NULL ? NULL : (str = (g_string_free (str, TRUE), NULL))), (i = (g_free (i), NULL)), _tmp4);
}


static ValaProperty* vala_gidl_parser_parse_property (ValaGIdlParser* self, GIdlNodeProperty* prop_node) {
	GIdlNode* node;
	ValaPropertyAccessor* get_acc;
	ValaPropertyAccessor* set_acc;
	ValaUnresolvedType* _tmp4;
	char* _tmp3;
	ValaProperty* _tmp5;
	ValaProperty* prop;
	char** _tmp9;
	gint attributes_length1;
	char* _tmp7;
	char* _tmp6;
	char** _tmp8;
	char** attributes;
	ValaProperty* _tmp15;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	node = ((GIdlNode*) prop_node);
	if (prop_node->deprecated) {
		return NULL;
	}
	if (!prop_node->readable && !prop_node->writable) {
		/* buggy GIDL definition*/
		prop_node->readable = TRUE;
		prop_node->writable = TRUE;
	}
	get_acc = NULL;
	set_acc = NULL;
	if (prop_node->readable) {
		ValaPropertyAccessor* _tmp1;
		_tmp1 = NULL;
		get_acc = (_tmp1 = vala_property_accessor_new (TRUE, FALSE, FALSE, NULL, NULL), (get_acc == NULL ? NULL : (get_acc = (g_object_unref (get_acc), NULL))), _tmp1);
	}
	if (prop_node->writable) {
		ValaPropertyAccessor* _tmp2;
		_tmp2 = NULL;
		set_acc = (_tmp2 = vala_property_accessor_new (FALSE, FALSE, FALSE, NULL, NULL), (set_acc == NULL ? NULL : (set_acc = (g_object_unref (set_acc), NULL))), _tmp2);
		if (prop_node->construct_only) {
			vala_property_accessor_set_construction (set_acc, TRUE);
		} else {
			vala_property_accessor_set_writable (set_acc, TRUE);
			vala_property_accessor_set_construction (set_acc, prop_node->construct);
		}
	}
	_tmp4 = NULL;
	_tmp3 = NULL;
	_tmp5 = NULL;
	prop = (_tmp5 = vala_property_new ((_tmp3 = vala_gidl_parser_fix_prop_name (self, node->name)), VALA_DATA_TYPE ((_tmp4 = vala_gidl_parser_parse_type (self, prop_node->type))), get_acc, set_acc, self->priv->current_source_reference), (_tmp4 == NULL ? NULL : (_tmp4 = (g_object_unref (_tmp4), NULL))), (_tmp3 = (g_free (_tmp3), NULL)), _tmp5);
	vala_symbol_set_access (VALA_SYMBOL (prop), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	vala_property_set_interface_only (prop, TRUE);
	_tmp9 = NULL;
	_tmp7 = NULL;
	_tmp6 = NULL;
	_tmp8 = NULL;
	attributes = (_tmp9 = (_tmp8 = vala_gidl_parser_get_attributes (self, (_tmp7 = g_strdup_printf ("%s:%s", (_tmp6 = vala_typesymbol_get_cname (self->priv->current_data_type, FALSE)), node->name))), (_tmp7 = (g_free (_tmp7), NULL)), (_tmp6 = (g_free (_tmp6), NULL)), _tmp8), attributes_length1 = -1, _tmp9);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_it;
			attr_collection = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
				const char* _tmp14;
				char* attr;
				_tmp14 = NULL;
				attr = (_tmp14 = attr_collection[attr_it], (_tmp14 == NULL ? NULL : g_strdup (_tmp14)));
				{
					char** _tmp10;
					gint nv_length1;
					char** nv;
					_tmp10 = NULL;
					nv = (_tmp10 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp10);
					if (_vala_strcmp0 (nv[0], "hidden") == 0) {
						char* _tmp11;
						gboolean _tmp12;
						_tmp11 = NULL;
						if ((_tmp12 = _vala_strcmp0 ((_tmp11 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp11 = (g_free (_tmp11), NULL)), _tmp12)) {
							gpointer _tmp13;
							return (_tmp13 = NULL, (attr = (g_free (attr), NULL)), (nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL)), (get_acc == NULL ? NULL : (get_acc = (g_object_unref (get_acc), NULL))), (set_acc == NULL ? NULL : (set_acc = (g_object_unref (set_acc), NULL))), (prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL))), (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp13);
						}
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL);
				}
			}
		}
	}
	if (self->priv->current_type_symbol_set != NULL) {
		gee_collection_add (GEE_COLLECTION (self->priv->current_type_symbol_set), vala_symbol_get_name (VALA_SYMBOL (prop)));
	}
	_tmp15 = NULL;
	return (_tmp15 = prop, (get_acc == NULL ? NULL : (get_acc = (g_object_unref (get_acc), NULL))), (set_acc == NULL ? NULL : (set_acc = (g_object_unref (set_acc), NULL))), (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp15);
}


static ValaConstant* vala_gidl_parser_parse_constant (ValaGIdlParser* self, GIdlNodeConstant* const_node) {
	GIdlNode* node;
	ValaUnresolvedType* type;
	ValaConstant* c;
	ValaConstant* _tmp1;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	node = ((GIdlNode*) const_node);
	type = vala_gidl_parser_parse_type (self, const_node->type);
	if (type == NULL) {
		gpointer _tmp0;
		return (_tmp0 = NULL, (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), _tmp0);
	}
	c = vala_constant_new (node->name, VALA_DATA_TYPE (type), NULL, self->priv->current_source_reference);
	vala_symbol_set_access (VALA_SYMBOL (c), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	_tmp1 = NULL;
	return (_tmp1 = c, (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), _tmp1);
}


static ValaField* vala_gidl_parser_parse_field (ValaGIdlParser* self, GIdlNodeField* field_node) {
	GIdlNode* node;
	ValaUnresolvedType* type;
	char** _tmp4;
	gint attributes_length1;
	char* _tmp2;
	char* _tmp1;
	char** _tmp3;
	char** attributes;
	const char* _tmp14;
	char* field_name;
	ValaField* field;
	ValaField* _tmp16;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	node = ((GIdlNode*) field_node);
	type = vala_gidl_parser_parse_type (self, field_node->type);
	if (type == NULL) {
		gpointer _tmp0;
		return (_tmp0 = NULL, (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), _tmp0);
	}
	_tmp4 = NULL;
	_tmp2 = NULL;
	_tmp1 = NULL;
	_tmp3 = NULL;
	attributes = (_tmp4 = (_tmp3 = vala_gidl_parser_get_attributes (self, (_tmp2 = g_strdup_printf ("%s.%s", (_tmp1 = vala_typesymbol_get_cname (self->priv->current_data_type, FALSE)), node->name))), (_tmp2 = (g_free (_tmp2), NULL)), (_tmp1 = (g_free (_tmp1), NULL)), _tmp3), attributes_length1 = -1, _tmp4);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_it;
			attr_collection = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
				const char* _tmp13;
				char* attr;
				_tmp13 = NULL;
				attr = (_tmp13 = attr_collection[attr_it], (_tmp13 == NULL ? NULL : g_strdup (_tmp13)));
				{
					char** _tmp5;
					gint nv_length1;
					char** nv;
					_tmp5 = NULL;
					nv = (_tmp5 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp5);
					if (_vala_strcmp0 (nv[0], "hidden") == 0) {
						char* _tmp6;
						gboolean _tmp7;
						_tmp6 = NULL;
						if ((_tmp7 = _vala_strcmp0 ((_tmp6 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp6 = (g_free (_tmp6), NULL)), _tmp7)) {
							gpointer _tmp8;
							return (_tmp8 = NULL, (attr = (g_free (attr), NULL)), (nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL)), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp8);
						}
					} else {
						if (_vala_strcmp0 (nv[0], "is_array") == 0) {
							char* _tmp9;
							gboolean _tmp10;
							_tmp9 = NULL;
							if ((_tmp10 = _vala_strcmp0 ((_tmp9 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp9 = (g_free (_tmp9), NULL)), _tmp10)) {
								vala_unresolved_type_set_array_rank (type, 1);
							}
						} else {
							if (_vala_strcmp0 (nv[0], "weak") == 0) {
								char* _tmp11;
								gboolean _tmp12;
								_tmp11 = NULL;
								if ((_tmp12 = _vala_strcmp0 ((_tmp11 = vala_gidl_parser_eval (self, nv[1])), "0") == 0, (_tmp11 = (g_free (_tmp11), NULL)), _tmp12)) {
									vala_data_type_set_takes_ownership (VALA_DATA_TYPE (type), TRUE);
								}
							}
						}
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL);
				}
			}
		}
	}
	if (self->priv->current_type_symbol_set != NULL) {
		gee_collection_add (GEE_COLLECTION (self->priv->current_type_symbol_set), node->name);
	}
	_tmp14 = NULL;
	field_name = (_tmp14 = node->name, (_tmp14 == NULL ? NULL : g_strdup (_tmp14)));
	if (_vala_strcmp0 (field_name, "string") == 0) {
		char* _tmp15;
		/* avoid conflict with string type*/
		_tmp15 = NULL;
		field_name = (_tmp15 = g_strdup ("str"), (field_name = (g_free (field_name), NULL)), _tmp15);
	}
	field = vala_field_new (field_name, VALA_DATA_TYPE (type), NULL, self->priv->current_source_reference);
	vala_symbol_set_access (VALA_SYMBOL (field), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	vala_field_set_no_array_length (field, TRUE);
	_tmp16 = NULL;
	return (_tmp16 = field, (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), (field_name = (g_free (field_name), NULL)), _tmp16);
}


static char** vala_gidl_parser_get_attributes (ValaGIdlParser* self, const char* codenode) {
	char* attributes;
	char** _tmp2;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	attributes = ((char*) gee_map_get (self->priv->codenode_attributes_map, codenode));
	if (attributes == NULL) {
		gboolean dot_required;
		gboolean colon_required;
		GeeSet* pattern_specs;
		dot_required = (NULL != g_utf8_strchr (codenode, ((glong) -1), ((gunichar) '.')));
		colon_required = (NULL != g_utf8_strchr (codenode, ((glong) -1), ((gunichar) ':')));
		pattern_specs = gee_map_get_keys (self->priv->codenode_attributes_patterns);
		{
			GeeSet* pattern_collection;
			GeeIterator* pattern_it;
			pattern_collection = pattern_specs;
			pattern_it = gee_iterable_iterator (GEE_ITERABLE (pattern_collection));
			while (gee_iterator_next (pattern_it)) {
				GPatternSpec* pattern;
				pattern = gee_iterator_get (pattern_it);
				{
					char* pspec;
					pspec = ((char*) gee_map_get (((GeeMap*) self->priv->codenode_attributes_patterns), pattern));
					if ((dot_required && NULL == g_utf8_strchr (pspec, ((glong) -1), ((gunichar) '.'))) || (colon_required && NULL == g_utf8_strchr (pspec, ((glong) -1), ((gunichar) ':')))) {
						pspec = (g_free (pspec), NULL);
						continue;
					}
					if (g_pattern_match_string (pattern, codenode)) {
						char** _tmp0;
						_tmp0 = NULL;
						return (_tmp0 = vala_gidl_parser_get_attributes (self, pspec), (pspec = (g_free (pspec), NULL)), (pattern_it == NULL ? NULL : (pattern_it = (g_object_unref (pattern_it), NULL))), (pattern_specs == NULL ? NULL : (pattern_specs = (g_object_unref (pattern_specs), NULL))), (attributes = (g_free (attributes), NULL)), _tmp0);
					}
					pspec = (g_free (pspec), NULL);
				}
			}
			(pattern_it == NULL ? NULL : (pattern_it = (g_object_unref (pattern_it), NULL)));
		}
		(pattern_specs == NULL ? NULL : (pattern_specs = (g_object_unref (pattern_specs), NULL)));
	}
	if (attributes == NULL) {
		gpointer _tmp1;
		return (_tmp1 = NULL, (attributes = (g_free (attributes), NULL)), _tmp1);
	}
	_tmp2 = NULL;
	return (_tmp2 = g_strsplit (attributes, " ", 0), (attributes = (g_free (attributes), NULL)), _tmp2);
}


static char* vala_gidl_parser_eval (ValaGIdlParser* self, const char* s) {
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	return g_strndup (g_utf8_offset_to_pointer (s, ((glong) 1)), ((gulong) strlen (s) - 2));
}


static ValaSignal* vala_gidl_parser_parse_signal (ValaGIdlParser* self, GIdlNodeSignal* sig_node) {
	GIdlNode* node;
	ValaUnresolvedType* _tmp2;
	char* _tmp1;
	ValaSignal* _tmp3;
	ValaSignal* sig;
	char** _tmp7;
	gint attributes_length1;
	char* _tmp5;
	char* _tmp4;
	char** _tmp6;
	char** attributes;
	gboolean first;
	ValaSignal* _tmp34;
	g_return_val_if_fail (VALA_IS_GIDL_PARSER (self), NULL);
	node = ((GIdlNode*) sig_node);
	if (sig_node->deprecated || sig_node->result == NULL) {
		return NULL;
	}
	_tmp2 = NULL;
	_tmp1 = NULL;
	_tmp3 = NULL;
	sig = (_tmp3 = vala_signal_new ((_tmp1 = vala_gidl_parser_fix_prop_name (self, node->name)), VALA_DATA_TYPE ((_tmp2 = vala_gidl_parser_parse_param (self, sig_node->result))), self->priv->current_source_reference), (_tmp2 == NULL ? NULL : (_tmp2 = (g_object_unref (_tmp2), NULL))), (_tmp1 = (g_free (_tmp1), NULL)), _tmp3);
	vala_symbol_set_access (VALA_SYMBOL (sig), VALA_SYMBOL_ACCESSIBILITY_PUBLIC);
	_tmp7 = NULL;
	_tmp5 = NULL;
	_tmp4 = NULL;
	_tmp6 = NULL;
	attributes = (_tmp7 = (_tmp6 = vala_gidl_parser_get_attributes (self, (_tmp5 = g_strdup_printf ("%s::%s", (_tmp4 = vala_typesymbol_get_cname (self->priv->current_data_type, FALSE)), vala_symbol_get_name (VALA_SYMBOL (sig))))), (_tmp5 = (g_free (_tmp5), NULL)), (_tmp4 = (g_free (_tmp4), NULL)), _tmp6), attributes_length1 = -1, _tmp7);
	if (attributes != NULL) {
		{
			char** attr_collection;
			int attr_it;
			attr_collection = attributes;
			for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
				const char* _tmp15;
				char* attr;
				_tmp15 = NULL;
				attr = (_tmp15 = attr_collection[attr_it], (_tmp15 == NULL ? NULL : g_strdup (_tmp15)));
				{
					char** _tmp8;
					gint nv_length1;
					char** nv;
					_tmp8 = NULL;
					nv = (_tmp8 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp8);
					if (_vala_strcmp0 (nv[0], "name") == 0) {
						char* _tmp9;
						vala_signal_set_cname (sig, vala_symbol_get_name (VALA_SYMBOL (sig)));
						_tmp9 = NULL;
						vala_symbol_set_name (VALA_SYMBOL (sig), (_tmp9 = vala_gidl_parser_eval (self, nv[1])));
						_tmp9 = (g_free (_tmp9), NULL);
					} else {
						char* _tmp10;
						gboolean _tmp11;
						_tmp10 = NULL;
						if ((_tmp11 = _vala_strcmp0 (nv[0], "has_emitter") == 0 && _vala_strcmp0 ((_tmp10 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp10 = (g_free (_tmp10), NULL)), _tmp11)) {
							vala_signal_set_has_emitter (sig, TRUE);
						} else {
							if (_vala_strcmp0 (nv[0], "hidden") == 0) {
								char* _tmp12;
								gboolean _tmp13;
								_tmp12 = NULL;
								if ((_tmp13 = _vala_strcmp0 ((_tmp12 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp12 = (g_free (_tmp12), NULL)), _tmp13)) {
									gpointer _tmp14;
									return (_tmp14 = NULL, (attr = (g_free (attr), NULL)), (nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL)), (sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL))), (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp14);
								}
							}
						}
					}
					attr = (g_free (attr), NULL);
					nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL);
				}
			}
		}
	}
	first = TRUE;
	{
		GList* param_collection;
		GList* param_it;
		param_collection = sig_node->parameters;
		for (param_it = param_collection; param_it != NULL; param_it = param_it->next) {
			GIdlNodeParam* param;
			param = ((GIdlNodeParam*) param_it->data);
			{
				GIdlNode* param_node;
				ValaUnresolvedType* param_type;
				ValaFormalParameter* p;
				char** _tmp19;
				gint attributes_length1;
				char* _tmp17;
				char* _tmp16;
				char** _tmp18;
				char** attributes;
				if (first) {
					/* ignore implicit first signal parameter (sender)*/
					first = FALSE;
					continue;
				}
				param_node = ((GIdlNode*) param);
				param_type = vala_gidl_parser_parse_param (self, param);
				p = vala_formal_parameter_new (param_node->name, VALA_DATA_TYPE (param_type), NULL);
				vala_signal_add_parameter (sig, p);
				_tmp19 = NULL;
				_tmp17 = NULL;
				_tmp16 = NULL;
				_tmp18 = NULL;
				attributes = (_tmp19 = (_tmp18 = vala_gidl_parser_get_attributes (self, (_tmp17 = g_strdup_printf ("%s::%s.%s", (_tmp16 = vala_typesymbol_get_cname (self->priv->current_data_type, FALSE)), vala_symbol_get_name (VALA_SYMBOL (sig)), param_node->name))), (_tmp17 = (g_free (_tmp17), NULL)), (_tmp16 = (g_free (_tmp16), NULL)), _tmp18), attributes_length1 = -1, _tmp19);
				if (attributes != NULL) {
					char* ns_name;
					ns_name = NULL;
					{
						char** attr_collection;
						int attr_it;
						attr_collection = attributes;
						for (attr_it = 0; (attributes_length1 != -1 && attr_it < attributes_length1) || (attributes_length1 == -1 && attr_collection[attr_it] != NULL); attr_it = attr_it + 1) {
							const char* _tmp32;
							char* attr;
							_tmp32 = NULL;
							attr = (_tmp32 = attr_collection[attr_it], (_tmp32 == NULL ? NULL : g_strdup (_tmp32)));
							{
								char** _tmp20;
								gint nv_length1;
								char** nv;
								_tmp20 = NULL;
								nv = (_tmp20 = g_strsplit (attr, "=", 2), nv_length1 = -1, _tmp20);
								if (_vala_strcmp0 (nv[0], "is_array") == 0) {
									char* _tmp21;
									gboolean _tmp22;
									_tmp21 = NULL;
									if ((_tmp22 = _vala_strcmp0 ((_tmp21 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp21 = (g_free (_tmp21), NULL)), _tmp22)) {
										vala_unresolved_type_set_array_rank (param_type, 1);
										vala_data_type_set_is_out (VALA_DATA_TYPE (param_type), FALSE);
									}
								} else {
									if (_vala_strcmp0 (nv[0], "is_out") == 0) {
										char* _tmp23;
										gboolean _tmp24;
										_tmp23 = NULL;
										if ((_tmp24 = _vala_strcmp0 ((_tmp23 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp23 = (g_free (_tmp23), NULL)), _tmp24)) {
											vala_data_type_set_is_out (VALA_DATA_TYPE (param_type), TRUE);
										}
									} else {
										if (_vala_strcmp0 (nv[0], "is_ref") == 0) {
											char* _tmp25;
											gboolean _tmp26;
											_tmp25 = NULL;
											if ((_tmp26 = _vala_strcmp0 ((_tmp25 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp25 = (g_free (_tmp25), NULL)), _tmp26)) {
												vala_data_type_set_is_ref (VALA_DATA_TYPE (param_type), TRUE);
											}
										} else {
											if (_vala_strcmp0 (nv[0], "nullable") == 0) {
												char* _tmp27;
												gboolean _tmp28;
												_tmp27 = NULL;
												if ((_tmp28 = _vala_strcmp0 ((_tmp27 = vala_gidl_parser_eval (self, nv[1])), "1") == 0, (_tmp27 = (g_free (_tmp27), NULL)), _tmp28)) {
													vala_data_type_set_nullable (VALA_DATA_TYPE (param_type), TRUE);
													vala_data_type_set_requires_null_check (VALA_DATA_TYPE (param_type), TRUE);
												}
											} else {
												if (_vala_strcmp0 (nv[0], "type_name") == 0) {
													ValaUnresolvedSymbol* _tmp30;
													char* _tmp29;
													_tmp30 = NULL;
													_tmp29 = NULL;
													vala_unresolved_type_set_unresolved_symbol (param_type, (_tmp30 = vala_unresolved_symbol_new (NULL, (_tmp29 = vala_gidl_parser_eval (self, nv[1])), NULL)));
													(_tmp30 == NULL ? NULL : (_tmp30 = (g_object_unref (_tmp30), NULL)));
													_tmp29 = (g_free (_tmp29), NULL);
												} else {
													if (_vala_strcmp0 (nv[0], "namespace_name") == 0) {
														char* _tmp31;
														_tmp31 = NULL;
														ns_name = (_tmp31 = vala_gidl_parser_eval (self, nv[1]), (ns_name = (g_free (ns_name), NULL)), _tmp31);
													}
												}
											}
										}
									}
								}
								attr = (g_free (attr), NULL);
								nv = (_vala_array_free (nv, nv_length1, ((GDestroyNotify) g_free)), NULL);
							}
						}
					}
					if (ns_name != NULL) {
						ValaUnresolvedSymbol* _tmp33;
						_tmp33 = NULL;
						vala_unresolved_symbol_set_inner (vala_unresolved_type_get_unresolved_symbol (param_type), (_tmp33 = vala_unresolved_symbol_new (NULL, ns_name, NULL)));
						(_tmp33 == NULL ? NULL : (_tmp33 = (g_object_unref (_tmp33), NULL)));
					}
					ns_name = (g_free (ns_name), NULL);
				}
				(param_type == NULL ? NULL : (param_type = (g_object_unref (param_type), NULL)));
				(p == NULL ? NULL : (p = (g_object_unref (p), NULL)));
				attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL);
			}
		}
	}
	_tmp34 = NULL;
	return (_tmp34 = sig, (attributes = (_vala_array_free (attributes, attributes_length1, ((GDestroyNotify) g_free)), NULL)), _tmp34);
}


/**
 * Code visitor parsing all GIDL files.
 */
ValaGIdlParser* vala_gidl_parser_new (void) {
	ValaGIdlParser * self;
	self = g_object_newv (VALA_TYPE_GIDL_PARSER, 0, NULL);
	return self;
}


static void vala_gidl_parser_class_init (ValaGIdlParserClass * klass) {
	vala_gidl_parser_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaGIdlParserPrivate));
	G_OBJECT_CLASS (klass)->dispose = vala_gidl_parser_dispose;
	VALA_CODE_VISITOR_CLASS (klass)->visit_namespace = vala_gidl_parser_real_visit_namespace;
	VALA_CODE_VISITOR_CLASS (klass)->visit_class = vala_gidl_parser_real_visit_class;
	VALA_CODE_VISITOR_CLASS (klass)->visit_struct = vala_gidl_parser_real_visit_struct;
	VALA_CODE_VISITOR_CLASS (klass)->visit_interface = vala_gidl_parser_real_visit_interface;
	VALA_CODE_VISITOR_CLASS (klass)->visit_enum = vala_gidl_parser_real_visit_enum;
	VALA_CODE_VISITOR_CLASS (klass)->visit_delegate = vala_gidl_parser_real_visit_delegate;
	VALA_CODE_VISITOR_CLASS (klass)->visit_source_file = vala_gidl_parser_real_visit_source_file;
}


static void vala_gidl_parser_init (ValaGIdlParser * self) {
	self->priv = VALA_GIDL_PARSER_GET_PRIVATE (self);
}


static void vala_gidl_parser_dispose (GObject * obj) {
	ValaGIdlParser * self;
	self = VALA_GIDL_PARSER (obj);
	(self->priv->context == NULL ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL)));
	(self->priv->current_source_file == NULL ? NULL : (self->priv->current_source_file = (g_object_unref (self->priv->current_source_file), NULL)));
	(self->priv->current_source_reference == NULL ? NULL : (self->priv->current_source_reference = (g_object_unref (self->priv->current_source_reference), NULL)));
	(self->priv->current_namespace == NULL ? NULL : (self->priv->current_namespace = (g_object_unref (self->priv->current_namespace), NULL)));
	(self->priv->current_data_type == NULL ? NULL : (self->priv->current_data_type = (g_object_unref (self->priv->current_data_type), NULL)));
	(self->priv->codenode_attributes_map == NULL ? NULL : (self->priv->codenode_attributes_map = (g_object_unref (self->priv->codenode_attributes_map), NULL)));
	(self->priv->codenode_attributes_patterns == NULL ? NULL : (self->priv->codenode_attributes_patterns = (g_object_unref (self->priv->codenode_attributes_patterns), NULL)));
	(self->priv->current_type_symbol_set == NULL ? NULL : (self->priv->current_type_symbol_set = (g_object_unref (self->priv->current_type_symbol_set), NULL)));
	(self->priv->cname_type_map == NULL ? NULL : (self->priv->cname_type_map = (g_object_unref (self->priv->cname_type_map), NULL)));
	G_OBJECT_CLASS (vala_gidl_parser_parent_class)->dispose (obj);
}


GType vala_gidl_parser_get_type (void) {
	static GType vala_gidl_parser_type_id = 0;
	if (G_UNLIKELY (vala_gidl_parser_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaGIdlParserClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_gidl_parser_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaGIdlParser), 0, (GInstanceInitFunc) vala_gidl_parser_init };
		vala_gidl_parser_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaGIdlParser", &g_define_type_info, 0);
	}
	return vala_gidl_parser_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if (array != NULL && destroy_func != NULL) {
		int i;
		if (array_length >= 0)
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL)
			destroy_func (((gpointer*) array)[i]);
		}
		else
		for (i = 0; ((gpointer*) array)[i] != NULL; i = i + 1) {
			destroy_func (((gpointer*) array)[i]);
		}
	}
	g_free (array);
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return (str1 != str2);
	}
	return strcmp (str1, str2);
}




