/* valaunaryexpression.vala
 *
 * Copyright (C) 2006  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valaunaryexpression.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>

struct _ValaUnaryExpressionPrivate {
	ValaUnaryOperator _operator;
	ValaExpression* _inner;
};
#define VALA_UNARY_EXPRESSION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_UNARY_EXPRESSION, ValaUnaryExpressionPrivate))
enum  {
	VALA_UNARY_EXPRESSION_DUMMY_PROPERTY,
	VALA_UNARY_EXPRESSION_OPERATOR,
	VALA_UNARY_EXPRESSION_INNER,
	VALA_UNARY_EXPRESSION_OPERATOR_STRING
};
static void vala_unary_expression_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_unary_expression_real_replace (ValaCodeNode* base, ValaCodeNode* old_node, ValaCodeNode* new_node);
static char* vala_unary_expression_real_to_string (ValaCodeNode* base);
static gpointer vala_unary_expression_parent_class = NULL;
static void vala_unary_expression_dispose (GObject * obj);


/**
 * Creates a new unary expression.
 *
 * @param op     unary operator
 * @param inner  operand
 * @param source reference to source code
 * @return       newly created binary expression
 */
ValaUnaryExpression* vala_unary_expression_new (ValaUnaryOperator op, ValaExpression* _inner, ValaSourceReference* source) {
	GParameter * __params;
	GParameter * __params_it;
	ValaUnaryExpression * self;
	g_return_val_if_fail (VALA_IS_EXPRESSION (_inner), NULL);
	g_return_val_if_fail (source == NULL || VALA_IS_SOURCE_REFERENCE (source), NULL);
	__params = g_new0 (GParameter, 1);
	__params_it = __params;
	(__params_it->name = "inner", g_value_init (&__params_it->value, VALA_TYPE_EXPRESSION), g_value_set_object (&__params_it->value, _inner), __params_it++);
	self = g_object_newv (VALA_TYPE_UNARY_EXPRESSION, __params_it - __params, __params);
	vala_unary_expression_set_operator (self, op);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


static void vala_unary_expression_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaUnaryExpression * self;
	self = VALA_UNARY_EXPRESSION (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_node_accept (VALA_CODE_NODE (vala_unary_expression_get_inner (self)), visitor);
	vala_code_visitor_visit_unary_expression (visitor, self);
}


static void vala_unary_expression_real_replace (ValaCodeNode* base, ValaCodeNode* old_node, ValaCodeNode* new_node) {
	ValaUnaryExpression * self;
	self = VALA_UNARY_EXPRESSION (base);
	g_return_if_fail (VALA_IS_CODE_NODE (old_node));
	g_return_if_fail (VALA_IS_CODE_NODE (new_node));
	if (VALA_CODE_NODE (vala_unary_expression_get_inner (self)) == old_node) {
		vala_unary_expression_set_inner (self, VALA_EXPRESSION (new_node));
	}
}


static char* vala_unary_expression_real_to_string (ValaCodeNode* base) {
	ValaUnaryExpression * self;
	char* _tmp0;
	char* _tmp1;
	self = VALA_UNARY_EXPRESSION (base);
	_tmp0 = NULL;
	_tmp1 = NULL;
	return (_tmp1 = g_strconcat (vala_unary_expression_get_operator_string (self), (_tmp0 = vala_code_node_to_string (VALA_CODE_NODE (self->priv->_inner))), NULL), (_tmp0 = (g_free (_tmp0), NULL)), _tmp1);
}


ValaUnaryOperator vala_unary_expression_get_operator (ValaUnaryExpression* self) {
	g_return_val_if_fail (VALA_IS_UNARY_EXPRESSION (self), 0);
	return self->priv->_operator;
}


void vala_unary_expression_set_operator (ValaUnaryExpression* self, ValaUnaryOperator value) {
	g_return_if_fail (VALA_IS_UNARY_EXPRESSION (self));
	self->priv->_operator = value;
}


ValaExpression* vala_unary_expression_get_inner (ValaUnaryExpression* self) {
	g_return_val_if_fail (VALA_IS_UNARY_EXPRESSION (self), NULL);
	return self->priv->_inner;
}


void vala_unary_expression_set_inner (ValaUnaryExpression* self, ValaExpression* value) {
	ValaExpression* _tmp1;
	g_return_if_fail (VALA_IS_UNARY_EXPRESSION (self));
	_tmp1 = NULL;
	self->priv->_inner = (_tmp1 = g_object_ref (value), (self->priv->_inner == NULL ? NULL : (self->priv->_inner = (g_object_unref (self->priv->_inner), NULL))), _tmp1);
	vala_code_node_set_parent_node (VALA_CODE_NODE (self->priv->_inner), VALA_CODE_NODE (self));
}


char* vala_unary_expression_get_operator_string (ValaUnaryExpression* self) {
	ValaUnaryOperator _tmp8;
	g_return_val_if_fail (VALA_IS_UNARY_EXPRESSION (self), NULL);
	_tmp8 = self->priv->_operator;
	if (_tmp8 == VALA_UNARY_OPERATOR_PLUS)
	do {
		return "+";
	} while (0); else if (_tmp8 == VALA_UNARY_OPERATOR_MINUS)
	do {
		return "-";
	} while (0); else if (_tmp8 == VALA_UNARY_OPERATOR_LOGICAL_NEGATION)
	do {
		return "!";
	} while (0); else if (_tmp8 == VALA_UNARY_OPERATOR_BITWISE_COMPLEMENT)
	do {
		return "~";
	} while (0); else if (_tmp8 == VALA_UNARY_OPERATOR_INCREMENT)
	do {
		return "++";
	} while (0); else if (_tmp8 == VALA_UNARY_OPERATOR_DECREMENT)
	do {
		return "--";
	} while (0); else if (_tmp8 == VALA_UNARY_OPERATOR_REF)
	do {
		return "ref ";
	} while (0); else if (_tmp8 == VALA_UNARY_OPERATOR_OUT)
	do {
		return "out ";
	} while (0);
	g_assert_not_reached ();
}


static void vala_unary_expression_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaUnaryExpression * self;
	self = VALA_UNARY_EXPRESSION (object);
	switch (property_id) {
		case VALA_UNARY_EXPRESSION_OPERATOR:
		g_value_set_int (value, vala_unary_expression_get_operator (self));
		break;
		case VALA_UNARY_EXPRESSION_INNER:
		g_value_set_object (value, vala_unary_expression_get_inner (self));
		break;
		case VALA_UNARY_EXPRESSION_OPERATOR_STRING:
		g_value_set_string (value, vala_unary_expression_get_operator_string (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_unary_expression_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaUnaryExpression * self;
	self = VALA_UNARY_EXPRESSION (object);
	switch (property_id) {
		case VALA_UNARY_EXPRESSION_OPERATOR:
		vala_unary_expression_set_operator (self, g_value_get_int (value));
		break;
		case VALA_UNARY_EXPRESSION_INNER:
		vala_unary_expression_set_inner (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_unary_expression_class_init (ValaUnaryExpressionClass * klass) {
	vala_unary_expression_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaUnaryExpressionPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_unary_expression_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_unary_expression_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_unary_expression_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_unary_expression_real_accept;
	VALA_CODE_NODE_CLASS (klass)->replace = vala_unary_expression_real_replace;
	VALA_CODE_NODE_CLASS (klass)->to_string = vala_unary_expression_real_to_string;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_UNARY_EXPRESSION_OPERATOR, g_param_spec_int ("operator", "operator", "operator", G_MININT, G_MAXINT, 0, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_UNARY_EXPRESSION_INNER, g_param_spec_object ("inner", "inner", "inner", VALA_TYPE_EXPRESSION, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_UNARY_EXPRESSION_OPERATOR_STRING, g_param_spec_string ("operator-string", "operator-string", "operator-string", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE));
}


static void vala_unary_expression_init (ValaUnaryExpression * self) {
	self->priv = VALA_UNARY_EXPRESSION_GET_PRIVATE (self);
}


static void vala_unary_expression_dispose (GObject * obj) {
	ValaUnaryExpression * self;
	self = VALA_UNARY_EXPRESSION (obj);
	(self->priv->_inner == NULL ? NULL : (self->priv->_inner = (g_object_unref (self->priv->_inner), NULL)));
	G_OBJECT_CLASS (vala_unary_expression_parent_class)->dispose (obj);
}


GType vala_unary_expression_get_type (void) {
	static GType vala_unary_expression_type_id = 0;
	if (G_UNLIKELY (vala_unary_expression_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaUnaryExpressionClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_unary_expression_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaUnaryExpression), 0, (GInstanceInitFunc) vala_unary_expression_init };
		vala_unary_expression_type_id = g_type_register_static (VALA_TYPE_EXPRESSION, "ValaUnaryExpression", &g_define_type_info, 0);
	}
	return vala_unary_expression_type_id;
}




