/* valasemanticanalyzer.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <rasa@gmx.ch>
 */

#include <vala/valasemanticanalyzer.h>
#include <gee/collection.h>
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/iterable.h>
#include <gee/iterator.h>
#include <vala/valadatatype.h>
#include <vala/valascope.h>
#include <vala/valareport.h>
#include <vala/valasourcereference.h>
#include <vala/valanamespace.h>
#include <vala/valaattribute.h>
#include <vala/valaexpression.h>
#include <vala/valastatement.h>
#include <vala/valaarray.h>
#include <vala/valaswitchlabel.h>
#include <vala/valalockable.h>
#include <vala/valaliteral.h>
#include <vala/valatypeparameter.h>
#include <vala/valaenumvalue.h>
#include <vala/valanamespacereference.h>
#include <vala/valapointer.h>
#include <vala/valaenum.h>
#include <vala/valainvokable.h>
#include <vala/valamemberinitializer.h>

struct _ValaSemanticAnalyzerPrivate {
	gboolean _memory_management;
	ValaCodeContext* context;
	ValaSymbol* root_symbol;
	ValaSymbol* current_symbol;
	ValaSourceFile* current_source_file;
	ValaTypeReference* current_return_type;
	ValaClass* current_class;
	ValaStruct* current_struct;
	GeeCollection* current_using_directives;
	ValaTypeReference* bool_type;
	ValaTypeReference* string_type;
	ValaTypeReference* int_type;
	ValaTypeReference* uint_type;
	ValaTypeReference* ulong_type;
	ValaTypeReference* unichar_type;
	ValaTypeReference* type_type;
	ValaDataType* pointer_type;
	ValaDataType* object_type;
	ValaDataType* initially_unowned_type;
	ValaDataType* glist_type;
	ValaDataType* gslist_type;
	ValaDataType* gerror_type;
	ValaDataType* iterable_type;
	ValaDataType* iterator_type;
	ValaDataType* list_type;
	ValaDataType* map_type;
	gint next_lambda_id;
	GeeCollection* binding_providers;
};
#define VALA_SEMANTIC_ANALYZER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_SEMANTIC_ANALYZER, ValaSemanticAnalyzerPrivate))
enum  {
	VALA_SEMANTIC_ANALYZER_DUMMY_PROPERTY,
	VALA_SEMANTIC_ANALYZER_MEMORY_MANAGEMENT
};
static void vala_semantic_analyzer_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* file);
static void vala_semantic_analyzer_real_visit_class (ValaCodeVisitor* base, ValaClass* cl);
static void vala_semantic_analyzer_get_all_prerequisites (ValaSemanticAnalyzer* self, ValaInterface* iface, GeeCollection* list);
static gboolean vala_semantic_analyzer_class_is_a (ValaSemanticAnalyzer* self, ValaClass* cl, ValaDataType* t);
static void vala_semantic_analyzer_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st);
static void vala_semantic_analyzer_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface);
static void vala_semantic_analyzer_real_visit_callback (ValaCodeVisitor* base, ValaCallback* cb);
static void vala_semantic_analyzer_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c);
static void vala_semantic_analyzer_real_visit_field (ValaCodeVisitor* base, ValaField* f);
static void vala_semantic_analyzer_real_visit_method (ValaCodeVisitor* base, ValaMethod* m);
static void vala_semantic_analyzer_find_base_class_method (ValaSemanticAnalyzer* self, ValaMethod* m, ValaClass* cl);
static void vala_semantic_analyzer_find_base_interface_method (ValaSemanticAnalyzer* self, ValaMethod* m, ValaClass* cl);
static void vala_semantic_analyzer_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m);
static void vala_semantic_analyzer_real_visit_formal_parameter (ValaCodeVisitor* base, ValaFormalParameter* p);
static void vala_semantic_analyzer_find_base_class_property (ValaSemanticAnalyzer* self, ValaProperty* prop, ValaClass* cl);
static void vala_semantic_analyzer_find_base_interface_property (ValaSemanticAnalyzer* self, ValaProperty* prop, ValaClass* cl);
static void vala_semantic_analyzer_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop);
static void vala_semantic_analyzer_real_visit_property_accessor (ValaCodeVisitor* base, ValaPropertyAccessor* acc);
static void vala_semantic_analyzer_real_visit_signal (ValaCodeVisitor* base, ValaSignal* sig);
static void vala_semantic_analyzer_real_visit_constructor (ValaCodeVisitor* base, ValaConstructor* c);
static void vala_semantic_analyzer_real_visit_destructor (ValaCodeVisitor* base, ValaDestructor* d);
static void vala_semantic_analyzer_real_visit_named_argument (ValaCodeVisitor* base, ValaNamedArgument* n);
static void vala_semantic_analyzer_real_visit_block (ValaCodeVisitor* base, ValaBlock* b);
static void vala_semantic_analyzer_real_visit_variable_declarator (ValaCodeVisitor* base, ValaVariableDeclarator* decl);
static void vala_semantic_analyzer_real_visit_initializer_list (ValaCodeVisitor* base, ValaInitializerList* list);
static void vala_semantic_analyzer_real_visit_expression_statement (ValaCodeVisitor* base, ValaExpressionStatement* stmt);
static void vala_semantic_analyzer_real_visit_if_statement (ValaCodeVisitor* base, ValaIfStatement* stmt);
static void vala_semantic_analyzer_real_visit_switch_section (ValaCodeVisitor* base, ValaSwitchSection* section);
static void vala_semantic_analyzer_real_visit_while_statement (ValaCodeVisitor* base, ValaWhileStatement* stmt);
static void vala_semantic_analyzer_real_visit_for_statement (ValaCodeVisitor* base, ValaForStatement* stmt);
static void vala_semantic_analyzer_real_visit_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt);
static void vala_semantic_analyzer_real_visit_return_statement (ValaCodeVisitor* base, ValaReturnStatement* stmt);
static void vala_semantic_analyzer_real_visit_throw_statement (ValaCodeVisitor* base, ValaThrowStatement* stmt);
static void vala_semantic_analyzer_real_visit_try_statement (ValaCodeVisitor* base, ValaTryStatement* stmt);
static void vala_semantic_analyzer_real_visit_catch_clause (ValaCodeVisitor* base, ValaCatchClause* clause);
static void vala_semantic_analyzer_real_visit_lock_statement (ValaCodeVisitor* base, ValaLockStatement* stmt);
static gint vala_semantic_analyzer_create_sizes_from_initializer_list (ValaSemanticAnalyzer* self, ValaInitializerList* il, gint rank, GeeList* sl);
static void vala_semantic_analyzer_real_visit_array_creation_expression (ValaCodeVisitor* base, ValaArrayCreationExpression* expr);
static void vala_semantic_analyzer_real_visit_boolean_literal (ValaCodeVisitor* base, ValaBooleanLiteral* expr);
static void vala_semantic_analyzer_real_visit_character_literal (ValaCodeVisitor* base, ValaCharacterLiteral* expr);
static void vala_semantic_analyzer_real_visit_integer_literal (ValaCodeVisitor* base, ValaIntegerLiteral* expr);
static void vala_semantic_analyzer_real_visit_real_literal (ValaCodeVisitor* base, ValaRealLiteral* expr);
static void vala_semantic_analyzer_real_visit_string_literal (ValaCodeVisitor* base, ValaStringLiteral* expr);
static void vala_semantic_analyzer_real_visit_null_literal (ValaCodeVisitor* base, ValaNullLiteral* expr);
static void vala_semantic_analyzer_real_visit_literal_expression (ValaCodeVisitor* base, ValaLiteralExpression* expr);
static ValaTypeReference* vala_semantic_analyzer_get_static_type_for_symbol (ValaSemanticAnalyzer* self, ValaSymbol* sym);
static void vala_semantic_analyzer_real_visit_parenthesized_expression (ValaCodeVisitor* base, ValaParenthesizedExpression* expr);
static void vala_semantic_analyzer_real_visit_member_access (ValaCodeVisitor* base, ValaMemberAccess* expr);
static gboolean vala_semantic_analyzer_is_type_compatible (ValaSemanticAnalyzer* self, ValaTypeReference* expression_type, ValaTypeReference* expected_type);
static void vala_semantic_analyzer_real_visit_invocation_expression (ValaCodeVisitor* base, ValaInvocationExpression* expr);
static gboolean vala_semantic_analyzer_check_arguments (ValaSemanticAnalyzer* self, ValaExpression* expr, ValaSymbol* msym, GeeCollection* params, GeeCollection* args);
static void vala_semantic_analyzer_real_visit_element_access (ValaCodeVisitor* base, ValaElementAccess* expr);
static void vala_semantic_analyzer_real_visit_base_access (ValaCodeVisitor* base, ValaBaseAccess* expr);
static void vala_semantic_analyzer_real_visit_postfix_expression (ValaCodeVisitor* base, ValaPostfixExpression* expr);
static void vala_semantic_analyzer_real_visit_object_creation_expression (ValaCodeVisitor* base, ValaObjectCreationExpression* expr);
static void vala_semantic_analyzer_real_visit_sizeof_expression (ValaCodeVisitor* base, ValaSizeofExpression* expr);
static void vala_semantic_analyzer_real_visit_typeof_expression (ValaCodeVisitor* base, ValaTypeofExpression* expr);
static gboolean vala_semantic_analyzer_is_numeric_type (ValaSemanticAnalyzer* self, ValaTypeReference* type);
static gboolean vala_semantic_analyzer_is_integer_type (ValaSemanticAnalyzer* self, ValaTypeReference* type);
static void vala_semantic_analyzer_real_visit_unary_expression (ValaCodeVisitor* base, ValaUnaryExpression* expr);
static ValaMemberAccess* vala_semantic_analyzer_find_member_access (ValaSemanticAnalyzer* self, ValaExpression* expr);
static void vala_semantic_analyzer_real_visit_cast_expression (ValaCodeVisitor* base, ValaCastExpression* expr);
static void vala_semantic_analyzer_real_visit_pointer_indirection (ValaCodeVisitor* base, ValaPointerIndirection* expr);
static void vala_semantic_analyzer_real_visit_addressof_expression (ValaCodeVisitor* base, ValaAddressofExpression* expr);
static void vala_semantic_analyzer_real_visit_reference_transfer_expression (ValaCodeVisitor* base, ValaReferenceTransferExpression* expr);
static ValaTypeReference* vala_semantic_analyzer_get_arithmetic_result_type (ValaSemanticAnalyzer* self, ValaTypeReference* left_type, ValaTypeReference* right_type);
static void vala_semantic_analyzer_real_visit_binary_expression (ValaCodeVisitor* base, ValaBinaryExpression* expr);
static void vala_semantic_analyzer_real_visit_type_check (ValaCodeVisitor* base, ValaTypeCheck* expr);
static ValaTypeReference* vala_semantic_analyzer_compute_common_base_type (ValaSemanticAnalyzer* self, GeeCollection* types, ValaSourceReference* source_reference);
static void vala_semantic_analyzer_real_visit_conditional_expression (ValaCodeVisitor* base, ValaConditionalExpression* expr);
static char* vala_semantic_analyzer_get_lambda_name (ValaSemanticAnalyzer* self);
static ValaMethod* vala_semantic_analyzer_find_current_method (ValaSemanticAnalyzer* self);
static gboolean vala_semantic_analyzer_is_in_constructor (ValaSemanticAnalyzer* self);
static void vala_semantic_analyzer_real_visit_lambda_expression (ValaCodeVisitor* base, ValaLambdaExpression* l);
static void vala_semantic_analyzer_real_visit_assignment (ValaCodeVisitor* base, ValaAssignment* a);
static gpointer vala_semantic_analyzer_parent_class = NULL;
static void vala_semantic_analyzer_dispose (GObject * obj);


ValaSemanticAnalyzer* vala_semantic_analyzer_new (gboolean manage_memory) {
	ValaSemanticAnalyzer * self;
	self = g_object_newv (VALA_TYPE_SEMANTIC_ANALYZER, 0, NULL);
	vala_semantic_analyzer_set_memory_management (self, manage_memory);
	return self;
}


void vala_semantic_analyzer_add_binding_provider (ValaSemanticAnalyzer* self, ValaBindingProvider* binding_provider) {
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (VALA_IS_BINDING_PROVIDER (binding_provider));
	gee_collection_add (self->priv->binding_providers, binding_provider);
}


/**
 * Analyze and check code in the specified context.
 *
 * @param context a code context
 */
void vala_semantic_analyzer_analyze (ValaSemanticAnalyzer* self, ValaCodeContext* context) {
	ValaCodeContext* _tmp0;
	ValaSymbol* _tmp1;
	ValaTypeReference* _tmp2;
	ValaDataType* _tmp3;
	ValaTypeReference* _tmp4;
	ValaDataType* _tmp5;
	ValaDataType* _tmp6;
	ValaTypeReference* _tmp7;
	ValaDataType* _tmp8;
	ValaTypeReference* _tmp9;
	ValaDataType* _tmp10;
	ValaTypeReference* _tmp11;
	ValaDataType* _tmp12;
	ValaTypeReference* _tmp13;
	ValaDataType* _tmp14;
	ValaSymbol* glib_ns;
	ValaSymbol* gee_ns;
	ValaSymbol* _tmp27;
	ValaSymbol* _tmp26;
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (VALA_IS_CODE_CONTEXT (context));
	_tmp0 = NULL;
	self->priv->context = (_tmp0 = g_object_ref (context), (self->priv->context == NULL ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL))), _tmp0);
	_tmp1 = NULL;
	self->priv->root_symbol = (_tmp1 = VALA_SYMBOL (g_object_ref (vala_code_context_get_root (context))), (self->priv->root_symbol == NULL ? NULL : (self->priv->root_symbol = (g_object_unref (self->priv->root_symbol), NULL))), _tmp1);
	_tmp2 = NULL;
	self->priv->bool_type = (_tmp2 = vala_type_reference_new (), (self->priv->bool_type == NULL ? NULL : (self->priv->bool_type = (g_object_unref (self->priv->bool_type), NULL))), _tmp2);
	_tmp3 = NULL;
	vala_type_reference_set_data_type (self->priv->bool_type, (_tmp3 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), "bool"))));
	(_tmp3 == NULL ? NULL : (_tmp3 = (g_object_unref (_tmp3), NULL)));
	_tmp4 = NULL;
	self->priv->string_type = (_tmp4 = vala_type_reference_new (), (self->priv->string_type == NULL ? NULL : (self->priv->string_type = (g_object_unref (self->priv->string_type), NULL))), _tmp4);
	_tmp5 = NULL;
	vala_type_reference_set_data_type (self->priv->string_type, (_tmp5 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), "string"))));
	(_tmp5 == NULL ? NULL : (_tmp5 = (g_object_unref (_tmp5), NULL)));
	_tmp6 = NULL;
	self->priv->pointer_type = (_tmp6 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), "pointer")), (self->priv->pointer_type == NULL ? NULL : (self->priv->pointer_type = (g_object_unref (self->priv->pointer_type), NULL))), _tmp6);
	_tmp7 = NULL;
	self->priv->int_type = (_tmp7 = vala_type_reference_new (), (self->priv->int_type == NULL ? NULL : (self->priv->int_type = (g_object_unref (self->priv->int_type), NULL))), _tmp7);
	_tmp8 = NULL;
	vala_type_reference_set_data_type (self->priv->int_type, (_tmp8 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), "int"))));
	(_tmp8 == NULL ? NULL : (_tmp8 = (g_object_unref (_tmp8), NULL)));
	_tmp9 = NULL;
	self->priv->uint_type = (_tmp9 = vala_type_reference_new (), (self->priv->uint_type == NULL ? NULL : (self->priv->uint_type = (g_object_unref (self->priv->uint_type), NULL))), _tmp9);
	_tmp10 = NULL;
	vala_type_reference_set_data_type (self->priv->uint_type, (_tmp10 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), "uint"))));
	(_tmp10 == NULL ? NULL : (_tmp10 = (g_object_unref (_tmp10), NULL)));
	_tmp11 = NULL;
	self->priv->ulong_type = (_tmp11 = vala_type_reference_new (), (self->priv->ulong_type == NULL ? NULL : (self->priv->ulong_type = (g_object_unref (self->priv->ulong_type), NULL))), _tmp11);
	_tmp12 = NULL;
	vala_type_reference_set_data_type (self->priv->ulong_type, (_tmp12 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), "ulong"))));
	(_tmp12 == NULL ? NULL : (_tmp12 = (g_object_unref (_tmp12), NULL)));
	_tmp13 = NULL;
	self->priv->unichar_type = (_tmp13 = vala_type_reference_new (), (self->priv->unichar_type == NULL ? NULL : (self->priv->unichar_type = (g_object_unref (self->priv->unichar_type), NULL))), _tmp13);
	_tmp14 = NULL;
	vala_type_reference_set_data_type (self->priv->unichar_type, (_tmp14 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), "unichar"))));
	(_tmp14 == NULL ? NULL : (_tmp14 = (g_object_unref (_tmp14), NULL)));
	/* TODO: don't require GLib namespace in semantic analyzer*/
	glib_ns = vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), "GLib");
	if (glib_ns != NULL) {
		ValaDataType* _tmp15;
		ValaDataType* _tmp16;
		ValaTypeReference* _tmp17;
		ValaDataType* _tmp18;
		ValaDataType* _tmp19;
		ValaDataType* _tmp20;
		ValaDataType* _tmp21;
		_tmp15 = NULL;
		self->priv->object_type = (_tmp15 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "Object")), (self->priv->object_type == NULL ? NULL : (self->priv->object_type = (g_object_unref (self->priv->object_type), NULL))), _tmp15);
		_tmp16 = NULL;
		self->priv->initially_unowned_type = (_tmp16 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "InitiallyUnowned")), (self->priv->initially_unowned_type == NULL ? NULL : (self->priv->initially_unowned_type = (g_object_unref (self->priv->initially_unowned_type), NULL))), _tmp16);
		_tmp17 = NULL;
		self->priv->type_type = (_tmp17 = vala_type_reference_new (), (self->priv->type_type == NULL ? NULL : (self->priv->type_type = (g_object_unref (self->priv->type_type), NULL))), _tmp17);
		_tmp18 = NULL;
		vala_type_reference_set_data_type (self->priv->type_type, (_tmp18 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "Type"))));
		(_tmp18 == NULL ? NULL : (_tmp18 = (g_object_unref (_tmp18), NULL)));
		_tmp19 = NULL;
		self->priv->glist_type = (_tmp19 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "List")), (self->priv->glist_type == NULL ? NULL : (self->priv->glist_type = (g_object_unref (self->priv->glist_type), NULL))), _tmp19);
		_tmp20 = NULL;
		self->priv->gslist_type = (_tmp20 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "SList")), (self->priv->gslist_type == NULL ? NULL : (self->priv->gslist_type = (g_object_unref (self->priv->gslist_type), NULL))), _tmp20);
		_tmp21 = NULL;
		self->priv->gerror_type = (_tmp21 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (glib_ns), "Error")), (self->priv->gerror_type == NULL ? NULL : (self->priv->gerror_type = (g_object_unref (self->priv->gerror_type), NULL))), _tmp21);
	}
	gee_ns = vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), "Gee");
	if (gee_ns != NULL) {
		ValaDataType* _tmp22;
		ValaDataType* _tmp23;
		ValaDataType* _tmp24;
		ValaDataType* _tmp25;
		_tmp22 = NULL;
		self->priv->iterable_type = (_tmp22 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (gee_ns), "Iterable")), (self->priv->iterable_type == NULL ? NULL : (self->priv->iterable_type = (g_object_unref (self->priv->iterable_type), NULL))), _tmp22);
		_tmp23 = NULL;
		self->priv->iterator_type = (_tmp23 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (gee_ns), "Iterator")), (self->priv->iterator_type == NULL ? NULL : (self->priv->iterator_type = (g_object_unref (self->priv->iterator_type), NULL))), _tmp23);
		_tmp24 = NULL;
		self->priv->list_type = (_tmp24 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (gee_ns), "List")), (self->priv->list_type == NULL ? NULL : (self->priv->list_type = (g_object_unref (self->priv->list_type), NULL))), _tmp24);
		_tmp25 = NULL;
		self->priv->map_type = (_tmp25 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (gee_ns), "Map")), (self->priv->map_type == NULL ? NULL : (self->priv->map_type = (g_object_unref (self->priv->map_type), NULL))), _tmp25);
	}
	_tmp27 = NULL;
	_tmp26 = NULL;
	self->priv->current_symbol = (_tmp27 = (_tmp26 = self->priv->root_symbol, (_tmp26 == NULL ? NULL : g_object_ref (_tmp26))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp27);
	vala_code_context_accept (context, VALA_CODE_VISITOR (self));
	(glib_ns == NULL ? NULL : (glib_ns = (g_object_unref (glib_ns), NULL)));
	(gee_ns == NULL ? NULL : (gee_ns = (g_object_unref (gee_ns), NULL)));
}


static void vala_semantic_analyzer_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* file) {
	ValaSemanticAnalyzer * self;
	ValaSourceFile* _tmp0;
	GeeCollection* _tmp1;
	GeeCollection* _tmp2;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_SOURCE_FILE (file));
	_tmp0 = NULL;
	self->priv->current_source_file = (_tmp0 = g_object_ref (file), (self->priv->current_source_file == NULL ? NULL : (self->priv->current_source_file = (g_object_unref (self->priv->current_source_file), NULL))), _tmp0);
	_tmp1 = NULL;
	self->priv->current_using_directives = (_tmp1 = vala_source_file_get_using_directives (file), (self->priv->current_using_directives == NULL ? NULL : (self->priv->current_using_directives = (g_object_unref (self->priv->current_using_directives), NULL))), _tmp1);
	self->priv->next_lambda_id = 0;
	vala_source_file_accept_children (file, VALA_CODE_VISITOR (self));
	_tmp2 = NULL;
	self->priv->current_using_directives = (_tmp2 = NULL, (self->priv->current_using_directives == NULL ? NULL : (self->priv->current_using_directives = (g_object_unref (self->priv->current_using_directives), NULL))), _tmp2);
}


static void vala_semantic_analyzer_real_visit_class (ValaCodeVisitor* base, ValaClass* cl) {
	ValaSemanticAnalyzer * self;
	ValaSymbol* _tmp0;
	ValaClass* _tmp1;
	GeeList* prerequisites;
	GeeList* missing_prereqs;
	ValaSymbol* _tmp19;
	ValaSymbol* _tmp18;
	ValaClass* _tmp20;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CLASS (cl));
	_tmp0 = NULL;
	self->priv->current_symbol = (_tmp0 = VALA_SYMBOL (g_object_ref (cl)), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp0);
	_tmp1 = NULL;
	self->priv->current_class = (_tmp1 = g_object_ref (cl), (self->priv->current_class == NULL ? NULL : (self->priv->current_class = (g_object_unref (self->priv->current_class), NULL))), _tmp1);
	if (vala_class_get_base_class (cl) != NULL) {
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_class_get_base_class (cl)), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_FULL);
	}
	{
		GeeCollection* base_type_reference_collection;
		GeeIterator* base_type_reference_it;
		base_type_reference_collection = vala_class_get_base_types (cl);
		base_type_reference_it = gee_iterable_iterator (GEE_ITERABLE (base_type_reference_collection));
		while (gee_iterator_next (base_type_reference_it)) {
			ValaTypeReference* base_type_reference;
			base_type_reference = gee_iterator_get (base_type_reference_it);
			{
				vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (base_type_reference)), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_FULL);
				(base_type_reference == NULL ? NULL : (base_type_reference = (g_object_unref (base_type_reference), NULL)));
			}
		}
		(base_type_reference_collection == NULL ? NULL : (base_type_reference_collection = (g_object_unref (base_type_reference_collection), NULL)));
		(base_type_reference_it == NULL ? NULL : (base_type_reference_it = (g_object_unref (base_type_reference_it), NULL)));
	}
	vala_code_node_accept_children (VALA_CODE_NODE (cl), VALA_CODE_VISITOR (self));
	/* gather all prerequisites */
	prerequisites = GEE_LIST (gee_array_list_new (VALA_TYPE_DATA_TYPE, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	{
		GeeCollection* base_type_collection;
		GeeIterator* base_type_it;
		base_type_collection = vala_class_get_base_types (cl);
		base_type_it = gee_iterable_iterator (GEE_ITERABLE (base_type_collection));
		while (gee_iterator_next (base_type_it)) {
			ValaTypeReference* base_type;
			base_type = gee_iterator_get (base_type_it);
			{
				if (VALA_IS_INTERFACE (vala_type_reference_get_data_type (base_type))) {
					vala_semantic_analyzer_get_all_prerequisites (self, VALA_INTERFACE (vala_type_reference_get_data_type (base_type)), GEE_COLLECTION (prerequisites));
				}
				(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
			}
		}
		(base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL)));
		(base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL)));
	}
	/* check whether all prerequisites are met */
	missing_prereqs = GEE_LIST (gee_array_list_new (G_TYPE_STRING, ((GBoxedCopyFunc) g_strdup), g_free, g_direct_equal));
	{
		GeeList* prereq_collection;
		GeeIterator* prereq_it;
		prereq_collection = prerequisites;
		prereq_it = gee_iterable_iterator (GEE_ITERABLE (prereq_collection));
		while (gee_iterator_next (prereq_it)) {
			ValaDataType* prereq;
			prereq = gee_iterator_get (prereq_it);
			{
				if (!vala_semantic_analyzer_class_is_a (self, cl, prereq)) {
					char* _tmp2;
					_tmp2 = NULL;
					gee_list_insert (missing_prereqs, 0, (_tmp2 = vala_symbol_get_full_name (VALA_SYMBOL (prereq))));
					(_tmp2 = (g_free (_tmp2), NULL));
				}
				(prereq == NULL ? NULL : (prereq = (g_object_unref (prereq), NULL)));
			}
		}
		(prereq_it == NULL ? NULL : (prereq_it = (g_object_unref (prereq_it), NULL)));
	}
	/* report any missing prerequisites */
	if (gee_collection_get_size (GEE_COLLECTION (missing_prereqs)) > 0) {
		char* _tmp3;
		char* _tmp4;
		char* error_string;
		gboolean first;
		char* _tmp7;
		vala_code_node_set_error (VALA_CODE_NODE (cl), TRUE);
		_tmp3 = NULL;
		_tmp4 = NULL;
		error_string = (_tmp4 = g_strdup_printf ("%s: some prerequisites (", (_tmp3 = vala_symbol_get_full_name (VALA_SYMBOL (cl)))), (_tmp3 = (g_free (_tmp3), NULL)), _tmp4);
		first = TRUE;
		{
			GeeList* s_collection;
			GeeIterator* s_it;
			s_collection = missing_prereqs;
			s_it = gee_iterable_iterator (GEE_ITERABLE (s_collection));
			while (gee_iterator_next (s_it)) {
				char* s;
				s = gee_iterator_get (s_it);
				{
					if (first) {
						char* _tmp5;
						_tmp5 = NULL;
						error_string = (_tmp5 = g_strdup_printf ("%s`%s'", error_string, s), (error_string = (g_free (error_string), NULL)), _tmp5);
						first = FALSE;
					} else {
						char* _tmp6;
						_tmp6 = NULL;
						error_string = (_tmp6 = g_strdup_printf ("%s, `%s'", error_string, s), (error_string = (g_free (error_string), NULL)), _tmp6);
					}
					(s = (g_free (s), NULL));
				}
			}
			(s_it == NULL ? NULL : (s_it = (g_object_unref (s_it), NULL)));
		}
		_tmp7 = NULL;
		error_string = (_tmp7 = g_strconcat (error_string, (") are not met"), NULL), (error_string = (g_free (error_string), NULL)), _tmp7);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (cl)), error_string);
		(error_string = (g_free (error_string), NULL));
	}
	/* VAPI classes don't have to specify overridden methods */
	if (!vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (cl))))) {
		{
			GeeCollection* base_type_collection;
			GeeIterator* base_type_it;
			base_type_collection = vala_class_get_base_types (cl);
			base_type_it = gee_iterable_iterator (GEE_ITERABLE (base_type_collection));
			while (gee_iterator_next (base_type_it)) {
				ValaTypeReference* base_type;
				base_type = gee_iterator_get (base_type_it);
				{
					/* all abstract symbols defined in base types have to be at least defined (or implemented) also in this type */
					if (VALA_IS_INTERFACE (vala_type_reference_get_data_type (base_type))) {
						ValaInterface* _tmp8;
						ValaInterface* iface;
						_tmp8 = NULL;
						iface = (_tmp8 = VALA_INTERFACE (vala_type_reference_get_data_type (base_type)), (_tmp8 == NULL ? NULL : g_object_ref (_tmp8)));
						{
							GeeCollection* m_collection;
							GeeIterator* m_it;
							m_collection = vala_interface_get_methods (iface);
							m_it = gee_iterable_iterator (GEE_ITERABLE (m_collection));
							while (gee_iterator_next (m_it)) {
								ValaMethod* m;
								m = gee_iterator_get (m_it);
								{
									/* We do not need to do expensive equality checking here since this is done
									 * already. We only need to guarantee the symbols are present.
									 
									 check methods */
									if (vala_method_get_is_abstract (m)) {
										ValaSymbol* sym;
										sym = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (cl)), vala_symbol_get_name (VALA_SYMBOL (m)));
										if (sym == NULL || !(VALA_IS_METHOD (sym)) || vala_method_get_base_interface_method ((VALA_METHOD (sym))) != m) {
											char* _tmp11;
											char* _tmp10;
											char* _tmp9;
											vala_code_node_set_error (VALA_CODE_NODE (cl), TRUE);
											_tmp11 = NULL;
											_tmp10 = NULL;
											_tmp9 = NULL;
											vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (cl)), (_tmp11 = g_strdup_printf ("`%s' does not implement interface method `%s'", (_tmp9 = vala_symbol_get_full_name (VALA_SYMBOL (cl))), (_tmp10 = vala_symbol_get_full_name (VALA_SYMBOL (m))))));
											(_tmp11 = (g_free (_tmp11), NULL));
											(_tmp10 = (g_free (_tmp10), NULL));
											(_tmp9 = (g_free (_tmp9), NULL));
										}
										(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
									}
									(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
								}
							}
							(m_collection == NULL ? NULL : (m_collection = (g_object_unref (m_collection), NULL)));
							(m_it == NULL ? NULL : (m_it = (g_object_unref (m_it), NULL)));
						}
						(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
					}
					(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
				}
			}
			(base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL)));
			(base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL)));
		}
		/* all abstract symbols defined in base classes have to be implemented in non-abstract classes */
		if (!vala_class_get_is_abstract (cl)) {
			ValaClass* _tmp12;
			ValaClass* base_class;
			_tmp12 = NULL;
			base_class = (_tmp12 = vala_class_get_base_class (cl), (_tmp12 == NULL ? NULL : g_object_ref (_tmp12)));
			while (base_class != NULL && vala_class_get_is_abstract (base_class)) {
				ValaClass* _tmp17;
				ValaClass* _tmp16;
				{
					GeeCollection* m_collection;
					GeeIterator* m_it;
					m_collection = vala_class_get_methods (base_class);
					m_it = gee_iterable_iterator (GEE_ITERABLE (m_collection));
					while (gee_iterator_next (m_it)) {
						ValaMethod* m;
						m = gee_iterator_get (m_it);
						{
							if (vala_method_get_is_abstract (m)) {
								ValaSymbol* sym;
								sym = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (cl)), vala_symbol_get_name (VALA_SYMBOL (m)));
								if (sym == NULL || !(VALA_IS_METHOD (sym)) || vala_method_get_base_method ((VALA_METHOD (sym))) != m) {
									char* _tmp15;
									char* _tmp14;
									char* _tmp13;
									vala_code_node_set_error (VALA_CODE_NODE (cl), TRUE);
									_tmp15 = NULL;
									_tmp14 = NULL;
									_tmp13 = NULL;
									vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (cl)), (_tmp15 = g_strdup_printf ("`%s' does not implement abstract method `%s'", (_tmp13 = vala_symbol_get_full_name (VALA_SYMBOL (cl))), (_tmp14 = vala_symbol_get_full_name (VALA_SYMBOL (m))))));
									(_tmp15 = (g_free (_tmp15), NULL));
									(_tmp14 = (g_free (_tmp14), NULL));
									(_tmp13 = (g_free (_tmp13), NULL));
								}
								(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
							}
							(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
						}
					}
					(m_collection == NULL ? NULL : (m_collection = (g_object_unref (m_collection), NULL)));
					(m_it == NULL ? NULL : (m_it = (g_object_unref (m_it), NULL)));
				}
				_tmp17 = NULL;
				_tmp16 = NULL;
				base_class = (_tmp17 = (_tmp16 = vala_class_get_base_class (base_class), (_tmp16 == NULL ? NULL : g_object_ref (_tmp16))), (base_class == NULL ? NULL : (base_class = (g_object_unref (base_class), NULL))), _tmp17);
			}
			(base_class == NULL ? NULL : (base_class = (g_object_unref (base_class), NULL)));
		}
	}
	_tmp19 = NULL;
	_tmp18 = NULL;
	self->priv->current_symbol = (_tmp19 = (_tmp18 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (_tmp18 == NULL ? NULL : g_object_ref (_tmp18))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp19);
	_tmp20 = NULL;
	self->priv->current_class = (_tmp20 = NULL, (self->priv->current_class == NULL ? NULL : (self->priv->current_class = (g_object_unref (self->priv->current_class), NULL))), _tmp20);
	(prerequisites == NULL ? NULL : (prerequisites = (g_object_unref (prerequisites), NULL)));
	(missing_prereqs == NULL ? NULL : (missing_prereqs = (g_object_unref (missing_prereqs), NULL)));
}


static void vala_semantic_analyzer_get_all_prerequisites (ValaSemanticAnalyzer* self, ValaInterface* iface, GeeCollection* list) {
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (VALA_IS_INTERFACE (iface));
	g_return_if_fail (list == NULL || GEE_IS_COLLECTION (list));
	{
		GeeCollection* prereq_collection;
		GeeIterator* prereq_it;
		prereq_collection = vala_interface_get_prerequisites (iface);
		prereq_it = gee_iterable_iterator (GEE_ITERABLE (prereq_collection));
		while (gee_iterator_next (prereq_it)) {
			ValaTypeReference* prereq;
			prereq = gee_iterator_get (prereq_it);
			{
				ValaDataType* _tmp0;
				ValaDataType* type;
				_tmp0 = NULL;
				type = (_tmp0 = vala_type_reference_get_data_type (prereq), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
				/* skip on previous errors */
				if (type == NULL) {
					(prereq == NULL ? NULL : (prereq = (g_object_unref (prereq), NULL)));
					(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
					continue;
				}
				gee_collection_add (list, type);
				if (VALA_IS_INTERFACE (type)) {
					vala_semantic_analyzer_get_all_prerequisites (self, VALA_INTERFACE (type), list);
				}
				(prereq == NULL ? NULL : (prereq = (g_object_unref (prereq), NULL)));
				(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			}
		}
		(prereq_collection == NULL ? NULL : (prereq_collection = (g_object_unref (prereq_collection), NULL)));
		(prereq_it == NULL ? NULL : (prereq_it = (g_object_unref (prereq_it), NULL)));
	}
}


static gboolean vala_semantic_analyzer_class_is_a (ValaSemanticAnalyzer* self, ValaClass* cl, ValaDataType* t) {
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	g_return_val_if_fail (VALA_IS_CLASS (cl), FALSE);
	g_return_val_if_fail (VALA_IS_DATA_TYPE (t), FALSE);
	if (VALA_DATA_TYPE (cl) == t) {
		return TRUE;
	}
	{
		GeeCollection* base_type_collection;
		GeeIterator* base_type_it;
		base_type_collection = vala_class_get_base_types (cl);
		base_type_it = gee_iterable_iterator (GEE_ITERABLE (base_type_collection));
		while (gee_iterator_next (base_type_it)) {
			ValaTypeReference* base_type;
			base_type = gee_iterator_get (base_type_it);
			{
				if (VALA_IS_CLASS (vala_type_reference_get_data_type (base_type))) {
					if (vala_semantic_analyzer_class_is_a (self, VALA_CLASS (vala_type_reference_get_data_type (base_type)), t)) {
						gboolean _tmp1;
						return (_tmp1 = TRUE, (base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL))), (base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL))), (base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL))), _tmp1);
					}
				} else {
					if (vala_type_reference_get_data_type (base_type) == t) {
						gboolean _tmp2;
						return (_tmp2 = TRUE, (base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL))), (base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL))), (base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL))), _tmp2);
					}
				}
				(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
			}
		}
		(base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL)));
		(base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL)));
	}
	return FALSE;
}


static void vala_semantic_analyzer_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st) {
	ValaSemanticAnalyzer * self;
	ValaSymbol* _tmp0;
	ValaStruct* _tmp1;
	ValaSymbol* _tmp3;
	ValaSymbol* _tmp2;
	ValaStruct* _tmp4;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_STRUCT (st));
	_tmp0 = NULL;
	self->priv->current_symbol = (_tmp0 = VALA_SYMBOL (g_object_ref (st)), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp0);
	_tmp1 = NULL;
	self->priv->current_struct = (_tmp1 = g_object_ref (st), (self->priv->current_struct == NULL ? NULL : (self->priv->current_struct = (g_object_unref (self->priv->current_struct), NULL))), _tmp1);
	vala_code_node_accept_children (VALA_CODE_NODE (st), VALA_CODE_VISITOR (self));
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->current_symbol = (_tmp3 = (_tmp2 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp3);
	_tmp4 = NULL;
	self->priv->current_struct = (_tmp4 = NULL, (self->priv->current_struct == NULL ? NULL : (self->priv->current_struct = (g_object_unref (self->priv->current_struct), NULL))), _tmp4);
}


static void vala_semantic_analyzer_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface) {
	ValaSemanticAnalyzer * self;
	ValaSymbol* _tmp0;
	ValaClass* prereq_class;
	ValaSymbol* _tmp9;
	ValaSymbol* _tmp8;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_INTERFACE (iface));
	_tmp0 = NULL;
	self->priv->current_symbol = (_tmp0 = VALA_SYMBOL (g_object_ref (iface)), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp0);
	{
		GeeCollection* prerequisite_reference_collection;
		GeeIterator* prerequisite_reference_it;
		prerequisite_reference_collection = vala_interface_get_prerequisites (iface);
		prerequisite_reference_it = gee_iterable_iterator (GEE_ITERABLE (prerequisite_reference_collection));
		while (gee_iterator_next (prerequisite_reference_it)) {
			ValaTypeReference* prerequisite_reference;
			prerequisite_reference = gee_iterator_get (prerequisite_reference_it);
			{
				vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (prerequisite_reference)), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_FULL);
				(prerequisite_reference == NULL ? NULL : (prerequisite_reference = (g_object_unref (prerequisite_reference), NULL)));
			}
		}
		(prerequisite_reference_collection == NULL ? NULL : (prerequisite_reference_collection = (g_object_unref (prerequisite_reference_collection), NULL)));
		(prerequisite_reference_it == NULL ? NULL : (prerequisite_reference_it = (g_object_unref (prerequisite_reference_it), NULL)));
	}
	/* check prerequisites */
	prereq_class = NULL;
	{
		GeeCollection* prereq_collection;
		GeeIterator* prereq_it;
		prereq_collection = vala_interface_get_prerequisites (iface);
		prereq_it = gee_iterable_iterator (GEE_ITERABLE (prereq_collection));
		while (gee_iterator_next (prereq_it)) {
			ValaTypeReference* prereq;
			prereq = gee_iterator_get (prereq_it);
			{
				ValaDataType* _tmp1;
				ValaDataType* class_or_interface;
				_tmp1 = NULL;
				class_or_interface = (_tmp1 = vala_type_reference_get_data_type (prereq), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1)));
				/* skip on previous errors */
				if (class_or_interface == NULL) {
					vala_code_node_set_error (VALA_CODE_NODE (iface), TRUE);
					(prereq == NULL ? NULL : (prereq = (g_object_unref (prereq), NULL)));
					(class_or_interface == NULL ? NULL : (class_or_interface = (g_object_unref (class_or_interface), NULL)));
					continue;
				}
				/* interfaces are not allowed to have multiple instantiable prerequisites */
				if (VALA_IS_CLASS (class_or_interface)) {
					ValaClass* _tmp7;
					ValaClass* _tmp6;
					if (prereq_class != NULL) {
						char* _tmp5;
						char* _tmp4;
						char* _tmp3;
						char* _tmp2;
						vala_code_node_set_error (VALA_CODE_NODE (iface), TRUE);
						_tmp5 = NULL;
						_tmp4 = NULL;
						_tmp3 = NULL;
						_tmp2 = NULL;
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (iface)), (_tmp5 = g_strdup_printf ("%s: Interfaces cannot have multiple instantiable prerequisites (`%s' and `%s')", (_tmp2 = vala_symbol_get_full_name (VALA_SYMBOL (iface))), (_tmp3 = vala_symbol_get_full_name (VALA_SYMBOL (class_or_interface))), (_tmp4 = vala_symbol_get_full_name (VALA_SYMBOL (prereq_class))))));
						(_tmp5 = (g_free (_tmp5), NULL));
						(_tmp4 = (g_free (_tmp4), NULL));
						(_tmp3 = (g_free (_tmp3), NULL));
						(_tmp2 = (g_free (_tmp2), NULL));
						(prereq == NULL ? NULL : (prereq = (g_object_unref (prereq), NULL)));
						(class_or_interface == NULL ? NULL : (class_or_interface = (g_object_unref (class_or_interface), NULL)));
						(prereq_collection == NULL ? NULL : (prereq_collection = (g_object_unref (prereq_collection), NULL)));
						(prereq_it == NULL ? NULL : (prereq_it = (g_object_unref (prereq_it), NULL)));
						(prereq_class == NULL ? NULL : (prereq_class = (g_object_unref (prereq_class), NULL)));
						return;
					}
					_tmp7 = NULL;
					_tmp6 = NULL;
					prereq_class = (_tmp7 = (_tmp6 = VALA_CLASS (class_or_interface), (_tmp6 == NULL ? NULL : g_object_ref (_tmp6))), (prereq_class == NULL ? NULL : (prereq_class = (g_object_unref (prereq_class), NULL))), _tmp7);
				}
				(prereq == NULL ? NULL : (prereq = (g_object_unref (prereq), NULL)));
				(class_or_interface == NULL ? NULL : (class_or_interface = (g_object_unref (class_or_interface), NULL)));
			}
		}
		(prereq_collection == NULL ? NULL : (prereq_collection = (g_object_unref (prereq_collection), NULL)));
		(prereq_it == NULL ? NULL : (prereq_it = (g_object_unref (prereq_it), NULL)));
	}
	vala_code_node_accept_children (VALA_CODE_NODE (iface), VALA_CODE_VISITOR (self));
	_tmp9 = NULL;
	_tmp8 = NULL;
	self->priv->current_symbol = (_tmp9 = (_tmp8 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (_tmp8 == NULL ? NULL : g_object_ref (_tmp8))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp9);
	(prereq_class == NULL ? NULL : (prereq_class = (g_object_unref (prereq_class), NULL)));
}


static void vala_semantic_analyzer_real_visit_callback (ValaCodeVisitor* base, ValaCallback* cb) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CALLBACK (cb));
	vala_code_node_accept_children (VALA_CODE_NODE (cb), VALA_CODE_VISITOR (self));
}


static void vala_semantic_analyzer_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CONSTANT (c));
	vala_code_node_accept_children (VALA_CODE_NODE (c), VALA_CODE_VISITOR (self));
	if (!vala_source_file_get_pkg (self->priv->current_source_file)) {
		if (vala_constant_get_initializer (c) == NULL) {
			vala_code_node_set_error (VALA_CODE_NODE (c), TRUE);
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (c)), "A const field requires a initializer to be provided");
		}
	}
}


static void vala_semantic_analyzer_real_visit_field (ValaCodeVisitor* base, ValaField* f) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_FIELD (f));
	vala_code_node_accept_children (VALA_CODE_NODE (f), VALA_CODE_VISITOR (self));
	if (!vala_symbol_is_internal_symbol (VALA_SYMBOL (f))) {
		if (vala_type_reference_get_data_type (vala_field_get_type_reference (f)) != NULL) {
			/* is null if it references a type parameter */
			vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (vala_field_get_type_reference (f))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW);
		}
	} else {
		if (VALA_IS_NAMESPACE (vala_symbol_get_parent_symbol (VALA_SYMBOL (f)))) {
			vala_code_node_set_error (VALA_CODE_NODE (f), TRUE);
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (f)), "Namespaces may not have private members");
			return;
		}
		if (vala_type_reference_get_data_type (vala_field_get_type_reference (f)) != NULL) {
			/* is null if it references a type parameter */
			vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (vala_field_get_type_reference (f))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
		}
	}
}


static void vala_semantic_analyzer_real_visit_method (ValaCodeVisitor* base, ValaMethod* m) {
	ValaSemanticAnalyzer * self;
	ValaSymbol* _tmp0;
	ValaTypeReference* _tmp2;
	ValaTypeReference* _tmp1;
	ValaAttribute* init_attr;
	ValaSymbol* _tmp4;
	ValaSymbol* _tmp3;
	ValaTypeReference* _tmp5;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_METHOD (m));
	_tmp0 = NULL;
	self->priv->current_symbol = (_tmp0 = VALA_SYMBOL (g_object_ref (m)), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp0);
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->current_return_type = (_tmp2 = (_tmp1 = vala_method_get_return_type (m), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), _tmp2);
	init_attr = vala_code_node_get_attribute (VALA_CODE_NODE (m), "ModuleInit");
	if (init_attr != NULL) {
		vala_code_context_set_module_init_method (vala_source_file_get_context (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (m)))), m);
	}
	if (vala_type_reference_get_data_type (vala_method_get_return_type (m)) != NULL) {
		/* is null if it is void or a reference to a type parameter */
		if (!vala_symbol_is_internal_symbol (VALA_SYMBOL (m))) {
			vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (vala_method_get_return_type (m))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW);
		}
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (vala_method_get_return_type (m))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	}
	vala_code_node_accept_children (VALA_CODE_NODE (m), VALA_CODE_VISITOR (self));
	_tmp4 = NULL;
	_tmp3 = NULL;
	self->priv->current_symbol = (_tmp4 = (_tmp3 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (_tmp3 == NULL ? NULL : g_object_ref (_tmp3))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp4);
	_tmp5 = NULL;
	self->priv->current_return_type = (_tmp5 = NULL, (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), _tmp5);
	if (VALA_IS_METHOD (vala_symbol_get_parent_symbol (self->priv->current_symbol))) {
		ValaMethod* _tmp6;
		ValaMethod* up_method;
		ValaTypeReference* _tmp8;
		ValaTypeReference* _tmp7;
		/* lambda expressions produce nested methods */
		_tmp6 = NULL;
		up_method = (_tmp6 = VALA_METHOD (vala_symbol_get_parent_symbol (self->priv->current_symbol)), (_tmp6 == NULL ? NULL : g_object_ref (_tmp6)));
		_tmp8 = NULL;
		_tmp7 = NULL;
		self->priv->current_return_type = (_tmp8 = (_tmp7 = vala_method_get_return_type (up_method), (_tmp7 == NULL ? NULL : g_object_ref (_tmp7))), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), _tmp8);
		(up_method == NULL ? NULL : (up_method = (g_object_unref (up_method), NULL)));
	}
	if (VALA_IS_CLASS (self->priv->current_symbol)) {
		if (!(VALA_IS_CREATION_METHOD (m))) {
			vala_semantic_analyzer_find_base_interface_method (self, m, VALA_CLASS (self->priv->current_symbol));
			if (vala_method_get_is_virtual (m) || vala_method_get_overrides (m)) {
				vala_semantic_analyzer_find_base_class_method (self, m, VALA_CLASS (self->priv->current_symbol));
				if (vala_method_get_base_method (m) == NULL) {
					char* _tmp10;
					char* _tmp9;
					_tmp10 = NULL;
					_tmp9 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), (_tmp10 = g_strdup_printf ("%s: no suitable method found to override", (_tmp9 = vala_symbol_get_full_name (VALA_SYMBOL (m))))));
					(_tmp10 = (g_free (_tmp10), NULL));
					(_tmp9 = (g_free (_tmp9), NULL));
				}
			}
		}
	} else {
		if (VALA_IS_STRUCT (self->priv->current_symbol)) {
			if (vala_method_get_is_abstract (m) || vala_method_get_is_virtual (m) || vala_method_get_overrides (m)) {
				char* _tmp12;
				char* _tmp11;
				_tmp12 = NULL;
				_tmp11 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), (_tmp12 = g_strdup_printf ("A struct member `%s' cannot be marked as override, virtual, or abstract", (_tmp11 = vala_symbol_get_full_name (VALA_SYMBOL (m))))));
				(_tmp12 = (g_free (_tmp12), NULL));
				(_tmp11 = (g_free (_tmp11), NULL));
				(init_attr == NULL ? NULL : (init_attr = (g_object_unref (init_attr), NULL)));
				return;
			}
		}
	}
	(init_attr == NULL ? NULL : (init_attr = (g_object_unref (init_attr), NULL)));
}


static void vala_semantic_analyzer_find_base_class_method (ValaSemanticAnalyzer* self, ValaMethod* m, ValaClass* cl) {
	ValaSymbol* sym;
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (VALA_IS_METHOD (m));
	g_return_if_fail (VALA_IS_CLASS (cl));
	sym = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (cl)), vala_symbol_get_name (VALA_SYMBOL (m)));
	if (VALA_IS_METHOD (sym)) {
		ValaMethod* _tmp0;
		ValaMethod* base_method;
		_tmp0 = NULL;
		base_method = (_tmp0 = VALA_METHOD (sym), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
		if (vala_method_get_is_abstract (base_method) || vala_method_get_is_virtual (base_method)) {
			if (!vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (cl)))) && !vala_method_equals (m, base_method)) {
				char* _tmp3;
				char* _tmp2;
				char* _tmp1;
				vala_code_node_set_error (VALA_CODE_NODE (m), TRUE);
				_tmp3 = NULL;
				_tmp2 = NULL;
				_tmp1 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), (_tmp3 = g_strdup_printf ("Return type and/or parameters of overriding method `%s' do not match overridden method `%s'.", (_tmp1 = vala_symbol_get_full_name (VALA_SYMBOL (m))), (_tmp2 = vala_symbol_get_full_name (VALA_SYMBOL (base_method))))));
				(_tmp3 = (g_free (_tmp3), NULL));
				(_tmp2 = (g_free (_tmp2), NULL));
				(_tmp1 = (g_free (_tmp1), NULL));
				(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
				(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
				return;
			}
			vala_method_set_base_method (m, base_method);
			(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
			(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
			return;
		}
		(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
	}
	if (vala_class_get_base_class (cl) != NULL) {
		vala_semantic_analyzer_find_base_class_method (self, m, vala_class_get_base_class (cl));
	}
	(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
}


static void vala_semantic_analyzer_find_base_interface_method (ValaSemanticAnalyzer* self, ValaMethod* m, ValaClass* cl) {
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (VALA_IS_METHOD (m));
	g_return_if_fail (VALA_IS_CLASS (cl));
	{
		GeeCollection* type_collection;
		GeeIterator* type_it;
		type_collection = vala_class_get_base_types (cl);
		type_it = gee_iterable_iterator (GEE_ITERABLE (type_collection));
		while (gee_iterator_next (type_it)) {
			ValaTypeReference* type;
			type = gee_iterator_get (type_it);
			{
				/* FIXME report error if multiple possible base methods are found*/
				if (VALA_IS_INTERFACE (vala_type_reference_get_data_type (type))) {
					ValaSymbol* sym;
					sym = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (vala_type_reference_get_data_type (type))), vala_symbol_get_name (VALA_SYMBOL (m)));
					if (VALA_IS_METHOD (sym)) {
						ValaMethod* _tmp0;
						ValaMethod* base_method;
						_tmp0 = NULL;
						base_method = (_tmp0 = VALA_METHOD (sym), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
						if (vala_method_get_is_abstract (base_method)) {
							if (!vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (cl)))) && !vala_method_equals (m, base_method)) {
								char* _tmp3;
								char* _tmp2;
								char* _tmp1;
								vala_code_node_set_error (VALA_CODE_NODE (m), TRUE);
								_tmp3 = NULL;
								_tmp2 = NULL;
								_tmp1 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), (_tmp3 = g_strdup_printf ("Return type and/or parameters of overriding method `%s' do not match overridden method `%s'.", (_tmp1 = vala_symbol_get_full_name (VALA_SYMBOL (m))), (_tmp2 = vala_symbol_get_full_name (VALA_SYMBOL (base_method))))));
								(_tmp3 = (g_free (_tmp3), NULL));
								(_tmp2 = (g_free (_tmp2), NULL));
								(_tmp1 = (g_free (_tmp1), NULL));
								(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
								(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
								(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
								(type_collection == NULL ? NULL : (type_collection = (g_object_unref (type_collection), NULL)));
								(type_it == NULL ? NULL : (type_it = (g_object_unref (type_it), NULL)));
								return;
							}
							vala_method_set_base_interface_method (m, base_method);
							(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
							(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
							(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
							(type_collection == NULL ? NULL : (type_collection = (g_object_unref (type_collection), NULL)));
							(type_it == NULL ? NULL : (type_it = (g_object_unref (type_it), NULL)));
							return;
						}
						(base_method == NULL ? NULL : (base_method = (g_object_unref (base_method), NULL)));
					}
					(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
				}
				(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			}
		}
		(type_collection == NULL ? NULL : (type_collection = (g_object_unref (type_collection), NULL)));
		(type_it == NULL ? NULL : (type_it = (g_object_unref (type_it), NULL)));
	}
}


static void vala_semantic_analyzer_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m) {
	ValaSemanticAnalyzer * self;
	ValaTypeReference* _tmp0;
	ValaSymbol* _tmp6;
	ValaTypeReference* _tmp8;
	ValaTypeReference* _tmp7;
	ValaSymbol* _tmp10;
	ValaSymbol* _tmp9;
	ValaTypeReference* _tmp11;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CREATION_METHOD (m));
	_tmp0 = NULL;
	vala_method_set_return_type (VALA_METHOD (m), (_tmp0 = vala_type_reference_new ()));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	if (vala_creation_method_get_type_name (m) != NULL && g_utf8_collate (vala_creation_method_get_type_name (m), vala_symbol_get_name (self->priv->current_symbol)) != 0) {
		char* _tmp2;
		char* _tmp1;
		/* type_name is null for constructors generated by GIdlParser*/
		_tmp2 = NULL;
		_tmp1 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), (_tmp2 = g_strdup_printf ("missing return type in method `%s.%s´", (_tmp1 = vala_symbol_get_full_name (self->priv->current_symbol)), vala_creation_method_get_type_name (m))));
		(_tmp2 = (g_free (_tmp2), NULL));
		(_tmp1 = (g_free (_tmp1), NULL));
		vala_code_node_set_error (VALA_CODE_NODE (m), TRUE);
		return;
	}
	if (VALA_IS_CLASS (self->priv->current_symbol)) {
		ValaClass* _tmp3;
		ValaClass* cl;
		vala_type_reference_set_data_type (vala_method_get_return_type (VALA_METHOD (m)), VALA_DATA_TYPE (vala_symbol_get_parent_symbol (VALA_SYMBOL (m))));
		vala_type_reference_set_transfers_ownership (vala_method_get_return_type (VALA_METHOD (m)), TRUE);
		/* check for floating reference*/
		_tmp3 = NULL;
		cl = (_tmp3 = VALA_CLASS (self->priv->current_symbol), (_tmp3 == NULL ? NULL : g_object_ref (_tmp3)));
		while (cl != NULL) {
			ValaClass* _tmp5;
			ValaClass* _tmp4;
			if (VALA_DATA_TYPE (cl) == self->priv->initially_unowned_type) {
				vala_type_reference_set_floating_reference (vala_method_get_return_type (VALA_METHOD (m)), TRUE);
				break;
			}
			_tmp5 = NULL;
			_tmp4 = NULL;
			cl = (_tmp5 = (_tmp4 = vala_class_get_base_class (cl), (_tmp4 == NULL ? NULL : g_object_ref (_tmp4))), (cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL))), _tmp5);
		}
		(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
	}
	_tmp6 = NULL;
	self->priv->current_symbol = (_tmp6 = VALA_SYMBOL (g_object_ref (m)), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp6);
	_tmp8 = NULL;
	_tmp7 = NULL;
	self->priv->current_return_type = (_tmp8 = (_tmp7 = vala_method_get_return_type (VALA_METHOD (m)), (_tmp7 == NULL ? NULL : g_object_ref (_tmp7))), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), _tmp8);
	vala_code_node_accept_children (VALA_CODE_NODE (m), VALA_CODE_VISITOR (self));
	_tmp10 = NULL;
	_tmp9 = NULL;
	self->priv->current_symbol = (_tmp10 = (_tmp9 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (_tmp9 == NULL ? NULL : g_object_ref (_tmp9))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp10);
	_tmp11 = NULL;
	self->priv->current_return_type = (_tmp11 = NULL, (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), _tmp11);
	if (VALA_IS_METHOD (vala_symbol_get_parent_symbol (self->priv->current_symbol))) {
		ValaMethod* _tmp12;
		ValaMethod* up_method;
		ValaTypeReference* _tmp14;
		ValaTypeReference* _tmp13;
		/* lambda expressions produce nested methods */
		_tmp12 = NULL;
		up_method = (_tmp12 = VALA_METHOD (vala_symbol_get_parent_symbol (self->priv->current_symbol)), (_tmp12 == NULL ? NULL : g_object_ref (_tmp12)));
		_tmp14 = NULL;
		_tmp13 = NULL;
		self->priv->current_return_type = (_tmp14 = (_tmp13 = vala_method_get_return_type (up_method), (_tmp13 == NULL ? NULL : g_object_ref (_tmp13))), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), _tmp14);
		(up_method == NULL ? NULL : (up_method = (g_object_unref (up_method), NULL)));
	}
	if (vala_method_get_is_abstract (VALA_METHOD (m)) || vala_method_get_is_virtual (VALA_METHOD (m)) || vala_method_get_overrides (VALA_METHOD (m))) {
		char* _tmp16;
		char* _tmp15;
		_tmp16 = NULL;
		_tmp15 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (m)), (_tmp16 = g_strdup_printf ("The creation method `%s' cannot be marked as override, virtual, or abstract", (_tmp15 = vala_symbol_get_full_name (VALA_SYMBOL (m))))));
		(_tmp16 = (g_free (_tmp16), NULL));
		(_tmp15 = (g_free (_tmp15), NULL));
		return;
	}
}


static void vala_semantic_analyzer_real_visit_formal_parameter (ValaCodeVisitor* base, ValaFormalParameter* p) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_FORMAL_PARAMETER (p));
	vala_code_node_accept_children (VALA_CODE_NODE (p), VALA_CODE_VISITOR (self));
	if (!vala_formal_parameter_get_ellipsis (p)) {
		if (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (p)) != NULL) {
			/* is null if it references a type parameter */
			if (!vala_symbol_is_internal_symbol (VALA_SYMBOL (p))) {
				vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (p))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW);
			}
			vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (p))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
		}
	}
	/* special treatment for construct formal parameters used in creation methods */
	if (vala_formal_parameter_get_construct_parameter (p)) {
		ValaBlock* _tmp0;
		ValaBlock* method_body;
		ValaMemberAccess* _tmp1;
		ValaMemberAccess* _tmp2;
		ValaMemberAccess* left;
		ValaMemberAccess* right;
		ValaExpressionStatement* _tmp4;
		ValaAssignment* _tmp3;
		if (!(VALA_IS_CREATION_METHOD (vala_symbol_get_parent_symbol (VALA_SYMBOL (p))))) {
			vala_code_node_set_error (VALA_CODE_NODE (p), TRUE);
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (p)), "construct parameters are only allowed in type creation methods");
			return;
		}
		_tmp0 = NULL;
		method_body = (_tmp0 = vala_method_get_body (VALA_METHOD ((VALA_CREATION_METHOD (vala_symbol_get_parent_symbol (VALA_SYMBOL (p)))))), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
		_tmp1 = NULL;
		_tmp2 = NULL;
		left = (_tmp2 = vala_member_access_new (VALA_EXPRESSION ((_tmp1 = vala_member_access_new_simple ("this", NULL))), vala_symbol_get_name (VALA_SYMBOL (p)), NULL), (_tmp1 == NULL ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL))), _tmp2);
		right = vala_member_access_new_simple (vala_symbol_get_name (VALA_SYMBOL (p)), NULL);
		_tmp4 = NULL;
		_tmp3 = NULL;
		vala_block_add_statement (method_body, VALA_STATEMENT ((_tmp4 = vala_expression_statement_new (VALA_EXPRESSION ((_tmp3 = vala_code_context_create_assignment (self->priv->context, VALA_EXPRESSION (left), VALA_EXPRESSION (right), VALA_ASSIGNMENT_OPERATOR_SIMPLE, NULL))), vala_code_node_get_source_reference (VALA_CODE_NODE (p))))));
		(_tmp4 == NULL ? NULL : (_tmp4 = (g_object_unref (_tmp4), NULL)));
		(_tmp3 = (g_object_unref (_tmp3), NULL));
		(method_body == NULL ? NULL : (method_body = (g_object_unref (method_body), NULL)));
		(left == NULL ? NULL : (left = (g_object_unref (left), NULL)));
		(right == NULL ? NULL : (right = (g_object_unref (right), NULL)));
	}
}


static void vala_semantic_analyzer_find_base_class_property (ValaSemanticAnalyzer* self, ValaProperty* prop, ValaClass* cl) {
	ValaSymbol* sym;
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (VALA_IS_PROPERTY (prop));
	g_return_if_fail (VALA_IS_CLASS (cl));
	sym = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (cl)), vala_symbol_get_name (VALA_SYMBOL (prop)));
	if (VALA_IS_PROPERTY (sym)) {
		ValaProperty* _tmp0;
		ValaProperty* base_property;
		_tmp0 = NULL;
		base_property = (_tmp0 = VALA_PROPERTY (sym), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
		if (vala_property_get_is_abstract (base_property) || vala_property_get_is_virtual (base_property)) {
			if (!vala_property_equals (prop, base_property)) {
				char* _tmp3;
				char* _tmp2;
				char* _tmp1;
				vala_code_node_set_error (VALA_CODE_NODE (prop), TRUE);
				_tmp3 = NULL;
				_tmp2 = NULL;
				_tmp1 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (prop)), (_tmp3 = g_strdup_printf ("Type and/or accessors of overriding property `%s' do not match overridden property `%s'.", (_tmp1 = vala_symbol_get_full_name (VALA_SYMBOL (prop))), (_tmp2 = vala_symbol_get_full_name (VALA_SYMBOL (base_property))))));
				(_tmp3 = (g_free (_tmp3), NULL));
				(_tmp2 = (g_free (_tmp2), NULL));
				(_tmp1 = (g_free (_tmp1), NULL));
				(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
				(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
				return;
			}
			vala_property_set_base_property (prop, base_property);
			(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
			(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
			return;
		}
		(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
	}
	if (vala_class_get_base_class (cl) != NULL) {
		vala_semantic_analyzer_find_base_class_property (self, prop, vala_class_get_base_class (cl));
	}
	(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
}


static void vala_semantic_analyzer_find_base_interface_property (ValaSemanticAnalyzer* self, ValaProperty* prop, ValaClass* cl) {
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	g_return_if_fail (VALA_IS_PROPERTY (prop));
	g_return_if_fail (VALA_IS_CLASS (cl));
	{
		GeeCollection* type_collection;
		GeeIterator* type_it;
		type_collection = vala_class_get_base_types (cl);
		type_it = gee_iterable_iterator (GEE_ITERABLE (type_collection));
		while (gee_iterator_next (type_it)) {
			ValaTypeReference* type;
			type = gee_iterator_get (type_it);
			{
				/* FIXME report error if multiple possible base properties are found*/
				if (VALA_IS_INTERFACE (vala_type_reference_get_data_type (type))) {
					ValaSymbol* sym;
					sym = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (vala_type_reference_get_data_type (type))), vala_symbol_get_name (VALA_SYMBOL (prop)));
					if (VALA_IS_PROPERTY (sym)) {
						ValaProperty* _tmp0;
						ValaProperty* base_property;
						_tmp0 = NULL;
						base_property = (_tmp0 = VALA_PROPERTY (sym), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
						if (vala_property_get_is_abstract (base_property)) {
							if (!vala_property_equals (prop, base_property)) {
								char* _tmp3;
								char* _tmp2;
								char* _tmp1;
								vala_code_node_set_error (VALA_CODE_NODE (prop), TRUE);
								_tmp3 = NULL;
								_tmp2 = NULL;
								_tmp1 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (prop)), (_tmp3 = g_strdup_printf ("Type and/or accessors of overriding property `%s' do not match overridden property `%s'.", (_tmp1 = vala_symbol_get_full_name (VALA_SYMBOL (prop))), (_tmp2 = vala_symbol_get_full_name (VALA_SYMBOL (base_property))))));
								(_tmp3 = (g_free (_tmp3), NULL));
								(_tmp2 = (g_free (_tmp2), NULL));
								(_tmp1 = (g_free (_tmp1), NULL));
								(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
								(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
								(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
								(type_collection == NULL ? NULL : (type_collection = (g_object_unref (type_collection), NULL)));
								(type_it == NULL ? NULL : (type_it = (g_object_unref (type_it), NULL)));
								return;
							}
							vala_property_set_base_interface_property (prop, base_property);
							(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
							(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
							(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
							(type_collection == NULL ? NULL : (type_collection = (g_object_unref (type_collection), NULL)));
							(type_it == NULL ? NULL : (type_it = (g_object_unref (type_it), NULL)));
							return;
						}
						(base_property == NULL ? NULL : (base_property = (g_object_unref (base_property), NULL)));
					}
					(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
				}
				(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			}
		}
		(type_collection == NULL ? NULL : (type_collection = (g_object_unref (type_collection), NULL)));
		(type_it == NULL ? NULL : (type_it = (g_object_unref (type_it), NULL)));
	}
}


static void vala_semantic_analyzer_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop) {
	ValaSemanticAnalyzer * self;
	ValaSymbol* _tmp0;
	ValaSymbol* _tmp5;
	ValaSymbol* _tmp4;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_PROPERTY (prop));
	_tmp0 = NULL;
	self->priv->current_symbol = (_tmp0 = VALA_SYMBOL (g_object_ref (prop)), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp0);
	vala_code_node_accept_children (VALA_CODE_NODE (prop), VALA_CODE_VISITOR (self));
	/* abstract/virtual properties using reference types without
	 * reference counting need to transfer ownership of their
	 * return values because of limitations in the GObject property
	 * system (g_object_get always returns strong references).
	 * Reference counting types can simulate to return a weak
	 * reference */
	if ((vala_property_get_is_abstract (prop) || vala_property_get_is_virtual (prop)) && vala_data_type_is_reference_type (vala_type_reference_get_data_type (vala_property_get_type_reference (prop))) && !vala_data_type_is_reference_counting (vala_type_reference_get_data_type (vala_property_get_type_reference (prop))) && !vala_type_reference_get_transfers_ownership (vala_property_get_type_reference (prop))) {
		char* _tmp3;
		char* _tmp2;
		char* _tmp1;
		_tmp3 = NULL;
		_tmp2 = NULL;
		_tmp1 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (prop)), (_tmp3 = g_strdup_printf ("%s: abstract or virtual properties using reference types not supporting reference counting, like `%s', have to mark their return value to transfer ownership.", (_tmp1 = vala_symbol_get_full_name (VALA_SYMBOL (prop))), (_tmp2 = vala_symbol_get_full_name (VALA_SYMBOL (vala_type_reference_get_data_type (vala_property_get_type_reference (prop))))))));
		(_tmp3 = (g_free (_tmp3), NULL));
		(_tmp2 = (g_free (_tmp2), NULL));
		(_tmp1 = (g_free (_tmp1), NULL));
		vala_code_node_set_error (VALA_CODE_NODE (prop), TRUE);
	}
	_tmp5 = NULL;
	_tmp4 = NULL;
	self->priv->current_symbol = (_tmp5 = (_tmp4 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (_tmp4 == NULL ? NULL : g_object_ref (_tmp4))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp5);
	if (vala_type_reference_get_data_type (vala_property_get_type_reference (prop)) != NULL) {
		/* is null if it references a type parameter */
		if (!vala_symbol_is_internal_symbol (VALA_SYMBOL (prop))) {
			vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (vala_property_get_type_reference (prop))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW);
		}
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (vala_property_get_type_reference (prop))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	}
	if (VALA_IS_CLASS (vala_symbol_get_parent_symbol (VALA_SYMBOL (prop)))) {
		ValaClass* _tmp6;
		ValaClass* cl;
		_tmp6 = NULL;
		cl = (_tmp6 = VALA_CLASS (vala_symbol_get_parent_symbol (VALA_SYMBOL (prop))), (_tmp6 == NULL ? NULL : g_object_ref (_tmp6)));
		vala_semantic_analyzer_find_base_interface_property (self, prop, cl);
		if (vala_property_get_is_virtual (prop) || vala_property_get_overrides (prop)) {
			vala_semantic_analyzer_find_base_class_property (self, prop, cl);
			if (vala_property_get_base_property (prop) == NULL) {
				char* _tmp8;
				char* _tmp7;
				vala_code_node_set_error (VALA_CODE_NODE (prop), TRUE);
				_tmp8 = NULL;
				_tmp7 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (prop)), (_tmp8 = g_strdup_printf ("%s: no suitable property found to override", (_tmp7 = vala_symbol_get_full_name (VALA_SYMBOL (prop))))));
				(_tmp8 = (g_free (_tmp8), NULL));
				(_tmp7 = (g_free (_tmp7), NULL));
			}
		}
		(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
	}
}


static void vala_semantic_analyzer_real_visit_property_accessor (ValaCodeVisitor* base, ValaPropertyAccessor* acc) {
	ValaSemanticAnalyzer * self;
	ValaTypeReference* _tmp12;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_PROPERTY_ACCESSOR (acc));
	vala_property_accessor_set_prop (acc, VALA_PROPERTY (self->priv->current_symbol));
	if (vala_property_accessor_get_readable (acc)) {
		ValaTypeReference* _tmp0;
		_tmp0 = NULL;
		self->priv->current_return_type = (_tmp0 = g_object_ref (vala_property_get_type_reference (vala_property_accessor_get_prop (acc))), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), _tmp0);
	} else {
		ValaTypeReference* _tmp1;
		/* void*/
		_tmp1 = NULL;
		self->priv->current_return_type = (_tmp1 = vala_type_reference_new (), (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), _tmp1);
	}
	if (!vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (acc))))) {
		if (vala_property_accessor_get_body (acc) == NULL && !vala_property_get_interface_only (vala_property_accessor_get_prop (acc)) && !vala_property_get_is_abstract (vala_property_accessor_get_prop (acc))) {
			ValaBlock* _tmp2;
			/* no accessor body specified, insert default body */
			_tmp2 = NULL;
			vala_property_accessor_set_body (acc, (_tmp2 = vala_block_new (NULL)));
			(_tmp2 == NULL ? NULL : (_tmp2 = (g_object_unref (_tmp2), NULL)));
			if (vala_property_accessor_get_readable (acc)) {
				ValaReturnStatement* _tmp5;
				ValaMemberAccess* _tmp4;
				char* _tmp3;
				_tmp5 = NULL;
				_tmp4 = NULL;
				_tmp3 = NULL;
				vala_block_add_statement (vala_property_accessor_get_body (acc), VALA_STATEMENT ((_tmp5 = vala_return_statement_new (VALA_EXPRESSION ((_tmp4 = vala_member_access_new_simple ((_tmp3 = g_strdup_printf ("_%s", vala_symbol_get_name (VALA_SYMBOL (vala_property_accessor_get_prop (acc))))), NULL))), vala_code_node_get_source_reference (VALA_CODE_NODE (acc))))));
				(_tmp5 == NULL ? NULL : (_tmp5 = (g_object_unref (_tmp5), NULL)));
				(_tmp4 == NULL ? NULL : (_tmp4 = (g_object_unref (_tmp4), NULL)));
				(_tmp3 = (g_free (_tmp3), NULL));
			} else {
				ValaExpressionStatement* _tmp10;
				ValaAssignment* _tmp9;
				ValaMemberAccess* _tmp8;
				ValaMemberAccess* _tmp7;
				char* _tmp6;
				_tmp10 = NULL;
				_tmp9 = NULL;
				_tmp8 = NULL;
				_tmp7 = NULL;
				_tmp6 = NULL;
				vala_block_add_statement (vala_property_accessor_get_body (acc), VALA_STATEMENT ((_tmp10 = vala_expression_statement_new (VALA_EXPRESSION ((_tmp9 = vala_code_context_create_assignment (self->priv->context, VALA_EXPRESSION ((_tmp7 = vala_member_access_new_simple ((_tmp6 = g_strdup_printf ("_%s", vala_symbol_get_name (VALA_SYMBOL (vala_property_accessor_get_prop (acc))))), NULL))), VALA_EXPRESSION ((_tmp8 = vala_member_access_new_simple ("value", NULL))), VALA_ASSIGNMENT_OPERATOR_SIMPLE, NULL))), vala_code_node_get_source_reference (VALA_CODE_NODE (acc))))));
				(_tmp10 == NULL ? NULL : (_tmp10 = (g_object_unref (_tmp10), NULL)));
				(_tmp9 = (g_object_unref (_tmp9), NULL));
				(_tmp8 == NULL ? NULL : (_tmp8 = (g_object_unref (_tmp8), NULL)));
				(_tmp7 == NULL ? NULL : (_tmp7 = (g_object_unref (_tmp7), NULL)));
				(_tmp6 = (g_free (_tmp6), NULL));
			}
		}
		if (vala_property_accessor_get_body (acc) != NULL && (vala_property_accessor_get_writable (acc) || vala_property_accessor_get_construction (acc))) {
			ValaFormalParameter* _tmp11;
			_tmp11 = NULL;
			vala_property_accessor_set_value_parameter (acc, (_tmp11 = vala_formal_parameter_new ("value", vala_property_get_type_reference (vala_property_accessor_get_prop (acc)), vala_code_node_get_source_reference (VALA_CODE_NODE (acc)))));
			(_tmp11 == NULL ? NULL : (_tmp11 = (g_object_unref (_tmp11), NULL)));
			vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (vala_property_accessor_get_body (acc))), vala_symbol_get_name (VALA_SYMBOL (vala_property_accessor_get_value_parameter (acc))), VALA_SYMBOL (vala_property_accessor_get_value_parameter (acc)));
		}
	}
	vala_code_node_accept_children (VALA_CODE_NODE (acc), VALA_CODE_VISITOR (self));
	_tmp12 = NULL;
	self->priv->current_return_type = (_tmp12 = NULL, (self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL))), _tmp12);
}


static void vala_semantic_analyzer_real_visit_signal (ValaCodeVisitor* base, ValaSignal* sig) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_SIGNAL (sig));
	vala_code_node_accept_children (VALA_CODE_NODE (sig), VALA_CODE_VISITOR (self));
}


static void vala_semantic_analyzer_real_visit_constructor (ValaCodeVisitor* base, ValaConstructor* c) {
	ValaSemanticAnalyzer * self;
	ValaFormalParameter* _tmp1;
	ValaTypeReference* _tmp0;
	ValaSymbol* _tmp2;
	ValaSymbol* _tmp4;
	ValaSymbol* _tmp3;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CONSTRUCTOR (c));
	_tmp1 = NULL;
	_tmp0 = NULL;
	vala_constructor_set_this_parameter (c, (_tmp1 = vala_formal_parameter_new ("this", (_tmp0 = vala_type_reference_new ()), NULL)));
	(_tmp1 == NULL ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	vala_type_reference_set_data_type (vala_formal_parameter_get_type_reference (vala_constructor_get_this_parameter (c)), VALA_DATA_TYPE (self->priv->current_symbol));
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (c)), vala_symbol_get_name (VALA_SYMBOL (vala_constructor_get_this_parameter (c))), VALA_SYMBOL (vala_constructor_get_this_parameter (c)));
	vala_symbol_set_owner (VALA_SYMBOL (c), vala_symbol_get_scope (self->priv->current_symbol));
	_tmp2 = NULL;
	self->priv->current_symbol = (_tmp2 = VALA_SYMBOL (g_object_ref (c)), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp2);
	vala_code_node_accept_children (VALA_CODE_NODE (c), VALA_CODE_VISITOR (self));
	_tmp4 = NULL;
	_tmp3 = NULL;
	self->priv->current_symbol = (_tmp4 = (_tmp3 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (_tmp3 == NULL ? NULL : g_object_ref (_tmp3))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp4);
}


static void vala_semantic_analyzer_real_visit_destructor (ValaCodeVisitor* base, ValaDestructor* d) {
	ValaSemanticAnalyzer * self;
	ValaSymbol* _tmp0;
	ValaSymbol* _tmp2;
	ValaSymbol* _tmp1;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_DESTRUCTOR (d));
	vala_symbol_set_owner (VALA_SYMBOL (d), vala_symbol_get_scope (self->priv->current_symbol));
	_tmp0 = NULL;
	self->priv->current_symbol = (_tmp0 = VALA_SYMBOL (g_object_ref (d)), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp0);
	vala_code_node_accept_children (VALA_CODE_NODE (d), VALA_CODE_VISITOR (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->current_symbol = (_tmp2 = (_tmp1 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp2);
}


static void vala_semantic_analyzer_real_visit_named_argument (ValaCodeVisitor* base, ValaNamedArgument* n) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_NAMED_ARGUMENT (n));
}


static void vala_semantic_analyzer_real_visit_block (ValaCodeVisitor* base, ValaBlock* b) {
	ValaSemanticAnalyzer * self;
	ValaSymbol* _tmp0;
	ValaSymbol* _tmp2;
	ValaSymbol* _tmp1;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_BLOCK (b));
	vala_symbol_set_owner (VALA_SYMBOL (b), vala_symbol_get_scope (self->priv->current_symbol));
	_tmp0 = NULL;
	self->priv->current_symbol = (_tmp0 = VALA_SYMBOL (g_object_ref (b)), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp0);
	vala_code_node_accept_children (VALA_CODE_NODE (b), VALA_CODE_VISITOR (self));
	{
		GeeCollection* decl_collection;
		GeeIterator* decl_it;
		decl_collection = vala_block_get_local_variables (b);
		decl_it = gee_iterable_iterator (GEE_ITERABLE (decl_collection));
		while (gee_iterator_next (decl_it)) {
			ValaVariableDeclarator* decl;
			decl = gee_iterator_get (decl_it);
			{
				vala_symbol_set_active (VALA_SYMBOL (decl), FALSE);
				(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
			}
		}
		(decl_collection == NULL ? NULL : (decl_collection = (g_object_unref (decl_collection), NULL)));
		(decl_it == NULL ? NULL : (decl_it = (g_object_unref (decl_it), NULL)));
	}
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->current_symbol = (_tmp2 = (_tmp1 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp2);
}


static void vala_semantic_analyzer_real_visit_variable_declarator (ValaCodeVisitor* base, ValaVariableDeclarator* decl) {
	ValaSemanticAnalyzer * self;
	ValaBlock* _tmp6;
	ValaBlock* block;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_VARIABLE_DECLARATOR (decl));
	if (vala_variable_declarator_get_initializer (decl) != NULL) {
		vala_expression_set_expected_type (vala_variable_declarator_get_initializer (decl), vala_variable_declarator_get_type_reference (decl));
	}
	vala_code_node_accept_children (VALA_CODE_NODE (decl), VALA_CODE_VISITOR (self));
	if (vala_variable_declarator_get_type_reference (decl) == NULL) {
		ValaTypeReference* _tmp0;
		/* var type */
		if (vala_variable_declarator_get_initializer (decl) == NULL) {
			vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE);
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (decl)), "var declaration not allowed without initializer");
			return;
		}
		if (vala_expression_get_static_type (vala_variable_declarator_get_initializer (decl)) == NULL) {
			vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE);
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (decl)), "var declaration not allowed with non-typed initializer");
			return;
		}
		_tmp0 = NULL;
		vala_variable_declarator_set_type_reference (decl, (_tmp0 = vala_type_reference_copy (vala_expression_get_static_type (vala_variable_declarator_get_initializer (decl)))));
		(_tmp0 = (g_object_unref (_tmp0), NULL));
		vala_type_reference_set_takes_ownership (vala_variable_declarator_get_type_reference (decl), (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl)) == NULL || vala_data_type_is_reference_type (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl)))));
		vala_type_reference_set_transfers_ownership (vala_variable_declarator_get_type_reference (decl), FALSE);
	}
	if (vala_variable_declarator_get_initializer (decl) != NULL) {
		if (vala_expression_get_static_type (vala_variable_declarator_get_initializer (decl)) == NULL) {
			if (!(VALA_IS_MEMBER_ACCESS (vala_variable_declarator_get_initializer (decl)))) {
				vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE);
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (decl)), "expression type not allowed as initializer");
				return;
			}
			if (VALA_IS_METHOD (vala_expression_get_symbol_reference (vala_variable_declarator_get_initializer (decl))) && VALA_IS_CALLBACK (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl)))) {
				ValaMethod* _tmp1;
				ValaMethod* m;
				ValaCallback* _tmp2;
				ValaCallback* cb;
				_tmp1 = NULL;
				m = (_tmp1 = VALA_METHOD (vala_expression_get_symbol_reference (vala_variable_declarator_get_initializer (decl))), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1)));
				_tmp2 = NULL;
				cb = (_tmp2 = VALA_CALLBACK (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl))), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2)));
				/* check whether method matches callback type */
				if (!vala_callback_matches_method (cb, m)) {
					char* _tmp5;
					char* _tmp4;
					char* _tmp3;
					vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE);
					_tmp5 = NULL;
					_tmp4 = NULL;
					_tmp3 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (decl)), (_tmp5 = g_strdup_printf ("declaration of method `%s' doesn't match declaration of callback `%s'", (_tmp3 = vala_symbol_get_full_name (VALA_SYMBOL (m))), (_tmp4 = vala_symbol_get_full_name (VALA_SYMBOL (cb))))));
					(_tmp5 = (g_free (_tmp5), NULL));
					(_tmp4 = (g_free (_tmp4), NULL));
					(_tmp3 = (g_free (_tmp3), NULL));
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
					(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
					return;
				}
				vala_expression_set_static_type (vala_variable_declarator_get_initializer (decl), vala_variable_declarator_get_type_reference (decl));
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
			} else {
				vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE);
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (decl)), "expression type not allowed as initializer");
				return;
			}
		}
		if (vala_semantic_analyzer_get_memory_management (self)) {
			if (vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (vala_variable_declarator_get_initializer (decl)))) {
				/* rhs transfers ownership of the expression */
				if (!vala_type_reference_get_takes_ownership (vala_variable_declarator_get_type_reference (decl))) {
					/* lhs doesn't own the value */
					vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE);
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (decl)), "Invalid assignment from owned expression to unowned variable");
					return;
				}
			}
		}
	}
	if (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl)) != NULL) {
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	}
	vala_scope_add (vala_symbol_get_scope (self->priv->current_symbol), vala_symbol_get_name (VALA_SYMBOL (decl)), VALA_SYMBOL (decl));
	_tmp6 = NULL;
	block = (_tmp6 = VALA_BLOCK (self->priv->current_symbol), (_tmp6 == NULL ? NULL : g_object_ref (_tmp6)));
	vala_block_add_local_variable (block, decl);
	vala_symbol_set_active (VALA_SYMBOL (decl), TRUE);
	(block == NULL ? NULL : (block = (g_object_unref (block), NULL)));
}


/**
 * Visit operation called for initializer lists
 *
 * @param list an initializer list
 */
static void vala_semantic_analyzer_real_visit_initializer_list (ValaCodeVisitor* base, ValaInitializerList* list) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_INITIALIZER_LIST (list));
	if (vala_expression_get_expected_type (VALA_EXPRESSION (list)) != NULL && VALA_IS_ARRAY (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (list))))) {
		ValaArray* _tmp0;
		ValaArray* edt;
		GeeCollection* inits;
		gint rank;
		ValaTypeReference* child_type;
		/* initializer is used as array initializer */
		_tmp0 = NULL;
		edt = (_tmp0 = VALA_ARRAY (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (list)))), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
		inits = vala_initializer_list_get_initializers (list);
		rank = vala_array_get_rank ((VALA_ARRAY (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (list))))));
		child_type = vala_type_reference_copy (vala_expression_get_expected_type (VALA_EXPRESSION (list)));
		if (rank > 1) {
			ValaArray* _tmp1;
			_tmp1 = NULL;
			vala_type_reference_set_data_type (child_type, VALA_DATA_TYPE ((_tmp1 = vala_data_type_get_array (vala_array_get_element_type (edt), rank - 1))));
			(_tmp1 = (g_object_unref (_tmp1), NULL));
		} else {
			vala_type_reference_set_data_type (child_type, vala_array_get_element_type (edt));
		}
		{
			GeeCollection* e_collection;
			GeeIterator* e_it;
			e_collection = inits;
			e_it = gee_iterable_iterator (GEE_ITERABLE (e_collection));
			while (gee_iterator_next (e_it)) {
				ValaExpression* e;
				e = gee_iterator_get (e_it);
				{
					ValaTypeReference* _tmp2;
					_tmp2 = NULL;
					vala_expression_set_expected_type (e, (_tmp2 = vala_type_reference_copy (child_type)));
					(_tmp2 = (g_object_unref (_tmp2), NULL));
					(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
				}
			}
			(e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL)));
		}
		(edt == NULL ? NULL : (edt = (g_object_unref (edt), NULL)));
		(inits == NULL ? NULL : (inits = (g_object_unref (inits), NULL)));
		(child_type = (g_object_unref (child_type), NULL));
	}
	vala_code_node_accept_children (VALA_CODE_NODE (list), VALA_CODE_VISITOR (self));
	if (vala_expression_get_expected_type (VALA_EXPRESSION (list)) != NULL && VALA_IS_ARRAY (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (list))))) {
		ValaArray* _tmp3;
		ValaArray* edt;
		GeeCollection* inits;
		gint rank;
		ValaTypeReference* child_type;
		gboolean error;
		_tmp3 = NULL;
		edt = (_tmp3 = VALA_ARRAY (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (list)))), (_tmp3 == NULL ? NULL : g_object_ref (_tmp3)));
		inits = vala_initializer_list_get_initializers (list);
		rank = vala_array_get_rank (edt);
		child_type = vala_type_reference_copy (vala_expression_get_expected_type (VALA_EXPRESSION (list)));
		error = FALSE;
		if (rank > 1) {
			ValaArray* _tmp4;
			_tmp4 = NULL;
			vala_type_reference_set_data_type (child_type, VALA_DATA_TYPE ((_tmp4 = vala_data_type_get_array (vala_array_get_element_type (edt), rank - 1))));
			(_tmp4 = (g_object_unref (_tmp4), NULL));
			{
				GeeCollection* e_collection;
				GeeIterator* e_it;
				e_collection = inits;
				e_it = gee_iterable_iterator (GEE_ITERABLE (e_collection));
				while (gee_iterator_next (e_it)) {
					ValaExpression* e;
					e = gee_iterator_get (e_it);
					{
						if (vala_expression_get_static_type (e) == NULL) {
							error = TRUE;
							(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
							continue;
						}
						if (!(VALA_IS_INITIALIZER_LIST (e))) {
							error = TRUE;
							vala_code_node_set_error (VALA_CODE_NODE (e), TRUE);
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (e)), "Initializer list expected");
							(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
							continue;
						}
						if (!vala_type_reference_equals (vala_expression_get_static_type (e), child_type)) {
							char* _tmp5;
							error = TRUE;
							vala_code_node_set_error (VALA_CODE_NODE (e), TRUE);
							_tmp5 = NULL;
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (e)), (_tmp5 = g_strdup_printf ("Expected initializer list of type `%s' but got `%s'", vala_symbol_get_name (VALA_SYMBOL (vala_type_reference_get_data_type (child_type))), vala_symbol_get_name (VALA_SYMBOL (vala_type_reference_get_data_type (vala_expression_get_static_type (e)))))));
							(_tmp5 = (g_free (_tmp5), NULL));
						}
						(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
					}
				}
				(e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL)));
			}
		} else {
			vala_type_reference_set_data_type (child_type, vala_array_get_element_type (edt));
			{
				GeeCollection* e_collection;
				GeeIterator* e_it;
				e_collection = inits;
				e_it = gee_iterable_iterator (GEE_ITERABLE (e_collection));
				while (gee_iterator_next (e_it)) {
					ValaExpression* e;
					e = gee_iterator_get (e_it);
					{
						if (vala_expression_get_static_type (e) == NULL) {
							error = TRUE;
							(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
							continue;
						}
						if (!vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (e), child_type)) {
							char* _tmp6;
							error = TRUE;
							vala_code_node_set_error (VALA_CODE_NODE (e), TRUE);
							_tmp6 = NULL;
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (e)), (_tmp6 = g_strdup_printf ("Expected initializer of type `%s' but got `%s'", vala_symbol_get_name (VALA_SYMBOL (vala_type_reference_get_data_type (child_type))), vala_symbol_get_name (VALA_SYMBOL (vala_type_reference_get_data_type (vala_expression_get_static_type (e)))))));
							(_tmp6 = (g_free (_tmp6), NULL));
						}
						(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
					}
				}
				(e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL)));
			}
		}
		if (!error) {
			/* everything seems to be correct */
			vala_expression_set_static_type (VALA_EXPRESSION (list), vala_expression_get_expected_type (VALA_EXPRESSION (list)));
		}
		(edt == NULL ? NULL : (edt = (g_object_unref (edt), NULL)));
		(inits == NULL ? NULL : (inits = (g_object_unref (inits), NULL)));
		(child_type = (g_object_unref (child_type), NULL));
	}
}


static void vala_semantic_analyzer_real_visit_expression_statement (ValaCodeVisitor* base, ValaExpressionStatement* stmt) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_EXPRESSION_STATEMENT (stmt));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_expression_statement_get_expression (stmt)))) {
		/* ignore inner error*/
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		return;
	}
	vala_code_node_set_tree_can_fail (VALA_CODE_NODE (stmt), vala_code_node_get_tree_can_fail (VALA_CODE_NODE (vala_expression_statement_get_expression (stmt))));
}


static void vala_semantic_analyzer_real_visit_if_statement (ValaCodeVisitor* base, ValaIfStatement* stmt) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_IF_STATEMENT (stmt));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_if_statement_get_condition (stmt)))) {
		/* if there was an error in the condition, skip this check */
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		return;
	}
	if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_if_statement_get_condition (stmt))) != vala_type_reference_get_data_type (self->priv->bool_type)) {
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_if_statement_get_condition (stmt))), "Condition must be boolean");
		return;
	}
}


static void vala_semantic_analyzer_real_visit_switch_section (ValaCodeVisitor* base, ValaSwitchSection* section) {
	ValaSemanticAnalyzer * self;
	ValaSymbol* _tmp0;
	ValaSymbol* _tmp2;
	ValaSymbol* _tmp1;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_SWITCH_SECTION (section));
	{
		GeeCollection* label_collection;
		GeeIterator* label_it;
		label_collection = vala_switch_section_get_labels (section);
		label_it = gee_iterable_iterator (GEE_ITERABLE (label_collection));
		while (gee_iterator_next (label_it)) {
			ValaSwitchLabel* label;
			label = gee_iterator_get (label_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (label), VALA_CODE_VISITOR (self));
				(label == NULL ? NULL : (label = (g_object_unref (label), NULL)));
			}
		}
		(label_collection == NULL ? NULL : (label_collection = (g_object_unref (label_collection), NULL)));
		(label_it == NULL ? NULL : (label_it = (g_object_unref (label_it), NULL)));
	}
	vala_symbol_set_owner (VALA_SYMBOL (section), vala_symbol_get_scope (self->priv->current_symbol));
	_tmp0 = NULL;
	self->priv->current_symbol = (_tmp0 = VALA_SYMBOL (g_object_ref (section)), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp0);
	{
		GeeCollection* st_collection;
		GeeIterator* st_it;
		st_collection = vala_switch_section_get_statements (section);
		st_it = gee_iterable_iterator (GEE_ITERABLE (st_collection));
		while (gee_iterator_next (st_it)) {
			ValaStatement* st;
			st = gee_iterator_get (st_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (st), VALA_CODE_VISITOR (self));
				(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
			}
		}
		(st_collection == NULL ? NULL : (st_collection = (g_object_unref (st_collection), NULL)));
		(st_it == NULL ? NULL : (st_it = (g_object_unref (st_it), NULL)));
	}
	{
		GeeCollection* decl_collection;
		GeeIterator* decl_it;
		decl_collection = vala_block_get_local_variables (VALA_BLOCK (section));
		decl_it = gee_iterable_iterator (GEE_ITERABLE (decl_collection));
		while (gee_iterator_next (decl_it)) {
			ValaVariableDeclarator* decl;
			decl = gee_iterator_get (decl_it);
			{
				vala_symbol_set_active (VALA_SYMBOL (decl), FALSE);
				(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
			}
		}
		(decl_collection == NULL ? NULL : (decl_collection = (g_object_unref (decl_collection), NULL)));
		(decl_it == NULL ? NULL : (decl_it = (g_object_unref (decl_it), NULL)));
	}
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->current_symbol = (_tmp2 = (_tmp1 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp2);
}


static void vala_semantic_analyzer_real_visit_while_statement (ValaCodeVisitor* base, ValaWhileStatement* stmt) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_WHILE_STATEMENT (stmt));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_while_statement_get_condition (stmt)))) {
		/* if there was an error in the condition, skip this check */
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		return;
	}
	if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_while_statement_get_condition (stmt))) != vala_type_reference_get_data_type (self->priv->bool_type)) {
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_while_statement_get_condition (stmt))), "Condition must be boolean");
		return;
	}
}


static void vala_semantic_analyzer_real_visit_for_statement (ValaCodeVisitor* base, ValaForStatement* stmt) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_FOR_STATEMENT (stmt));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_for_statement_get_condition (stmt)))) {
		/* if there was an error in the condition, skip this check */
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		return;
	}
	if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_for_statement_get_condition (stmt))) != vala_type_reference_get_data_type (self->priv->bool_type)) {
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_for_statement_get_condition (stmt))), "Condition must be boolean");
		return;
	}
}


static void vala_semantic_analyzer_real_visit_foreach_statement (ValaCodeVisitor* base, ValaForeachStatement* stmt) {
	ValaSemanticAnalyzer * self;
	ValaVariableDeclarator* _tmp0;
	ValaSymbol* _tmp1;
	ValaSymbol* _tmp3;
	ValaSymbol* _tmp2;
	ValaVariableDeclarator* _tmp5;
	char* _tmp4;
	ValaTypeReference* _tmp6;
	ValaDataType* _tmp7;
	ValaDataType* collection_type;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_FOREACH_STATEMENT (stmt));
	if (vala_type_reference_get_data_type (vala_foreach_statement_get_type_reference (stmt)) != NULL) {
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (vala_foreach_statement_get_type_reference (stmt))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	}
	_tmp0 = NULL;
	vala_foreach_statement_set_variable_declarator (stmt, (_tmp0 = vala_variable_declarator_new (vala_foreach_statement_get_variable_name (stmt), NULL, NULL)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	vala_variable_declarator_set_type_reference (vala_foreach_statement_get_variable_declarator (stmt), vala_foreach_statement_get_type_reference (stmt));
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (vala_foreach_statement_get_body (stmt))), vala_foreach_statement_get_variable_name (stmt), VALA_SYMBOL (vala_foreach_statement_get_variable_declarator (stmt)));
	vala_block_add_local_variable (vala_foreach_statement_get_body (stmt), vala_foreach_statement_get_variable_declarator (stmt));
	vala_symbol_set_active (VALA_SYMBOL (vala_foreach_statement_get_variable_declarator (stmt)), TRUE);
	vala_symbol_set_owner (VALA_SYMBOL (stmt), vala_symbol_get_scope (self->priv->current_symbol));
	_tmp1 = NULL;
	self->priv->current_symbol = (_tmp1 = VALA_SYMBOL (g_object_ref (stmt)), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp1);
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
	{
		GeeCollection* decl_collection;
		GeeIterator* decl_it;
		decl_collection = vala_block_get_local_variables (VALA_BLOCK (stmt));
		decl_it = gee_iterable_iterator (GEE_ITERABLE (decl_collection));
		while (gee_iterator_next (decl_it)) {
			ValaVariableDeclarator* decl;
			decl = gee_iterator_get (decl_it);
			{
				vala_symbol_set_active (VALA_SYMBOL (decl), FALSE);
				(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
			}
		}
		(decl_collection == NULL ? NULL : (decl_collection = (g_object_unref (decl_collection), NULL)));
		(decl_it == NULL ? NULL : (decl_it = (g_object_unref (decl_it), NULL)));
	}
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->current_symbol = (_tmp3 = (_tmp2 = vala_symbol_get_parent_symbol (self->priv->current_symbol), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), _tmp3);
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_foreach_statement_get_collection (stmt)))) {
		/* ignore inner error*/
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		return;
	}
	_tmp5 = NULL;
	_tmp4 = NULL;
	vala_foreach_statement_set_collection_variable_declarator (stmt, (_tmp5 = vala_variable_declarator_new ((_tmp4 = g_strdup_printf ("%s_collection", vala_foreach_statement_get_variable_name (stmt))), NULL, NULL)));
	(_tmp5 == NULL ? NULL : (_tmp5 = (g_object_unref (_tmp5), NULL)));
	(_tmp4 = (g_free (_tmp4), NULL));
	_tmp6 = NULL;
	vala_variable_declarator_set_type_reference (vala_foreach_statement_get_collection_variable_declarator (stmt), (_tmp6 = vala_type_reference_copy (vala_expression_get_static_type (vala_foreach_statement_get_collection (stmt)))));
	(_tmp6 = (g_object_unref (_tmp6), NULL));
	vala_type_reference_set_transfers_ownership (vala_variable_declarator_get_type_reference (vala_foreach_statement_get_collection_variable_declarator (stmt)), FALSE);
	vala_type_reference_set_takes_ownership (vala_variable_declarator_get_type_reference (vala_foreach_statement_get_collection_variable_declarator (stmt)), vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (vala_foreach_statement_get_collection (stmt))));
	vala_block_add_local_variable (VALA_BLOCK (stmt), vala_foreach_statement_get_collection_variable_declarator (stmt));
	vala_symbol_set_active (VALA_SYMBOL (vala_foreach_statement_get_collection_variable_declarator (stmt)), TRUE);
	_tmp7 = NULL;
	collection_type = (_tmp7 = vala_type_reference_get_data_type (vala_expression_get_static_type (vala_foreach_statement_get_collection (stmt))), (_tmp7 == NULL ? NULL : g_object_ref (_tmp7)));
	if (self->priv->iterable_type != NULL && vala_data_type_is_subtype_of (collection_type, self->priv->iterable_type)) {
		ValaVariableDeclarator* _tmp9;
		char* _tmp8;
		ValaTypeReference* _tmp10;
		_tmp9 = NULL;
		_tmp8 = NULL;
		vala_foreach_statement_set_iterator_variable_declarator (stmt, (_tmp9 = vala_variable_declarator_new ((_tmp8 = g_strdup_printf ("%s_it", vala_foreach_statement_get_variable_name (stmt))), NULL, NULL)));
		(_tmp9 == NULL ? NULL : (_tmp9 = (g_object_unref (_tmp9), NULL)));
		(_tmp8 = (g_free (_tmp8), NULL));
		_tmp10 = NULL;
		vala_variable_declarator_set_type_reference (vala_foreach_statement_get_iterator_variable_declarator (stmt), (_tmp10 = vala_type_reference_new ()));
		(_tmp10 == NULL ? NULL : (_tmp10 = (g_object_unref (_tmp10), NULL)));
		vala_type_reference_set_data_type (vala_variable_declarator_get_type_reference (vala_foreach_statement_get_iterator_variable_declarator (stmt)), self->priv->iterator_type);
		vala_type_reference_set_takes_ownership (vala_variable_declarator_get_type_reference (vala_foreach_statement_get_iterator_variable_declarator (stmt)), TRUE);
		vala_type_reference_add_type_argument (vala_variable_declarator_get_type_reference (vala_foreach_statement_get_iterator_variable_declarator (stmt)), vala_foreach_statement_get_type_reference (stmt));
		vala_block_add_local_variable (VALA_BLOCK (stmt), vala_foreach_statement_get_iterator_variable_declarator (stmt));
		vala_symbol_set_active (VALA_SYMBOL (vala_foreach_statement_get_iterator_variable_declarator (stmt)), TRUE);
	} else {
		if (!(VALA_IS_ARRAY (collection_type) || collection_type == self->priv->glist_type || collection_type == self->priv->gslist_type)) {
			vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Collection not iterable");
			(collection_type == NULL ? NULL : (collection_type = (g_object_unref (collection_type), NULL)));
			return;
		}
	}
	vala_code_node_set_tree_can_fail (VALA_CODE_NODE (stmt), vala_code_node_get_tree_can_fail (VALA_CODE_NODE (vala_foreach_statement_get_collection (stmt))) || vala_code_node_get_tree_can_fail (VALA_CODE_NODE (vala_foreach_statement_get_body (stmt))));
	(collection_type == NULL ? NULL : (collection_type = (g_object_unref (collection_type), NULL)));
}


static void vala_semantic_analyzer_real_visit_return_statement (ValaCodeVisitor* base, ValaReturnStatement* stmt) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_RETURN_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
	if (vala_return_statement_get_return_expression (stmt) != NULL && vala_code_node_get_error (VALA_CODE_NODE (vala_return_statement_get_return_expression (stmt)))) {
		/* ignore inner error*/
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		return;
	}
	if (self->priv->current_return_type == NULL) {
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Return not allowed in this context");
		return;
	}
	if (vala_return_statement_get_return_expression (stmt) == NULL && vala_type_reference_get_data_type (self->priv->current_return_type) != NULL) {
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Return without value in non-void function");
		return;
	}
	if (vala_return_statement_get_return_expression (stmt) != NULL && vala_type_reference_get_data_type (self->priv->current_return_type) == NULL && vala_type_reference_get_type_parameter (self->priv->current_return_type) == NULL) {
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Return with value in void function");
		return;
	}
	if (vala_return_statement_get_return_expression (stmt) != NULL && !vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (vala_return_statement_get_return_expression (stmt)), self->priv->current_return_type)) {
		char* _tmp2;
		char* _tmp1;
		char* _tmp0;
		_tmp2 = NULL;
		_tmp1 = NULL;
		_tmp0 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), (_tmp2 = g_strdup_printf ("Return: Cannot convert from `%s' to `%s'", (_tmp0 = vala_type_reference_to_string (vala_expression_get_static_type (vala_return_statement_get_return_expression (stmt)))), (_tmp1 = vala_type_reference_to_string (self->priv->current_return_type)))));
		(_tmp2 = (g_free (_tmp2), NULL));
		(_tmp1 = (g_free (_tmp1), NULL));
		(_tmp0 = (g_free (_tmp0), NULL));
		return;
	}
	if (vala_return_statement_get_return_expression (stmt) != NULL && vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (vala_return_statement_get_return_expression (stmt))) && !vala_type_reference_get_transfers_ownership (self->priv->current_return_type)) {
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Return value transfers ownership but method return type hasn't been declared to transfer ownership");
		return;
	}
	if (vala_return_statement_get_return_expression (stmt) != NULL && VALA_IS_VARIABLE_DECLARATOR (vala_expression_get_symbol_reference (vala_return_statement_get_return_expression (stmt))) && vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_return_statement_get_return_expression (stmt))) && !vala_type_reference_get_transfers_ownership (self->priv->current_return_type)) {
		vala_report_warning (vala_code_node_get_source_reference (VALA_CODE_NODE (stmt)), "Local variable with strong reference used as return value and method return type hasn't been declared to transfer ownership");
	}
}


static void vala_semantic_analyzer_real_visit_throw_statement (ValaCodeVisitor* base, ValaThrowStatement* stmt) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_THROW_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
}


static void vala_semantic_analyzer_real_visit_try_statement (ValaCodeVisitor* base, ValaTryStatement* stmt) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_TRY_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
}


static void vala_semantic_analyzer_real_visit_catch_clause (ValaCodeVisitor* base, ValaCatchClause* clause) {
	ValaSemanticAnalyzer * self;
	ValaVariableDeclarator* _tmp0;
	ValaTypeReference* _tmp1;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CATCH_CLAUSE (clause));
	if (vala_type_reference_get_data_type (vala_catch_clause_get_type_reference (clause)) != NULL) {
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (vala_catch_clause_get_type_reference (clause))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	}
	_tmp0 = NULL;
	vala_catch_clause_set_variable_declarator (clause, (_tmp0 = vala_variable_declarator_new (vala_catch_clause_get_variable_name (clause), NULL, NULL)));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	_tmp1 = NULL;
	vala_variable_declarator_set_type_reference (vala_catch_clause_get_variable_declarator (clause), (_tmp1 = vala_type_reference_new ()));
	(_tmp1 == NULL ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL)));
	vala_type_reference_set_data_type (vala_variable_declarator_get_type_reference (vala_catch_clause_get_variable_declarator (clause)), self->priv->gerror_type);
	vala_scope_add (vala_symbol_get_scope (VALA_SYMBOL (vala_catch_clause_get_body (clause))), vala_catch_clause_get_variable_name (clause), VALA_SYMBOL (vala_catch_clause_get_variable_declarator (clause)));
	vala_code_node_accept_children (VALA_CODE_NODE (clause), VALA_CODE_VISITOR (self));
}


/**
 * Visit operation called for lock statements.
 *
 * @param stmt a lock statement
 */
static void vala_semantic_analyzer_real_visit_lock_statement (ValaCodeVisitor* base, ValaLockStatement* stmt) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_LOCK_STATEMENT (stmt));
	/* resource must be a member access and denote a Lockable */
	if (!(VALA_IS_MEMBER_ACCESS (vala_lock_statement_get_resource (stmt)) && VALA_IS_LOCKABLE (vala_expression_get_symbol_reference (vala_lock_statement_get_resource (stmt))))) {
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		vala_code_node_set_error (VALA_CODE_NODE (vala_lock_statement_get_resource (stmt)), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_lock_statement_get_resource (stmt))), "Expression is either not a member access or does not denote a lockable member");
		return;
	}
	/* parent symbol must be the current class */
	if (vala_symbol_get_parent_symbol (vala_expression_get_symbol_reference (vala_lock_statement_get_resource (stmt))) != VALA_SYMBOL (self->priv->current_class)) {
		vala_code_node_set_error (VALA_CODE_NODE (stmt), TRUE);
		vala_code_node_set_error (VALA_CODE_NODE (vala_lock_statement_get_resource (stmt)), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_lock_statement_get_resource (stmt))), "Only members of the current class are lockable");
	}
	vala_lockable_set_lock_used ((VALA_LOCKABLE (vala_expression_get_symbol_reference (vala_lock_statement_get_resource (stmt)))), TRUE);
}


static gint vala_semantic_analyzer_create_sizes_from_initializer_list (ValaSemanticAnalyzer* self, ValaInitializerList* il, gint rank, GeeList* sl) {
	ValaIntegerLiteral* _tmp1;
	char* _tmp0;
	ValaLiteralExpression* _tmp2;
	ValaLiteralExpression* init;
	gint subsize;
	gint _tmp8;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), 0);
	g_return_val_if_fail (VALA_IS_INITIALIZER_LIST (il), 0);
	g_return_val_if_fail (GEE_IS_LIST (sl), 0);
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp2 = NULL;
	init = (_tmp2 = vala_literal_expression_new (VALA_LITERAL ((_tmp1 = vala_integer_literal_new ((_tmp0 = g_strdup_printf ("%i", vala_initializer_list_get_size (il))), vala_code_node_get_source_reference (VALA_CODE_NODE (il))))), vala_code_node_get_source_reference (VALA_CODE_NODE (il))), (_tmp1 == NULL ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL))), (_tmp0 = (g_free (_tmp0), NULL)), _tmp2);
	vala_code_node_accept (VALA_CODE_NODE (init), VALA_CODE_VISITOR (self));
	gee_collection_add (GEE_COLLECTION (sl), init);
	subsize = -1;
	{
		GeeCollection* e_collection;
		GeeIterator* e_it;
		e_collection = vala_initializer_list_get_initializers (il);
		e_it = gee_iterable_iterator (GEE_ITERABLE (e_collection));
		while (gee_iterator_next (e_it)) {
			ValaExpression* e;
			e = gee_iterator_get (e_it);
			{
				if (VALA_IS_INITIALIZER_LIST (e)) {
					gint size;
					if (rank == 1) {
						gint _tmp3;
						vala_code_node_set_error (VALA_CODE_NODE (il), TRUE);
						vala_code_node_set_error (VALA_CODE_NODE (e), TRUE);
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (e)), "Expected array element, got array initializer list");
						return (_tmp3 = -1, (e == NULL ? NULL : (e = (g_object_unref (e), NULL))), (e_collection == NULL ? NULL : (e_collection = (g_object_unref (e_collection), NULL))), (e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL))), (init == NULL ? NULL : (init = (g_object_unref (init), NULL))), _tmp3);
					}
					size = vala_semantic_analyzer_create_sizes_from_initializer_list (self, VALA_INITIALIZER_LIST (e), rank - 1, sl);
					if (size == -1) {
						gint _tmp4;
						return (_tmp4 = -1, (e == NULL ? NULL : (e = (g_object_unref (e), NULL))), (e_collection == NULL ? NULL : (e_collection = (g_object_unref (e_collection), NULL))), (e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL))), (init == NULL ? NULL : (init = (g_object_unref (init), NULL))), _tmp4);
					}
					if (subsize >= 0 && subsize != size) {
						char* _tmp5;
						gint _tmp6;
						vala_code_node_set_error (VALA_CODE_NODE (il), TRUE);
						_tmp5 = NULL;
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (il)), (_tmp5 = g_strdup_printf ("Expected initializer list of size %d, got size %d", subsize, size)));
						(_tmp5 = (g_free (_tmp5), NULL));
						return (_tmp6 = -1, (e == NULL ? NULL : (e = (g_object_unref (e), NULL))), (e_collection == NULL ? NULL : (e_collection = (g_object_unref (e_collection), NULL))), (e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL))), (init == NULL ? NULL : (init = (g_object_unref (init), NULL))), _tmp6);
					} else {
						subsize = size;
					}
				} else {
					if (rank != 1) {
						gint _tmp7;
						vala_code_node_set_error (VALA_CODE_NODE (il), TRUE);
						vala_code_node_set_error (VALA_CODE_NODE (e), TRUE);
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (e)), "Expected array initializer list, got array element");
						return (_tmp7 = -1, (e == NULL ? NULL : (e = (g_object_unref (e), NULL))), (e_collection == NULL ? NULL : (e_collection = (g_object_unref (e_collection), NULL))), (e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL))), (init == NULL ? NULL : (init = (g_object_unref (init), NULL))), _tmp7);
					}
				}
				(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
			}
		}
		(e_collection == NULL ? NULL : (e_collection = (g_object_unref (e_collection), NULL)));
		(e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL)));
	}
	return (_tmp8 = vala_initializer_list_get_size (il), (init == NULL ? NULL : (init = (g_object_unref (init), NULL))), _tmp8);
	(init == NULL ? NULL : (init = (g_object_unref (init), NULL)));
}


/**
 * Visit operations called for array creation expresions.
 *
 * @param expr an array creation expression
 */
static void vala_semantic_analyzer_real_visit_array_creation_expression (ValaCodeVisitor* base, ValaArrayCreationExpression* expr) {
	ValaSemanticAnalyzer * self;
	GeeCollection* size;
	ValaInitializerList* _tmp0;
	ValaInitializerList* initlist;
	GeeArrayList* calc_sizes;
	ValaTypeReference* _tmp3;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_ARRAY_CREATION_EXPRESSION (expr));
	size = GEE_COLLECTION (vala_array_creation_expression_get_sizes (expr));
	_tmp0 = NULL;
	initlist = (_tmp0 = vala_array_creation_expression_get_initializer_list (expr), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
	if (vala_array_creation_expression_get_element_type (expr) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_array_creation_expression_get_element_type (expr)), VALA_CODE_VISITOR (self));
	}
	{
		GeeCollection* e_collection;
		GeeIterator* e_it;
		e_collection = size;
		e_it = gee_iterable_iterator (GEE_ITERABLE (e_collection));
		while (gee_iterator_next (e_it)) {
			ValaExpression* e;
			e = gee_iterator_get (e_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (e), VALA_CODE_VISITOR (self));
				(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
			}
		}
		(e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL)));
	}
	calc_sizes = gee_array_list_new (VALA_TYPE_LITERAL_EXPRESSION, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal);
	if (initlist != NULL) {
		ValaTypeReference* _tmp1;
		ValaArray* _tmp2;
		gint ret;
		_tmp1 = NULL;
		vala_expression_set_expected_type (VALA_EXPRESSION (initlist), (_tmp1 = vala_type_reference_copy (vala_array_creation_expression_get_element_type (expr))));
		(_tmp1 = (g_object_unref (_tmp1), NULL));
		_tmp2 = NULL;
		vala_type_reference_set_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (initlist)), VALA_DATA_TYPE ((_tmp2 = vala_data_type_get_array (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (initlist))), vala_array_creation_expression_get_rank (expr)))));
		(_tmp2 = (g_object_unref (_tmp2), NULL));
		/* FIXME: add element type to type_argument*/
		vala_code_node_accept (VALA_CODE_NODE (initlist), VALA_CODE_VISITOR (self));
		ret = vala_semantic_analyzer_create_sizes_from_initializer_list (self, initlist, vala_array_creation_expression_get_rank (expr), GEE_LIST (calc_sizes));
		if (ret == -1) {
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		}
	}
	if (gee_collection_get_size (size) > 0) {
		{
			GeeCollection* e_collection;
			GeeIterator* e_it;
			e_collection = size;
			e_it = gee_iterable_iterator (GEE_ITERABLE (e_collection));
			while (gee_iterator_next (e_it)) {
				ValaExpression* e;
				e = gee_iterator_get (e_it);
				{
					/* check for errors in the size list */
					if (vala_expression_get_static_type (e) == NULL) {
						(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
						(e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL)));
						(size == NULL ? NULL : (size = (g_object_unref (size), NULL)));
						(initlist == NULL ? NULL : (initlist = (g_object_unref (initlist), NULL)));
						(calc_sizes == NULL ? NULL : (calc_sizes = (g_object_unref (calc_sizes), NULL)));
						return;
					} else {
						/* return on previous error */
						if (!(VALA_IS_STRUCT (vala_type_reference_get_data_type (vala_expression_get_static_type (e)))) || !vala_struct_is_integer_type ((VALA_STRUCT (vala_type_reference_get_data_type (vala_expression_get_static_type (e)))))) {
							vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (e)), "Expression of integer type expected");
						}
					}
					(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
				}
			}
			(e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL)));
		}
	} else {
		if (initlist == NULL) {
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			/* this is an internal error because it is already handeld by the parser */
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "internal error: initializer list expected");
		} else {
			{
				GeeArrayList* size_collection;
				GeeIterator* size_it;
				size_collection = calc_sizes;
				size_it = gee_iterable_iterator (GEE_ITERABLE (size_collection));
				while (gee_iterator_next (size_it)) {
					ValaExpression* size;
					size = gee_iterator_get (size_it);
					{
						vala_array_creation_expression_append_size (expr, size);
						(size == NULL ? NULL : (size = (g_object_unref (size), NULL)));
					}
				}
				(size_it == NULL ? NULL : (size_it = (g_object_unref (size_it), NULL)));
			}
		}
	}
	if (vala_code_node_get_error (VALA_CODE_NODE (expr))) {
		(size == NULL ? NULL : (size = (g_object_unref (size), NULL)));
		(initlist == NULL ? NULL : (initlist = (g_object_unref (initlist), NULL)));
		(calc_sizes == NULL ? NULL : (calc_sizes = (g_object_unref (calc_sizes), NULL)));
		return;
	}
	/* check for wrong elements inside the initializer */
	if (vala_array_creation_expression_get_initializer_list (expr) != NULL && vala_expression_get_static_type (VALA_EXPRESSION (vala_array_creation_expression_get_initializer_list (expr))) == NULL) {
		(size == NULL ? NULL : (size = (g_object_unref (size), NULL)));
		(initlist == NULL ? NULL : (initlist = (g_object_unref (initlist), NULL)));
		(calc_sizes == NULL ? NULL : (calc_sizes = (g_object_unref (calc_sizes), NULL)));
		return;
	}
	/* try to construct the type of the array */
	if (vala_array_creation_expression_get_element_type (expr) == NULL) {
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Cannot determine the element type of the created array");
		(size == NULL ? NULL : (size = (g_object_unref (size), NULL)));
		(initlist == NULL ? NULL : (initlist = (g_object_unref (initlist), NULL)));
		(calc_sizes == NULL ? NULL : (calc_sizes = (g_object_unref (calc_sizes), NULL)));
		return;
	}
	_tmp3 = NULL;
	vala_expression_set_static_type (VALA_EXPRESSION (expr), (_tmp3 = vala_type_reference_copy (vala_array_creation_expression_get_element_type (expr))));
	(_tmp3 = (g_object_unref (_tmp3), NULL));
	if (vala_type_reference_get_data_type (vala_array_creation_expression_get_element_type (expr)) != NULL) {
		ValaArray* _tmp4;
		_tmp4 = NULL;
		vala_type_reference_set_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr)), VALA_DATA_TYPE ((_tmp4 = vala_data_type_get_array (vala_type_reference_get_data_type (vala_array_creation_expression_get_element_type (expr)), vala_array_creation_expression_get_rank (expr)))));
		(_tmp4 = (g_object_unref (_tmp4), NULL));
	} else {
		ValaArray* _tmp5;
		_tmp5 = NULL;
		vala_type_reference_set_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr)), VALA_DATA_TYPE ((_tmp5 = vala_type_parameter_get_array (vala_type_reference_get_type_parameter (vala_array_creation_expression_get_element_type (expr)), vala_array_creation_expression_get_rank (expr)))));
		(_tmp5 = (g_object_unref (_tmp5), NULL));
	}
	vala_type_reference_set_transfers_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), TRUE);
	vala_type_reference_set_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), TRUE);
	vala_type_reference_add_type_argument (vala_expression_get_static_type (VALA_EXPRESSION (expr)), vala_array_creation_expression_get_element_type (expr));
	vala_type_reference_set_takes_ownership (vala_array_creation_expression_get_element_type (expr), TRUE);
	(size == NULL ? NULL : (size = (g_object_unref (size), NULL)));
	(initlist == NULL ? NULL : (initlist = (g_object_unref (initlist), NULL)));
	(calc_sizes == NULL ? NULL : (calc_sizes = (g_object_unref (calc_sizes), NULL)));
}


static void vala_semantic_analyzer_real_visit_boolean_literal (ValaCodeVisitor* base, ValaBooleanLiteral* expr) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_BOOLEAN_LITERAL (expr));
	vala_literal_set_static_type (VALA_LITERAL (expr), self->priv->bool_type);
}


static void vala_semantic_analyzer_real_visit_character_literal (ValaCodeVisitor* base, ValaCharacterLiteral* expr) {
	ValaSemanticAnalyzer * self;
	ValaTypeReference* _tmp0;
	ValaDataType* _tmp1;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CHARACTER_LITERAL (expr));
	_tmp0 = NULL;
	vala_literal_set_static_type (VALA_LITERAL (expr), (_tmp0 = vala_type_reference_new ()));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	_tmp1 = NULL;
	vala_type_reference_set_data_type (vala_literal_get_static_type (VALA_LITERAL (expr)), (_tmp1 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), "char"))));
	(_tmp1 == NULL ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL)));
}


static void vala_semantic_analyzer_real_visit_integer_literal (ValaCodeVisitor* base, ValaIntegerLiteral* expr) {
	ValaSemanticAnalyzer * self;
	ValaTypeReference* _tmp0;
	ValaDataType* _tmp2;
	char* _tmp1;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_INTEGER_LITERAL (expr));
	_tmp0 = NULL;
	vala_literal_set_static_type (VALA_LITERAL (expr), (_tmp0 = vala_type_reference_new ()));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	_tmp2 = NULL;
	_tmp1 = NULL;
	vala_type_reference_set_data_type (vala_literal_get_static_type (VALA_LITERAL (expr)), (_tmp2 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), (_tmp1 = vala_integer_literal_get_type_name (expr))))));
	(_tmp2 == NULL ? NULL : (_tmp2 = (g_object_unref (_tmp2), NULL)));
	(_tmp1 = (g_free (_tmp1), NULL));
}


static void vala_semantic_analyzer_real_visit_real_literal (ValaCodeVisitor* base, ValaRealLiteral* expr) {
	ValaSemanticAnalyzer * self;
	ValaTypeReference* _tmp0;
	ValaDataType* _tmp2;
	char* _tmp1;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_REAL_LITERAL (expr));
	_tmp0 = NULL;
	vala_literal_set_static_type (VALA_LITERAL (expr), (_tmp0 = vala_type_reference_new ()));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	_tmp2 = NULL;
	_tmp1 = NULL;
	vala_type_reference_set_data_type (vala_literal_get_static_type (VALA_LITERAL (expr)), (_tmp2 = VALA_DATA_TYPE (vala_scope_lookup (vala_symbol_get_scope (self->priv->root_symbol), (_tmp1 = vala_real_literal_get_type_name (expr))))));
	(_tmp2 == NULL ? NULL : (_tmp2 = (g_object_unref (_tmp2), NULL)));
	(_tmp1 = (g_free (_tmp1), NULL));
}


static void vala_semantic_analyzer_real_visit_string_literal (ValaCodeVisitor* base, ValaStringLiteral* expr) {
	ValaSemanticAnalyzer * self;
	ValaTypeReference* _tmp0;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_STRING_LITERAL (expr));
	_tmp0 = NULL;
	vala_literal_set_static_type (VALA_LITERAL (expr), (_tmp0 = vala_type_reference_copy (self->priv->string_type)));
	(_tmp0 = (g_object_unref (_tmp0), NULL));
	vala_type_reference_set_non_null (vala_literal_get_static_type (VALA_LITERAL (expr)), TRUE);
}


static void vala_semantic_analyzer_real_visit_null_literal (ValaCodeVisitor* base, ValaNullLiteral* expr) {
	ValaSemanticAnalyzer * self;
	ValaTypeReference* _tmp0;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_NULL_LITERAL (expr));
	/* empty TypeReference represents null */
	_tmp0 = NULL;
	vala_literal_set_static_type (VALA_LITERAL (expr), (_tmp0 = vala_type_reference_new ()));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
}


static void vala_semantic_analyzer_real_visit_literal_expression (ValaCodeVisitor* base, ValaLiteralExpression* expr) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_LITERAL_EXPRESSION (expr));
	vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_literal_get_static_type (vala_literal_expression_get_literal (expr)));
}


static ValaTypeReference* vala_semantic_analyzer_get_static_type_for_symbol (ValaSemanticAnalyzer* self, ValaSymbol* sym) {
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	g_return_val_if_fail (VALA_IS_SYMBOL (sym), NULL);
	if (VALA_IS_FIELD (sym)) {
		ValaField* _tmp0;
		ValaField* f;
		ValaTypeReference* _tmp1;
		_tmp0 = NULL;
		f = (_tmp0 = VALA_FIELD (sym), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
		_tmp1 = NULL;
		return (_tmp1 = g_object_ref (vala_field_get_type_reference (f)), (f == NULL ? NULL : (f = (g_object_unref (f), NULL))), _tmp1);
		(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
	} else {
		if (VALA_IS_CONSTANT (sym)) {
			ValaConstant* _tmp2;
			ValaConstant* c;
			ValaTypeReference* _tmp3;
			_tmp2 = NULL;
			c = (_tmp2 = VALA_CONSTANT (sym), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2)));
			_tmp3 = NULL;
			return (_tmp3 = g_object_ref (vala_constant_get_type_reference (c)), (c == NULL ? NULL : (c = (g_object_unref (c), NULL))), _tmp3);
			(c == NULL ? NULL : (c = (g_object_unref (c), NULL)));
		} else {
			if (VALA_IS_PROPERTY (sym)) {
				ValaProperty* _tmp4;
				ValaProperty* prop;
				ValaTypeReference* type;
				ValaTypeReference* _tmp5;
				_tmp4 = NULL;
				prop = (_tmp4 = VALA_PROPERTY (sym), (_tmp4 == NULL ? NULL : g_object_ref (_tmp4)));
				type = vala_type_reference_copy (vala_property_get_type_reference (prop));
				vala_type_reference_set_takes_ownership (type, FALSE);
				_tmp5 = NULL;
				return (_tmp5 = type, (prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL))), _tmp5);
				(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
				(type = (g_object_unref (type), NULL));
			} else {
				if (VALA_IS_FORMAL_PARAMETER (sym)) {
					ValaFormalParameter* _tmp6;
					ValaFormalParameter* p;
					ValaTypeReference* _tmp7;
					ValaTypeReference* _tmp8;
					_tmp6 = NULL;
					p = (_tmp6 = VALA_FORMAL_PARAMETER (sym), (_tmp6 == NULL ? NULL : g_object_ref (_tmp6)));
					_tmp7 = NULL;
					_tmp8 = NULL;
					return (_tmp8 = (_tmp7 = vala_formal_parameter_get_type_reference (p), (_tmp7 == NULL ? NULL : g_object_ref (_tmp7))), (p == NULL ? NULL : (p = (g_object_unref (p), NULL))), _tmp8);
					(p == NULL ? NULL : (p = (g_object_unref (p), NULL)));
				} else {
					if (VALA_IS_TYPE_REFERENCE (sym)) {
						ValaTypeReference* _tmp9;
						_tmp9 = NULL;
						return (_tmp9 = VALA_TYPE_REFERENCE (sym), (_tmp9 == NULL ? NULL : g_object_ref (_tmp9)));
					} else {
						if (VALA_IS_VARIABLE_DECLARATOR (sym)) {
							ValaVariableDeclarator* _tmp11;
							ValaVariableDeclarator* decl;
							ValaTypeReference* _tmp12;
							ValaTypeReference* _tmp13;
							_tmp11 = NULL;
							decl = (_tmp11 = VALA_VARIABLE_DECLARATOR (sym), (_tmp11 == NULL ? NULL : g_object_ref (_tmp11)));
							_tmp12 = NULL;
							_tmp13 = NULL;
							return (_tmp13 = (_tmp12 = vala_variable_declarator_get_type_reference (decl), (_tmp12 == NULL ? NULL : g_object_ref (_tmp12))), (decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL))), _tmp13);
							(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
						} else {
							if (VALA_IS_ENUM_VALUE (sym)) {
								ValaTypeReference* type;
								type = vala_type_reference_new ();
								vala_type_reference_set_data_type (type, VALA_DATA_TYPE (vala_symbol_get_parent_symbol (sym)));
								return type;
								(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
							}
						}
					}
				}
			}
		}
	}
	return NULL;
}


ValaSymbol* vala_semantic_analyzer_symbol_lookup_inherited (ValaSymbol* sym, const char* name) {
	ValaSymbol* result;
	gpointer _tmp10;
	g_return_val_if_fail (VALA_IS_SYMBOL (sym), NULL);
	g_return_val_if_fail (name != NULL, NULL);
	result = vala_scope_lookup (vala_symbol_get_scope (sym), name);
	if (result != NULL) {
		return result;
	}
	if (VALA_IS_CLASS (sym)) {
		ValaClass* _tmp1;
		ValaClass* cl;
		_tmp1 = NULL;
		cl = (_tmp1 = VALA_CLASS (sym), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1)));
		{
			GeeCollection* base_type_collection;
			GeeIterator* base_type_it;
			base_type_collection = vala_class_get_base_types (cl);
			base_type_it = gee_iterable_iterator (GEE_ITERABLE (base_type_collection));
			while (gee_iterator_next (base_type_it)) {
				ValaTypeReference* base_type;
				base_type = gee_iterator_get (base_type_it);
				{
					ValaSymbol* _tmp2;
					_tmp2 = NULL;
					result = (_tmp2 = vala_semantic_analyzer_symbol_lookup_inherited (VALA_SYMBOL (vala_type_reference_get_data_type (base_type)), name), (result == NULL ? NULL : (result = (g_object_unref (result), NULL))), _tmp2);
					if (result != NULL) {
						ValaSymbol* _tmp3;
						_tmp3 = NULL;
						return (_tmp3 = result, (base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL))), (base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL))), (base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL))), (cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL))), _tmp3);
					}
					(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
				}
			}
			(base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL)));
			(base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL)));
		}
		(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
	} else {
		if (VALA_IS_STRUCT (sym)) {
			ValaStruct* _tmp4;
			ValaStruct* st;
			_tmp4 = NULL;
			st = (_tmp4 = VALA_STRUCT (sym), (_tmp4 == NULL ? NULL : g_object_ref (_tmp4)));
			{
				GeeCollection* base_type_collection;
				GeeIterator* base_type_it;
				base_type_collection = vala_struct_get_base_types (st);
				base_type_it = gee_iterable_iterator (GEE_ITERABLE (base_type_collection));
				while (gee_iterator_next (base_type_it)) {
					ValaTypeReference* base_type;
					base_type = gee_iterator_get (base_type_it);
					{
						ValaSymbol* _tmp5;
						_tmp5 = NULL;
						result = (_tmp5 = vala_semantic_analyzer_symbol_lookup_inherited (VALA_SYMBOL (vala_type_reference_get_data_type (base_type)), name), (result == NULL ? NULL : (result = (g_object_unref (result), NULL))), _tmp5);
						if (result != NULL) {
							ValaSymbol* _tmp6;
							_tmp6 = NULL;
							return (_tmp6 = result, (base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL))), (base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL))), (base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL))), (st == NULL ? NULL : (st = (g_object_unref (st), NULL))), _tmp6);
						}
						(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
					}
				}
				(base_type_collection == NULL ? NULL : (base_type_collection = (g_object_unref (base_type_collection), NULL)));
				(base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL)));
			}
			(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
		} else {
			if (VALA_IS_INTERFACE (sym)) {
				ValaInterface* _tmp7;
				ValaInterface* iface;
				_tmp7 = NULL;
				iface = (_tmp7 = VALA_INTERFACE (sym), (_tmp7 == NULL ? NULL : g_object_ref (_tmp7)));
				{
					GeeCollection* prerequisite_collection;
					GeeIterator* prerequisite_it;
					prerequisite_collection = vala_interface_get_prerequisites (iface);
					prerequisite_it = gee_iterable_iterator (GEE_ITERABLE (prerequisite_collection));
					while (gee_iterator_next (prerequisite_it)) {
						ValaTypeReference* prerequisite;
						prerequisite = gee_iterator_get (prerequisite_it);
						{
							ValaSymbol* _tmp8;
							_tmp8 = NULL;
							result = (_tmp8 = vala_semantic_analyzer_symbol_lookup_inherited (VALA_SYMBOL (vala_type_reference_get_data_type (prerequisite)), name), (result == NULL ? NULL : (result = (g_object_unref (result), NULL))), _tmp8);
							if (result != NULL) {
								ValaSymbol* _tmp9;
								_tmp9 = NULL;
								return (_tmp9 = result, (prerequisite == NULL ? NULL : (prerequisite = (g_object_unref (prerequisite), NULL))), (prerequisite_collection == NULL ? NULL : (prerequisite_collection = (g_object_unref (prerequisite_collection), NULL))), (prerequisite_it == NULL ? NULL : (prerequisite_it = (g_object_unref (prerequisite_it), NULL))), (iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL))), _tmp9);
							}
							(prerequisite == NULL ? NULL : (prerequisite = (g_object_unref (prerequisite), NULL)));
						}
					}
					(prerequisite_collection == NULL ? NULL : (prerequisite_collection = (g_object_unref (prerequisite_collection), NULL)));
					(prerequisite_it == NULL ? NULL : (prerequisite_it = (g_object_unref (prerequisite_it), NULL)));
				}
				(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
			}
		}
	}
	return (_tmp10 = NULL, (result == NULL ? NULL : (result = (g_object_unref (result), NULL))), _tmp10);
	(result == NULL ? NULL : (result = (g_object_unref (result), NULL)));
}


static void vala_semantic_analyzer_real_visit_parenthesized_expression (ValaCodeVisitor* base, ValaParenthesizedExpression* expr) {
	ValaSemanticAnalyzer * self;
	ValaTypeReference* _tmp0;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_PARENTHESIZED_EXPRESSION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_parenthesized_expression_get_inner (expr)))) {
		/* ignore inner error*/
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		return;
	}
	if (vala_expression_get_static_type (vala_parenthesized_expression_get_inner (expr)) == NULL) {
		/* static type may be null for method references*/
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_parenthesized_expression_get_inner (expr))), "Invalid expression type");
		return;
	}
	_tmp0 = NULL;
	vala_expression_set_static_type (VALA_EXPRESSION (expr), (_tmp0 = vala_type_reference_copy (vala_expression_get_static_type (vala_parenthesized_expression_get_inner (expr)))));
	(_tmp0 = (g_object_unref (_tmp0), NULL));
	/* don't call g_object_ref_sink on inner and outer expression*/
	vala_type_reference_set_floating_reference (vala_expression_get_static_type (VALA_EXPRESSION (expr)), FALSE);
}


static void vala_semantic_analyzer_real_visit_member_access (ValaCodeVisitor* base, ValaMemberAccess* expr) {
	ValaSemanticAnalyzer * self;
	ValaSymbol* base_symbol;
	gboolean may_access_instance_members;
	ValaSymbol* _tmp22;
	ValaSymbol* member;
	ValaSymbolAccessibility access;
	gboolean instance;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_MEMBER_ACCESS (expr));
	base_symbol = NULL;
	may_access_instance_members = FALSE;
	vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), NULL);
	if (vala_member_access_get_inner (expr) == NULL) {
		ValaSymbol* _tmp1;
		ValaSymbol* _tmp0;
		ValaSymbol* _tmp2;
		ValaSymbol* sym;
		_tmp1 = NULL;
		_tmp0 = NULL;
		base_symbol = (_tmp1 = (_tmp0 = self->priv->current_symbol, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0))), (base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL))), _tmp1);
		_tmp2 = NULL;
		sym = (_tmp2 = self->priv->current_symbol, (_tmp2 == NULL ? NULL : g_object_ref (_tmp2)));
		while (sym != NULL && vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL) {
			ValaSymbol* _tmp4;
			ValaSymbol* _tmp6;
			ValaSymbol* _tmp5;
			if (VALA_IS_CREATION_METHOD (sym) || VALA_IS_PROPERTY (sym) || VALA_IS_CONSTRUCTOR (sym) || VALA_IS_DESTRUCTOR (sym)) {
				may_access_instance_members = TRUE;
			} else {
				if (VALA_IS_METHOD (sym)) {
					ValaMethod* _tmp3;
					ValaMethod* m;
					_tmp3 = NULL;
					m = (_tmp3 = VALA_METHOD (sym), (_tmp3 == NULL ? NULL : g_object_ref (_tmp3)));
					may_access_instance_members = vala_method_get_instance (m);
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				}
			}
			_tmp4 = NULL;
			vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), (_tmp4 = vala_semantic_analyzer_symbol_lookup_inherited (sym, vala_member_access_get_member_name (expr))));
			(_tmp4 == NULL ? NULL : (_tmp4 = (g_object_unref (_tmp4), NULL)));
			_tmp6 = NULL;
			_tmp5 = NULL;
			sym = (_tmp6 = (_tmp5 = vala_symbol_get_parent_symbol (sym), (_tmp5 == NULL ? NULL : g_object_ref (_tmp5))), (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), _tmp6);
		}
		if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL) {
			{
				GeeCollection* ns_collection;
				GeeIterator* ns_it;
				ns_collection = self->priv->current_using_directives;
				ns_it = gee_iterable_iterator (GEE_ITERABLE (ns_collection));
				while (gee_iterator_next (ns_it)) {
					ValaNamespaceReference* ns;
					ns = gee_iterator_get (ns_it);
					{
						ValaSymbol* local_sym;
						local_sym = vala_scope_lookup (vala_symbol_get_scope (vala_namespace_reference_get_namespace_symbol (ns)), vala_member_access_get_member_name (expr));
						if (local_sym != NULL) {
							if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) != NULL) {
								char* _tmp9;
								char* _tmp8;
								char* _tmp7;
								vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
								_tmp9 = NULL;
								_tmp8 = NULL;
								_tmp7 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp9 = g_strdup_printf ("`%s' is an ambiguous reference between `%s' and `%s'", vala_member_access_get_member_name (expr), (_tmp7 = vala_symbol_get_full_name (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)))), (_tmp8 = vala_symbol_get_full_name (local_sym)))));
								(_tmp9 = (g_free (_tmp9), NULL));
								(_tmp8 = (g_free (_tmp8), NULL));
								(_tmp7 = (g_free (_tmp7), NULL));
								(ns == NULL ? NULL : (ns = (g_object_unref (ns), NULL)));
								(local_sym == NULL ? NULL : (local_sym = (g_object_unref (local_sym), NULL)));
								(ns_it == NULL ? NULL : (ns_it = (g_object_unref (ns_it), NULL)));
								(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
								(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
								return;
							}
							vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), local_sym);
						}
						(ns == NULL ? NULL : (ns = (g_object_unref (ns), NULL)));
						(local_sym == NULL ? NULL : (local_sym = (g_object_unref (local_sym), NULL)));
					}
				}
				(ns_it == NULL ? NULL : (ns_it = (g_object_unref (ns_it), NULL)));
			}
		}
		(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
	} else {
		if (vala_code_node_get_error (VALA_CODE_NODE (vala_member_access_get_inner (expr)))) {
			/* if there was an error in the inner expression, skip this check */
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
			return;
		}
		if (VALA_IS_MEMBER_ACCESS (vala_member_access_get_inner (expr))) {
			ValaMemberAccess* _tmp10;
			ValaMemberAccess* ma;
			_tmp10 = NULL;
			ma = (_tmp10 = VALA_MEMBER_ACCESS (vala_member_access_get_inner (expr)), (_tmp10 == NULL ? NULL : g_object_ref (_tmp10)));
			if (vala_member_access_get_prototype_access (ma)) {
				char* _tmp12;
				char* _tmp11;
				vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
				_tmp12 = NULL;
				_tmp11 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp12 = g_strdup_printf ("Access to instance member `%s' denied", (_tmp11 = vala_symbol_get_full_name (vala_expression_get_symbol_reference (vala_member_access_get_inner (expr)))))));
				(_tmp12 = (g_free (_tmp12), NULL));
				(_tmp11 = (g_free (_tmp11), NULL));
				(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
				(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
				return;
			}
			(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
		}
		if (VALA_IS_MEMBER_ACCESS (vala_member_access_get_inner (expr)) || VALA_IS_BASE_ACCESS (vala_member_access_get_inner (expr))) {
			ValaSymbol* _tmp14;
			ValaSymbol* _tmp13;
			_tmp14 = NULL;
			_tmp13 = NULL;
			base_symbol = (_tmp14 = (_tmp13 = vala_expression_get_symbol_reference (vala_member_access_get_inner (expr)), (_tmp13 == NULL ? NULL : g_object_ref (_tmp13))), (base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL))), _tmp14);
			if (VALA_IS_NAMESPACE (base_symbol) || VALA_IS_DATA_TYPE (base_symbol)) {
				ValaSymbol* _tmp15;
				_tmp15 = NULL;
				vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), (_tmp15 = vala_scope_lookup (vala_symbol_get_scope (base_symbol), vala_member_access_get_member_name (expr))));
				(_tmp15 == NULL ? NULL : (_tmp15 = (g_object_unref (_tmp15), NULL)));
				if (VALA_IS_BASE_ACCESS (vala_member_access_get_inner (expr))) {
					/* inner expression is base access
					 access to instance members of the base type possible*/
					may_access_instance_members = TRUE;
				}
			}
		}
		if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL && vala_expression_get_static_type (vala_member_access_get_inner (expr)) != NULL) {
			ValaSymbol* _tmp17;
			ValaDataType* _tmp16;
			ValaSymbol* _tmp18;
			_tmp17 = NULL;
			_tmp16 = NULL;
			base_symbol = (_tmp17 = VALA_SYMBOL ((_tmp16 = vala_type_reference_get_data_type (vala_expression_get_static_type (vala_member_access_get_inner (expr))), (_tmp16 == NULL ? NULL : g_object_ref (_tmp16)))), (base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL))), _tmp17);
			_tmp18 = NULL;
			vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), (_tmp18 = vala_semantic_analyzer_symbol_lookup_inherited (base_symbol, vala_member_access_get_member_name (expr))));
			(_tmp18 == NULL ? NULL : (_tmp18 = (g_object_unref (_tmp18), NULL)));
			if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) != NULL) {
				/* inner expression is variable, field, or parameter
				 access to instance members of the corresponding type possible*/
				may_access_instance_members = TRUE;
			}
		}
	}
	if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL) {
		{
			GeeCollection* binding_provider_collection;
			GeeIterator* binding_provider_it;
			binding_provider_collection = self->priv->binding_providers;
			binding_provider_it = gee_iterable_iterator (GEE_ITERABLE (binding_provider_collection));
			while (gee_iterator_next (binding_provider_it)) {
				ValaBindingProvider* binding_provider;
				binding_provider = gee_iterator_get (binding_provider_it);
				{
					ValaSymbol* _tmp19;
					/* allow plug-ins to provide custom member bindings */
					_tmp19 = NULL;
					vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), (_tmp19 = vala_binding_provider_get_binding (binding_provider, expr)));
					(_tmp19 == NULL ? NULL : (_tmp19 = (g_object_unref (_tmp19), NULL)));
					if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) != NULL) {
						may_access_instance_members = TRUE;
						(binding_provider == NULL ? NULL : (binding_provider = (g_object_unref (binding_provider), NULL)));
						break;
					}
					(binding_provider == NULL ? NULL : (binding_provider = (g_object_unref (binding_provider), NULL)));
				}
			}
			(binding_provider_it == NULL ? NULL : (binding_provider_it = (g_object_unref (binding_provider_it), NULL)));
		}
		if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL) {
			char* _tmp21;
			char* _tmp20;
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			_tmp21 = NULL;
			_tmp20 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp21 = g_strdup_printf ("The name `%s' does not exist in the context of `%s'", vala_member_access_get_member_name (expr), (_tmp20 = vala_symbol_get_full_name (base_symbol)))));
			(_tmp21 = (g_free (_tmp21), NULL));
			(_tmp20 = (g_free (_tmp20), NULL));
			(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
			return;
		}
	}
	_tmp22 = NULL;
	member = (_tmp22 = vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)), (_tmp22 == NULL ? NULL : g_object_ref (_tmp22)));
	access = VALA_SYMBOL_ACCESSIBILITY_PUBLIC;
	instance = FALSE;
	if (VALA_IS_FIELD (member)) {
		ValaField* _tmp23;
		ValaField* f;
		_tmp23 = NULL;
		f = (_tmp23 = VALA_FIELD (member), (_tmp23 == NULL ? NULL : g_object_ref (_tmp23)));
		access = vala_symbol_get_access (VALA_SYMBOL (f));
		instance = vala_field_get_instance (f);
		(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
	} else {
		if (VALA_IS_METHOD (member)) {
			ValaMethod* _tmp24;
			ValaMethod* m;
			_tmp24 = NULL;
			m = (_tmp24 = VALA_METHOD (member), (_tmp24 == NULL ? NULL : g_object_ref (_tmp24)));
			access = vala_symbol_get_access (VALA_SYMBOL (m));
			instance = vala_method_get_instance (m);
			(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
		} else {
			if (VALA_IS_PROPERTY (member) || VALA_IS_SIGNAL (member)) {
				instance = TRUE;
			}
		}
	}
	if (access == VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
		ValaSymbol* _tmp25;
		ValaSymbol* target_type;
		gboolean in_target_type;
		_tmp25 = NULL;
		target_type = (_tmp25 = vala_symbol_get_parent_symbol (member), (_tmp25 == NULL ? NULL : g_object_ref (_tmp25)));
		in_target_type = FALSE;
		{
			ValaSymbol* this_symbol;
			ValaSymbol* _tmp29;
			ValaSymbol* _tmp28;
			ValaSymbol* _tmp27;
			ValaSymbol* _tmp26;
			this_symbol = NULL;
			_tmp29 = NULL;
			_tmp28 = NULL;
			_tmp27 = NULL;
			_tmp26 = NULL;
			for (this_symbol = (_tmp27 = (_tmp26 = self->priv->current_symbol, (_tmp26 == NULL ? NULL : g_object_ref (_tmp26))), (this_symbol == NULL ? NULL : (this_symbol = (g_object_unref (this_symbol), NULL))), _tmp27); this_symbol != NULL; this_symbol = (_tmp29 = (_tmp28 = vala_symbol_get_parent_symbol (this_symbol), (_tmp28 == NULL ? NULL : g_object_ref (_tmp28))), (this_symbol == NULL ? NULL : (this_symbol = (g_object_unref (this_symbol), NULL))), _tmp29)) {
				if (target_type == this_symbol) {
					in_target_type = TRUE;
					break;
				}
			}
			(this_symbol == NULL ? NULL : (this_symbol = (g_object_unref (this_symbol), NULL)));
		}
		if (!in_target_type) {
			char* _tmp31;
			char* _tmp30;
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			_tmp31 = NULL;
			_tmp30 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp31 = g_strdup_printf ("Access to private member `%s' denied", (_tmp30 = vala_symbol_get_full_name (member)))));
			(_tmp31 = (g_free (_tmp31), NULL));
			(_tmp30 = (g_free (_tmp30), NULL));
			(target_type == NULL ? NULL : (target_type = (g_object_unref (target_type), NULL)));
			(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
			(member == NULL ? NULL : (member = (g_object_unref (member), NULL)));
			return;
		}
		(target_type == NULL ? NULL : (target_type = (g_object_unref (target_type), NULL)));
	}
	if (instance && !may_access_instance_members) {
		vala_member_access_set_prototype_access (expr, TRUE);
	} else {
		ValaTypeReference* _tmp32;
		/* no static type for prototype access*/
		_tmp32 = NULL;
		vala_expression_set_static_type (VALA_EXPRESSION (expr), (_tmp32 = vala_semantic_analyzer_get_static_type_for_symbol (self, vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)))));
		(_tmp32 == NULL ? NULL : (_tmp32 = (g_object_unref (_tmp32), NULL)));
	}
	vala_source_file_add_symbol_dependency (self->priv->current_source_file, vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	(base_symbol == NULL ? NULL : (base_symbol = (g_object_unref (base_symbol), NULL)));
	(member == NULL ? NULL : (member = (g_object_unref (member), NULL)));
}


static gboolean vala_semantic_analyzer_is_type_compatible (ValaSemanticAnalyzer* self, ValaTypeReference* expression_type, ValaTypeReference* expected_type) {
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (expression_type), FALSE);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (expected_type), FALSE);
	/* only null is compatible to null */
	if (vala_type_reference_get_data_type (expected_type) == NULL && vala_type_reference_get_type_parameter (expected_type) == NULL) {
		return (vala_type_reference_get_data_type (expression_type) == NULL && vala_type_reference_get_type_parameter (expected_type) == NULL);
	}
	if (vala_type_reference_get_data_type (expression_type) == NULL) {
		/* null can be cast to any reference or array type or pointer type */
		if (vala_type_reference_get_type_parameter (expected_type) != NULL || vala_data_type_is_reference_type (vala_type_reference_get_data_type (expected_type)) || vala_type_reference_get_is_out (expected_type) || VALA_IS_POINTER (vala_type_reference_get_data_type (expected_type)) || VALA_IS_ARRAY (vala_type_reference_get_data_type (expected_type)) || VALA_IS_CALLBACK (vala_type_reference_get_data_type (expected_type)) || vala_type_reference_get_data_type (expected_type) == self->priv->pointer_type) {
			return TRUE;
		}
		return FALSE;
	}
	/* null is not compatible with any other type (i.e. value types) */
	if (vala_type_reference_get_data_type (expected_type) == self->priv->pointer_type) {
		/* any reference or array type or pointer type can be cast to a generic pointer */
		if (vala_type_reference_get_type_parameter (expression_type) != NULL || vala_data_type_is_reference_type (vala_type_reference_get_data_type (expression_type)) || VALA_IS_POINTER (vala_type_reference_get_data_type (expression_type)) || VALA_IS_ARRAY (vala_type_reference_get_data_type (expression_type)) || VALA_IS_CALLBACK (vala_type_reference_get_data_type (expression_type)) || vala_type_reference_get_data_type (expression_type) == self->priv->pointer_type) {
			return TRUE;
		}
		return FALSE;
	}
	/* temporarily ignore type parameters */
	if (vala_type_reference_get_type_parameter (expected_type) != NULL) {
		return TRUE;
	}
	if (VALA_IS_ARRAY (vala_type_reference_get_data_type (expression_type)) != VALA_IS_ARRAY (vala_type_reference_get_data_type (expected_type))) {
		return FALSE;
	}
	if (VALA_IS_ENUM (vala_type_reference_get_data_type (expression_type)) && vala_type_reference_get_data_type (expected_type) == vala_type_reference_get_data_type (self->priv->int_type)) {
		return TRUE;
	}
	if (vala_type_reference_get_data_type (expression_type) == vala_type_reference_get_data_type (expected_type)) {
		return TRUE;
	}
	if (VALA_IS_STRUCT (vala_type_reference_get_data_type (expression_type)) && VALA_IS_STRUCT (vala_type_reference_get_data_type (expected_type))) {
		ValaStruct* _tmp9;
		ValaStruct* expr_struct;
		ValaStruct* _tmp10;
		ValaStruct* expect_struct;
		_tmp9 = NULL;
		expr_struct = (_tmp9 = VALA_STRUCT (vala_type_reference_get_data_type (expression_type)), (_tmp9 == NULL ? NULL : g_object_ref (_tmp9)));
		_tmp10 = NULL;
		expect_struct = (_tmp10 = VALA_STRUCT (vala_type_reference_get_data_type (expected_type)), (_tmp10 == NULL ? NULL : g_object_ref (_tmp10)));
		/* integer types may be implicitly cast to floating point types */
		if (vala_struct_is_integer_type (expr_struct) && vala_struct_is_floating_type (expect_struct)) {
			gboolean _tmp11;
			return (_tmp11 = TRUE, (expr_struct == NULL ? NULL : (expr_struct = (g_object_unref (expr_struct), NULL))), (expect_struct == NULL ? NULL : (expect_struct = (g_object_unref (expect_struct), NULL))), _tmp11);
		}
		if ((vala_struct_is_integer_type (expr_struct) && vala_struct_is_integer_type (expect_struct)) || (vala_struct_is_floating_type (expr_struct) && vala_struct_is_floating_type (expect_struct))) {
			if (vala_struct_get_rank (expr_struct) <= vala_struct_get_rank (expect_struct)) {
				gboolean _tmp12;
				return (_tmp12 = TRUE, (expr_struct == NULL ? NULL : (expr_struct = (g_object_unref (expr_struct), NULL))), (expect_struct == NULL ? NULL : (expect_struct = (g_object_unref (expect_struct), NULL))), _tmp12);
			}
		}
		(expr_struct == NULL ? NULL : (expr_struct = (g_object_unref (expr_struct), NULL)));
		(expect_struct == NULL ? NULL : (expect_struct = (g_object_unref (expect_struct), NULL)));
	}
	return vala_data_type_is_subtype_of (vala_type_reference_get_data_type (expression_type), vala_type_reference_get_data_type (expected_type));
}


static void vala_semantic_analyzer_real_visit_invocation_expression (ValaCodeVisitor* base, ValaInvocationExpression* expr) {
	ValaSemanticAnalyzer * self;
	ValaSymbol* _tmp3;
	ValaSymbol* msym;
	GeeCollection* params;
	GeeCollection* args;
	GeeIterator* arg_it;
	ValaTypeReference* ret_type;
	GeeCollection* _tmp13;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_INVOCATION_EXPRESSION (expr));
	vala_code_node_accept (VALA_CODE_NODE (vala_invocation_expression_get_call (expr)), VALA_CODE_VISITOR (self));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_invocation_expression_get_call (expr)))) {
		/* if method resolving didn't succeed, skip this check */
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		return;
	}
	if (VALA_IS_MEMBER_ACCESS (vala_invocation_expression_get_call (expr))) {
		ValaMemberAccess* _tmp0;
		ValaMemberAccess* ma;
		_tmp0 = NULL;
		ma = (_tmp0 = VALA_MEMBER_ACCESS (vala_invocation_expression_get_call (expr)), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
		if (vala_member_access_get_prototype_access (ma)) {
			char* _tmp2;
			char* _tmp1;
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			_tmp2 = NULL;
			_tmp1 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp2 = g_strdup_printf ("Access to instance member `%s' denied", (_tmp1 = vala_symbol_get_full_name (vala_expression_get_symbol_reference (vala_invocation_expression_get_call (expr)))))));
			(_tmp2 = (g_free (_tmp2), NULL));
			(_tmp1 = (g_free (_tmp1), NULL));
			(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
			return;
		}
		(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
	}
	_tmp3 = NULL;
	msym = (_tmp3 = vala_expression_get_symbol_reference (vala_invocation_expression_get_call (expr)), (_tmp3 == NULL ? NULL : g_object_ref (_tmp3)));
	if (msym == NULL) {
		/* if no symbol found, skip this check */
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
		return;
	}
	params = NULL;
	if (VALA_IS_INVOKABLE (msym)) {
		ValaInvokable* _tmp4;
		ValaInvokable* m;
		_tmp4 = NULL;
		m = (_tmp4 = VALA_INVOKABLE (msym), (_tmp4 == NULL ? NULL : g_object_ref (_tmp4)));
		if (vala_invokable_is_invokable (m)) {
			GeeCollection* _tmp5;
			_tmp5 = NULL;
			params = (_tmp5 = vala_invokable_get_parameters (m), (params == NULL ? NULL : (params = (g_object_unref (params), NULL))), _tmp5);
		} else {
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "invocation not supported in this context");
			(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
			(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
			(params == NULL ? NULL : (params = (g_object_unref (params), NULL)));
			return;
		}
		(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
	} else {
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "invocation not supported in this context");
		(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
		(params == NULL ? NULL : (params = (g_object_unref (params), NULL)));
		return;
	}
	args = vala_invocation_expression_get_argument_list (expr);
	arg_it = gee_iterable_iterator (GEE_ITERABLE (args));
	{
		GeeCollection* param_collection;
		GeeIterator* param_it;
		param_collection = params;
		param_it = gee_iterable_iterator (GEE_ITERABLE (param_collection));
		while (gee_iterator_next (param_it)) {
			ValaFormalParameter* param;
			param = gee_iterator_get (param_it);
			{
				if (vala_formal_parameter_get_ellipsis (param)) {
					(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
					break;
				}
				if (gee_iterator_next (arg_it)) {
					ValaExpression* arg;
					arg = gee_iterator_get (arg_it);
					/* store expected type for callback parameters */
					vala_expression_set_expected_type (arg, vala_formal_parameter_get_type_reference (param));
					(arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL)));
				}
				(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
			}
		}
		(param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL)));
	}
	{
		GeeCollection* arg_collection;
		GeeIterator* arg_it;
		arg_collection = vala_invocation_expression_get_argument_list (expr);
		arg_it = gee_iterable_iterator (GEE_ITERABLE (arg_collection));
		while (gee_iterator_next (arg_it)) {
			ValaExpression* arg;
			arg = gee_iterator_get (arg_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (arg), VALA_CODE_VISITOR (self));
				(arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL)));
			}
		}
		(arg_collection == NULL ? NULL : (arg_collection = (g_object_unref (arg_collection), NULL)));
		(arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL)));
	}
	ret_type = NULL;
	if (VALA_IS_INVOKABLE (msym)) {
		ValaInvokable* _tmp6;
		ValaInvokable* m;
		ValaTypeReference* _tmp7;
		GeeCollection* _tmp8;
		_tmp6 = NULL;
		m = (_tmp6 = VALA_INVOKABLE (msym), (_tmp6 == NULL ? NULL : g_object_ref (_tmp6)));
		_tmp7 = NULL;
		ret_type = (_tmp7 = vala_invokable_get_return_type (m), (ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL))), _tmp7);
		_tmp8 = NULL;
		params = (_tmp8 = vala_invokable_get_parameters (m), (params == NULL ? NULL : (params = (g_object_unref (params), NULL))), _tmp8);
		if (vala_type_reference_get_data_type (ret_type) == NULL && vala_type_reference_get_type_parameter (ret_type) == NULL) {
			/* void return type*/
			if (!(VALA_IS_EXPRESSION_STATEMENT (vala_code_node_get_parent_node (VALA_CODE_NODE (expr))))) {
				vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "invocation of void method not allowed as expression");
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
				(params == NULL ? NULL : (params = (g_object_unref (params), NULL)));
				(args == NULL ? NULL : (args = (g_object_unref (args), NULL)));
				(arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL)));
				(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
				return;
			}
		}
		/* resolve generic return values*/
		if (vala_type_reference_get_type_parameter (ret_type) != NULL) {
			ValaMemberAccess* _tmp9;
			ValaMemberAccess* ma;
			if (!(VALA_IS_MEMBER_ACCESS (vala_invocation_expression_get_call (expr)))) {
				vala_report_error (vala_code_node_get_source_reference ((VALA_CODE_NODE (m))), "internal error: unsupported generic return value");
				vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
				(params == NULL ? NULL : (params = (g_object_unref (params), NULL)));
				(args == NULL ? NULL : (args = (g_object_unref (args), NULL)));
				(arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL)));
				(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
				return;
			}
			_tmp9 = NULL;
			ma = (_tmp9 = VALA_MEMBER_ACCESS (vala_invocation_expression_get_call (expr)), (_tmp9 == NULL ? NULL : g_object_ref (_tmp9)));
			if (vala_member_access_get_inner (ma) == NULL) {
				/* TODO resolve generic return values within the type hierarchy if possible*/
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "internal error: resolving generic return values within type hierarchy not supported yet");
				vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
				(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
				(params == NULL ? NULL : (params = (g_object_unref (params), NULL)));
				(args == NULL ? NULL : (args = (g_object_unref (args), NULL)));
				(arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL)));
				(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
				return;
			} else {
				ValaTypeReference* _tmp10;
				_tmp10 = NULL;
				ret_type = (_tmp10 = vala_semantic_analyzer_get_actual_type (vala_expression_get_static_type (vala_member_access_get_inner (ma)), msym, ret_type, VALA_CODE_NODE (expr)), (ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL))), _tmp10);
				if (ret_type == NULL) {
					(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
					(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
					(params == NULL ? NULL : (params = (g_object_unref (params), NULL)));
					(args == NULL ? NULL : (args = (g_object_unref (args), NULL)));
					(arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL)));
					(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
					return;
				}
			}
			(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
		}
		(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
	}
	if (VALA_IS_METHOD (msym)) {
		ValaMethod* _tmp11;
		ValaMethod* m;
		GeeCollection* _tmp12;
		_tmp11 = NULL;
		m = (_tmp11 = VALA_METHOD (msym), (_tmp11 == NULL ? NULL : g_object_ref (_tmp11)));
		_tmp12 = NULL;
		vala_code_node_set_tree_can_fail (VALA_CODE_NODE (expr), (vala_expression_set_can_fail (VALA_EXPRESSION (expr), (gee_collection_get_size ((_tmp12 = vala_method_get_error_domains (m))) > 0)), vala_expression_get_can_fail (VALA_EXPRESSION (expr))));
		(_tmp12 == NULL ? NULL : (_tmp12 = (g_object_unref (_tmp12), NULL)));
		(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
	}
	vala_expression_set_static_type (VALA_EXPRESSION (expr), ret_type);
	_tmp13 = NULL;
	vala_semantic_analyzer_check_arguments (self, VALA_EXPRESSION (expr), msym, params, (_tmp13 = vala_invocation_expression_get_argument_list (expr)));
	(_tmp13 == NULL ? NULL : (_tmp13 = (g_object_unref (_tmp13), NULL)));
	(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
	(params == NULL ? NULL : (params = (g_object_unref (params), NULL)));
	(args == NULL ? NULL : (args = (g_object_unref (args), NULL)));
	(arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL)));
	(ret_type == NULL ? NULL : (ret_type = (g_object_unref (ret_type), NULL)));
}


static gboolean vala_semantic_analyzer_check_arguments (ValaSemanticAnalyzer* self, ValaExpression* expr, ValaSymbol* msym, GeeCollection* params, GeeCollection* args) {
	ValaExpression* prev_arg;
	GeeIterator* arg_it;
	ValaAttribute* _tmp0;
	gboolean _tmp1;
	gboolean diag;
	gboolean ellipsis;
	gint i;
	gboolean _tmp23;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	g_return_val_if_fail (VALA_IS_EXPRESSION (expr), FALSE);
	g_return_val_if_fail (VALA_IS_SYMBOL (msym), FALSE);
	g_return_val_if_fail (params == NULL || GEE_IS_COLLECTION (params), FALSE);
	g_return_val_if_fail (args == NULL || GEE_IS_COLLECTION (args), FALSE);
	prev_arg = NULL;
	arg_it = gee_iterable_iterator (GEE_ITERABLE (args));
	_tmp0 = NULL;
	diag = (_tmp1 = ((_tmp0 = vala_code_node_get_attribute (VALA_CODE_NODE (msym), "Diagnostics")) != NULL), (_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL))), _tmp1);
	ellipsis = FALSE;
	i = 0;
	{
		GeeCollection* param_collection;
		GeeIterator* param_it;
		param_collection = params;
		param_it = gee_iterable_iterator (GEE_ITERABLE (param_collection));
		while (gee_iterator_next (param_it)) {
			ValaFormalParameter* param;
			param = gee_iterator_get (param_it);
			{
				if (vala_formal_parameter_get_ellipsis (param)) {
					ellipsis = TRUE;
					(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
					break;
				}
				/* header file necessary if we need to cast argument */
				if (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (param)) != NULL) {
					vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (param))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
				}
				if (!gee_iterator_next (arg_it)) {
					if (vala_formal_parameter_get_default_expression (param) == NULL) {
						char* _tmp3;
						char* _tmp2;
						gboolean _tmp4;
						vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
						_tmp3 = NULL;
						_tmp2 = NULL;
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp3 = g_strdup_printf ("Too few arguments, method `%s' does not take %d arguments", (_tmp2 = vala_symbol_get_full_name (msym)), gee_collection_get_size (args))));
						(_tmp3 = (g_free (_tmp3), NULL));
						(_tmp2 = (g_free (_tmp2), NULL));
						return (_tmp4 = FALSE, (param == NULL ? NULL : (param = (g_object_unref (param), NULL))), (param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL))), (prev_arg == NULL ? NULL : (prev_arg = (g_object_unref (prev_arg), NULL))), (arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL))), _tmp4);
					}
				} else {
					ValaExpression* arg;
					ValaExpression* _tmp13;
					ValaExpression* _tmp12;
					arg = gee_iterator_get (arg_it);
					if (vala_code_node_get_error (VALA_CODE_NODE (arg))) {
						gboolean _tmp5;
						/* ignore inner error*/
						vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
						return (_tmp5 = FALSE, (arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL))), (param == NULL ? NULL : (param = (g_object_unref (param), NULL))), (param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL))), (prev_arg == NULL ? NULL : (prev_arg = (g_object_unref (prev_arg), NULL))), (arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL))), _tmp5);
					} else {
						if (vala_expression_get_static_type (arg) == NULL) {
							/* disallow untyped arguments except for type inference of callbacks*/
							if (!(VALA_IS_CALLBACK (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (param)))) || !(VALA_IS_METHOD (vala_expression_get_symbol_reference (arg)))) {
								char* _tmp6;
								gboolean _tmp7;
								vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
								_tmp6 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp6 = g_strdup_printf ("Invalid type for argument %d", i + 1)));
								(_tmp6 = (g_free (_tmp6), NULL));
								return (_tmp7 = FALSE, (arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL))), (param == NULL ? NULL : (param = (g_object_unref (param), NULL))), (param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL))), (prev_arg == NULL ? NULL : (prev_arg = (g_object_unref (prev_arg), NULL))), (arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL))), _tmp7);
							}
						} else {
							if (!vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (arg), vala_formal_parameter_get_type_reference (param))) {
								char* _tmp10;
								char* _tmp9;
								char* _tmp8;
								gboolean _tmp11;
								vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
								_tmp10 = NULL;
								_tmp9 = NULL;
								_tmp8 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp10 = g_strdup_printf ("Argument %d: Cannot convert from `%s' to `%s'", i + 1, (_tmp8 = vala_type_reference_to_string (vala_expression_get_static_type (arg))), (_tmp9 = vala_type_reference_to_string (vala_formal_parameter_get_type_reference (param))))));
								(_tmp10 = (g_free (_tmp10), NULL));
								(_tmp9 = (g_free (_tmp9), NULL));
								(_tmp8 = (g_free (_tmp8), NULL));
								return (_tmp11 = FALSE, (arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL))), (param == NULL ? NULL : (param = (g_object_unref (param), NULL))), (param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL))), (prev_arg == NULL ? NULL : (prev_arg = (g_object_unref (prev_arg), NULL))), (arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL))), _tmp11);
							}
						}
					}
					_tmp13 = NULL;
					_tmp12 = NULL;
					prev_arg = (_tmp13 = (_tmp12 = arg, (_tmp12 == NULL ? NULL : g_object_ref (_tmp12))), (prev_arg == NULL ? NULL : (prev_arg = (g_object_unref (prev_arg), NULL))), _tmp13);
					i++;
					(arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL)));
				}
				(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
			}
		}
		(param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL)));
	}
	if (ellipsis) {
		while (gee_iterator_next (arg_it)) {
			ValaExpression* arg;
			arg = gee_iterator_get (arg_it);
			if (vala_code_node_get_error (VALA_CODE_NODE (arg))) {
				gboolean _tmp14;
				/* ignore inner error*/
				vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
				return (_tmp14 = FALSE, (arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL))), (prev_arg == NULL ? NULL : (prev_arg = (g_object_unref (prev_arg), NULL))), (arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL))), _tmp14);
			} else {
				if (vala_expression_get_static_type (arg) == NULL) {
					/* disallow untyped arguments except for type inference of callbacks*/
					if (!(VALA_IS_METHOD (vala_expression_get_symbol_reference (arg)))) {
						char* _tmp15;
						gboolean _tmp16;
						vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
						_tmp15 = NULL;
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp15 = g_strdup_printf ("Invalid type for argument %d", i + 1)));
						(_tmp15 = (g_free (_tmp15), NULL));
						return (_tmp16 = FALSE, (arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL))), (prev_arg == NULL ? NULL : (prev_arg = (g_object_unref (prev_arg), NULL))), (arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL))), _tmp16);
					}
				}
			}
			i++;
			(arg == NULL ? NULL : (arg = (g_object_unref (arg), NULL)));
		}
	} else {
		if (!ellipsis && gee_iterator_next (arg_it)) {
			char* _tmp18;
			char* _tmp17;
			gboolean _tmp19;
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			_tmp18 = NULL;
			_tmp17 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp18 = g_strdup_printf ("Too many arguments, method `%s' does not take %d arguments", (_tmp17 = vala_symbol_get_full_name (msym)), gee_collection_get_size (args))));
			(_tmp18 = (g_free (_tmp18), NULL));
			(_tmp17 = (g_free (_tmp17), NULL));
			return (_tmp19 = FALSE, (prev_arg == NULL ? NULL : (prev_arg = (g_object_unref (prev_arg), NULL))), (arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL))), _tmp19);
		}
	}
	if (diag && prev_arg != NULL) {
		ValaExpression* _tmp20;
		ValaExpression* format_arg;
		_tmp20 = NULL;
		format_arg = (_tmp20 = prev_arg, (_tmp20 == NULL ? NULL : g_object_ref (_tmp20)));
		if (VALA_IS_LITERAL_EXPRESSION (format_arg)) {
			ValaStringLiteral* _tmp21;
			ValaStringLiteral* format_lit;
			char* _tmp22;
			_tmp21 = NULL;
			format_lit = (_tmp21 = VALA_STRING_LITERAL (vala_literal_expression_get_literal ((VALA_LITERAL_EXPRESSION (format_arg)))), (_tmp21 == NULL ? NULL : g_object_ref (_tmp21)));
			_tmp22 = NULL;
			vala_string_literal_set_value (format_lit, (_tmp22 = g_strdup_printf ("\"%s:%d: %s", vala_source_file_get_filename (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)))), vala_source_reference_get_first_line (vala_code_node_get_source_reference (VALA_CODE_NODE (expr))), g_utf8_offset_to_pointer (vala_string_literal_get_value (format_lit), 1))));
			(_tmp22 = (g_free (_tmp22), NULL));
			(format_lit == NULL ? NULL : (format_lit = (g_object_unref (format_lit), NULL)));
		}
		(format_arg == NULL ? NULL : (format_arg = (g_object_unref (format_arg), NULL)));
	}
	return (_tmp23 = TRUE, (prev_arg == NULL ? NULL : (prev_arg = (g_object_unref (prev_arg), NULL))), (arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL))), _tmp23);
	(prev_arg == NULL ? NULL : (prev_arg = (g_object_unref (prev_arg), NULL)));
	(arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL)));
}


ValaTypeReference* vala_semantic_analyzer_get_actual_type (ValaTypeReference* derived_instance_type, ValaSymbol* generic_member, ValaTypeReference* generic_type, ValaCodeNode* node_reference) {
	ValaTypeReference* _tmp0;
	ValaTypeReference* instance_type;
	gint param_index;
	ValaTypeReference* actual_type;
	GeeList* _tmp17;
	gboolean _tmp18;
	ValaTypeReference* _tmp23;
	ValaTypeReference* _tmp24;
	g_return_val_if_fail (derived_instance_type == NULL || VALA_IS_TYPE_REFERENCE (derived_instance_type), NULL);
	g_return_val_if_fail (generic_member == NULL || VALA_IS_SYMBOL (generic_member), NULL);
	g_return_val_if_fail (generic_type == NULL || VALA_IS_TYPE_REFERENCE (generic_type), NULL);
	g_return_val_if_fail (node_reference == NULL || VALA_IS_CODE_NODE (node_reference), NULL);
	_tmp0 = NULL;
	instance_type = (_tmp0 = derived_instance_type, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
	while (VALA_SYMBOL (vala_type_reference_get_data_type (instance_type)) != vala_symbol_get_parent_symbol (generic_member)) {
		GeeCollection* base_types;
		/* trace type arguments back to the datatype where the method has been declared*/
		base_types = NULL;
		if (VALA_IS_CLASS (vala_type_reference_get_data_type (instance_type))) {
			ValaClass* _tmp1;
			ValaClass* cl;
			GeeCollection* _tmp2;
			_tmp1 = NULL;
			cl = (_tmp1 = VALA_CLASS (vala_type_reference_get_data_type (instance_type)), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1)));
			_tmp2 = NULL;
			base_types = (_tmp2 = vala_class_get_base_types (cl), (base_types == NULL ? NULL : (base_types = (g_object_unref (base_types), NULL))), _tmp2);
			(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
		} else {
			if (VALA_IS_INTERFACE (vala_type_reference_get_data_type (instance_type))) {
				ValaInterface* _tmp3;
				ValaInterface* iface;
				GeeCollection* _tmp4;
				_tmp3 = NULL;
				iface = (_tmp3 = VALA_INTERFACE (vala_type_reference_get_data_type (instance_type)), (_tmp3 == NULL ? NULL : g_object_ref (_tmp3)));
				_tmp4 = NULL;
				base_types = (_tmp4 = vala_interface_get_prerequisites (iface), (base_types == NULL ? NULL : (base_types = (g_object_unref (base_types), NULL))), _tmp4);
				(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
			} else {
				gpointer _tmp5;
				vala_report_error (vala_code_node_get_source_reference (node_reference), "internal error: unsupported generic type");
				vala_code_node_set_error (node_reference, TRUE);
				return (_tmp5 = NULL, (base_types == NULL ? NULL : (base_types = (g_object_unref (base_types), NULL))), (instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL))), _tmp5);
			}
		}
		{
			GeeCollection* base_type_collection;
			GeeIterator* base_type_it;
			base_type_collection = base_types;
			base_type_it = gee_iterable_iterator (GEE_ITERABLE (base_type_collection));
			while (gee_iterator_next (base_type_it)) {
				ValaTypeReference* base_type;
				base_type = gee_iterator_get (base_type_it);
				{
					ValaSymbol* _tmp6;
					gboolean _tmp7;
					_tmp6 = NULL;
					if ((_tmp7 = (_tmp6 = vala_semantic_analyzer_symbol_lookup_inherited (VALA_SYMBOL (vala_type_reference_get_data_type (base_type)), vala_symbol_get_name (generic_member))) != NULL, (_tmp6 == NULL ? NULL : (_tmp6 = (g_object_unref (_tmp6), NULL))), _tmp7)) {
						ValaTypeReference* instance_base_type;
						ValaTypeReference* _tmp13;
						ValaTypeReference* _tmp12;
						/* construct a new type reference for the base type with correctly linked type arguments*/
						instance_base_type = vala_type_reference_new ();
						vala_type_reference_set_data_type (instance_base_type, vala_type_reference_get_data_type (base_type));
						{
							GeeList* type_arg_collection;
							GeeIterator* type_arg_it;
							type_arg_collection = vala_type_reference_get_type_arguments (base_type);
							type_arg_it = gee_iterable_iterator (GEE_ITERABLE (type_arg_collection));
							while (gee_iterator_next (type_arg_it)) {
								ValaTypeReference* type_arg;
								type_arg = gee_iterator_get (type_arg_it);
								{
									if (vala_type_reference_get_type_parameter (type_arg) != NULL) {
										gint param_index;
										ValaTypeReference* _tmp11;
										GeeList* _tmp10;
										/* link to type argument of derived type*/
										param_index = vala_data_type_get_type_parameter_index (vala_type_reference_get_data_type (instance_type), vala_symbol_get_name (VALA_SYMBOL (vala_type_reference_get_type_parameter (type_arg))));
										if (param_index == -1) {
											char* _tmp8;
											gpointer _tmp9;
											_tmp8 = NULL;
											vala_report_error (vala_code_node_get_source_reference (node_reference), (_tmp8 = g_strdup_printf ("internal error: unknown type parameter %s", vala_symbol_get_name (VALA_SYMBOL (vala_type_reference_get_type_parameter (type_arg))))));
											(_tmp8 = (g_free (_tmp8), NULL));
											vala_code_node_set_error (node_reference, TRUE);
											return (_tmp9 = NULL, (type_arg == NULL ? NULL : (type_arg = (g_object_unref (type_arg), NULL))), (type_arg_collection == NULL ? NULL : (type_arg_collection = (g_object_unref (type_arg_collection), NULL))), (type_arg_it == NULL ? NULL : (type_arg_it = (g_object_unref (type_arg_it), NULL))), (instance_base_type == NULL ? NULL : (instance_base_type = (g_object_unref (instance_base_type), NULL))), (base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL))), (base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL))), (base_types == NULL ? NULL : (base_types = (g_object_unref (base_types), NULL))), (instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL))), _tmp9);
										}
										_tmp11 = NULL;
										_tmp10 = NULL;
										type_arg = (_tmp11 = gee_list_get ((_tmp10 = vala_type_reference_get_type_arguments (instance_type)), param_index), (type_arg == NULL ? NULL : (type_arg = (g_object_unref (type_arg), NULL))), _tmp11);
										(_tmp10 == NULL ? NULL : (_tmp10 = (g_object_unref (_tmp10), NULL)));
									}
									vala_type_reference_add_type_argument (instance_base_type, type_arg);
									(type_arg == NULL ? NULL : (type_arg = (g_object_unref (type_arg), NULL)));
								}
							}
							(type_arg_collection == NULL ? NULL : (type_arg_collection = (g_object_unref (type_arg_collection), NULL)));
							(type_arg_it == NULL ? NULL : (type_arg_it = (g_object_unref (type_arg_it), NULL)));
						}
						_tmp13 = NULL;
						_tmp12 = NULL;
						instance_type = (_tmp13 = (_tmp12 = instance_base_type, (_tmp12 == NULL ? NULL : g_object_ref (_tmp12))), (instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL))), _tmp13);
						(instance_base_type == NULL ? NULL : (instance_base_type = (g_object_unref (instance_base_type), NULL)));
					}
					(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
				}
			}
			(base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL)));
		}
		(base_types == NULL ? NULL : (base_types = (g_object_unref (base_types), NULL)));
	}
	if (VALA_SYMBOL (vala_type_reference_get_data_type (instance_type)) != vala_symbol_get_parent_symbol (generic_member)) {
		gpointer _tmp14;
		vala_report_error (vala_code_node_get_source_reference (node_reference), "internal error: generic type parameter tracing not supported yet");
		vala_code_node_set_error (node_reference, TRUE);
		return (_tmp14 = NULL, (instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL))), _tmp14);
	}
	param_index = vala_data_type_get_type_parameter_index (vala_type_reference_get_data_type (instance_type), vala_symbol_get_name (VALA_SYMBOL (vala_type_reference_get_type_parameter (generic_type))));
	if (param_index == -1) {
		char* _tmp15;
		gpointer _tmp16;
		_tmp15 = NULL;
		vala_report_error (vala_code_node_get_source_reference (node_reference), (_tmp15 = g_strdup_printf ("internal error: unknown type parameter %s", vala_symbol_get_name (VALA_SYMBOL (vala_type_reference_get_type_parameter (generic_type))))));
		(_tmp15 = (g_free (_tmp15), NULL));
		vala_code_node_set_error (node_reference, TRUE);
		return (_tmp16 = NULL, (instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL))), _tmp16);
	}
	actual_type = NULL;
	_tmp17 = NULL;
	if ((_tmp18 = param_index < gee_collection_get_size (GEE_COLLECTION ((_tmp17 = vala_type_reference_get_type_arguments (instance_type)))), (_tmp17 == NULL ? NULL : (_tmp17 = (g_object_unref (_tmp17), NULL))), _tmp18)) {
		ValaTypeReference* _tmp20;
		GeeList* _tmp19;
		_tmp20 = NULL;
		_tmp19 = NULL;
		actual_type = (_tmp20 = VALA_TYPE_REFERENCE (gee_list_get ((_tmp19 = vala_type_reference_get_type_arguments (instance_type)), param_index)), (actual_type == NULL ? NULL : (actual_type = (g_object_unref (actual_type), NULL))), _tmp20);
		(_tmp19 == NULL ? NULL : (_tmp19 = (g_object_unref (_tmp19), NULL)));
	}
	if (actual_type == NULL) {
		char* _tmp21;
		gpointer _tmp22;
		_tmp21 = NULL;
		vala_report_error (vala_code_node_get_source_reference (node_reference), (_tmp21 = g_strdup_printf ("internal error: no actual argument found for type parameter %s", vala_symbol_get_name (VALA_SYMBOL (vala_type_reference_get_type_parameter (generic_type))))));
		(_tmp21 = (g_free (_tmp21), NULL));
		vala_code_node_set_error (node_reference, TRUE);
		return (_tmp22 = NULL, (instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL))), (actual_type == NULL ? NULL : (actual_type = (g_object_unref (actual_type), NULL))), _tmp22);
	}
	_tmp23 = NULL;
	actual_type = (_tmp23 = vala_type_reference_copy (actual_type), (actual_type == NULL ? NULL : (actual_type = (g_object_unref (actual_type), NULL))), _tmp23);
	vala_type_reference_set_transfers_ownership (actual_type, vala_type_reference_get_takes_ownership (actual_type) && vala_type_reference_get_transfers_ownership (generic_type));
	vala_type_reference_set_takes_ownership (actual_type, vala_type_reference_get_takes_ownership (actual_type) && vala_type_reference_get_takes_ownership (generic_type));
	_tmp24 = NULL;
	return (_tmp24 = actual_type, (instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL))), _tmp24);
	(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
	(actual_type == NULL ? NULL : (actual_type = (g_object_unref (actual_type), NULL)));
}


static void vala_semantic_analyzer_real_visit_element_access (ValaCodeVisitor* base, ValaElementAccess* expr) {
	ValaSemanticAnalyzer * self;
	ValaDataType* _tmp0;
	ValaDataType* container_type;
	gboolean index_int_type_check;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_ELEMENT_ACCESS (expr));
	if (vala_expression_get_static_type (vala_element_access_get_container (expr)) == NULL) {
		/* don't proceed if a child expression failed */
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		return;
	}
	_tmp0 = NULL;
	container_type = (_tmp0 = vala_type_reference_get_data_type (vala_expression_get_static_type (vala_element_access_get_container (expr))), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
	index_int_type_check = TRUE;
	/* assign a static_type when possible */
	if (VALA_IS_ARRAY (container_type)) {
		GeeList* args;
		ValaTypeReference* _tmp2;
		args = vala_type_reference_get_type_arguments (vala_expression_get_static_type (vala_element_access_get_container (expr)));
		if (gee_collection_get_size (GEE_COLLECTION (args)) != 1) {
			char* _tmp1;
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			_tmp1 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp1 = g_strdup_printf ("internal error: array reference with %d type arguments, expected 1", gee_collection_get_size (GEE_COLLECTION (args)))));
			(_tmp1 = (g_free (_tmp1), NULL));
			(args == NULL ? NULL : (args = (g_object_unref (args), NULL)));
			(container_type == NULL ? NULL : (container_type = (g_object_unref (container_type), NULL)));
			return;
		}
		_tmp2 = NULL;
		vala_expression_set_static_type (VALA_EXPRESSION (expr), (_tmp2 = gee_list_get (args, 0)));
		(_tmp2 == NULL ? NULL : (_tmp2 = (g_object_unref (_tmp2), NULL)));
		(args == NULL ? NULL : (args = (g_object_unref (args), NULL)));
	} else {
		if (container_type == vala_type_reference_get_data_type (self->priv->string_type)) {
			GeeList* _tmp3;
			gboolean _tmp4;
			_tmp3 = NULL;
			if ((_tmp4 = gee_collection_get_size (GEE_COLLECTION ((_tmp3 = vala_element_access_get_indices (expr)))) != 1, (_tmp3 == NULL ? NULL : (_tmp3 = (g_object_unref (_tmp3), NULL))), _tmp4)) {
				vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Element access with more than one dimension is not supported for strings");
				(container_type == NULL ? NULL : (container_type = (g_object_unref (container_type), NULL)));
				return;
			}
			vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->unichar_type);
		} else {
			if (container_type != NULL && self->priv->list_type != NULL && self->priv->map_type != NULL && (vala_data_type_is_subtype_of (container_type, self->priv->list_type) || vala_data_type_is_subtype_of (container_type, self->priv->map_type))) {
				GeeCollection* indices;
				GeeIterator* indices_it;
				ValaExpression* index;
				ValaSymbol* get_sym;
				ValaMethod* _tmp5;
				ValaMethod* get_method;
				GeeCollection* get_params;
				GeeIterator* get_params_it;
				ValaFormalParameter* get_param;
				ValaTypeReference* _tmp6;
				ValaTypeReference* index_type;
				ValaTypeReference* _tmp12;
				ValaTypeReference* _tmp11;
				indices = GEE_COLLECTION (vala_element_access_get_indices (expr));
				if (gee_collection_get_size (indices) != 1) {
					vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Element access with more than one dimension is not supported for the specified type");
					(indices == NULL ? NULL : (indices = (g_object_unref (indices), NULL)));
					(container_type == NULL ? NULL : (container_type = (g_object_unref (container_type), NULL)));
					return;
				}
				indices_it = gee_iterable_iterator (GEE_ITERABLE (indices));
				gee_iterator_next (indices_it);
				index = gee_iterator_get (indices_it);
				index_int_type_check = FALSE;
				get_sym = vala_scope_lookup (vala_symbol_get_scope (VALA_SYMBOL (container_type)), "get");
				if (!(VALA_IS_METHOD (get_sym))) {
					vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "invalid get method in specified collection type");
					(indices == NULL ? NULL : (indices = (g_object_unref (indices), NULL)));
					(indices_it == NULL ? NULL : (indices_it = (g_object_unref (indices_it), NULL)));
					(index == NULL ? NULL : (index = (g_object_unref (index), NULL)));
					(get_sym == NULL ? NULL : (get_sym = (g_object_unref (get_sym), NULL)));
					(container_type == NULL ? NULL : (container_type = (g_object_unref (container_type), NULL)));
					return;
				}
				_tmp5 = NULL;
				get_method = (_tmp5 = VALA_METHOD (get_sym), (_tmp5 == NULL ? NULL : g_object_ref (_tmp5)));
				get_params = vala_invokable_get_parameters (VALA_INVOKABLE (get_method));
				get_params_it = gee_iterable_iterator (GEE_ITERABLE (get_params));
				gee_iterator_next (get_params_it);
				get_param = gee_iterator_get (get_params_it);
				_tmp6 = NULL;
				index_type = (_tmp6 = vala_formal_parameter_get_type_reference (get_param), (_tmp6 == NULL ? NULL : g_object_ref (_tmp6)));
				if (vala_type_reference_get_type_parameter (index_type) != NULL) {
					ValaTypeReference* _tmp7;
					_tmp7 = NULL;
					index_type = (_tmp7 = vala_semantic_analyzer_get_actual_type (vala_expression_get_static_type (vala_element_access_get_container (expr)), VALA_SYMBOL (get_method), vala_formal_parameter_get_type_reference (get_param), VALA_CODE_NODE (expr)), (index_type == NULL ? NULL : (index_type = (g_object_unref (index_type), NULL))), _tmp7);
				}
				if (!vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (index), index_type)) {
					char* _tmp10;
					char* _tmp9;
					char* _tmp8;
					vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
					_tmp10 = NULL;
					_tmp9 = NULL;
					_tmp8 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp10 = g_strdup_printf ("index expression: Cannot convert from `%s' to `%s'", (_tmp8 = vala_type_reference_to_string (vala_expression_get_static_type (index))), (_tmp9 = vala_type_reference_to_string (index_type)))));
					(_tmp10 = (g_free (_tmp10), NULL));
					(_tmp9 = (g_free (_tmp9), NULL));
					(_tmp8 = (g_free (_tmp8), NULL));
					(indices == NULL ? NULL : (indices = (g_object_unref (indices), NULL)));
					(indices_it == NULL ? NULL : (indices_it = (g_object_unref (indices_it), NULL)));
					(index == NULL ? NULL : (index = (g_object_unref (index), NULL)));
					(get_sym == NULL ? NULL : (get_sym = (g_object_unref (get_sym), NULL)));
					(get_method == NULL ? NULL : (get_method = (g_object_unref (get_method), NULL)));
					(get_params == NULL ? NULL : (get_params = (g_object_unref (get_params), NULL)));
					(get_params_it == NULL ? NULL : (get_params_it = (g_object_unref (get_params_it), NULL)));
					(get_param == NULL ? NULL : (get_param = (g_object_unref (get_param), NULL)));
					(index_type == NULL ? NULL : (index_type = (g_object_unref (index_type), NULL)));
					(container_type == NULL ? NULL : (container_type = (g_object_unref (container_type), NULL)));
					return;
				}
				_tmp12 = NULL;
				_tmp11 = NULL;
				vala_expression_set_static_type (VALA_EXPRESSION (expr), (_tmp12 = vala_type_reference_copy ((_tmp11 = vala_semantic_analyzer_get_actual_type (vala_expression_get_static_type (vala_element_access_get_container (expr)), VALA_SYMBOL (get_method), vala_method_get_return_type (get_method), VALA_CODE_NODE (expr))))));
				(_tmp12 = (g_object_unref (_tmp12), NULL));
				(_tmp11 == NULL ? NULL : (_tmp11 = (g_object_unref (_tmp11), NULL)));
				vala_type_reference_set_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), FALSE);
				(indices == NULL ? NULL : (indices = (g_object_unref (indices), NULL)));
				(indices_it == NULL ? NULL : (indices_it = (g_object_unref (indices_it), NULL)));
				(index == NULL ? NULL : (index = (g_object_unref (index), NULL)));
				(get_sym == NULL ? NULL : (get_sym = (g_object_unref (get_sym), NULL)));
				(get_method == NULL ? NULL : (get_method = (g_object_unref (get_method), NULL)));
				(get_params == NULL ? NULL : (get_params = (g_object_unref (get_params), NULL)));
				(get_params_it == NULL ? NULL : (get_params_it = (g_object_unref (get_params_it), NULL)));
				(get_param == NULL ? NULL : (get_param = (g_object_unref (get_param), NULL)));
				(index_type == NULL ? NULL : (index_type = (g_object_unref (index_type), NULL)));
			} else {
				char* _tmp14;
				char* _tmp13;
				vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
				_tmp14 = NULL;
				_tmp13 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp14 = g_strdup_printf ("The expression `%s' does not denote an Array", (_tmp13 = vala_type_reference_to_string (vala_expression_get_static_type (vala_element_access_get_container (expr)))))));
				(_tmp14 = (g_free (_tmp14), NULL));
				(_tmp13 = (g_free (_tmp13), NULL));
			}
		}
	}
	if (index_int_type_check) {
		{
			GeeList* e_collection;
			GeeIterator* e_it;
			e_collection = vala_element_access_get_indices (expr);
			e_it = gee_iterable_iterator (GEE_ITERABLE (e_collection));
			while (gee_iterator_next (e_it)) {
				ValaExpression* e;
				e = gee_iterator_get (e_it);
				{
					/* check if the index is of type integer 
					 don't proceed if a child expression failed */
					if (vala_expression_get_static_type (e) == NULL) {
						(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
						(e_collection == NULL ? NULL : (e_collection = (g_object_unref (e_collection), NULL)));
						(e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL)));
						(container_type == NULL ? NULL : (container_type = (g_object_unref (container_type), NULL)));
						return;
					}
					/* check if the index is of type integer */
					if (!(VALA_IS_STRUCT (vala_type_reference_get_data_type (vala_expression_get_static_type (e)))) || !vala_struct_is_integer_type ((VALA_STRUCT (vala_type_reference_get_data_type (vala_expression_get_static_type (e)))))) {
						vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (e)), "Expression of integer type expected");
					}
					(e == NULL ? NULL : (e = (g_object_unref (e), NULL)));
				}
			}
			(e_collection == NULL ? NULL : (e_collection = (g_object_unref (e_collection), NULL)));
			(e_it == NULL ? NULL : (e_it = (g_object_unref (e_it), NULL)));
		}
	}
	(container_type == NULL ? NULL : (container_type = (g_object_unref (container_type), NULL)));
}


static void vala_semantic_analyzer_real_visit_base_access (ValaCodeVisitor* base, ValaBaseAccess* expr) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_BASE_ACCESS (expr));
	if (self->priv->current_class == NULL) {
		GeeCollection* _tmp4;
		GeeIterator* _tmp5;
		GeeIterator* base_type_it;
		ValaTypeReference* _tmp6;
		if (self->priv->current_struct == NULL) {
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Base access invalid outside of class and struct");
			return;
		} else {
			GeeCollection* _tmp0;
			gboolean _tmp1;
			_tmp0 = NULL;
			if ((_tmp1 = gee_collection_get_size ((_tmp0 = vala_struct_get_base_types (self->priv->current_struct))) != 1, (_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL))), _tmp1)) {
				char* _tmp3;
				GeeCollection* _tmp2;
				vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
				_tmp3 = NULL;
				_tmp2 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp3 = g_strdup_printf ("Base access invalid without base type %d", gee_collection_get_size ((_tmp2 = vala_struct_get_base_types (self->priv->current_struct))))));
				(_tmp3 = (g_free (_tmp3), NULL));
				(_tmp2 == NULL ? NULL : (_tmp2 = (g_object_unref (_tmp2), NULL)));
				return;
			}
		}
		_tmp4 = NULL;
		_tmp5 = NULL;
		base_type_it = (_tmp5 = gee_iterable_iterator (GEE_ITERABLE ((_tmp4 = vala_struct_get_base_types (self->priv->current_struct)))), (_tmp4 == NULL ? NULL : (_tmp4 = (g_object_unref (_tmp4), NULL))), _tmp5);
		gee_iterator_next (base_type_it);
		_tmp6 = NULL;
		vala_expression_set_static_type (VALA_EXPRESSION (expr), (_tmp6 = gee_iterator_get (base_type_it)));
		(_tmp6 == NULL ? NULL : (_tmp6 = (g_object_unref (_tmp6), NULL)));
		(base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL)));
	} else {
		ValaTypeReference* _tmp7;
		_tmp7 = NULL;
		vala_expression_set_static_type (VALA_EXPRESSION (expr), (_tmp7 = vala_type_reference_new ()));
		(_tmp7 == NULL ? NULL : (_tmp7 = (g_object_unref (_tmp7), NULL)));
		vala_type_reference_set_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr)), VALA_DATA_TYPE (vala_class_get_base_class (self->priv->current_class)));
	}
	vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), VALA_SYMBOL (vala_type_reference_get_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr)))));
}


static void vala_semantic_analyzer_real_visit_postfix_expression (ValaCodeVisitor* base, ValaPostfixExpression* expr) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_POSTFIX_EXPRESSION (expr));
	vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_postfix_expression_get_inner (expr)));
}


static void vala_semantic_analyzer_real_visit_object_creation_expression (ValaCodeVisitor* base, ValaObjectCreationExpression* expr) {
	ValaSemanticAnalyzer * self;
	ValaDataType* type;
	ValaTypeReference* _tmp19;
	GeeCollection* _tmp26;
	gboolean _tmp27;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_OBJECT_CREATION_EXPRESSION (expr));
	vala_code_node_accept_children (VALA_CODE_NODE (expr), VALA_CODE_VISITOR (self));
	type = NULL;
	if (vala_object_creation_expression_get_type_reference (expr) == NULL) {
		ValaSymbol* _tmp0;
		ValaSymbol* constructor_sym;
		ValaSymbol* _tmp1;
		ValaSymbol* type_sym;
		GeeCollection* type_args;
		ValaTypeReference* _tmp16;
		if (vala_object_creation_expression_get_member_name (expr) == NULL) {
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Incomplete object creation expression");
			(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			return;
		}
		if (vala_expression_get_symbol_reference (VALA_EXPRESSION (vala_object_creation_expression_get_member_name (expr))) == NULL) {
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			return;
		}
		_tmp0 = NULL;
		constructor_sym = (_tmp0 = vala_expression_get_symbol_reference (VALA_EXPRESSION (vala_object_creation_expression_get_member_name (expr))), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
		_tmp1 = NULL;
		type_sym = (_tmp1 = vala_expression_get_symbol_reference (VALA_EXPRESSION (vala_object_creation_expression_get_member_name (expr))), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1)));
		type_args = vala_member_access_get_type_arguments (vala_object_creation_expression_get_member_name (expr));
		if (VALA_IS_METHOD (constructor_sym)) {
			ValaSymbol* _tmp3;
			ValaSymbol* _tmp2;
			ValaMethod* _tmp4;
			ValaMethod* constructor;
			GeeCollection* _tmp7;
			_tmp3 = NULL;
			_tmp2 = NULL;
			type_sym = (_tmp3 = (_tmp2 = vala_symbol_get_parent_symbol (constructor_sym), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2))), (type_sym == NULL ? NULL : (type_sym = (g_object_unref (type_sym), NULL))), _tmp3);
			_tmp4 = NULL;
			constructor = (_tmp4 = VALA_METHOD (constructor_sym), (_tmp4 == NULL ? NULL : g_object_ref (_tmp4)));
			if (!(VALA_IS_CREATION_METHOD (constructor_sym))) {
				char* _tmp6;
				char* _tmp5;
				vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
				_tmp6 = NULL;
				_tmp5 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp6 = g_strdup_printf ("`%s' is not a creation method", (_tmp5 = vala_symbol_get_full_name (VALA_SYMBOL (constructor))))));
				(_tmp6 = (g_free (_tmp6), NULL));
				(_tmp5 = (g_free (_tmp5), NULL));
				(constructor == NULL ? NULL : (constructor = (g_object_unref (constructor), NULL)));
				(constructor_sym == NULL ? NULL : (constructor_sym = (g_object_unref (constructor_sym), NULL)));
				(type_sym == NULL ? NULL : (type_sym = (g_object_unref (type_sym), NULL)));
				(type_args == NULL ? NULL : (type_args = (g_object_unref (type_args), NULL)));
				(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
				return;
			}
			vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), VALA_SYMBOL (constructor));
			_tmp7 = NULL;
			type_args = (_tmp7 = vala_member_access_get_type_arguments ((VALA_MEMBER_ACCESS (vala_member_access_get_inner (vala_object_creation_expression_get_member_name (expr))))), (type_args == NULL ? NULL : (type_args = (g_object_unref (type_args), NULL))), _tmp7);
			(constructor == NULL ? NULL : (constructor = (g_object_unref (constructor), NULL)));
		} else {
			if (VALA_IS_ENUM_VALUE (constructor_sym)) {
				ValaSymbol* _tmp9;
				ValaSymbol* _tmp8;
				_tmp9 = NULL;
				_tmp8 = NULL;
				type_sym = (_tmp9 = (_tmp8 = vala_symbol_get_parent_symbol (constructor_sym), (_tmp8 == NULL ? NULL : g_object_ref (_tmp8))), (type_sym == NULL ? NULL : (type_sym = (g_object_unref (type_sym), NULL))), _tmp9);
				vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), constructor_sym);
			}
		}
		if (VALA_IS_CLASS (type_sym) || VALA_IS_STRUCT (type_sym)) {
			ValaDataType* _tmp11;
			ValaDataType* _tmp10;
			_tmp11 = NULL;
			_tmp10 = NULL;
			type = (_tmp11 = (_tmp10 = VALA_DATA_TYPE (type_sym), (_tmp10 == NULL ? NULL : g_object_ref (_tmp10))), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), _tmp11);
		} else {
			if (VALA_IS_ENUM (type_sym) && vala_enum_get_error_domain ((VALA_ENUM (type_sym)))) {
				ValaDataType* _tmp13;
				ValaDataType* _tmp12;
				_tmp13 = NULL;
				_tmp12 = NULL;
				type = (_tmp13 = (_tmp12 = VALA_DATA_TYPE (type_sym), (_tmp12 == NULL ? NULL : g_object_ref (_tmp12))), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), _tmp13);
			} else {
				char* _tmp15;
				char* _tmp14;
				vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
				_tmp15 = NULL;
				_tmp14 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp15 = g_strdup_printf ("`%s' is not a class, struct, or error domain", (_tmp14 = vala_symbol_get_full_name (type_sym)))));
				(_tmp15 = (g_free (_tmp15), NULL));
				(_tmp14 = (g_free (_tmp14), NULL));
				(constructor_sym == NULL ? NULL : (constructor_sym = (g_object_unref (constructor_sym), NULL)));
				(type_sym == NULL ? NULL : (type_sym = (g_object_unref (type_sym), NULL)));
				(type_args == NULL ? NULL : (type_args = (g_object_unref (type_args), NULL)));
				(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
				return;
			}
		}
		_tmp16 = NULL;
		vala_object_creation_expression_set_type_reference (expr, (_tmp16 = vala_type_reference_new ()));
		(_tmp16 == NULL ? NULL : (_tmp16 = (g_object_unref (_tmp16), NULL)));
		vala_type_reference_set_data_type (vala_object_creation_expression_get_type_reference (expr), type);
		{
			GeeCollection* type_arg_collection;
			GeeIterator* type_arg_it;
			type_arg_collection = type_args;
			type_arg_it = gee_iterable_iterator (GEE_ITERABLE (type_arg_collection));
			while (gee_iterator_next (type_arg_it)) {
				ValaTypeReference* type_arg;
				type_arg = gee_iterator_get (type_arg_it);
				{
					vala_type_reference_add_type_argument (vala_object_creation_expression_get_type_reference (expr), type_arg);
					if (vala_type_reference_get_data_type (type_arg) != NULL) {
						vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (type_arg)), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
					}
					(type_arg == NULL ? NULL : (type_arg = (g_object_unref (type_arg), NULL)));
				}
			}
			(type_arg_it == NULL ? NULL : (type_arg_it = (g_object_unref (type_arg_it), NULL)));
		}
		(constructor_sym == NULL ? NULL : (constructor_sym = (g_object_unref (constructor_sym), NULL)));
		(type_sym == NULL ? NULL : (type_sym = (g_object_unref (type_sym), NULL)));
		(type_args == NULL ? NULL : (type_args = (g_object_unref (type_args), NULL)));
	} else {
		ValaDataType* _tmp18;
		ValaDataType* _tmp17;
		_tmp18 = NULL;
		_tmp17 = NULL;
		type = (_tmp18 = (_tmp17 = vala_type_reference_get_data_type (vala_object_creation_expression_get_type_reference (expr)), (_tmp17 == NULL ? NULL : g_object_ref (_tmp17))), (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), _tmp18);
	}
	vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (type), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	_tmp19 = NULL;
	vala_expression_set_static_type (VALA_EXPRESSION (expr), (_tmp19 = vala_type_reference_copy (vala_object_creation_expression_get_type_reference (expr))));
	(_tmp19 = (g_object_unref (_tmp19), NULL));
	vala_type_reference_set_transfers_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), TRUE);
	if (VALA_IS_CLASS (type)) {
		ValaClass* _tmp20;
		ValaClass* cl;
		_tmp20 = NULL;
		cl = (_tmp20 = VALA_CLASS (type), (_tmp20 == NULL ? NULL : g_object_ref (_tmp20)));
		if (vala_class_get_is_abstract (cl)) {
			char* _tmp22;
			char* _tmp21;
			vala_expression_set_static_type (VALA_EXPRESSION (expr), NULL);
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			_tmp22 = NULL;
			_tmp21 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp22 = g_strdup_printf ("Can't create instance of abstract class `%s'", (_tmp21 = vala_symbol_get_full_name (VALA_SYMBOL (cl))))));
			(_tmp22 = (g_free (_tmp22), NULL));
			(_tmp21 = (g_free (_tmp21), NULL));
			(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
			(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			return;
		}
		if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL) {
			vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), VALA_SYMBOL (vala_class_get_default_construction_method (cl)));
		}
		while (cl != NULL) {
			ValaClass* _tmp24;
			ValaClass* _tmp23;
			if (VALA_DATA_TYPE (cl) == self->priv->initially_unowned_type) {
				vala_type_reference_set_floating_reference (vala_expression_get_static_type (VALA_EXPRESSION (expr)), TRUE);
				break;
			}
			_tmp24 = NULL;
			_tmp23 = NULL;
			cl = (_tmp24 = (_tmp23 = vala_class_get_base_class (cl), (_tmp23 == NULL ? NULL : g_object_ref (_tmp23))), (cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL))), _tmp24);
		}
		(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
	} else {
		if (VALA_IS_STRUCT (type)) {
			ValaStruct* _tmp25;
			ValaStruct* st;
			_tmp25 = NULL;
			st = (_tmp25 = VALA_STRUCT (type), (_tmp25 == NULL ? NULL : g_object_ref (_tmp25)));
			vala_type_reference_set_transfers_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), FALSE);
			if (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL) {
				vala_expression_set_symbol_reference (VALA_EXPRESSION (expr), VALA_SYMBOL (vala_struct_get_default_construction_method (st)));
			}
			(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
		}
	}
	_tmp26 = NULL;
	if ((_tmp27 = vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)) == NULL && gee_collection_get_size ((_tmp26 = vala_object_creation_expression_get_argument_list (expr))) != 0, (_tmp26 == NULL ? NULL : (_tmp26 = (g_object_unref (_tmp26), NULL))), _tmp27)) {
		char* _tmp29;
		char* _tmp28;
		vala_expression_set_static_type (VALA_EXPRESSION (expr), NULL);
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		_tmp29 = NULL;
		_tmp28 = NULL;
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp29 = g_strdup_printf ("No arguments allowed when constructing type `%s'", (_tmp28 = vala_symbol_get_full_name (VALA_SYMBOL (type))))));
		(_tmp29 = (g_free (_tmp29), NULL));
		(_tmp28 = (g_free (_tmp28), NULL));
		(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
		return;
	}
	if (VALA_IS_METHOD (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr)))) {
		ValaMethod* _tmp30;
		ValaMethod* m;
		GeeCollection* _tmp32;
		GeeCollection* _tmp31;
		GeeCollection* _tmp33;
		_tmp30 = NULL;
		m = (_tmp30 = VALA_METHOD (vala_expression_get_symbol_reference (VALA_EXPRESSION (expr))), (_tmp30 == NULL ? NULL : g_object_ref (_tmp30)));
		_tmp32 = NULL;
		_tmp31 = NULL;
		vala_semantic_analyzer_check_arguments (self, VALA_EXPRESSION (expr), VALA_SYMBOL (m), (_tmp31 = vala_invokable_get_parameters (VALA_INVOKABLE (m))), (_tmp32 = vala_object_creation_expression_get_argument_list (expr)));
		(_tmp32 == NULL ? NULL : (_tmp32 = (g_object_unref (_tmp32), NULL)));
		(_tmp31 == NULL ? NULL : (_tmp31 = (g_object_unref (_tmp31), NULL)));
		_tmp33 = NULL;
		vala_code_node_set_tree_can_fail (VALA_CODE_NODE (expr), (vala_expression_set_can_fail (VALA_EXPRESSION (expr), (gee_collection_get_size ((_tmp33 = vala_method_get_error_domains (m))) > 0)), vala_expression_get_can_fail (VALA_EXPRESSION (expr))));
		(_tmp33 == NULL ? NULL : (_tmp33 = (g_object_unref (_tmp33), NULL)));
		(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
	} else {
		if (VALA_IS_ENUM (type)) {
			GeeCollection* _tmp34;
			gboolean _tmp35;
			ValaTypeReference* _tmp38;
			_tmp34 = NULL;
			if ((_tmp35 = gee_collection_get_size ((_tmp34 = vala_object_creation_expression_get_argument_list (expr))) == 0, (_tmp34 == NULL ? NULL : (_tmp34 = (g_object_unref (_tmp34), NULL))), _tmp35)) {
				vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Too few arguments, errors need at least 1 argument");
			} else {
				GeeCollection* _tmp36;
				GeeIterator* _tmp37;
				GeeIterator* arg_it;
				ValaExpression* ex;
				_tmp36 = NULL;
				_tmp37 = NULL;
				arg_it = (_tmp37 = gee_iterable_iterator (GEE_ITERABLE ((_tmp36 = vala_object_creation_expression_get_argument_list (expr)))), (_tmp36 == NULL ? NULL : (_tmp36 = (g_object_unref (_tmp36), NULL))), _tmp37);
				gee_iterator_next (arg_it);
				ex = gee_iterator_get (arg_it);
				if (vala_expression_get_static_type (ex) == NULL || !vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (ex), self->priv->string_type)) {
					vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Invalid type for argument 1");
				}
				(arg_it == NULL ? NULL : (arg_it = (g_object_unref (arg_it), NULL)));
				(ex == NULL ? NULL : (ex = (g_object_unref (ex), NULL)));
			}
			_tmp38 = NULL;
			vala_expression_set_static_type (VALA_EXPRESSION (expr), (_tmp38 = vala_type_reference_new ()));
			(_tmp38 == NULL ? NULL : (_tmp38 = (g_object_unref (_tmp38), NULL)));
		}
	}
	{
		GeeCollection* init_collection;
		GeeIterator* init_it;
		init_collection = vala_object_creation_expression_get_object_initializer (expr);
		init_it = gee_iterable_iterator (GEE_ITERABLE (init_collection));
		while (gee_iterator_next (init_it)) {
			ValaMemberInitializer* init;
			init = gee_iterator_get (init_it);
			{
				ValaSymbol* _tmp39;
				ValaTypeReference* member_type;
				_tmp39 = NULL;
				vala_member_initializer_set_symbol_reference (init, (_tmp39 = vala_semantic_analyzer_symbol_lookup_inherited (VALA_SYMBOL (vala_type_reference_get_data_type (vala_object_creation_expression_get_type_reference (expr))), vala_member_initializer_get_name (init))));
				(_tmp39 == NULL ? NULL : (_tmp39 = (g_object_unref (_tmp39), NULL)));
				if (!(VALA_IS_FIELD (vala_member_initializer_get_symbol_reference (init)) || VALA_IS_PROPERTY (vala_member_initializer_get_symbol_reference (init)))) {
					char* _tmp41;
					char* _tmp40;
					vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
					_tmp41 = NULL;
					_tmp40 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp41 = g_strdup_printf ("Invalid member `%s' in `%s'", vala_member_initializer_get_name (init), (_tmp40 = vala_symbol_get_full_name (VALA_SYMBOL (vala_type_reference_get_data_type (vala_object_creation_expression_get_type_reference (expr))))))));
					(_tmp41 = (g_free (_tmp41), NULL));
					(_tmp40 = (g_free (_tmp40), NULL));
					(init == NULL ? NULL : (init = (g_object_unref (init), NULL)));
					(init_collection == NULL ? NULL : (init_collection = (g_object_unref (init_collection), NULL)));
					(init_it == NULL ? NULL : (init_it = (g_object_unref (init_it), NULL)));
					(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
					return;
				}
				if (vala_symbol_get_access (vala_member_initializer_get_symbol_reference (init)) != VALA_SYMBOL_ACCESSIBILITY_PUBLIC) {
					char* _tmp43;
					char* _tmp42;
					vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
					_tmp43 = NULL;
					_tmp42 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp43 = g_strdup_printf ("Access to private member `%s' denied", (_tmp42 = vala_symbol_get_full_name (vala_member_initializer_get_symbol_reference (init))))));
					(_tmp43 = (g_free (_tmp43), NULL));
					(_tmp42 = (g_free (_tmp42), NULL));
					(init == NULL ? NULL : (init = (g_object_unref (init), NULL)));
					(init_collection == NULL ? NULL : (init_collection = (g_object_unref (init_collection), NULL)));
					(init_it == NULL ? NULL : (init_it = (g_object_unref (init_it), NULL)));
					(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
					return;
				}
				member_type = NULL;
				if (VALA_IS_FIELD (vala_member_initializer_get_symbol_reference (init))) {
					ValaField* _tmp44;
					ValaField* f;
					ValaTypeReference* _tmp45;
					_tmp44 = NULL;
					f = (_tmp44 = VALA_FIELD (vala_member_initializer_get_symbol_reference (init)), (_tmp44 == NULL ? NULL : g_object_ref (_tmp44)));
					_tmp45 = NULL;
					member_type = (_tmp45 = g_object_ref (vala_field_get_type_reference (f)), (member_type == NULL ? NULL : (member_type = (g_object_unref (member_type), NULL))), _tmp45);
					(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
				} else {
					if (VALA_IS_PROPERTY (vala_member_initializer_get_symbol_reference (init))) {
						ValaProperty* _tmp46;
						ValaProperty* prop;
						ValaTypeReference* _tmp47;
						_tmp46 = NULL;
						prop = (_tmp46 = VALA_PROPERTY (vala_member_initializer_get_symbol_reference (init)), (_tmp46 == NULL ? NULL : g_object_ref (_tmp46)));
						_tmp47 = NULL;
						member_type = (_tmp47 = g_object_ref (vala_property_get_type_reference (prop)), (member_type == NULL ? NULL : (member_type = (g_object_unref (member_type), NULL))), _tmp47);
						if (vala_property_get_set_accessor (prop) == NULL || !vala_property_accessor_get_writable (vala_property_get_set_accessor (prop))) {
							char* _tmp49;
							char* _tmp48;
							vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
							_tmp49 = NULL;
							_tmp48 = NULL;
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp49 = g_strdup_printf ("Property `%s' is read-only", (_tmp48 = vala_symbol_get_full_name (VALA_SYMBOL (prop))))));
							(_tmp49 = (g_free (_tmp49), NULL));
							(_tmp48 = (g_free (_tmp48), NULL));
							(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
							(init == NULL ? NULL : (init = (g_object_unref (init), NULL)));
							(member_type == NULL ? NULL : (member_type = (g_object_unref (member_type), NULL)));
							(init_collection == NULL ? NULL : (init_collection = (g_object_unref (init_collection), NULL)));
							(init_it == NULL ? NULL : (init_it = (g_object_unref (init_it), NULL)));
							(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
							return;
						}
						(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
					}
				}
				if (vala_expression_get_static_type (vala_member_initializer_get_initializer (init)) == NULL || !vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (vala_member_initializer_get_initializer (init)), member_type)) {
					char* _tmp50;
					vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
					_tmp50 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (init)), (_tmp50 = g_strdup_printf ("Invalid type for member `%s'", vala_member_initializer_get_name (init))));
					(_tmp50 = (g_free (_tmp50), NULL));
					(init == NULL ? NULL : (init = (g_object_unref (init), NULL)));
					(member_type == NULL ? NULL : (member_type = (g_object_unref (member_type), NULL)));
					(init_collection == NULL ? NULL : (init_collection = (g_object_unref (init_collection), NULL)));
					(init_it == NULL ? NULL : (init_it = (g_object_unref (init_it), NULL)));
					(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
					return;
				}
				(init == NULL ? NULL : (init = (g_object_unref (init), NULL)));
				(member_type == NULL ? NULL : (member_type = (g_object_unref (member_type), NULL)));
			}
		}
		(init_collection == NULL ? NULL : (init_collection = (g_object_unref (init_collection), NULL)));
		(init_it == NULL ? NULL : (init_it = (g_object_unref (init_it), NULL)));
	}
	(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
}


static void vala_semantic_analyzer_real_visit_sizeof_expression (ValaCodeVisitor* base, ValaSizeofExpression* expr) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_SIZEOF_EXPRESSION (expr));
	vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->ulong_type);
}


static void vala_semantic_analyzer_real_visit_typeof_expression (ValaCodeVisitor* base, ValaTypeofExpression* expr) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_TYPEOF_EXPRESSION (expr));
	vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->type_type);
}


static gboolean vala_semantic_analyzer_is_numeric_type (ValaSemanticAnalyzer* self, ValaTypeReference* type) {
	ValaStruct* _tmp1;
	ValaStruct* st;
	gboolean _tmp2;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (type), FALSE);
	if (!(VALA_IS_STRUCT (vala_type_reference_get_data_type (type)))) {
		return FALSE;
	}
	_tmp1 = NULL;
	st = (_tmp1 = VALA_STRUCT (vala_type_reference_get_data_type (type)), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1)));
	return (_tmp2 = vala_struct_is_integer_type (st) || vala_struct_is_floating_type (st), (st == NULL ? NULL : (st = (g_object_unref (st), NULL))), _tmp2);
	(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
}


static gboolean vala_semantic_analyzer_is_integer_type (ValaSemanticAnalyzer* self, ValaTypeReference* type) {
	ValaStruct* _tmp1;
	ValaStruct* st;
	gboolean _tmp2;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (type), FALSE);
	if (!(VALA_IS_STRUCT (vala_type_reference_get_data_type (type)))) {
		return FALSE;
	}
	_tmp1 = NULL;
	st = (_tmp1 = VALA_STRUCT (vala_type_reference_get_data_type (type)), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1)));
	return (_tmp2 = vala_struct_is_integer_type (st), (st == NULL ? NULL : (st = (g_object_unref (st), NULL))), _tmp2);
	(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
}


static void vala_semantic_analyzer_real_visit_unary_expression (ValaCodeVisitor* base, ValaUnaryExpression* expr) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_UNARY_EXPRESSION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_unary_expression_get_inner (expr)))) {
		/* if there was an error in the inner expression, skip type check */
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		return;
	}
	if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_PLUS || vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_MINUS) {
		/* integer or floating point type*/
		if (!vala_semantic_analyzer_is_numeric_type (self, vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))) {
			char* _tmp1;
			char* _tmp0;
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			_tmp1 = NULL;
			_tmp0 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp1 = g_strdup_printf ("Operator not supported for `%s'", (_tmp0 = vala_type_reference_to_string (vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))))));
			(_tmp1 = (g_free (_tmp1), NULL));
			(_tmp0 = (g_free (_tmp0), NULL));
			return;
		}
		vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_unary_expression_get_inner (expr)));
	} else {
		if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_LOGICAL_NEGATION) {
			/* boolean type*/
			if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_unary_expression_get_inner (expr))) != vala_type_reference_get_data_type (self->priv->bool_type)) {
				char* _tmp3;
				char* _tmp2;
				vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
				_tmp3 = NULL;
				_tmp2 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp3 = g_strdup_printf ("Operator not supported for `%s'", (_tmp2 = vala_type_reference_to_string (vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))))));
				(_tmp3 = (g_free (_tmp3), NULL));
				(_tmp2 = (g_free (_tmp2), NULL));
				return;
			}
			vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_unary_expression_get_inner (expr)));
		} else {
			if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_BITWISE_COMPLEMENT) {
				/* integer type*/
				if (!vala_semantic_analyzer_is_integer_type (self, vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))) {
					char* _tmp5;
					char* _tmp4;
					vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
					_tmp5 = NULL;
					_tmp4 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp5 = g_strdup_printf ("Operator not supported for `%s'", (_tmp4 = vala_type_reference_to_string (vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))))));
					(_tmp5 = (g_free (_tmp5), NULL));
					(_tmp4 = (g_free (_tmp4), NULL));
					return;
				}
				vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_unary_expression_get_inner (expr)));
			} else {
				if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_INCREMENT || vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_DECREMENT) {
					ValaMemberAccess* ma;
					ValaMemberAccess* old_value;
					ValaLiteralExpression* _tmp9;
					ValaIntegerLiteral* _tmp8;
					ValaBinaryExpression* _tmp10;
					ValaBinaryExpression* bin;
					ValaAssignment* assignment;
					ValaParenthesizedExpression* parenthexp;
					/* integer type*/
					if (!vala_semantic_analyzer_is_integer_type (self, vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))) {
						char* _tmp7;
						char* _tmp6;
						vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
						_tmp7 = NULL;
						_tmp6 = NULL;
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp7 = g_strdup_printf ("Operator not supported for `%s'", (_tmp6 = vala_type_reference_to_string (vala_expression_get_static_type (vala_unary_expression_get_inner (expr)))))));
						(_tmp7 = (g_free (_tmp7), NULL));
						(_tmp6 = (g_free (_tmp6), NULL));
						return;
					}
					ma = vala_semantic_analyzer_find_member_access (self, vala_unary_expression_get_inner (expr));
					if (ma == NULL) {
						vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Prefix operators not supported for this expression");
						(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
						return;
					}
					old_value = vala_member_access_new (vala_member_access_get_inner (ma), vala_member_access_get_member_name (ma), vala_code_node_get_source_reference (VALA_CODE_NODE (vala_unary_expression_get_inner (expr))));
					_tmp9 = NULL;
					_tmp8 = NULL;
					_tmp10 = NULL;
					bin = (_tmp10 = vala_binary_expression_new ((vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_INCREMENT ? VALA_BINARY_OPERATOR_PLUS : VALA_BINARY_OPERATOR_MINUS), VALA_EXPRESSION (old_value), VALA_EXPRESSION ((_tmp9 = vala_literal_expression_new (VALA_LITERAL ((_tmp8 = vala_integer_literal_new ("1", NULL))), NULL))), vala_code_node_get_source_reference (VALA_CODE_NODE (expr))), (_tmp9 == NULL ? NULL : (_tmp9 = (g_object_unref (_tmp9), NULL))), (_tmp8 == NULL ? NULL : (_tmp8 = (g_object_unref (_tmp8), NULL))), _tmp10);
					assignment = vala_code_context_create_assignment (self->priv->context, VALA_EXPRESSION (ma), VALA_EXPRESSION (bin), VALA_ASSIGNMENT_OPERATOR_SIMPLE, vala_code_node_get_source_reference (VALA_CODE_NODE (expr)));
					parenthexp = vala_parenthesized_expression_new (VALA_EXPRESSION (assignment), vala_code_node_get_source_reference (VALA_CODE_NODE (expr)));
					vala_code_node_replace (vala_code_node_get_parent_node (VALA_CODE_NODE (expr)), VALA_CODE_NODE (expr), VALA_CODE_NODE (parenthexp));
					vala_code_node_accept (VALA_CODE_NODE (parenthexp), VALA_CODE_VISITOR (self));
					(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
					(old_value == NULL ? NULL : (old_value = (g_object_unref (old_value), NULL)));
					(bin == NULL ? NULL : (bin = (g_object_unref (bin), NULL)));
					(assignment = (g_object_unref (assignment), NULL));
					(parenthexp == NULL ? NULL : (parenthexp = (g_object_unref (parenthexp), NULL)));
					return;
					(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
					(old_value == NULL ? NULL : (old_value = (g_object_unref (old_value), NULL)));
					(bin == NULL ? NULL : (bin = (g_object_unref (bin), NULL)));
					(assignment = (g_object_unref (assignment), NULL));
					(parenthexp == NULL ? NULL : (parenthexp = (g_object_unref (parenthexp), NULL)));
				} else {
					if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_REF) {
						/* value type*/
						vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_unary_expression_get_inner (expr)));
					} else {
						if (vala_unary_expression_get_operator (expr) == VALA_UNARY_OPERATOR_OUT) {
							/* reference type*/
							vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_unary_expression_get_inner (expr)));
						} else {
							vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "internal error: unsupported unary operator");
							return;
						}
					}
				}
			}
		}
	}
}


static ValaMemberAccess* vala_semantic_analyzer_find_member_access (ValaSemanticAnalyzer* self, ValaExpression* expr) {
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	g_return_val_if_fail (VALA_IS_EXPRESSION (expr), NULL);
	if (VALA_IS_PARENTHESIZED_EXPRESSION (expr)) {
		ValaParenthesizedExpression* _tmp0;
		ValaParenthesizedExpression* pe;
		ValaMemberAccess* _tmp1;
		_tmp0 = NULL;
		pe = (_tmp0 = VALA_PARENTHESIZED_EXPRESSION (expr), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
		_tmp1 = NULL;
		return (_tmp1 = vala_semantic_analyzer_find_member_access (self, vala_parenthesized_expression_get_inner (pe)), (pe == NULL ? NULL : (pe = (g_object_unref (pe), NULL))), _tmp1);
		(pe == NULL ? NULL : (pe = (g_object_unref (pe), NULL)));
	}
	if (VALA_IS_MEMBER_ACCESS (expr)) {
		ValaMemberAccess* _tmp2;
		_tmp2 = NULL;
		return (_tmp2 = VALA_MEMBER_ACCESS (expr), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2)));
	}
	return NULL;
}


static void vala_semantic_analyzer_real_visit_cast_expression (ValaCodeVisitor* base, ValaCastExpression* expr) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CAST_EXPRESSION (expr));
	if (vala_type_reference_get_data_type (vala_cast_expression_get_type_reference (expr)) == NULL && vala_type_reference_get_type_parameter (vala_cast_expression_get_type_reference (expr)) == NULL) {
		return;
	}
	/* if type resolving didn't succeed, skip this check */
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_cast_expression_get_inner (expr)))) {
		return;
	}
	/* FIXME: check whether cast is allowed*/
	if (vala_type_reference_get_data_type (vala_cast_expression_get_type_reference (expr)) != NULL) {
		vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (vala_cast_expression_get_type_reference (expr))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	}
	vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_cast_expression_get_type_reference (expr));
	vala_type_reference_set_transfers_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (vala_cast_expression_get_inner (expr))));
}


static void vala_semantic_analyzer_real_visit_pointer_indirection (ValaCodeVisitor* base, ValaPointerIndirection* expr) {
	ValaSemanticAnalyzer * self;
	ValaPointer* _tmp0;
	ValaPointer* pointer;
	ValaTypeReference* _tmp1;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_POINTER_INDIRECTION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_pointer_indirection_get_inner (expr)))) {
		return;
	}
	if (vala_expression_get_static_type (vala_pointer_indirection_get_inner (expr)) == NULL) {
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "internal error: unknown type of inner expression");
		return;
	}
	if (!(VALA_IS_POINTER (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_pointer_indirection_get_inner (expr)))))) {
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Pointer indirection not supported for this expression");
		return;
	}
	_tmp0 = NULL;
	pointer = (_tmp0 = VALA_POINTER (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_pointer_indirection_get_inner (expr)))), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
	_tmp1 = NULL;
	vala_expression_set_static_type (VALA_EXPRESSION (expr), (_tmp1 = vala_type_reference_new ()));
	(_tmp1 == NULL ? NULL : (_tmp1 = (g_object_unref (_tmp1), NULL)));
	vala_type_reference_set_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr)), vala_pointer_get_referent_type (pointer));
	vala_type_reference_set_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_pointer_indirection_get_inner (expr))));
	(pointer == NULL ? NULL : (pointer = (g_object_unref (pointer), NULL)));
}


static void vala_semantic_analyzer_real_visit_addressof_expression (ValaCodeVisitor* base, ValaAddressofExpression* expr) {
	ValaSemanticAnalyzer * self;
	ValaTypeReference* _tmp0;
	ValaPointer* _tmp1;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_ADDRESSOF_EXPRESSION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_addressof_expression_get_inner (expr)))) {
		return;
	}
	if (vala_expression_get_static_type (vala_addressof_expression_get_inner (expr)) == NULL) {
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "internal error: unknown type of inner expression");
		return;
	}
	if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_addressof_expression_get_inner (expr))) == NULL) {
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Address-of operator not supported for this expression");
		return;
	}
	_tmp0 = NULL;
	vala_expression_set_static_type (VALA_EXPRESSION (expr), (_tmp0 = vala_type_reference_new ()));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	_tmp1 = NULL;
	vala_type_reference_set_data_type (vala_expression_get_static_type (VALA_EXPRESSION (expr)), VALA_DATA_TYPE ((_tmp1 = vala_data_type_get_pointer (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_addressof_expression_get_inner (expr)))))));
	(_tmp1 = (g_object_unref (_tmp1), NULL));
	vala_type_reference_set_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_addressof_expression_get_inner (expr))));
}


static void vala_semantic_analyzer_real_visit_reference_transfer_expression (ValaCodeVisitor* base, ValaReferenceTransferExpression* expr) {
	ValaSemanticAnalyzer * self;
	ValaTypeReference* _tmp0;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_REFERENCE_TRANSFER_EXPRESSION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_reference_transfer_expression_get_inner (expr)))) {
		/* if there was an error in the inner expression, skip type check */
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		return;
	}
	if (!(VALA_IS_MEMBER_ACCESS (vala_reference_transfer_expression_get_inner (expr)) || VALA_IS_ELEMENT_ACCESS (vala_reference_transfer_expression_get_inner (expr)))) {
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Reference transfer not supported for this expression");
		return;
	}
	if (!vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_reference_transfer_expression_get_inner (expr)))) {
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "No reference to be transferred");
		return;
	}
	_tmp0 = NULL;
	vala_expression_set_static_type (VALA_EXPRESSION (expr), (_tmp0 = vala_type_reference_copy (vala_expression_get_static_type (vala_reference_transfer_expression_get_inner (expr)))));
	(_tmp0 = (g_object_unref (_tmp0), NULL));
	vala_type_reference_set_transfers_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), TRUE);
	vala_type_reference_set_takes_ownership (vala_expression_get_static_type (VALA_EXPRESSION (expr)), FALSE);
}


static ValaTypeReference* vala_semantic_analyzer_get_arithmetic_result_type (ValaSemanticAnalyzer* self, ValaTypeReference* left_type, ValaTypeReference* right_type) {
	ValaStruct* _tmp1;
	ValaStruct* left;
	ValaStruct* _tmp2;
	ValaStruct* right;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (left_type), NULL);
	g_return_val_if_fail (VALA_IS_TYPE_REFERENCE (right_type), NULL);
	if (!(VALA_IS_STRUCT (vala_type_reference_get_data_type (left_type))) || !(VALA_IS_STRUCT (vala_type_reference_get_data_type (right_type)))) {
		return NULL;
	}
	/* at least one operand not struct*/
	_tmp1 = NULL;
	left = (_tmp1 = VALA_STRUCT (vala_type_reference_get_data_type (left_type)), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1)));
	_tmp2 = NULL;
	right = (_tmp2 = VALA_STRUCT (vala_type_reference_get_data_type (right_type)), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2)));
	if ((!vala_struct_is_floating_type (left) && !vala_struct_is_integer_type (left)) || (!vala_struct_is_floating_type (right) && !vala_struct_is_integer_type (right))) {
		gpointer _tmp3;
		return (_tmp3 = NULL, (left == NULL ? NULL : (left = (g_object_unref (left), NULL))), (right == NULL ? NULL : (right = (g_object_unref (right), NULL))), _tmp3);
	}
	/* at least one operand not numeric*/
	if (vala_struct_is_floating_type (left) == vala_struct_is_floating_type (right)) {
		/* both operands integer or floating type*/
		if (vala_struct_get_rank (left) >= vala_struct_get_rank (right)) {
			ValaTypeReference* _tmp4;
			_tmp4 = NULL;
			return (_tmp4 = g_object_ref (left_type), (left == NULL ? NULL : (left = (g_object_unref (left), NULL))), (right == NULL ? NULL : (right = (g_object_unref (right), NULL))), _tmp4);
		} else {
			ValaTypeReference* _tmp5;
			_tmp5 = NULL;
			return (_tmp5 = g_object_ref (right_type), (left == NULL ? NULL : (left = (g_object_unref (left), NULL))), (right == NULL ? NULL : (right = (g_object_unref (right), NULL))), _tmp5);
		}
	} else {
		/* one integer and one floating type operand*/
		if (vala_struct_is_floating_type (left)) {
			ValaTypeReference* _tmp6;
			_tmp6 = NULL;
			return (_tmp6 = g_object_ref (left_type), (left == NULL ? NULL : (left = (g_object_unref (left), NULL))), (right == NULL ? NULL : (right = (g_object_unref (right), NULL))), _tmp6);
		} else {
			ValaTypeReference* _tmp7;
			_tmp7 = NULL;
			return (_tmp7 = g_object_ref (right_type), (left == NULL ? NULL : (left = (g_object_unref (left), NULL))), (right == NULL ? NULL : (right = (g_object_unref (right), NULL))), _tmp7);
		}
	}
	(left == NULL ? NULL : (left = (g_object_unref (left), NULL)));
	(right == NULL ? NULL : (right = (g_object_unref (right), NULL)));
}


static void vala_semantic_analyzer_real_visit_binary_expression (ValaCodeVisitor* base, ValaBinaryExpression* expr) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_BINARY_EXPRESSION (expr));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_binary_expression_get_left (expr))) || vala_code_node_get_error (VALA_CODE_NODE (vala_binary_expression_get_right (expr)))) {
		/* if there were any errors in inner expressions, skip type check */
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		return;
	}
	if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_left (expr))) == vala_type_reference_get_data_type (self->priv->string_type) && vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_PLUS) {
		ValaMemberAccess* _tmp0;
		ValaInvocationExpression* _tmp1;
		ValaInvocationExpression* concat_call;
		if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_right (expr))) != vala_type_reference_get_data_type (self->priv->string_type)) {
			vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Operands must be strings");
			return;
		}
		/* string concatenation: convert to a.concat (b) */
		_tmp0 = NULL;
		_tmp1 = NULL;
		concat_call = (_tmp1 = vala_invocation_expression_new (VALA_EXPRESSION ((_tmp0 = vala_member_access_new (vala_binary_expression_get_left (expr), "concat", NULL))), NULL), (_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL))), _tmp1);
		vala_invocation_expression_add_argument (concat_call, vala_binary_expression_get_right (expr));
		vala_code_node_replace (vala_code_node_get_parent_node (VALA_CODE_NODE (expr)), VALA_CODE_NODE (expr), VALA_CODE_NODE (concat_call));
		vala_code_node_accept (VALA_CODE_NODE (concat_call), VALA_CODE_VISITOR (self));
		(concat_call == NULL ? NULL : (concat_call = (g_object_unref (concat_call), NULL)));
	} else {
		if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_PLUS || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_MINUS || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_MUL || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_DIV) {
			ValaTypeReference* _tmp2;
			_tmp2 = NULL;
			vala_expression_set_static_type (VALA_EXPRESSION (expr), (_tmp2 = vala_semantic_analyzer_get_arithmetic_result_type (self, vala_expression_get_static_type (vala_binary_expression_get_left (expr)), vala_expression_get_static_type (vala_binary_expression_get_right (expr)))));
			(_tmp2 == NULL ? NULL : (_tmp2 = (g_object_unref (_tmp2), NULL)));
			if (vala_expression_get_static_type (VALA_EXPRESSION (expr)) == NULL) {
				char* _tmp5;
				char* _tmp4;
				char* _tmp3;
				vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
				_tmp5 = NULL;
				_tmp4 = NULL;
				_tmp3 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp5 = g_strdup_printf ("Arithmetic operation not supported for types `%s' and `%s'", (_tmp3 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_left (expr)))), (_tmp4 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_right (expr)))))));
				(_tmp5 = (g_free (_tmp5), NULL));
				(_tmp4 = (g_free (_tmp4), NULL));
				(_tmp3 = (g_free (_tmp3), NULL));
				return;
			}
		} else {
			if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_MOD || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_SHIFT_LEFT || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_SHIFT_RIGHT || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_BITWISE_XOR) {
				ValaTypeReference* _tmp6;
				_tmp6 = NULL;
				vala_expression_set_static_type (VALA_EXPRESSION (expr), (_tmp6 = vala_semantic_analyzer_get_arithmetic_result_type (self, vala_expression_get_static_type (vala_binary_expression_get_left (expr)), vala_expression_get_static_type (vala_binary_expression_get_right (expr)))));
				(_tmp6 == NULL ? NULL : (_tmp6 = (g_object_unref (_tmp6), NULL)));
				if (vala_expression_get_static_type (VALA_EXPRESSION (expr)) == NULL) {
					char* _tmp9;
					char* _tmp8;
					char* _tmp7;
					vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
					_tmp9 = NULL;
					_tmp8 = NULL;
					_tmp7 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp9 = g_strdup_printf ("Arithmetic operation not supported for types `%s' and `%s'", (_tmp7 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_left (expr)))), (_tmp8 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_right (expr)))))));
					(_tmp9 = (g_free (_tmp9), NULL));
					(_tmp8 = (g_free (_tmp8), NULL));
					(_tmp7 = (g_free (_tmp7), NULL));
					return;
				}
			} else {
				if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_LESS_THAN || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_GREATER_THAN || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_LESS_THAN_OR_EQUAL || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_GREATER_THAN_OR_EQUAL) {
					if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_left (expr))) == vala_type_reference_get_data_type (self->priv->string_type) && vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_right (expr))) == vala_type_reference_get_data_type (self->priv->string_type)) {
						ValaMemberAccess* _tmp10;
						ValaInvocationExpression* _tmp11;
						ValaInvocationExpression* cmp_call;
						ValaLiteralExpression* _tmp13;
						ValaIntegerLiteral* _tmp12;
						/* string comparison: convert to a.collate (b) OP 0 */
						_tmp10 = NULL;
						_tmp11 = NULL;
						cmp_call = (_tmp11 = vala_invocation_expression_new (VALA_EXPRESSION ((_tmp10 = vala_member_access_new (vala_binary_expression_get_left (expr), "collate", NULL))), NULL), (_tmp10 == NULL ? NULL : (_tmp10 = (g_object_unref (_tmp10), NULL))), _tmp11);
						vala_invocation_expression_add_argument (cmp_call, vala_binary_expression_get_right (expr));
						vala_binary_expression_set_left (expr, VALA_EXPRESSION (cmp_call));
						_tmp13 = NULL;
						_tmp12 = NULL;
						vala_binary_expression_set_right (expr, VALA_EXPRESSION ((_tmp13 = vala_literal_expression_new (VALA_LITERAL ((_tmp12 = vala_integer_literal_new ("0", NULL))), NULL))));
						(_tmp13 == NULL ? NULL : (_tmp13 = (g_object_unref (_tmp13), NULL)));
						(_tmp12 == NULL ? NULL : (_tmp12 = (g_object_unref (_tmp12), NULL)));
						vala_code_node_accept (VALA_CODE_NODE (vala_binary_expression_get_left (expr)), VALA_CODE_VISITOR (self));
						(cmp_call == NULL ? NULL : (cmp_call = (g_object_unref (cmp_call), NULL)));
					} else {
						ValaTypeReference* resulting_type;
						resulting_type = vala_semantic_analyzer_get_arithmetic_result_type (self, vala_expression_get_static_type (vala_binary_expression_get_left (expr)), vala_expression_get_static_type (vala_binary_expression_get_right (expr)));
						if (resulting_type == NULL) {
							char* _tmp16;
							char* _tmp15;
							char* _tmp14;
							vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
							_tmp16 = NULL;
							_tmp15 = NULL;
							_tmp14 = NULL;
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp16 = g_strdup_printf ("Relational operation not supported for types `%s' and `%s'", (_tmp14 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_left (expr)))), (_tmp15 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_right (expr)))))));
							(_tmp16 = (g_free (_tmp16), NULL));
							(_tmp15 = (g_free (_tmp15), NULL));
							(_tmp14 = (g_free (_tmp14), NULL));
							(resulting_type == NULL ? NULL : (resulting_type = (g_object_unref (resulting_type), NULL)));
							return;
						}
						(resulting_type == NULL ? NULL : (resulting_type = (g_object_unref (resulting_type), NULL)));
					}
					vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->bool_type);
				} else {
					if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_EQUALITY || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_INEQUALITY) {
						/* relational operation */
						if (!vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (vala_binary_expression_get_right (expr)), vala_expression_get_static_type (vala_binary_expression_get_left (expr))) && !vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (vala_binary_expression_get_left (expr)), vala_expression_get_static_type (vala_binary_expression_get_right (expr)))) {
							char* _tmp19;
							char* _tmp18;
							char* _tmp17;
							_tmp19 = NULL;
							_tmp18 = NULL;
							_tmp17 = NULL;
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (_tmp19 = g_strdup_printf ("Equality operation: `%s' and `%s' are incompatible, comparison would always evaluate to false", (_tmp17 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_right (expr)))), (_tmp18 = vala_type_reference_to_string (vala_expression_get_static_type (vala_binary_expression_get_left (expr)))))));
							(_tmp19 = (g_free (_tmp19), NULL));
							(_tmp18 = (g_free (_tmp18), NULL));
							(_tmp17 = (g_free (_tmp17), NULL));
							vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
							return;
						}
						if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_left (expr))) == vala_type_reference_get_data_type (self->priv->string_type) && vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_right (expr))) == vala_type_reference_get_data_type (self->priv->string_type)) {
							ValaMemberAccess* _tmp20;
							ValaInvocationExpression* _tmp21;
							ValaInvocationExpression* cmp_call;
							ValaLiteralExpression* _tmp23;
							ValaIntegerLiteral* _tmp22;
							/* string comparison: convert to a.collate (b) OP 0 */
							_tmp20 = NULL;
							_tmp21 = NULL;
							cmp_call = (_tmp21 = vala_invocation_expression_new (VALA_EXPRESSION ((_tmp20 = vala_member_access_new (vala_binary_expression_get_left (expr), "collate", NULL))), NULL), (_tmp20 == NULL ? NULL : (_tmp20 = (g_object_unref (_tmp20), NULL))), _tmp21);
							vala_invocation_expression_add_argument (cmp_call, vala_binary_expression_get_right (expr));
							vala_binary_expression_set_left (expr, VALA_EXPRESSION (cmp_call));
							_tmp23 = NULL;
							_tmp22 = NULL;
							vala_binary_expression_set_right (expr, VALA_EXPRESSION ((_tmp23 = vala_literal_expression_new (VALA_LITERAL ((_tmp22 = vala_integer_literal_new ("0", NULL))), NULL))));
							(_tmp23 == NULL ? NULL : (_tmp23 = (g_object_unref (_tmp23), NULL)));
							(_tmp22 == NULL ? NULL : (_tmp22 = (g_object_unref (_tmp22), NULL)));
							vala_code_node_accept (VALA_CODE_NODE (vala_binary_expression_get_left (expr)), VALA_CODE_VISITOR (self));
							(cmp_call == NULL ? NULL : (cmp_call = (g_object_unref (cmp_call), NULL)));
						}
						vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->bool_type);
					} else {
						if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_BITWISE_AND || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_BITWISE_OR) {
							/* integer type or flags type*/
							vala_expression_set_static_type (VALA_EXPRESSION (expr), vala_expression_get_static_type (vala_binary_expression_get_left (expr)));
						} else {
							if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_AND || vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_OR) {
								if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_left (expr))) != vala_type_reference_get_data_type (self->priv->bool_type) || vala_type_reference_get_data_type (vala_expression_get_static_type (vala_binary_expression_get_right (expr))) != vala_type_reference_get_data_type (self->priv->bool_type)) {
									vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
									vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "Operands must be boolean");
								}
								vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->bool_type);
							} else {
								if (vala_binary_expression_get_operator (expr) == VALA_BINARY_OPERATOR_IN) {
									/* integer type or flags type*/
									vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->bool_type);
								} else {
									g_assert_not_reached ();
								}
							}
						}
					}
				}
			}
		}
	}
}


static void vala_semantic_analyzer_real_visit_type_check (ValaCodeVisitor* base, ValaTypeCheck* expr) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_TYPE_CHECK (expr));
	if (vala_type_reference_get_data_type (vala_type_check_get_type_reference (expr)) == NULL) {
		/* if type resolving didn't succeed, skip this check */
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		return;
	}
	vala_source_file_add_symbol_dependency (self->priv->current_source_file, VALA_SYMBOL (vala_type_reference_get_data_type (vala_type_check_get_type_reference (expr))), VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE);
	vala_expression_set_static_type (VALA_EXPRESSION (expr), self->priv->bool_type);
}


static ValaTypeReference* vala_semantic_analyzer_compute_common_base_type (ValaSemanticAnalyzer* self, GeeCollection* types, ValaSourceReference* source_reference) {
	gboolean null_found;
	gboolean class_or_iface_found;
	gboolean type_param_found;
	gboolean ref_struct_found;
	gboolean val_struct_found;
	gboolean enum_found;
	gboolean callback_found;
	ValaTypeReference* base_type;
	ValaTypeReference* last_type;
	ValaTypeReference* _tmp13;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	g_return_val_if_fail (types == NULL || GEE_IS_COLLECTION (types), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	null_found = FALSE;
	class_or_iface_found = FALSE;
	type_param_found = FALSE;
	ref_struct_found = FALSE;
	val_struct_found = FALSE;
	enum_found = FALSE;
	callback_found = FALSE;
	base_type = NULL;
	last_type = NULL;
	{
		GeeCollection* type_collection;
		GeeIterator* type_it;
		type_collection = types;
		type_it = gee_iterable_iterator (GEE_ITERABLE (type_collection));
		while (gee_iterator_next (type_it)) {
			ValaTypeReference* type;
			type = gee_iterator_get (type_it);
			{
				ValaTypeReference* _tmp1;
				ValaTypeReference* _tmp0;
				_tmp1 = NULL;
				_tmp0 = NULL;
				last_type = (_tmp1 = (_tmp0 = type, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0))), (last_type == NULL ? NULL : (last_type = (g_object_unref (last_type), NULL))), _tmp1);
				if (vala_code_node_get_error (VALA_CODE_NODE (type))) {
					ValaTypeReference* _tmp2;
					ValaTypeReference* _tmp3;
					_tmp2 = NULL;
					base_type = (_tmp2 = vala_type_reference_new (), (base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL))), _tmp2);
					vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE);
					_tmp3 = NULL;
					return (_tmp3 = base_type, (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), (type_it == NULL ? NULL : (type_it = (g_object_unref (type_it), NULL))), (last_type == NULL ? NULL : (last_type = (g_object_unref (last_type), NULL))), _tmp3);
				}
				if (vala_type_reference_get_data_type (type) == NULL && vala_type_reference_get_type_parameter (type) == NULL) {
					if (!null_found) {
						null_found = TRUE;
						if (val_struct_found || enum_found) {
							vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE);
							(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
							break;
						}
					}
				} else {
					if (VALA_IS_CLASS (vala_type_reference_get_data_type (type)) || VALA_IS_INTERFACE (vala_type_reference_get_data_type (type))) {
						if (!class_or_iface_found) {
							class_or_iface_found = TRUE;
							if (type_param_found || ref_struct_found || val_struct_found || enum_found || callback_found) {
								vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE);
								(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
								break;
							}
						}
					} else {
						if (vala_type_reference_get_type_parameter (type) != NULL) {
							if (!type_param_found) {
								type_param_found = TRUE;
								if (class_or_iface_found || ref_struct_found || val_struct_found || enum_found || callback_found) {
									vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE);
									(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
									break;
								}
							}
						} else {
							if (VALA_IS_STRUCT (vala_type_reference_get_data_type (type))) {
								ValaStruct* _tmp4;
								ValaStruct* st;
								_tmp4 = NULL;
								st = (_tmp4 = VALA_STRUCT (vala_type_reference_get_data_type (type)), (_tmp4 == NULL ? NULL : g_object_ref (_tmp4)));
								if (vala_data_type_is_reference_type (VALA_DATA_TYPE (st))) {
									if (!ref_struct_found) {
										ref_struct_found = TRUE;
										if (class_or_iface_found || type_param_found || val_struct_found || enum_found || callback_found) {
											vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE);
											(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
											(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
											break;
										}
									}
								} else {
									if (!val_struct_found) {
										val_struct_found = TRUE;
										if (class_or_iface_found || type_param_found || ref_struct_found || enum_found || callback_found) {
											vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE);
											(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
											(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
											break;
										}
									}
								}
								(st == NULL ? NULL : (st = (g_object_unref (st), NULL)));
							} else {
								if (VALA_IS_ENUM (vala_type_reference_get_data_type (type))) {
									if (!enum_found) {
										enum_found = TRUE;
										if (class_or_iface_found || type_param_found || ref_struct_found || val_struct_found) {
											vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE);
											(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
											break;
										}
									}
								} else {
									if (VALA_IS_CALLBACK (vala_type_reference_get_data_type (type))) {
										if (!callback_found) {
											callback_found = TRUE;
											if (class_or_iface_found || type_param_found || ref_struct_found || val_struct_found || enum_found) {
												vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE);
												(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
												break;
											}
										}
									} else {
										ValaTypeReference* _tmp5;
										char* _tmp7;
										char* _tmp6;
										ValaTypeReference* _tmp8;
										_tmp5 = NULL;
										base_type = (_tmp5 = vala_type_reference_new (), (base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL))), _tmp5);
										vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE);
										_tmp7 = NULL;
										_tmp6 = NULL;
										vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (type)), (_tmp7 = g_strdup_printf ("internal error: unsupported type `%s'", (_tmp6 = vala_type_reference_to_string (type)))));
										(_tmp7 = (g_free (_tmp7), NULL));
										(_tmp6 = (g_free (_tmp6), NULL));
										_tmp8 = NULL;
										return (_tmp8 = base_type, (type == NULL ? NULL : (type = (g_object_unref (type), NULL))), (type_it == NULL ? NULL : (type_it = (g_object_unref (type_it), NULL))), (last_type == NULL ? NULL : (last_type = (g_object_unref (last_type), NULL))), _tmp8);
									}
								}
							}
						}
					}
				}
				if (base_type == NULL) {
					ValaTypeReference* _tmp9;
					_tmp9 = NULL;
					base_type = (_tmp9 = vala_type_reference_new (), (base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL))), _tmp9);
					vala_type_reference_set_data_type (base_type, vala_type_reference_get_data_type (type));
					vala_type_reference_set_type_parameter (base_type, vala_type_reference_get_type_parameter (type));
					vala_type_reference_set_non_null (base_type, vala_type_reference_get_non_null (type));
					vala_type_reference_set_is_null (base_type, vala_type_reference_get_is_null (type));
					vala_type_reference_set_transfers_ownership (base_type, vala_type_reference_get_transfers_ownership (type));
				} else {
					if (vala_type_reference_get_data_type (base_type) != vala_type_reference_get_data_type (type)) {
						if (vala_semantic_analyzer_is_type_compatible (self, type, base_type)) {
						} else {
							if (vala_semantic_analyzer_is_type_compatible (self, base_type, type)) {
								vala_type_reference_set_data_type (base_type, vala_type_reference_get_data_type (type));
							} else {
								vala_code_node_set_error (VALA_CODE_NODE (base_type), TRUE);
								(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
								break;
							}
						}
					}
					vala_type_reference_set_non_null (base_type, vala_type_reference_get_non_null (base_type) && vala_type_reference_get_non_null (type));
					vala_type_reference_set_is_null (base_type, vala_type_reference_get_is_null (base_type) && vala_type_reference_get_is_null (type));
					/* if one subexpression transfers ownership, all subexpressions must transfer ownership
					 FIXME add ref calls to subexpressions that don't transfer ownership*/
					vala_type_reference_set_transfers_ownership (base_type, vala_type_reference_get_transfers_ownership (base_type) || vala_type_reference_get_transfers_ownership (type));
				}
				(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
			}
		}
		(type_it == NULL ? NULL : (type_it = (g_object_unref (type_it), NULL)));
	}
	if (base_type != NULL && vala_code_node_get_error (VALA_CODE_NODE (base_type))) {
		char* _tmp12;
		char* _tmp11;
		char* _tmp10;
		_tmp12 = NULL;
		_tmp11 = NULL;
		_tmp10 = NULL;
		vala_report_error (source_reference, (_tmp12 = g_strdup_printf ("`%s' is incompatible with `%s'", (_tmp10 = vala_type_reference_to_string (last_type)), (_tmp11 = vala_type_reference_to_string (base_type)))));
		(_tmp12 = (g_free (_tmp12), NULL));
		(_tmp11 = (g_free (_tmp11), NULL));
		(_tmp10 = (g_free (_tmp10), NULL));
	}
	_tmp13 = NULL;
	return (_tmp13 = base_type, (last_type == NULL ? NULL : (last_type = (g_object_unref (last_type), NULL))), _tmp13);
	(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
	(last_type == NULL ? NULL : (last_type = (g_object_unref (last_type), NULL)));
}


static void vala_semantic_analyzer_real_visit_conditional_expression (ValaCodeVisitor* base, ValaConditionalExpression* expr) {
	ValaSemanticAnalyzer * self;
	GeeList* types;
	ValaTypeReference* _tmp0;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_CONDITIONAL_EXPRESSION (expr));
	if (vala_type_reference_get_data_type (vala_expression_get_static_type (vala_conditional_expression_get_condition (expr))) != vala_type_reference_get_data_type (self->priv->bool_type)) {
		vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_conditional_expression_get_condition (expr))), "Condition must be boolean");
		return;
	}
	/* FIXME: support memory management */
	types = GEE_LIST (gee_array_list_new (VALA_TYPE_TYPE_REFERENCE, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
	gee_collection_add (GEE_COLLECTION (types), vala_expression_get_static_type (vala_conditional_expression_get_true_expression (expr)));
	gee_collection_add (GEE_COLLECTION (types), vala_expression_get_static_type (vala_conditional_expression_get_false_expression (expr)));
	_tmp0 = NULL;
	vala_expression_set_static_type (VALA_EXPRESSION (expr), (_tmp0 = vala_semantic_analyzer_compute_common_base_type (self, GEE_COLLECTION (types), vala_code_node_get_source_reference (VALA_CODE_NODE (expr)))));
	(_tmp0 == NULL ? NULL : (_tmp0 = (g_object_unref (_tmp0), NULL)));
	(types == NULL ? NULL : (types = (g_object_unref (types), NULL)));
}


static char* vala_semantic_analyzer_get_lambda_name (ValaSemanticAnalyzer* self) {
	char* result;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	result = g_strdup_printf ("__lambda%d", self->priv->next_lambda_id);
	self->priv->next_lambda_id++;
	return result;
	(result = (g_free (result), NULL));
}


static ValaMethod* vala_semantic_analyzer_find_current_method (ValaSemanticAnalyzer* self) {
	ValaSymbol* _tmp0;
	ValaSymbol* sym;
	gpointer _tmp5;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), NULL);
	_tmp0 = NULL;
	sym = (_tmp0 = self->priv->current_symbol, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
	while (sym != NULL) {
		ValaSymbol* _tmp4;
		ValaSymbol* _tmp3;
		if (VALA_IS_METHOD (sym)) {
			ValaMethod* _tmp1;
			ValaMethod* _tmp2;
			_tmp1 = NULL;
			_tmp2 = NULL;
			return (_tmp2 = (_tmp1 = VALA_METHOD (sym), (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), _tmp2);
		}
		_tmp4 = NULL;
		_tmp3 = NULL;
		sym = (_tmp4 = (_tmp3 = vala_symbol_get_parent_symbol (sym), (_tmp3 == NULL ? NULL : g_object_ref (_tmp3))), (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), _tmp4);
	}
	return (_tmp5 = NULL, (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), _tmp5);
	(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
}


static gboolean vala_semantic_analyzer_is_in_constructor (ValaSemanticAnalyzer* self) {
	ValaSymbol* _tmp0;
	ValaSymbol* sym;
	gboolean _tmp4;
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	_tmp0 = NULL;
	sym = (_tmp0 = self->priv->current_symbol, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
	while (sym != NULL) {
		ValaSymbol* _tmp3;
		ValaSymbol* _tmp2;
		if (VALA_IS_CONSTRUCTOR (sym)) {
			gboolean _tmp1;
			return (_tmp1 = TRUE, (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), _tmp1);
		}
		_tmp3 = NULL;
		_tmp2 = NULL;
		sym = (_tmp3 = (_tmp2 = vala_symbol_get_parent_symbol (sym), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2))), (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), _tmp3);
	}
	return (_tmp4 = FALSE, (sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL))), _tmp4);
	(sym == NULL ? NULL : (sym = (g_object_unref (sym), NULL)));
}


static void vala_semantic_analyzer_real_visit_lambda_expression (ValaCodeVisitor* base, ValaLambdaExpression* l) {
	ValaSemanticAnalyzer * self;
	gboolean in_instance_method;
	ValaMethod* current_method;
	ValaCallback* _tmp0;
	ValaCallback* cb;
	ValaMethod* _tmp2;
	char* _tmp1;
	GeeCollection* lambda_params;
	GeeIterator* lambda_param_it;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_LAMBDA_EXPRESSION (l));
	if (vala_expression_get_expected_type (VALA_EXPRESSION (l)) == NULL || !(VALA_IS_CALLBACK (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (l)))))) {
		vala_code_node_set_error (VALA_CODE_NODE (l), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (l)), "lambda expression not allowed in this context");
		return;
	}
	in_instance_method = FALSE;
	current_method = vala_semantic_analyzer_find_current_method (self);
	if (current_method != NULL) {
		in_instance_method = vala_method_get_instance (current_method);
	} else {
		in_instance_method = vala_semantic_analyzer_is_in_constructor (self);
	}
	_tmp0 = NULL;
	cb = (_tmp0 = VALA_CALLBACK (vala_type_reference_get_data_type (vala_expression_get_expected_type (VALA_EXPRESSION (l)))), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
	_tmp2 = NULL;
	_tmp1 = NULL;
	vala_lambda_expression_set_method (l, (_tmp2 = vala_method_new ((_tmp1 = vala_semantic_analyzer_get_lambda_name (self)), vala_callback_get_return_type (cb), NULL)));
	(_tmp2 == NULL ? NULL : (_tmp2 = (g_object_unref (_tmp2), NULL)));
	(_tmp1 = (g_free (_tmp1), NULL));
	vala_method_set_instance (vala_lambda_expression_get_method (l), vala_callback_get_instance (cb) && in_instance_method);
	vala_symbol_set_owner (VALA_SYMBOL (vala_lambda_expression_get_method (l)), vala_symbol_get_scope (self->priv->current_symbol));
	lambda_params = vala_lambda_expression_get_parameters (l);
	lambda_param_it = gee_iterable_iterator (GEE_ITERABLE (lambda_params));
	{
		GeeCollection* cb_param_collection;
		GeeIterator* cb_param_it;
		cb_param_collection = vala_callback_get_parameters (cb);
		cb_param_it = gee_iterable_iterator (GEE_ITERABLE (cb_param_collection));
		while (gee_iterator_next (cb_param_it)) {
			ValaFormalParameter* cb_param;
			cb_param = gee_iterator_get (cb_param_it);
			{
				char* lambda_param;
				ValaFormalParameter* param;
				if (!gee_iterator_next (lambda_param_it)) {
					(cb_param == NULL ? NULL : (cb_param = (g_object_unref (cb_param), NULL)));
					break;
				}
				/* lambda expressions are allowed to have less parameters */
				lambda_param = gee_iterator_get (lambda_param_it);
				param = vala_formal_parameter_new (lambda_param, vala_formal_parameter_get_type_reference (cb_param), NULL);
				vala_method_add_parameter (vala_lambda_expression_get_method (l), param);
				(cb_param == NULL ? NULL : (cb_param = (g_object_unref (cb_param), NULL)));
				(lambda_param = (g_free (lambda_param), NULL));
				(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
			}
		}
		(cb_param_collection == NULL ? NULL : (cb_param_collection = (g_object_unref (cb_param_collection), NULL)));
		(cb_param_it == NULL ? NULL : (cb_param_it = (g_object_unref (cb_param_it), NULL)));
	}
	if (gee_iterator_next (lambda_param_it)) {
		/* lambda expressions may not expect more parameters */
		vala_code_node_set_error (VALA_CODE_NODE (l), TRUE);
		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (l)), "lambda expression: too many parameters");
		(current_method == NULL ? NULL : (current_method = (g_object_unref (current_method), NULL)));
		(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
		(lambda_params == NULL ? NULL : (lambda_params = (g_object_unref (lambda_params), NULL)));
		(lambda_param_it == NULL ? NULL : (lambda_param_it = (g_object_unref (lambda_param_it), NULL)));
		return;
	}
	if (vala_lambda_expression_get_expression_body (l) != NULL) {
		ValaBlock* block;
		block = vala_block_new (NULL);
		vala_scope_set_parent_scope (vala_symbol_get_scope (VALA_SYMBOL (block)), vala_symbol_get_scope (VALA_SYMBOL (vala_lambda_expression_get_method (l))));
		if (vala_type_reference_get_data_type (vala_method_get_return_type (vala_lambda_expression_get_method (l))) != NULL) {
			ValaReturnStatement* _tmp3;
			_tmp3 = NULL;
			vala_block_add_statement (block, VALA_STATEMENT ((_tmp3 = vala_return_statement_new (vala_lambda_expression_get_expression_body (l), vala_code_node_get_source_reference (VALA_CODE_NODE (l))))));
			(_tmp3 == NULL ? NULL : (_tmp3 = (g_object_unref (_tmp3), NULL)));
		} else {
			ValaExpressionStatement* _tmp4;
			_tmp4 = NULL;
			vala_block_add_statement (block, VALA_STATEMENT ((_tmp4 = vala_expression_statement_new (vala_lambda_expression_get_expression_body (l), vala_code_node_get_source_reference (VALA_CODE_NODE (l))))));
			(_tmp4 == NULL ? NULL : (_tmp4 = (g_object_unref (_tmp4), NULL)));
		}
		vala_method_set_body (vala_lambda_expression_get_method (l), block);
		(block == NULL ? NULL : (block = (g_object_unref (block), NULL)));
	} else {
		vala_method_set_body (vala_lambda_expression_get_method (l), vala_lambda_expression_get_statement_body (l));
	}
	vala_symbol_set_owner (VALA_SYMBOL (vala_method_get_body (vala_lambda_expression_get_method (l))), vala_symbol_get_scope (VALA_SYMBOL (vala_lambda_expression_get_method (l))));
	/* lambda expressions should be usable like MemberAccess of a method */
	vala_expression_set_symbol_reference (VALA_EXPRESSION (l), VALA_SYMBOL (vala_lambda_expression_get_method (l)));
	vala_code_node_accept_children (VALA_CODE_NODE (l), VALA_CODE_VISITOR (self));
	(current_method == NULL ? NULL : (current_method = (g_object_unref (current_method), NULL)));
	(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
	(lambda_params == NULL ? NULL : (lambda_params = (g_object_unref (lambda_params), NULL)));
	(lambda_param_it == NULL ? NULL : (lambda_param_it = (g_object_unref (lambda_param_it), NULL)));
}


static void vala_semantic_analyzer_real_visit_assignment (ValaCodeVisitor* base, ValaAssignment* a) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (base);
	g_return_if_fail (VALA_IS_ASSIGNMENT (a));
	vala_code_node_accept (VALA_CODE_NODE (vala_assignment_get_left (a)), VALA_CODE_VISITOR (self));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_assignment_get_left (a)))) {
		/* skip on error in inner expression*/
		vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
		return;
	}
	if (VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (a))) {
		ValaMemberAccess* _tmp0;
		ValaMemberAccess* ma;
		_tmp0 = NULL;
		ma = (_tmp0 = VALA_MEMBER_ACCESS (vala_assignment_get_left (a)), (_tmp0 == NULL ? NULL : g_object_ref (_tmp0)));
		if (!(VALA_IS_SIGNAL (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))) && vala_expression_get_static_type (VALA_EXPRESSION (ma)) == NULL) {
			vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), "unsupported lvalue in assignment");
			(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
			return;
		}
		if (vala_member_access_get_prototype_access (ma)) {
			char* _tmp2;
			char* _tmp1;
			vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
			_tmp2 = NULL;
			_tmp1 = NULL;
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), (_tmp2 = g_strdup_printf ("Access to instance member `%s' denied", (_tmp1 = vala_symbol_get_full_name (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))))));
			(_tmp2 = (g_free (_tmp2), NULL));
			(_tmp1 = (g_free (_tmp1), NULL));
			(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
			return;
		}
		if (vala_code_node_get_error (VALA_CODE_NODE (ma)) || vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)) == NULL) {
			vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
			(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
			return;
		}
		/* if no symbol found, skip this check */
		if (VALA_IS_SIGNAL (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))) {
			ValaSignal* _tmp3;
			ValaSignal* sig;
			ValaTypeReference* _tmp4;
			ValaCallback* _tmp5;
			_tmp3 = NULL;
			sig = (_tmp3 = VALA_SIGNAL (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma))), (_tmp3 == NULL ? NULL : g_object_ref (_tmp3)));
			_tmp4 = NULL;
			vala_expression_set_expected_type (vala_assignment_get_right (a), (_tmp4 = vala_type_reference_new ()));
			(_tmp4 == NULL ? NULL : (_tmp4 = (g_object_unref (_tmp4), NULL)));
			_tmp5 = NULL;
			vala_type_reference_set_data_type (vala_expression_get_expected_type (vala_assignment_get_right (a)), VALA_DATA_TYPE ((_tmp5 = vala_signal_get_callback (sig))));
			(_tmp5 = (g_object_unref (_tmp5), NULL));
			(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
		} else {
			vala_expression_set_expected_type (vala_assignment_get_right (a), vala_expression_get_static_type (VALA_EXPRESSION (ma)));
		}
		(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
	} else {
		if (VALA_IS_ELEMENT_ACCESS (vala_assignment_get_left (a))) {
		} else {
			/* do nothing*/
			if (VALA_IS_POINTER_INDIRECTION (vala_assignment_get_left (a))) {
			} else {
				/* do nothing*/
				vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), "unsupported lvalue in assignment");
				return;
			}
		}
	}
	vala_code_node_accept (VALA_CODE_NODE (vala_assignment_get_right (a)), VALA_CODE_VISITOR (self));
	if (vala_code_node_get_error (VALA_CODE_NODE (vala_assignment_get_right (a)))) {
		/* skip on error in inner expression*/
		vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
		return;
	}
	if (vala_assignment_get_operator (a) != VALA_ASSIGNMENT_OPERATOR_SIMPLE && VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (a))) {
		ValaMemberAccess* _tmp6;
		ValaMemberAccess* ma;
		/* transform into simple assignment
		 FIXME: only do this if the backend doesn't support
		 the assignment natively*/
		_tmp6 = NULL;
		ma = (_tmp6 = VALA_MEMBER_ACCESS (vala_assignment_get_left (a)), (_tmp6 == NULL ? NULL : g_object_ref (_tmp6)));
		if (!(VALA_IS_SIGNAL (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma))))) {
			ValaMemberAccess* old_value;
			ValaParenthesizedExpression* _tmp7;
			ValaBinaryExpression* _tmp8;
			ValaBinaryExpression* bin;
			old_value = vala_member_access_new (vala_member_access_get_inner (ma), vala_member_access_get_member_name (ma), NULL);
			_tmp7 = NULL;
			_tmp8 = NULL;
			bin = (_tmp8 = vala_binary_expression_new (VALA_BINARY_OPERATOR_PLUS, VALA_EXPRESSION (old_value), VALA_EXPRESSION ((_tmp7 = vala_parenthesized_expression_new (vala_assignment_get_right (a), vala_code_node_get_source_reference (VALA_CODE_NODE (vala_assignment_get_right (a)))))), NULL), (_tmp7 == NULL ? NULL : (_tmp7 = (g_object_unref (_tmp7), NULL))), _tmp8);
			if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_BITWISE_OR) {
				vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_OR);
			} else {
				if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_BITWISE_AND) {
					vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_AND);
				} else {
					if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_BITWISE_XOR) {
						vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_BITWISE_XOR);
					} else {
						if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_ADD) {
							vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_PLUS);
						} else {
							if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_SUB) {
								vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MINUS);
							} else {
								if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_MUL) {
									vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MUL);
								} else {
									if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_DIV) {
										vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_DIV);
									} else {
										if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_PERCENT) {
											vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_MOD);
										} else {
											if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_SHIFT_LEFT) {
												vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_SHIFT_LEFT);
											} else {
												if (vala_assignment_get_operator (a) == VALA_ASSIGNMENT_OPERATOR_SHIFT_RIGHT) {
													vala_binary_expression_set_operator (bin, VALA_BINARY_OPERATOR_SHIFT_RIGHT);
												}
											}
										}
									}
								}
							}
						}
					}
				}
			}
			vala_assignment_set_right (a, VALA_EXPRESSION (bin));
			vala_code_node_accept (VALA_CODE_NODE (vala_assignment_get_right (a)), VALA_CODE_VISITOR (self));
			vala_assignment_set_operator (a, VALA_ASSIGNMENT_OPERATOR_SIMPLE);
			(old_value == NULL ? NULL : (old_value = (g_object_unref (old_value), NULL)));
			(bin == NULL ? NULL : (bin = (g_object_unref (bin), NULL)));
		}
		(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
	}
	if (VALA_IS_MEMBER_ACCESS (vala_assignment_get_left (a))) {
		ValaMemberAccess* _tmp9;
		ValaMemberAccess* ma;
		_tmp9 = NULL;
		ma = (_tmp9 = VALA_MEMBER_ACCESS (vala_assignment_get_left (a)), (_tmp9 == NULL ? NULL : g_object_ref (_tmp9)));
		if (VALA_IS_SIGNAL (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))) {
			ValaSignal* _tmp10;
			ValaSignal* sig;
			ValaMethod* _tmp11;
			ValaMethod* m;
			_tmp10 = NULL;
			sig = (_tmp10 = VALA_SIGNAL (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma))), (_tmp10 == NULL ? NULL : g_object_ref (_tmp10)));
			if (vala_expression_get_symbol_reference (vala_assignment_get_right (a)) == NULL) {
				vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_assignment_get_right (a))), "unsupported expression for signal handler");
				(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
				(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
				return;
			}
			_tmp11 = NULL;
			m = (_tmp11 = VALA_METHOD (vala_expression_get_symbol_reference (vala_assignment_get_right (a))), (_tmp11 == NULL ? NULL : g_object_ref (_tmp11)));
			if (vala_method_get_instance (m) && vala_symbol_get_access (VALA_SYMBOL (m)) != VALA_SYMBOL_ACCESSIBILITY_PRIVATE) {
				/* TODO: generate wrapper function */
				vala_code_node_set_error (VALA_CODE_NODE (ma), TRUE);
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (vala_assignment_get_right (a))), "public instance methods not yet supported as signal handlers");
				(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
				return;
			}
			if (vala_method_get_instance (m)) {
				/* instance signal handlers must have the self
				 * parameter at the end
				 * do not use G_CONNECT_SWAPPED as this would
				 * rearrange the parameters for instance
				 * methods and non-instance methods
				 */
				vala_method_set_instance_last (m, TRUE);
			}
			(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
			(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
		} else {
			if (VALA_IS_PROPERTY (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma)))) {
				ValaProperty* _tmp12;
				ValaProperty* prop;
				_tmp12 = NULL;
				prop = (_tmp12 = VALA_PROPERTY (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma))), (_tmp12 == NULL ? NULL : g_object_ref (_tmp12)));
				if (vala_property_get_set_accessor (prop) == NULL) {
					char* _tmp14;
					char* _tmp13;
					vala_code_node_set_error (VALA_CODE_NODE (ma), TRUE);
					_tmp14 = NULL;
					_tmp13 = NULL;
					vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (ma)), (_tmp14 = g_strdup_printf ("Property `%s' is read-only", (_tmp13 = vala_symbol_get_full_name (VALA_SYMBOL (prop))))));
					(_tmp14 = (g_free (_tmp14), NULL));
					(_tmp13 = (g_free (_tmp13), NULL));
					(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
					(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
					return;
				}
				(prop == NULL ? NULL : (prop = (g_object_unref (prop), NULL)));
			} else {
				if (VALA_IS_VARIABLE_DECLARATOR (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma))) && vala_expression_get_static_type (vala_assignment_get_right (a)) == NULL) {
					ValaVariableDeclarator* _tmp15;
					ValaVariableDeclarator* decl;
					ValaMemberAccess* _tmp16;
					ValaMemberAccess* right_ma;
					_tmp15 = NULL;
					decl = (_tmp15 = VALA_VARIABLE_DECLARATOR (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma))), (_tmp15 == NULL ? NULL : g_object_ref (_tmp15)));
					_tmp16 = NULL;
					right_ma = (_tmp16 = VALA_MEMBER_ACCESS (vala_assignment_get_right (a)), (_tmp16 == NULL ? NULL : g_object_ref (_tmp16)));
					if (VALA_IS_METHOD (vala_expression_get_symbol_reference (VALA_EXPRESSION (right_ma))) && VALA_IS_CALLBACK (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl)))) {
						ValaMethod* _tmp17;
						ValaMethod* m;
						ValaCallback* _tmp18;
						ValaCallback* cb;
						_tmp17 = NULL;
						m = (_tmp17 = VALA_METHOD (vala_expression_get_symbol_reference (VALA_EXPRESSION (right_ma))), (_tmp17 == NULL ? NULL : g_object_ref (_tmp17)));
						_tmp18 = NULL;
						cb = (_tmp18 = VALA_CALLBACK (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl))), (_tmp18 == NULL ? NULL : g_object_ref (_tmp18)));
						/* check whether method matches callback type */
						if (!vala_callback_matches_method (cb, m)) {
							char* _tmp21;
							char* _tmp20;
							char* _tmp19;
							vala_code_node_set_error (VALA_CODE_NODE (decl), TRUE);
							_tmp21 = NULL;
							_tmp20 = NULL;
							_tmp19 = NULL;
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), (_tmp21 = g_strdup_printf ("declaration of method `%s' doesn't match declaration of callback `%s'", (_tmp19 = vala_symbol_get_full_name (VALA_SYMBOL (m))), (_tmp20 = vala_symbol_get_full_name (VALA_SYMBOL (cb))))));
							(_tmp21 = (g_free (_tmp21), NULL));
							(_tmp20 = (g_free (_tmp20), NULL));
							(_tmp19 = (g_free (_tmp19), NULL));
							(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
							(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
							(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
							(right_ma == NULL ? NULL : (right_ma = (g_object_unref (right_ma), NULL)));
							(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
							return;
						}
						vala_expression_set_static_type (vala_assignment_get_right (a), vala_variable_declarator_get_type_reference (decl));
						(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
						(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
					} else {
						vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), "Assignment: Invalid callback assignment attempt");
						(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
						(right_ma == NULL ? NULL : (right_ma = (g_object_unref (right_ma), NULL)));
						(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
						return;
					}
					(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
					(right_ma == NULL ? NULL : (right_ma = (g_object_unref (right_ma), NULL)));
				} else {
					if (VALA_IS_FIELD (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma))) && vala_expression_get_static_type (vala_assignment_get_right (a)) == NULL) {
						ValaField* _tmp22;
						ValaField* f;
						ValaMemberAccess* _tmp23;
						ValaMemberAccess* right_ma;
						_tmp22 = NULL;
						f = (_tmp22 = VALA_FIELD (vala_expression_get_symbol_reference (VALA_EXPRESSION (ma))), (_tmp22 == NULL ? NULL : g_object_ref (_tmp22)));
						_tmp23 = NULL;
						right_ma = (_tmp23 = VALA_MEMBER_ACCESS (vala_assignment_get_right (a)), (_tmp23 == NULL ? NULL : g_object_ref (_tmp23)));
						if (VALA_IS_METHOD (vala_expression_get_symbol_reference (VALA_EXPRESSION (right_ma))) && VALA_IS_CALLBACK (vala_type_reference_get_data_type (vala_field_get_type_reference (f)))) {
							ValaMethod* _tmp24;
							ValaMethod* m;
							ValaCallback* _tmp25;
							ValaCallback* cb;
							_tmp24 = NULL;
							m = (_tmp24 = VALA_METHOD (vala_expression_get_symbol_reference (VALA_EXPRESSION (right_ma))), (_tmp24 == NULL ? NULL : g_object_ref (_tmp24)));
							_tmp25 = NULL;
							cb = (_tmp25 = VALA_CALLBACK (vala_type_reference_get_data_type (vala_field_get_type_reference (f))), (_tmp25 == NULL ? NULL : g_object_ref (_tmp25)));
							/* check whether method matches callback type */
							if (!vala_callback_matches_method (cb, m)) {
								char* _tmp28;
								char* _tmp27;
								char* _tmp26;
								vala_code_node_set_error (VALA_CODE_NODE (f), TRUE);
								_tmp28 = NULL;
								_tmp27 = NULL;
								_tmp26 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), (_tmp28 = g_strdup_printf ("declaration of method `%s' doesn't match declaration of callback `%s'", (_tmp26 = vala_symbol_get_full_name (VALA_SYMBOL (m))), (_tmp27 = vala_symbol_get_full_name (VALA_SYMBOL (cb))))));
								(_tmp28 = (g_free (_tmp28), NULL));
								(_tmp27 = (g_free (_tmp27), NULL));
								(_tmp26 = (g_free (_tmp26), NULL));
								(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
								(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
								(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
								(right_ma == NULL ? NULL : (right_ma = (g_object_unref (right_ma), NULL)));
								(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
								return;
							}
							vala_expression_set_static_type (vala_assignment_get_right (a), vala_field_get_type_reference (f));
							(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
							(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
						} else {
							vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
							vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), "Assignment: Invalid callback assignment attempt");
							(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
							(right_ma == NULL ? NULL : (right_ma = (g_object_unref (right_ma), NULL)));
							(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
							return;
						}
						(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
						(right_ma == NULL ? NULL : (right_ma = (g_object_unref (right_ma), NULL)));
					} else {
						if (vala_expression_get_static_type (vala_assignment_get_left (a)) != NULL && vala_expression_get_static_type (vala_assignment_get_right (a)) != NULL) {
							if (!vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (vala_assignment_get_right (a)), vala_expression_get_static_type (vala_assignment_get_left (a)))) {
								char* _tmp31;
								char* _tmp30;
								char* _tmp29;
								/* if there was an error on either side,
								 * i.e. a.{left|right}.static_type == null, skip type check */
								vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
								_tmp31 = NULL;
								_tmp30 = NULL;
								_tmp29 = NULL;
								vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), (_tmp31 = g_strdup_printf ("Assignment: Cannot convert from `%s' to `%s'", (_tmp29 = vala_type_reference_to_string (vala_expression_get_static_type (vala_assignment_get_right (a)))), (_tmp30 = vala_type_reference_to_string (vala_expression_get_static_type (vala_assignment_get_left (a)))))));
								(_tmp31 = (g_free (_tmp31), NULL));
								(_tmp30 = (g_free (_tmp30), NULL));
								(_tmp29 = (g_free (_tmp29), NULL));
								(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
								return;
							}
							if (vala_semantic_analyzer_get_memory_management (self)) {
								if (vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (vala_assignment_get_right (a)))) {
									/* rhs transfers ownership of the expression */
									if (!vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_assignment_get_left (a)))) {
										/* lhs doesn't own the value */
										vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
										vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), "Invalid assignment from owned expression to unowned variable");
									}
								} else {
									if (vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_assignment_get_left (a)))) {
									}
								}
							}
						}
					}
				}
			}
		}
		(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
	} else {
		/* lhs wants to own the value
		 * rhs doesn't transfer the ownership
		 * code generator needs to add reference
		 * increment calls */
		if (VALA_IS_ELEMENT_ACCESS (vala_assignment_get_left (a))) {
			ValaElementAccess* _tmp32;
			ValaElementAccess* ea;
			_tmp32 = NULL;
			ea = (_tmp32 = VALA_ELEMENT_ACCESS (vala_assignment_get_left (a)), (_tmp32 == NULL ? NULL : g_object_ref (_tmp32)));
			if (!vala_semantic_analyzer_is_type_compatible (self, vala_expression_get_static_type (vala_assignment_get_right (a)), vala_expression_get_static_type (vala_assignment_get_left (a)))) {
				char* _tmp35;
				char* _tmp34;
				char* _tmp33;
				/* if there was an error on either side,
				 * i.e. a.{left|right}.static_type == null, skip type check */
				vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
				_tmp35 = NULL;
				_tmp34 = NULL;
				_tmp33 = NULL;
				vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), (_tmp35 = g_strdup_printf ("Assignment: Cannot convert from `%s' to `%s'", (_tmp33 = vala_type_reference_to_string (vala_expression_get_static_type (vala_assignment_get_right (a)))), (_tmp34 = vala_type_reference_to_string (vala_expression_get_static_type (vala_assignment_get_left (a)))))));
				(_tmp35 = (g_free (_tmp35), NULL));
				(_tmp34 = (g_free (_tmp34), NULL));
				(_tmp33 = (g_free (_tmp33), NULL));
				(ea == NULL ? NULL : (ea = (g_object_unref (ea), NULL)));
				return;
			}
			if (vala_semantic_analyzer_get_memory_management (self)) {
				if (vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (vala_assignment_get_right (a)))) {
					GeeList* args;
					ValaTypeReference* element_type;
					/* rhs transfers ownership of the expression */
					args = vala_type_reference_get_type_arguments (vala_expression_get_static_type (vala_element_access_get_container (ea)));
					if (gee_collection_get_size (GEE_COLLECTION (args)) != 1) {
						vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (ea)), "internal error: array reference without type arguments");
						(args == NULL ? NULL : (args = (g_object_unref (args), NULL)));
						(ea == NULL ? NULL : (ea = (g_object_unref (ea), NULL)));
						return;
					}
					element_type = gee_list_get (args, 0);
					if (!vala_type_reference_get_takes_ownership (element_type)) {
						/* lhs doesn't own the value */
						vala_code_node_set_error (VALA_CODE_NODE (a), TRUE);
						vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (a)), "Invalid assignment from owned expression to unowned variable");
						(args == NULL ? NULL : (args = (g_object_unref (args), NULL)));
						(element_type == NULL ? NULL : (element_type = (g_object_unref (element_type), NULL)));
						(ea == NULL ? NULL : (ea = (g_object_unref (ea), NULL)));
						return;
					}
					(args == NULL ? NULL : (args = (g_object_unref (args), NULL)));
					(element_type == NULL ? NULL : (element_type = (g_object_unref (element_type), NULL)));
				} else {
					if (vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_assignment_get_left (a)))) {
					}
				}
			}
			(ea == NULL ? NULL : (ea = (g_object_unref (ea), NULL)));
		} else {
			return;
		}
	}
	/* lhs wants to own the value
	 * rhs doesn't transfer the ownership
	 * code generator needs to add reference
	 * increment calls */
	if (vala_expression_get_static_type (vala_assignment_get_left (a)) != NULL) {
		ValaTypeReference* _tmp36;
		_tmp36 = NULL;
		vala_expression_set_static_type (VALA_EXPRESSION (a), (_tmp36 = vala_type_reference_copy (vala_expression_get_static_type (vala_assignment_get_left (a)))));
		(_tmp36 = (g_object_unref (_tmp36), NULL));
		if (VALA_IS_EXPRESSION_STATEMENT (vala_code_node_get_parent_node (VALA_CODE_NODE (a)))) {
			/* Gee.List.get () transfers ownership but void function Gee.List.set () doesn't*/
			vala_type_reference_set_transfers_ownership (vala_expression_get_static_type (VALA_EXPRESSION (a)), FALSE);
		}
	} else {
		vala_expression_set_static_type (VALA_EXPRESSION (a), NULL);
	}
}


gboolean vala_semantic_analyzer_get_memory_management (ValaSemanticAnalyzer* self) {
	g_return_val_if_fail (VALA_IS_SEMANTIC_ANALYZER (self), FALSE);
	return self->priv->_memory_management;
}


void vala_semantic_analyzer_set_memory_management (ValaSemanticAnalyzer* self, gboolean value) {
	g_return_if_fail (VALA_IS_SEMANTIC_ANALYZER (self));
	self->priv->_memory_management = value;
}


static void vala_semantic_analyzer_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (object);
	switch (property_id) {
		case VALA_SEMANTIC_ANALYZER_MEMORY_MANAGEMENT:
		g_value_set_boolean (value, vala_semantic_analyzer_get_memory_management (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_semantic_analyzer_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (object);
	switch (property_id) {
		case VALA_SEMANTIC_ANALYZER_MEMORY_MANAGEMENT:
		vala_semantic_analyzer_set_memory_management (self, g_value_get_boolean (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_semantic_analyzer_class_init (ValaSemanticAnalyzerClass * klass) {
	vala_semantic_analyzer_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaSemanticAnalyzerPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_semantic_analyzer_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_semantic_analyzer_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_semantic_analyzer_dispose;
	VALA_CODE_VISITOR_CLASS (klass)->visit_source_file = vala_semantic_analyzer_real_visit_source_file;
	VALA_CODE_VISITOR_CLASS (klass)->visit_class = vala_semantic_analyzer_real_visit_class;
	VALA_CODE_VISITOR_CLASS (klass)->visit_struct = vala_semantic_analyzer_real_visit_struct;
	VALA_CODE_VISITOR_CLASS (klass)->visit_interface = vala_semantic_analyzer_real_visit_interface;
	VALA_CODE_VISITOR_CLASS (klass)->visit_callback = vala_semantic_analyzer_real_visit_callback;
	VALA_CODE_VISITOR_CLASS (klass)->visit_constant = vala_semantic_analyzer_real_visit_constant;
	VALA_CODE_VISITOR_CLASS (klass)->visit_field = vala_semantic_analyzer_real_visit_field;
	VALA_CODE_VISITOR_CLASS (klass)->visit_method = vala_semantic_analyzer_real_visit_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_creation_method = vala_semantic_analyzer_real_visit_creation_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_formal_parameter = vala_semantic_analyzer_real_visit_formal_parameter;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property = vala_semantic_analyzer_real_visit_property;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property_accessor = vala_semantic_analyzer_real_visit_property_accessor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_signal = vala_semantic_analyzer_real_visit_signal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_constructor = vala_semantic_analyzer_real_visit_constructor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_destructor = vala_semantic_analyzer_real_visit_destructor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_named_argument = vala_semantic_analyzer_real_visit_named_argument;
	VALA_CODE_VISITOR_CLASS (klass)->visit_block = vala_semantic_analyzer_real_visit_block;
	VALA_CODE_VISITOR_CLASS (klass)->visit_variable_declarator = vala_semantic_analyzer_real_visit_variable_declarator;
	VALA_CODE_VISITOR_CLASS (klass)->visit_initializer_list = vala_semantic_analyzer_real_visit_initializer_list;
	VALA_CODE_VISITOR_CLASS (klass)->visit_expression_statement = vala_semantic_analyzer_real_visit_expression_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_if_statement = vala_semantic_analyzer_real_visit_if_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_switch_section = vala_semantic_analyzer_real_visit_switch_section;
	VALA_CODE_VISITOR_CLASS (klass)->visit_while_statement = vala_semantic_analyzer_real_visit_while_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_for_statement = vala_semantic_analyzer_real_visit_for_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_foreach_statement = vala_semantic_analyzer_real_visit_foreach_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_return_statement = vala_semantic_analyzer_real_visit_return_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_throw_statement = vala_semantic_analyzer_real_visit_throw_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_try_statement = vala_semantic_analyzer_real_visit_try_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_catch_clause = vala_semantic_analyzer_real_visit_catch_clause;
	VALA_CODE_VISITOR_CLASS (klass)->visit_lock_statement = vala_semantic_analyzer_real_visit_lock_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_array_creation_expression = vala_semantic_analyzer_real_visit_array_creation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_boolean_literal = vala_semantic_analyzer_real_visit_boolean_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_character_literal = vala_semantic_analyzer_real_visit_character_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_integer_literal = vala_semantic_analyzer_real_visit_integer_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_real_literal = vala_semantic_analyzer_real_visit_real_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_string_literal = vala_semantic_analyzer_real_visit_string_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_null_literal = vala_semantic_analyzer_real_visit_null_literal;
	VALA_CODE_VISITOR_CLASS (klass)->visit_literal_expression = vala_semantic_analyzer_real_visit_literal_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_parenthesized_expression = vala_semantic_analyzer_real_visit_parenthesized_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_member_access = vala_semantic_analyzer_real_visit_member_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_invocation_expression = vala_semantic_analyzer_real_visit_invocation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_element_access = vala_semantic_analyzer_real_visit_element_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_base_access = vala_semantic_analyzer_real_visit_base_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_postfix_expression = vala_semantic_analyzer_real_visit_postfix_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_object_creation_expression = vala_semantic_analyzer_real_visit_object_creation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_sizeof_expression = vala_semantic_analyzer_real_visit_sizeof_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_typeof_expression = vala_semantic_analyzer_real_visit_typeof_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_unary_expression = vala_semantic_analyzer_real_visit_unary_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_cast_expression = vala_semantic_analyzer_real_visit_cast_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_pointer_indirection = vala_semantic_analyzer_real_visit_pointer_indirection;
	VALA_CODE_VISITOR_CLASS (klass)->visit_addressof_expression = vala_semantic_analyzer_real_visit_addressof_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_reference_transfer_expression = vala_semantic_analyzer_real_visit_reference_transfer_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_binary_expression = vala_semantic_analyzer_real_visit_binary_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_type_check = vala_semantic_analyzer_real_visit_type_check;
	VALA_CODE_VISITOR_CLASS (klass)->visit_conditional_expression = vala_semantic_analyzer_real_visit_conditional_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_lambda_expression = vala_semantic_analyzer_real_visit_lambda_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_assignment = vala_semantic_analyzer_real_visit_assignment;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SEMANTIC_ANALYZER_MEMORY_MANAGEMENT, g_param_spec_boolean ("memory-management", "memory-management", "memory-management", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_semantic_analyzer_init (ValaSemanticAnalyzer * self) {
	self->priv = VALA_SEMANTIC_ANALYZER_GET_PRIVATE (self);
	self->priv->next_lambda_id = 0;
	self->priv->binding_providers = GEE_COLLECTION (gee_array_list_new (VALA_TYPE_BINDING_PROVIDER, ((GBoxedCopyFunc) g_object_ref), g_object_unref, g_direct_equal));
}


static void vala_semantic_analyzer_dispose (GObject * obj) {
	ValaSemanticAnalyzer * self;
	self = VALA_SEMANTIC_ANALYZER (obj);
	(self->priv->context == NULL ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL)));
	(self->priv->root_symbol == NULL ? NULL : (self->priv->root_symbol = (g_object_unref (self->priv->root_symbol), NULL)));
	(self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL)));
	(self->priv->current_source_file == NULL ? NULL : (self->priv->current_source_file = (g_object_unref (self->priv->current_source_file), NULL)));
	(self->priv->current_return_type == NULL ? NULL : (self->priv->current_return_type = (g_object_unref (self->priv->current_return_type), NULL)));
	(self->priv->current_class == NULL ? NULL : (self->priv->current_class = (g_object_unref (self->priv->current_class), NULL)));
	(self->priv->current_struct == NULL ? NULL : (self->priv->current_struct = (g_object_unref (self->priv->current_struct), NULL)));
	(self->priv->current_using_directives == NULL ? NULL : (self->priv->current_using_directives = (g_object_unref (self->priv->current_using_directives), NULL)));
	(self->priv->bool_type == NULL ? NULL : (self->priv->bool_type = (g_object_unref (self->priv->bool_type), NULL)));
	(self->priv->string_type == NULL ? NULL : (self->priv->string_type = (g_object_unref (self->priv->string_type), NULL)));
	(self->priv->int_type == NULL ? NULL : (self->priv->int_type = (g_object_unref (self->priv->int_type), NULL)));
	(self->priv->uint_type == NULL ? NULL : (self->priv->uint_type = (g_object_unref (self->priv->uint_type), NULL)));
	(self->priv->ulong_type == NULL ? NULL : (self->priv->ulong_type = (g_object_unref (self->priv->ulong_type), NULL)));
	(self->priv->unichar_type == NULL ? NULL : (self->priv->unichar_type = (g_object_unref (self->priv->unichar_type), NULL)));
	(self->priv->type_type == NULL ? NULL : (self->priv->type_type = (g_object_unref (self->priv->type_type), NULL)));
	(self->priv->pointer_type == NULL ? NULL : (self->priv->pointer_type = (g_object_unref (self->priv->pointer_type), NULL)));
	(self->priv->object_type == NULL ? NULL : (self->priv->object_type = (g_object_unref (self->priv->object_type), NULL)));
	(self->priv->initially_unowned_type == NULL ? NULL : (self->priv->initially_unowned_type = (g_object_unref (self->priv->initially_unowned_type), NULL)));
	(self->priv->glist_type == NULL ? NULL : (self->priv->glist_type = (g_object_unref (self->priv->glist_type), NULL)));
	(self->priv->gslist_type == NULL ? NULL : (self->priv->gslist_type = (g_object_unref (self->priv->gslist_type), NULL)));
	(self->priv->gerror_type == NULL ? NULL : (self->priv->gerror_type = (g_object_unref (self->priv->gerror_type), NULL)));
	(self->priv->iterable_type == NULL ? NULL : (self->priv->iterable_type = (g_object_unref (self->priv->iterable_type), NULL)));
	(self->priv->iterator_type == NULL ? NULL : (self->priv->iterator_type = (g_object_unref (self->priv->iterator_type), NULL)));
	(self->priv->list_type == NULL ? NULL : (self->priv->list_type = (g_object_unref (self->priv->list_type), NULL)));
	(self->priv->map_type == NULL ? NULL : (self->priv->map_type = (g_object_unref (self->priv->map_type), NULL)));
	(self->priv->binding_providers == NULL ? NULL : (self->priv->binding_providers = (g_object_unref (self->priv->binding_providers), NULL)));
	G_OBJECT_CLASS (vala_semantic_analyzer_parent_class)->dispose (obj);
}


GType vala_semantic_analyzer_get_type (void) {
	static GType vala_semantic_analyzer_type_id = 0;
	if (G_UNLIKELY (vala_semantic_analyzer_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaSemanticAnalyzerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_semantic_analyzer_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaSemanticAnalyzer), 0, (GInstanceInitFunc) vala_semantic_analyzer_init };
		vala_semantic_analyzer_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaSemanticAnalyzer", &g_define_type_info, 0);
	}
	return vala_semantic_analyzer_type_id;
}




