/* valarealliteral.vala
 *
 * Copyright (C) 2006  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valarealliteral.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>

struct _ValaRealLiteralPrivate {
	char* _value;
};
#define VALA_REAL_LITERAL_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_REAL_LITERAL, ValaRealLiteralPrivate))
enum  {
	VALA_REAL_LITERAL_DUMMY_PROPERTY,
	VALA_REAL_LITERAL_VALUE
};
static void vala_real_literal_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gpointer vala_real_literal_parent_class = NULL;
static void vala_real_literal_dispose (GObject * obj);


/**
 * Creates a new real literal.
 *
 * @param r      literal value
 * @param source reference to source code
 * @return       newly created real literal
 */
ValaRealLiteral* vala_real_literal_new (const char* r, ValaSourceReference* source) {
	ValaRealLiteral * self;
	g_return_val_if_fail (source == NULL || VALA_IS_SOURCE_REFERENCE (source), NULL);
	self = g_object_newv (VALA_TYPE_REAL_LITERAL, 0, NULL);
	vala_real_literal_set_value (self, r);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source);
	return self;
}


static void vala_real_literal_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaRealLiteral * self;
	self = VALA_REAL_LITERAL (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_real_literal (visitor, self);
}


/**
 * Returns the type name of the value this literal represents.
 *
 * @return the name of literal type
 */
char* vala_real_literal_get_type_name (ValaRealLiteral* self) {
	g_return_val_if_fail (VALA_IS_REAL_LITERAL (self), NULL);
	if (g_str_has_suffix (vala_real_literal_get_value (self), "F")) {
		return g_strdup ("float");
	}
	return g_strdup ("double");
}


char* vala_real_literal_get_value (ValaRealLiteral* self) {
	g_return_val_if_fail (VALA_IS_REAL_LITERAL (self), NULL);
	return self->priv->_value;
}


void vala_real_literal_set_value (ValaRealLiteral* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (VALA_IS_REAL_LITERAL (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_value = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (self->priv->_value = (g_free (self->priv->_value), NULL)), _tmp2);
}


static void vala_real_literal_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaRealLiteral * self;
	self = VALA_REAL_LITERAL (object);
	switch (property_id) {
		case VALA_REAL_LITERAL_VALUE:
		g_value_set_string (value, vala_real_literal_get_value (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_real_literal_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaRealLiteral * self;
	self = VALA_REAL_LITERAL (object);
	switch (property_id) {
		case VALA_REAL_LITERAL_VALUE:
		vala_real_literal_set_value (self, g_value_get_string (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_real_literal_class_init (ValaRealLiteralClass * klass) {
	vala_real_literal_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaRealLiteralPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_real_literal_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_real_literal_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_real_literal_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_real_literal_real_accept;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_REAL_LITERAL_VALUE, g_param_spec_string ("value", "value", "value", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_real_literal_init (ValaRealLiteral * self) {
	self->priv = VALA_REAL_LITERAL_GET_PRIVATE (self);
}


static void vala_real_literal_dispose (GObject * obj) {
	ValaRealLiteral * self;
	self = VALA_REAL_LITERAL (obj);
	(self->priv->_value = (g_free (self->priv->_value), NULL));
	G_OBJECT_CLASS (vala_real_literal_parent_class)->dispose (obj);
}


GType vala_real_literal_get_type (void) {
	static GType vala_real_literal_type_id = 0;
	if (G_UNLIKELY (vala_real_literal_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaRealLiteralClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_real_literal_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaRealLiteral), 0, (GInstanceInitFunc) vala_real_literal_init };
		vala_real_literal_type_id = g_type_register_static (VALA_TYPE_LITERAL, "ValaRealLiteral", &g_define_type_info, 0);
	}
	return vala_real_literal_type_id;
}




