/* arraylist.vala
 *
 * Copyright (C) 2004-2005  Novell, Inc
 * Copyright (C) 2005  David Waite
 * Copyright (C) 2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include "arraylist.h"
#include <gee/arraylist.h>
#include <string.h>

typedef struct _GeeArrayListIterator GeeArrayListIterator;
typedef struct _GeeArrayListIteratorClass GeeArrayListIteratorClass;
typedef struct _GeeArrayListIteratorPrivate GeeArrayListIteratorPrivate;
struct _GeeArrayListPrivate {
	gpointer* _items;
	gint _items_length1;
	gint _size;
	GEqualFunc _equal_func;
	gint _stamp;
	GBoxedCopyFunc g_dup_func;
	GDestroyNotify g_destroy_func;
};
#define GEE_ARRAY_LIST_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_TYPE_ARRAY_LIST, GeeArrayListPrivate))
enum  {
	GEE_ARRAY_LIST_DUMMY_PROPERTY,
	GEE_ARRAY_LIST_SIZE,
	GEE_ARRAY_LIST_EQUAL_FUNC,
	GEE_ARRAY_LIST_G_DUP_FUNC,
	GEE_ARRAY_LIST_G_DESTROY_FUNC
};
static GeeIterator* gee_array_list_real_iterator (GeeIterable* base);
static gboolean gee_array_list_real_contains (GeeCollection* base, gpointer item);
static gint gee_array_list_real_index_of (GeeList* base, gpointer item);
static gpointer gee_array_list_real_get (GeeList* base, gint index);
static void gee_array_list_real_set (GeeList* base, gint index, gpointer item);
static gboolean gee_array_list_real_add (GeeCollection* base, gpointer item);
static void gee_array_list_real_insert (GeeList* base, gint index, gpointer item);
static gboolean gee_array_list_real_remove (GeeCollection* base, gpointer item);
static void gee_array_list_real_remove_at (GeeList* base, gint index);
static void gee_array_list_real_clear (GeeCollection* base);
static void gee_array_list_shift (GeeArrayList* self, gint start, gint delta);
static void gee_array_list_grow_if_needed (GeeArrayList* self, gint new_count);
static void gee_array_list_set_capacity (GeeArrayList* self, gint value);

#define GEE_ARRAY_LIST_TYPE_ITERATOR (gee_array_list_iterator_get_type ())
#define GEE_ARRAY_LIST_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GEE_ARRAY_LIST_TYPE_ITERATOR, GeeArrayListIterator))
#define GEE_ARRAY_LIST_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GEE_ARRAY_LIST_TYPE_ITERATOR, GeeArrayListIteratorClass))
#define GEE_ARRAY_LIST_IS_ITERATOR(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GEE_ARRAY_LIST_TYPE_ITERATOR))
#define GEE_ARRAY_LIST_IS_ITERATOR_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GEE_ARRAY_LIST_TYPE_ITERATOR))
#define GEE_ARRAY_LIST_ITERATOR_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GEE_ARRAY_LIST_TYPE_ITERATOR, GeeArrayListIteratorClass))

struct _GeeArrayListIterator {
	GObject parent;
	GeeArrayListIteratorPrivate * priv;
	gint _stamp;
};
struct _GeeArrayListIteratorClass {
	GObjectClass parent;
};
struct _GeeArrayListIteratorPrivate {
	GeeArrayList* _list;
	gint _index;
	GBoxedCopyFunc g_dup_func;
	GDestroyNotify g_destroy_func;
};
#define GEE_ARRAY_LIST_ITERATOR_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GEE_ARRAY_LIST_TYPE_ITERATOR, GeeArrayListIteratorPrivate))
enum  {
	GEE_ARRAY_LIST_ITERATOR_DUMMY_PROPERTY,
	GEE_ARRAY_LIST_ITERATOR_LIST,
	GEE_ARRAY_LIST_ITERATOR_G_DUP_FUNC,
	GEE_ARRAY_LIST_ITERATOR_G_DESTROY_FUNC
};
static GeeArrayListIterator* gee_array_list_iterator_new (GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GeeArrayList* list);
static gboolean gee_array_list_iterator_real_next (GeeIterator* base);
static gpointer gee_array_list_iterator_real_get (GeeIterator* base);
static void gee_array_list_iterator_set_list (GeeArrayListIterator* self, GeeArrayList* value);
static gpointer gee_array_list_iterator_parent_class = NULL;
static GeeIteratorIface* gee_array_list_iterator_gee_iterator_parent_iface = NULL;
static void gee_array_list_iterator_dispose (GObject * obj);
static GType gee_array_list_iterator_get_type (void);
static gpointer gee_array_list_parent_class = NULL;
static GeeIterableIface* gee_array_list_gee_iterable_parent_iface = NULL;
static GeeCollectionIface* gee_array_list_gee_collection_parent_iface = NULL;
static GeeListIface* gee_array_list_gee_list_parent_iface = NULL;
static void gee_array_list_dispose (GObject * obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static void _vala_array_move (gpointer array, gsize element_size, gint src, gint dest, gint length);


GeeArrayList* gee_array_list_new (GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GEqualFunc equal_func)
{
	GeeArrayList * self;
	self = g_object_newv (GEE_TYPE_ARRAY_LIST, 0, NULL);
	gee_array_list_set_equal_func (self, equal_func);
	self->priv->g_dup_func = g_dup_func;
	self->priv->g_destroy_func = g_destroy_func;
	return self;
}


static GeeIterator* gee_array_list_real_iterator (GeeIterable* base)
{
	GeeArrayList * self;
	self = ((GeeArrayList*) base);
	return GEE_ITERATOR (gee_array_list_iterator_new (self->priv->g_dup_func, self->priv->g_destroy_func, self));
}


static gboolean gee_array_list_real_contains (GeeCollection* base, gpointer item)
{
	GeeArrayList * self;
	self = ((GeeArrayList*) base);
	return (gee_list_index_of (GEE_LIST (self), item) != -1);
}


static gint gee_array_list_real_index_of (GeeList* base, gpointer item)
{
	GeeArrayList * self;
	self = ((GeeArrayList*) base);
	{
		gint index;
		index = 0;
		for (index = 0; index < self->priv->_size; index++) {
			if (self->priv->_equal_func (self->priv->_items[index], item)) {
				return index;
			}
		}
	}
	return -1;
}


static gpointer gee_array_list_real_get (GeeList* base, gint index)
{
	GeeArrayList * self;
	gpointer _tmp0;
	self = ((GeeArrayList*) base);
	g_assert (index >= 0 && index < self->priv->_size);
	return (_tmp0 = self->priv->_items[index], (_tmp0 == NULL || self->priv->g_dup_func == NULL ? _tmp0 : self->priv->g_dup_func (_tmp0)));
}


static void gee_array_list_real_set (GeeList* base, gint index, gpointer item)
{
	GeeArrayList * self;
	gpointer _tmp1;
	gpointer _tmp0;
	self = ((GeeArrayList*) base);
	g_assert (index >= 0 && index < self->priv->_size);
	self->priv->_items[index] = (_tmp1 = (_tmp0 = item, (_tmp0 == NULL || self->priv->g_dup_func == NULL ? _tmp0 : self->priv->g_dup_func (_tmp0))), (self->priv->_items[index] == NULL || self->priv->g_destroy_func == NULL ? NULL : (self->priv->_items[index] = (self->priv->g_destroy_func (self->priv->_items[index]), NULL))), _tmp1);
}


static gboolean gee_array_list_real_add (GeeCollection* base, gpointer item)
{
	GeeArrayList * self;
	gint _tmp2;
	gpointer _tmp1;
	gpointer _tmp0;
	self = ((GeeArrayList*) base);
	if (self->priv->_size == self->priv->_items_length1) {
		gee_array_list_grow_if_needed (self, 1);
	}
	(_tmp2 = self->priv->_size++, self->priv->_items[_tmp2] = (_tmp1 = (_tmp0 = item, (_tmp0 == NULL || self->priv->g_dup_func == NULL ? _tmp0 : self->priv->g_dup_func (_tmp0))), (self->priv->_items[_tmp2] == NULL || self->priv->g_destroy_func == NULL ? NULL : (self->priv->_items[_tmp2] = (self->priv->g_destroy_func (self->priv->_items[_tmp2]), NULL))), _tmp1));
	self->priv->_stamp++;
	return TRUE;
}


static void gee_array_list_real_insert (GeeList* base, gint index, gpointer item)
{
	GeeArrayList * self;
	gpointer _tmp1;
	gpointer _tmp0;
	self = ((GeeArrayList*) base);
	g_assert (index >= 0 && index <= self->priv->_size);
	if (self->priv->_size == self->priv->_items_length1) {
		gee_array_list_grow_if_needed (self, 1);
	}
	gee_array_list_shift (self, index, 1);
	self->priv->_items[index] = (_tmp1 = (_tmp0 = item, (_tmp0 == NULL || self->priv->g_dup_func == NULL ? _tmp0 : self->priv->g_dup_func (_tmp0))), (self->priv->_items[index] == NULL || self->priv->g_destroy_func == NULL ? NULL : (self->priv->_items[index] = (self->priv->g_destroy_func (self->priv->_items[index]), NULL))), _tmp1);
	self->priv->_stamp++;
}


static gboolean gee_array_list_real_remove (GeeCollection* base, gpointer item)
{
	GeeArrayList * self;
	self = ((GeeArrayList*) base);
	{
		gint index;
		index = 0;
		for (index = 0; index < self->priv->_size; index++) {
			if (self->priv->_equal_func (self->priv->_items[index], item)) {
				gee_list_remove_at (GEE_LIST (self), index);
				return TRUE;
			}
		}
	}
	return FALSE;
}


static void gee_array_list_real_remove_at (GeeList* base, gint index)
{
	GeeArrayList * self;
	gpointer _tmp0;
	self = ((GeeArrayList*) base);
	g_assert (index >= 0 && index < self->priv->_size);
	self->priv->_items[index] = (_tmp0 = NULL, (self->priv->_items[index] == NULL || self->priv->g_destroy_func == NULL ? NULL : (self->priv->_items[index] = (self->priv->g_destroy_func (self->priv->_items[index]), NULL))), _tmp0);
	gee_array_list_shift (self, index + 1, -1);
	self->priv->_stamp++;
}


static void gee_array_list_real_clear (GeeCollection* base)
{
	GeeArrayList * self;
	self = ((GeeArrayList*) base);
	{
		gint index;
		index = 0;
		for (index = 0; index < self->priv->_size; index++) {
			gpointer _tmp0;
			self->priv->_items[index] = (_tmp0 = NULL, (self->priv->_items[index] == NULL || self->priv->g_destroy_func == NULL ? NULL : (self->priv->_items[index] = (self->priv->g_destroy_func (self->priv->_items[index]), NULL))), _tmp0);
		}
	}
	self->priv->_size = 0;
	self->priv->_stamp++;
}


static void gee_array_list_shift (GeeArrayList* self, gint start, gint delta)
{
	g_return_if_fail (GEE_IS_ARRAY_LIST (self));
	g_assert (start >= 0 && start <= self->priv->_size && start >= -delta);
	_vala_array_move (self->priv->_items, sizeof (gpointer), start, start + delta, self->priv->_size - start);
	self->priv->_size = self->priv->_size + (delta);
}


static void gee_array_list_grow_if_needed (GeeArrayList* self, gint new_count)
{
	gint minimum_size;
	g_return_if_fail (GEE_IS_ARRAY_LIST (self));
	g_assert (new_count >= 0);
	minimum_size = self->priv->_size + new_count;
	if (minimum_size > self->priv->_items_length1) {
		/* double the capacity unless we add even more items at this time*/
		gee_array_list_set_capacity (self, (new_count > self->priv->_items_length1 ? minimum_size : 2 * self->priv->_items_length1));
	}
}


static void gee_array_list_set_capacity (GeeArrayList* self, gint value)
{
	gint _tmp0;
	g_return_if_fail (GEE_IS_ARRAY_LIST (self));
	g_assert (value >= self->priv->_size);
	(_tmp0 = value, self->priv->_items = g_renew (gpointer, self->priv->_items, value), (_tmp0 > self->priv->_items_length1 ? memset (self->priv->_items + self->priv->_items_length1, 0, sizeof (gpointer) * (_tmp0 - self->priv->_items_length1)) : NULL), self->priv->_items_length1 = _tmp0);
}


static gint gee_array_list_real_get_size (GeeArrayList* self)
{
	g_return_val_if_fail (GEE_IS_ARRAY_LIST (self), 0);
	return self->priv->_size;
}


void gee_array_list_set_equal_func (GeeArrayList* self, GEqualFunc value)
{
	g_return_if_fail (GEE_IS_ARRAY_LIST (self));
	self->priv->_equal_func = value;
}


static GeeArrayListIterator* gee_array_list_iterator_new (GBoxedCopyFunc g_dup_func, GDestroyNotify g_destroy_func, GeeArrayList* list)
{
	GeeArrayListIterator * self;
	g_return_val_if_fail (GEE_IS_ARRAY_LIST (list), NULL);
	self = g_object_newv (GEE_ARRAY_LIST_TYPE_ITERATOR, 0, NULL);
	gee_array_list_iterator_set_list (self, list);
	self->priv->g_dup_func = g_dup_func;
	self->priv->g_destroy_func = g_destroy_func;
	return self;
}


static gboolean gee_array_list_iterator_real_next (GeeIterator* base)
{
	GeeArrayListIterator * self;
	self = ((GeeArrayListIterator*) base);
	g_assert (self->_stamp == self->priv->_list->priv->_stamp);
	if (self->priv->_index < self->priv->_list->priv->_size) {
		self->priv->_index++;
	}
	return (self->priv->_index < self->priv->_list->priv->_size);
}


static gpointer gee_array_list_iterator_real_get (GeeIterator* base)
{
	GeeArrayListIterator * self;
	self = ((GeeArrayListIterator*) base);
	g_assert (self->_stamp == self->priv->_list->priv->_stamp);
	if (self->priv->_index < 0 || self->priv->_index >= self->priv->_list->priv->_size) {
		return NULL;
	}
	return gee_list_get (GEE_LIST (self->priv->_list), self->priv->_index);
}


static void gee_array_list_iterator_set_list (GeeArrayListIterator* self, GeeArrayList* value)
{
	GeeArrayList* _tmp1;
	GeeArrayList* _tmp0;
	g_return_if_fail (GEE_ARRAY_LIST_IS_ITERATOR (self));
	_tmp1 = NULL;
	_tmp0 = NULL;
	self->priv->_list = (_tmp1 = (_tmp0 = value, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0))), (self->priv->_list == NULL ? NULL : (self->priv->_list = (g_object_unref (self->priv->_list), NULL))), _tmp1);
	self->_stamp = self->priv->_list->priv->_stamp;
}


static void gee_array_list_iterator_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	GeeArrayListIterator * self;
	self = GEE_ARRAY_LIST_ITERATOR (object);
	switch (property_id) {
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void gee_array_list_iterator_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	GeeArrayListIterator * self;
	self = GEE_ARRAY_LIST_ITERATOR (object);
	switch (property_id) {
		case GEE_ARRAY_LIST_ITERATOR_LIST:
		gee_array_list_iterator_set_list (self, g_value_get_object (value));
		break;
		case GEE_ARRAY_LIST_ITERATOR_G_DESTROY_FUNC:
		self->priv->g_destroy_func = g_value_get_pointer (value);
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void gee_array_list_iterator_class_init (GeeArrayListIteratorClass * klass)
{
	gee_array_list_iterator_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (GeeArrayListIteratorPrivate));
	G_OBJECT_CLASS (klass)->get_property = gee_array_list_iterator_get_property;
	G_OBJECT_CLASS (klass)->set_property = gee_array_list_iterator_set_property;
	G_OBJECT_CLASS (klass)->dispose = gee_array_list_iterator_dispose;
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_ARRAY_LIST_ITERATOR_G_DUP_FUNC, g_param_spec_pointer ("g-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_ARRAY_LIST_ITERATOR_G_DESTROY_FUNC, g_param_spec_pointer ("g-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_ARRAY_LIST_ITERATOR_LIST, g_param_spec_object ("list", "foo", "bar", GEE_TYPE_ARRAY_LIST, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
}


static void gee_array_list_iterator_gee_iterator_interface_init (GeeIteratorIface * iface)
{
	gee_array_list_iterator_gee_iterator_parent_iface = g_type_interface_peek_parent (iface);
	iface->next = gee_array_list_iterator_real_next;
	iface->get = gee_array_list_iterator_real_get;
}


static void gee_array_list_iterator_init (GeeArrayListIterator * self)
{
	self->priv = GEE_ARRAY_LIST_ITERATOR_GET_PRIVATE (self);
	self->priv->_index = -1;
	self->_stamp = 0;
}


static void gee_array_list_iterator_dispose (GObject * obj)
{
	GeeArrayListIterator * self;
	GeeArrayListIteratorClass * klass;
	GObjectClass * parent_class;
	self = GEE_ARRAY_LIST_ITERATOR (obj);
	(self->priv->_list == NULL ? NULL : (self->priv->_list = (g_object_unref (self->priv->_list), NULL)));
	klass = GEE_ARRAY_LIST_ITERATOR_CLASS (g_type_class_peek (GEE_ARRAY_LIST_TYPE_ITERATOR));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


static GType gee_array_list_iterator_get_type (void)
{
	static GType gee_array_list_iterator_type_id = 0;
	if (G_UNLIKELY (gee_array_list_iterator_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeArrayListIteratorClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_array_list_iterator_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeArrayListIterator), 0, (GInstanceInitFunc) gee_array_list_iterator_init };
		static const GInterfaceInfo gee_iterator_info = { (GInterfaceInitFunc) gee_array_list_iterator_gee_iterator_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_array_list_iterator_type_id = g_type_register_static (G_TYPE_OBJECT, "GeeArrayListIterator", &g_define_type_info, 0);
		g_type_add_interface_static (gee_array_list_iterator_type_id, GEE_TYPE_ITERATOR, &gee_iterator_info);
	}
	return gee_array_list_iterator_type_id;
}


static void gee_array_list_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	GeeArrayList * self;
	self = GEE_ARRAY_LIST (object);
	switch (property_id) {
		case GEE_ARRAY_LIST_SIZE:
		g_value_set_int (value, gee_array_list_real_get_size (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void gee_array_list_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	GeeArrayList * self;
	self = GEE_ARRAY_LIST (object);
	switch (property_id) {
		case GEE_ARRAY_LIST_EQUAL_FUNC:
		gee_array_list_set_equal_func (self, g_value_get_pointer (value));
		break;
		case GEE_ARRAY_LIST_G_DESTROY_FUNC:
		self->priv->g_destroy_func = g_value_get_pointer (value);
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void gee_array_list_class_init (GeeArrayListClass * klass)
{
	gee_array_list_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (GeeArrayListPrivate));
	G_OBJECT_CLASS (klass)->get_property = gee_array_list_get_property;
	G_OBJECT_CLASS (klass)->set_property = gee_array_list_set_property;
	G_OBJECT_CLASS (klass)->dispose = gee_array_list_dispose;
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_ARRAY_LIST_G_DUP_FUNC, g_param_spec_pointer ("g-dup-func", "dup func", "dup func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_ARRAY_LIST_G_DESTROY_FUNC, g_param_spec_pointer ("g-destroy-func", "destroy func", "destroy func", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
	g_object_class_override_property (G_OBJECT_CLASS (klass), GEE_ARRAY_LIST_SIZE, "size");
	g_object_class_install_property (G_OBJECT_CLASS (klass), GEE_ARRAY_LIST_EQUAL_FUNC, g_param_spec_pointer ("equal-func", "foo", "bar", G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_WRITABLE));
}


static void gee_array_list_gee_iterable_interface_init (GeeIterableIface * iface)
{
	gee_array_list_gee_iterable_parent_iface = g_type_interface_peek_parent (iface);
	iface->iterator = gee_array_list_real_iterator;
}


static void gee_array_list_gee_collection_interface_init (GeeCollectionIface * iface)
{
	gee_array_list_gee_collection_parent_iface = g_type_interface_peek_parent (iface);
	iface->contains = gee_array_list_real_contains;
	iface->add = gee_array_list_real_add;
	iface->remove = gee_array_list_real_remove;
	iface->clear = gee_array_list_real_clear;
}


static void gee_array_list_gee_list_interface_init (GeeListIface * iface)
{
	gee_array_list_gee_list_parent_iface = g_type_interface_peek_parent (iface);
	iface->index_of = gee_array_list_real_index_of;
	iface->get = gee_array_list_real_get;
	iface->set = gee_array_list_real_set;
	iface->insert = gee_array_list_real_insert;
	iface->remove_at = gee_array_list_real_remove_at;
}


static void gee_array_list_init (GeeArrayList * self)
{
	self->priv = GEE_ARRAY_LIST_GET_PRIVATE (self);
	self->priv->_items = g_new0 (gpointer, 4);
	self->priv->_items_length1 = 4;
	self->priv->_stamp = 0;
}


static void gee_array_list_dispose (GObject * obj)
{
	GeeArrayList * self;
	GeeArrayListClass * klass;
	GObjectClass * parent_class;
	self = GEE_ARRAY_LIST (obj);
	(self->priv->_items = (_vala_array_free (self->priv->_items, self->priv->_items_length1, ((GDestroyNotify) self->priv->g_destroy_func)), NULL));
	klass = GEE_ARRAY_LIST_CLASS (g_type_class_peek (GEE_TYPE_ARRAY_LIST));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType gee_array_list_get_type (void)
{
	static GType gee_array_list_type_id = 0;
	if (G_UNLIKELY (gee_array_list_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (GeeArrayListClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gee_array_list_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GeeArrayList), 0, (GInstanceInitFunc) gee_array_list_init };
		static const GInterfaceInfo gee_iterable_info = { (GInterfaceInitFunc) gee_array_list_gee_iterable_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		static const GInterfaceInfo gee_collection_info = { (GInterfaceInitFunc) gee_array_list_gee_collection_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		static const GInterfaceInfo gee_list_info = { (GInterfaceInitFunc) gee_array_list_gee_list_interface_init, (GInterfaceFinalizeFunc) NULL, NULL};
		gee_array_list_type_id = g_type_register_static (G_TYPE_OBJECT, "GeeArrayList", &g_define_type_info, 0);
		g_type_add_interface_static (gee_array_list_type_id, GEE_TYPE_ITERABLE, &gee_iterable_info);
		g_type_add_interface_static (gee_array_list_type_id, GEE_TYPE_COLLECTION, &gee_collection_info);
		g_type_add_interface_static (gee_array_list_type_id, GEE_TYPE_LIST, &gee_list_info);
	}
	return gee_array_list_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func)
{
	if (array != NULL && destroy_func != NULL) {
		int i;
		if (array_length >= 0)
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL)
			destroy_func (((gpointer*) array)[i]);
		}
		else
		for (i = 0; ((gpointer*) array)[i] != NULL; i = i + 1) {
			destroy_func (((gpointer*) array)[i]);
		}
	}
	g_free (array);
}


static void _vala_array_move (gpointer array, gsize element_size, gint src, gint dest, gint length)
{
	g_memmove (array + dest * element_size, array + src * element_size, length * element_size);
	if (src < dest) {
		memset (array + src * element_size, 0, (dest - src) * element_size);
	} else {
		memset (array + (dest + length) * element_size, 0, (src - dest) * element_size);
	}
}




