/* valamemorymanager.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <rasa@gmx.ch>
 */
#define VALA_FREE_CHECKED(o,f) ((o) == NULL ? NULL : ((o) = (f (o), NULL)))
#define VALA_FREE_UNCHECKED(o,f) ((o) = (f (o), NULL))

#include "valamemorymanager.h"
#include <stdlib.h>
#include <string.h>
#include <vala/valasymbol.h>
#include <vala/valamemorymanager.h>
#include <vala/valatypereference.h>
#include <vala/valadatatype.h>
#include <vala/valacodenode.h>
#include <vala/valacallback.h>
#include <vala/valaformalparameter.h>
#include <vala/valasignal.h>
#include <vala/valareport.h>
#include <vala/valasourcereference.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valatypeparameter.h>
#include <vala/valapointerindirection.h>

struct _ValaMemoryManagerPrivate {
	ValaSymbol* current_symbol;
};
#define VALA_MEMORY_MANAGER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_MEMORY_MANAGER, ValaMemoryManagerPrivate))
enum  {
	VALA_MEMORY_MANAGER_DUMMY_PROPERTY
};
static void vala_memory_manager_visit_possibly_leaked_expression (ValaMemoryManager* self, ValaExpression* expr);
static void vala_memory_manager_visit_possibly_missing_copy_expression (ValaMemoryManager* self, ValaExpression* expr);
static void vala_memory_manager_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file);
static void vala_memory_manager_real_visit_namespace (ValaCodeVisitor* base, ValaNamespace* ns);
static void vala_memory_manager_real_visit_class (ValaCodeVisitor* base, ValaClass* cl);
static void vala_memory_manager_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st);
static void vala_memory_manager_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface);
static void vala_memory_manager_real_visit_field (ValaCodeVisitor* base, ValaField* f);
static void vala_memory_manager_real_visit_method (ValaCodeVisitor* base, ValaMethod* m);
static void vala_memory_manager_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m);
static void vala_memory_manager_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop);
static void vala_memory_manager_real_visit_property_accessor (ValaCodeVisitor* base, ValaPropertyAccessor* acc);
static void vala_memory_manager_real_visit_constructor (ValaCodeVisitor* base, ValaConstructor* c);
static void vala_memory_manager_real_visit_destructor (ValaCodeVisitor* base, ValaDestructor* d);
static void vala_memory_manager_real_visit_named_argument (ValaCodeVisitor* base, ValaNamedArgument* n);
static void vala_memory_manager_real_visit_variable_declarator (ValaCodeVisitor* base, ValaVariableDeclarator* decl);
static void vala_memory_manager_real_visit_expression_statement (ValaCodeVisitor* base, ValaExpressionStatement* stmt);
static void vala_memory_manager_real_visit_end_return_statement (ValaCodeVisitor* base, ValaReturnStatement* stmt);
static void vala_memory_manager_real_visit_throw_statement (ValaCodeVisitor* base, ValaThrowStatement* stmt);
static void vala_memory_manager_real_visit_try_statement (ValaCodeVisitor* base, ValaTryStatement* stmt);
static void vala_memory_manager_real_visit_catch_clause (ValaCodeVisitor* base, ValaCatchClause* clause);
static void vala_memory_manager_real_visit_member_access (ValaCodeVisitor* base, ValaMemberAccess* expr);
static void vala_memory_manager_real_visit_end_invocation_expression (ValaCodeVisitor* base, ValaInvocationExpression* expr);
static void vala_memory_manager_real_visit_binary_expression (ValaCodeVisitor* base, ValaBinaryExpression* expr);
static void vala_memory_manager_real_visit_end_assignment (ValaCodeVisitor* base, ValaAssignment* a);
static gpointer vala_memory_manager_parent_class = NULL;
static void vala_memory_manager_dispose (GObject * obj);


/**
 * Analyze memory usage in the specified code context.
 *
 * @param context a code context
 */
void vala_memory_manager_analyze (ValaMemoryManager* self, ValaCodeContext* context)
{
	g_return_if_fail (VALA_IS_MEMORY_MANAGER (self));
	g_return_if_fail (VALA_IS_CODE_CONTEXT (context));
	vala_code_context_accept (context, VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_visit_possibly_leaked_expression (ValaMemoryManager* self, ValaExpression* expr)
{
	g_return_if_fail (VALA_IS_MEMORY_MANAGER (self));
	g_return_if_fail (VALA_IS_EXPRESSION (expr));
	if (vala_expression_get_static_type (expr) != NULL && ((vala_type_reference_get_data_type (vala_expression_get_static_type (expr)) != NULL && vala_data_type_is_reference_type (vala_type_reference_get_data_type (vala_expression_get_static_type (expr)))) || vala_type_reference_get_type_parameter (vala_expression_get_static_type (expr)) != NULL) && vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (expr))) {
		/* mark reference as leaked */
		(vala_expression_set_ref_leaked (expr, TRUE), vala_expression_get_ref_leaked (expr));
	}
}


static void vala_memory_manager_visit_possibly_missing_copy_expression (ValaMemoryManager* self, ValaExpression* expr)
{
	g_return_if_fail (VALA_IS_MEMORY_MANAGER (self));
	g_return_if_fail (VALA_IS_EXPRESSION (expr));
	if (vala_expression_get_static_type (expr) != NULL && ((vala_type_reference_get_data_type (vala_expression_get_static_type (expr)) != NULL && vala_data_type_is_reference_type (vala_type_reference_get_data_type (vala_expression_get_static_type (expr)))) || vala_type_reference_get_type_parameter (vala_expression_get_static_type (expr)) != NULL) && !vala_type_reference_get_transfers_ownership (vala_expression_get_static_type (expr))) {
		/* mark reference as missing */
		(vala_expression_set_ref_missing (expr, TRUE), vala_expression_get_ref_missing (expr));
	}
}


static void vala_memory_manager_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file)
{
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_SOURCE_FILE (source_file));
	vala_source_file_accept_children (source_file, VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_namespace (ValaCodeVisitor* base, ValaNamespace* ns)
{
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_NAMESPACE (ns));
	vala_code_node_accept_children (VALA_CODE_NODE (ns), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_class (ValaCodeVisitor* base, ValaClass* cl)
{
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_CLASS (cl));
	vala_code_node_accept_children (VALA_CODE_NODE (cl), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st)
{
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_STRUCT (st));
	vala_code_node_accept_children (VALA_CODE_NODE (st), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface)
{
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_INTERFACE (iface));
	vala_code_node_accept_children (VALA_CODE_NODE (iface), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_field (ValaCodeVisitor* base, ValaField* f)
{
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_FIELD (f));
	if (vala_field_get_initializer (f) != NULL) {
		if (vala_type_reference_get_takes_ownership (vala_field_get_type_reference (f))) {
			vala_memory_manager_visit_possibly_missing_copy_expression (self, vala_field_get_initializer (f));
		} else {
			vala_memory_manager_visit_possibly_leaked_expression (self, vala_field_get_initializer (f));
		}
	}
}


static void vala_memory_manager_real_visit_method (ValaCodeVisitor* base, ValaMethod* m)
{
	ValaMemoryManager * self;
	ValaSymbol* __temp1;
	ValaSymbol* __temp0;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_METHOD (m));
	__temp1 = NULL;
	__temp0 = NULL;
	self->priv->current_symbol = (__temp1 = (__temp0 = vala_code_node_get_symbol (VALA_CODE_NODE (m)), (__temp0 == NULL ? NULL : g_object_ref (__temp0))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp1);
	vala_code_node_accept_children (VALA_CODE_NODE (m), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m)
{
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_CREATION_METHOD (m));
	vala_code_visitor_visit_method (VALA_CODE_VISITOR (self), VALA_METHOD (m));
}


static void vala_memory_manager_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop)
{
	ValaMemoryManager * self;
	ValaSymbol* __temp3;
	ValaSymbol* __temp2;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_PROPERTY (prop));
	__temp3 = NULL;
	__temp2 = NULL;
	self->priv->current_symbol = (__temp3 = (__temp2 = vala_code_node_get_symbol (VALA_CODE_NODE (prop)), (__temp2 == NULL ? NULL : g_object_ref (__temp2))), (self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL))), __temp3);
	vala_code_node_accept_children (VALA_CODE_NODE (prop), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_property_accessor (ValaCodeVisitor* base, ValaPropertyAccessor* acc)
{
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_PROPERTY_ACCESSOR (acc));
	vala_code_node_accept_children (VALA_CODE_NODE (acc), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_constructor (ValaCodeVisitor* base, ValaConstructor* c)
{
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_CONSTRUCTOR (c));
	vala_code_node_accept_children (VALA_CODE_NODE (c), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_destructor (ValaCodeVisitor* base, ValaDestructor* d)
{
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_DESTRUCTOR (d));
	vala_code_node_accept_children (VALA_CODE_NODE (d), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_named_argument (ValaCodeVisitor* base, ValaNamedArgument* n)
{
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_NAMED_ARGUMENT (n));
	vala_memory_manager_visit_possibly_leaked_expression (self, vala_named_argument_get_argument (n));
}


static void vala_memory_manager_real_visit_variable_declarator (ValaCodeVisitor* base, ValaVariableDeclarator* decl)
{
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_VARIABLE_DECLARATOR (decl));
	if (vala_variable_declarator_get_initializer (decl) != NULL) {
		if (vala_type_reference_get_takes_ownership (vala_variable_declarator_get_type_reference (decl))) {
			vala_memory_manager_visit_possibly_missing_copy_expression (self, vala_variable_declarator_get_initializer (decl));
		} else {
			vala_memory_manager_visit_possibly_leaked_expression (self, vala_variable_declarator_get_initializer (decl));
		}
	}
}


static void vala_memory_manager_real_visit_expression_statement (ValaCodeVisitor* base, ValaExpressionStatement* stmt)
{
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_EXPRESSION_STATEMENT (stmt));
	vala_memory_manager_visit_possibly_leaked_expression (self, vala_expression_statement_get_expression (stmt));
}


static void vala_memory_manager_real_visit_end_return_statement (ValaCodeVisitor* base, ValaReturnStatement* stmt)
{
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_RETURN_STATEMENT (stmt));
	if (vala_return_statement_get_return_expression (stmt) != NULL) {
		if (VALA_IS_METHOD (vala_symbol_get_node (self->priv->current_symbol))) {
			ValaMethod* __temp4;
			ValaMethod* m;
			__temp4 = NULL;
			m = (__temp4 = VALA_METHOD (vala_symbol_get_node (self->priv->current_symbol)), (__temp4 == NULL ? NULL : g_object_ref (__temp4)));
			if (vala_type_reference_get_transfers_ownership (vala_method_get_return_type (m))) {
				vala_memory_manager_visit_possibly_missing_copy_expression (self, vala_return_statement_get_return_expression (stmt));
			} else {
				vala_memory_manager_visit_possibly_leaked_expression (self, vala_return_statement_get_return_expression (stmt));
			}
			(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
		} else {
			/* property get accessor */
			vala_memory_manager_visit_possibly_leaked_expression (self, vala_return_statement_get_return_expression (stmt));
		}
	}
}


static void vala_memory_manager_real_visit_throw_statement (ValaCodeVisitor* base, ValaThrowStatement* stmt)
{
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_THROW_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_try_statement (ValaCodeVisitor* base, ValaTryStatement* stmt)
{
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_TRY_STATEMENT (stmt));
	vala_code_node_accept_children (VALA_CODE_NODE (stmt), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_catch_clause (ValaCodeVisitor* base, ValaCatchClause* clause)
{
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_CATCH_CLAUSE (clause));
	vala_code_node_accept_children (VALA_CODE_NODE (clause), VALA_CODE_VISITOR (self));
}


static void vala_memory_manager_real_visit_member_access (ValaCodeVisitor* base, ValaMemberAccess* expr)
{
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_MEMBER_ACCESS (expr));
	if (vala_member_access_get_inner (expr) != NULL) {
		vala_memory_manager_visit_possibly_leaked_expression (self, vala_member_access_get_inner (expr));
	}
}


static void vala_memory_manager_real_visit_end_invocation_expression (ValaCodeVisitor* base, ValaInvocationExpression* expr)
{
	ValaMemoryManager * self;
	GList* params;
	ValaSymbol* __temp5;
	ValaSymbol* msym;
	GList* params_it;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_INVOCATION_EXPRESSION (expr));
	params = NULL;
	__temp5 = NULL;
	msym = (__temp5 = vala_expression_get_symbol_reference (vala_invocation_expression_get_call (expr)), (__temp5 == NULL ? NULL : g_object_ref (__temp5)));
	if (VALA_IS_VARIABLE_DECLARATOR (vala_symbol_get_node (msym))) {
		ValaVariableDeclarator* __temp6;
		ValaVariableDeclarator* decl;
		ValaCallback* __temp7;
		ValaCallback* cb;
		GList* __temp8;
		__temp6 = NULL;
		decl = (__temp6 = VALA_VARIABLE_DECLARATOR (vala_symbol_get_node (msym)), (__temp6 == NULL ? NULL : g_object_ref (__temp6)));
		__temp7 = NULL;
		cb = (__temp7 = VALA_CALLBACK (vala_type_reference_get_data_type (vala_variable_declarator_get_type_reference (decl))), (__temp7 == NULL ? NULL : g_object_ref (__temp7)));
		__temp8 = NULL;
		params = (__temp8 = vala_callback_get_parameters (cb), (params == NULL ? NULL : (params = (g_list_free (params), NULL))), __temp8);
		(decl == NULL ? NULL : (decl = (g_object_unref (decl), NULL)));
		(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
	} else {
		if (VALA_IS_FORMAL_PARAMETER (vala_symbol_get_node (msym))) {
			ValaFormalParameter* __temp9;
			ValaFormalParameter* param;
			ValaCallback* __temp10;
			ValaCallback* cb;
			GList* __temp11;
			__temp9 = NULL;
			param = (__temp9 = VALA_FORMAL_PARAMETER (vala_symbol_get_node (msym)), (__temp9 == NULL ? NULL : g_object_ref (__temp9)));
			__temp10 = NULL;
			cb = (__temp10 = VALA_CALLBACK (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (param))), (__temp10 == NULL ? NULL : g_object_ref (__temp10)));
			__temp11 = NULL;
			params = (__temp11 = vala_callback_get_parameters (cb), (params == NULL ? NULL : (params = (g_list_free (params), NULL))), __temp11);
			(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
			(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
		} else {
			if (VALA_IS_FIELD (vala_symbol_get_node (msym))) {
				ValaField* __temp12;
				ValaField* f;
				ValaCallback* __temp13;
				ValaCallback* cb;
				GList* __temp14;
				__temp12 = NULL;
				f = (__temp12 = VALA_FIELD (vala_symbol_get_node (msym)), (__temp12 == NULL ? NULL : g_object_ref (__temp12)));
				__temp13 = NULL;
				cb = (__temp13 = VALA_CALLBACK (vala_type_reference_get_data_type (vala_field_get_type_reference (f))), (__temp13 == NULL ? NULL : g_object_ref (__temp13)));
				__temp14 = NULL;
				params = (__temp14 = vala_callback_get_parameters (cb), (params == NULL ? NULL : (params = (g_list_free (params), NULL))), __temp14);
				(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
				(cb == NULL ? NULL : (cb = (g_object_unref (cb), NULL)));
			} else {
				if (VALA_IS_METHOD (vala_symbol_get_node (msym))) {
					ValaMethod* __temp15;
					ValaMethod* m;
					GList* __temp16;
					__temp15 = NULL;
					m = (__temp15 = VALA_METHOD (vala_symbol_get_node (msym)), (__temp15 == NULL ? NULL : g_object_ref (__temp15)));
					__temp16 = NULL;
					params = (__temp16 = vala_invokable_get_parameters (VALA_INVOKABLE (m)), (params == NULL ? NULL : (params = (g_list_free (params), NULL))), __temp16);
					(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
				} else {
					if (VALA_IS_SIGNAL (vala_symbol_get_node (msym))) {
						ValaSignal* __temp17;
						ValaSignal* sig;
						GList* __temp18;
						__temp17 = NULL;
						sig = (__temp17 = VALA_SIGNAL (vala_symbol_get_node (msym)), (__temp17 == NULL ? NULL : g_object_ref (__temp17)));
						__temp18 = NULL;
						params = (__temp18 = vala_invokable_get_parameters (VALA_INVOKABLE (sig)), (params == NULL ? NULL : (params = (g_list_free (params), NULL))), __temp18);
						(sig == NULL ? NULL : (sig = (g_object_unref (sig), NULL)));
					}
				}
			}
		}
	}
	params_it = params;
	{
		GList* __temp39;
		GList* arg_it;
		__temp39 = NULL;
		__temp39 = vala_invocation_expression_get_argument_list (expr);
		for (arg_it = __temp39; arg_it != NULL; arg_it = arg_it->next) {
			ValaExpression* arg;
			arg = arg_it->data;
			{
				if (params_it != NULL) {
					ValaFormalParameter* __temp19;
					ValaFormalParameter* param;
					__temp19 = NULL;
					param = (__temp19 = VALA_FORMAL_PARAMETER (params_it->data), (__temp19 == NULL ? NULL : g_object_ref (__temp19)));
					if (!vala_formal_parameter_get_ellipsis (param) && ((vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (param)) != NULL && vala_data_type_is_reference_type (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (param)))) || vala_type_reference_get_type_parameter (vala_formal_parameter_get_type_reference (param)) != NULL)) {
						gboolean is_ref;
						is_ref = vala_type_reference_get_takes_ownership (vala_formal_parameter_get_type_reference (param));
						if (is_ref && vala_type_reference_get_type_parameter (vala_formal_parameter_get_type_reference (param)) != NULL) {
							/* TODO move this to semantic analyzer*/
							if (VALA_IS_MEMBER_ACCESS (vala_invocation_expression_get_call (expr))) {
								ValaMemberAccess* __temp20;
								ValaMemberAccess* ma;
								ValaTypeReference* __temp21;
								ValaTypeReference* instance_type;
								gint param_index;
								ValaTypeReference* __temp36;
								GList* __temp35;
								ValaTypeReference* __temp37;
								ValaTypeReference* param_type;
								__temp20 = NULL;
								ma = (__temp20 = VALA_MEMBER_ACCESS (vala_invocation_expression_get_call (expr)), (__temp20 == NULL ? NULL : g_object_ref (__temp20)));
								__temp21 = NULL;
								instance_type = (__temp21 = vala_expression_get_static_type (vala_member_access_get_inner (ma)), (__temp21 == NULL ? NULL : g_object_ref (__temp21)));
								while (VALA_CODE_NODE (vala_type_reference_get_data_type (instance_type)) != vala_symbol_get_node (vala_symbol_get_parent_symbol (msym))) {
									GList* base_types;
									/* trace type arguments back to the datatype where the method has been declared*/
									base_types = NULL;
									if (VALA_IS_CLASS (vala_type_reference_get_data_type (instance_type))) {
										ValaClass* __temp22;
										ValaClass* cl;
										GList* __temp23;
										__temp22 = NULL;
										cl = (__temp22 = VALA_CLASS (vala_type_reference_get_data_type (instance_type)), (__temp22 == NULL ? NULL : g_object_ref (__temp22)));
										__temp23 = NULL;
										base_types = (__temp23 = vala_class_get_base_types (cl), (base_types == NULL ? NULL : (base_types = (g_list_free (base_types), NULL))), __temp23);
										(cl == NULL ? NULL : (cl = (g_object_unref (cl), NULL)));
									} else {
										if (VALA_IS_INTERFACE (vala_type_reference_get_data_type (instance_type))) {
											ValaInterface* __temp24;
											ValaInterface* iface;
											GList* __temp25;
											__temp24 = NULL;
											iface = (__temp24 = VALA_INTERFACE (vala_type_reference_get_data_type (instance_type)), (__temp24 == NULL ? NULL : g_object_ref (__temp24)));
											__temp25 = NULL;
											base_types = (__temp25 = vala_interface_get_prerequisites (iface), (base_types == NULL ? NULL : (base_types = (g_list_free (base_types), NULL))), __temp25);
											(iface == NULL ? NULL : (iface = (g_object_unref (iface), NULL)));
										} else {
											vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "internal error: unsupported generic type");
											(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
											(base_types == NULL ? NULL : (base_types = (g_list_free (base_types), NULL)));
											(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
											(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
											(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
											(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
											(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
											return;
										}
									}
									{
										GList* __temp33;
										GList* base_type_it;
										__temp33 = NULL;
										__temp33 = base_types;
										for (base_type_it = __temp33; base_type_it != NULL; base_type_it = base_type_it->next) {
											ValaTypeReference* base_type;
											base_type = base_type_it->data;
											{
												ValaSymbol* __temp26;
												gboolean __temp27;
												__temp26 = NULL;
												if ((__temp27 = (__temp26 = vala_semantic_analyzer_symbol_lookup_inherited (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (base_type))), vala_symbol_get_name (msym))) != NULL, (__temp26 == NULL ? NULL : (__temp26 = (g_object_unref (__temp26), NULL))), __temp27)) {
													ValaTypeReference* instance_base_type;
													ValaTypeReference* __temp32;
													ValaTypeReference* __temp31;
													/* construct a new type reference for the base type with correctly linked type arguments*/
													instance_base_type = vala_type_reference_new ();
													(vala_type_reference_set_data_type (instance_base_type, vala_type_reference_get_data_type (base_type)), vala_type_reference_get_data_type (instance_base_type));
													{
														GList* __temp30;
														GList* type_arg_it;
														__temp30 = NULL;
														__temp30 = vala_type_reference_get_type_arguments (base_type);
														for (type_arg_it = __temp30; type_arg_it != NULL; type_arg_it = type_arg_it->next) {
															ValaTypeReference* type_arg;
															type_arg = type_arg_it->data;
															{
																if (vala_type_reference_get_type_parameter (type_arg) != NULL) {
																	gint param_index;
																	GList* __temp29;
																	/* link to type argument of derived type*/
																	param_index = vala_data_type_get_type_parameter_index (vala_type_reference_get_data_type (instance_type), vala_type_parameter_get_name (vala_type_reference_get_type_parameter (type_arg)));
																	if (param_index == -1) {
																		char* __temp28;
																		__temp28 = NULL;
																		vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp28 = g_strdup_printf ("internal error: unknown type parameter %s", vala_type_parameter_get_name (vala_type_reference_get_type_parameter (type_arg)))));
																		(__temp28 = (g_free (__temp28), NULL));
																		(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
																		(instance_base_type == NULL ? NULL : (instance_base_type = (g_object_unref (instance_base_type), NULL)));
																		(base_types == NULL ? NULL : (base_types = (g_list_free (base_types), NULL)));
																		(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
																		(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
																		(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
																		(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
																		(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
																		return;
																	}
																	__temp29 = NULL;
																	type_arg = g_list_nth_data ((__temp29 = vala_type_reference_get_type_arguments (instance_type)), param_index);
																	(__temp29 == NULL ? NULL : (__temp29 = (g_list_free (__temp29), NULL)));
																}
																vala_type_reference_add_type_argument (instance_base_type, type_arg);
															}
														}
														(__temp30 == NULL ? NULL : (__temp30 = (g_list_free (__temp30), NULL)));
													}
													__temp32 = NULL;
													__temp31 = NULL;
													instance_type = (__temp32 = (__temp31 = instance_base_type, (__temp31 == NULL ? NULL : g_object_ref (__temp31))), (instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL))), __temp32);
													(instance_base_type == NULL ? NULL : (instance_base_type = (g_object_unref (instance_base_type), NULL)));
												}
											}
										}
									}
									(base_types == NULL ? NULL : (base_types = (g_list_free (base_types), NULL)));
								}
								if (VALA_CODE_NODE (vala_type_reference_get_data_type (instance_type)) != vala_symbol_get_node (vala_symbol_get_parent_symbol (msym))) {
									vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), "internal error: generic type parameter tracing not supported yet");
									(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
									(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
									(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
									(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
									(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
									(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
									return;
								}
								param_index = vala_data_type_get_type_parameter_index (vala_type_reference_get_data_type (instance_type), vala_type_parameter_get_name (vala_type_reference_get_type_parameter (vala_formal_parameter_get_type_reference (param))));
								if (param_index == -1) {
									char* __temp34;
									__temp34 = NULL;
									vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp34 = g_strdup_printf ("internal error: unknown type parameter %s", vala_type_parameter_get_name (vala_type_reference_get_type_parameter (vala_formal_parameter_get_type_reference (param))))));
									(__temp34 = (g_free (__temp34), NULL));
									(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
									(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
									(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
									(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
									(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
									(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
									return;
								}
								__temp36 = NULL;
								__temp35 = NULL;
								__temp37 = NULL;
								param_type = (__temp37 = (__temp36 = VALA_TYPE_REFERENCE (g_list_nth_data ((__temp35 = vala_type_reference_get_type_arguments (instance_type)), param_index)), (__temp36 == NULL ? NULL : g_object_ref (__temp36))), (__temp35 == NULL ? NULL : (__temp35 = (g_list_free (__temp35), NULL))), __temp37);
								if (param_type == NULL) {
									char* __temp38;
									__temp38 = NULL;
									vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (expr)), (__temp38 = g_strdup_printf ("internal error: no actual argument found for type parameter %s", vala_type_parameter_get_name (vala_type_reference_get_type_parameter (vala_formal_parameter_get_type_reference (param))))));
									(__temp38 = (g_free (__temp38), NULL));
									(vala_code_node_set_error (VALA_CODE_NODE (expr), TRUE), vala_code_node_get_error (VALA_CODE_NODE (expr)));
									(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
									(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
									(param_type == NULL ? NULL : (param_type = (g_object_unref (param_type), NULL)));
									(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
									(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
									(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
									return;
								}
								is_ref = vala_type_reference_get_takes_ownership (param_type);
								(ma == NULL ? NULL : (ma = (g_object_unref (ma), NULL)));
								(instance_type == NULL ? NULL : (instance_type = (g_object_unref (instance_type), NULL)));
								(param_type == NULL ? NULL : (param_type = (g_object_unref (param_type), NULL)));
							}
						}
						if (is_ref) {
							vala_memory_manager_visit_possibly_missing_copy_expression (self, arg);
						} else {
							vala_memory_manager_visit_possibly_leaked_expression (self, arg);
						}
					} else {
						vala_memory_manager_visit_possibly_leaked_expression (self, arg);
					}
					params_it = params_it->next;
					(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
				} else {
					vala_memory_manager_visit_possibly_leaked_expression (self, arg);
				}
			}
		}
		(__temp39 == NULL ? NULL : (__temp39 = (g_list_free (__temp39), NULL)));
	}
	(params == NULL ? NULL : (params = (g_list_free (params), NULL)));
	(msym == NULL ? NULL : (msym = (g_object_unref (msym), NULL)));
}


static void vala_memory_manager_real_visit_binary_expression (ValaCodeVisitor* base, ValaBinaryExpression* expr)
{
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_BINARY_EXPRESSION (expr));
	vala_memory_manager_visit_possibly_leaked_expression (self, vala_binary_expression_get_left (expr));
	vala_memory_manager_visit_possibly_leaked_expression (self, vala_binary_expression_get_right (expr));
}


static void vala_memory_manager_real_visit_end_assignment (ValaCodeVisitor* base, ValaAssignment* a)
{
	ValaMemoryManager * self;
	self = VALA_MEMORY_MANAGER (base);
	g_return_if_fail (VALA_IS_ASSIGNMENT (a));
	if (VALA_IS_POINTER_INDIRECTION (vala_assignment_get_left (a)) || (vala_expression_get_symbol_reference (vala_assignment_get_left (a)) != NULL && VALA_IS_SIGNAL (vala_symbol_get_node (vala_expression_get_symbol_reference (vala_assignment_get_left (a)))))) {
	} else {
		if (vala_type_reference_get_takes_ownership (vala_expression_get_static_type (vala_assignment_get_left (a)))) {
			vala_memory_manager_visit_possibly_missing_copy_expression (self, vala_assignment_get_right (a));
		} else {
			vala_memory_manager_visit_possibly_leaked_expression (self, vala_assignment_get_right (a));
		}
	}
}


static void vala_memory_manager_class_init (ValaMemoryManagerClass * klass)
{
	vala_memory_manager_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaMemoryManagerPrivate));
	G_OBJECT_CLASS (klass)->dispose = vala_memory_manager_dispose;
	VALA_CODE_VISITOR_CLASS (klass)->visit_source_file = vala_memory_manager_real_visit_source_file;
	VALA_CODE_VISITOR_CLASS (klass)->visit_namespace = vala_memory_manager_real_visit_namespace;
	VALA_CODE_VISITOR_CLASS (klass)->visit_class = vala_memory_manager_real_visit_class;
	VALA_CODE_VISITOR_CLASS (klass)->visit_struct = vala_memory_manager_real_visit_struct;
	VALA_CODE_VISITOR_CLASS (klass)->visit_interface = vala_memory_manager_real_visit_interface;
	VALA_CODE_VISITOR_CLASS (klass)->visit_field = vala_memory_manager_real_visit_field;
	VALA_CODE_VISITOR_CLASS (klass)->visit_method = vala_memory_manager_real_visit_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_creation_method = vala_memory_manager_real_visit_creation_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property = vala_memory_manager_real_visit_property;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property_accessor = vala_memory_manager_real_visit_property_accessor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_constructor = vala_memory_manager_real_visit_constructor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_destructor = vala_memory_manager_real_visit_destructor;
	VALA_CODE_VISITOR_CLASS (klass)->visit_named_argument = vala_memory_manager_real_visit_named_argument;
	VALA_CODE_VISITOR_CLASS (klass)->visit_variable_declarator = vala_memory_manager_real_visit_variable_declarator;
	VALA_CODE_VISITOR_CLASS (klass)->visit_expression_statement = vala_memory_manager_real_visit_expression_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_return_statement = vala_memory_manager_real_visit_end_return_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_throw_statement = vala_memory_manager_real_visit_throw_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_try_statement = vala_memory_manager_real_visit_try_statement;
	VALA_CODE_VISITOR_CLASS (klass)->visit_catch_clause = vala_memory_manager_real_visit_catch_clause;
	VALA_CODE_VISITOR_CLASS (klass)->visit_member_access = vala_memory_manager_real_visit_member_access;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_invocation_expression = vala_memory_manager_real_visit_end_invocation_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_binary_expression = vala_memory_manager_real_visit_binary_expression;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_assignment = vala_memory_manager_real_visit_end_assignment;
}


static void vala_memory_manager_init (ValaMemoryManager * self)
{
	self->priv = VALA_MEMORY_MANAGER_GET_PRIVATE (self);
}


static void vala_memory_manager_dispose (GObject * obj)
{
	ValaMemoryManager * self;
	ValaMemoryManagerClass * klass;
	GObjectClass * parent_class;
	self = VALA_MEMORY_MANAGER (obj);
	(self->priv->current_symbol == NULL ? NULL : (self->priv->current_symbol = (g_object_unref (self->priv->current_symbol), NULL)));
	klass = VALA_MEMORY_MANAGER_CLASS (g_type_class_peek (VALA_TYPE_MEMORY_MANAGER));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_memory_manager_get_type ()
{
	static GType vala_memory_manager_type_id = 0;
	if (G_UNLIKELY (vala_memory_manager_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaMemoryManagerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_memory_manager_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaMemoryManager), 0, (GInstanceInitFunc) vala_memory_manager_init };
		vala_memory_manager_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaMemoryManager", &g_define_type_info, 0);
	}
	return vala_memory_manager_type_id;
}




