/* valainterfacewriter.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <rasa@gmx.ch>
 */
#define VALA_FREE_CHECKED(o,f) ((o) == NULL ? NULL : ((o) = (f (o), NULL)))
#define VALA_FREE_UNCHECKED(o,f) ((o) = (f (o), NULL))

#include "valainterfacewriter.h"
#include <stdio.h>
#include <vala/valainterfacewriter.h>
#include <vala/valadatatype.h>
#include <vala/valamemberaccessibility.h>
#include <vala/valatypereference.h>
#include <vala/valacodenode.h>
#include <vala/valasymbol.h>
#include <vala/valaarray.h>
#include <vala/valaformalparameter.h>
#include <vala/valapropertyaccessor.h>

struct _ValaInterfaceWriterPrivate {
	ValaCodeContext* context;
	FILE* stream;
	gint indent;
	gboolean bol;
	char* current_cheader_filename;
};
#define VALA_INTERFACE_WRITER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_INTERFACE_WRITER, ValaInterfaceWriterPrivate))
enum  {
	VALA_INTERFACE_WRITER_DUMMY_PROPERTY
};
static void vala_interface_writer_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file);
static void vala_interface_writer_real_visit_namespace (ValaCodeVisitor* base, ValaNamespace* ns);
static void vala_interface_writer_real_visit_class (ValaCodeVisitor* base, ValaClass* cl);
static void vala_interface_writer_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st);
static void vala_interface_writer_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface);
static void vala_interface_writer_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en);
static void vala_interface_writer_real_visit_enum_value (ValaCodeVisitor* base, ValaEnumValue* ev);
static void vala_interface_writer_real_visit_flags (ValaCodeVisitor* base, ValaFlags* fl);
static void vala_interface_writer_real_visit_flags_value (ValaCodeVisitor* base, ValaFlagsValue* fv);
static void vala_interface_writer_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c);
static void vala_interface_writer_real_visit_field (ValaCodeVisitor* base, ValaField* f);
static void vala_interface_writer_write_params (ValaInterfaceWriter* self, GList* params);
static void vala_interface_writer_real_visit_callback (ValaCodeVisitor* base, ValaCallback* cb);
static void vala_interface_writer_real_visit_method (ValaCodeVisitor* base, ValaMethod* m);
static void vala_interface_writer_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m);
static void vala_interface_writer_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop);
static void vala_interface_writer_real_visit_signal (ValaCodeVisitor* base, ValaSignal* sig);
static void vala_interface_writer_write_indent (ValaInterfaceWriter* self);
static void vala_interface_writer_write_identifier (ValaInterfaceWriter* self, const char* s);
static void vala_interface_writer_write_string (ValaInterfaceWriter* self, const char* s);
static void vala_interface_writer_write_newline (ValaInterfaceWriter* self);
static void vala_interface_writer_write_begin_block (ValaInterfaceWriter* self);
static void vala_interface_writer_write_end_block (ValaInterfaceWriter* self);
static gpointer vala_interface_writer_parent_class = NULL;
static void vala_interface_writer_dispose (GObject * obj);


/**
 * Writes the public interface of the specified code context into the
 * specified file.
 *
 * @param context  a code context
 * @param filename a relative or absolute filename
 */
void vala_interface_writer_write_file (ValaInterfaceWriter* self, ValaCodeContext* context, const char* filename)
{
	ValaCodeContext* __temp0;
	FILE* __temp1;
	FILE* __temp3;
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	g_return_if_fail (VALA_IS_CODE_CONTEXT (context));
	g_return_if_fail (filename != NULL);
	__temp0 = NULL;
	self->priv->context = (__temp0 = g_object_ref (context), (self->priv->context == NULL ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL))), __temp0);
	__temp1 = NULL;
	self->priv->stream = (__temp1 = fopen (filename, "w"), (self->priv->stream == NULL ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL))), __temp1);
	{
		GList* __temp2;
		GList* file_it;
		__temp2 = NULL;
		__temp2 = vala_code_context_get_source_files (context);
		for (file_it = __temp2; file_it != NULL; file_it = file_it->next) {
			ValaSourceFile* file;
			file = file_it->data;
			{
				/* we're only interested in non-pkg source files */
				if (!vala_source_file_get_pkg (file)) {
					vala_source_file_accept (file, VALA_CODE_VISITOR (self));
				}
			}
		}
		(__temp2 == NULL ? NULL : (__temp2 = (g_list_free (__temp2), NULL)));
	}
	__temp3 = NULL;
	self->priv->stream = (__temp3 = NULL, (self->priv->stream == NULL ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL))), __temp3);
}


static void vala_interface_writer_real_visit_source_file (ValaCodeVisitor* base, ValaSourceFile* source_file)
{
	ValaInterfaceWriter * self;
	self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_SOURCE_FILE (source_file));
	vala_source_file_accept_children (source_file, VALA_CODE_VISITOR (self));
}


static void vala_interface_writer_real_visit_namespace (ValaCodeVisitor* base, ValaNamespace* ns)
{
	ValaInterfaceWriter * self;
	char* __temp4;
	char* __temp7;
	char* __temp6;
	char* __temp5;
	self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_NAMESPACE (ns));
	if (vala_namespace_get_name (ns) == NULL) {
		vala_code_node_accept_children (VALA_CODE_NODE (ns), VALA_CODE_VISITOR (self));
		return;
	}
	__temp4 = NULL;
	self->priv->current_cheader_filename = (__temp4 = vala_namespace_get_cheader_filename (ns), (self->priv->current_cheader_filename = (g_free (self->priv->current_cheader_filename), NULL)), __temp4);
	vala_interface_writer_write_indent (self);
	__temp7 = NULL;
	__temp6 = NULL;
	__temp5 = NULL;
	vala_interface_writer_write_string (self, (__temp7 = g_strdup_printf ("[CCode (cprefix = \"%s\", lower_case_cprefix = \"%s\", cheader_filename = \"%s\")]", (__temp5 = vala_namespace_get_cprefix (ns)), (__temp6 = vala_namespace_get_lower_case_cprefix (ns)), self->priv->current_cheader_filename)));
	(__temp7 = (g_free (__temp7), NULL));
	(__temp6 = (g_free (__temp6), NULL));
	(__temp5 = (g_free (__temp5), NULL));
	vala_interface_writer_write_newline (self);
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "namespace ");
	vala_interface_writer_write_identifier (self, vala_namespace_get_name (ns));
	vala_interface_writer_write_begin_block (self);
	vala_code_node_accept_children (VALA_CODE_NODE (ns), VALA_CODE_VISITOR (self));
	vala_interface_writer_write_end_block (self);
	vala_interface_writer_write_newline (self);
}


static void vala_interface_writer_real_visit_class (ValaCodeVisitor* base, ValaClass* cl)
{
	ValaInterfaceWriter * self;
	gboolean first;
	char* cheaders;
	char* __temp12;
	GList* base_types;
	self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_CLASS (cl));
	if (VALA_DATA_TYPE (cl)->access == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		return;
	}
	vala_interface_writer_write_indent (self);
	first = TRUE;
	cheaders = NULL;
	{
		GList* __temp11;
		GList* cheader_it;
		__temp11 = NULL;
		__temp11 = vala_data_type_get_cheader_filenames (VALA_DATA_TYPE (cl));
		for (cheader_it = __temp11; cheader_it != NULL; cheader_it = cheader_it->next) {
			char* cheader;
			cheader = cheader_it->data;
			{
				if (first) {
					char* __temp9;
					const char* __temp8;
					__temp9 = NULL;
					__temp8 = NULL;
					cheaders = (__temp9 = (__temp8 = cheader, (__temp8 == NULL ? NULL : g_strdup (__temp8))), (cheaders = (g_free (cheaders), NULL)), __temp9);
					first = FALSE;
				} else {
					char* __temp10;
					__temp10 = NULL;
					cheaders = (__temp10 = g_strdup_printf ("%s, %s", cheaders, cheader), (cheaders = (g_free (cheaders), NULL)), __temp10);
				}
			}
		}
		(__temp11 == NULL ? NULL : (__temp11 = (g_list_foreach (__temp11, (GFunc) g_free, NULL), g_list_free (__temp11), NULL)));
	}
	__temp12 = NULL;
	vala_interface_writer_write_string (self, (__temp12 = g_strdup_printf ("[CCode (cheader_filename = \"%s\")]", cheaders)));
	(__temp12 = (g_free (__temp12), NULL));
	vala_interface_writer_write_newline (self);
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public ");
	if (vala_class_get_is_abstract (cl)) {
		vala_interface_writer_write_string (self, "abstract ");
	}
	vala_interface_writer_write_string (self, "class ");
	vala_interface_writer_write_identifier (self, vala_data_type_get_name (VALA_DATA_TYPE (cl)));
	base_types = vala_class_get_base_types (cl);
	if (base_types != NULL) {
		gboolean first;
		vala_interface_writer_write_string (self, " : ");
		first = TRUE;
		{
			GList* __temp14;
			GList* base_type_it;
			__temp14 = NULL;
			__temp14 = base_types;
			for (base_type_it = __temp14; base_type_it != NULL; base_type_it = base_type_it->next) {
				ValaTypeReference* base_type;
				base_type = base_type_it->data;
				{
					char* __temp13;
					if (!first) {
						vala_interface_writer_write_string (self, ", ");
					} else {
						first = FALSE;
					}
					__temp13 = NULL;
					vala_interface_writer_write_string (self, (__temp13 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (base_type))))));
					(__temp13 = (g_free (__temp13), NULL));
				}
			}
		}
	}
	vala_interface_writer_write_begin_block (self);
	vala_code_node_accept_children (VALA_CODE_NODE (cl), VALA_CODE_VISITOR (self));
	vala_interface_writer_write_end_block (self);
	vala_interface_writer_write_newline (self);
	(cheaders = (g_free (cheaders), NULL));
	(base_types == NULL ? NULL : (base_types = (g_list_free (base_types), NULL)));
}


static void vala_interface_writer_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st)
{
	ValaInterfaceWriter * self;
	self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_STRUCT (st));
	if (VALA_DATA_TYPE (st)->access == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		return;
	}
	if (vala_data_type_is_reference_type (VALA_DATA_TYPE (st))) {
		vala_interface_writer_write_indent (self);
		vala_interface_writer_write_string (self, "[ReferenceType]");
	}
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public struct ");
	vala_interface_writer_write_identifier (self, vala_data_type_get_name (VALA_DATA_TYPE (st)));
	vala_interface_writer_write_begin_block (self);
	vala_code_node_accept_children (VALA_CODE_NODE (st), VALA_CODE_VISITOR (self));
	vala_interface_writer_write_end_block (self);
	vala_interface_writer_write_newline (self);
}


static void vala_interface_writer_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface)
{
	ValaInterfaceWriter * self;
	gboolean first;
	char* cheaders;
	char* __temp19;
	self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_INTERFACE (iface));
	if (VALA_DATA_TYPE (iface)->access == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		return;
	}
	vala_interface_writer_write_indent (self);
	first = TRUE;
	cheaders = NULL;
	{
		GList* __temp18;
		GList* cheader_it;
		__temp18 = NULL;
		__temp18 = vala_data_type_get_cheader_filenames (VALA_DATA_TYPE (iface));
		for (cheader_it = __temp18; cheader_it != NULL; cheader_it = cheader_it->next) {
			char* cheader;
			cheader = cheader_it->data;
			{
				if (first) {
					char* __temp16;
					const char* __temp15;
					__temp16 = NULL;
					__temp15 = NULL;
					cheaders = (__temp16 = (__temp15 = cheader, (__temp15 == NULL ? NULL : g_strdup (__temp15))), (cheaders = (g_free (cheaders), NULL)), __temp16);
					first = FALSE;
				} else {
					char* __temp17;
					__temp17 = NULL;
					cheaders = (__temp17 = g_strdup_printf ("%s, %s", cheaders, cheader), (cheaders = (g_free (cheaders), NULL)), __temp17);
				}
			}
		}
		(__temp18 == NULL ? NULL : (__temp18 = (g_list_foreach (__temp18, (GFunc) g_free, NULL), g_list_free (__temp18), NULL)));
	}
	__temp19 = NULL;
	vala_interface_writer_write_string (self, (__temp19 = g_strdup_printf ("[CCode (cheader_filename = \"%s\")]", cheaders)));
	(__temp19 = (g_free (__temp19), NULL));
	vala_interface_writer_write_newline (self);
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public ");
	vala_interface_writer_write_string (self, "interface ");
	vala_interface_writer_write_identifier (self, vala_data_type_get_name (VALA_DATA_TYPE (iface)));
	vala_interface_writer_write_begin_block (self);
	vala_code_node_accept_children (VALA_CODE_NODE (iface), VALA_CODE_VISITOR (self));
	vala_interface_writer_write_end_block (self);
	vala_interface_writer_write_newline (self);
	(cheaders = (g_free (cheaders), NULL));
}


static void vala_interface_writer_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en)
{
	ValaInterfaceWriter * self;
	char* __temp21;
	char* __temp20;
	self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_ENUM (en));
	if (VALA_DATA_TYPE (en)->access == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		return;
	}
	vala_interface_writer_write_indent (self);
	__temp21 = NULL;
	__temp20 = NULL;
	vala_interface_writer_write_string (self, (__temp21 = g_strdup_printf ("[CCode (cprefix = \"%s\")]", (__temp20 = vala_enum_get_cprefix (en)))));
	(__temp21 = (g_free (__temp21), NULL));
	(__temp20 = (g_free (__temp20), NULL));
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public enum ");
	vala_interface_writer_write_identifier (self, vala_data_type_get_name (VALA_DATA_TYPE (en)));
	vala_interface_writer_write_begin_block (self);
	vala_code_node_accept_children (VALA_CODE_NODE (en), VALA_CODE_VISITOR (self));
	vala_interface_writer_write_end_block (self);
	vala_interface_writer_write_newline (self);
}


static void vala_interface_writer_real_visit_enum_value (ValaCodeVisitor* base, ValaEnumValue* ev)
{
	ValaInterfaceWriter * self;
	self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_ENUM_VALUE (ev));
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_identifier (self, vala_enum_value_get_name (ev));
	vala_interface_writer_write_string (self, ",");
	vala_interface_writer_write_newline (self);
}


static void vala_interface_writer_real_visit_flags (ValaCodeVisitor* base, ValaFlags* fl)
{
	ValaInterfaceWriter * self;
	char* __temp23;
	char* __temp22;
	self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_FLAGS (fl));
	if (VALA_DATA_TYPE (fl)->access == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		return;
	}
	vala_interface_writer_write_indent (self);
	__temp23 = NULL;
	__temp22 = NULL;
	vala_interface_writer_write_string (self, (__temp23 = g_strdup_printf ("[CCode (cprefix = \"%s\")]", (__temp22 = vala_flags_get_cprefix (fl)))));
	(__temp23 = (g_free (__temp23), NULL));
	(__temp22 = (g_free (__temp22), NULL));
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public flags ");
	vala_interface_writer_write_identifier (self, vala_data_type_get_name (VALA_DATA_TYPE (fl)));
	vala_interface_writer_write_begin_block (self);
	vala_code_node_accept_children (VALA_CODE_NODE (fl), VALA_CODE_VISITOR (self));
	vala_interface_writer_write_end_block (self);
	vala_interface_writer_write_newline (self);
}


static void vala_interface_writer_real_visit_flags_value (ValaCodeVisitor* base, ValaFlagsValue* fv)
{
	ValaInterfaceWriter * self;
	self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_FLAGS_VALUE (fv));
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_identifier (self, vala_flags_value_get_name (fv));
	vala_interface_writer_write_string (self, ",");
	vala_interface_writer_write_newline (self);
}


static void vala_interface_writer_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c)
{
	ValaInterfaceWriter * self;
	char* __temp24;
	self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_CONSTANT (c));
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public const ");
	__temp24 = NULL;
	vala_interface_writer_write_string (self, (__temp24 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_constant_get_type_reference (c)))))));
	(__temp24 = (g_free (__temp24), NULL));
	vala_interface_writer_write_string (self, " ");
	vala_interface_writer_write_identifier (self, vala_constant_get_name (c));
	vala_interface_writer_write_string (self, ";");
	vala_interface_writer_write_newline (self);
}


static void vala_interface_writer_real_visit_field (ValaCodeVisitor* base, ValaField* f)
{
	ValaInterfaceWriter * self;
	char* __temp25;
	GList* type_args;
	self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_FIELD (f));
	if (f->access == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		return;
	}
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public ");
	if (vala_type_reference_get_data_type (vala_field_get_type_reference (f)) != NULL && vala_data_type_is_reference_type (vala_type_reference_get_data_type (vala_field_get_type_reference (f))) && !vala_type_reference_get_takes_ownership (vala_field_get_type_reference (f))) {
		vala_interface_writer_write_string (self, "weak ");
	}
	__temp25 = NULL;
	vala_interface_writer_write_string (self, (__temp25 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_field_get_type_reference (f)))))));
	(__temp25 = (g_free (__temp25), NULL));
	type_args = vala_type_reference_get_type_arguments (vala_field_get_type_reference (f));
	if (!(VALA_IS_ARRAY (vala_type_reference_get_data_type (vala_field_get_type_reference (f)))) && type_args != NULL) {
		vala_interface_writer_write_string (self, "<");
		{
			GList* __temp27;
			GList* type_arg_it;
			__temp27 = NULL;
			__temp27 = type_args;
			for (type_arg_it = __temp27; type_arg_it != NULL; type_arg_it = type_arg_it->next) {
				ValaTypeReference* type_arg;
				type_arg = type_arg_it->data;
				{
					char* __temp26;
					if (!vala_type_reference_get_takes_ownership (type_arg)) {
						vala_interface_writer_write_string (self, "weak ");
					}
					__temp26 = NULL;
					vala_interface_writer_write_string (self, (__temp26 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (type_arg))))));
					(__temp26 = (g_free (__temp26), NULL));
				}
			}
		}
		vala_interface_writer_write_string (self, ">");
	}
	vala_interface_writer_write_string (self, " ");
	vala_interface_writer_write_identifier (self, vala_field_get_name (f));
	vala_interface_writer_write_string (self, ";");
	vala_interface_writer_write_newline (self);
	(type_args == NULL ? NULL : (type_args = (g_list_free (type_args), NULL)));
}


static void vala_interface_writer_write_params (ValaInterfaceWriter* self, GList* params)
{
	gboolean first;
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	vala_interface_writer_write_string (self, "(");
	first = TRUE;
	{
		GList* __temp32;
		GList* param_it;
		__temp32 = NULL;
		__temp32 = params;
		for (param_it = __temp32; param_it != NULL; param_it = param_it->next) {
			ValaFormalParameter* param;
			param = param_it->data;
			{
				char* __temp28;
				GList* type_args;
				if (!first) {
					vala_interface_writer_write_string (self, ", ");
				} else {
					first = FALSE;
				}
				if (vala_formal_parameter_get_ellipsis (param)) {
					vala_interface_writer_write_string (self, "...");
					continue;
				}
				if (vala_type_reference_get_is_ref (vala_formal_parameter_get_type_reference (param))) {
					vala_interface_writer_write_string (self, "ref ");
				} else {
					if (vala_type_reference_get_is_out (vala_formal_parameter_get_type_reference (param))) {
						vala_interface_writer_write_string (self, "out ");
					}
				}
				__temp28 = NULL;
				vala_interface_writer_write_string (self, (__temp28 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (param)))))));
				(__temp28 = (g_free (__temp28), NULL));
				type_args = vala_type_reference_get_type_arguments (vala_formal_parameter_get_type_reference (param));
				if (!(VALA_IS_ARRAY (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (param)))) && type_args != NULL) {
					vala_interface_writer_write_string (self, "<");
					{
						GList* __temp30;
						GList* type_arg_it;
						__temp30 = NULL;
						__temp30 = type_args;
						for (type_arg_it = __temp30; type_arg_it != NULL; type_arg_it = type_arg_it->next) {
							ValaTypeReference* type_arg;
							type_arg = type_arg_it->data;
							{
								char* __temp29;
								if (!vala_type_reference_get_takes_ownership (type_arg)) {
									vala_interface_writer_write_string (self, "weak ");
								}
								__temp29 = NULL;
								vala_interface_writer_write_string (self, (__temp29 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (type_arg))))));
								(__temp29 = (g_free (__temp29), NULL));
							}
						}
					}
					vala_interface_writer_write_string (self, ">");
				}
				if (vala_type_reference_get_non_null (vala_formal_parameter_get_type_reference (param))) {
					vala_interface_writer_write_string (self, "!");
				}
				if (vala_type_reference_get_takes_ownership (vala_formal_parameter_get_type_reference (param))) {
					vala_interface_writer_write_string (self, "#");
				}
				vala_interface_writer_write_string (self, " ");
				vala_interface_writer_write_identifier (self, vala_formal_parameter_get_name (param));
				if (vala_formal_parameter_get_default_expression (param) != NULL) {
					char* __temp31;
					vala_interface_writer_write_string (self, " = ");
					__temp31 = NULL;
					vala_interface_writer_write_string (self, (__temp31 = vala_code_node_to_string (VALA_CODE_NODE (vala_formal_parameter_get_default_expression (param)))));
					(__temp31 = (g_free (__temp31), NULL));
				}
				(type_args == NULL ? NULL : (type_args = (g_list_free (type_args), NULL)));
			}
		}
	}
	vala_interface_writer_write_string (self, ")");
}


static void vala_interface_writer_real_visit_callback (ValaCodeVisitor* base, ValaCallback* cb)
{
	ValaInterfaceWriter * self;
	ValaDataType* __temp33;
	ValaDataType* type;
	GList* __temp35;
	self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_CALLBACK (cb));
	if (VALA_DATA_TYPE (cb)->access == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		return;
	}
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public static delegate ");
	__temp33 = NULL;
	type = (__temp33 = vala_type_reference_get_data_type (vala_callback_get_return_type (cb)), (__temp33 == NULL ? NULL : g_object_ref (__temp33)));
	if (type == NULL) {
		vala_interface_writer_write_string (self, "void");
	} else {
		char* __temp34;
		if (vala_type_reference_get_transfers_ownership (vala_callback_get_return_type (cb))) {
			vala_interface_writer_write_string (self, "ref ");
		}
		__temp34 = NULL;
		vala_interface_writer_write_string (self, (__temp34 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_callback_get_return_type (cb)))))));
		(__temp34 = (g_free (__temp34), NULL));
	}
	vala_interface_writer_write_string (self, " ");
	vala_interface_writer_write_identifier (self, vala_data_type_get_name (VALA_DATA_TYPE (cb)));
	vala_interface_writer_write_string (self, " ");
	__temp35 = NULL;
	vala_interface_writer_write_params (self, (__temp35 = vala_callback_get_parameters (cb)));
	(__temp35 == NULL ? NULL : (__temp35 = (g_list_free (__temp35), NULL)));
	vala_interface_writer_write_string (self, ";");
	vala_interface_writer_write_newline (self);
	(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
}


static void vala_interface_writer_real_visit_method (ValaCodeVisitor* base, ValaMethod* m)
{
	ValaInterfaceWriter * self;
	char* __temp38;
	char* __temp37;
	gboolean __temp39;
	GList* __temp45;
	self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_METHOD (m));
	if (m->access == VALA_MEMBER_ACCESSIBILITY_PRIVATE || vala_method_get_overrides (m)) {
		return;
	}
	if (vala_method_get_no_array_length (m)) {
		gboolean array_found;
		array_found = (vala_method_get_return_type (m) != NULL && VALA_IS_ARRAY (vala_type_reference_get_data_type (vala_method_get_return_type (m))));
		{
			GList* __temp36;
			GList* param_it;
			__temp36 = NULL;
			__temp36 = vala_invokable_get_parameters (VALA_INVOKABLE (m));
			for (param_it = __temp36; param_it != NULL; param_it = param_it->next) {
				ValaFormalParameter* param;
				param = param_it->data;
				{
					if (vala_formal_parameter_get_type_reference (param) != NULL && VALA_IS_ARRAY (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (param)))) {
						array_found = TRUE;
						break;
					}
				}
			}
			(__temp36 == NULL ? NULL : (__temp36 = (g_list_free (__temp36), NULL)));
		}
		if (array_found) {
			vala_interface_writer_write_indent (self);
			vala_interface_writer_write_string (self, "[NoArrayLength]");
		}
	}
	if (vala_method_get_instance_last (m)) {
		vala_interface_writer_write_indent (self);
		vala_interface_writer_write_string (self, "[InstanceLast]");
	}
	if (vala_method_get_instance_by_reference (m)) {
		vala_interface_writer_write_indent (self);
		vala_interface_writer_write_string (self, "[InstanceByReference]");
	}
	__temp38 = NULL;
	__temp37 = NULL;
	if ((__temp39 = g_utf8_collate ((__temp37 = vala_method_get_cname (m)), (__temp38 = vala_method_get_default_cname (m))) != 0, (__temp38 = (g_free (__temp38), NULL)), (__temp37 = (g_free (__temp37), NULL)), __temp39)) {
		char* __temp41;
		char* __temp40;
		vala_interface_writer_write_indent (self);
		__temp41 = NULL;
		__temp40 = NULL;
		vala_interface_writer_write_string (self, (__temp41 = g_strdup_printf ("[CCode (cname = \"%s\")]", (__temp40 = vala_method_get_cname (m)))));
		(__temp41 = (g_free (__temp41), NULL));
		(__temp40 = (g_free (__temp40), NULL));
	}
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public");
	if (VALA_IS_CREATION_METHOD (m)) {
		ValaDataType* __temp42;
		ValaDataType* datatype;
		vala_interface_writer_write_string (self, " ");
		__temp42 = NULL;
		datatype = (__temp42 = VALA_DATA_TYPE (vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (m))))), (__temp42 == NULL ? NULL : g_object_ref (__temp42)));
		vala_interface_writer_write_identifier (self, vala_data_type_get_name (datatype));
		if (vala_method_get_name (m) != NULL) {
			vala_interface_writer_write_string (self, ".");
			vala_interface_writer_write_identifier (self, vala_method_get_name (m));
		}
		(datatype == NULL ? NULL : (datatype = (g_object_unref (datatype), NULL)));
	} else {
		if (!vala_method_get_instance (m)) {
			vala_interface_writer_write_string (self, " static");
		} else {
			if (vala_method_get_is_abstract (m)) {
				vala_interface_writer_write_string (self, " abstract");
			} else {
				if (vala_method_get_is_virtual (m)) {
					vala_interface_writer_write_string (self, " virtual");
				}
			}
		}
	}
	if (!(VALA_IS_CREATION_METHOD (m))) {
		ValaDataType* __temp43;
		ValaDataType* type;
		vala_interface_writer_write_string (self, " ");
		__temp43 = NULL;
		type = (__temp43 = vala_type_reference_get_data_type (vala_method_get_return_type (m)), (__temp43 == NULL ? NULL : g_object_ref (__temp43)));
		if (type == NULL) {
			vala_interface_writer_write_string (self, "void");
		} else {
			char* __temp44;
			if (vala_type_reference_get_transfers_ownership (vala_method_get_return_type (m))) {
			} else {
				if ((vala_type_reference_get_data_type (vala_method_get_return_type (m)) != NULL && vala_data_type_is_reference_type (vala_type_reference_get_data_type (vala_method_get_return_type (m)))) || vala_type_reference_get_type_parameter (vala_method_get_return_type (m)) != NULL) {
					vala_interface_writer_write_string (self, "weak ");
				}
			}
			__temp44 = NULL;
			vala_interface_writer_write_string (self, (__temp44 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_method_get_return_type (m)))))));
			(__temp44 = (g_free (__temp44), NULL));
			if (vala_type_reference_get_non_null (vala_method_get_return_type (m))) {
				vala_interface_writer_write_string (self, "!");
			}
		}
		vala_interface_writer_write_string (self, " ");
		vala_interface_writer_write_identifier (self, vala_method_get_name (m));
		(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
	}
	vala_interface_writer_write_string (self, " ");
	__temp45 = NULL;
	vala_interface_writer_write_params (self, (__temp45 = vala_invokable_get_parameters (VALA_INVOKABLE (m))));
	(__temp45 == NULL ? NULL : (__temp45 = (g_list_free (__temp45), NULL)));
	vala_interface_writer_write_string (self, ";");
	vala_interface_writer_write_newline (self);
}


static void vala_interface_writer_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m)
{
	ValaInterfaceWriter * self;
	self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_CREATION_METHOD (m));
	vala_code_visitor_visit_method (VALA_CODE_VISITOR (self), VALA_METHOD (m));
}


static void vala_interface_writer_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop)
{
	ValaInterfaceWriter * self;
	char* __temp46;
	GList* type_args;
	self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_PROPERTY (prop));
	if (vala_property_get_no_accessor_method (prop)) {
		vala_interface_writer_write_indent (self);
		vala_interface_writer_write_string (self, "[NoAccessorMethod]");
	}
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public ");
	if (!vala_type_reference_get_takes_ownership (vala_property_get_type_reference (prop))) {
		vala_interface_writer_write_string (self, "weak ");
	}
	__temp46 = NULL;
	vala_interface_writer_write_string (self, (__temp46 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_property_get_type_reference (prop)))))));
	(__temp46 = (g_free (__temp46), NULL));
	type_args = vala_type_reference_get_type_arguments (vala_property_get_type_reference (prop));
	if (!(VALA_IS_ARRAY (vala_type_reference_get_data_type (vala_property_get_type_reference (prop)))) && type_args != NULL) {
		vala_interface_writer_write_string (self, "<");
		{
			GList* __temp48;
			GList* type_arg_it;
			__temp48 = NULL;
			__temp48 = type_args;
			for (type_arg_it = __temp48; type_arg_it != NULL; type_arg_it = type_arg_it->next) {
				ValaTypeReference* type_arg;
				type_arg = type_arg_it->data;
				{
					char* __temp47;
					if (!vala_type_reference_get_takes_ownership (type_arg)) {
						vala_interface_writer_write_string (self, "weak ");
					}
					__temp47 = NULL;
					vala_interface_writer_write_string (self, (__temp47 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (type_arg))))));
					(__temp47 = (g_free (__temp47), NULL));
				}
			}
		}
		vala_interface_writer_write_string (self, ">");
	}
	vala_interface_writer_write_string (self, " ");
	vala_interface_writer_write_identifier (self, vala_property_get_name (prop));
	vala_interface_writer_write_string (self, " {");
	if (vala_property_get_get_accessor (prop) != NULL) {
		vala_interface_writer_write_string (self, " get;");
	}
	if (vala_property_get_set_accessor (prop) != NULL) {
		if (vala_property_accessor_get_writable (vala_property_get_set_accessor (prop))) {
			vala_interface_writer_write_string (self, " set");
		}
		if (vala_property_accessor_get_construction (vala_property_get_set_accessor (prop))) {
			vala_interface_writer_write_string (self, " construct");
		}
		vala_interface_writer_write_string (self, ";");
	}
	vala_interface_writer_write_string (self, " }");
	vala_interface_writer_write_newline (self);
	(type_args == NULL ? NULL : (type_args = (g_list_free (type_args), NULL)));
}


static void vala_interface_writer_real_visit_signal (ValaCodeVisitor* base, ValaSignal* sig)
{
	ValaInterfaceWriter * self;
	ValaDataType* __temp49;
	ValaDataType* type;
	GList* __temp51;
	self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_SIGNAL (sig));
	if (vala_signal_get_access (sig) == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		return;
	}
	if (vala_signal_get_has_emitter (sig)) {
		vala_interface_writer_write_indent (self);
		vala_interface_writer_write_string (self, "[HasEmitter]");
	}
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public signal ");
	__temp49 = NULL;
	type = (__temp49 = vala_type_reference_get_data_type (vala_signal_get_return_type (sig)), (__temp49 == NULL ? NULL : g_object_ref (__temp49)));
	if (type == NULL) {
		vala_interface_writer_write_string (self, "void");
	} else {
		char* __temp50;
		if (vala_type_reference_get_transfers_ownership (vala_signal_get_return_type (sig))) {
			vala_interface_writer_write_string (self, "ref ");
		}
		__temp50 = NULL;
		vala_interface_writer_write_string (self, (__temp50 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_signal_get_return_type (sig)))))));
		(__temp50 = (g_free (__temp50), NULL));
		if (vala_type_reference_get_non_null (vala_signal_get_return_type (sig))) {
			vala_interface_writer_write_string (self, "!");
		}
	}
	vala_interface_writer_write_string (self, " ");
	vala_interface_writer_write_identifier (self, vala_signal_get_name (sig));
	vala_interface_writer_write_string (self, " ");
	__temp51 = NULL;
	vala_interface_writer_write_params (self, (__temp51 = vala_invokable_get_parameters (VALA_INVOKABLE (sig))));
	(__temp51 == NULL ? NULL : (__temp51 = (g_list_free (__temp51), NULL)));
	vala_interface_writer_write_string (self, ";");
	vala_interface_writer_write_newline (self);
	(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
}


static void vala_interface_writer_write_indent (ValaInterfaceWriter* self)
{
	gint i;
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	i = 0;
	if (!self->priv->bol) {
		fputc ('\n', self->priv->stream);
	}
	for (i = 0; i < self->priv->indent; i++) {
		fputc ('\t', self->priv->stream);
	}
	self->priv->bol = FALSE;
}


static void vala_interface_writer_write_identifier (ValaInterfaceWriter* self, const char* s)
{
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	g_return_if_fail (s != NULL);
	if (g_utf8_collate (s, "base") == 0 || g_utf8_collate (s, "callback") == 0 || g_utf8_collate (s, "class") == 0 || g_utf8_collate (s, "construct") == 0 || g_utf8_collate (s, "flags") == 0 || g_utf8_collate (s, "foreach") == 0 || g_utf8_collate (s, "in") == 0 || g_utf8_collate (s, "interface") == 0 || g_utf8_collate (s, "lock") == 0 || g_utf8_collate (s, "namespace") == 0 || g_utf8_collate (s, "out") == 0 || g_utf8_collate (s, "ref") == 0) {
		fputc ('@', self->priv->stream);
	}
	vala_interface_writer_write_string (self, s);
}


static void vala_interface_writer_write_string (ValaInterfaceWriter* self, const char* s)
{
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	g_return_if_fail (s != NULL);
	fprintf (self->priv->stream, "%s", s);
	self->priv->bol = FALSE;
}


static void vala_interface_writer_write_newline (ValaInterfaceWriter* self)
{
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	fputc ('\n', self->priv->stream);
	self->priv->bol = TRUE;
}


static void vala_interface_writer_write_begin_block (ValaInterfaceWriter* self)
{
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	if (!self->priv->bol) {
		fputc (' ', self->priv->stream);
	} else {
		vala_interface_writer_write_indent (self);
	}
	fputc ('{', self->priv->stream);
	vala_interface_writer_write_newline (self);
	self->priv->indent++;
}


static void vala_interface_writer_write_end_block (ValaInterfaceWriter* self)
{
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	self->priv->indent--;
	vala_interface_writer_write_indent (self);
	fprintf (self->priv->stream, "}");
}


static void vala_interface_writer_class_init (ValaInterfaceWriterClass * klass)
{
	vala_interface_writer_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaInterfaceWriterPrivate));
	G_OBJECT_CLASS (klass)->dispose = vala_interface_writer_dispose;
	VALA_CODE_VISITOR_CLASS (klass)->visit_source_file = vala_interface_writer_real_visit_source_file;
	VALA_CODE_VISITOR_CLASS (klass)->visit_namespace = vala_interface_writer_real_visit_namespace;
	VALA_CODE_VISITOR_CLASS (klass)->visit_class = vala_interface_writer_real_visit_class;
	VALA_CODE_VISITOR_CLASS (klass)->visit_struct = vala_interface_writer_real_visit_struct;
	VALA_CODE_VISITOR_CLASS (klass)->visit_interface = vala_interface_writer_real_visit_interface;
	VALA_CODE_VISITOR_CLASS (klass)->visit_enum = vala_interface_writer_real_visit_enum;
	VALA_CODE_VISITOR_CLASS (klass)->visit_enum_value = vala_interface_writer_real_visit_enum_value;
	VALA_CODE_VISITOR_CLASS (klass)->visit_flags = vala_interface_writer_real_visit_flags;
	VALA_CODE_VISITOR_CLASS (klass)->visit_flags_value = vala_interface_writer_real_visit_flags_value;
	VALA_CODE_VISITOR_CLASS (klass)->visit_constant = vala_interface_writer_real_visit_constant;
	VALA_CODE_VISITOR_CLASS (klass)->visit_field = vala_interface_writer_real_visit_field;
	VALA_CODE_VISITOR_CLASS (klass)->visit_callback = vala_interface_writer_real_visit_callback;
	VALA_CODE_VISITOR_CLASS (klass)->visit_method = vala_interface_writer_real_visit_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_creation_method = vala_interface_writer_real_visit_creation_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property = vala_interface_writer_real_visit_property;
	VALA_CODE_VISITOR_CLASS (klass)->visit_signal = vala_interface_writer_real_visit_signal;
}


static void vala_interface_writer_init (ValaInterfaceWriter * self)
{
	self->priv = VALA_INTERFACE_WRITER_GET_PRIVATE (self);
	self->priv->bol = TRUE;
}


static void vala_interface_writer_dispose (GObject * obj)
{
	ValaInterfaceWriter * self;
	ValaInterfaceWriterClass * klass;
	GObjectClass * parent_class;
	self = VALA_INTERFACE_WRITER (obj);
	(self->priv->context == NULL ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL)));
	(self->priv->stream == NULL ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL)));
	(self->priv->current_cheader_filename = (g_free (self->priv->current_cheader_filename), NULL));
	klass = VALA_INTERFACE_WRITER_CLASS (g_type_class_peek (VALA_TYPE_INTERFACE_WRITER));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_interface_writer_get_type ()
{
	static GType vala_interface_writer_type_id = 0;
	if (G_UNLIKELY (vala_interface_writer_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaInterfaceWriterClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_interface_writer_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaInterfaceWriter), 0, (GInstanceInitFunc) vala_interface_writer_init };
		vala_interface_writer_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaInterfaceWriter", &g_define_type_info, 0);
	}
	return vala_interface_writer_type_id;
}




