/* valanamespace.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */
#define VALA_FREE_CHECKED(o,f) ((o) == NULL ? NULL : ((o) = (f (o), NULL)))
#define VALA_FREE_UNCHECKED(o,f) ((o) = (f (o), NULL))

#include "valanamespace.h"
#include <vala/valasourcereference.h>
#include <vala/valanamespace.h>
#include <vala/valaclass.h>
#include <vala/valadatatype.h>
#include <vala/valainterface.h>
#include <vala/valastruct.h>
#include <vala/valaenum.h>
#include <vala/valaflags.h>
#include <vala/valacallback.h>
#include <vala/valaconstant.h>
#include <vala/valafield.h>
#include <vala/valamethod.h>
#include <vala/valacodevisitor.h>
#include <vala/valasourcefile.h>
#include <vala/valaattribute.h>

struct _ValaNamespacePrivate {
	char* _name;
	GList* classes;
	GList* interfaces;
	GList* structs;
	GList* enums;
	GList* flags_;
	GList* callbacks;
	GList* constants;
	GList* fields;
	GList* methods;
	char* cprefix;
	char* lower_case_cprefix;
	GList* cheader_filenames;
};
#define VALA_NAMESPACE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_NAMESPACE, ValaNamespacePrivate))
enum  {
	VALA_NAMESPACE_DUMMY_PROPERTY,
	VALA_NAMESPACE_NAME
};
static void vala_namespace_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_namespace_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_namespace_process_ccode_attribute (ValaNamespace* self, ValaAttribute* a);
static gpointer vala_namespace_parent_class = NULL;
static void vala_namespace_dispose (GObject * obj);


/**
 * Creates a new namespace.
 *
 * @param name   namespace name
 * @param source reference to source code
 * @return       newly created namespace
 */
ValaNamespace* vala_namespace_new (const char* _name, ValaSourceReference* source)
{
	GParameter * __params;
	GParameter * __params_it;
	ValaNamespace * self;
	g_return_val_if_fail (source == NULL || VALA_IS_SOURCE_REFERENCE (source), NULL);
	__params = g_new0 (GParameter, 2);
	__params_it = __params;
	(__params_it->name = "name", g_value_init (&__params_it->value, G_TYPE_STRING), g_value_set_string (&__params_it->value, _name), __params_it++);
	(__params_it->name = "source-reference", g_value_init (&__params_it->value, VALA_TYPE_SOURCE_REFERENCE), g_value_set_object (&__params_it->value, source), __params_it++);
	self = g_object_newv (VALA_TYPE_NAMESPACE, __params_it - __params, __params);
	return self;
}


/**
 * Adds the specified class to this namespace.
 *
 * @param cl a class
 */
void vala_namespace_add_class (ValaNamespace* self, ValaClass* cl)
{
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_CLASS (cl));
	self->priv->classes = g_list_append (self->priv->classes, g_object_ref (cl));
	VALA_DATA_TYPE (cl)->namespace = self;
}


/**
 * Removes the specified class from this namespace.
 *
 * @param cl a class
 */
void vala_namespace_remove_class (ValaNamespace* self, ValaClass* cl)
{
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_CLASS (cl));
	VALA_DATA_TYPE (cl)->namespace = NULL;
	self->priv->classes = g_list_remove (self->priv->classes, cl);
}


/**
 * Adds the specified interface to this namespace.
 *
 * @param iface an interface
 */
void vala_namespace_add_interface (ValaNamespace* self, ValaInterface* iface)
{
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_INTERFACE (iface));
	self->priv->interfaces = g_list_append (self->priv->interfaces, g_object_ref (iface));
	VALA_DATA_TYPE (iface)->namespace = self;
}


/**
 * Adds the specified struct to this namespace.
 *
 * @param st a struct
 */
void vala_namespace_add_struct (ValaNamespace* self, ValaStruct* st)
{
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_STRUCT (st));
	self->priv->structs = g_list_append (self->priv->structs, g_object_ref (st));
	VALA_DATA_TYPE (st)->namespace = self;
}


/**
 * Adds the specified enum to this namespace.
 *
 * @param en an enum
 */
void vala_namespace_add_enum (ValaNamespace* self, ValaEnum* en)
{
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_ENUM (en));
	self->priv->enums = g_list_append (self->priv->enums, g_object_ref (en));
	VALA_DATA_TYPE (en)->namespace = self;
}


/**
 * Adds the specified flags to this namespace.
 *
 * @param fl a flags
 */
void vala_namespace_add_flags (ValaNamespace* self, ValaFlags* fl)
{
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_FLAGS (fl));
	self->priv->flags_ = g_list_append (self->priv->flags_, g_object_ref (fl));
	VALA_DATA_TYPE (fl)->namespace = self;
}


/**
 * Adds the specified callback to this namespace.
 *
 * @param cb a callback
 */
void vala_namespace_add_callback (ValaNamespace* self, ValaCallback* cb)
{
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_CALLBACK (cb));
	self->priv->callbacks = g_list_append (self->priv->callbacks, g_object_ref (cb));
	VALA_DATA_TYPE (cb)->namespace = self;
}


/**
 * Returns a copy of the list of structs.
 *
 * @return struct list
 */
GList* vala_namespace_get_structs (ValaNamespace* self)
{
	g_return_val_if_fail (VALA_IS_NAMESPACE (self), NULL);
	return g_list_copy (self->priv->structs);
}


/**
 * Returns a copy of the list of classes.
 *
 * @return class list
 */
GList* vala_namespace_get_classes (ValaNamespace* self)
{
	g_return_val_if_fail (VALA_IS_NAMESPACE (self), NULL);
	return g_list_copy (self->priv->classes);
}


/**
 * Returns a copy of the list of interfaces.
 *
 * @return interface list
 */
GList* vala_namespace_get_interfaces (ValaNamespace* self)
{
	g_return_val_if_fail (VALA_IS_NAMESPACE (self), NULL);
	return g_list_copy (self->priv->interfaces);
}


/**
 * Adds the specified constant to this namespace.
 *
 * @param constant a constant
 */
void vala_namespace_add_constant (ValaNamespace* self, ValaConstant* constant)
{
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_CONSTANT (constant));
	self->priv->constants = g_list_append (self->priv->constants, g_object_ref (constant));
}


/**
 * Adds the specified field to this namespace.
 *
 * @param f a field
 */
void vala_namespace_add_field (ValaNamespace* self, ValaField* f)
{
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_FIELD (f));
	self->priv->fields = g_list_append (self->priv->fields, g_object_ref (f));
}


/**
 * Adds the specified method to this namespace.
 *
 * @param m a method
 */
void vala_namespace_add_method (ValaNamespace* self, ValaMethod* m)
{
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_METHOD (m));
	self->priv->methods = g_list_append (self->priv->methods, g_object_ref (m));
}


static void vala_namespace_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaNamespace * self;
	self = VALA_NAMESPACE (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_namespace (visitor, self);
}


static void vala_namespace_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaNamespace * self;
	self = VALA_NAMESPACE (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	{
		GList* __temp3;
		GList* en_it;
		__temp3 = NULL;
		__temp3 = self->priv->enums;
		for (en_it = __temp3; en_it != NULL; en_it = en_it->next) {
			ValaEnum* en;
			en = en_it->data;
			{
				/* process enums and flags first to avoid order problems in C code */
				vala_code_node_accept (VALA_CODE_NODE (en), visitor);
			}
		}
	}
	{
		GList* __temp4;
		GList* fl_it;
		__temp4 = NULL;
		__temp4 = self->priv->flags_;
		for (fl_it = __temp4; fl_it != NULL; fl_it = fl_it->next) {
			ValaFlags* fl;
			fl = fl_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (fl), visitor);
			}
		}
	}
	{
		GList* __temp5;
		GList* cl_it;
		__temp5 = NULL;
		__temp5 = self->priv->classes;
		for (cl_it = __temp5; cl_it != NULL; cl_it = cl_it->next) {
			ValaClass* cl;
			cl = cl_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (cl), visitor);
			}
		}
	}
	{
		GList* __temp6;
		GList* iface_it;
		__temp6 = NULL;
		__temp6 = self->priv->interfaces;
		for (iface_it = __temp6; iface_it != NULL; iface_it = iface_it->next) {
			ValaInterface* iface;
			iface = iface_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (iface), visitor);
			}
		}
	}
	{
		GList* __temp7;
		GList* st_it;
		__temp7 = NULL;
		__temp7 = self->priv->structs;
		for (st_it = __temp7; st_it != NULL; st_it = st_it->next) {
			ValaStruct* st;
			st = st_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (st), visitor);
			}
		}
	}
	{
		GList* __temp8;
		GList* cb_it;
		__temp8 = NULL;
		__temp8 = self->priv->callbacks;
		for (cb_it = __temp8; cb_it != NULL; cb_it = cb_it->next) {
			ValaCallback* cb;
			cb = cb_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (cb), visitor);
			}
		}
	}
	{
		GList* __temp9;
		GList* c_it;
		__temp9 = NULL;
		__temp9 = self->priv->constants;
		for (c_it = __temp9; c_it != NULL; c_it = c_it->next) {
			ValaConstant* c;
			c = c_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (c), visitor);
			}
		}
	}
	{
		GList* __temp10;
		GList* f_it;
		__temp10 = NULL;
		__temp10 = self->priv->fields;
		for (f_it = __temp10; f_it != NULL; f_it = f_it->next) {
			ValaField* f;
			f = f_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (f), visitor);
			}
		}
	}
	{
		GList* __temp11;
		GList* m_it;
		__temp11 = NULL;
		__temp11 = self->priv->methods;
		for (m_it = __temp11; m_it != NULL; m_it = m_it->next) {
			ValaMethod* m;
			m = m_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (m), visitor);
			}
		}
	}
}


/**
 * Converts a string from CamelCase to lower_case.
 *
 * @param camel_case a string in camel case
 * @return           the specified string converted to lower case
 */
char* vala_namespace_camel_case_to_lower_case (const char* camel_case)
{
	GString* result;
	const char* i;
	gboolean first;
	const char* __temp12;
	char* __temp13;
	g_return_val_if_fail (camel_case != NULL, NULL);
	result = g_string_new ("");
	i = camel_case;
	first = TRUE;
	while (g_utf8_strlen (i, -1) > 0) {
		gunichar c;
		c = g_utf8_get_char (i);
		if (g_unichar_isupper (c) && !first) {
			const char* t;
			gboolean prev_upper;
			gboolean next_upper;
			/* current character is upper case and
			 * we're not at the beginning */
			t = g_utf8_prev_char (i);
			prev_upper = g_unichar_isupper (g_utf8_get_char (t));
			t = g_utf8_next_char (i);
			next_upper = g_unichar_isupper (g_utf8_get_char (t));
			if (!prev_upper || (g_utf8_strlen (i, -1) >= 2 && !next_upper)) {
				gint len;
				/* previous character wasn't upper case or
				 * next character isn't upper case*/
				len = g_utf8_strlen (result->str, -1);
				if (len != 1 && g_utf8_get_char (g_utf8_offset_to_pointer (result->str, len - 2)) != '_') {
					/* we're not creating 1 character words */
					g_string_append_c (result, '_');
				}
			}
		}
		g_string_append_unichar (result, g_unichar_tolower (c));
		first = FALSE;
		i = g_utf8_next_char (i);
	}
	__temp12 = NULL;
	__temp13 = NULL;
	return (__temp13 = (__temp12 = result->str, (__temp12 == NULL ? NULL : g_strdup (__temp12))), (result == NULL ? NULL : (result = (g_string_free (result, TRUE), NULL))), __temp13);
	(result == NULL ? NULL : (result = (g_string_free (result, TRUE), NULL)));
}


/**
 * Returns the camel case string to be prepended to the name of members
 * of this namespace when used in C code.
 *
 * @return the camel case prefix to be used in C code
 */
char* vala_namespace_get_cprefix (ValaNamespace* self)
{
	const char* __temp17;
	g_return_val_if_fail (VALA_IS_NAMESPACE (self), NULL);
	if (self->priv->cprefix == NULL) {
		if (vala_namespace_get_name (self) == NULL) {
			char* __temp14;
			__temp14 = NULL;
			self->priv->cprefix = (__temp14 = g_strdup (""), (self->priv->cprefix = (g_free (self->priv->cprefix), NULL)), __temp14);
		} else {
			char* __temp16;
			const char* __temp15;
			__temp16 = NULL;
			__temp15 = NULL;
			self->priv->cprefix = (__temp16 = (__temp15 = vala_namespace_get_name (self), (__temp15 == NULL ? NULL : g_strdup (__temp15))), (self->priv->cprefix = (g_free (self->priv->cprefix), NULL)), __temp16);
		}
	}
	__temp17 = NULL;
	return (__temp17 = self->priv->cprefix, (__temp17 == NULL ? NULL : g_strdup (__temp17)));
}


/**
 * Sets the camel case string to be prepended to the name of members of
 * this namespace when used in C code.
 *
 * @param cprefix the camel case prefix to be used in C code
 */
void vala_namespace_set_cprefix (ValaNamespace* self, const char* cprefix)
{
	char* __temp20;
	const char* __temp19;
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	__temp20 = NULL;
	__temp19 = NULL;
	self->priv->cprefix = (__temp20 = (__temp19 = cprefix, (__temp19 == NULL ? NULL : g_strdup (__temp19))), (self->priv->cprefix = (g_free (self->priv->cprefix), NULL)), __temp20);
}


/**
 * Returns the lower case string to be prepended to the name of members
 * of this namespace when used in C code.
 *
 * @return the lower case prefix to be used in C code
 */
char* vala_namespace_get_lower_case_cprefix (ValaNamespace* self)
{
	const char* __temp24;
	g_return_val_if_fail (VALA_IS_NAMESPACE (self), NULL);
	if (self->priv->lower_case_cprefix == NULL) {
		if (vala_namespace_get_name (self) == NULL) {
			char* __temp21;
			__temp21 = NULL;
			self->priv->lower_case_cprefix = (__temp21 = g_strdup (""), (self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL)), __temp21);
		} else {
			char* __temp23;
			char* __temp22;
			__temp23 = NULL;
			__temp22 = NULL;
			self->priv->lower_case_cprefix = (__temp23 = g_strdup_printf ("%s_", (__temp22 = vala_namespace_camel_case_to_lower_case (vala_namespace_get_name (self)))), (self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL)), __temp23);
			(__temp22 = (g_free (__temp22), NULL));
		}
	}
	__temp24 = NULL;
	return (__temp24 = self->priv->lower_case_cprefix, (__temp24 == NULL ? NULL : g_strdup (__temp24)));
}


/**
 * Sets the lower case string to be prepended to the name of members of
 * this namespace when used in C code.
 *
 * @param cprefix the lower case prefix to be used in C code
 */
void vala_namespace_set_lower_case_cprefix (ValaNamespace* self, const char* cprefix)
{
	char* __temp27;
	const char* __temp26;
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	__temp27 = NULL;
	__temp26 = NULL;
	self->priv->lower_case_cprefix = (__temp27 = (__temp26 = cprefix, (__temp26 == NULL ? NULL : g_strdup (__temp26))), (self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL)), __temp27);
}


/**
 * Returns a list of C header filenames users of this namespace must
 * include.
 *
 * @return list of C header filenames for this namespace
 */
GList* vala_namespace_get_cheader_filenames (ValaNamespace* self)
{
	g_return_val_if_fail (VALA_IS_NAMESPACE (self), NULL);
	if (self->priv->cheader_filenames == NULL) {
		if (!vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (self))))) {
			/* don't add default include directives for VAPI files*/
			self->priv->cheader_filenames = g_list_append (self->priv->cheader_filenames, vala_source_file_get_cinclude_filename (vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (self)))));
		}
	}
	return g_list_copy (self->priv->cheader_filenames);
}


/**
 * Returns the C header filename of this namespace.
 *
 * @return header filename
 */
char* vala_namespace_get_cheader_filename (ValaNamespace* self)
{
	GString* s;
	gboolean first;
	const char* __temp30;
	char* __temp31;
	g_return_val_if_fail (VALA_IS_NAMESPACE (self), NULL);
	s = g_string_new ("");
	first = TRUE;
	{
		GList* __temp29;
		GList* cheader_filename_it;
		__temp29 = NULL;
		__temp29 = vala_namespace_get_cheader_filenames (self);
		for (cheader_filename_it = __temp29; cheader_filename_it != NULL; cheader_filename_it = cheader_filename_it->next) {
			char* cheader_filename;
			cheader_filename = cheader_filename_it->data;
			{
				if (first) {
					first = FALSE;
				} else {
					g_string_append_c (s, ',');
				}
				g_string_append (s, cheader_filename);
			}
		}
		(__temp29 == NULL ? NULL : (__temp29 = (g_list_free (__temp29), NULL)));
	}
	__temp30 = NULL;
	__temp31 = NULL;
	return (__temp31 = (__temp30 = s->str, (__temp30 == NULL ? NULL : g_strdup (__temp30))), (s == NULL ? NULL : (s = (g_string_free (s, TRUE), NULL))), __temp31);
	(s == NULL ? NULL : (s = (g_string_free (s, TRUE), NULL)));
}


/**
 * Sets the C header filename of this namespace to the specified
 * filename.
 *
 * @param cheader_filename header filename
 */
void vala_namespace_set_cheader_filename (ValaNamespace* self, const char* cheader_filename)
{
	GList* __temp32;
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (cheader_filename != NULL);
	__temp32 = NULL;
	self->priv->cheader_filenames = (__temp32 = NULL, (self->priv->cheader_filenames == NULL ? NULL : (self->priv->cheader_filenames = (g_list_foreach (self->priv->cheader_filenames, (GFunc) g_free, NULL), g_list_free (self->priv->cheader_filenames), NULL))), __temp32);
	self->priv->cheader_filenames = g_list_append (self->priv->cheader_filenames, g_strdup (cheader_filename));
}


static void vala_namespace_process_ccode_attribute (ValaNamespace* self, ValaAttribute* a)
{
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	g_return_if_fail (VALA_IS_ATTRIBUTE (a));
	if (vala_attribute_has_argument (a, "cprefix")) {
		char* __temp33;
		__temp33 = NULL;
		vala_namespace_set_cprefix (self, (__temp33 = vala_attribute_get_string (a, "cprefix")));
		(__temp33 = (g_free (__temp33), NULL));
	}
	if (vala_attribute_has_argument (a, "lower_case_cprefix")) {
		char* __temp34;
		__temp34 = NULL;
		vala_namespace_set_lower_case_cprefix (self, (__temp34 = vala_attribute_get_string (a, "lower_case_cprefix")));
		(__temp34 = (g_free (__temp34), NULL));
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** __temp36;
			char** filename_it;
			__temp36 = NULL;
			__temp36 = g_strsplit (val, ",", 0);
			for (filename_it = __temp36; *filename_it != NULL; filename_it = filename_it + 1) {
				char* filename;
				filename = *filename_it;
				{
					const char* __temp35;
					__temp35 = NULL;
					self->priv->cheader_filenames = g_list_append (self->priv->cheader_filenames, (__temp35 = filename, (__temp35 == NULL ? NULL : g_strdup (__temp35))));
				}
			}
			(__temp36 = (g_free (__temp36), NULL));
		}
		(val = (g_free (val), NULL));
	}
}


/**
 * Process all associated attributes.
 */
void vala_namespace_process_attributes (ValaNamespace* self)
{
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	{
		GList* __temp37;
		GList* a_it;
		__temp37 = NULL;
		__temp37 = VALA_CODE_NODE (self)->attributes;
		for (a_it = __temp37; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* a;
			a = a_it->data;
			{
				if (g_utf8_collate (vala_attribute_get_name (a), "CCode") == 0) {
					vala_namespace_process_ccode_attribute (self, a);
				}
			}
		}
	}
}


char* vala_namespace_get_name (ValaNamespace* self)
{
	g_return_val_if_fail (VALA_IS_NAMESPACE (self), NULL);
	return self->priv->_name;
}


void vala_namespace_set_name (ValaNamespace* self, const char* value)
{
	char* __temp40;
	const char* __temp39;
	g_return_if_fail (VALA_IS_NAMESPACE (self));
	__temp40 = NULL;
	__temp39 = NULL;
	self->priv->_name = (__temp40 = (__temp39 = value, (__temp39 == NULL ? NULL : g_strdup (__temp39))), (self->priv->_name = (g_free (self->priv->_name), NULL)), __temp40);
}


static void vala_namespace_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaNamespace * self;
	self = VALA_NAMESPACE (object);
	switch (property_id) {
		case VALA_NAMESPACE_NAME:
		g_value_set_string (value, vala_namespace_get_name (self));
		break;
	}
}


static void vala_namespace_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaNamespace * self;
	self = VALA_NAMESPACE (object);
	switch (property_id) {
		case VALA_NAMESPACE_NAME:
		vala_namespace_set_name (self, g_value_get_string (value));
		break;
	}
}


static void vala_namespace_class_init (ValaNamespaceClass * klass)
{
	vala_namespace_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaNamespacePrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_namespace_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_namespace_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_namespace_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_namespace_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_namespace_real_accept_children;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_NAMESPACE_NAME, g_param_spec_string ("name", "foo", "bar", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_namespace_init (ValaNamespace * self)
{
	self->priv = VALA_NAMESPACE_GET_PRIVATE (self);
}


static void vala_namespace_dispose (GObject * obj)
{
	ValaNamespace * self;
	ValaNamespaceClass * klass;
	GObjectClass * parent_class;
	self = VALA_NAMESPACE (obj);
	(self->priv->_name = (g_free (self->priv->_name), NULL));
	(self->priv->classes == NULL ? NULL : (self->priv->classes = (g_list_foreach (self->priv->classes, (GFunc) g_object_unref, NULL), g_list_free (self->priv->classes), NULL)));
	(self->priv->interfaces == NULL ? NULL : (self->priv->interfaces = (g_list_foreach (self->priv->interfaces, (GFunc) g_object_unref, NULL), g_list_free (self->priv->interfaces), NULL)));
	(self->priv->structs == NULL ? NULL : (self->priv->structs = (g_list_foreach (self->priv->structs, (GFunc) g_object_unref, NULL), g_list_free (self->priv->structs), NULL)));
	(self->priv->enums == NULL ? NULL : (self->priv->enums = (g_list_foreach (self->priv->enums, (GFunc) g_object_unref, NULL), g_list_free (self->priv->enums), NULL)));
	(self->priv->flags_ == NULL ? NULL : (self->priv->flags_ = (g_list_foreach (self->priv->flags_, (GFunc) g_object_unref, NULL), g_list_free (self->priv->flags_), NULL)));
	(self->priv->callbacks == NULL ? NULL : (self->priv->callbacks = (g_list_foreach (self->priv->callbacks, (GFunc) g_object_unref, NULL), g_list_free (self->priv->callbacks), NULL)));
	(self->priv->constants == NULL ? NULL : (self->priv->constants = (g_list_foreach (self->priv->constants, (GFunc) g_object_unref, NULL), g_list_free (self->priv->constants), NULL)));
	(self->priv->fields == NULL ? NULL : (self->priv->fields = (g_list_foreach (self->priv->fields, (GFunc) g_object_unref, NULL), g_list_free (self->priv->fields), NULL)));
	(self->priv->methods == NULL ? NULL : (self->priv->methods = (g_list_foreach (self->priv->methods, (GFunc) g_object_unref, NULL), g_list_free (self->priv->methods), NULL)));
	(self->priv->cprefix = (g_free (self->priv->cprefix), NULL));
	(self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL));
	(self->priv->cheader_filenames == NULL ? NULL : (self->priv->cheader_filenames = (g_list_foreach (self->priv->cheader_filenames, (GFunc) g_free, NULL), g_list_free (self->priv->cheader_filenames), NULL)));
	klass = VALA_NAMESPACE_CLASS (g_type_class_peek (VALA_TYPE_NAMESPACE));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_namespace_get_type ()
{
	static GType vala_namespace_type_id = 0;
	if (G_UNLIKELY (vala_namespace_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaNamespaceClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_namespace_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaNamespace), 0, (GInstanceInitFunc) vala_namespace_init };
		vala_namespace_type_id = g_type_register_static (VALA_TYPE_CODE_NODE, "ValaNamespace", &g_define_type_info, 0);
	}
	return vala_namespace_type_id;
}




