/* valaccodewriter.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */
#define VALA_FREE_CHECKED(o,f) ((o) == NULL ? NULL : ((o) = (f (o), NULL)))
#define VALA_FREE_UNCHECKED(o,f) ((o) = (f (o), NULL))

#include "valaccodewriter.h"
#include <stdio.h>
#include <glib/gstdio.h>
#include <ccode/valaccodewriter.h>

struct _ValaCCodeWriterPrivate {
	char* _filename;
	char* temp_filename;
	gboolean file_exists;
	FILE* stream;
	gint indent;
	gboolean _bol;
};
#define VALA_CCODE_WRITER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CCODE_WRITER, ValaCCodeWriterPrivate))
enum  {
	VALA_CCODE_WRITER_DUMMY_PROPERTY,
	VALA_CCODE_WRITER_FILENAME,
	VALA_CCODE_WRITER_BOL
};
static gpointer vala_ccode_writer_parent_class = NULL;
static void vala_ccode_writer_dispose (GObject * obj);


ValaCCodeWriter* vala_ccode_writer_new (const char* _filename)
{
	GParameter * __params;
	GParameter * __params_it;
	ValaCCodeWriter * self;
	g_return_val_if_fail (_filename != NULL, NULL);
	__params = g_new0 (GParameter, 1);
	__params_it = __params;
	(__params_it->name = "filename", g_value_init (&__params_it->value, G_TYPE_STRING), g_value_set_string (&__params_it->value, _filename), __params_it++);
	self = g_object_newv (VALA_TYPE_CCODE_WRITER, __params_it - __params, __params);
	return self;
}


/**
 * Closes the file.
 */
void vala_ccode_writer_close (ValaCCodeWriter* self)
{
	GError * inner_error;
	FILE* __temp0;
	g_return_if_fail (VALA_IS_CCODE_WRITER (self));
	inner_error = NULL;
	__temp0 = NULL;
	self->priv->stream = (__temp0 = NULL, (self->priv->stream == NULL ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL))), __temp0);
	if (self->priv->file_exists) {
		gboolean changed;
		changed = TRUE;
		{
			GMappedFile* old_file;
			GMappedFile* new_file;
			glong len;
			GMappedFile* __temp1;
			GMappedFile* __temp2;
			old_file = g_mapped_file_new (self->priv->_filename, FALSE, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == G_FILE_ERROR) {
					goto __catch0_g_file_error;
				}
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
			}
			new_file = g_mapped_file_new (self->priv->temp_filename, FALSE, &inner_error);
			if (inner_error != NULL) {
				if (inner_error->domain == G_FILE_ERROR) {
					goto __catch0_g_file_error;
				}
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
			}
			len = g_mapped_file_get_length (old_file);
			if (len == g_mapped_file_get_length (new_file)) {
				if (memcmp (g_mapped_file_get_contents (old_file), g_mapped_file_get_contents (new_file), len) == 0) {
					changed = FALSE;
				}
			}
			__temp1 = NULL;
			old_file = (__temp1 = NULL, (old_file == NULL ? NULL : (old_file = (g_mapped_file_free (old_file), NULL))), __temp1);
			__temp2 = NULL;
			new_file = (__temp2 = NULL, (new_file == NULL ? NULL : (new_file = (g_mapped_file_free (new_file), NULL))), __temp2);
			(old_file == NULL ? NULL : (old_file = (g_mapped_file_free (old_file), NULL)));
			(new_file == NULL ? NULL : (new_file = (g_mapped_file_free (new_file), NULL)));
		}
		goto __finally0;
		__catch0_g_file_error:
		{
			GError * e;
			e = inner_error;
			inner_error = NULL;
			{
			}
		}
		__finally0:
		;
		/* assume changed if mmap comparison doesn't work*/
		if (changed) {
			g_rename (self->priv->temp_filename, self->priv->_filename);
		} else {
			g_unlink (self->priv->temp_filename);
		}
	}
}


/**
 * Writes tabs according to the current indent level.
 */
void vala_ccode_writer_write_indent (ValaCCodeWriter* self)
{
	gint i;
	g_return_if_fail (VALA_IS_CCODE_WRITER (self));
	i = 0;
	if (!vala_ccode_writer_get_bol (self)) {
		fputc ('\n', self->priv->stream);
	}
	for (i = 0; i < self->priv->indent; i++) {
		fputc ('\t', self->priv->stream);
	}
	self->priv->_bol = FALSE;
}


/**
 * Writes the specified string.
 *
 * @param s a string
 */
void vala_ccode_writer_write_string (ValaCCodeWriter* self, const char* s)
{
	g_return_if_fail (VALA_IS_CCODE_WRITER (self));
	g_return_if_fail (s != NULL);
	fprintf (self->priv->stream, "%s", s);
	self->priv->_bol = FALSE;
}


/**
 * Writes a newline.
 */
void vala_ccode_writer_write_newline (ValaCCodeWriter* self)
{
	g_return_if_fail (VALA_IS_CCODE_WRITER (self));
	fputc ('\n', self->priv->stream);
	self->priv->_bol = TRUE;
}


/**
 * Opens a new block, increasing the indent level.
 */
void vala_ccode_writer_write_begin_block (ValaCCodeWriter* self)
{
	g_return_if_fail (VALA_IS_CCODE_WRITER (self));
	if (!vala_ccode_writer_get_bol (self)) {
		fputc (' ', self->priv->stream);
	} else {
		vala_ccode_writer_write_indent (self);
	}
	fputc ('{', self->priv->stream);
	vala_ccode_writer_write_newline (self);
	self->priv->indent++;
}


/**
 * Closes the current block, decreasing the indent level.
 */
void vala_ccode_writer_write_end_block (ValaCCodeWriter* self)
{
	g_return_if_fail (VALA_IS_CCODE_WRITER (self));
	g_assert (self->priv->indent > 0);
	self->priv->indent--;
	vala_ccode_writer_write_indent (self);
	fprintf (self->priv->stream, "}");
}


/**
 * Writes the specified text as comment.
 *
 * @param text the comment text
 */
void vala_ccode_writer_write_comment (ValaCCodeWriter* self, const char* text)
{
	gboolean first;
	char** __temp3;
	gint lines_length1;
	char** lines;
	g_return_if_fail (VALA_IS_CCODE_WRITER (self));
	g_return_if_fail (text != NULL);
	vala_ccode_writer_write_indent (self);
	fprintf (self->priv->stream, "/*");
	first = TRUE;
	/* separate declaration due to missing memory management in foreach statements */
	__temp3 = NULL;
	lines = (__temp3 = g_strsplit (text, "\n", 0), lines_length1 = -1, __temp3);
	{
		char** __temp4;
		int line_it;
		__temp4 = NULL;
		__temp4 = lines;
		for (line_it = 0; (lines_length1 != -1 && line_it < lines_length1) || (lines_length1 == -1 && __temp4[line_it] != NULL); line_it = line_it + 1) {
			char* line;
			line = __temp4[line_it];
			{
				if (!first) {
					vala_ccode_writer_write_indent (self);
				} else {
					first = FALSE;
				}
				fprintf (self->priv->stream, "%s", line);
			}
		}
	}
	fprintf (self->priv->stream, "*/");
	vala_ccode_writer_write_newline (self);
	(lines = (g_free (lines), NULL));
}


char* vala_ccode_writer_get_filename (ValaCCodeWriter* self)
{
	g_return_val_if_fail (VALA_IS_CCODE_WRITER (self), NULL);
	return self->priv->_filename;
}


void vala_ccode_writer_set_filename (ValaCCodeWriter* self, const char* value)
{
	char* __temp7;
	const char* __temp6;
	g_return_if_fail (VALA_IS_CCODE_WRITER (self));
	__temp7 = NULL;
	__temp6 = NULL;
	self->priv->_filename = (__temp7 = (__temp6 = value, (__temp6 == NULL ? NULL : g_strdup (__temp6))), (self->priv->_filename = (g_free (self->priv->_filename), NULL)), __temp7);
	self->priv->file_exists = g_file_test (self->priv->_filename, G_FILE_TEST_EXISTS);
	if (self->priv->file_exists) {
		char* __temp8;
		FILE* __temp9;
		__temp8 = NULL;
		self->priv->temp_filename = (__temp8 = g_strdup_printf ("%s.valatmp", self->priv->_filename), (self->priv->temp_filename = (g_free (self->priv->temp_filename), NULL)), __temp8);
		__temp9 = NULL;
		self->priv->stream = (__temp9 = fopen (self->priv->temp_filename, "w"), (self->priv->stream == NULL ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL))), __temp9);
	} else {
		FILE* __temp10;
		__temp10 = NULL;
		self->priv->stream = (__temp10 = fopen (self->priv->_filename, "w"), (self->priv->stream == NULL ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL))), __temp10);
	}
}


gboolean vala_ccode_writer_get_bol (ValaCCodeWriter* self)
{
	g_return_val_if_fail (VALA_IS_CCODE_WRITER (self), FALSE);
	return self->priv->_bol;
}


static void vala_ccode_writer_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaCCodeWriter * self;
	self = VALA_CCODE_WRITER (object);
	switch (property_id) {
		case VALA_CCODE_WRITER_FILENAME:
		g_value_set_string (value, vala_ccode_writer_get_filename (self));
		break;
		case VALA_CCODE_WRITER_BOL:
		g_value_set_boolean (value, vala_ccode_writer_get_bol (self));
		break;
	}
}


static void vala_ccode_writer_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaCCodeWriter * self;
	self = VALA_CCODE_WRITER (object);
	switch (property_id) {
		case VALA_CCODE_WRITER_FILENAME:
		vala_ccode_writer_set_filename (self, g_value_get_string (value));
		break;
	}
}


static void vala_ccode_writer_class_init (ValaCCodeWriterClass * klass)
{
	vala_ccode_writer_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaCCodeWriterPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_ccode_writer_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_ccode_writer_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_ccode_writer_dispose;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CCODE_WRITER_FILENAME, g_param_spec_string ("filename", "foo", "bar", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CCODE_WRITER_BOL, g_param_spec_boolean ("bol", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE));
}


static void vala_ccode_writer_init (ValaCCodeWriter * self)
{
	self->priv = VALA_CCODE_WRITER_GET_PRIVATE (self);
	self->priv->_bol = TRUE;
}


static void vala_ccode_writer_dispose (GObject * obj)
{
	ValaCCodeWriter * self;
	ValaCCodeWriterClass * klass;
	GObjectClass * parent_class;
	self = VALA_CCODE_WRITER (obj);
	(self->priv->_filename = (g_free (self->priv->_filename), NULL));
	(self->priv->temp_filename = (g_free (self->priv->temp_filename), NULL));
	(self->priv->stream == NULL ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL)));
	klass = VALA_CCODE_WRITER_CLASS (g_type_class_peek (VALA_TYPE_CCODE_WRITER));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_ccode_writer_get_type ()
{
	static GType vala_ccode_writer_type_id = 0;
	if (G_UNLIKELY (vala_ccode_writer_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCCodeWriterClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_ccode_writer_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCCodeWriter), 0, (GInstanceInitFunc) vala_ccode_writer_init };
		vala_ccode_writer_type_id = g_type_register_static (G_TYPE_OBJECT, "ValaCCodeWriter", &g_define_type_info, 0);
	}
	return vala_ccode_writer_type_id;
}




