/* valatype.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <rasa@gmx.ch>
 */
#define VALA_FREE_CHECKED(o,f) ((o) == NULL ? NULL : ((o) = (f (o), NULL)))
#define VALA_FREE_UNCHECKED(o,f) ((o) = (f (o), NULL))

#include "valadatatype.h"
#include <vala/valapointer.h>
#include <vala/valadatatype.h>
#include <vala/valanamespace.h>
#include <vala/valasourcereference.h>
#include <vala/valasymbol.h>
#include <vala/valaarray.h>
#include <vala/valafield.h>
#include <vala/valamethod.h>

struct _ValaDataTypePrivate {
	char* _name;
	GList* cheader_filenames;
	ValaPointer* pointer_type;
	GHashTable* array_types;
};
#define VALA_DATA_TYPE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_DATA_TYPE, ValaDataTypePrivate))
enum  {
	VALA_DATA_TYPE_DUMMY_PROPERTY,
	VALA_DATA_TYPE_NAME
};
static gboolean vala_data_type_real_is_reference_type (ValaDataType* self);
static char* vala_data_type_real_get_dup_function (ValaDataType* self);
static char* vala_data_type_real_get_free_function (ValaDataType* self);
static gboolean vala_data_type_real_is_reference_counting (ValaDataType* self);
static char* vala_data_type_real_get_ref_function (ValaDataType* self);
static char* vala_data_type_real_get_unref_function (ValaDataType* self);
static char* vala_data_type_real_get_type_id (ValaDataType* self);
static char* vala_data_type_real_get_marshaller_type_name (ValaDataType* self);
static char* vala_data_type_real_get_get_value_function (ValaDataType* self);
static char* vala_data_type_real_get_set_value_function (ValaDataType* self);
static char* vala_data_type_real_get_upper_case_cname (ValaDataType* self, const char* infix);
static char* vala_data_type_real_get_lower_case_cname (ValaDataType* self, const char* infix);
static char* vala_data_type_real_get_lower_case_cprefix (ValaDataType* self);
static char* vala_data_type_real_get_default_value (ValaDataType* self);
static GList* vala_data_type_real_get_cheader_filenames (ValaDataType* self);
static gboolean vala_data_type_real_is_subtype_of (ValaDataType* self, ValaDataType* t);
static gint vala_data_type_real_get_type_parameter_index (ValaDataType* self, const char* name);
static gpointer vala_data_type_parent_class = NULL;
static void vala_data_type_dispose (GObject * obj);


char* vala_data_type_get_cname (ValaDataType* self, gboolean const_type)
{
	return VALA_DATA_TYPE_GET_CLASS (self)->get_cname (self, const_type);
}


/**
 * Checks whether this data type has value or reference type semantics.
 *
 * @return true if this data type has reference type semantics
 */
static gboolean vala_data_type_real_is_reference_type (ValaDataType* self)
{
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), FALSE);
	return FALSE;
}


gboolean vala_data_type_is_reference_type (ValaDataType* self)
{
	return VALA_DATA_TYPE_GET_CLASS (self)->is_reference_type (self);
}


/**
 * Returns the C function name that duplicates instances of this data
 * type. The specified C function must accept one argument referencing
 * the instance of this data type and return a reference to the
 * duplicate.
 *
 * @return the name of the C function if supported or null otherwise
 */
static char* vala_data_type_real_get_dup_function (ValaDataType* self)
{
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), NULL);
	return NULL;
}


char* vala_data_type_get_dup_function (ValaDataType* self)
{
	return VALA_DATA_TYPE_GET_CLASS (self)->get_dup_function (self);
}


/**
 * Returns the C function name that frees instances of this data type.
 * This is only valid for data types with reference type semantics that
 * do not support reference counting. The specified C function must
 * accept one argument pointing to the instance to be freed.
 *
 * @return the name of the C function or null if this data type is not a
 *         reference type or if it supports reference counting
 */
static char* vala_data_type_real_get_free_function (ValaDataType* self)
{
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), NULL);
	return NULL;
}


char* vala_data_type_get_free_function (ValaDataType* self)
{
	return VALA_DATA_TYPE_GET_CLASS (self)->get_free_function (self);
}


/**
 * Checks whether this data type supports reference counting. This is
 * only valid for reference types.
 *
 * @return true if this data type supports reference counting
 */
static gboolean vala_data_type_real_is_reference_counting (ValaDataType* self)
{
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), FALSE);
	return FALSE;
}


gboolean vala_data_type_is_reference_counting (ValaDataType* self)
{
	return VALA_DATA_TYPE_GET_CLASS (self)->is_reference_counting (self);
}


/**
 * Returns the C function name that increments the reference count of
 * instances of this data type. This is only valid for data types
 * supporting reference counting. The specified C function must accept
 * one argument referencing the instance of this data type and return
 * the reference.
 *
 * @return the name of the C function or null if this data type does not
 *         support reference counting
 */
static char* vala_data_type_real_get_ref_function (ValaDataType* self)
{
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), NULL);
	return NULL;
}


char* vala_data_type_get_ref_function (ValaDataType* self)
{
	return VALA_DATA_TYPE_GET_CLASS (self)->get_ref_function (self);
}


/**
 * Returns the C function name that decrements the reference count of
 * instances of this data type. This is only valid for data types
 * supporting reference counting. The specified C function must accept
 * one argument referencing the instance of this data type.
 *
 * @return the name of the C function or null if this data type does not
 *         support reference counting
 */
static char* vala_data_type_real_get_unref_function (ValaDataType* self)
{
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), NULL);
	return NULL;
}


char* vala_data_type_get_unref_function (ValaDataType* self)
{
	return VALA_DATA_TYPE_GET_CLASS (self)->get_unref_function (self);
}


/**
 * Returns the C symbol representing the runtime type id for this data
 * type. The specified symbol must express a registered GType.
 *
 * @return the name of the GType name in C code or null if this data
 *         type is not registered with GType
 */
static char* vala_data_type_real_get_type_id (ValaDataType* self)
{
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), NULL);
	return NULL;
}


char* vala_data_type_get_type_id (ValaDataType* self)
{
	return VALA_DATA_TYPE_GET_CLASS (self)->get_type_id (self);
}


/**
 * Returns the name of this data type as used in C code marshallers
 *
 * @return type name for marshallers
 */
static char* vala_data_type_real_get_marshaller_type_name (ValaDataType* self)
{
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), NULL);
	return NULL;
}


char* vala_data_type_get_marshaller_type_name (ValaDataType* self)
{
	return VALA_DATA_TYPE_GET_CLASS (self)->get_marshaller_type_name (self);
}


/**
 * Returns the cname of the GValue getter function,
 */
static char* vala_data_type_real_get_get_value_function (ValaDataType* self)
{
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), NULL);
	return NULL;
}


char* vala_data_type_get_get_value_function (ValaDataType* self)
{
	return VALA_DATA_TYPE_GET_CLASS (self)->get_get_value_function (self);
}


/**
 * Returns the cname of the GValue setter function,
 */
static char* vala_data_type_real_get_set_value_function (ValaDataType* self)
{
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), NULL);
	return NULL;
}


char* vala_data_type_get_set_value_function (ValaDataType* self)
{
	return VALA_DATA_TYPE_GET_CLASS (self)->get_set_value_function (self);
}


/**
 * Returns the C name of this data type in upper case. Words are
 * separated by underscores. The upper case C name of the namespace is
 * prefix of the result.
 *
 * @param infix a string to be placed between namespace and data type
 *              name or null
 * @return      the upper case name to be used in C code
 */
static char* vala_data_type_real_get_upper_case_cname (ValaDataType* self, const char* infix)
{
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), NULL);
	return NULL;
}


char* vala_data_type_get_upper_case_cname (ValaDataType* self, const char* infix)
{
	return VALA_DATA_TYPE_GET_CLASS (self)->get_upper_case_cname (self, infix);
}


/**
 * Returns the C name of this data type in lower case. Words are
 * separated by underscores. The lower case C name of the namespace is
 * prefix of the result.
 *
 * @param infix a string to be placed between namespace and data type
 *              name or null
 * @return      the lower case name to be used in C code
 */
static char* vala_data_type_real_get_lower_case_cname (ValaDataType* self, const char* infix)
{
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), NULL);
	return NULL;
}


char* vala_data_type_get_lower_case_cname (ValaDataType* self, const char* infix)
{
	return VALA_DATA_TYPE_GET_CLASS (self)->get_lower_case_cname (self, infix);
}


/**
 * Returns the string to be prefixed to members of this data type in
 * lower case when used in C code.
 *
 * @return      the lower case prefix to be used in C code
 */
static char* vala_data_type_real_get_lower_case_cprefix (ValaDataType* self)
{
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), NULL);
	return NULL;
}


char* vala_data_type_get_lower_case_cprefix (ValaDataType* self)
{
	return VALA_DATA_TYPE_GET_CLASS (self)->get_lower_case_cprefix (self);
}


/**
 * Returns the default value for the given type. Returning null means
 * there is no default value (i.e. not that the default name is NULL).
 *
 * @returnthe name of the default value
 */
static char* vala_data_type_real_get_default_value (ValaDataType* self)
{
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), NULL);
	return NULL;
}


char* vala_data_type_get_default_value (ValaDataType* self)
{
	return VALA_DATA_TYPE_GET_CLASS (self)->get_default_value (self);
}


/**
 * Returns a list of C header filenames users of this data type must
 * include.
 *
 * @return list of C header filenames for this data type
 */
static GList* vala_data_type_real_get_cheader_filenames (ValaDataType* self)
{
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), NULL);
	if (self->priv->cheader_filenames == NULL) {
		{
			GList* __temp14;
			GList* filename_it;
			__temp14 = NULL;
			__temp14 = vala_namespace_get_cheader_filenames (self->namespace);
			for (filename_it = __temp14; filename_it != NULL; filename_it = filename_it->next) {
				char* filename;
				filename = filename_it->data;
				{
					/* default to header filenames of the namespace */
					vala_data_type_add_cheader_filename (self, filename);
				}
			}
			(__temp14 == NULL ? NULL : (__temp14 = (g_list_free (__temp14), NULL)));
		}
	}
	return g_list_copy (self->priv->cheader_filenames);
}


GList* vala_data_type_get_cheader_filenames (ValaDataType* self)
{
	return VALA_DATA_TYPE_GET_CLASS (self)->get_cheader_filenames (self);
}


/**
 * Adds a filename to the list of C header filenames users of this data
 * type must include.
 *
 * @param filename a C header filename
 */
void vala_data_type_add_cheader_filename (ValaDataType* self, const char* filename)
{
	g_return_if_fail (VALA_IS_DATA_TYPE (self));
	g_return_if_fail (filename != NULL);
	self->priv->cheader_filenames = g_list_append (self->priv->cheader_filenames, g_strdup (filename));
}


/**
 * Returns the pointer type of this data type.
 *
 * @return pointer-type for this data type
 */
ValaPointer* vala_data_type_get_pointer (ValaDataType* self)
{
	ValaPointer* __temp18;
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), NULL);
	if (self->priv->pointer_type == NULL) {
		ValaPointer* __temp16;
		ValaSymbol* __temp17;
		__temp16 = NULL;
		self->priv->pointer_type = (__temp16 = vala_pointer_new (self, vala_code_node_get_source_reference (VALA_CODE_NODE (self))), (self->priv->pointer_type == NULL ? NULL : (self->priv->pointer_type = (g_object_unref (self->priv->pointer_type), NULL))), __temp16);
		/* create a new Symbol */
		__temp17 = NULL;
		(vala_code_node_set_symbol (VALA_CODE_NODE (self->priv->pointer_type), (__temp17 = vala_symbol_new (VALA_CODE_NODE (self->priv->pointer_type)))), vala_code_node_get_symbol (VALA_CODE_NODE (self->priv->pointer_type)));
		(__temp17 == NULL ? NULL : (__temp17 = (g_object_unref (__temp17), NULL)));
		vala_symbol_add (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (self))), vala_data_type_get_name (VALA_DATA_TYPE (self->priv->pointer_type)), vala_code_node_get_symbol (VALA_CODE_NODE (self->priv->pointer_type)));
		/* link the namespace */
		VALA_DATA_TYPE (self->priv->pointer_type)->namespace = self->namespace;
	}
	__temp18 = NULL;
	return (__temp18 = self->priv->pointer_type, (__temp18 == NULL ? NULL : g_object_ref (__temp18)));
}


/**
 * Returns the array type for elements of this data type.
 *
 * @param rank the rank the array should be of
 * @return array type for this data type
 */
ValaArray* vala_data_type_get_array (ValaDataType* self, gint rank)
{
	ValaArray* __temp21;
	char* __temp20;
	ValaArray* __temp22;
	ValaArray* array_type;
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), NULL);
	__temp21 = NULL;
	__temp20 = NULL;
	__temp22 = NULL;
	array_type = (__temp22 = (__temp21 = VALA_ARRAY (g_hash_table_lookup (self->priv->array_types, (__temp20 = g_strdup_printf ("%i", rank)))), (__temp21 == NULL ? NULL : g_object_ref (__temp21))), (__temp20 = (g_free (__temp20), NULL)), __temp22);
	if (array_type == NULL) {
		ValaArray* new_array_type;
		ValaSymbol* __temp23;
		ValaArrayLengthField* __temp25;
		ValaArrayLengthField* __temp24;
		ValaArrayResizeMethod* __temp27;
		ValaArrayResizeMethod* __temp26;
		ValaArray* __temp28;
		ValaArray* __temp30;
		ValaArray* __temp29;
		new_array_type = vala_array_new (self, rank, vala_code_node_get_source_reference (VALA_CODE_NODE (self)));
		/* create a new Symbol */
		__temp23 = NULL;
		(vala_code_node_set_symbol (VALA_CODE_NODE (new_array_type), (__temp23 = vala_symbol_new (VALA_CODE_NODE (new_array_type)))), vala_code_node_get_symbol (VALA_CODE_NODE (new_array_type)));
		(__temp23 == NULL ? NULL : (__temp23 = (g_object_unref (__temp23), NULL)));
		vala_symbol_add (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (self))), vala_data_type_get_name (VALA_DATA_TYPE (new_array_type)), vala_code_node_get_symbol (VALA_CODE_NODE (new_array_type)));
		/* add internal length field */
		__temp25 = NULL;
		__temp24 = NULL;
		vala_symbol_add (vala_code_node_get_symbol (VALA_CODE_NODE (new_array_type)), vala_field_get_name (VALA_FIELD ((__temp24 = vala_array_get_length_field (new_array_type)))), vala_code_node_get_symbol (VALA_CODE_NODE ((__temp25 = vala_array_get_length_field (new_array_type)))));
		(__temp25 == NULL ? NULL : (__temp25 = (g_object_unref (__temp25), NULL)));
		(__temp24 == NULL ? NULL : (__temp24 = (g_object_unref (__temp24), NULL)));
		/* add internal resize method */
		__temp27 = NULL;
		__temp26 = NULL;
		vala_symbol_add (vala_code_node_get_symbol (VALA_CODE_NODE (new_array_type)), vala_method_get_name (VALA_METHOD ((__temp26 = vala_array_get_resize_method (new_array_type)))), vala_code_node_get_symbol (VALA_CODE_NODE ((__temp27 = vala_array_get_resize_method (new_array_type)))));
		(__temp27 == NULL ? NULL : (__temp27 = (g_object_unref (__temp27), NULL)));
		(__temp26 == NULL ? NULL : (__temp26 = (g_object_unref (__temp26), NULL)));
		/* link the array type to the same source as the container type */
		(vala_code_node_set_source_reference (VALA_CODE_NODE (new_array_type), vala_code_node_get_source_reference (VALA_CODE_NODE (self))), vala_code_node_get_source_reference (VALA_CODE_NODE (new_array_type)));
		/* link the namespace */
		VALA_DATA_TYPE (new_array_type)->namespace = self->namespace;
		__temp28 = NULL;
		g_hash_table_insert (self->priv->array_types, g_strdup_printf ("%i", rank), (__temp28 = new_array_type, (__temp28 == NULL ? NULL : g_object_ref (__temp28))));
		__temp30 = NULL;
		__temp29 = NULL;
		array_type = (__temp30 = (__temp29 = new_array_type, (__temp29 == NULL ? NULL : g_object_ref (__temp29))), (array_type == NULL ? NULL : (array_type = (g_object_unref (array_type), NULL))), __temp30);
		(new_array_type == NULL ? NULL : (new_array_type = (g_object_unref (new_array_type), NULL)));
	}
	return array_type;
	(array_type == NULL ? NULL : (array_type = (g_object_unref (array_type), NULL)));
}


/**
 * Checks whether this data type is a subtype of the specified data
 * type.
 *
 * @param t a data type
 * @return  true if t is a supertype of this data type, false otherwise
 */
static gboolean vala_data_type_real_is_subtype_of (ValaDataType* self, ValaDataType* t)
{
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), FALSE);
	g_return_val_if_fail (VALA_IS_DATA_TYPE (t), FALSE);
	return FALSE;
}


gboolean vala_data_type_is_subtype_of (ValaDataType* self, ValaDataType* t)
{
	return VALA_DATA_TYPE_GET_CLASS (self)->is_subtype_of (self, t);
}


/**
 * Return the index of the specified type parameter name.
 */
static gint vala_data_type_real_get_type_parameter_index (ValaDataType* self, const char* name)
{
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), 0);
	g_return_val_if_fail (name != NULL, 0);
	return -1;
}


gint vala_data_type_get_type_parameter_index (ValaDataType* self, const char* name)
{
	return VALA_DATA_TYPE_GET_CLASS (self)->get_type_parameter_index (self, name);
}


char* vala_data_type_get_name (ValaDataType* self)
{
	g_return_val_if_fail (VALA_IS_DATA_TYPE (self), NULL);
	return self->priv->_name;
}


void vala_data_type_set_name (ValaDataType* self, const char* value)
{
	char* __temp36;
	const char* __temp35;
	g_return_if_fail (VALA_IS_DATA_TYPE (self));
	__temp36 = NULL;
	__temp35 = NULL;
	self->priv->_name = (__temp36 = (__temp35 = value, (__temp35 == NULL ? NULL : g_strdup (__temp35))), (self->priv->_name = (g_free (self->priv->_name), NULL)), __temp36);
}


static void vala_data_type_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaDataType * self;
	self = VALA_DATA_TYPE (object);
	switch (property_id) {
		case VALA_DATA_TYPE_NAME:
		g_value_set_string (value, vala_data_type_get_name (self));
		break;
	}
}


static void vala_data_type_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaDataType * self;
	self = VALA_DATA_TYPE (object);
	switch (property_id) {
		case VALA_DATA_TYPE_NAME:
		vala_data_type_set_name (self, g_value_get_string (value));
		break;
	}
}


static void vala_data_type_class_init (ValaDataTypeClass * klass)
{
	vala_data_type_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaDataTypePrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_data_type_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_data_type_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_data_type_dispose;
	VALA_DATA_TYPE_CLASS (klass)->is_reference_type = vala_data_type_real_is_reference_type;
	VALA_DATA_TYPE_CLASS (klass)->get_dup_function = vala_data_type_real_get_dup_function;
	VALA_DATA_TYPE_CLASS (klass)->get_free_function = vala_data_type_real_get_free_function;
	VALA_DATA_TYPE_CLASS (klass)->is_reference_counting = vala_data_type_real_is_reference_counting;
	VALA_DATA_TYPE_CLASS (klass)->get_ref_function = vala_data_type_real_get_ref_function;
	VALA_DATA_TYPE_CLASS (klass)->get_unref_function = vala_data_type_real_get_unref_function;
	VALA_DATA_TYPE_CLASS (klass)->get_type_id = vala_data_type_real_get_type_id;
	VALA_DATA_TYPE_CLASS (klass)->get_marshaller_type_name = vala_data_type_real_get_marshaller_type_name;
	VALA_DATA_TYPE_CLASS (klass)->get_get_value_function = vala_data_type_real_get_get_value_function;
	VALA_DATA_TYPE_CLASS (klass)->get_set_value_function = vala_data_type_real_get_set_value_function;
	VALA_DATA_TYPE_CLASS (klass)->get_upper_case_cname = vala_data_type_real_get_upper_case_cname;
	VALA_DATA_TYPE_CLASS (klass)->get_lower_case_cname = vala_data_type_real_get_lower_case_cname;
	VALA_DATA_TYPE_CLASS (klass)->get_lower_case_cprefix = vala_data_type_real_get_lower_case_cprefix;
	VALA_DATA_TYPE_CLASS (klass)->get_default_value = vala_data_type_real_get_default_value;
	VALA_DATA_TYPE_CLASS (klass)->get_cheader_filenames = vala_data_type_real_get_cheader_filenames;
	VALA_DATA_TYPE_CLASS (klass)->is_subtype_of = vala_data_type_real_is_subtype_of;
	VALA_DATA_TYPE_CLASS (klass)->get_type_parameter_index = vala_data_type_real_get_type_parameter_index;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_DATA_TYPE_NAME, g_param_spec_string ("name", "foo", "bar", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_data_type_init (ValaDataType * self)
{
	self->priv = VALA_DATA_TYPE_GET_PRIVATE (self);
	self->priv->array_types = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_object_unref);
}


static void vala_data_type_dispose (GObject * obj)
{
	ValaDataType * self;
	ValaDataTypeClass * klass;
	GObjectClass * parent_class;
	self = VALA_DATA_TYPE (obj);
	(self->priv->_name = (g_free (self->priv->_name), NULL));
	(self->priv->cheader_filenames == NULL ? NULL : (self->priv->cheader_filenames = (g_list_foreach (self->priv->cheader_filenames, (GFunc) g_free, NULL), g_list_free (self->priv->cheader_filenames), NULL)));
	(self->priv->pointer_type == NULL ? NULL : (self->priv->pointer_type = (g_object_unref (self->priv->pointer_type), NULL)));
	(self->priv->array_types == NULL ? NULL : (self->priv->array_types = (g_hash_table_unref (self->priv->array_types), NULL)));
	klass = VALA_DATA_TYPE_CLASS (g_type_class_peek (VALA_TYPE_DATA_TYPE));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_data_type_get_type ()
{
	static GType vala_data_type_type_id = 0;
	if (G_UNLIKELY (vala_data_type_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaDataTypeClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_data_type_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaDataType), 0, (GInstanceInitFunc) vala_data_type_init };
		vala_data_type_type_id = g_type_register_static (VALA_TYPE_CODE_NODE, "ValaDataType", &g_define_type_info, G_TYPE_FLAG_ABSTRACT);
	}
	return vala_data_type_type_id;
}




