/* valacharacterliteral.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <raffaele@sandrini.ch>
 */

#include <vala/valacharacterliteral.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>
#include <vala/valareport.h>

struct _ValaCharacterLiteralPrivate {
	char* _value;
};
#define VALA_CHARACTER_LITERAL_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_CHARACTER_LITERAL, ValaCharacterLiteralPrivate))
enum  {
	VALA_CHARACTER_LITERAL_DUMMY_PROPERTY,
	VALA_CHARACTER_LITERAL_VALUE
};
static void vala_character_literal_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static GObject * vala_character_literal_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
static gpointer vala_character_literal_parent_class = NULL;
static void vala_character_literal_dispose (GObject * obj);


/**
 * Creates a new character literal.
 *
 * @param c      character
 * @param source reference to source code
 * @return       newly created character literal
 */
ValaCharacterLiteral* vala_character_literal_new (const char* c, ValaSourceReference* source) {
	GParameter * __params;
	GParameter * __params_it;
	ValaCharacterLiteral * self;
	g_return_val_if_fail (source == NULL || VALA_IS_SOURCE_REFERENCE (source), NULL);
	__params = g_new0 (GParameter, 1);
	__params_it = __params;
	__params_it->name = "value";
	g_value_init (&__params_it->value, G_TYPE_STRING);
	g_value_set_string (&__params_it->value, c);
	__params_it++;
	self = g_object_newv (VALA_TYPE_CHARACTER_LITERAL, __params_it - __params, __params);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


static void vala_character_literal_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaCharacterLiteral * self;
	self = VALA_CHARACTER_LITERAL (base);
	g_return_if_fail (visitor == NULL || VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_character_literal (visitor, self);
}


/**
 * Returns the unicode character value this character literal
 * represents.
 *
 * @return unicode character value
 */
gunichar vala_character_literal_get_char (ValaCharacterLiteral* self) {
	g_return_val_if_fail (VALA_IS_CHARACTER_LITERAL (self), 0U);
	return g_utf8_get_char (g_utf8_next_char (vala_character_literal_get_value (self)));
}


char* vala_character_literal_get_value (ValaCharacterLiteral* self) {
	g_return_val_if_fail (VALA_IS_CHARACTER_LITERAL (self), NULL);
	return self->priv->_value;
}


void vala_character_literal_set_value (ValaCharacterLiteral* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (VALA_IS_CHARACTER_LITERAL (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_value = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (self->priv->_value = (g_free (self->priv->_value), NULL)), _tmp2);
	if (!g_utf8_validate (value, -1, NULL) || (g_utf8_strlen (value, -1) != 3 && g_utf8_get_char (g_utf8_next_char (value)) != '\\')) {
		vala_code_node_set_error (VALA_CODE_NODE (self), TRUE);
	}
}


static GObject * vala_character_literal_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	ValaCharacterLiteralClass * klass;
	GObjectClass * parent_class;
	ValaCharacterLiteral * self;
	klass = VALA_CHARACTER_LITERAL_CLASS (g_type_class_peek (VALA_TYPE_CHARACTER_LITERAL));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = VALA_CHARACTER_LITERAL (obj);
	{
		if (vala_code_node_get_error (VALA_CODE_NODE (self))) {
			vala_report_error (vala_code_node_get_source_reference (VALA_CODE_NODE (self)), "invalid character literal");
		}
	}
	return obj;
}


static void vala_character_literal_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaCharacterLiteral * self;
	self = VALA_CHARACTER_LITERAL (object);
	switch (property_id) {
		case VALA_CHARACTER_LITERAL_VALUE:
		g_value_set_string (value, vala_character_literal_get_value (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_character_literal_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaCharacterLiteral * self;
	self = VALA_CHARACTER_LITERAL (object);
	switch (property_id) {
		case VALA_CHARACTER_LITERAL_VALUE:
		vala_character_literal_set_value (self, g_value_get_string (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_character_literal_class_init (ValaCharacterLiteralClass * klass) {
	vala_character_literal_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaCharacterLiteralPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_character_literal_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_character_literal_set_property;
	G_OBJECT_CLASS (klass)->constructor = vala_character_literal_constructor;
	G_OBJECT_CLASS (klass)->dispose = vala_character_literal_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_character_literal_real_accept;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_CHARACTER_LITERAL_VALUE, g_param_spec_string ("value", "value", "value", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
}


static void vala_character_literal_init (ValaCharacterLiteral * self) {
	self->priv = VALA_CHARACTER_LITERAL_GET_PRIVATE (self);
}


static void vala_character_literal_dispose (GObject * obj) {
	ValaCharacterLiteral * self;
	self = VALA_CHARACTER_LITERAL (obj);
	self->priv->_value = (g_free (self->priv->_value), NULL);
	G_OBJECT_CLASS (vala_character_literal_parent_class)->dispose (obj);
}


GType vala_character_literal_get_type (void) {
	static GType vala_character_literal_type_id = 0;
	if (G_UNLIKELY (vala_character_literal_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCharacterLiteralClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_character_literal_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCharacterLiteral), 0, (GInstanceInitFunc) vala_character_literal_init };
		vala_character_literal_type_id = g_type_register_static (VALA_TYPE_LITERAL, "ValaCharacterLiteral", &g_define_type_info, 0);
	}
	return vala_character_literal_type_id;
}




