/* valablock.vala
 *
 * Copyright (C) 2006  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */
#define VALA_FREE_CHECKED(o,f) ((o) == NULL ? NULL : ((o) = (f (o), NULL)))
#define VALA_FREE_UNCHECKED(o,f) ((o) = (f (o), NULL))

#include "valablock.h"
#include <vala/valasourcereference.h>
#include <vala/valablock.h>
#include <vala/valavariabledeclarator.h>
#include <vala/valacodevisitor.h>

struct _ValaBlockPrivate {
	gboolean _contains_jump_statement;
	GList* statement_list;
	GList* local_variables;
};
#define VALA_BLOCK_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_BLOCK, ValaBlockPrivate))
enum  {
	VALA_BLOCK_DUMMY_PROPERTY,
	VALA_BLOCK_CONTAINS_JUMP_STATEMENT,
};
static void vala_block_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static gpointer vala_block_parent_class = NULL;
static void vala_block_dispose (GObject * obj);


/**
 * Creates a new block.
 *
 * @param source reference to source code
 */
ValaBlock* vala_block_new (ValaSourceReference* source)
{
	g_return_val_if_fail (source == NULL || VALA_IS_SOURCE_REFERENCE (source), NULL);
	GParameter * __params = g_new0 (GParameter, 1);
	GParameter * __params_it = __params;
	(__params_it->name = "source-reference", g_value_init (&__params_it->value, VALA_TYPE_SOURCE_REFERENCE), g_value_set_object (&__params_it->value, source), __params_it++);
	ValaBlock * self = g_object_newv (VALA_TYPE_BLOCK, __params_it - __params, __params);
	return self;
}


/**
 * Append a statement to this block.
 *
 * @param stmt a statement
 */
void vala_block_add_statement (ValaBlock* self, ValaStatement* stmt)
{
	g_return_if_fail (VALA_IS_BLOCK (self));
	g_return_if_fail (VALA_IS_STATEMENT (stmt));
	self->priv->statement_list = g_list_append (self->priv->statement_list, g_object_ref (stmt));
}


/**
 * Returns a copy of the list of statements.
 *
 * @return statement list
 */
GList* vala_block_get_statements (ValaBlock* self)
{
	g_return_val_if_fail (VALA_IS_BLOCK (self), NULL);
	return g_list_copy (self->priv->statement_list);
}


/**
 * Add a local variable to this block.
 *
 * @param decl a variable declarator
 */
void vala_block_add_local_variable (ValaBlock* self, ValaVariableDeclarator* decl)
{
	g_return_if_fail (VALA_IS_BLOCK (self));
	g_return_if_fail (VALA_IS_VARIABLE_DECLARATOR (decl));
	self->priv->local_variables = g_list_append (self->priv->local_variables, g_object_ref (decl));
}


/**
 * Returns a copy of the list of local variables.
 *
 * @return variable declarator list
 */
GList* vala_block_get_local_variables (ValaBlock* self)
{
	g_return_val_if_fail (VALA_IS_BLOCK (self), NULL);
	return g_list_copy (self->priv->local_variables);
}


static void vala_block_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaBlock * self = VALA_BLOCK (base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_begin_block (visitor, self);
	{
		GList* __temp2 = NULL;
		__temp2 = self->priv->statement_list;
		GList* stmt_it;
		for (stmt_it = __temp2; stmt_it != NULL; stmt_it = stmt_it->next) {
			ValaStatement* stmt = stmt_it->data;
			{
				vala_code_node_accept (VALA_CODE_NODE (stmt), visitor);
			}
		}
	}
	vala_code_visitor_visit_end_block (visitor, self);
}


gboolean vala_block_get_contains_jump_statement (ValaBlock* self)
{
	g_return_val_if_fail (VALA_IS_BLOCK (self), FALSE);
	return self->priv->_contains_jump_statement;
}


void vala_block_set_contains_jump_statement (ValaBlock* self, gboolean value)
{
	g_return_if_fail (VALA_IS_BLOCK (self));
	self->priv->_contains_jump_statement = value;
}


static void vala_block_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaBlock * self = VALA_BLOCK (object);
	switch (property_id) {
		case VALA_BLOCK_CONTAINS_JUMP_STATEMENT:
		g_value_set_boolean (value, vala_block_get_contains_jump_statement (self));
		break;
	}
}


static void vala_block_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaBlock * self = VALA_BLOCK (object);
	switch (property_id) {
		case VALA_BLOCK_CONTAINS_JUMP_STATEMENT:
		vala_block_set_contains_jump_statement (self, g_value_get_boolean (value));
		break;
	}
}


static void vala_block_class_init (ValaBlockClass * klass)
{
	vala_block_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaBlockPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_block_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_block_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_block_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_block_real_accept;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_BLOCK_CONTAINS_JUMP_STATEMENT, g_param_spec_boolean ("contains-jump-statement", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_block_init (ValaBlock * self)
{
	self->priv = VALA_BLOCK_GET_PRIVATE (self);
}


static void vala_block_dispose (GObject * obj)
{
	ValaBlock * self = VALA_BLOCK (obj);
	(self->priv->statement_list == NULL ? NULL : (self->priv->statement_list = (g_list_foreach (self->priv->statement_list, (GFunc) g_object_unref, NULL), g_list_free (self->priv->statement_list), NULL)));
	(self->priv->local_variables == NULL ? NULL : (self->priv->local_variables = (g_list_foreach (self->priv->local_variables, (GFunc) g_object_unref, NULL), g_list_free (self->priv->local_variables), NULL)));
	ValaBlockClass * klass;
	GObjectClass * parent_class;
	klass = VALA_BLOCK_CLASS (g_type_class_peek (VALA_TYPE_BLOCK));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_block_get_type ()
{
	static GType vala_block_type_id = 0;
	if (G_UNLIKELY (vala_block_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaBlockClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_block_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaBlock), 0, (GInstanceInitFunc) vala_block_init };
		vala_block_type_id = g_type_register_static (VALA_TYPE_STATEMENT, "ValaBlock", &g_define_type_info, 0);
	}
	return vala_block_type_id;
}




