/* valacompiler.vala
 *
 * Copyright (C) 2006  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */
#define VALA_FREE_CHECKED(o,f) ((o) == NULL ? NULL : ((o) = (f (o), NULL)))
#define VALA_FREE_UNCHECKED(o,f) ((o) = (f (o), NULL))

#include "valacompiler.h"
#include <vala/valacodecontext.h>
#include <vala/valareport.h>
#include <stdio.h>
#include <glib/gstdio.h>
#include <vala/valasourcefile.h>
#include <vala/valasourcereference.h>
#include <vala/valaparser.h>
#include <vala/valasymbolbuilder.h>
#include <vala/valaattributeprocessor.h>
#include <vala/valasymbolresolver.h>
#include <vala/valasemanticanalyzer.h>
#include <vala/valamemorymanager.h>
#include <vala/valacodegenerator.h>
#include <vala/valainterfacewriter.h>
#include "valacompiler.h"
#include "config.h"

struct _ValaCompilerPrivate {
	ValaCodeContext* context;
	GList* added_packages;
};
#define VALA_COMPILER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_COMPILER, ValaCompilerPrivate))
enum  {
	VALA_COMPILER_DUMMY_PROPERTY,
};
static char* vala_compiler_directory;
static gboolean vala_compiler_version;
static char** vala_compiler_sources;
static char** vala_compiler_vapi_directories;
static char* vala_compiler_library;
static char** vala_compiler_packages;
static gboolean vala_compiler_disable_memory_management;
static const GOptionEntry VALA_COMPILER_options[] = {{"vapidir", 0, 0, G_OPTION_ARG_FILENAME_ARRAY, &vala_compiler_vapi_directories, "Look for package bindings in DIRECTORY", "DIRECTORY..."}, {"pkg", 0, 0, G_OPTION_ARG_STRING_ARRAY, &vala_compiler_packages, "Include binding for PACKAGE", "PACKAGE..."}, {"library", 0, 0, G_OPTION_ARG_STRING, &vala_compiler_library, "Library name", "NAME"}, {"directory", 'd', 0, G_OPTION_ARG_FILENAME, &vala_compiler_directory, "Output directory", "DIRECTORY"}, {"version", 0, 0, G_OPTION_ARG_NONE, &vala_compiler_version, "Display version number", NULL}, {"disable-memory-management", 0, 0, G_OPTION_ARG_NONE, &vala_compiler_disable_memory_management, "Disable memory management", NULL}, {"", 0, 0, G_OPTION_ARG_FILENAME_ARRAY, &vala_compiler_sources, NULL, "FILE..."}, {NULL}};
static gint vala_compiler_quit (ValaCompiler* self);
static char* vala_compiler_get_package_path (ValaCompiler* self, const char* pkg);
static gboolean vala_compiler_add_package (ValaCompiler* self, const char* pkg);
static gint vala_compiler_run (ValaCompiler* self);
static gint vala_compiler_main (int args_length1, char** args);
static gpointer vala_compiler_parent_class = NULL;
static void vala_compiler_dispose (GObject * obj);


static gint vala_compiler_quit (ValaCompiler* self)
{
	g_return_val_if_fail (VALA_IS_COMPILER (self), 0);
	if (vala_report_get_errors () == 0) {
		fprintf (stdout, "Compilation succeeded - %d warning(s)\n", vala_report_get_warnings ());
		return 0;
	} else {
		fprintf (stdout, "Compilation failed: %d error(s), %d warning(s)\n", vala_report_get_errors (), vala_report_get_warnings ());
		return 1;
	}
}


static char* vala_compiler_get_package_path (ValaCompiler* self, const char* pkg)
{
	g_return_val_if_fail (VALA_IS_COMPILER (self), NULL);
	g_return_val_if_fail (pkg != NULL, NULL);
	char* basename = g_strdup_printf ("%s.vala", pkg);
	if (vala_compiler_vapi_directories != NULL) {
		{
			char** __temp3 = NULL;
			__temp3 = vala_compiler_vapi_directories;
			char** vapidir_it;
			for (vapidir_it = __temp3; *vapidir_it != NULL; vapidir_it = vapidir_it + 1) {
				char* vapidir = *vapidir_it;
				{
					char* filename = g_build_filename (vapidir, basename, NULL);
					if (g_file_test (filename, G_FILE_TEST_EXISTS)) {
						char* __temp2 = NULL;
						return (__temp2 = filename, (basename = (g_free (basename), NULL)), __temp2);
					}
					(filename = (g_free (filename), NULL));
				}
			}
		}
	}
	char* filename = g_build_filename (PACKAGE_DATADIR, "vapi", basename, NULL);
	if (g_file_test (filename, G_FILE_TEST_EXISTS)) {
		char* __temp4 = NULL;
		return (__temp4 = filename, (basename = (g_free (basename), NULL)), __temp4);
	}
	char* __temp5 = NULL;
	filename = (__temp5 = g_build_filename ("/usr/local/share/vala/vapi", basename, NULL), (filename = (g_free (filename), NULL)), __temp5);
	if (g_file_test (filename, G_FILE_TEST_EXISTS)) {
		char* __temp6 = NULL;
		return (__temp6 = filename, (basename = (g_free (basename), NULL)), __temp6);
	}
	char* __temp7 = NULL;
	filename = (__temp7 = g_build_filename ("/usr/share/vala/vapi", basename, NULL), (filename = (g_free (filename), NULL)), __temp7);
	if (g_file_test (filename, G_FILE_TEST_EXISTS)) {
		char* __temp8 = NULL;
		return (__temp8 = filename, (basename = (g_free (basename), NULL)), __temp8);
	}
	gpointer __temp9;
	return (__temp9 = NULL, (basename = (g_free (basename), NULL)), (filename = (g_free (filename), NULL)), __temp9);
	(basename = (g_free (basename), NULL));
	(filename = (g_free (filename), NULL));
}


static gboolean vala_compiler_add_package (ValaCompiler* self, const char* pkg)
{
	g_return_val_if_fail (VALA_IS_COMPILER (self), FALSE);
	g_return_val_if_fail (pkg != NULL, FALSE);
	if (g_list_find_custom (self->priv->added_packages, pkg, ((GCompareFunc) strcmp)) != NULL) {
		return TRUE;
	}
	/* ignore multiple occurences of the same package*/
	char* package_path = vala_compiler_get_package_path (self, pkg);
	if (package_path == NULL) {
		gboolean __temp11;
		return (__temp11 = FALSE, (package_path = (g_free (package_path), NULL)), __temp11);
	}
	self->priv->added_packages = g_list_append (self->priv->added_packages, g_strdup (pkg));
	ValaSourceFile* __temp12 = NULL;
	vala_code_context_add_source_file (self->priv->context, (__temp12 = vala_source_file_new (self->priv->context, package_path, TRUE)));
	(__temp12 == NULL ? NULL : (__temp12 = (g_object_unref (__temp12), NULL)));
	char* __temp14 = NULL;
	char* __temp13 = NULL;
	char* __temp15 = NULL;
	char* deps_filename = (__temp15 = g_build_filename ((__temp13 = g_path_get_dirname (package_path)), (__temp14 = g_strdup_printf ("%s.deps", pkg)), NULL), (__temp14 = (g_free (__temp14), NULL)), (__temp13 = (g_free (__temp13), NULL)), __temp15);
	if (g_file_test (deps_filename, G_FILE_TEST_EXISTS)) {
		char* deps_content = NULL;
		g_file_get_contents (deps_filename, &deps_content, NULL, NULL);
		{
			char** __temp17 = NULL;
			__temp17 = g_strsplit (deps_content, "\n", 0);
			char** dep_it;
			for (dep_it = __temp17; *dep_it != NULL; dep_it = dep_it + 1) {
				char* dep = *dep_it;
				{
					if (g_utf8_collate (dep, "") != 0) {
						if (!vala_compiler_add_package (self, dep)) {
							char* __temp16 = NULL;
							vala_report_error (NULL, (__temp16 = g_strdup_printf ("%s, dependency of %s, not found in specified Vala API directories", dep, pkg)));
							(__temp16 = (g_free (__temp16), NULL));
						}
					}
				}
			}
			(__temp17 = (g_free (__temp17), NULL));
		}
		(deps_content = (g_free (deps_content), NULL));
	}
	gboolean __temp18;
	return (__temp18 = TRUE, (package_path = (g_free (package_path), NULL)), (deps_filename = (g_free (deps_filename), NULL)), __temp18);
	(package_path = (g_free (package_path), NULL));
	(deps_filename = (g_free (deps_filename), NULL));
}


static gint vala_compiler_run (ValaCompiler* self)
{
	g_return_val_if_fail (VALA_IS_COMPILER (self), 0);
	ValaCodeContext* __temp19 = NULL;
	self->priv->context = (__temp19 = g_object_new (VALA_TYPE_CODE_CONTEXT, NULL), (self->priv->context == NULL ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL))), __temp19);
	(vala_code_context_set_library (self->priv->context, vala_compiler_library), vala_code_context_get_library (self->priv->context));
	/* default package */
	if (!vala_compiler_add_package (self, "glib-2.0")) {
		vala_report_error (NULL, "glib-2.0 not found in specified Vala API directories");
	}
	if (vala_compiler_packages != NULL) {
		{
			char** __temp21 = NULL;
			__temp21 = vala_compiler_packages;
			char** package_it;
			for (package_it = __temp21; *package_it != NULL; package_it = package_it + 1) {
				char* package = *package_it;
				{
					if (!vala_compiler_add_package (self, package)) {
						char* __temp20 = NULL;
						vala_report_error (NULL, (__temp20 = g_strdup_printf ("%s not found in specified Vala API directories", package)));
						(__temp20 = (g_free (__temp20), NULL));
					}
				}
			}
		}
		char** __temp22 = NULL;
		vala_compiler_packages = (__temp22 = NULL, (vala_compiler_packages = (g_free (vala_compiler_packages), NULL)), __temp22);
	}
	if (vala_report_get_errors () > 0) {
		return vala_compiler_quit (self);
	}
	{
		char** __temp26 = NULL;
		__temp26 = vala_compiler_sources;
		char** source_it;
		for (source_it = __temp26; *source_it != NULL; source_it = source_it + 1) {
			char* source = *source_it;
			{
				if (g_file_test (source, G_FILE_TEST_EXISTS)) {
					ValaSourceFile* __temp24 = NULL;
					vala_code_context_add_source_file (self->priv->context, (__temp24 = vala_source_file_new (self->priv->context, source, FALSE)));
					(__temp24 == NULL ? NULL : (__temp24 = (g_object_unref (__temp24), NULL)));
				} else {
					char* __temp25 = NULL;
					vala_report_error (NULL, (__temp25 = g_strdup_printf ("%s not found", source)));
					(__temp25 = (g_free (__temp25), NULL));
				}
			}
		}
	}
	char** __temp27 = NULL;
	vala_compiler_sources = (__temp27 = NULL, (vala_compiler_sources = (g_free (vala_compiler_sources), NULL)), __temp27);
	if (vala_report_get_errors () > 0) {
		return vala_compiler_quit (self);
	}
	ValaParser* parser = g_object_new (VALA_TYPE_PARSER, NULL);
	vala_parser_parse (parser, self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint __temp29;
		return (__temp29 = vala_compiler_quit (self), (parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL))), __temp29);
	}
	ValaSymbolBuilder* builder = g_object_new (VALA_TYPE_SYMBOL_BUILDER, NULL);
	vala_symbol_builder_build (builder, self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint __temp30;
		return (__temp30 = vala_compiler_quit (self), (parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL))), (builder == NULL ? NULL : (builder = (g_object_unref (builder), NULL))), __temp30);
	}
	ValaAttributeProcessor* attributeprocessor = g_object_new (VALA_TYPE_ATTRIBUTE_PROCESSOR, NULL);
	vala_attribute_processor_process (attributeprocessor, self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint __temp31;
		return (__temp31 = vala_compiler_quit (self), (parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL))), (builder == NULL ? NULL : (builder = (g_object_unref (builder), NULL))), (attributeprocessor == NULL ? NULL : (attributeprocessor = (g_object_unref (attributeprocessor), NULL))), __temp31);
	}
	ValaSymbolResolver* resolver = g_object_new (VALA_TYPE_SYMBOL_RESOLVER, NULL);
	vala_symbol_resolver_resolve (resolver, self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint __temp32;
		return (__temp32 = vala_compiler_quit (self), (parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL))), (builder == NULL ? NULL : (builder = (g_object_unref (builder), NULL))), (attributeprocessor == NULL ? NULL : (attributeprocessor = (g_object_unref (attributeprocessor), NULL))), (resolver == NULL ? NULL : (resolver = (g_object_unref (resolver), NULL))), __temp32);
	}
	ValaSemanticAnalyzer* analyzer = vala_semantic_analyzer_new (!vala_compiler_disable_memory_management);
	vala_semantic_analyzer_analyze (analyzer, self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint __temp33;
		return (__temp33 = vala_compiler_quit (self), (parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL))), (builder == NULL ? NULL : (builder = (g_object_unref (builder), NULL))), (attributeprocessor == NULL ? NULL : (attributeprocessor = (g_object_unref (attributeprocessor), NULL))), (resolver == NULL ? NULL : (resolver = (g_object_unref (resolver), NULL))), (analyzer == NULL ? NULL : (analyzer = (g_object_unref (analyzer), NULL))), __temp33);
	}
	if (!vala_compiler_disable_memory_management) {
		ValaMemoryManager* memory_manager = g_object_new (VALA_TYPE_MEMORY_MANAGER, NULL);
		vala_memory_manager_analyze (memory_manager, self->priv->context);
		if (vala_report_get_errors () > 0) {
			gint __temp34;
			return (__temp34 = vala_compiler_quit (self), (memory_manager == NULL ? NULL : (memory_manager = (g_object_unref (memory_manager), NULL))), __temp34);
		}
		(memory_manager == NULL ? NULL : (memory_manager = (g_object_unref (memory_manager), NULL)));
	}
	ValaCodeGenerator* code_generator = vala_code_generator_new (!vala_compiler_disable_memory_management);
	vala_code_generator_emit (code_generator, self->priv->context);
	if (vala_report_get_errors () > 0) {
		gint __temp35;
		return (__temp35 = vala_compiler_quit (self), (parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL))), (builder == NULL ? NULL : (builder = (g_object_unref (builder), NULL))), (attributeprocessor == NULL ? NULL : (attributeprocessor = (g_object_unref (attributeprocessor), NULL))), (resolver == NULL ? NULL : (resolver = (g_object_unref (resolver), NULL))), (analyzer == NULL ? NULL : (analyzer = (g_object_unref (analyzer), NULL))), (code_generator == NULL ? NULL : (code_generator = (g_object_unref (code_generator), NULL))), __temp35);
	}
	if (vala_compiler_library != NULL) {
		ValaInterfaceWriter* interface_writer = g_object_new (VALA_TYPE_INTERFACE_WRITER, NULL);
		char* __temp36 = NULL;
		vala_interface_writer_write_file (interface_writer, self->priv->context, (__temp36 = g_strdup_printf ("%s.vala", vala_compiler_library)));
		(__temp36 = (g_free (__temp36), NULL));
		char* __temp37 = NULL;
		vala_compiler_library = (__temp37 = NULL, (vala_compiler_library = (g_free (vala_compiler_library), NULL)), __temp37);
		(interface_writer == NULL ? NULL : (interface_writer = (g_object_unref (interface_writer), NULL)));
	}
	gint __temp38;
	return (__temp38 = vala_compiler_quit (self), (parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL))), (builder == NULL ? NULL : (builder = (g_object_unref (builder), NULL))), (attributeprocessor == NULL ? NULL : (attributeprocessor = (g_object_unref (attributeprocessor), NULL))), (resolver == NULL ? NULL : (resolver = (g_object_unref (resolver), NULL))), (analyzer == NULL ? NULL : (analyzer = (g_object_unref (analyzer), NULL))), (code_generator == NULL ? NULL : (code_generator = (g_object_unref (code_generator), NULL))), __temp38);
	(parser == NULL ? NULL : (parser = (g_object_unref (parser), NULL)));
	(builder == NULL ? NULL : (builder = (g_object_unref (builder), NULL)));
	(attributeprocessor == NULL ? NULL : (attributeprocessor = (g_object_unref (attributeprocessor), NULL)));
	(resolver == NULL ? NULL : (resolver = (g_object_unref (resolver), NULL)));
	(analyzer == NULL ? NULL : (analyzer = (g_object_unref (analyzer), NULL)));
	(code_generator == NULL ? NULL : (code_generator = (g_object_unref (code_generator), NULL)));
}


static gint vala_compiler_main (int args_length1, char** args)
{
	GError* err = NULL;
	GOptionContext* opt_context = g_option_context_new ("- Vala Compiler");
	g_option_context_set_help_enabled (opt_context, TRUE);
	g_option_context_add_main_entries (opt_context, VALA_COMPILER_options, NULL);
	g_option_context_parse (opt_context, &args_length1, &args, &err);
	if (err != NULL) {
		fprintf (stdout, "%s\n", err->message);
		fprintf (stdout, "Run '%s --help' to see a full list of available command line options.\n", args[0]);
		gint __temp39;
		return (__temp39 = 1, (err == NULL ? NULL : (err = (g_error_free (err), NULL))), (opt_context == NULL ? NULL : (opt_context = (g_option_context_free (opt_context), NULL))), __temp39);
	}
	if (vala_compiler_version) {
		fprintf (stdout, "Vala %s\n", PACKAGE_VERSION);
		gint __temp40;
		return (__temp40 = 0, (err == NULL ? NULL : (err = (g_error_free (err), NULL))), (opt_context == NULL ? NULL : (opt_context = (g_option_context_free (opt_context), NULL))), __temp40);
	}
	if (vala_compiler_sources == NULL) {
		fprintf (stderr, "No source file specified.\n");
		gint __temp41;
		return (__temp41 = 1, (err == NULL ? NULL : (err = (g_error_free (err), NULL))), (opt_context == NULL ? NULL : (opt_context = (g_option_context_free (opt_context), NULL))), __temp41);
	}
	{
		char** __temp43 = NULL;
		__temp43 = vala_compiler_sources;
		char** source_it;
		for (source_it = __temp43; *source_it != NULL; source_it = source_it + 1) {
			char* source = *source_it;
			{
				if (!g_str_has_suffix (source, ".vala")) {
					fprintf (stderr, "Only .vala source files supported.\n");
					gint __temp42;
					return (__temp42 = 1, (err == NULL ? NULL : (err = (g_error_free (err), NULL))), (opt_context == NULL ? NULL : (opt_context = (g_option_context_free (opt_context), NULL))), __temp42);
				}
			}
		}
	}
	ValaCompiler* compiler = g_object_new (VALA_TYPE_COMPILER, NULL);
	gint __temp44;
	return (__temp44 = vala_compiler_run (compiler), (err == NULL ? NULL : (err = (g_error_free (err), NULL))), (opt_context == NULL ? NULL : (opt_context = (g_option_context_free (opt_context), NULL))), (compiler == NULL ? NULL : (compiler = (g_object_unref (compiler), NULL))), __temp44);
	(err == NULL ? NULL : (err = (g_error_free (err), NULL)));
	(opt_context == NULL ? NULL : (opt_context = (g_option_context_free (opt_context), NULL)));
	(compiler == NULL ? NULL : (compiler = (g_object_unref (compiler), NULL)));
}


int main (int argc, char ** argv)
{
	g_type_init ();
	return vala_compiler_main (argc, argv);
}


static void vala_compiler_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaCompiler * self = VALA_COMPILER (object);
	switch (property_id) {
	}
}


static void vala_compiler_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaCompiler * self = VALA_COMPILER (object);
	switch (property_id) {
	}
}


static void vala_compiler_class_init (ValaCompilerClass * klass)
{
	vala_compiler_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaCompilerPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_compiler_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_compiler_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_compiler_dispose;
}


static void vala_compiler_init (ValaCompiler * self)
{
	self->priv = VALA_COMPILER_GET_PRIVATE (self);
}


static void vala_compiler_dispose (GObject * obj)
{
	ValaCompiler * self = VALA_COMPILER (obj);
	(self->priv->context == NULL ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL)));
	(self->priv->added_packages == NULL ? NULL : (self->priv->added_packages = (g_list_foreach (self->priv->added_packages, (GFunc) g_free, NULL), g_list_free (self->priv->added_packages), NULL)));
	ValaCompilerClass * klass;
	GObjectClass * parent_class;
	klass = VALA_COMPILER_CLASS (g_type_class_peek (VALA_TYPE_COMPILER));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_compiler_get_type ()
{
	static GType vala_compiler_type_id = 0;
	if (G_UNLIKELY (vala_compiler_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaCompilerClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_compiler_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaCompiler), 0, (GInstanceInitFunc) vala_compiler_init };
		vala_compiler_type_id = g_type_register_static (G_TYPE_OBJECT, "ValaCompiler", &g_define_type_info, 0);
	}
	return vala_compiler_type_id;
}




