/* valainterfacewriter.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <rasa@gmx.ch>
 */
#define VALA_FREE_CHECKED(o,f) ((o) == NULL ? NULL : ((o) = (f (o), NULL)))
#define VALA_FREE_UNCHECKED(o,f) ((o) = (f (o), NULL))

#include "valainterfacewriter.h"
#include <stdio.h>
#include <glib/gstdio.h>
#include <vala/valainterfacewriter.h>
#include <vala/valasourcefile.h>
#include <vala/valadatatype.h>
#include <vala/valamemberaccessibility.h>
#include <vala/valatypereference.h>
#include <vala/valacodenode.h>
#include <vala/valasymbol.h>
#include <vala/valaarray.h>
#include <vala/valaformalparameter.h>
#include <vala/valapropertyaccessor.h>

struct _ValaInterfaceWriterPrivate {
	ValaCodeContext* context;
	FILE* stream;
	gint indent;
	gboolean bol;
	gboolean internal_scope;
	char* current_cheader_filename;
};
#define VALA_INTERFACE_WRITER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_INTERFACE_WRITER, ValaInterfaceWriterPrivate))
enum  {
	VALA_INTERFACE_WRITER_DUMMY_PROPERTY,
};
static void vala_interface_writer_real_visit_begin_namespace (ValaCodeVisitor* base, ValaNamespace* ns);
static void vala_interface_writer_real_visit_end_namespace (ValaCodeVisitor* base, ValaNamespace* ns);
static void vala_interface_writer_real_visit_begin_class (ValaCodeVisitor* base, ValaClass* cl);
static void vala_interface_writer_real_visit_end_class (ValaCodeVisitor* base, ValaClass* cl);
static void vala_interface_writer_real_visit_begin_struct (ValaCodeVisitor* base, ValaStruct* st);
static void vala_interface_writer_real_visit_end_struct (ValaCodeVisitor* base, ValaStruct* st);
static void vala_interface_writer_real_visit_begin_interface (ValaCodeVisitor* base, ValaInterface* iface);
static void vala_interface_writer_real_visit_end_interface (ValaCodeVisitor* base, ValaInterface* iface);
static void vala_interface_writer_real_visit_begin_enum (ValaCodeVisitor* base, ValaEnum* en);
static void vala_interface_writer_real_visit_end_enum (ValaCodeVisitor* base, ValaEnum* en);
static void vala_interface_writer_real_visit_enum_value (ValaCodeVisitor* base, ValaEnumValue* ev);
static void vala_interface_writer_real_visit_begin_flags (ValaCodeVisitor* base, ValaFlags* fl);
static void vala_interface_writer_real_visit_end_flags (ValaCodeVisitor* base, ValaFlags* fl);
static void vala_interface_writer_real_visit_flags_value (ValaCodeVisitor* base, ValaFlagsValue* fv);
static void vala_interface_writer_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c);
static void vala_interface_writer_real_visit_field (ValaCodeVisitor* base, ValaField* f);
static void vala_interface_writer_write_params (ValaInterfaceWriter* self, GList* params);
static void vala_interface_writer_real_visit_begin_callback (ValaCodeVisitor* base, ValaCallback* cb);
static void vala_interface_writer_real_visit_begin_method (ValaCodeVisitor* base, ValaMethod* m);
static void vala_interface_writer_real_visit_begin_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m);
static void vala_interface_writer_real_visit_begin_property (ValaCodeVisitor* base, ValaProperty* prop);
static void vala_interface_writer_real_visit_begin_signal (ValaCodeVisitor* base, ValaSignal* sig);
static void vala_interface_writer_write_indent (ValaInterfaceWriter* self);
static void vala_interface_writer_write_identifier (ValaInterfaceWriter* self, const char* s);
static void vala_interface_writer_write_string (ValaInterfaceWriter* self, const char* s);
static void vala_interface_writer_write_newline (ValaInterfaceWriter* self);
static void vala_interface_writer_write_begin_block (ValaInterfaceWriter* self);
static void vala_interface_writer_write_end_block (ValaInterfaceWriter* self);
static gpointer vala_interface_writer_parent_class = NULL;
static void vala_interface_writer_dispose (GObject * obj);


/**
 * Writes the public interface of the specified code context into the
 * specified file.
 *
 * @param context  a code context
 * @param filename a relative or absolute filename
 */
void vala_interface_writer_write_file (ValaInterfaceWriter* self, ValaCodeContext* context, const char* filename)
{
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	g_return_if_fail (VALA_IS_CODE_CONTEXT (context));
	g_return_if_fail (filename != NULL);
	ValaCodeContext* __temp0 = NULL;
	self->priv->context = (__temp0 = g_object_ref (context), (self->priv->context == NULL ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL))), __temp0);
	FILE* __temp1 = NULL;
	self->priv->stream = (__temp1 = fopen (filename, "w"), (self->priv->stream == NULL ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL))), __temp1);
	{
		GList* __temp2 = NULL;
		__temp2 = vala_code_context_get_source_files (context);
		GList* file_it;
		for (file_it = __temp2; file_it != NULL; file_it = file_it->next) {
			ValaSourceFile* file = file_it->data;
			{
				/* we're only interested in non-pkg source files */
				if (!vala_source_file_get_pkg (file)) {
					vala_source_file_accept (file, VALA_CODE_VISITOR (self));
				}
			}
		}
		(__temp2 == NULL ? NULL : (__temp2 = (g_list_free (__temp2), NULL)));
	}
	FILE* __temp3 = NULL;
	self->priv->stream = (__temp3 = NULL, (self->priv->stream == NULL ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL))), __temp3);
}


static void vala_interface_writer_real_visit_begin_namespace (ValaCodeVisitor* base, ValaNamespace* ns)
{
	ValaInterfaceWriter * self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_NAMESPACE (ns));
	if (vala_namespace_get_name (ns) == NULL) {
		return;
	}
	char* __temp4 = NULL;
	self->priv->current_cheader_filename = (__temp4 = vala_namespace_get_cheader_filename (ns), (self->priv->current_cheader_filename = (g_free (self->priv->current_cheader_filename), NULL)), __temp4);
	vala_interface_writer_write_indent (self);
	char* __temp5 = NULL;
	vala_interface_writer_write_string (self, (__temp5 = g_strdup_printf ("[CCode (cheader_filename = \"%s\")]", self->priv->current_cheader_filename)));
	(__temp5 = (g_free (__temp5), NULL));
	vala_interface_writer_write_newline (self);
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "namespace ");
	vala_interface_writer_write_identifier (self, vala_namespace_get_name (ns));
	vala_interface_writer_write_begin_block (self);
}


static void vala_interface_writer_real_visit_end_namespace (ValaCodeVisitor* base, ValaNamespace* ns)
{
	ValaInterfaceWriter * self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_NAMESPACE (ns));
	if (vala_namespace_get_name (ns) == NULL) {
		return;
	}
	vala_interface_writer_write_end_block (self);
	vala_interface_writer_write_newline (self);
}


static void vala_interface_writer_real_visit_begin_class (ValaCodeVisitor* base, ValaClass* cl)
{
	ValaInterfaceWriter * self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_CLASS (cl));
	if (VALA_DATA_TYPE (cl)->access == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		self->priv->internal_scope = TRUE;
		return;
	}
	vala_interface_writer_write_indent (self);
	gboolean first = TRUE;
	char* cheaders = NULL;
	{
		GList* __temp9 = NULL;
		__temp9 = vala_data_type_get_cheader_filenames (VALA_DATA_TYPE (cl));
		GList* cheader_it;
		for (cheader_it = __temp9; cheader_it != NULL; cheader_it = cheader_it->next) {
			char* cheader = cheader_it->data;
			{
				if (first) {
					char* __temp7 = NULL;
					const char* __temp6 = NULL;
					cheaders = (__temp7 = (__temp6 = cheader, (__temp6 == NULL ? NULL : g_strdup (__temp6))), (cheaders = (g_free (cheaders), NULL)), __temp7);
					first = FALSE;
				} else {
					char* __temp8 = NULL;
					cheaders = (__temp8 = g_strdup_printf ("%s, %s", cheaders, cheader), (cheaders = (g_free (cheaders), NULL)), __temp8);
				}
			}
		}
		(__temp9 == NULL ? NULL : (__temp9 = (g_list_foreach (__temp9, (GFunc) g_free, NULL), g_list_free (__temp9), NULL)));
	}
	char* __temp10 = NULL;
	vala_interface_writer_write_string (self, (__temp10 = g_strdup_printf ("[CCode (cheader_filename = \"%s\")]", cheaders)));
	(__temp10 = (g_free (__temp10), NULL));
	vala_interface_writer_write_newline (self);
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public ");
	if (vala_class_get_is_abstract (cl)) {
		vala_interface_writer_write_string (self, "abstract ");
	}
	vala_interface_writer_write_string (self, "class ");
	vala_interface_writer_write_identifier (self, vala_data_type_get_name (VALA_DATA_TYPE (cl)));
	GList* base_types = vala_class_get_base_types (cl);
	if (base_types != NULL) {
		vala_interface_writer_write_string (self, " : ");
		gboolean first = TRUE;
		{
			GList* __temp12 = NULL;
			__temp12 = base_types;
			GList* base_type_it;
			for (base_type_it = __temp12; base_type_it != NULL; base_type_it = base_type_it->next) {
				ValaTypeReference* base_type = base_type_it->data;
				{
					if (!first) {
						vala_interface_writer_write_string (self, ", ");
					} else {
						first = FALSE;
					}
					char* __temp11 = NULL;
					vala_interface_writer_write_string (self, (__temp11 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (base_type))))));
					(__temp11 = (g_free (__temp11), NULL));
				}
			}
		}
	}
	vala_interface_writer_write_begin_block (self);
	(cheaders = (g_free (cheaders), NULL));
	(base_types == NULL ? NULL : (base_types = (g_list_free (base_types), NULL)));
}


static void vala_interface_writer_real_visit_end_class (ValaCodeVisitor* base, ValaClass* cl)
{
	ValaInterfaceWriter * self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_CLASS (cl));
	if (VALA_DATA_TYPE (cl)->access == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		self->priv->internal_scope = FALSE;
		return;
	}
	vala_interface_writer_write_end_block (self);
	vala_interface_writer_write_newline (self);
}


static void vala_interface_writer_real_visit_begin_struct (ValaCodeVisitor* base, ValaStruct* st)
{
	ValaInterfaceWriter * self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_STRUCT (st));
	if (VALA_DATA_TYPE (st)->access == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		self->priv->internal_scope = TRUE;
		return;
	}
	if (vala_data_type_is_reference_type (VALA_DATA_TYPE (st))) {
		vala_interface_writer_write_indent (self);
		vala_interface_writer_write_string (self, "[ReferenceType]");
	}
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public struct ");
	vala_interface_writer_write_identifier (self, vala_data_type_get_name (VALA_DATA_TYPE (st)));
	vala_interface_writer_write_begin_block (self);
}


static void vala_interface_writer_real_visit_end_struct (ValaCodeVisitor* base, ValaStruct* st)
{
	ValaInterfaceWriter * self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_STRUCT (st));
	if (VALA_DATA_TYPE (st)->access == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		self->priv->internal_scope = FALSE;
		return;
	}
	vala_interface_writer_write_end_block (self);
	vala_interface_writer_write_newline (self);
}


static void vala_interface_writer_real_visit_begin_interface (ValaCodeVisitor* base, ValaInterface* iface)
{
	ValaInterfaceWriter * self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_INTERFACE (iface));
	if (VALA_DATA_TYPE (iface)->access == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		self->priv->internal_scope = TRUE;
		return;
	}
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public ");
	vala_interface_writer_write_string (self, "interface ");
	vala_interface_writer_write_identifier (self, vala_data_type_get_name (VALA_DATA_TYPE (iface)));
	vala_interface_writer_write_begin_block (self);
}


static void vala_interface_writer_real_visit_end_interface (ValaCodeVisitor* base, ValaInterface* iface)
{
	ValaInterfaceWriter * self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_INTERFACE (iface));
	if (VALA_DATA_TYPE (iface)->access == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		self->priv->internal_scope = FALSE;
		return;
	}
	vala_interface_writer_write_end_block (self);
	vala_interface_writer_write_newline (self);
}


static void vala_interface_writer_real_visit_begin_enum (ValaCodeVisitor* base, ValaEnum* en)
{
	ValaInterfaceWriter * self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_ENUM (en));
	if (VALA_DATA_TYPE (en)->access == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		self->priv->internal_scope = TRUE;
		return;
	}
	vala_interface_writer_write_indent (self);
	char* __temp14 = NULL;
	char* __temp13 = NULL;
	vala_interface_writer_write_string (self, (__temp14 = g_strdup_printf ("[CCode (cprefix = \"%s\")]", (__temp13 = vala_enum_get_cprefix (en)))));
	(__temp14 = (g_free (__temp14), NULL));
	(__temp13 = (g_free (__temp13), NULL));
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public enum ");
	vala_interface_writer_write_identifier (self, vala_data_type_get_name (VALA_DATA_TYPE (en)));
	vala_interface_writer_write_begin_block (self);
}


static void vala_interface_writer_real_visit_end_enum (ValaCodeVisitor* base, ValaEnum* en)
{
	ValaInterfaceWriter * self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_ENUM (en));
	if (VALA_DATA_TYPE (en)->access == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		self->priv->internal_scope = FALSE;
		return;
	}
	vala_interface_writer_write_end_block (self);
	vala_interface_writer_write_newline (self);
}


static void vala_interface_writer_real_visit_enum_value (ValaCodeVisitor* base, ValaEnumValue* ev)
{
	ValaInterfaceWriter * self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_ENUM_VALUE (ev));
	if (self->priv->internal_scope) {
		return;
	}
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_identifier (self, vala_enum_value_get_name (ev));
	vala_interface_writer_write_string (self, ",");
	vala_interface_writer_write_newline (self);
}


static void vala_interface_writer_real_visit_begin_flags (ValaCodeVisitor* base, ValaFlags* fl)
{
	ValaInterfaceWriter * self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_FLAGS (fl));
	if (VALA_DATA_TYPE (fl)->access == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		self->priv->internal_scope = TRUE;
		return;
	}
	vala_interface_writer_write_indent (self);
	char* __temp16 = NULL;
	char* __temp15 = NULL;
	vala_interface_writer_write_string (self, (__temp16 = g_strdup_printf ("[CCode (cprefix = \"%s\")]", (__temp15 = vala_flags_get_cprefix (fl)))));
	(__temp16 = (g_free (__temp16), NULL));
	(__temp15 = (g_free (__temp15), NULL));
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public flags ");
	vala_interface_writer_write_identifier (self, vala_data_type_get_name (VALA_DATA_TYPE (fl)));
	vala_interface_writer_write_begin_block (self);
}


static void vala_interface_writer_real_visit_end_flags (ValaCodeVisitor* base, ValaFlags* fl)
{
	ValaInterfaceWriter * self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_FLAGS (fl));
	if (VALA_DATA_TYPE (fl)->access == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		self->priv->internal_scope = FALSE;
		return;
	}
	vala_interface_writer_write_end_block (self);
	vala_interface_writer_write_newline (self);
}


static void vala_interface_writer_real_visit_flags_value (ValaCodeVisitor* base, ValaFlagsValue* fv)
{
	ValaInterfaceWriter * self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_FLAGS_VALUE (fv));
	if (self->priv->internal_scope) {
		return;
	}
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_identifier (self, vala_flags_value_get_name (fv));
	vala_interface_writer_write_string (self, ",");
	vala_interface_writer_write_newline (self);
}


static void vala_interface_writer_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c)
{
	ValaInterfaceWriter * self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_CONSTANT (c));
	if (self->priv->internal_scope) {
		return;
	}
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public const ");
	char* __temp17 = NULL;
	vala_interface_writer_write_string (self, (__temp17 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_constant_get_type_reference (c)))))));
	(__temp17 = (g_free (__temp17), NULL));
	vala_interface_writer_write_string (self, " ");
	vala_interface_writer_write_identifier (self, vala_constant_get_name (c));
	vala_interface_writer_write_string (self, ";");
	vala_interface_writer_write_newline (self);
}


static void vala_interface_writer_real_visit_field (ValaCodeVisitor* base, ValaField* f)
{
	ValaInterfaceWriter * self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_FIELD (f));
	if (self->priv->internal_scope || f->access == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		return;
	}
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public ");
	if (vala_type_reference_get_data_type (vala_field_get_type_reference (f)) != NULL && vala_data_type_is_reference_type (vala_type_reference_get_data_type (vala_field_get_type_reference (f))) && !vala_type_reference_get_takes_ownership (vala_field_get_type_reference (f))) {
		vala_interface_writer_write_string (self, "weak ");
	}
	char* __temp18 = NULL;
	vala_interface_writer_write_string (self, (__temp18 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_field_get_type_reference (f)))))));
	(__temp18 = (g_free (__temp18), NULL));
	GList* type_args = vala_type_reference_get_type_arguments (vala_field_get_type_reference (f));
	if (!(VALA_IS_ARRAY (vala_type_reference_get_data_type (vala_field_get_type_reference (f)))) && type_args != NULL) {
		vala_interface_writer_write_string (self, "<");
		{
			GList* __temp20 = NULL;
			__temp20 = type_args;
			GList* type_arg_it;
			for (type_arg_it = __temp20; type_arg_it != NULL; type_arg_it = type_arg_it->next) {
				ValaTypeReference* type_arg = type_arg_it->data;
				{
					if (!vala_type_reference_get_takes_ownership (type_arg)) {
						vala_interface_writer_write_string (self, "weak ");
					}
					char* __temp19 = NULL;
					vala_interface_writer_write_string (self, (__temp19 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (type_arg))))));
					(__temp19 = (g_free (__temp19), NULL));
				}
			}
		}
		vala_interface_writer_write_string (self, ">");
	}
	vala_interface_writer_write_string (self, " ");
	vala_interface_writer_write_identifier (self, vala_field_get_name (f));
	vala_interface_writer_write_string (self, ";");
	vala_interface_writer_write_newline (self);
	(type_args == NULL ? NULL : (type_args = (g_list_free (type_args), NULL)));
}


static void vala_interface_writer_write_params (ValaInterfaceWriter* self, GList* params)
{
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	vala_interface_writer_write_string (self, "(");
	gboolean first = TRUE;
	{
		GList* __temp25 = NULL;
		__temp25 = params;
		GList* param_it;
		for (param_it = __temp25; param_it != NULL; param_it = param_it->next) {
			ValaFormalParameter* param = param_it->data;
			{
				if (!first) {
					vala_interface_writer_write_string (self, ", ");
				} else {
					first = FALSE;
				}
				if (vala_formal_parameter_get_ellipsis (param)) {
					vala_interface_writer_write_string (self, "...");
					continue;
				}
				if (vala_type_reference_get_reference_to_value_type (vala_formal_parameter_get_type_reference (param)) || vala_type_reference_get_takes_ownership (vala_formal_parameter_get_type_reference (param))) {
					vala_interface_writer_write_string (self, "ref ");
				} else {
					if (vala_type_reference_get_is_out (vala_formal_parameter_get_type_reference (param))) {
						vala_interface_writer_write_string (self, "out ");
					}
				}
				char* __temp21 = NULL;
				vala_interface_writer_write_string (self, (__temp21 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (param)))))));
				(__temp21 = (g_free (__temp21), NULL));
				GList* type_args = vala_type_reference_get_type_arguments (vala_formal_parameter_get_type_reference (param));
				if (!(VALA_IS_ARRAY (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (param)))) && type_args != NULL) {
					vala_interface_writer_write_string (self, "<");
					{
						GList* __temp23 = NULL;
						__temp23 = type_args;
						GList* type_arg_it;
						for (type_arg_it = __temp23; type_arg_it != NULL; type_arg_it = type_arg_it->next) {
							ValaTypeReference* type_arg = type_arg_it->data;
							{
								if (vala_type_reference_get_takes_ownership (type_arg)) {
									vala_interface_writer_write_string (self, "ref ");
								}
								char* __temp22 = NULL;
								vala_interface_writer_write_string (self, (__temp22 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (type_arg))))));
								(__temp22 = (g_free (__temp22), NULL));
							}
						}
					}
					vala_interface_writer_write_string (self, ">");
				}
				if (vala_type_reference_get_non_null (vala_formal_parameter_get_type_reference (param))) {
					vala_interface_writer_write_string (self, "!");
				}
				vala_interface_writer_write_string (self, " ");
				vala_interface_writer_write_identifier (self, vala_formal_parameter_get_name (param));
				if (vala_formal_parameter_get_default_expression (param) != NULL) {
					vala_interface_writer_write_string (self, " = ");
					char* __temp24 = NULL;
					vala_interface_writer_write_string (self, (__temp24 = vala_code_node_to_string (VALA_CODE_NODE (vala_formal_parameter_get_default_expression (param)))));
					(__temp24 = (g_free (__temp24), NULL));
				}
				(type_args == NULL ? NULL : (type_args = (g_list_free (type_args), NULL)));
			}
		}
	}
	vala_interface_writer_write_string (self, ")");
}


static void vala_interface_writer_real_visit_begin_callback (ValaCodeVisitor* base, ValaCallback* cb)
{
	ValaInterfaceWriter * self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_CALLBACK (cb));
	if (self->priv->internal_scope || VALA_DATA_TYPE (cb)->access == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		return;
	}
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public callback ");
	ValaDataType* __temp26 = NULL;
	ValaDataType* type = (__temp26 = vala_type_reference_get_data_type (vala_callback_get_return_type (cb)), (__temp26 == NULL ? NULL : g_object_ref (__temp26)));
	if (type == NULL) {
		vala_interface_writer_write_string (self, "void");
	} else {
		if (vala_type_reference_get_transfers_ownership (vala_callback_get_return_type (cb))) {
			vala_interface_writer_write_string (self, "ref ");
		}
		char* __temp27 = NULL;
		vala_interface_writer_write_string (self, (__temp27 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_callback_get_return_type (cb)))))));
		(__temp27 = (g_free (__temp27), NULL));
	}
	vala_interface_writer_write_string (self, " ");
	vala_interface_writer_write_identifier (self, vala_data_type_get_name (VALA_DATA_TYPE (cb)));
	vala_interface_writer_write_string (self, " ");
	GList* __temp28 = NULL;
	vala_interface_writer_write_params (self, (__temp28 = vala_callback_get_parameters (cb)));
	(__temp28 == NULL ? NULL : (__temp28 = (g_list_free (__temp28), NULL)));
	vala_interface_writer_write_string (self, ";");
	vala_interface_writer_write_newline (self);
	(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
}


static void vala_interface_writer_real_visit_begin_method (ValaCodeVisitor* base, ValaMethod* m)
{
	ValaInterfaceWriter * self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_METHOD (m));
	if (self->priv->internal_scope || m->access == VALA_MEMBER_ACCESSIBILITY_PRIVATE || vala_method_get_overrides (m)) {
		return;
	}
	if (vala_method_get_no_array_length (m)) {
		gboolean array_found = (vala_method_get_return_type (m) != NULL && VALA_IS_ARRAY (vala_type_reference_get_data_type (vala_method_get_return_type (m))));
		{
			GList* __temp29 = NULL;
			__temp29 = vala_invokable_get_parameters (VALA_INVOKABLE (m));
			GList* param_it;
			for (param_it = __temp29; param_it != NULL; param_it = param_it->next) {
				ValaFormalParameter* param = param_it->data;
				{
					if (vala_formal_parameter_get_type_reference (param) != NULL && VALA_IS_ARRAY (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (param)))) {
						array_found = TRUE;
						break;
					}
				}
			}
			(__temp29 == NULL ? NULL : (__temp29 = (g_list_free (__temp29), NULL)));
		}
		if (array_found) {
			vala_interface_writer_write_indent (self);
			vala_interface_writer_write_string (self, "[NoArrayLength]");
		}
	}
	if (vala_method_get_instance_last (m)) {
		vala_interface_writer_write_indent (self);
		vala_interface_writer_write_string (self, "[InstanceLast]");
	}
	if (vala_method_get_instance_by_reference (m)) {
		vala_interface_writer_write_indent (self);
		vala_interface_writer_write_string (self, "[InstanceByReference]");
	}
	char* __temp31 = NULL;
	char* __temp30 = NULL;
	gboolean __temp32;
	if ((__temp32 = g_utf8_collate ((__temp30 = vala_method_get_cname (m)), (__temp31 = vala_method_get_default_cname (m))) != 0, (__temp31 = (g_free (__temp31), NULL)), (__temp30 = (g_free (__temp30), NULL)), __temp32)) {
		vala_interface_writer_write_indent (self);
		char* __temp34 = NULL;
		char* __temp33 = NULL;
		vala_interface_writer_write_string (self, (__temp34 = g_strdup_printf ("[CCode (cname = \"%s\")]", (__temp33 = vala_method_get_cname (m)))));
		(__temp34 = (g_free (__temp34), NULL));
		(__temp33 = (g_free (__temp33), NULL));
	}
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public");
	if (VALA_IS_CREATION_METHOD (m)) {
		vala_interface_writer_write_string (self, " ");
		ValaDataType* __temp35 = NULL;
		ValaDataType* datatype = (__temp35 = VALA_DATA_TYPE (vala_symbol_get_node (vala_symbol_get_parent_symbol (vala_code_node_get_symbol (VALA_CODE_NODE (m))))), (__temp35 == NULL ? NULL : g_object_ref (__temp35)));
		vala_interface_writer_write_identifier (self, vala_data_type_get_name (datatype));
		if (vala_method_get_name (m) != NULL) {
			vala_interface_writer_write_string (self, ".");
			vala_interface_writer_write_identifier (self, vala_method_get_name (m));
		}
		(datatype == NULL ? NULL : (datatype = (g_object_unref (datatype), NULL)));
	} else {
		if (!vala_method_get_instance (m)) {
			vala_interface_writer_write_string (self, " static");
		} else {
			if (vala_method_get_is_abstract (m)) {
				vala_interface_writer_write_string (self, " abstract");
			} else {
				if (vala_method_get_is_virtual (m)) {
					vala_interface_writer_write_string (self, " virtual");
				}
			}
		}
	}
	if (!(VALA_IS_CREATION_METHOD (m))) {
		vala_interface_writer_write_string (self, " ");
		ValaDataType* __temp36 = NULL;
		ValaDataType* type = (__temp36 = vala_type_reference_get_data_type (vala_method_get_return_type (m)), (__temp36 == NULL ? NULL : g_object_ref (__temp36)));
		if (type == NULL) {
			vala_interface_writer_write_string (self, "void");
		} else {
			if (vala_type_reference_get_transfers_ownership (vala_method_get_return_type (m))) {
			} else {
				if ((vala_type_reference_get_data_type (vala_method_get_return_type (m)) != NULL && vala_data_type_is_reference_type (vala_type_reference_get_data_type (vala_method_get_return_type (m)))) || vala_type_reference_get_type_parameter (vala_method_get_return_type (m)) != NULL) {
					vala_interface_writer_write_string (self, "weak ");
				}
			}
			char* __temp37 = NULL;
			vala_interface_writer_write_string (self, (__temp37 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_method_get_return_type (m)))))));
			(__temp37 = (g_free (__temp37), NULL));
			if (vala_type_reference_get_non_null (vala_method_get_return_type (m))) {
				vala_interface_writer_write_string (self, "!");
			}
		}
		vala_interface_writer_write_string (self, " ");
		vala_interface_writer_write_identifier (self, vala_method_get_name (m));
		(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
	}
	vala_interface_writer_write_string (self, " ");
	GList* __temp38 = NULL;
	vala_interface_writer_write_params (self, (__temp38 = vala_invokable_get_parameters (VALA_INVOKABLE (m))));
	(__temp38 == NULL ? NULL : (__temp38 = (g_list_free (__temp38), NULL)));
	vala_interface_writer_write_string (self, ";");
	vala_interface_writer_write_newline (self);
}


static void vala_interface_writer_real_visit_begin_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m)
{
	ValaInterfaceWriter * self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_CREATION_METHOD (m));
	vala_code_visitor_visit_begin_method (VALA_CODE_VISITOR (self), VALA_METHOD (m));
}


static void vala_interface_writer_real_visit_begin_property (ValaCodeVisitor* base, ValaProperty* prop)
{
	ValaInterfaceWriter * self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_PROPERTY (prop));
	if (self->priv->internal_scope) {
		return;
	}
	if (vala_property_get_no_accessor_method (prop)) {
		vala_interface_writer_write_indent (self);
		vala_interface_writer_write_string (self, "[NoAccessorMethod]");
	}
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public ");
	if (!vala_type_reference_get_takes_ownership (vala_property_get_type_reference (prop))) {
		vala_interface_writer_write_string (self, "weak ");
	}
	char* __temp39 = NULL;
	vala_interface_writer_write_string (self, (__temp39 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_property_get_type_reference (prop)))))));
	(__temp39 = (g_free (__temp39), NULL));
	GList* type_args = vala_type_reference_get_type_arguments (vala_property_get_type_reference (prop));
	if (!(VALA_IS_ARRAY (vala_type_reference_get_data_type (vala_property_get_type_reference (prop)))) && type_args != NULL) {
		vala_interface_writer_write_string (self, "<");
		{
			GList* __temp41 = NULL;
			__temp41 = type_args;
			GList* type_arg_it;
			for (type_arg_it = __temp41; type_arg_it != NULL; type_arg_it = type_arg_it->next) {
				ValaTypeReference* type_arg = type_arg_it->data;
				{
					if (!vala_type_reference_get_takes_ownership (type_arg)) {
						vala_interface_writer_write_string (self, "weak ");
					}
					char* __temp40 = NULL;
					vala_interface_writer_write_string (self, (__temp40 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (type_arg))))));
					(__temp40 = (g_free (__temp40), NULL));
				}
			}
		}
		vala_interface_writer_write_string (self, ">");
	}
	vala_interface_writer_write_string (self, " ");
	vala_interface_writer_write_identifier (self, vala_property_get_name (prop));
	vala_interface_writer_write_string (self, " {");
	if (vala_property_get_get_accessor (prop) != NULL) {
		vala_interface_writer_write_string (self, " get;");
	}
	if (vala_property_get_set_accessor (prop) != NULL) {
		if (vala_property_accessor_get_writable (vala_property_get_set_accessor (prop))) {
			vala_interface_writer_write_string (self, " set");
		}
		if (vala_property_accessor_get_construction (vala_property_get_set_accessor (prop))) {
			vala_interface_writer_write_string (self, " construct");
		}
		vala_interface_writer_write_string (self, ";");
	}
	vala_interface_writer_write_string (self, " }");
	vala_interface_writer_write_newline (self);
	(type_args == NULL ? NULL : (type_args = (g_list_free (type_args), NULL)));
}


static void vala_interface_writer_real_visit_begin_signal (ValaCodeVisitor* base, ValaSignal* sig)
{
	ValaInterfaceWriter * self = VALA_INTERFACE_WRITER (base);
	g_return_if_fail (VALA_IS_SIGNAL (sig));
	if (self->priv->internal_scope || vala_signal_get_access (sig) == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		return;
	}
	if (vala_signal_get_has_emitter (sig)) {
		vala_interface_writer_write_indent (self);
		vala_interface_writer_write_string (self, "[HasEmitter]");
	}
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public signal ");
	ValaDataType* __temp42 = NULL;
	ValaDataType* type = (__temp42 = vala_type_reference_get_data_type (vala_signal_get_return_type (sig)), (__temp42 == NULL ? NULL : g_object_ref (__temp42)));
	if (type == NULL) {
		vala_interface_writer_write_string (self, "void");
	} else {
		if (vala_type_reference_get_transfers_ownership (vala_signal_get_return_type (sig))) {
			vala_interface_writer_write_string (self, "ref ");
		}
		char* __temp43 = NULL;
		vala_interface_writer_write_string (self, (__temp43 = vala_symbol_get_full_name (vala_code_node_get_symbol (VALA_CODE_NODE (vala_type_reference_get_data_type (vala_signal_get_return_type (sig)))))));
		(__temp43 = (g_free (__temp43), NULL));
		if (vala_type_reference_get_non_null (vala_signal_get_return_type (sig))) {
			vala_interface_writer_write_string (self, "!");
		}
	}
	vala_interface_writer_write_string (self, " ");
	vala_interface_writer_write_identifier (self, vala_signal_get_name (sig));
	vala_interface_writer_write_string (self, " ");
	GList* __temp44 = NULL;
	vala_interface_writer_write_params (self, (__temp44 = vala_invokable_get_parameters (VALA_INVOKABLE (sig))));
	(__temp44 == NULL ? NULL : (__temp44 = (g_list_free (__temp44), NULL)));
	vala_interface_writer_write_string (self, ";");
	vala_interface_writer_write_newline (self);
	(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
}


static void vala_interface_writer_write_indent (ValaInterfaceWriter* self)
{
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	gint i = 0;
	if (!self->priv->bol) {
		fputc ('\n', self->priv->stream);
	}
	for (i = 0; i < self->priv->indent; i++) {
		fputc ('\t', self->priv->stream);
	}
	self->priv->bol = FALSE;
}


static void vala_interface_writer_write_identifier (ValaInterfaceWriter* self, const char* s)
{
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	g_return_if_fail (s != NULL);
	if (g_utf8_collate (s, "base") == 0 || g_utf8_collate (s, "callback") == 0 || g_utf8_collate (s, "class") == 0 || g_utf8_collate (s, "construct") == 0 || g_utf8_collate (s, "flags") == 0 || g_utf8_collate (s, "foreach") == 0 || g_utf8_collate (s, "in") == 0 || g_utf8_collate (s, "interface") == 0 || g_utf8_collate (s, "lock") == 0 || g_utf8_collate (s, "namespace") == 0 || g_utf8_collate (s, "out") == 0 || g_utf8_collate (s, "ref") == 0) {
		fputc ('@', self->priv->stream);
	}
	vala_interface_writer_write_string (self, s);
}


static void vala_interface_writer_write_string (ValaInterfaceWriter* self, const char* s)
{
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	g_return_if_fail (s != NULL);
	fprintf (self->priv->stream, "%s", s);
	self->priv->bol = FALSE;
}


static void vala_interface_writer_write_newline (ValaInterfaceWriter* self)
{
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	fputc ('\n', self->priv->stream);
	self->priv->bol = TRUE;
}


static void vala_interface_writer_write_begin_block (ValaInterfaceWriter* self)
{
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	if (!self->priv->bol) {
		fputc (' ', self->priv->stream);
	} else {
		vala_interface_writer_write_indent (self);
	}
	fputc ('{', self->priv->stream);
	vala_interface_writer_write_newline (self);
	self->priv->indent++;
}


static void vala_interface_writer_write_end_block (ValaInterfaceWriter* self)
{
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	self->priv->indent--;
	vala_interface_writer_write_indent (self);
	fprintf (self->priv->stream, "}");
}


static void vala_interface_writer_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaInterfaceWriter * self = VALA_INTERFACE_WRITER (object);
	switch (property_id) {
	}
}


static void vala_interface_writer_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaInterfaceWriter * self = VALA_INTERFACE_WRITER (object);
	switch (property_id) {
	}
}


static void vala_interface_writer_class_init (ValaInterfaceWriterClass * klass)
{
	vala_interface_writer_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaInterfaceWriterPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_interface_writer_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_interface_writer_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_interface_writer_dispose;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_namespace = vala_interface_writer_real_visit_begin_namespace;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_namespace = vala_interface_writer_real_visit_end_namespace;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_class = vala_interface_writer_real_visit_begin_class;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_class = vala_interface_writer_real_visit_end_class;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_struct = vala_interface_writer_real_visit_begin_struct;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_struct = vala_interface_writer_real_visit_end_struct;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_interface = vala_interface_writer_real_visit_begin_interface;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_interface = vala_interface_writer_real_visit_end_interface;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_enum = vala_interface_writer_real_visit_begin_enum;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_enum = vala_interface_writer_real_visit_end_enum;
	VALA_CODE_VISITOR_CLASS (klass)->visit_enum_value = vala_interface_writer_real_visit_enum_value;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_flags = vala_interface_writer_real_visit_begin_flags;
	VALA_CODE_VISITOR_CLASS (klass)->visit_end_flags = vala_interface_writer_real_visit_end_flags;
	VALA_CODE_VISITOR_CLASS (klass)->visit_flags_value = vala_interface_writer_real_visit_flags_value;
	VALA_CODE_VISITOR_CLASS (klass)->visit_constant = vala_interface_writer_real_visit_constant;
	VALA_CODE_VISITOR_CLASS (klass)->visit_field = vala_interface_writer_real_visit_field;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_callback = vala_interface_writer_real_visit_begin_callback;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_method = vala_interface_writer_real_visit_begin_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_creation_method = vala_interface_writer_real_visit_begin_creation_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_property = vala_interface_writer_real_visit_begin_property;
	VALA_CODE_VISITOR_CLASS (klass)->visit_begin_signal = vala_interface_writer_real_visit_begin_signal;
}


static void vala_interface_writer_init (ValaInterfaceWriter * self)
{
	self->priv = VALA_INTERFACE_WRITER_GET_PRIVATE (self);
	self->priv->bol = TRUE;
	self->priv->internal_scope = FALSE;
}


static void vala_interface_writer_dispose (GObject * obj)
{
	ValaInterfaceWriter * self = VALA_INTERFACE_WRITER (obj);
	(self->priv->context == NULL ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL)));
	(self->priv->stream == NULL ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL)));
	(self->priv->current_cheader_filename = (g_free (self->priv->current_cheader_filename), NULL));
	ValaInterfaceWriterClass * klass;
	GObjectClass * parent_class;
	klass = VALA_INTERFACE_WRITER_CLASS (g_type_class_peek (VALA_TYPE_INTERFACE_WRITER));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_interface_writer_get_type ()
{
	static GType vala_interface_writer_type_id = 0;
	if (G_UNLIKELY (vala_interface_writer_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaInterfaceWriterClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_interface_writer_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaInterfaceWriter), 0, (GInstanceInitFunc) vala_interface_writer_init };
		vala_interface_writer_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaInterfaceWriter", &g_define_type_info, 0);
	}
	return vala_interface_writer_type_id;
}




