/*
 *  arch-tag: Header for core RBNode tree-structure entry
 *
 *  Copyright (C) 2002 Jorn Baayen <jorn@nl.linux.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */


#ifndef RB_NODE_H
#define RB_NODE_H

#include <libxml/tree.h>

G_BEGIN_DECLS

/* convenience macro to check node validity */

#define RB_IS_NODE(o)	(o != NULL)

typedef struct RBNode RBNode;

typedef enum
{
	RB_NODE_DESTROY,           /* RBNode *node */
	RB_NODE_RESTORED,          /* RBNode *node */
	RB_NODE_CHILD_ADDED,       /* RBNode *node, RBNode *child */
	RB_NODE_CHILD_CHANGED,     /* RBNode *node, RBNode *child */
	RB_NODE_CHILD_REMOVED,     /* RBNode *node, RBNode *child, guint old_index */
	RB_NODE_CHILDREN_REORDERED /* RBNode *node, int *new_order */
} RBNodeSignalType;

#include "rb-node-db.h"

#define RB_NODE_MAX_SAVED_PROP 512

/* properties */
enum
{
	RB_NODE_PROP_NAME               = 0,
	RB_NODE_PROP_GENRE              = 2,
	RB_NODE_PROP_ARTIST             = 3,
	RB_NODE_PROP_ALBUM              = 4,
	RB_NODE_PROP_REAL_GENRE         = 5,
	RB_NODE_PROP_REAL_ARTIST        = 6,
	RB_NODE_PROP_REAL_ALBUM         = 7,
	RB_NODE_PROP_TRACK_NUMBER       = 8,
	RB_NODE_PROP_DURATION		= 9,
	RB_NODE_PROP_DURATION_STR       = 10,
	RB_NODE_PROP_FILE_SIZE          = 11,
	RB_NODE_PROP_LOCATION           = 12,
	RB_NODE_PROP_MTIME              = 13,
	RB_NODE_PROP_RESERVED           = 14,
	RB_NODE_PROP_RATING	        = 15,
	RB_NODE_PROP_PLAY_COUNT         = 16,
	RB_NODE_PROP_LAST_PLAYED        = 17,
	RB_NODE_PROP_LAST_PLAYED_STR	= 18,
	RB_NODE_PROP_QUALITY            = 21,
	RB_NODE_PROP_ALT_LOCATIONS      = 22,
	RB_NODE_PROP_PRIORITY		= 23,
	RB_NODE_PROP_IRADIO_SOURCE	= 24,
	/* ... */
	RB_NODE_PROP_UNUSED		= RB_NODE_MAX_SAVED_PROP,
	/* These properties will not be saved. */
	RB_NODE_PROP_NAME_SORT_KEY,
	RB_NODE_PROP_ARTIST_SORT_KEY,
	RB_NODE_PROP_ALBUM_SORT_KEY,
};

typedef void (*RBNodeCallback) (RBNode *node, ...);

RBNode   *rb_node_new                   (RBNodeDb *db);

RBNode   *rb_node_new_with_id           (RBNodeDb *db,
					     gulong reserved_id);

RBNodeDb *rb_node_get_db		    (RBNode *node);

/* unique node ID */
long        rb_node_get_id                (RBNode *node);

/* refcounting */
void        rb_node_ref                   (RBNode *node);
void        rb_node_unref                 (RBNode *node);
void	    rb_node_unref_with_locked_child (RBNode *node, RBNode *child);

/* locking */
void        rb_node_freeze                (RBNode *node);
void        rb_node_thaw                  (RBNode *node);

/* signals */
int         rb_node_signal_connect_object (RBNode *node,
					     RBNodeSignalType type,
					     RBNodeCallback callback,
					     GObject *object);

void        rb_node_signal_disconnect     (RBNode *node,
					     int signal_id);

/* properties */
void        rb_node_set_property          (RBNode *node,
				             guint property_id,
				             const GValue *value);
void        rb_node_set_property_unlocked (RBNode *node,
					   guint property_id,
					   const GValue *value);
gboolean    rb_node_get_property          (RBNode *node,
				             guint property_id,
				             GValue *value);

const char *rb_node_get_property_string   (RBNode *node,
					     guint property_id);
gboolean    rb_node_get_property_boolean  (RBNode *node,
					     guint property_id);
long        rb_node_get_property_long     (RBNode *node,
					     guint property_id);
int         rb_node_get_property_int      (RBNode *node,
					     guint property_id);
double      rb_node_get_property_double   (RBNode *node,
					     guint property_id);
float       rb_node_get_property_float    (RBNode *node,
					     guint property_id);
gpointer    rb_node_get_property_pointer  (RBNode *node,
					   guint property_id);
/* free return value */
char       *rb_node_get_property_time     (RBNode *node,
					     guint property_id);

/* xml storage */
void          rb_node_save_to_xml         (RBNode *node,
					     xmlNodePtr parent_xml_node);
RBNode     *rb_node_new_from_xml        (RBNodeDb *db,
					     xmlNodePtr xml_node);

/* DAG structure */
void        rb_node_add_child             (RBNode *node,
					   RBNode *child);
void        rb_node_add_child_unlocked    (RBNode *node,
					   RBNode *child);

void        rb_node_remove_child          (RBNode *node,
					   RBNode *child);
void        rb_node_remove_child_unlocked (RBNode *node,
					   RBNode *child);
void	    rb_node_sort_children	    (RBNode *node,
					     GCompareFunc compare_func);
gboolean    rb_node_has_child             (RBNode *node,
					   RBNode *child);

void	    rb_node_reorder_children	    (RBNode *node,
					     int *new_order);

/* Note that rb_node_get_children freezes the node; you'll have to thaw it when done.
 * This is to prevent the data getting changed from another thread. */
GPtrArray    *rb_node_get_children        (RBNode *node);
int           rb_node_get_n_children      (RBNode *node);
RBNode     *rb_node_get_nth_child       (RBNode *node,
					     guint n);
int           rb_node_get_child_index     (RBNode *node,
					     RBNode *child);
RBNode     *rb_node_get_next_child      (RBNode *node,
					     RBNode *child);
RBNode     *rb_node_get_previous_child  (RBNode *node,
					     RBNode *child);

/* Update 'play' statistics */
void        rb_node_update_play_statistics (RBNode *node);

G_END_DECLS

#endif /* __RB_NODE_H */
