// RadioActive Copyright (C) 1999 RDI Gerg <cactus@cactus.rulez.org>
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2
// (included in the RadioActive distribution in doc/GPL) as published by
// the Free Software Foundation.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

//$Id: radioactive.cc,v 1.36 2001/02/24 19:42:35 cactus Exp $ -*- c++ -*-

#include "radioactive.h"

#include "common/file.h"
#include "common/driver.h"
#include <getopt.h>
#include <stdexcept>

#include "common/properties.h"

#include "common/wndMain.h"

using namespace SigC;

namespace cmd
{
    bool mute = false;
    bool console = false;
}

void usage(const string& myname)
{
    char* dev_help_str_i18n = _("Radio device (default is %s)");
    char* dev_help_str = new char(strlen(dev_help_str_i18n) +
				  arg::devfile.length());
    sprintf(dev_help_str, dev_help_str_i18n, arg::devfile.c_str());
    
    cerr << _("Usage:") << " " << myname << " [-f " << _("freq")
	 << "|-m|-v] [-d " << _("devicefile") << "] [-c]" << endl << endl;
    cerr << "-f\t" << _("Specify the initial frequency") << endl
	 << "-d\t" << dev_help_str << endl
	 << "-c\t" << _("Use command line-only interface (default if not under X)")
	 << endl
	 << "-m\t" << _("Just mute radio and exit") << endl
	 << "-v\t" << _("Print version information and exit") << endl
	 << endl;
}

void parse(int argc, char **argv)
{
    int c;
    while ((c = getopt (argc, argv, "vf:d:cmh")) != -1)
	switch (c)
	{
	case 'v':
	    cout << PROGNAME << " " << VERSION
		 << " (C) 1999-2001 ERDI Gergo <cactus@cactus.rulez.org>" << endl;
	    exit(0);
	case 'f':
	    arg::freq = atof(optarg);
	    break;
	case 'd':
	    arg::devfile = optarg;
	    break;
	case 'c':
	    cmd::console = true;
	    break;
	case 'm':
	    cmd::mute = true;
	    break;
	case 'h':
	    usage(argv[0]);
	    exit(0);
	case '?':
	    usage(argv[0]);
	    exit(1);
	}
}

int stop(GdkEventAny* e)
{
    Gnome::Main::quit();
    return 0;
}

int main(int argc, char** argv)
{    
    bindtextdomain(PACKAGE, GNOMELOCALEDIR);
    textdomain(PACKAGE);
    
    char **arg_dummy = new char*[1];
    arg_dummy[0] = argv[0];
    arg_dummy[1] = NULL;
    
    gnomelib_init(PROGNAME, VERSION);
    
    // Load configuration
    read_config();
    
    parse(argc, argv);
    Radio* radio;

    try
    {
	radio = new Radio(arg::devfile);
    }
    catch (runtime_error e)
    {
	cerr << e.what() << endl;
	exit(1);
    }

    // Muting
    if (cmd::mute)
    {
	radio->mute(true);
	exit(0);
    }
	
    // Load previous values
    radio->set(arg::freq);
    radio->set_vol(arg::volume);
    radio->mute(arg::mute);

    // User interface
    Gnome::Main* m = 0;
    if (getenv("DISPLAY") && !cmd::console)
    {
	m = new Gnome::Main(PROGNAME, VERSION, 1, arg_dummy);
	wndMain* w = new wndMain(*radio);
	w->delete_event.connect(slot(&stop));
	vector<pair<float, string> > stations =
 	    load_button_values(arg::progbutton_col * arg::progbutton_row);
	
	w->set_speedtuner(stations);
	if (arg::save_pos)
	    w->set_uposition(arg::pos_x, arg::pos_y);
	w->show();

	m->run();
	
	save_button_values(w->get_speedtuner());
	if (arg::save_pos)
	    w->get_position(arg::pos_x, arg::pos_y);
    }
    
    // Save current values
    arg::freq = radio->get();
    arg::volume = radio->get_vol();
    arg::mute = radio->get_mute();
    save_config();

    if (arg::mute_exit)
	radio->mute(true);

    delete radio;
    delete[] arg_dummy;
//    if (m) delete m;
    return 0;
}
