#ifndef PDF_H			// -*- c++ -*-
#define PDF_H
///
// Copyright (C) 2002 - 2004, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "fonts/fontinfo.hh"
#include <glibmm/ustring.h>
#include <sstream>
#include <vector>
#include <map>

namespace PDF {
  
  class Object {
  public:
    Object() {}
    virtual ~Object();
    
    virtual std::ostream& write(std::ostream& out) = 0;
  protected:
    // Remember a dynamically allocated part that is to be deleted when 
    // this part is deleted:
    template<class AnyPart>
    AnyPart *manage(AnyPart *part)
      {managed.push_back(dynamic_cast<Object*>(part)); return part;}
    
  private:
    Object(Object&);
    void operator = (Object&);

    typedef std::vector<Object*> Managed;
    Managed managed;
  };
  
  std::ostream& operator << (std::ostream& out, Object& obj);
  
  class Ref : public Object {
  public:
    typedef unsigned Num; 
    typedef unsigned Generation; 
    
    Ref(Num num, Generation generation)
      : num_(num), generation_(generation) {}
    
    Num get_num() const { return num_; }
    Generation get_generation() const { return generation_; }

    std::ostream& write(std::ostream& out) {
      return out << num_ << ' ' << generation_ << " R";
    }

  private:
    Num num_;
    Generation generation_;
  }; 

  class ReferencedObject : public Object {
  public:
    ReferencedObject(Ref* ref, Object* obj);
    
    Ref* get_ref();
    Object* get_obj() { return obj_; }
    std::streampos get_offset() const;
    
    std::ostream& write(std::ostream& out);
    
  private:
    Object* obj_;
    Ref *ref_;
    std::streampos offset_;
  };
  
  class Dictionary: public Object {
  public:
    explicit Dictionary();
    
    Object* set_entry(std::string name, Object *object);
    const Object* get_entry(const std::string& name) const;
    
    std::ostream& write(std::ostream& out);
    
  private:
    typedef std::map<std::string, Object*> Entries;
    Entries entries;
  };

  class Stream : public Dictionary {
  public:
    Stream() {}
    
    std::ostream& data() { return data_; }
    std::ostream& write(std::ostream& out);
    
  private:
    std::ostringstream data_;
  };

  class Page;
  
  class Content: public Stream{ // only text so far
  public:   
    explicit Content(Page* page);
    
    std::ostream& write(std::ostream& out);
    
    void selectfont(const font::FontInfo& font);
    void setgray(float gray);
    void moveto(float xpos, float ypos);
    void textRise(float rise);
    void setWordSpace(const float& w);
    void setCharSpace(const float& w);
    
    void show(const Glib::ustring& s);
    void whitespace();
    void whitespace(float width);
    
  private:
    void commitText();
    
    Page* page_;
    float last_xpos, last_ypos, cur_charspace, cur_wordspace;
    std::ostringstream textbuf;
  };
  
  class Document;
  
  class XRefs: public Object {
    friend class Ref;
  public:
    explicit XRefs(Document *document);
    ReferencedObject* add_object(Object* object);
    std::streampos get_xref_offset();
    std::ostream& write(std::ostream& out);
    Ref::Num get_num_of_refs() const;

  private:
    Document* document_;
    std::streampos xref_offset;

    typedef std::vector<Ref*> Refs;
    Refs refs;  
    typedef std::vector<ReferencedObject*> Objects;
    Objects objects;  
  };

  class Document: public Object {
  public:
    Document();
    Content* add_page(int width, int height);
    std::ostream& write(std::ostream& out);
    XRefs *get_xrefs() {return &xrefs;}

    ReferencedObject* getFontObject(const font::FontInfo& fontname);
    
  private:
    typedef std::vector<ReferencedObject*> Pages;
    XRefs xrefs;
    Pages pages;
    
    ReferencedObject* page_tree_root;

    std::map<std::string, ReferencedObject*> used_font;
  };

};

#endif
