///
// Copyright (C) 2002, 2003, Fredrik Arnerup & Rasmus Kaj, See COPYING
///
#include "pagent.h"
#include "group.h"
#include "pptout/action.h" // Todo
#include <util/stringutil.h>
#include <sigc++/sigc++.h>

Error::InvalidPageNum::InvalidPageNum(int num) 
  : logic_error("Invalid page number: " + tostr(num))
{}

Pagent::Pagent(Group *_parent, const std::string& _name) :
  parent(_parent), resizable(false),
  locked(false), flow_around(false), name(_name),
  obstacle_margin(0), width(1), height(1)
{
  // a geometry change implies a properties change
  geometry_changed_signal.connect(props_changed_signal.slot());
  // a geometry change does NOT imply a redraw

  connect_to_parent();
}


Pagent::~Pagent() {}

namespace {
  class SetMatrixAction: public Action {
  public:
    SetMatrixAction(const Matrix &matrix, const std::string &description,
		      Pagent &_pagent): 
      Action(description),
      pagent(_pagent),
      new_matrix(matrix), old_matrix(pagent.get_matrix())
    {}
    void undo() const {pagent.set_matrix(new_matrix);}
    void redo() const {pagent.set_matrix(old_matrix);}
  private:
    Pagent &pagent;
    Matrix new_matrix, old_matrix;
  };
};

void Pagent::set_translation(const Vector& v, undo::Undoable* history) {
  Matrix m = get_matrix(); m.set_tr(v);
  if(history)
    history->push_action(new SetMatrixAction(m, "Move", *this));
  // important to create action object before matrix is set
  set_matrix(m);
}

void Pagent::set_rotation(float angle, undo::Undoable* history) {
  Matrix m = get_matrix(); m.set_rot(angle);
  if(history)
    history->push_action(new SetMatrixAction(m, "Rotate", *this));
  // important to create action object before matrix is set
  set_matrix(m);
}

void Pagent::set_scaling(float xfactor, float yfactor, 
			 undo::Undoable* history) {
  Matrix m = get_matrix(); m.set_scale(xfactor, yfactor);
  if(history)
    history->push_action(new SetMatrixAction(m, "Scale", *this));
  // important to create action object before matrix is set
  set_matrix(m);
}

namespace {
  class SetSizeAction: public Action {
  public:
    SetSizeAction(float w, float h, Pagent &_pagent): 
      Action("Resize"), 
      pagent(_pagent),
      new_w(w), new_h(h), 
      old_w(pagent.get_width()), old_h(pagent.get_height())
    {}
    void undo() const {pagent.set_size(new_w, new_h, false);}
    void redo() const {pagent.set_size(old_w, old_h, false);}
  private:
    Pagent &pagent;
    float new_w, new_h, old_w, old_h;
  };
};

void Pagent::set_size(float w, float h, undo::Undoable* history) {
//   if(!is_resizable())
//     return;
  if(w != width || h != height) {
    if(history)
      history->push_action(new SetSizeAction(w, h, *this));
    // important to create action object before size is changed
    width = w; height = h;
    geometry_changed_signal(this);
  }
}

namespace {
  class SetLockAction: public Action {
  public:
    SetLockAction(bool locked, Pagent &_pagent): 
      Action(locked ? "Lock" : "Unlock"),
      pagent(_pagent)
    {}
    void undo() const {pagent.set_lock(!pagent.get_lock(), false);}
    void redo() const {undo();}
  private:
    Pagent &pagent;
  };
};

void Pagent::set_lock(bool _locked, undo::Undoable* history) {
  if(_locked != locked) {
    locked = _locked;
    if(history)
      history->push_action(new SetLockAction(_locked, *this));
    props_changed_signal(this);
    ready_to_draw_signal(this);
  }
}

namespace {
  class SetFlowAroundAction: public Action {
  public:
    SetFlowAroundAction(bool flow_around, Pagent &_pagent): 
      Action(flow_around 
	     ? "Enable text avoids object"
	     : "disable text avoids object"),
      pagent(_pagent)
    {}
    void undo() const 
    {pagent.set_flow_around(!pagent.get_flow_around(), false);}
    void redo() const {undo();}
  private:
    Pagent &pagent;
  };
};

void Pagent::set_flow_around(bool _flow_around, undo::Undoable* history) {
  if(flow_around != _flow_around) {
    flow_around = _flow_around;
    if(history)
      history->push_action(new SetFlowAroundAction(_flow_around, *this));
    props_changed_signal(this);
  }
}

namespace {
  class SetMarginAction: public Action {
  public:
    SetMarginAction(float margin, Pagent &_pagent): 
      Action("Change text avoidance margin"),
      pagent(_pagent), 
      old_margin(pagent.get_obstacle_margin()), new_margin(margin)
    {}
    void undo() const {pagent.set_obstacle_margin(old_margin, false);}
    void redo() const {pagent.set_obstacle_margin(new_margin, false);}
  private:
    Pagent &pagent;
    float old_margin, new_margin;
  };
};


void Pagent::set_obstacle_margin(float margin, undo::Undoable* history) {
  if(margin != obstacle_margin) {
    if(history)
      history->push_action(new SetMarginAction(margin, *this));
    // important to create action object before margin is changed
    obstacle_margin = margin;
    props_changed_signal(this);
  }
}

namespace {
  class SetNameAction: public Action {
  public:
    SetNameAction(std::string name, Pagent &_pagent): 
      Action("Rename"),
      pagent(_pagent), old_name(pagent.get_name()), new_name(name)
    {}
    void undo() const {pagent.set_name(old_name, false);}
    void redo() const {pagent.set_name(new_name, false);}
  private:
    Pagent &pagent;
    std::string old_name, new_name;
  };
};

void Pagent::set_name(const std::string &_name, undo::Undoable* history) {
  if(_name != name) {
    if(history)
      history->push_action(new SetNameAction(_name, *this));
    // important to create action object before changing name
    name = _name;
    props_changed_signal(this);
  }
}

int Pagent::get_page_num() const {
  if(parent)
    return parent->get_page_num();
  else
    throw Error::NoParent();
}

void Pagent::set_parent(Group *_parent) {
  if(parent) {
    draw_connection.disconnect();
    geometry_connection.disconnect();
    props_connection.disconnect();
  }
  parent = _parent;
  connect_to_parent();
  props_changed_signal(this);
}

void Pagent::set_matrix(Matrix m) {
  matrix = m;
  geometry_changed_signal(this);
  ready_to_draw_signal(this);
}

Group& Pagent::get_parent() {
  if(!parent) 
    throw Error::NoParent(); 
  else
    return *parent;
}

const Group& Pagent::get_parent() const {
  if(!parent) 
    throw Error::NoParent(); 
  else
    return *parent;
}

void Pagent::connect_to_parent() {
  // Anything that happens to a pagent, happens to its parent.
  // It is simpler to put this in pagent than in group, because
  // a group would need to maintain a list of connections.
  // Note: we will get multible implicit signals.
  if(parent) {
    using SigC::slot;
      draw_connection = ready_to_draw_signal.connect
	(slot(*parent, &Group::child_ready_to_draw));

      geometry_connection = geometry_changed_signal.connect
	(slot(*parent, &Group::child_geometry_changed));

      props_connection = props_changed_signal.connect
	(slot(*parent, &Group::child_props_changed));
    }
}
