/*
-*- Mode: perl; tab-width: 5; indent-tabs-mode: nil; c-basic-offset: 5 -*-
 *
 * mtm-theme :
 *
 * Copyright (C) 2000-2001 Ximian, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors: Richard Hestilow <hestgray@ionet.net>
 *
 */

#include <config.h>
#include <dirent.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#if 0 /* xml-config , gnome-config --libs xml & pkg-config --libs xml are all broKEN
	  * i sent a patch to the list xml@gnome.org for now deal with their brokenes
	  * Chema
	  */
#include <gnome-xml/parser.h>
#include <gnome-xml/xmlmemory.h>
#else
#include <parser.h>
#include <xmlmemory.h>
#endif

#include <mtm/mtm-theme.h>
#include <mtm/mtm-ext-handler.h>
#include "internals.h"

#include <gdk/gdk.h>

static MtmStatefulClass *mtm_theme_parent_class;

struct _MtmThemePrivate {
	GList *exts;
};

enum
{
	ARG_0,
	ARG_DESC,
	ARG_NAME,
	ARG_FILENAME,
	ARG_PREVIEW,
	ARG_AUTHOR,
	ARG_AUTHOR_EMAIL
};

static void mtm_theme_set_arg (GtkObject *object, GtkArg *arg, guint arg_id);
static void mtm_theme_get_arg (GtkObject *object, GtkArg *arg, guint arg_id);

static void parse_args (MtmExt *ext, xmlNodePtr node);

static void
mtm_theme_destroy (GtkObject *object)
{
	MtmTheme *theme = MTM_THEME (object);
	GList *list = NULL;

	if (theme->desc)
		g_free (theme->desc);
	if (theme->name)
		g_free (theme->name);
	if (theme->filename)
		g_free (theme->filename);
	if (theme->preview)
		g_free (theme->preview);
	if (theme->author)
		g_free (theme->author);
	if (theme->author_email)
		g_free (theme->author_email);
	
	list = theme->priv->exts;
	while (list)
	{
		gtk_object_unref (GTK_OBJECT (list->data));
		list = list->next;
	}
	g_list_free (theme->priv->exts);
	g_free (theme->priv);

	if (GTK_OBJECT_CLASS(mtm_theme_parent_class)->destroy)
		(*GTK_OBJECT_CLASS(mtm_theme_parent_class)->destroy)(object);
}

static void
mtm_theme_class_init (GtkObjectClass *object_class)
{
	mtm_theme_parent_class = gtk_type_class (mtm_stateful_get_type ());

	object_class->set_arg = mtm_theme_set_arg;
	object_class->get_arg = mtm_theme_get_arg;

	gtk_object_add_arg_type ("MtmTheme::desc", GTK_TYPE_STRING,
						GTK_ARG_READWRITE, ARG_DESC);
	gtk_object_add_arg_type ("MtmTheme::name", GTK_TYPE_STRING,
						GTK_ARG_READWRITE, ARG_NAME);
	gtk_object_add_arg_type ("MtmTheme::filename", GTK_TYPE_STRING,
						GTK_ARG_READWRITE, ARG_FILENAME);
	gtk_object_add_arg_type ("MtmTheme::preview", GTK_TYPE_STRING,
						GTK_ARG_READWRITE, ARG_PREVIEW);
	gtk_object_add_arg_type ("MtmTheme::author", GTK_TYPE_STRING,
						GTK_ARG_READWRITE, ARG_AUTHOR);
	gtk_object_add_arg_type ("MtmTheme::author_email", GTK_TYPE_STRING,
						GTK_ARG_READWRITE, ARG_AUTHOR_EMAIL);

	object_class->destroy = mtm_theme_destroy;
}

static void
mtm_theme_init (GtkObject *object)
{
	MtmTheme *theme = MTM_THEME (object);

	theme->priv = g_new0 (MtmThemePrivate, 1);
}

/**
 * mtm_theme_get_type:
 * @void:
 *
 * Registers the #MtmTheme class if necessary, and returns the type ID
 * associated to it.
 * 
 * Return value: The type ID of the #MtmTheme class.
 **/
GtkType
mtm_theme_get_type (void)
{
	static GtkType type = 0;

	if (!type)
	{
		GtkTypeInfo info =
		{
			"MtmTheme",
			sizeof (MtmTheme),
			sizeof (MtmThemeClass),
			(GtkClassInitFunc) mtm_theme_class_init,
			(GtkObjectInitFunc) mtm_theme_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		type = gtk_type_unique (mtm_stateful_get_type (), &info);
	}

	return type;
}

/**
 * mtm_theme_activate:
 * @theme: The theme
 *
 * Activates all theme extensions that have the "activate" flag
 * set. This does not actually update the extensions.
 *
 * Return value: The result of the operation.
 **/
MtmResult
mtm_theme_activate (MtmTheme *theme)
{
	GList *list;
	MtmResult ret = MTM_OK;

	g_return_val_if_fail (theme != NULL, MTM_GENERAL_ERROR);

	
	list = theme->priv->exts; 
	while (list)
	{
		MtmExt *ext = list->data;
		MtmExt *pas;

		list = list->next;
		if (!ext->activate)
			continue;

		/*FIXME: Evil hack */
		if (gtk_object_get_data (GTK_OBJECT (ext), "default"))
			pas = NULL;
		else
			pas = ext;
			
		mtm_ext_handler_check_args (ext->handler);

		if (ext->handler->n_subclass_args)
		{
			gtk_object_set (GTK_OBJECT (ext->handler),
						 "ext_context", ext, NULL);
		}
		ret = ext->handler->activate (ext->handler, pas);
	}
	
	return ret;
}

/**
 * mtm_theme_update_exts:
 * @theme: The theme
 * 
 * Updates all theme extensions that have the "update_ext" flag set.
 * 
 * Return value: The result of the operation.
 */
MtmResult
mtm_theme_update_exts (MtmTheme *theme)
{
	GList *list;
	MtmResult ret = MTM_OK;

	g_return_val_if_fail (theme != NULL, MTM_GENERAL_ERROR);

	list = theme->priv->exts; 
	while (list)
	{
		MtmExt *ext = list->data;
		list = list->next;
		if (!ext->activate || !ext->update_ext)
			continue;

		mtm_ext_handler_check_args (ext->handler);
		
		if (ext->handler->n_subclass_args)
		{
			gtk_object_set (GTK_OBJECT (ext->handler),
					"ext_context", ext, NULL);
		}
		ret = ext->handler->update_ext (ext->handler);
	}
	return ret;
}

/**
 * mtm_theme_get_exts
 * @theme: The theme
 *
 * Queries the theme for a list of extensions.
 *
 * Return value: A list of the theme extensions
 */
GList *
mtm_theme_get_exts (MtmTheme *theme)
{
	g_return_val_if_fail (theme != NULL, NULL);

	return theme->priv->exts;
}

void
mtm_theme_add_ext (MtmTheme *theme, MtmExt *ext)
{

	g_return_if_fail (theme != NULL);
	g_return_if_fail (ext != NULL);
	
	theme->priv->exts = g_list_append (theme->priv->exts, ext);
	gtk_object_ref (GTK_OBJECT (ext));
	gtk_object_sink (GTK_OBJECT (ext));
}

void
mtm_theme_remove_ext (MtmTheme *theme, MtmExt *ext)
{
	GList *l;

	g_return_if_fail (theme != NULL);
	g_return_if_fail (ext != NULL);

	l = theme->priv->exts;
	while (l)
	{	
		GList *l2 = l->next;
		if (l->data == ext)
		{
			
			theme->priv->exts = g_list_remove_link (
				theme->priv->exts, l);
			gtk_object_unref (GTK_OBJECT (ext));
		}
		l = l2;
	}
}

MtmTheme *
mtm_theme_new (MtmEnv *env)
{
	MtmTheme *theme;

	g_return_val_if_fail (env != NULL, NULL);
	
	theme = gtk_type_new (mtm_theme_get_type ());
	mtm_theme_construct (theme, env);

	return theme;
}


MtmTheme *
mtm_theme_new_from_file (MtmEnv *env, const gchar *filename)
{
	MtmTheme *theme;

	g_return_val_if_fail (env != NULL, NULL);
	g_return_val_if_fail (filename != NULL, NULL); 

	theme = gtk_type_new (mtm_theme_get_type ());
	mtm_theme_construct_from_file (theme, env, filename);
	if (!theme->filename)
	{
		gtk_object_destroy (GTK_OBJECT (theme));
		theme = NULL;
	}

	return theme;
}

MtmTheme *
mtm_theme_new_from_defaults (MtmEnv *env)
{
	MtmTheme *theme;

	g_return_val_if_fail (MTM_IS_ENV (env), NULL);

	theme = gtk_type_new (mtm_theme_get_type ());
	mtm_theme_construct_from_defaults (theme, env);

	return theme;
}

void
mtm_theme_construct (MtmTheme *theme, MtmEnv *env)
{
	g_return_if_fail (theme != NULL);
	g_return_if_fail (env != NULL);

	MTM_STATEFUL (theme)->env = env;
}

/**
 * mtm_ext_construct_extension_from_node:
 * @node: 
 * 
 * Given a xml node that represents an extension it creates the extension from it
 * 
 * Return Value: the newly created extension, NULL on error
 **/
static MtmExt *
mtm_theme_construct_extension_from_node (xmlNodePtr node, MtmEnv *env, const gchar *themedir)
{
	MtmExt *ext;
	gchar *type = NULL;
	gchar *file = NULL;
	gchar *tmp;
			
	tmp = xmlGetProp (node, "type");
	if (!tmp) {
		g_warning ("Could not construct extension from node because the "
				 "type node was not found\n");
		return NULL;
	}
	
	type = g_strdup (tmp);
	xmlFree (tmp);
			
	tmp = xmlGetProp (node, "file");
	if (tmp && !(strcmp (tmp, "") == 0))
	{
		file = g_strconcat (themedir, "/", type, "/", tmp, NULL);
		xmlFree (tmp);
	}
	else
	{
		file = NULL;
	}

	ext = mtm_ext_new_with_type (env, type);
	g_free (type);
	ext->file = file;
	
	if (node->childs && ext->handler)
	{
		gtk_object_set (GTK_OBJECT (ext->handler),
					 "ext_context", ext, NULL);
		parse_args (ext, node->childs);
	}
	
	return ext;
}


void
mtm_theme_construct_from_file (MtmTheme *theme, MtmEnv *env, const gchar *filename)
{
	xmlDocPtr doc;
	xmlNodePtr node;
	gchar *realfile;

	g_return_if_fail (filename != NULL);

	MTM_STATEFUL (theme)->env = env;
	
	realfile = g_strconcat (filename, "/theme.xml", NULL);
	doc = xmlParseFile (realfile);
	g_free (realfile);
	g_return_if_fail (doc != NULL);
	
	/* MEMLEAK */
	node = doc->root;
	g_return_if_fail (node != NULL);
	g_return_if_fail (strcasecmp (node->childs->name, "metatheme") != 0);

	theme->priv->exts = NULL;
	
	node = node->childs;
	
	while (node != NULL)
	{
		if (strcasecmp (node->name, "name") == 0)
		{
			char *tmp = xmlNodeListGetString (doc, node->childs, 1);
			theme->name = g_strdup (tmp);
			xmlFree (tmp);
		}
		else if (strcasecmp (node->name, "desc") == 0)
		{
			char *tmp = xmlNodeListGetString (doc, node->childs, 1);
			theme->desc = g_strdup (tmp);
			xmlFree (tmp);
		}
		else if (strcasecmp (node->name, "preview") == 0)
		{
			char *tmp = xmlNodeListGetString (doc, node->childs, 1);
			theme->preview = g_strconcat (filename, "/", tmp, NULL);
			xmlFree (tmp);
		}
		else if (strcasecmp (node->name, "author") == 0)
		{
			char *tmp = xmlNodeListGetString (doc, node->childs, 1);
			char *email = xmlGetProp (node, "email");
			theme->author= g_strdup (tmp);
			xmlFree (tmp);
			if (email)
			{
				theme->author_email = g_strdup (email);
				xmlFree (email);
			}
		}
		else if (strcasecmp (node->name, "ext") == 0)
		{
			MtmExt *ext;

			ext = mtm_theme_construct_extension_from_node (node, env, filename);

			if (ext != NULL)
				mtm_theme_add_ext (theme, ext);
		}
		node = node->next;
	}
	xmlFreeDoc (doc);
	
	/* We do this at the end to make sure the theme file was ok 
	 * If we have bombed out earlier, filename will be NULL and
	 * ::new_from_file will know something bad happened */
	theme->filename = g_strdup (filename);
}

void
mtm_theme_construct_from_defaults (MtmTheme *theme, MtmEnv *env)
{
	GList *l;
	
	g_return_if_fail (MTM_IS_THEME (theme));
	g_return_if_fail (MTM_IS_ENV (env));

	MTM_STATEFUL (theme)->env = env;
	
	for (l = mtm_env_get_all_ext_handlers (env); l != NULL; l = l->next)
	{
		MtmHandler *handler = MTM_HANDLER (l->data);
		MtmExt *ext = mtm_ext_new_from_default (env, handler->key);

		mtm_theme_add_ext (theme, ext);
	}
}

MtmResult
mtm_theme_save_as (MtmTheme *theme, gchar *filename)
{
	char *indent = "\t";
	FILE *file;
	GList *exts;
	MtmExt *ext;
	gchar *realfile;

	g_return_val_if_fail (theme != NULL, MTM_GENERAL_ERROR);
	g_return_val_if_fail (filename != NULL, MTM_GENERAL_ERROR);

	mtm_check_dir (filename);

	realfile = g_strconcat (filename, "/theme.xml", NULL);
	file = fopen (realfile, "w");
	g_free (realfile);
	g_return_val_if_fail (file != NULL, MTM_NO_ACCESS);
	
	fprintf (file, "<?xml version=\"1.0\"?>\n");
	fprintf (file, "<metatheme>\n");
	if (theme->name)
		fprintf (file, "%s<name>%s</name>\n", indent, theme->name);
	if (theme->desc)
		fprintf (file, "%s<desc>%s</desc>\n", indent, theme->desc);
	if (theme->preview)
	{
		gchar *tofile = g_strconcat (filename, "/", g_basename (theme->preview), NULL);
		
		fprintf (file, "%s<preview>%s</preview>\n", indent, g_basename (theme->preview));
		mtm_copy_file (theme->preview, tofile);
		g_free (tofile);
	}
	if (theme->author)
	{
		fprintf (file, "%s<author", indent);
		if (theme->author_email)
			fprintf (file, " email=\"%s\"", theme->author_email);
		fprintf (file, ">%s</author>", theme->author);
	}
	
	exts = mtm_theme_get_exts (theme);
	
	while (exts)
	{
		ext = MTM_EXT (exts->data);
		fprintf (file, "%s<ext type=\"%s\"", indent, ext->type);
		if (ext->file)
		{
			gchar *tmp;
			
			if (mtm_file_is_targz (ext->file))
				tmp = mtm_strip_ext (g_basename (ext->file), ".tar.gz");
			else
				tmp = g_strdup (g_basename (ext->file));
			fprintf (file, " file=\"%s\"", tmp);
			
			g_free (tmp);
		}
		
		if (ext->handler && ext->handler->subclass_args)
		{
			char *tmp = g_strconcat (indent, "\t", NULL);
			guint n, i;
			GtkArg *args;
			GtkArg tmparg;
			args = ext->handler->subclass_args;
			n = ext->handler->n_subclass_args;
			
			fprintf (file, ">\n");

			gtk_object_set (GTK_OBJECT (ext->handler),
				"ext_context", ext, NULL);

			for (i = 0; i < n; i++)
			{
				fprintf (file, "%s<arg name=\"%s\" ",
					tmp,
					mtm_ext_handler_get_arg_name (ext->handler, args[i].name));
				tmparg.name = args[i].name;
				tmparg.type = args[i].type;
				gtk_object_getv (GTK_OBJECT (ext->handler),
					1, &tmparg);
				mtm_ext_handler_print_arg (ext->handler,
					file, &tmparg);
				fprintf (file, "/>\n");
			}
			g_free (tmp);
			fprintf (file, "%s</ext>\n", indent);
		}
		else
			fprintf (file, "/>\n");

		/* Save the actual extension */
		if (ext->handler && ext->handler->save && ext->file)
		{
			gchar *todir = g_strconcat (filename, "/", ext->type, NULL);
			gchar *savefile = g_strconcat (todir, "/", g_basename (ext->file), NULL);
			
			ext->handler->save (ext->handler, ext, todir);
			g_free (todir);
			g_free (savefile);
		}
		
		exts = exts->next;
	}

	fprintf (file, "</metatheme>\n");
	
	fclose (file);
	
	return MTM_OK;
}

MtmResult
mtm_theme_save (MtmTheme *theme)
{
	g_return_val_if_fail (theme != NULL, MTM_GENERAL_ERROR);

	return mtm_theme_save_as (theme, theme->filename);
}

static void
mtm_theme_set_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	MtmTheme *theme = MTM_THEME (object);

	switch (arg_id)
	{
		case ARG_DESC:
			mtm_theme_set_desc (theme, GTK_VALUE_STRING (*arg));
			break;
		case ARG_NAME:
			mtm_theme_set_name (theme, GTK_VALUE_STRING (*arg));
			break;
		case ARG_FILENAME:
			mtm_theme_set_filename (theme, GTK_VALUE_STRING (*arg));
			break;
		case ARG_PREVIEW:
			mtm_theme_set_preview (theme, GTK_VALUE_STRING (*arg));
			break;
		case ARG_AUTHOR:
			mtm_theme_set_author (theme, GTK_VALUE_STRING (*arg));
			break;
		case ARG_AUTHOR_EMAIL:
			mtm_theme_set_author_email (theme, GTK_VALUE_STRING (*arg));
			break;
	}
}

static void
mtm_theme_get_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	MtmTheme *theme = MTM_THEME (object);

	switch (arg_id)
	{
		case ARG_DESC:
			GTK_VALUE_STRING (*arg) = (gchar *) mtm_theme_get_desc (theme);
			break;
		case ARG_NAME:
			GTK_VALUE_STRING (*arg) = (gchar *) mtm_theme_get_name (theme);
			break;
		case ARG_FILENAME:
			GTK_VALUE_STRING (*arg) = (gchar *) mtm_theme_get_filename (theme);;
			break;
		case ARG_PREVIEW:
			GTK_VALUE_STRING (*arg) = (gchar *) mtm_theme_get_preview (theme);;
			break;
		case ARG_AUTHOR:
			GTK_VALUE_STRING (*arg) = (gchar *) mtm_theme_get_author (theme);;
			break;
		case ARG_AUTHOR_EMAIL:
			GTK_VALUE_STRING (*arg) = (gchar *) mtm_theme_get_author_email (theme);;
			break;
	}
}

const gchar *
mtm_theme_get_desc (MtmTheme *theme)
{
	g_return_val_if_fail (MTM_IS_THEME (theme), NULL);

	if (!theme->desc)
		return "Could not get theme description";
	    
	return theme->desc;
}

void
mtm_theme_set_desc (MtmTheme *theme, gchar *desc)
{
	g_return_if_fail (MTM_IS_THEME (theme));

	if (theme->desc)
		g_free (theme->desc);

	theme->desc = g_strdup (desc);
}

const gchar *
mtm_theme_get_name (MtmTheme *theme)
{
	g_return_val_if_fail (MTM_IS_THEME (theme), NULL);
	
	return theme->name;
}

void
mtm_theme_set_name (MtmTheme *theme, gchar *name)
{
	g_return_if_fail (MTM_IS_THEME (theme));
	
	if (theme->name)
		g_free (theme->name);
	theme->name = g_strdup (name);
}

const gchar *
mtm_theme_get_filename (MtmTheme *theme)
{
	g_return_val_if_fail (MTM_IS_THEME (theme), NULL);

	return theme->filename;
}

void
mtm_theme_set_filename (MtmTheme *theme, gchar *filename)
{
	int len;
	
	g_return_if_fail (MTM_IS_THEME (theme));
	
	if (theme->filename)
		g_free (theme->filename);

	if (!filename)
	{
		theme->filename = NULL;
		return;
	}
	
	theme->filename = g_strdup (filename);
	
	len = strlen (theme->filename);
	if (theme->filename[len - 1] == '/')
		theme->filename[len - 1] = '\0';
}

const gchar *
mtm_theme_get_preview (MtmTheme *theme)
{
	g_return_val_if_fail (MTM_IS_THEME (theme), NULL);

	return theme->preview;
}

void
mtm_theme_set_preview (MtmTheme *theme, gchar *preview)
{
	g_return_if_fail (MTM_IS_THEME (theme));

	if (theme->preview)
		g_free (theme->preview);
	theme->preview = g_strdup (preview);
}

const gchar*
mtm_theme_get_author (MtmTheme *theme)
{
	g_return_val_if_fail (MTM_IS_THEME (theme), NULL);

	return theme->author;
}

void
mtm_theme_set_author (MtmTheme *theme, gchar *author)
{
	g_return_if_fail (MTM_IS_THEME (theme));
	
	if (theme->author)
		g_free (theme->author);
	theme->author = g_strdup (author); 
}

const gchar*
mtm_theme_get_author_email (MtmTheme *theme)
{
	g_return_val_if_fail (MTM_IS_THEME (theme), NULL);
	
	return theme->author_email;
}

void mtm_theme_set_author_email (MtmTheme *theme, gchar *author_email)
{
	g_return_if_fail (MTM_IS_THEME (theme));
	
	if (theme->author_email)
		g_free (theme->author_email);
	theme->author = g_strdup (author_email); 
}

static void
parse_args (MtmExt *ext, xmlNodePtr node)
{
	GdkColor color;
	GtkType type;
	GtkArg arg;
	gchar *typename;
	gchar *tmp;

	while (node != NULL)
	{
		if (strcasecmp (node->name, "arg") == 0)
		{
			char *xmltmp;
			
			typename = xmlGetProp (node, "type");
			type = gtk_type_from_name (typename);
			if (type == GTK_TYPE_INVALID)
			{
				xmlFree (typename);
				node = node->next;
				continue;
			}
			arg.type = type;
			xmltmp = xmlGetProp (node, "name");
			arg.name = g_strdup (xmltmp);
			xmlFree (xmltmp);
			tmp = xmlGetProp (node, "value");
			if (type == GTK_TYPE_STRING)
				GTK_VALUE_STRING (arg) = tmp;
			else if (type == GTK_TYPE_BOOL)
				GTK_VALUE_BOOL (arg) = (tmp[0] != '0');
			else if (type == GTK_TYPE_GDK_COLOR)
			{
				gdk_color_parse (tmp, &color);
				GTK_VALUE_BOXED (arg) = &color;
			}
			else
				GTK_VALUE_ENUM (arg) = atoi (tmp);

			gtk_object_setv (GTK_OBJECT (ext->handler), 1, &arg);
			g_free (arg.name);
			xmlFree (tmp);
			xmlFree (typename);
		}
		node = node->next;
	}
}

gint
mtm_theme_compare (gconstpointer a, gconstpointer b)
{
	const MtmTheme *theme1, *theme2;
	gchar *str1, *str2;
	
	theme1 = a;
	theme2 = b;

	if (theme1->name)
		str1 = theme1->name;
	else
		str1 = g_basename (theme1->filename);

	if (theme2->name)
		str2 = theme2->name;
	else
		str2 = g_basename (theme2->filename);

	return g_strcasecmp (str1, str2);
}

gchar *
mtm_theme_dup_name (MtmTheme *theme)
{
	g_return_val_if_fail (MTM_IS_THEME (theme), g_strdup("Error"));

	if (!theme->name)
		return g_strdup (g_basename (theme->filename));

	return g_strdup (theme->name);
}
