/* value.c
 *
 * Copyright (C) 2001 - 2002 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


/*
 * This module represents QueryField of type QUERY_FIELD_FIELD:
 * a field of a QueryView (DbTable (table or view) or Query).
 */

#include <config.h>
#include "../query.h"
#include "../database.h"
#include "../query-field-private.h"
#include "../server-access.h"
#include "../item-selector.h"
#include "../data-entry.h"

static void        q_init            (QueryField *qf);
static void        q_finalize        (QueryField *qf);
static void        q_deactivate      (QueryField *qf);
static void        q_activate        (QueryField *qf);
static GtkWidget * q_get_edit_widget (QueryField *qf);
static GtkWidget * q_get_sel_widget  (QueryField *qf, QueryField *sel_qf, 
				      GCallback callback, gpointer data);
static gchar     * q_render_as_sql   (QueryField *qf, GSList * missing_values);
static xmlNodePtr  q_render_as_xml   (QueryField *qf, GSList * missing_values);
static gchar     * q_render_as_string(QueryField *qf, GSList * missing_values);
static void        q_save_to_xml     (QueryField *qf, xmlNodePtr node);
static void        q_load_from_xml   (QueryField *qf, xmlNodePtr node);
static void        q_copy_other_field(QueryField *qf, QueryField *other);
static gboolean    q_is_equal_to     (QueryField *qf, QueryField *other);
static GSList    * q_get_monitored_objects (QueryField *qf);
static void        q_replace_comp    (QueryField *qf, gint ref, GObject   *old, GObject   *new);

/* Weak ref from the refering object */
static void        q_obj_destroyed_cb (QueryField *qf, GObject *obj);

typedef struct {
	/* the type of value */
	ServerDataType *value_type;
	DataHandler    *dh;
	gboolean        weak_ref_dt; /* TRUE if the weak ref on the data type is no more */

	/* value, if any */
	GdaValue       *value;

	/* still ask for a value */
	gboolean        is_parameter;

	/* XML field name if we did not find it in the first place */
	gchar          *type_name;
} private_data;

#define QF_PRIVATE_DATA(qf) ((private_data *) qf->priv->private_data)

QueryFieldIface * 
query_field_value_get_iface()
{
	QueryFieldIface *iface;

	iface = g_new0 (QueryFieldIface, 1);
	iface->field_type = QUERY_FIELD_VALUE;
	iface->name = "value";
	iface->pretty_name = _("Constant value or parameter");
	iface->init = q_init;
	iface->destroy = q_finalize;
	iface->deactivate = q_deactivate;
	iface->activate = q_activate;
	iface->get_edit_widget = q_get_edit_widget;
	iface->get_sel_widget = q_get_sel_widget;
	iface->render_as_sql = q_render_as_sql;
	iface->render_as_xml = q_render_as_xml;
	iface->render_as_string = q_render_as_string;
	iface->save_to_xml = q_save_to_xml;
	iface->load_from_xml = q_load_from_xml;
	iface->copy_other_field = q_copy_other_field;
	iface->is_equal_to = q_is_equal_to;
	iface->get_monitored_objects = q_get_monitored_objects;
	iface->replace_comp = q_replace_comp;

	return iface;
}

static void        
q_init            (QueryField *qf)
{
	private_data *data;
	data = g_new0 (private_data, 1);
	qf->priv->private_data = (gpointer) data;
	QF_PRIVATE_DATA(qf)->value_type = NULL;
	QF_PRIVATE_DATA(qf)->dh = NULL;
	QF_PRIVATE_DATA(qf)->type_name = NULL;
	QF_PRIVATE_DATA(qf)->value = NULL;
	QF_PRIVATE_DATA(qf)->weak_ref_dt = FALSE;
	QF_PRIVATE_DATA(qf)->is_parameter = FALSE;
}

static void        
q_finalize         (QueryField *qf)
{
	query_field_deactivate (qf);
	if (qf->priv->private_data) {
		if (QF_PRIVATE_DATA(qf)->type_name)
			g_free (QF_PRIVATE_DATA(qf)->type_name);
		g_free (qf->priv->private_data);
		qf->priv->private_data = NULL;
	}
}


static gchar * qf_get_xml_id (QueryField *qf);
static void
q_deactivate      (QueryField *qf)
{
	if (! qf->priv->activated)
		return;

	/* This function disconnects any event handler from any object
	   this QueryField wants to receive events from.
	   Here we disconnect from the data type if we are connected */

	if (QF_PRIVATE_DATA(qf)->weak_ref_dt) {
		g_object_weak_unref (G_OBJECT (QF_PRIVATE_DATA(qf)->value_type),
				     (GWeakNotify) q_obj_destroyed_cb, qf);
		QF_PRIVATE_DATA(qf)->weak_ref_dt = FALSE;
	}

	if (QF_PRIVATE_DATA(qf)->type_name) {
		g_free (QF_PRIVATE_DATA(qf)->type_name);
		QF_PRIVATE_DATA(qf)->type_name = NULL;
	}

	QF_PRIVATE_DATA(qf)->type_name = qf_get_xml_id (qf);
	QF_PRIVATE_DATA(qf)->value_type = NULL;
	
	query_field_set_activated (qf, FALSE);
}

static gchar *
qf_get_xml_id (QueryField *qf)
{
	gchar *str = NULL;

	if (QF_PRIVATE_DATA(qf)->value_type) 
		str = server_data_type_get_xml_id (QF_PRIVATE_DATA(qf)->value_type);

	return str;
}

static void value_set_data_type (QueryField *qf, ServerDataType *dt, gboolean try_act);
static void
q_activate        (QueryField *qf)
{
	/* this function gets references to any object this QueryField wants to 
	   receive events from. */

	ServerDataType *value_type;
	
	if (qf->priv->activated)
		return;

	value_type = QF_PRIVATE_DATA(qf)->value_type;
	
	if (!value_type && QF_PRIVATE_DATA(qf)->type_name) 
		value_type = server_data_type_get_from_xml_id (qf->priv->query->conf->srv, 
							       QF_PRIVATE_DATA(qf)->type_name);


	if (value_type) {
		value_set_data_type (qf, value_type, FALSE);

		/* now it's activated */ 
		query_field_set_activated (qf, TRUE);
	}
}

static void        
q_obj_destroyed_cb (QueryField *qf, GObject *obj)
{
	QF_PRIVATE_DATA(qf)->weak_ref_dt = FALSE;

	q_deactivate (qf);
	/* if the data type disappear, then do nothing, just remain non activated */
	/*g_object_unref (G_OBJECT (qf));*/
}

static gchar *
get_data_type_name (ServerDataType *dt) {
	g_return_val_if_fail (dt && IS_SERVER_DATA_TYPE (dt), NULL);
	return g_strdup (dt->sqlname);
}

static void data_type_changed_cb (ItemSelector *is, ServerDataType *dt, QueryField *qf);
static void param_state_toggled_cb (GtkToggleButton *button, QueryField *qf);
static GtkWidget * 
q_get_edit_widget (QueryField *qf)
{
	GtkWidget *wid, *vb, *table, *vb2, *is, *param;
	gpointer selection;

	vb = gtk_vbox_new (FALSE, GNOME_PAD/2.);
	gtk_container_set_border_width (GTK_CONTAINER (vb), 0);

	table = gtk_table_new (3, 2, FALSE);
	gtk_container_set_border_width (GTK_CONTAINER (table), 0);
	gtk_box_pack_start (GTK_BOX (vb), table, TRUE, TRUE, GNOME_PAD/2.);

	wid = gtk_label_new (_("Type:"));
	gtk_misc_set_alignment (GTK_MISC (wid), 0, 0);
	gtk_table_attach (GTK_TABLE (table), wid, 0, 1, 0, 1, GTK_FILL, 0, 0, 0);

	is = item_selector_new (ITEM_SELECTOR_MENU);
	item_selector_set_dyn_content (ITEM_SELECTOR (is), G_OBJECT (qf->priv->query->conf->srv),
				       "data_type_added", "data_type_removed",
				       qf->priv->query->conf->srv->data_types,
				       (ItemSelectorFunc) get_data_type_name);
	g_signal_connect (G_OBJECT (is), "selection_changed",
			  G_CALLBACK (data_type_changed_cb), qf);

	gtk_table_attach (GTK_TABLE (table), is, 1, 2, 0, 1,
			  GTK_FILL | GTK_EXPAND | GTK_SHRINK, 0, GNOME_PAD/2., 0);

	wid = gtk_label_new (_("Parameter:"));
	gtk_misc_set_alignment (GTK_MISC (wid), 0, 0);
	gtk_table_attach (GTK_TABLE (table), wid, 0, 1, 1, 2, GTK_FILL, 0, 0, 0);

	param = gtk_check_button_new ();
	gtk_table_attach (GTK_TABLE (table), param, 1, 2, 1, 2,
			  GTK_FILL | GTK_EXPAND | GTK_SHRINK, 0, GNOME_PAD/2., GNOME_PAD/2.);
	g_signal_connect (G_OBJECT (param), "toggled",
			  G_CALLBACK (param_state_toggled_cb), qf);


	wid = gtk_label_new (_("Value:"));
	gtk_misc_set_alignment (GTK_MISC (wid), 0, 0);
	gtk_misc_set_padding (GTK_MISC (wid), 0, GNOME_PAD/2.);
	gtk_table_attach (GTK_TABLE (table), wid, 0, 1, 2, 3, 
			  GTK_FILL, GTK_FILL | GTK_EXPAND | GTK_SHRINK, 
			  GNOME_PAD/2., GNOME_PAD/2.);

	vb2 = gtk_vbox_new (FALSE, 0);
	gtk_table_attach_defaults (GTK_TABLE (table), vb2, 1, 2, 2, 3);
	g_object_set_data (G_OBJECT (is), "value_vbox", vb2);

	wid = gtk_label_new (_("To enter a value,\nfirst select a data type!"));
	gtk_box_pack_start (GTK_BOX (vb2), wid, FALSE, TRUE, 0);
	g_object_set_data (G_OBJECT (vb2), "msg", wid);

	gtk_widget_show_all (table);

	/* correct initial setting */
	if (QF_PRIVATE_DATA(qf)->value_type) {
		selection = QF_PRIVATE_DATA(qf)->value_type;
		item_selector_set_selection_ptr (ITEM_SELECTOR (is), selection);
	}
	else
		selection = item_selector_get_selection (ITEM_SELECTOR (is));
	g_signal_emit_by_name (G_OBJECT (is), "selection_changed", selection);
	

	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (param), QF_PRIVATE_DATA(qf)->is_parameter);	

	return vb;
}

static void de_value_changed_cb (DataEntry *de, QueryField *qf);
static void 
data_type_changed_cb (ItemSelector *is, ServerDataType *dt, QueryField *qf)
{
	GtkWidget *value_vbox, *de, *msg;

	value_vbox = g_object_get_data (G_OBJECT (is), "value_vbox");
	msg = g_object_get_data (G_OBJECT (value_vbox), "msg");
	de = g_object_get_data (G_OBJECT (value_vbox), "de");

	if (dt && (dt == QF_PRIVATE_DATA(qf)->value_type) && de)
		return;

	/* empty the vbox for the value */
	if (de) {
		gtk_widget_destroy (de);
		g_object_set_data (G_OBJECT (value_vbox), "de", NULL);
	}
	
	value_set_data_type (qf, dt, TRUE);

	if (dt) {
		GtkWidget *de;

		de = data_handler_get_widget_from_value (QF_PRIVATE_DATA(qf)->dh, NULL);
		data_entry_set_is_default_allowed (DATA_ENTRY (de), FALSE);
		data_entry_set_is_null_allowed (DATA_ENTRY (de), TRUE);
		data_entry_set_show_actions_menu (DATA_ENTRY (de), TRUE);
		
		data_entry_update_with_value (DATA_ENTRY (de), QF_PRIVATE_DATA(qf)->value, TRUE);

		gtk_box_pack_start (GTK_BOX (value_vbox), de, 
				    data_handler_get_expand (QF_PRIVATE_DATA(qf)->dh), TRUE, 0);
		g_object_set_data (G_OBJECT (value_vbox), "de", de);

		gtk_widget_show (de);
		gtk_widget_hide (msg);
		
		g_signal_connect (G_OBJECT (de), "contents_modified",
				  G_CALLBACK (de_value_changed_cb), qf);

		/* force the display update */
		de_value_changed_cb (DATA_ENTRY (de), qf);
		q_activate (qf);
	}
	else 
		gtk_widget_show (msg);
}

static void 
param_state_toggled_cb (GtkToggleButton *button, QueryField *qf)
{
	if (gtk_toggle_button_get_active (button) != QF_PRIVATE_DATA(qf)->is_parameter)
		QF_PRIVATE_DATA(qf)->is_parameter = gtk_toggle_button_get_active (button);
}

static void
value_set_data_type (QueryField *qf, ServerDataType *dt, gboolean try_act)
{
	if (try_act)
		q_deactivate (qf);

	if (dt) {
		DataHandler *dh;

		if (QF_PRIVATE_DATA(qf)->weak_ref_dt) {
			g_object_weak_unref (G_OBJECT (QF_PRIVATE_DATA(qf)->value_type),
					     (GWeakNotify) q_obj_destroyed_cb, qf);
			QF_PRIVATE_DATA(qf)->weak_ref_dt = FALSE;
		}

		dh = server_access_get_object_handler (qf->priv->query->conf->srv, G_OBJECT (dt));
		QF_PRIVATE_DATA(qf)->value_type = dt;
		QF_PRIVATE_DATA(qf)->dh = dh;

		/* weak ref the data type */
		g_object_weak_ref (G_OBJECT (QF_PRIVATE_DATA(qf)->value_type),
				   (GWeakNotify) q_obj_destroyed_cb, qf);
		QF_PRIVATE_DATA(qf)->weak_ref_dt = TRUE;

		/* check that the stored value is of the same type, otherwise set it to NULL */
		if (QF_PRIVATE_DATA(qf)->value &&
		    (dt->gda_type != gda_value_get_type (QF_PRIVATE_DATA(qf)->value))) {
			query_field_value_set_value (qf, NULL);
			g_print ("Value incompatible with DataType\n");
		}
	}
	else {
		QF_PRIVATE_DATA(qf)->value_type = NULL;
		QF_PRIVATE_DATA(qf)->dh = NULL;
		query_field_value_set_value (qf, NULL);
	}

	if (try_act)
		q_activate (qf);
}

void
query_field_value_set_value (QueryField *qf, const GdaValue *value)
{
	g_return_if_fail (qf && IS_QUERY_FIELD (qf));

	if ((value && QF_PRIVATE_DATA(qf)->value && !gda_value_compare (value, QF_PRIVATE_DATA(qf)->value)) ||
	    (! value && !QF_PRIVATE_DATA(qf)->value))
		return;

	if (QF_PRIVATE_DATA(qf)->value) {
		gda_value_free (QF_PRIVATE_DATA(qf)->value);
		QF_PRIVATE_DATA(qf)->value = NULL;
	}

	if (value) 
		QF_PRIVATE_DATA(qf)->value = gda_value_copy (value);
}


static void
de_value_changed_cb (DataEntry *de, QueryField *qf)
{
	if (QF_PRIVATE_DATA(qf)->value)
		gda_value_free (QF_PRIVATE_DATA(qf)->value);

	QF_PRIVATE_DATA(qf)->value = data_entry_get_gdavalue (de);

#ifdef debug_signal
	g_print (">> 'FIELD_MODIFIED' from de_value_changed_cb\n");
#endif
	g_signal_emit_by_name (G_OBJECT (qf), "field_modified");
#ifdef debug_signal
	g_print ("<< 'FIELD_MODIFIED' from de_value_changed_cb\n");
#endif	

}





static GtkWidget * 
q_get_sel_widget  (QueryField *qf, QueryField *sel_qf, GCallback callback, gpointer data)
{
	GtkWidget *button;
	gchar *str = NULL;

	if (qf->priv->activated) {
		if (QF_PRIVATE_DATA(qf)->value)
			str = q_render_as_sql (qf, NULL);
	}

	if (!str)
		str = g_strdup ("NULL");

	button = gtk_button_new_with_label (str);	
	g_free (str);
	g_signal_connect (G_OBJECT (button), "clicked", callback, data);
	g_object_set_data (G_OBJECT (button), "qf", qf);

	/* Set the "QF_obj_emit_sig" attribute so that we can attach attributes to that button
	   which will be transmitted when the user clicks on it */
	g_object_set_data (G_OBJECT (button), "QF_obj_emit_sig", button);

	/* state */
	if (sel_qf == qf) 
		gtk_widget_set_sensitive (button, FALSE);

	return button;
}



static gchar * 
q_render_as_sql   (QueryField *qf, GSList * missing_values)
{
	gchar *str = NULL;

	if (QF_PRIVATE_DATA(qf)->value && !gda_value_is_null (QF_PRIVATE_DATA(qf)->value)) {
		if (QF_PRIVATE_DATA(qf)->dh) 
			str = data_handler_get_sql_from_value (QF_PRIVATE_DATA(qf)->dh, 
							       QF_PRIVATE_DATA(qf)->value);
		else 
			g_warning ("Value without a Data Handler!");
	}

	return str;
}

static xmlNodePtr  
q_render_as_xml   (QueryField *qf, GSList * missing_values)
{
	return NULL;
}

static gchar * 
q_render_as_string(QueryField *qf, GSList * missing_values)
{
	GString *string;
	gchar *str = NULL;
	
	if (QF_PRIVATE_DATA(qf)->is_parameter)
		string = g_string_new (_("Parameter: "));
	else
		string = g_string_new (_("Constant value: "));
       
	str = q_render_as_sql (qf, NULL);	
	if (!str)
		str = g_strdup ("NULL");
	g_string_append (string, str);
	g_free (str);

	str = string->str;
	g_string_free (string, FALSE);

	return str;
}

static void  
q_save_to_xml     (QueryField *qf, xmlNodePtr node)
{
	if (qf->priv->activated) {
		gchar *str;
	
		/* node object ref */
		str = server_data_type_get_xml_id (QF_PRIVATE_DATA(qf)->value_type);
		xmlSetProp (node, "object", str);
		g_free (str);
		
		if (QF_PRIVATE_DATA(qf)->value) {
			xmlNodePtr valnode = gda_value_to_xml (QF_PRIVATE_DATA(qf)->value);
			xmlAddChild (node, valnode);
		}

		xmlSetProp (node, "type", "value");
	}
	else
		g_warning ("QueryField not activated; can't save\n");
}

static void        
q_load_from_xml   (QueryField *qf, xmlNodePtr node)
{
	query_field_deactivate (qf);
	
	/* check we have a QueryField */
	if (!strcmp (node->name, "QueryField")) {
		gchar *str;
		xmlNodePtr subnode;

		str = xmlGetProp (node, "type");
		if (!str || (str && strcmp (str, "value"))) {
			if (str) g_free (str);
			return;
		}

		str = xmlGetProp (node, "object");
		if (str) {
			if (QF_PRIVATE_DATA(qf)->type_name)
				g_free (QF_PRIVATE_DATA(qf)->type_name);
			QF_PRIVATE_DATA(qf)->type_name = str;
		}

		subnode = node->xmlChildrenNode;
		while (subnode) {
			if (!strcmp (subnode->name, "value")) 
				QF_PRIVATE_DATA(qf)->value = gda_value_new_from_xml (subnode);
			subnode = subnode->next;
		}
		query_field_activate (qf);
	}
}

static void        
q_copy_other_field(QueryField *qf, QueryField *other)
{
	/* we can't call q_destroy(qf) because we don't know what the type
	   of QueryField it was before. This is normally done by the
	   QueryField object before the copy */

	if (QF_PRIVATE_DATA(other)->value_type) {
		value_set_data_type (qf, QF_PRIVATE_DATA(other)->value_type, FALSE);
		query_field_activate (qf);
	}
	else {
		if (QF_PRIVATE_DATA(other)->type_name) {
			QF_PRIVATE_DATA(qf)->type_name = g_strdup (QF_PRIVATE_DATA(other)->type_name);
			query_field_activate (qf);
		}
	}

	/* copy the real values */
	query_field_value_set_value (qf, QF_PRIVATE_DATA(other)->value);
	QF_PRIVATE_DATA(qf)->is_parameter = QF_PRIVATE_DATA(other)->is_parameter;
}

static gboolean
q_is_equal_to (QueryField *qf, QueryField *other)
{
	gboolean retval = TRUE;

	if (qf->priv->activated && other->priv->activated) {
		if ((QF_PRIVATE_DATA(qf)->value_type != QF_PRIVATE_DATA(other)->value_type))
			retval = FALSE;
		else {
			if (QF_PRIVATE_DATA(qf)->value && QF_PRIVATE_DATA(other)->value) {
				if (QF_PRIVATE_DATA(qf)->value->type == QF_PRIVATE_DATA(other)->value->type) {
					if (gda_value_compare (QF_PRIVATE_DATA(qf)->value, 
							       QF_PRIVATE_DATA(other)->value))
						retval = FALSE;
				}
				else
					retval = FALSE;
			}
			else {
				if ((!QF_PRIVATE_DATA(qf)->value && QF_PRIVATE_DATA(other)->value) ||
				    (QF_PRIVATE_DATA(qf)->value && !QF_PRIVATE_DATA(other)->value))
					retval = FALSE;
			}
		}
		
		if ((QF_PRIVATE_DATA(qf)->is_parameter != QF_PRIVATE_DATA(other)->is_parameter))
			retval = FALSE;
	}

	return retval;
}


static GSList *
q_get_monitored_objects (QueryField *qf)
{
	GSList *list = NULL;

	if (qf->priv->activated) 
		list = g_slist_prepend (NULL, QF_PRIVATE_DATA(qf)->value_type);

	return list;
}

static void
q_replace_comp (QueryField *qf, gint ref, GObject   *old, GObject   *new)
{
	/* no reference to other QueryFields */
	return;
}


/* 
 * 
 * QueryField object's different implementations
 * 
 *
 */


ServerDataType  * 
query_field_value_get_data_type (QueryField *qf)
{
	g_assert (qf && IS_QUERY_FIELD (qf));
	g_assert (qf->priv->field_type == QUERY_FIELD_VALUE);

	if (! qf->priv->activated)
		return NULL;
	else 
		return QF_PRIVATE_DATA(qf)->value_type;
}

gboolean 
query_field_value_is_parameter (QueryField *qf)
{
	return QF_PRIVATE_DATA(qf)->is_parameter;
}

const GdaValue *
query_field_value_get_default_value (QueryField *qf)
{
	return QF_PRIVATE_DATA(qf)->value;
}
