/* ws-tables.c
 *
 * Copyright (C) 2004 - 2006 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "ws-tables.h"
#include "workspace-page.h"
#include <string.h>
#include <glib-object.h>
#include <gtk/gtk.h>
#include <libgnomedb/libgnomedb.h>
#include <libgda/gda-enums.h>
#include <glib/gi18n.h>
#include "utils.h"
#include "mg-extra-formgrid.h"
#include "mg-plugin-editor.h"

/* 
 * Main static functions 
 */
static void ws_tables_class_init (WsTablesClass * class);
static void ws_tables_init (WsTables *ws);
static void ws_tables_dispose (GObject *object);
static void ws_tables_finalize (GObject *object);

/* WorkspacePage interface */
static void            ws_tables_page_init       (WorkspacePageIface *iface);
static gchar          *ws_tables_get_name        (WorkspacePage *iface);
static gchar          *ws_tables_get_description (WorkspacePage *iface);
static GtkWidget      *ws_tables_get_sel_button  (WorkspacePage *iface);
static GtkWidget      *ws_tables_get_selector    (WorkspacePage *iface);
static GtkWidget      *ws_tables_get_work_area   (WorkspacePage *iface);
static GtkActionGroup *ws_tables_get_actions     (WorkspacePage *iface);
static const gchar    *ws_tables_get_actions_ui  (WorkspacePage *iface);

static void            opened_dialog_closed_cb   (GtkWidget *dlg, WsTables *ws);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* private structure */
struct _WsTablesPrivate
{
	GdaDict        *dict;
	GtkWidget      *selector;
	GtkWidget      *work_area;
	GtkActionGroup *actions;

	GtkWidget      *notebook;
	GtkWidget      *description;
	GtkWidget      *fields;
	GtkTextBuffer  *integ_descr;

	GObject        *sel_obj;
	GSList         *opened_dialogs;
};

GType
ws_tables_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (WsTablesClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) ws_tables_class_init,
			NULL,
			NULL,
			sizeof (WsTables),
			0,
			(GInstanceInitFunc) ws_tables_init
		};

		static const GInterfaceInfo workspace_page_info = {
			(GInterfaceInitFunc) ws_tables_page_init,
			NULL,
			NULL
		};

		type = g_type_register_static (G_TYPE_OBJECT, "WsTables", &info, 0);
		g_type_add_interface_static (type, WORKSPACE_PAGE_TYPE, &workspace_page_info);
	}
	return type;
}

static void 
ws_tables_page_init (WorkspacePageIface *iface)
{
	iface->get_name = ws_tables_get_name;
	iface->get_description = ws_tables_get_description;
	iface->get_sel_button = ws_tables_get_sel_button;
	iface->get_selector = ws_tables_get_selector;
	iface->get_work_area = ws_tables_get_work_area;
	iface->get_actions = ws_tables_get_actions;
	iface->get_actions_ui = ws_tables_get_actions_ui;
}

static void
ws_tables_class_init (WsTablesClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	object_class->dispose = ws_tables_dispose;
	object_class->finalize = ws_tables_finalize;
}

static void
ws_tables_init (WsTables *ws)
{
	ws->priv = g_new0 (WsTablesPrivate, 1);
}

static void ws_tables_initialize (WsTables *ws);

/**
 * ws_tables_new
 * @dict: a #GdaDict object
 *
 * Creates a new WsTables object
 *
 * Returns: the new object
 */
GObject*
ws_tables_new (GdaDict *dict)
{
	GObject  *obj;
	WsTables *ws;

	g_return_val_if_fail (dict && GDA_IS_DICT (dict), NULL);

	obj = g_object_new (WS_TABLES_TYPE, NULL);
	ws = WS_TABLES (obj);
	ws->priv->dict = dict;
	g_object_ref (G_OBJECT (dict));

	ws_tables_initialize (ws);

	return obj;
}


static void
ws_tables_dispose (GObject *object)
{
	WsTables *ws;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_WS_TABLES (object));

	ws = WS_TABLES (object);
	if (ws->priv) {
		while (ws->priv->opened_dialogs) 
			gtk_widget_destroy (GTK_WIDGET (ws->priv->opened_dialogs->data));

		if (ws->priv->selector) {
			gtk_widget_destroy (ws->priv->selector);
			ws->priv->selector = NULL;
		}

		if (ws->priv->work_area) {
			gtk_widget_destroy (ws->priv->work_area);
			ws->priv->work_area = NULL;
		}

		if (ws->priv->actions) {
			g_object_unref (ws->priv->actions);
			ws->priv->actions = NULL;
		}

		if (ws->priv->dict) {
			g_object_unref (G_OBJECT (ws->priv->dict));
			ws->priv->dict = NULL;
		}
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
ws_tables_finalize (GObject   * object)
{
	WsTables *ws;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_WS_TABLES (object));

	ws = WS_TABLES (object);
	if (ws->priv) {
		g_free (ws->priv);
		ws->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}

static void action_create_table_cb (GtkAction *action, WsTables *ws);
static void action_drop_table_cb (GtkAction *action, WsTables *ws);
static void action_rename_table_cb (GtkAction *action, WsTables *ws);
static void action_add_column_cb (GtkAction *action, WsTables *ws);
static void action_drop_column_cb (GtkAction *action, WsTables *ws);
static void action_table_contents_cb (GtkAction *action, WsTables *ws);

static GtkActionEntry ui_actions[] = {
	{ "Table", NULL, "_Table", NULL, "Table", NULL },
        { "CREATE_TABLE", GTK_STOCK_ADD, "_New table", NULL, "Create a new table", 
	  G_CALLBACK (action_create_table_cb)},
        { "DROP_TABLE", GTK_STOCK_DELETE, "_Delete", NULL, "Delete the selected table", 
	  G_CALLBACK (action_drop_table_cb)},
        { "RENAME_TABLE", NULL, "_Rename", NULL, "Rename the selected table", 
	  G_CALLBACK (action_rename_table_cb)},
        { "ADD_COLUMN", NULL, "_Add column", NULL, "Add a column to the selected table", 
	  G_CALLBACK (action_add_column_cb)},
        { "DROP_COLUMN", NULL, "_Delete column", NULL, "Delete a column from the selected table", 
	  G_CALLBACK (action_drop_column_cb)},
        { "TableContents", GTK_STOCK_EDIT, "_Contents", NULL, "Display the contents of the selected table", 
	  G_CALLBACK (action_table_contents_cb)},
};

static const gchar *ui_actions_info =
	"<ui>"
	"  <menubar name='MenuBar'>"
	"    <placeholder name='PageExtension'>"
        "      <menu name='Table' action='Table'>"
        "        <menuitem name='CREATE_TABLE' action= 'CREATE_TABLE'/>"
        "        <menuitem name='DROP_TABLE' action= 'DROP_TABLE'/>"
        "        <menuitem name='RENAME_TABLE' action= 'RENAME_TABLE'/>"
	"        <separator/>"
        "        <menuitem name='ADD_COLUMN' action= 'ADD_COLUMN'/>"
        "        <menuitem name='DROP_COLUMN' action= 'DROP_COLUMN'/>"
	"        <separator/>"
	"        <menuitem name='TableContents' action= 'TableContents'/>"
        "      </menu>"
	"    </placeholder>"
	"  </menubar>"
        "  <toolbar name='ToolBar'>"
        "    <toolitem action='CREATE_TABLE'/>"
        "    <toolitem action='DROP_TABLE'/>"
        "    <separator/>"
        "    <toolitem action='TableContents'/>"
        "  </toolbar>"
        "</ui>";


static void table_row_activated_cb (GnomeDbSelector *mgsel, GObject *sel_object, WsTables *ws);
static void table_selection_changed_cb (GnomeDbSelector *mgsel, GObject *sel_object, WsTables *ws);
static void field_selection_changed_cb (GnomeDbSelector *mgsel, GObject *sel_object, MgPluginEditor *editor);
static void table_updated_cb (GdaDictDatabase *db, GdaDictTable *table, WsTables *ws);
static void
ws_tables_initialize (WsTables *ws)
{
	GtkWidget *label, *vbox, *wid, *nb, *vp, *vbox2, *hbox, *sw, *exp;

	/* Selector part */
	wid = gnome_db_selector_new (ws->priv->dict, NULL,
			       GNOME_DB_SELECTOR_TABLES, GNOME_DB_SELECTOR_COLUMN_TYPE);
	ws->priv->selector = wid;
	g_signal_connect (G_OBJECT (ws->priv->selector), "selection_changed", 
			  G_CALLBACK (table_selection_changed_cb), ws);
	gtk_widget_set_size_request (wid, 200, -1);
	g_signal_connect (G_OBJECT (ws->priv->selector), "row_activated",
			  G_CALLBACK (table_row_activated_cb), ws);

	/* WorkArea part */
	vbox = gtk_vbox_new (FALSE, 5);
	ws->priv->work_area = vbox;
	
	nb = gtk_notebook_new ();
	gtk_notebook_set_show_border (GTK_NOTEBOOK (nb), FALSE);
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (nb), FALSE);
	gtk_box_pack_start (GTK_BOX (vbox), nb, TRUE, TRUE, 0);
	gtk_widget_show (nb);
	ws->priv->notebook = nb;
	
	label = gtk_label_new (_("Please select a table from the list on the left,\n"
				 "or create a new one using the 'Add' button below."));
	gtk_notebook_append_page (GTK_NOTEBOOK (nb), label, NULL);
	gtk_widget_show (label);
	
	vp = gtk_vpaned_new ();
	gtk_notebook_append_page (GTK_NOTEBOOK (nb), vp, NULL);
	gtk_widget_show (vp);

	vbox2 = gtk_vbox_new (FALSE, 5);
	gtk_paned_add1 (GTK_PANED (vp), vbox2);
	gtk_widget_show (vbox2);

	label = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL (label), _("<b>Description:</b>"));
	gtk_box_pack_start (GTK_BOX (vbox2), label, FALSE, FALSE, 0);
	gtk_widget_show (label);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);

	label = gtk_label_new (NULL);
	gtk_box_pack_start (GTK_BOX (vbox2), label, FALSE, FALSE, 0);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);
	gtk_widget_show (label);
	ws->priv->description = label;
	
	label = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL (label), _("<b>Fields:</b>"));
	gtk_box_pack_start (GTK_BOX (vbox2), label, FALSE, FALSE, 0);
	gtk_widget_show (label);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);

	hbox = gtk_hbox_new (FALSE, 0); /* HIG */
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 0);
	gtk_widget_show (hbox);
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);

	wid = gnome_db_selector_new (ws->priv->dict, NULL,
				     GNOME_DB_SELECTOR_FIELDS, 
				     GNOME_DB_SELECTOR_COLUMN_TYPE | GNOME_DB_SELECTOR_COLUMN_FIELD_LENGTH | 
				     GNOME_DB_SELECTOR_COLUMN_FIELD_NNUL |
				     GNOME_DB_SELECTOR_COLUMN_FIELD_DEFAULT | GNOME_DB_SELECTOR_COLUMN_COMMENTS);
	gtk_box_pack_start (GTK_BOX (hbox), wid, TRUE, TRUE, 0);
	gtk_widget_show (wid);
	ws->priv->fields = wid;

	vbox2 = gtk_vbox_new (FALSE, 5);
	gtk_paned_add2 (GTK_PANED (vp), vbox2);
	gtk_widget_show (vbox2);

	label = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL (label), _("<b>Constraints and integrity rules:</b>"));
	gtk_box_pack_start (GTK_BOX (vbox2), label, FALSE, FALSE, 0);
	gtk_widget_show (label);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);

	hbox = gtk_hbox_new (FALSE, 0); /* HIG */
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 0);
	gtk_widget_show (hbox);
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);

	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_box_pack_start (GTK_BOX (hbox), sw, TRUE, TRUE, 0);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw), GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_widget_show (sw);

	wid = gtk_text_view_new ();
	gtk_container_add (GTK_CONTAINER (sw), wid);
	gtk_text_view_set_left_margin (GTK_TEXT_VIEW (wid), 5);
	gtk_text_view_set_right_margin (GTK_TEXT_VIEW (wid), 5);
	gtk_text_view_set_editable (GTK_TEXT_VIEW (wid), FALSE);
	ws->priv->integ_descr = gtk_text_view_get_buffer (GTK_TEXT_VIEW (wid));
	gtk_text_buffer_set_text (ws->priv->integ_descr, "", -1);
	gtk_widget_show (wid);

	gtk_text_buffer_create_tag (ws->priv->integ_descr, "header",
				    "weight", PANGO_WEIGHT_BOLD,
				    "foreground", "red", NULL);

	gtk_text_buffer_create_tag (ws->priv->integ_descr, "section",
				    "weight", PANGO_WEIGHT_BOLD,
				    "foreground", "blue", NULL);

	g_signal_connect (G_OBJECT (gda_dict_get_database (ws->priv->dict)), "table_updated",
			  G_CALLBACK (table_updated_cb), ws);


	exp = gtk_expander_new (_("<b>Field's display specifications</b>"));
	gtk_expander_set_use_markup (GTK_EXPANDER (exp), TRUE);
	gtk_box_pack_start (GTK_BOX (vbox2), exp, FALSE, TRUE, 0);
	gtk_widget_show (exp);

	hbox = gtk_hbox_new (FALSE, 0); /* HIG */
	gtk_container_add (GTK_CONTAINER (exp), hbox);
	gtk_widget_show (hbox);
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);

	wid = mg_plugin_editor_new (ws->priv->dict);
	gtk_box_pack_start (GTK_BOX (hbox), wid, TRUE, TRUE, 0);
	gtk_widget_show (wid);
	g_signal_connect (G_OBJECT (ws->priv->fields), "selection-changed",
			  G_CALLBACK (field_selection_changed_cb), wid);

	/* actions part */
	ws->priv->actions = gtk_action_group_new ("Actions");
	gtk_action_group_add_actions (ws->priv->actions, ui_actions, G_N_ELEMENTS (ui_actions), ws);

	if (ws->priv->actions) {
		GtkAction *action;
		GdaConnection *cnc;
		GdaServerProvider *prov;

		cnc = gda_dict_get_connection (ws->priv->dict);
		prov = gda_connection_get_provider_obj (cnc);
		g_assert (prov);

		action = gtk_action_group_get_action (ws->priv->actions, "CREATE_TABLE");
		if (!gda_server_provider_supports_operation (prov, cnc, GDA_SERVER_OPERATION_CREATE_TABLE, NULL))
			gtk_action_set_visible (action, FALSE);

		action = gtk_action_group_get_action (ws->priv->actions, "DROP_TABLE");
		if (gda_server_provider_supports_operation (prov, cnc, GDA_SERVER_OPERATION_DROP_TABLE, NULL))
			gtk_action_set_sensitive (action, FALSE);
		else
			gtk_action_set_visible (action, FALSE);

		action = gtk_action_group_get_action (ws->priv->actions, "RENAME_TABLE");
		if (gda_server_provider_supports_operation (prov, cnc, GDA_SERVER_OPERATION_RENAME_TABLE, NULL))
			gtk_action_set_sensitive (action, FALSE);
		else
			gtk_action_set_visible (action, FALSE);

		action = gtk_action_group_get_action (ws->priv->actions, "ADD_COLUMN");
		if (gda_server_provider_supports_operation (prov, cnc, GDA_SERVER_OPERATION_ADD_COLUMN, NULL))
			gtk_action_set_sensitive (action, FALSE);
		else
			gtk_action_set_visible (action, FALSE);

		action = gtk_action_group_get_action (ws->priv->actions, "DROP_COLUMN");
		if (gda_server_provider_supports_operation (prov, cnc, GDA_SERVER_OPERATION_DROP_COLUMN, NULL))
			gtk_action_set_sensitive (action, FALSE);
		else
			gtk_action_set_visible (action, FALSE);

		action = gtk_action_group_get_action (ws->priv->actions, "TableContents");
		gtk_action_set_sensitive (action, FALSE);
	}
}

static void table_info_display_update (GdaDictTable *table, WsTables *ws);

static void
table_selection_changed_cb (GnomeDbSelector *mgsel, GObject *sel_object, WsTables *ws)
{
	GtkAction *action;

	if (sel_object && !GDA_IS_DICT_TABLE (sel_object))
		return;

	ws->priv->sel_obj = sel_object;
	gtk_notebook_set_current_page (GTK_NOTEBOOK (ws->priv->notebook), sel_object ? 1 : 0);

	action = gtk_action_group_get_action (ws->priv->actions, "DROP_TABLE");
	if (action)
		gtk_action_set_sensitive (action, sel_object ? TRUE : FALSE);

	action = gtk_action_group_get_action (ws->priv->actions, "RENAME_TABLE");
	if (action)
		gtk_action_set_sensitive (action, sel_object ? TRUE : FALSE);

	action = gtk_action_group_get_action (ws->priv->actions, "ADD_COLUMN");
	if (action)
		gtk_action_set_sensitive (action, sel_object ? TRUE : FALSE);

	action = gtk_action_group_get_action (ws->priv->actions, "DROP_COLUMN");
	if (action)
		gtk_action_set_sensitive (action, sel_object ? TRUE : FALSE);

	action = gtk_action_group_get_action (ws->priv->actions, "TableContents");
	gtk_action_set_sensitive (action, sel_object ? TRUE : FALSE);

	table_info_display_update ((GdaDictTable *) sel_object, ws);
}

static void show_table_contents (WsTables *ws, GdaDictTable *table);

static void
table_row_activated_cb (GnomeDbSelector *mgsel, GObject *sel_object, WsTables *ws)
{
	/* show contents of @sel_object */
	if (sel_object && !GDA_IS_DICT_TABLE (sel_object))
		return;
	show_table_contents (ws, GDA_DICT_TABLE (sel_object));
}

static void
table_updated_cb (GdaDictDatabase *db, GdaDictTable *table, WsTables *ws)
{
	if ((GObject *)table == ws->priv->sel_obj)
		table_info_display_update (table, ws);
}

static void
action_table_contents_cb (GtkAction *action, WsTables *ws)
{
	show_table_contents (ws, GDA_DICT_TABLE (ws->priv->sel_obj));
}

static void
show_table_contents (WsTables *ws, GdaDictTable *table)
{
	GdaQuery *query;
	gchar *sql;
	GdaDict *dict = ws->priv->dict;
	GError *error = NULL;
	GtkWidget *window = NULL;
	
	sql = g_strdup_printf ("SELECT * FROM %s\n", gda_object_get_name (GDA_OBJECT (table)));
	query = gda_query_new_from_sql (dict, sql, NULL);
	g_free (sql);

	if (gda_query_get_query_type (query) != GDA_QUERY_TYPE_NON_PARSED_SQL) {
		GtkWidget *grid, *label;
		gchar *str, *mstr, *warn = NULL;
		GdaDataModel *model;

		model = gda_data_model_query_new (query);
		if (! gda_dict_table_is_view (table)) {
			if (!gda_data_model_query_compute_modification_queries (GDA_DATA_MODEL_QUERY (model), 
										NULL, 0, &error)) {
				if (error && error->message)
					warn = g_strdup_printf (_("Data cannot be modified for the following reason:\n%s"), 
								error->message);
				else
					warn = g_strdup (_("Data cannot be modified"));
				g_error_free (error);
				error = NULL;
			}
		}
		else
			warn = g_strdup (_("Data cannot be modified because it is a view's data"));

		grid = mg_extra_formgrid_new (model);
		g_object_unref (model);

		if (gda_dict_table_is_view (table))
			str = g_strdup_printf (_("Contents of view '%s'"), gda_object_get_name (table));
		else
			str = g_strdup_printf (_("Contents of table '%s'"), gda_object_get_name (table));

		window = gtk_dialog_new_with_buttons (str, NULL, 0,
						      GTK_STOCK_CLOSE, GTK_RESPONSE_ACCEPT,
						      NULL);
		g_signal_connect_swapped (window,
					  "response", 
					  G_CALLBACK (gtk_widget_destroy),
					  window);

		if (warn) {
			mstr = g_strdup_printf ("<span weight=\"bold\" size=\"larger\">%s</span>\n"
						"<small>%s</small>\n", str, warn);
			g_free (warn);
		}
		else
			mstr = g_strdup_printf ("<span weight=\"bold\" size=\"larger\">%s</span>\n", str);
		g_free (str);
		label = gtk_label_new ("");
		gtk_label_set_markup (GTK_LABEL (label), mstr);
		gtk_misc_set_alignment (GTK_MISC (label), 0., -1);
		g_free (mstr);

		gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->vbox), label, FALSE, TRUE, 0);
		gtk_widget_show (label);

		gtk_box_pack_start (GTK_BOX (GTK_DIALOG (window)->vbox), grid, TRUE, TRUE, 0);
		gtk_widget_show (grid);

		gtk_window_set_default_size (GTK_WINDOW (window), 800, 300);
		gtk_widget_show (window);

		g_object_unref (G_OBJECT (query));
	}
	else {
		if (error) {
			g_print ("ERROR: %s\n", error->message);
			g_error_free (error);
		}
	}

	if (window) {
		ws->priv->opened_dialogs = g_slist_append (ws->priv->opened_dialogs, window);
		g_signal_connect (G_OBJECT (window), "destroy",
				  G_CALLBACK (opened_dialog_closed_cb), ws);
	}
}

static void
table_info_display_update (GdaDictTable *table, WsTables *ws)
{
	const gchar *str = NULL;
	gchar *title = NULL;
	GtkTextIter start, end;
	
	if (table)
		str = gda_object_get_description (GDA_OBJECT (table));
	if (str && *str) 
		gtk_label_set_text (GTK_LABEL (ws->priv->description), str);
	else
		gtk_label_set_text (GTK_LABEL (ws->priv->description), "---");
	
	gnome_db_selector_set_mode_columns (GNOME_DB_SELECTOR (ws->priv->fields), (GObject*) table,
					    GNOME_DB_SELECTOR_FIELDS, 
					    GNOME_DB_SELECTOR_COLUMN_TYPE | GNOME_DB_SELECTOR_COLUMN_FIELD_LENGTH | 
					    GNOME_DB_SELECTOR_COLUMN_FIELD_NNUL |
					    GNOME_DB_SELECTOR_COLUMN_FIELD_DEFAULT | GNOME_DB_SELECTOR_COLUMN_COMMENTS);

	/* global title update */
	title = ws_tables_get_description (WORKSPACE_PAGE (ws));
	g_signal_emit_by_name (G_OBJECT (ws), "description_changed", title);
	g_free (title);

	/* integrity rules update */
	gtk_text_buffer_get_start_iter (ws->priv->integ_descr, &start);
	gtk_text_buffer_get_end_iter (ws->priv->integ_descr, &end);
	gtk_text_buffer_delete (ws->priv->integ_descr, &start, &end);
	
	if (table) {
		GtkTextIter current;
		GSList *constraints, *list;

		gtk_text_buffer_get_start_iter (ws->priv->integ_descr, &current);		
		/* constraints list */
		constraints = gda_dict_table_get_constraints (table);
		
		/* PKey */
		list = constraints;
		while (list) {
			gboolean first = TRUE;
			if (gda_dict_constraint_get_constraint_type (GDA_DICT_CONSTRAINT (list->data)) == 
			    CONSTRAINT_PRIMARY_KEY) {
				GdaDictConstraint *cstr = GDA_DICT_CONSTRAINT (list->data);
				GSList *fields, *list2;
				gboolean header = FALSE;

				fields = gda_dict_constraint_pkey_get_fields (cstr);
				list2 = fields;
				while (list2) {
					if (!header) {
						header = TRUE;
						gtk_text_buffer_insert_with_tags_by_name (ws->priv->integ_descr,
											  &current, 
											  _("Primary key"), -1,
											  "section", NULL);
						gtk_text_buffer_insert (ws->priv->integ_descr, &current, "\n", -1);
					}

					if (first) 
						first = FALSE;
					else
						gtk_text_buffer_insert (ws->priv->integ_descr, &current, ", ", -1);

					gtk_text_buffer_insert (ws->priv->integ_descr, &current, 
								gda_object_get_name (GDA_OBJECT (list2->data)), -1);
					list2 = g_slist_next (list2);
				}
				g_slist_free (fields);
				gtk_text_buffer_insert (ws->priv->integ_descr, &current, "\n\n", -1);
			}
			list = g_slist_next (list);
		}

		/* FKey */
		list = constraints;
		while (list) {
			if (gda_dict_constraint_get_constraint_type (GDA_DICT_CONSTRAINT (list->data)) == 
			    CONSTRAINT_FOREIGN_KEY) {
				GdaDictConstraint *cstr = GDA_DICT_CONSTRAINT (list->data);
				GSList *fields, *list2;
				gboolean header = FALSE;

				fields = gda_dict_constraint_fkey_get_fields (cstr);
				list2 = fields;
				while (list2) {
					GdaEntity *ent;
					if (!header) {
						header = TRUE;
						gtk_text_buffer_insert_with_tags_by_name (ws->priv->integ_descr,
											  &current, 
											  _("Foreign key"), -1,
											  "section", NULL);
						gtk_text_buffer_insert (ws->priv->integ_descr, &current, "\n", -1);
					}
						
					str = gda_object_get_name (GDA_OBJECT (GDA_DICT_CONSTRAINT_FK_PAIR (list2->data)->fkey));
					gtk_text_buffer_insert (ws->priv->integ_descr, &current, str, -1);
					gtk_text_buffer_insert (ws->priv->integ_descr, &current, " --> ", -1);
					ent = gda_entity_field_get_entity (GDA_ENTITY_FIELD (GDA_DICT_CONSTRAINT_FK_PAIR (list2->data)->ref_pkey));
					str = gda_object_get_name (GDA_OBJECT (ent));
					gtk_text_buffer_insert (ws->priv->integ_descr, &current, str, -1);
					gtk_text_buffer_insert (ws->priv->integ_descr, &current, ".", -1);
					str = gda_object_get_name (GDA_OBJECT (GDA_DICT_CONSTRAINT_FK_PAIR (list2->data)->ref_pkey));
					gtk_text_buffer_insert (ws->priv->integ_descr, &current, str, -1);
					gtk_text_buffer_insert (ws->priv->integ_descr, &current, "\n", -1);
						
					list2 = g_slist_next (list2);
				}
				g_slist_free (fields);
				gtk_text_buffer_insert (ws->priv->integ_descr, &current, "\n", -1);
			}
			list = g_slist_next (list);
		}

		/* Unique */
		list = constraints;
		while (list) {
			if (gda_dict_constraint_get_constraint_type (GDA_DICT_CONSTRAINT (list->data)) == 
			    CONSTRAINT_UNIQUE) {
				GdaDictConstraint *cstr = GDA_DICT_CONSTRAINT (list->data);
				GSList *fields, *list2;
				gboolean header = FALSE;

				fields = gda_dict_constraint_unique_get_fields (cstr);
				list2 = fields;
				while (list2) {
					if (!header) {
						header = TRUE;
						gtk_text_buffer_insert_with_tags_by_name (ws->priv->integ_descr,
											  &current, 
											  _("UNIQUE constraint"), -1,
											  "section", NULL);
						gtk_text_buffer_insert (ws->priv->integ_descr, &current, "\n", -1);
					}
					else
						gtk_text_buffer_insert (ws->priv->integ_descr, &current, ", ", -1);

					gtk_text_buffer_insert (ws->priv->integ_descr, &current, 
								gda_object_get_name (GDA_OBJECT (list2->data)), -1);
						
					list2 = g_slist_next (list2);
				}
				g_slist_free (fields);
				gtk_text_buffer_insert (ws->priv->integ_descr, &current, "\n\n", -1);
			}
			list = g_slist_next (list);
		}

		/* check constraint: FIXME */

		g_slist_free (constraints);			
	}
}

static void
field_selection_changed_cb (GnomeDbSelector *mgsel, GObject *sel_object, MgPluginEditor *editor)
{
	mg_plugin_editor_set_work_object (editor, sel_object);
}


/* 
 * WorkspacePage interface implementation
 */
static gchar *
ws_tables_get_name (WorkspacePage *iface)
{
	g_return_val_if_fail (IS_WS_TABLES (iface), NULL);
	g_return_val_if_fail (WS_TABLES (iface)->priv, NULL);

	return g_strdup_printf (_("Tables and views"));
}

static gchar *
ws_tables_get_description (WorkspacePage *iface)
{
	gchar *str = NULL;
	WsTables *ws;

	g_return_val_if_fail (IS_WS_TABLES (iface), NULL);
	g_return_val_if_fail (WS_TABLES (iface)->priv, NULL);

	ws = WS_TABLES (iface);
	if (ws->priv->sel_obj) {
		if (gda_dict_table_is_view (GDA_DICT_TABLE (ws->priv->sel_obj)))
			str = g_strdup_printf ("View: <b>%s</b>", 
					       gda_object_get_name (GDA_OBJECT (ws->priv->sel_obj)));
		else
			str = g_strdup_printf ("Table: <b>%s</b>", 
					       gda_object_get_name (GDA_OBJECT (ws->priv->sel_obj)));
	}
	else
		str = g_strdup_printf ("<b>%s</b>", _("No table selected"));

	return str;
}

static GtkWidget *
ws_tables_get_sel_button (WorkspacePage *iface)
{
	GdkPixbuf *pixbuf;
	GtkWidget *button, *wid, *hbox;

	g_return_val_if_fail (IS_WS_TABLES (iface), NULL);
	g_return_val_if_fail (WS_TABLES (iface)->priv, NULL);

	hbox = gtk_hbox_new (FALSE, 0);

	pixbuf = gnome_db_stock_get_icon_pixbuf (GNOME_DB_STOCK_TABLES);
	wid = gtk_image_new_from_pixbuf (pixbuf);
	g_object_unref (pixbuf);
	gtk_box_pack_start (GTK_BOX (hbox), wid, FALSE, FALSE, 0);
	gtk_widget_show (wid);

	wid = gtk_label_new (_("Tables"));
	gtk_box_pack_start (GTK_BOX (hbox), wid, FALSE, FALSE, 5);
	gtk_widget_show (wid);

	button = gtk_toggle_button_new ();
	gtk_container_add (GTK_CONTAINER (button), hbox);
	gtk_widget_show (hbox);

	return button;
}

static GtkWidget *
ws_tables_get_selector (WorkspacePage *iface)
{
	g_return_val_if_fail (IS_WS_TABLES (iface), NULL);
	g_return_val_if_fail (WS_TABLES (iface)->priv, NULL);

	return WS_TABLES (iface)->priv->selector;
}

static GtkWidget *
ws_tables_get_work_area (WorkspacePage *iface)
{
	g_return_val_if_fail (IS_WS_TABLES (iface), NULL);
	g_return_val_if_fail (WS_TABLES (iface)->priv, NULL);

	return WS_TABLES (iface)->priv->work_area;
}

static GtkActionGroup *
ws_tables_get_actions (WorkspacePage *iface)
{
	g_return_val_if_fail (IS_WS_TABLES (iface), NULL);
	g_return_val_if_fail (WS_TABLES (iface)->priv, NULL);

	return WS_TABLES (iface)->priv->actions;
}

static const gchar *
ws_tables_get_actions_ui (WorkspacePage *iface)
{
	g_return_val_if_fail (IS_WS_TABLES (iface), NULL);
	g_return_val_if_fail (WS_TABLES (iface)->priv, NULL);

	return ui_actions_info;
}
static void action_done_cb (GtkWidget *dlg, gboolean succeed, WsTables *ws);
static void
action_create_table_cb (GtkAction *action, WsTables *ws)
{
	GtkWidget *dlg;

	dlg = mergeant_server_op_create_dialog (NULL, GDA_SERVER_OPERATION_CREATE_TABLE, 
						gda_dict_get_connection (ws->priv->dict), NULL,
						_("Create a table"), NULL,
						MERGEANT_SERVER_OP_CALLBACK (action_done_cb), ws);
	gtk_widget_show (dlg);

	ws->priv->opened_dialogs = g_slist_append (ws->priv->opened_dialogs, dlg);
	g_signal_connect (G_OBJECT (dlg), "destroy",
			  G_CALLBACK (opened_dialog_closed_cb), ws);
}

static void
action_drop_table_cb (GtkAction *action, WsTables *ws)
{
	GdaParameterList *options;
	GtkWidget *dlg;

	g_return_if_fail (GDA_IS_DICT_TABLE (ws->priv->sel_obj));

	options = gda_parameter_list_new (NULL);
	gda_parameter_list_add_param_from_string (options, "/TABLE_DESC_P/TABLE_NAME", G_TYPE_STRING,
						  gda_object_get_name (GDA_OBJECT (ws->priv->sel_obj)));

	dlg = mergeant_server_op_create_dialog (NULL, GDA_SERVER_OPERATION_DROP_TABLE, 
						gda_dict_get_connection (ws->priv->dict), options,
						_("Delete a table"), NULL,
						MERGEANT_SERVER_OP_CALLBACK (action_done_cb), ws);
	g_object_unref (options);

	gtk_widget_show (dlg);

	ws->priv->opened_dialogs = g_slist_append (ws->priv->opened_dialogs, dlg);
	g_signal_connect (G_OBJECT (dlg), "destroy",
			  G_CALLBACK (opened_dialog_closed_cb), ws);
}

static void
action_done_cb (GtkWidget *dlg, gboolean succeed, WsTables *ws)
{
	if (succeed) {
		GdaDictDatabase *db;
		GdaServerOperation *op;
		const GValue *value;
		const gchar *table_name;
		
		db = gda_dict_get_database (ws->priv->dict);
		op = g_object_get_data (G_OBJECT (dlg), "op");

		value = gda_server_operation_get_value_at (op, "/TABLE_DEF_P/TABLE_NAME");
		if (value) {
			table_name = g_value_get_string (value);
			gda_dict_database_update_dbms_data (db, GDA_TYPE_DICT_TABLE, table_name, NULL);
		}

		value = gda_server_operation_get_value_at (op, "/TABLE_DESC_P/TABLE_NAME");
		if (value) {
			table_name = g_value_get_string (value);
			gda_dict_database_update_dbms_data (db, GDA_TYPE_DICT_TABLE, table_name, NULL);
		}

		value = gda_server_operation_get_value_at (op, "/TABLE_DESC_P/TABLE_NEW_NAME");
		if (value) {
			table_name = g_value_get_string (value);
			gda_dict_database_update_dbms_data (db, GDA_TYPE_DICT_TABLE, table_name, NULL);
		}

		value = gda_server_operation_get_value_at (op, "/COLUMN_DEF_P/TABLE_NAME");
		if (value) {
			table_name = g_value_get_string (value);
			gda_dict_database_update_dbms_data (db, GDA_TYPE_DICT_TABLE, table_name, NULL);
		}
		value = gda_server_operation_get_value_at (op, "/COLUMN_DESC_P/TABLE_NAME");
		if (value) {
			table_name = g_value_get_string (value);
			gda_dict_database_update_dbms_data (db, GDA_TYPE_DICT_TABLE, table_name, NULL);
		}
	}
}

static void
action_rename_table_cb (GtkAction *action, WsTables *ws)
{
	GdaParameterList *options;
	GtkWidget *dlg;

	g_return_if_fail (GDA_IS_DICT_TABLE (ws->priv->sel_obj));

	options = gda_parameter_list_new (NULL);
	gda_parameter_list_add_param_from_string (options, "/TABLE_DESC_P/TABLE_NAME", G_TYPE_STRING,
						  gda_object_get_name (GDA_OBJECT (ws->priv->sel_obj)));

	dlg = mergeant_server_op_create_dialog (NULL, GDA_SERVER_OPERATION_RENAME_TABLE, 
						gda_dict_get_connection (ws->priv->dict), options,
						_("Rename a table"), NULL,
						MERGEANT_SERVER_OP_CALLBACK (action_done_cb), ws);
	g_object_unref (options);

	gtk_widget_show (dlg);

	ws->priv->opened_dialogs = g_slist_append (ws->priv->opened_dialogs, dlg);
	g_signal_connect (G_OBJECT (dlg), "destroy",
			  G_CALLBACK (opened_dialog_closed_cb), ws);
}

static void
action_add_column_cb (GtkAction *action, WsTables *ws)
{
	GdaParameterList *options;
	GtkWidget *dlg;

	g_return_if_fail (GDA_IS_DICT_TABLE (ws->priv->sel_obj));

	options = gda_parameter_list_new (NULL);
	gda_parameter_list_add_param_from_string (options, "/COLUMN_DEF_P/TABLE_NAME", G_TYPE_STRING,
						  gda_object_get_name (GDA_OBJECT (ws->priv->sel_obj)));

	dlg = mergeant_server_op_create_dialog (NULL, GDA_SERVER_OPERATION_ADD_COLUMN, 
						gda_dict_get_connection (ws->priv->dict), options,
						_("Add a column to a table"), NULL,
						MERGEANT_SERVER_OP_CALLBACK (action_done_cb), ws);
	g_object_unref (options);

	gtk_widget_show (dlg);

	ws->priv->opened_dialogs = g_slist_append (ws->priv->opened_dialogs, dlg);
	g_signal_connect (G_OBJECT (dlg), "destroy",
			  G_CALLBACK (opened_dialog_closed_cb), ws);
}

static void
action_drop_column_cb (GtkAction *action, WsTables *ws)
{
	GdaParameterList *options;
	GtkWidget *dlg;
	GdaDictField *sel_field;

	g_return_if_fail (GDA_IS_DICT_TABLE (ws->priv->sel_obj));

	options = gda_parameter_list_new (NULL);
	gda_parameter_list_add_param_from_string (options, "/COLUMN_DESC_P/TABLE_NAME", G_TYPE_STRING,
						  gda_object_get_name (GDA_OBJECT (ws->priv->sel_obj)));

	sel_field = gnome_db_selector_get_selected_object (GNOME_DB_SELECTOR (ws->priv->fields));
	if (sel_field) 
		gda_parameter_list_add_param_from_string (options, "/COLUMN_DESC_P/COLUMN_NAME", G_TYPE_STRING,
						  gda_object_get_name (GDA_OBJECT (sel_field)));

	dlg = mergeant_server_op_create_dialog (NULL, GDA_SERVER_OPERATION_DROP_COLUMN, 
						gda_dict_get_connection (ws->priv->dict), options,
						_("Delete a column drom a table"), NULL,
						MERGEANT_SERVER_OP_CALLBACK (action_done_cb), ws);
	g_object_unref (options);

	gtk_widget_show (dlg);

	ws->priv->opened_dialogs = g_slist_append (ws->priv->opened_dialogs, dlg);
	g_signal_connect (G_OBJECT (dlg), "destroy",
			  G_CALLBACK (opened_dialog_closed_cb), ws);
}

static void
opened_dialog_closed_cb (GtkWidget *dlg, WsTables *ws)
{
	ws->priv->opened_dialogs = g_slist_remove (ws->priv->opened_dialogs, dlg);
}
