/* mg-conf.h
 *
 * Copyright (C) 2003 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


/*
 * This object manages the objects used my mergeant (for tables, queries, etc),
 * but not what is specific to Mergeant's own GUI.
 */

#ifndef __MG_CONF_H_
#define __MG_CONF_H_

#include <glib-object.h>
#include "mg-defs.h"

G_BEGIN_DECLS

#define MG_CONF_TYPE          (mg_conf_get_type())
#define MG_CONF(obj)          G_TYPE_CHECK_INSTANCE_CAST (obj, mg_conf_get_type(), MgConf)
#define MG_CONF_CLASS(klass)  G_TYPE_CHECK_CLASS_CAST (klass, mg_conf_get_type (), MgConfClass)
#define IS_MG_CONF(obj)       G_TYPE_CHECK_INSTANCE_TYPE (obj, mg_conf_get_type ())


/* error reporting */
extern GQuark mg_conf_error_quark (void);
#define MG_CONF_ERROR mg_conf_error_quark ()

enum {
	MG_CONF_LOAD_FILE_NOT_EXIST_ERROR,
	MG_CONF_FILE_LOAD_ERROR,
	MG_CONF_FILE_SAVE_ERROR,
	MG_CONF_PROPOSED_FILE
};


/* struct for the object's data */
struct _MgConf
{
	GObject              object;
	MgConfPrivate       *priv;
};

/* struct for the object's class */
struct _MgConfClass
{
	GObjectClass            class;

        /* signal the addition or removal of a query in the queries list */
        void (*query_added)      (MgConf * conf, MgQuery *new_query);
        void (*query_removed)    (MgConf * conf, MgQuery *old_query);
        void (*query_updated)    (MgConf * conf, MgQuery *query);

        /* signal the addition or removal of a graph in the graphs list */
        void (*graph_added)      (MgConf * conf, MgGraph *new_graph);
        void (*graph_removed)    (MgConf * conf, MgGraph *old_graph);
        void (*graph_updated)    (MgConf * conf, MgGraph *graph);

        /* signal the addition or removal of a custom layout in the layouts list */
        void (*layout_added)      (MgConf * conf, MgCustomLayout *new_layout);
        void (*layout_removed)    (MgConf * conf, MgCustomLayout *old_layout);
        void (*layout_updated)    (MgConf * conf, MgCustomLayout *layout);

	/* signal that a change in the whole dictionnary has occured */
	void (*changed)          (MgConf * conf);
};

guint           mg_conf_get_type            (void);
GObject        *mg_conf_new                 (void);

guint           mg_conf_get_id_serial       (MgConf *conf);
void            mg_conf_set_id_serial       (MgConf *conf, guint value);

gchar          *mg_conf_compute_xml_filename(MgConf *conf, const gchar *datasource, const gchar *app_id, GError **error);
void            mg_conf_set_xml_filename    (MgConf *conf, const gchar *xmlfile);
const gchar    *mg_conf_get_xml_filename    (MgConf *conf);
gboolean        mg_conf_load_xml            (MgConf *conf, GError **error);
gboolean        mg_conf_save_xml            (MgConf *conf, GError **error);
gboolean        mg_conf_load_xml_file       (MgConf *conf, const gchar *xmlfile, GError **error);
gboolean        mg_conf_save_xml_file       (MgConf *conf, const gchar *xmlfile, GError **error);

void            mg_conf_declare_query       (MgConf *conf, MgQuery *query);
void            mg_conf_assume_query        (MgConf *conf, MgQuery *query);
void            mg_conf_unassume_query      (MgConf *conf, MgQuery *query);
GSList         *mg_conf_get_queries         (MgConf *conf);
MgQuery        *mg_conf_get_query_by_xml_id (MgConf *conf, const gchar *xml_id);

void            mg_conf_declare_graph       (MgConf *conf, MgGraph *graph);
void            mg_conf_assume_graph        (MgConf *conf, MgGraph *graph);
void            mg_conf_unassume_graph      (MgConf *conf, MgGraph *graph);
GSList         *mg_conf_get_graphs          (MgConf *conf);
MgGraph        *mg_conf_get_graph_by_xml_id (MgConf *conf, const gchar *xml_id);
MgGraph        *mg_conf_get_graph_for_object(MgConf *conf, GObject *obj);

void            mg_conf_declare_layout       (MgConf *conf, MgCustomLayout *layout);
void            mg_conf_assume_layout        (MgConf *conf, MgCustomLayout *layout);
void            mg_conf_unassume_layout      (MgConf *conf, MgCustomLayout *layout);
GSList         *mg_conf_get_layouts          (MgConf *conf);
MgCustomLayout *mg_conf_get_layout_by_xml_id (MgConf *conf, const gchar *xml_id);


MgServer       *mg_conf_get_server          (MgConf *conf);
MgDatabase     *mg_conf_get_database        (MgConf *conf);

GSList         *mg_conf_get_entities_fk_constraints (MgConf *conf, MgEntity *entity1, MgEntity *entity2,
						     gboolean entity1_has_fk);

#ifdef debug
void            mg_conf_dump                (MgConf *conf);
#endif
G_END_DECLS


#endif
