/* conds-view.c
 *
 * Copyright (C) 2002 - 2003 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "conds-view.h"
#include "canvas-query-cond.h"


/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;

static void conds_view_class_init (CondsViewClass * class);
static void conds_view_init       (CondsView * csv);
static void conds_view_dispose    (GObject   * object);
static void conds_view_post_init  (CondsView * csv);

struct _CondsViewPrivate
{
	GObject            *manager;
	GnomeCanvasItem    *item;

	/* currently displayed top QueryCond */
	QueryCond          *top_cond;

	/* pointer position when a context menu was last opened */
	gdouble             xmouse;
	gdouble             ymouse;
};

guint
conds_view_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (CondsViewClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) conds_view_class_init,
			NULL,
			NULL,
			sizeof (CondsView),
			0,
			(GInstanceInitFunc) conds_view_init
		};		

		type = g_type_register_static (GNOME_TYPE_CANVAS, "CondsView", &info, 0);
	}
	return type;
}

static void
conds_view_class_init (CondsViewClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	parent_class = g_type_class_peek_parent (class);

	object_class->dispose = conds_view_dispose;
}


static void
conds_view_init (CondsView * csv)
{
	csv->priv = g_new0 (CondsViewPrivate, 1);
	csv->priv->manager = NULL;
	csv->priv->item = NULL;

	csv->priv->xmouse = 50.;
	csv->priv->ymouse = 50.;
}

static void cond_changed_cb (GObject *manager, CondsView *csv);
static void manager_nullified_cb (GObject *manager, CondsView *csv);
static int canvas_event(GnomeCanvasItem *item, GdkEvent *event, CondsView * csv);

/* the "manager" object MUST emit a "cond_changed" when either the global
 * condition has changed or the top condition has been replaced
 */
GtkWidget *
conds_view_new (GObject *manager)
{
	GObject   *obj;
	CondsView *csv;

	g_return_val_if_fail (manager && G_IS_OBJECT (manager), NULL);

	obj = g_object_new (CONDS_VIEW_TYPE, NULL);
	/* obj = g_object_new (CONDS_VIEW_TYPE, "aa", TRUE, NULL); */
	csv = CONDS_VIEW (obj);

	csv->priv->manager = manager;
	conds_view_post_init (csv);

	/* signals from the manager */
	g_signal_connect (manager, "cond_changed",
			  G_CALLBACK (cond_changed_cb), csv);
	if (IS_QUERY_JOIN (manager)) /* FIXME: for the Query object */
		g_signal_connect (manager, "nullified",
				  G_CALLBACK (manager_nullified_cb), csv);

	g_signal_connect (G_OBJECT (csv),"event-after",
			  G_CALLBACK(canvas_event), csv);

	return GTK_WIDGET (obj);
}

static void set_top_cond (CondsView *csv);
static void redraw_items (CondsView *csv);

static void
cond_changed_cb (GObject *manager, CondsView *csv)
{
	set_top_cond (csv);
	redraw_items (csv);
}

static void
manager_nullified_cb (GObject *manager, CondsView *csv)
{
	csv->priv->manager = NULL;
	set_top_cond (csv);
	redraw_items (csv);
}

static void
set_top_cond (CondsView *csv)
{
	if (csv->priv->manager) {
		QueryCond *cond = NULL;

		if (IS_QUERY (csv->priv->manager))
			cond = QUERY (csv->priv->manager)->where_cond;
		if (IS_QUERY_JOIN (csv->priv->manager))
			cond = query_join_get_condition (QUERY_JOIN (csv->priv->manager));
		
		csv->priv->top_cond = cond;
	}
	else
		csv->priv->top_cond = NULL;
}


static int 
canvas_event(GnomeCanvasItem *item, GdkEvent *event, CondsView * csv)
{
	gboolean done = TRUE;
	GnomeCanvasItem *itemat;
	gdouble wx, wy;

	if (g_object_get_data (G_OBJECT (gnome_canvas_root (GNOME_CANVAS (csv))), "dragged_from")) {
		/* Dragging cancelled */
		g_object_set_data (G_OBJECT (gnome_canvas_root (GNOME_CANVAS (csv))), "dragged_from", NULL);
	}

	switch (event->type) {
	case GDK_BUTTON_PRESS:
		gnome_canvas_window_to_world (GNOME_CANVAS (csv), 
					      ((GdkEventButton *) event)->x, 
					      ((GdkEventButton *) event)->y,
					      &wx, &wy);
		itemat = gnome_canvas_get_item_at (GNOME_CANVAS (csv), wx, wy);
		
		if (!itemat) {
			if (((GdkEventButton *) event)->button == 3) {
				GtkWidget *menu;
				
				csv->priv->xmouse = wx;
				csv->priv->ymouse = wy;
				/* Display a menu here if necessary */
				done = TRUE;
			}
		}
		break;
	default:
		done = FALSE;
		break;
	}

	return done;	
}



static void drag_action_cb (CanvasBase *repport,
			    CanvasBase *drag_to, CanvasBase * drag_from, CondsView *csv);
static void
conds_view_dispose (GObject   * object)
{
	CondsView *csv;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_CONDS_VIEW (object));

	csv = CONDS_VIEW (object);

	if (csv->priv) {
		if (csv->priv->manager) {
			/* signals from the manager */
			g_signal_handlers_disconnect_by_func (csv->priv->manager,
							      G_CALLBACK (cond_changed_cb), csv);
			g_signal_handlers_disconnect_by_func (csv->priv->manager,
							      G_CALLBACK (manager_nullified_cb),
							      csv);
			csv->priv->manager = NULL;
		}
	
		/* get rid of the GnomeCanvasItem */
		if (csv->priv->item) {
			gtk_object_destroy (GTK_OBJECT (csv->priv->item));
			csv->priv->item = NULL;
		}
		g_free (csv->priv);
		csv->priv = NULL;
	}

	/* for the parent class */
	parent_class->dispose (object);
}


static void 
conds_view_post_init  (CondsView * csv)
{
	gdouble x1, y1, x2, y2;
	
	gnome_canvas_get_scroll_region (GNOME_CANVAS (csv), &x1, &y1, &x2, &y2);

	set_top_cond (csv);
	redraw_items (csv);
}

static void
redraw_items (CondsView *csv)
{
	double x1, y1, x2, y2;
	gfloat space = 10.;

	/* remove current items */
	if (csv->priv->item) 
		gtk_object_destroy (GTK_OBJECT (csv->priv->item));
	
	/* display the top QueryCond */
	csv->priv->item = gnome_canvas_item_new (gnome_canvas_root (GNOME_CANVAS (csv)),
						 CANVAS_QUERY_COND_TYPE,
						 "x", space,
						 "y", 0.,
						 "query_cond", csv->priv->top_cond,
						 "allow_move", FALSE,
						 NULL);
	g_object_add_weak_pointer (G_OBJECT (csv->priv->item), &(csv->priv->item));

	/* resize the scroll area */
	gnome_canvas_item_get_bounds (csv->priv->item, &x1, &y1, &x2, &y2);
	gnome_canvas_item_move (csv->priv->item, 0., - (y2-y1)/2.);
	gnome_canvas_set_scroll_region(GNOME_CANVAS (csv), -space, -(y2-y1)/2.,
				       (x2-x1)+space, (y2-y1)/2.);
}

static void 
drag_action_cb (CanvasBase *repport, CanvasBase *drag_from, CanvasBase *drag_to, CondsView *csv)
{
	g_return_if_fail (IS_CONDS_VIEW (csv));

	g_print ("Dragging not yet implemented\n");
}


static void vadj_changed_cb (GtkAdjustment *adj, CondsView *conds);
GtkWidget *
conds_view_new_in_sw (GObject *manager)
{
	GtkWidget *sw, *conds;
	GtkAdjustment *adj;

	g_return_val_if_fail (manager && G_IS_OBJECT (manager), NULL);

	sw = gtk_scrolled_window_new (NULL, NULL);
	conds = conds_view_new (manager);
	
	gtk_widget_set_usize(conds, 200, 200);
	gtk_container_add(GTK_CONTAINER(sw), conds);
	gtk_widget_show (conds);

	adj = gtk_scrolled_window_get_vadjustment (GTK_SCROLLED_WINDOW (sw));
	g_signal_connect (G_OBJECT (adj), "changed",
			  G_CALLBACK (vadj_changed_cb), conds);

	adj = gtk_scrolled_window_get_hadjustment (GTK_SCROLLED_WINDOW (sw));
	/*g_signal_connect (G_OBJECT (adj), "changed",
	  G_CALLBACK (adj_changed_cb), conds);*/

	return sw;
}

static void
vadj_changed_cb (GtkAdjustment *adj, CondsView *conds)
{
	g_signal_handlers_disconnect_by_func (G_OBJECT (adj), 
					      G_CALLBACK (vadj_changed_cb), conds);
	gtk_adjustment_set_value (adj, (adj->upper - adj->lower) / 2.);
}
