'''
Defines a device that uses the IBM TTS synthesizer via the pyibmtts package.
pyibmtts can be obtained from U{http://ibmtts-sdk.sf.net}.

@author: Larry Weiss
@author: Peter Parente
@author: Brett Clippingdale
@organization: IBM Corporation
@copyright: Copyright (c) 2006 IBM Corporation
@license: The BSD License

All rights reserved. This program and the accompanying materials are made
available under the terms of the BSD license which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/bsd-license.php}
'''
import AEOutput
import pyibmtts
from i18n import _

__uie__ = dict(kind='device')

class IBMSpeechStyle(AEOutput.StyleFlyweight):
  '''
  Overrides the base L{AEOutput.Style.StyleFlyweight} class to provide style
  settings relative to the default style for this device.
  '''
  ADDITIVE = ('Volume', 'Rate', 'Pitch')
  Rate = 0
  Volume = 0
  Pitch = 0
  
  def getSettings(self):
    '''
    Gets configurable relative settings affecting output from this device for
    particular types of information.
    
    @todo: PP: support in a later version
    
    @return: Group of all configurable settings per semantic information type
    @rtype: L{AEState.Setting.Group}
    '''
    raise NotImplementedError

class IBMSpeechStyleDefault(AEOutput.AudioStyleDefault):
  '''
  Overrides the base L{AEOutput.Style.AudioStyleDefault} class, filling in 
  fields for supported style properties with their appropriate values.
  '''
  MaxVoice = 8
  MinVoice = 1
  MaxVolume = 65535
  MinVolume = 1
  Volume = 58982 # about 90%
  MinPitch = 40
  MaxPitch = 442
  Pitch = 220
  MaxRate = 1297
  MinRate = 176
  Rate = 250
  Voice = 1

  def getSettings(self):
    '''
    Gets configurable absolute settings affecting all output from this device.
    
    @return: Group of all configurable settings
    @rtype: L{AEState.Setting.Group}
    '''
    g = self._newGroup()
    a = g.newGroup(_('Speech'))
    a.newRange('Pitch', _('Pitch'), self.MinPitch, self.MaxPitch, 0,
               _('Baseline voice pitch in Hertz'))
    a.newRange('Rate', _('Rate'), self.MinRate, self.MaxRate, 0,
               _('Speech rate in words per minute'))
    a.newPercent('Volume', _('Volume'), self.MinVolume, self.MaxVolume, 0,
                 _('Speech volume as a percentage'))
    # generate a group for standard word parsing settings
    self._newWordGroup(g)
    return g
  
class IBMSpeech(AEOutput.Audio):
  '''
  IBM TTS via the pyibmtts wrapper.
 
  @ivar last_style: Last style object to be applied to output
  @type last_style: L{AEOutput.Style}
  '''    
  USE_THREAD = False
  DEFAULT_STYLE = IBMSpeechStyleDefault
  FLYWEIGHT_STYLE = IBMSpeechStyle
  
  def _localize(self, text):
    '''
    Converts the given unicode text to latin-1, the encoding expected by the
    IBM TTS runtime.

    @todo: PP: Temporary kludge to convert all text to Latin-1 as IBM TTS is
      expecting that encoding instead of UTF-8. Should look at the style 
      language attribute in the future to do the correct encoding.
    @param text: Text to localize
    @type text: unicode
    @return: Localized string
    @rtype: string
    '''
    try:
      # kludge!
      return text.encode('latin-1', 'ignore')
    except (TypeError, UnicodeDecodeError), e:
      return text

  def init(self):
    '''
    Initializes the IBM TTSspeech driver through gnome-speech.
    
    @raise AEOutput.InitError: When the device can not be initialized
    '''
    self.last_style = None
    try:
      self.tts = pyibmtts.Engine()
    except pyibmtts.ECIException:
      raise AEOutput.InitError
    # use real units and start at some speedy rate
    self.tts.realWorldUnits = True
      
  def createDistinctStyles(self, num_groups, num_layers):
    '''
    Creates distinct styles following the guidelines set forth in the base
    class. Only distinguishes groups using voice, not layers.
    
    @param num_groups: Number of sematic groups the requestor would like to
      represent using distinct styles
    @type num_groups: integer
    @param num_layers: Number of content origins (e.g. output originating from
      a background task versus the focus) the requestor would like to represent
      using distinct styles
    @type num_layers: integer   
    @return: Styles
    @rtype: list of L{AEOutput.Style}
    ''' 
    styles = []
    for i in xrange(1, num_groups*num_layers+1):
      s = IBMSpeechStyle(self.default_style)
      v_num = (i) % s.MaxVoice
      v = self.tts.getVoice(v_num)
      s.Voice = v_num
      s.Rate = 0
      s.Pitch = v.pitchBaseline-self.default_style.Pitch
      s.Volume = v.volume-self.default_style.Volume
      styles.append(s)
    return styles

  def close(self):
    '''
    Closes the speech device.
    '''
    del self.tts

  def sendStop(self, style=None):
    '''
    Stops speech immediately.
    
    @param style: Ignored
    @type style: L{AEOutput.Style}
    '''
    try:
      self.tts.stop()
    except pyibmtts.ECIException:
      pass

  def sendTalk(self, style=None):
    '''
    Begins synthesis of the buffered text.
    
    @param style: Ignored
    @type style: L{AEOutput.Style}
    '''
    try:
      self.tts.synthesize()
    except pyibmtts.ECIException:
      pass
    
  def sendString(self, text, style):
    '''
    Adds the given string to the text to be synthesized.

    @param text: String to buffer on the device
    @type text: string
    @param style: Style with which this string should be output; None means no
      style change should be applied
    @type style: integer
    '''
    if style.isDirty() or self.last_style != style:
      self.last_style = style
      self._applyStyle(style)
    try:
      self.tts.addText(self._localize(text))
    except pyibmtts.ECIException:
      pass

  def isActive(self):
    '''
    Indicates whether the device is active meaning it is busy doing output or
    has text waiting to be output.

    @return: True when the speech device is synthesizing, False otherwise
    @rtype: boolean
    '''
    try:
      return self.tts.speaking()
    except pyibmtts.Exception:
      return False

  def getName(self):
    '''
    Gives the user displayable (localized) name for this output device.
    Relevant version and device status should be included.

    @return: Localized name for the device
    @rtype: string
    '''
    return 'IBM-TTS %s' % pyibmtts.getVersion()
  
  def sendIndex(self, style):
    '''
    Sends an index marker to the device driver. The driver should notify the
    device when the marker has been reached during output.
    
    @todo: PP: implement when index working

    @param style: Style indicating channel in which the marker will be appended
    @type style: L{AEOutput.Style}
    '''
    raise NotImplementedError

  def _applyStyle(self, style):
    '''
    Applies a given style to this output device. All output following this 
    call will be presented in the given style until this method is called again
    with a different style.
    
    @param style: Style to apply to future output
    @type style: L{AEOutput.Style}
    '''
    try:
      v = self.tts.getVoice(style.Voice)
      v = self.tts.setActiveVoice(v)
    except pyibmtts.ECIException:
      # just quit if we can't even get the voice to which style params should 
      # be applied
      return
    # restore voice independent parameters
    try:
      v.speed = min(max(style.Rate, style.MinRate), style.MaxRate)
    except pyibmtts.ECIException:
      pass
    try:
      v.volume = min(max(style.Volume, style.MinVolume), style.MaxVolume)
    except pyibmtts.ECIException:
      pass
    try:
      v.pitchBaseline = min(max(style.Pitch, style.MinPitch), style.MaxPitch)
    except pyibmtts.ECIException:
      pass  
