'''
Defines the base class for all L{Task}s.

@author: Peter Parente
@author: Larry Weiss
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import Tools, Perk, UIElement, UIRegistrar

class Task(Tools.All, UIElement.UIE):
  '''
  Most base class for all L{Task}s that execute in a L{Perk}. Stores a 
  string identifying a particular class instance. Implements L{UIElement}
  to provide information to the user about its purpose even though it is not
  managed by the L{UIRegistrar}.
  
  @ivar ident: Programmatic identifier for this L{Task}
  @type ident: string
  ''' 
  def __init__(self, ident=None):
    '''
    Stores the L{Task} identifier.
    
    @param ident: Programmatic identifier
    @type ident: string
    '''
    Tools.All.__init__(self)
    self.ident = ident
  
  def execute(self, **kwargs):
    '''
    Executes this L{Task} in response to an event. Called by 
    L{Tier.Tier._executeTask}.
    
    @param kwargs: Dictionary containing parameters passed to a task.
    @type kwargs: dictionary 
    @return: Should processing continue? Always returns True by default.
    @rtype: boolean
    '''
    return True
  
  def update(self, **kwargs):
    '''
    Allows this L{Task} to update itself in response to an event that was 
    consumed by a L{Task} that executed higher in the L{Perk} stack. Called by 
    L{Tier.Tier._executeTask}.
    
    @param kwargs: Dictionary containing parameters passed to a task.
    @type kwargs: dictionary 
    '''
    pass
  
  def getIdentity(self):
    '''
    Gets the unique string identity assigned to this L{Task}. Type checking
    is used to ensure the identity is a string to avoid complications with
    what previous versions of LSR required as the first parameter to this
    class.
    
    @return: Identity of this L{Task}
    @rtype: string
    @raise ValueError: When a string identity is not supplied
    '''
    if not isinstance(self.ident, str):
      raise ValueError
    return self.ident
  
  def getPerk(self):
    '''
    Gets the L{Perk} in which this L{Task} was registered as an event handler
    or as a named task.
    
    @return: Reference to the L{Perk}
    @rtype: L{Perk}
    '''
    return self.perk
  
  def getPath(self):
    '''
    @return: Absolute path location of the module containing the implementor
      of this interface
    @rtype: string
    '''
    return UIRegistrar.getPath(self.perk.getClassName())