'''
Defines an L{AEEvent} indicating that the selector has moved.

@author: Pete Brunet
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''

import Base, Constants

class SelectorChange(Base.AccessEngineEvent):
  '''
  Event that fires when the selector moves.
  
  This class registers its name and whether it should be monitored by default in
  an L{AEMonitor} using the L{Constants.registerEventType} function
  when this module is first imported. The L{AEMonitor} can use this
  information to build its menus.
  
  @ivar text: Accessible text associated with the event, typically the item 
    text or the selected text
  @type text: string
  @cvar ACTIVE_ITEM_SELECT: Active item changed inside a container-like object
  @type ACTIVE_ITEM_SELECT: integer
  @cvar ADD_ITEM_SELECT: A new item was added to a selection
  @type ADD_ITEM_SELECT: integer
  @cvar REMOVE_ITEM_SELECT: An item was removed from a selection
  @type REMOVE_ITEM_SELECT: integer
  @cvar CHANGE_TEXT_SELECT: A string of text was added or removed from a 
    selection
  @type CHANGE_TEXT_SELECT: integer
  '''
  ACTIVE_ITEM_SELECT = 0
  ADD_ITEM_SELECT = 1 
  REMOVE_ITEM_SELECT = 2
  CHANGE_TEXT_SELECT = 3
  
  Constants.registerEventType('SelectorChange', True)
  def __init__(self, por, text, kind=ACTIVE_ITEM_SELECT, **kwargs):
    '''
    Calls the base class (which set the event priority) and then stores the item
    to be passed along to the Tier as part of the event.
    
    @param por: Point-of-regard
    @type por: L{POR}
    @param text: Accessible text associated with the event, typically the 
      item text or the selected text
    @type text: string
    @param kind: Kind of selection event, one of  L{ACTIVE_ITEM_SELECT}, 
      L{ADD_ITEM_SELECT}, L{REMOVE_ITEM_SELECT}, or L{CHANGE_TEXT_SELECT}
    @type kind: integer
    '''
    Base.AccessEngineEvent.__init__(self, **kwargs)
    self.text = text
    self.por = por
    self.kind = kind
    
  def __str__(self):
    '''
    Returns a human readable representation of this event including its name,
    its L{POR}, and its item text.
    
    @return: Information about this event
    @rtype: string
    '''
    name = Base.AccessEngineEvent.__str__(self)
    if self.kind == self.ACTIVE_ITEM_SELECT:
      action = 'active'
    elif self.kind == self.ADD_ITEM_SELECT:
      action = 'add'
    elif self.kind == self.REMOVE_ITEM_SELECT:
      action = 'remove'
    else:
      action = 'text'
    return '%s:\n\tPOR: %s\n\titem text: %s\n\taction: %s' % \
      (name, self.por, self.text, action)
  
  def execute(self, tier_manager, **kwargs):
    '''
    Contacts the L{TierManager} so it can manage the selector change event.
    
    @param tier_manager: TierManager that will handle the event
    @type tier_manager: L{TierManager}
    @param kwargs: Packed references to other managers not of interest here
    @type kwargs: dictionary
    @return: Always True to indicate the event executed properly
    @rtype: boolean
    '''
    tier_manager.manageEvent(self)
    return True

  def getDataForTask(self):
    '''
    Fetches data out of this L{SelectorChange} for use by a
    L{Task.SelectorTask}.
    
    @return: Dictionary of parameters to be passed to a
      L{Task.SelectorTask} as follows:
        - por: The L{POR} pointing at the item at which selection changed or
               the current caret location for a text selection change
        - text: The text now selected or unselected
        - kind: The kind of selection change
    @rtype: dictionary
    '''
    return {'por':self.getPOR(), 'text':self.text, 'kind':self.kind}