'''
Defines a class for managing the L{Perk}s loaded for a single instance of an
application.

@author: Peter Parente
@author: Pete Brunet
@author: Larry Weiss
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import logging, weakref
import AEState, AEInput, Perk, Task
from POR import POR

log = logging.getLogger('Tier')

class Tier(object):
  '''
  Manages the L{Perk}s for a single instance of a running program (a process). 
  Supports the addition and removal of L{Perk}s from its stack.  Executes 
  L{Task}s registered in each L{Perk} in response to L{AEEvent}s. Provides
  methods to assist L{Perk}s and L{Task}s in finding registered L{Task}s 
  throughout the L{Tier}.

  @ivar perks: List of L{Perk}s treated as a stack (last in, first executed)
  @type perks: list
  @ivar wanted_events: Lookup table for what L{AEEvent}s are desired by any
    L{Task} in any L{Perk} in this L{Tier}. Used to optimize event dispatch and
    processing. Unwanted events are ignored.
  @type wanted_events: dictionary
  @ivar tier_manager: L{TierManager} that created this L{Tier}
  @type tier_manager: L{TierManager}
  @ivar pointer_por: Pointer of regard for the user focus
  @type pointer_por: L{POR}
  @ivar focus_por: Point of regard for the application focus
  @type focus_por: L{POR}
  @ivar name: Name of this L{Tier}
  @type name: string
  @ivar aid: ID uniquely identifying this L{Tier} from all other L{Tier}s
  @type aid: opaque
  '''
  def __init__(self, tier_manager, name, aid, por):
    '''
    Stores a reference to the L{TierManager} that created this L{Tier}. Creates
    an empty list of L{Perk}s. Initializes the stored focus L{POR} to None and
    the pointer L{POR} to the application. Creates the weak dictionary for 
    L{Perk} L{Task}s and stores other information about the L{Tier}.
    
    @param tier_manager: Manager that created this L{Tier}
    @type tier_manager: L{TierManager}
    @param name: Name of this L{Tier}
    @type name: string
    @param aid: ID uniquely identifying this L{Tier} from all other L{Tier}s
    @type aid: opaque
    @param por: First point of regard to the application represented by 
      this L{Tier}. Typically, the top most accessible.
    @type por: L{POR}
    '''
    self.tier_manager = tier_manager
    self.perks = []
    self.wanted_events = {}
    self.name = name
    self.aid = aid
    self.perk_refs = weakref.WeakValueDictionary()
    self.pointer_por = por
    self.focus_por = None

  def getPointer(self):
    '''
    Gets the user L{POR} for this L{Tier}.
    
    @return: Point of regard of user attention
    @rtype: L{POR}
    '''
    return self.pointer_por
  
  def setPointer(self, por):
    '''
    Sets the user L{POR} for this L{Tier}.
    
    @param por: Point of regard of user attention
    @type por: L{POR}
    '''
    self.pointer_por = por

  def getFocus(self):
    '''
    Gets the application focus L{POR} for this L{Tier}.
    
    @return: Point of regard of application focus
    @rtype: L{POR}
    '''
    return self.focus_por

  def addPerkRef(self, key, perk):
    '''
    Adds a key that can be used later to quickly look up a reference to a
    L{Perk} in this L{Tier}. Optimization for the L{getKeyedTask},
    L{getCommandTask}, and L{getNamedTask} methods.
    
    @param key: Key under which to hash the L{Perk}
    @type key: immutable
    @param perk: Perk to store in the hash
    @type perk: L{Perk}
    '''
    self.perk_refs[key] = perk
    
  def removePerkRef(self, key):
    '''
    Removes a key used to quickly look up a reference to a L{Perk} in this 
    L{Tier}. Keys are cleaned up automatically when L{Perk}s are destroyed.
    However, this method may be used to manually remove L{Perk}s at any time.
    
    @param key: Key under which to hash the L{Perk}
    @type key: immutable
    '''
    try:
      del self.perk_refs[key]
    except KeyError:
      pass
    
  def getName(self):
    '''
    @return: Name of this L{Tier} (i.e. name of the app with which it is 
      associated)
    @rtype: string
    '''
    return self.name
  
  def getIdentity(self):
    '''
    @return: Unique identifier for this L{Tier}
    @rtype: opaque
    '''
    return self.aid

  def pushPerk(self, ae, *perks):
    '''
    Adds one or more L{Perk}s to the top of the stack. If more than one L{Perk}
    is specified, the last specified L{Perk} will be at the top of the stack 
    when this method completes. That is, the behavior of pushing more than one
    L{Perk} at a time is the same as if each L{Perk} were pushed individually.
    
    @param ae: The AccessEngine context to use for initializing the Perk
    @type ae: L{AccessEngine}
    @param perks: L{Perk}s to add
    @type perks: list of L{Perk}
    '''
    self.insertPerk(ae, 0, *perks)
      
  def insertPerk(self, ae, index, *perks):
    '''
    Adds one L{Perk} to the stack at the insertion index. Negative indices are
    valid per Python list convention.
    
    @param ae: The AccessEngine context to use for initializing the Perk
    @type ae: L{AccessEngine}
    @param index: Index at which the L{Perk} should be inserted in the stack
    @type index: integer
    @param perks: L{Perk}s to add
    @type perks: list of L{Perk}
    '''
    for perk in perks:
      # put the Perk somewhere in the stack
      self.perks.insert(index, perk)
      # initialize the Perk base classes with important, persistent references
      perk.preInit(ae, self)
      # initialize this particular instance with temporary references
      perk.preExecute(Task.LAYER_FOCUS)
      try:
        # now let Perk specific code run
        perk.init()
      except Exception, e:
        # pop the perk from the stack if it fails to initialize
        self.perks.pop(index)
        # let it finalize
        perk.close()
        perk.postClose()
        # log the exception
        log.exception('Error in Perk.init')
      perk.postExecute(False)
    
  def popPerk(self, *indices):
    '''
    Removes one or more L{Perk}s from the stack given their indices. Negative
    indices are valid per Python list convention. Calls L{Perk.Perk.close} on 
    each to ensure they have a chance to persist state.
    
    @param indices: Indices of L{Perk}s to remove
    @type indices: list of integer
    @raise IndexError: When the stack is empty
    @raise ValueError: When a specific index is out of the bounds of the stack
    '''
    indices = list(indices)
    # have to pop in reverse order to avoid corrupting original indices
    indices.sort(reverse=True)
    perks = map(self.perks.pop, indices)
    map(Perk.Perk.close, perks)
    map(Perk.Perk.postClose, perks)
      
  def clearPerks(self):
    '''
    Removes all L{Perk}s from the stack. Calls L{Perk.Perk.close} on each to 
    ensure they have a chance to persist state.
    '''
    map(Perk.Perk.close, self.perks)
    map(Perk.Perk.postClose, self.perks)
    self.perks = []
      
  def getPerks(self):
    '''
    Gets all L{Perk}s pushed onto this L{Tier} in execution order.
    
    @return: All L{Perk}s pushed onto this L{Tier}
    @rtype: list of L{Perk}
    '''
    return self.perks
  
  def setEventInterest(self, kind, wants):
    '''    
    Updates the L{wanted_events} dictionary to indicate that a L{Task} in a
    L{Perk} in this L{Tier} has been registered for a particular kind of
    L{AEEvent}. This information is used for optimization purposes such that no
    processing will occur on the event unless at least one L{Task} is
    registered that will use it.
    
    @param kind: Kind of L{AEEvent} of interest to a L{Task}
    @type kind: L{AEEvent} class
    @param wants: Does a L{Perk} want an event (i.e. a L{Task} is registering 
      for it or no longer want an event (i.e. a L{Task} is unregistering for 
      it)?
    @type wants: boolean
    '''
    count = self.wanted_events.setdefault(kind, 0)
    if wants:
      count += 1
    else:
      count -= 1
    if count <= 0:
      del self.wanted_events[kind]
    else:
      self.wanted_events[kind] = count
    # inform the TierManager of the new event interest
    self.tier_manager.setEventInterest(kind, wants)
      
  def wantsEvent(self, event):
    '''
    Gets if this L{Tier} wants a particular kind of L{AEEvent} given that one of
    its L{Perk}s has a L{Task} that wants to handle it.
    
    @param event: Event to be tested
    @type event: L{AEEvent}
    '''
    return self.wanted_events.has_key(type(event))
  
  def getKeyedTask(self, key):
    '''
    Gets a L{Task} registered under a particular key set to execute in response
    to events. None is returned if not found.
    
    @param key: Unique ID for locating a registered L{Task}
    @type key: integer
    @return: L{Task} set to execute in response to the event or None
    @rtype: L{Task}
    '''
    try:
      perk = self.perk_refs[key]
    except KeyError:
      return None
    return perk.getKeyedTask(key)
  
  def getCommandTask(self, gesture_list):
    '''
    Gets a L{Task} registered to execute in response to the L{AEInput.Gesture}.
    None is returned if not found.
    
    @param gesture_list: Gestures and device expressed as a list of virtual key 
      codes
    @type gesture_list: L{AEInput.Gesture}
    @return: L{Task} set to execute in response to the input gesture or None
    @rtype: L{Task.InputTask.InputTask}
    '''
    try:
      perk = self.perk_refs[gesture_list]
    except KeyError:
      return None
    name = perk.getCommandTask(gesture_list)
    return self.getNamedTask(name)

  def getNamedTask(self, name):
    '''
    Gets a L{Task} with the given name by iterating through the registered 
    L{Perk}s until one is found containing a L{Task} registered under the given
    name. None is returned if not found.
    
    @param name: Name of the L{Task} to find 
    @type name: string
    @return: L{Task} with the given name or None
    @rtype: L{Task.Base.Task}
    '''
    try:
      perk = self.perk_refs[name]
    except KeyError:
      return None
    return perk.getNamedTask(name)
    
  def getEventTasks(self, event_type, task_layer):
    '''    
    Gets all L{Task}s registered to handle the given type of event on the
    given layer by iterating through the registered {Perk}s. The L{Task}s are
    returned in the order they will be executed both within and across
    L{Perk}s in this L{Tier}.

    @param event_type: Desired type of L{AEEvent}
    @type event_type: L{AEEvent} class
    @param task_layer: Layer on which the desired L{Task}s are registered, one 
      of L{Task.LAYER_FOCUS}, L{Task.LAYER_TIER}, or L{Task.LAYER_BACKGROUND}
    @type task_layer: integer
    @return: List of all L{Task}s registered to handle the given type of event
      on the given layer
    @rtype: list of L{Task.Base.Task}
    '''
    tasks = []
    map(tasks.extend, [p.getEventTasks(event_type, task_layer) for p in perks])
    return tasks
  
  def _executeTask(self, task, por, layer, task_data, propagate, timestamp=0):
    '''
    Executes the given L{Task} in response to the given L{AEEvent}.
    
    @param task: A L{Task} registered to handle the given event
    @type task: L{Task}
    @param por: Point of regard where the event occurred
    @type por: L{POR}
    @param propagate: Should this event be propagated to the 
      L{Task.Base.Task.execute} method or should we call 
      L{Task.Base.Task.update} instead?
    @type propagate: boolean
    @param layer: Layer on which the L{AEEvent} occurred, one of 
      L{AEEvent.LAYER_FOCUS}, L{AEEvent.LAYER_TIER}, 
      L{AEEvent.LAYER_BACKGROUND}
    @type layer: integer
    @param task_data: Keyword arguments to be provided to the L{Task}
    @type task_data: dictionary
    @param timestamp: Time at which the event occurred. Not available for all
      events. Defaults to zero.
    @type timestamp: float
    @return: Should the next registered L{Task} be executed?
    @rtype: boolean
    '''
    # pass important references to the Task
    if not task.preExecute(layer, por, timestamp):
      return True
    try:
      if propagate:
        # unpack the dictionary and execute the task
        rv = task.execute(layer=layer, **task_data)
      else:
        # unpack the dictionary and update the task
        task.update(layer=layer, **task_data)
        rv = False
    except Task.ToolsError, ex:
      # speak the Task.ToolsError if state.Trap is True
      if self.tier_manager.getState().Trap:
        task.sayError(text=str(ex))
      else:
        log.exception(ex)
      rv = True
    except Exception:
      # log any other Perk exception
      log.exception('%s: %s %s', task.getClassName(), layer, task_data)
      # continue processing
      rv = True
    if rv is None:
      # continue processing if no return value is specified
      rv = True
    # uninitialize the Task, potentially storing the POR only if the event was
    # in the focus layer and execution is still propagating
    task.postExecute(task.layer == Task.LAYER_FOCUS and propagate)
    return rv
  
  def manageEvent(self, event):
    '''
    Manages an event by iterating through the L{Perk} stack (top to bottom) and
    checking for registered L{Task}s of the given type. Executes the registered
    L{Task}s (last registered, first executed) in each L{Perk} until one of the 
    following conditions is met:
      - All L{Task}s have executed
      - A L{Task} returns False
      - A L{Task} raises an exception
    
    In the latter two cases, no additional L{Task}s in the current L{Perk} or
    additional L{Perk}s in this L{Tier} are executed. Instead the
    L{Task.Base.Task.update} methods are called to allow housekeeping
    operations (e.g updating state) to be performed.
    
    If a L{Task} returns neither True or False (e.g. it returns None) a 
    warning is logged and the return value is treated as if it were True. This
    likely means the L{Task} forgot to specify a return value.

    @param event: Event to process
    @type event: L{AEEvent.Base.AccessEngineEvent}
    '''
    try:
      # grab POR from any focus type event
      self.focus_por = event.getFocusPOR()
    except AttributeError:
      pass
    
    # use type as an indicator of what Task will handle
    if not self.wantsEvent(event):
      # quit immediately if no Task wants this event
      return
    
    # show the event in registered monitors
    self.tier_manager.showEvent(event, self.name)
    
    # get these values once to initialize, most are constant for all Tasks
    event_por = event.getPOR()
    task_layer = event.getLayer()
    task_data = event.getDataForTask()
    event_type = type(event)

    # run through all the registered Perks
    propagate = True
    for perk in self.perks:
      if not perk.preExecute(task_layer, event_por):
        # don't execute if the Perk isn't prepared
        continue
      # run through all Tasks of the given type in this Perk
      for task in perk.getEventTasks(event_type, task_layer):
        propagate = self._executeTask(task, event_por, task_layer, 
                                      task_data, propagate)
        self.tier_manager.showTask(event, perk, task, propagate)
      perk.postExecute(False)
        
  def manageGesture(self, event):
    '''
    Manages an event by getting the L{AEInput.Gesture} that triggered it and
    locating a L{Task} registered to execute in response to it. If a L{Task} 
    could not be found for the given event, the L{Task} registered for invalid
    gestures is executed instead.

    @param event: Event to process
    @type event: L{AEEvent.InputGesture}
    '''
    # look for a Perk with a Task registered to respond to this input command
    task = self.getCommandTask(event.getTaskKey())
    # show the event in registered monitors
    self.tier_manager.showEvent(event, self.name)
    if task is not None:
      layer = event.getLayer()
      timestamp = event.getTimestamp()
      perk = task.getPerk()
      if not perk.preExecute(layer):
        return
      # execute that Task
      self._executeTask(task, None, layer, {}, True, timestamp)
      perk.postExecute(False)
      
  def manageKeyedTask(self, event):
    '''
    Manages an event by locating a L{Task} under a particular key registered to
    execute in response to it.

    @param event: Event to process
    @type event: L{AEEvent}
    '''
    # look for a Perk with a Task registered to respond to this event
    task = self.getKeyedTask(event.getTaskKey())
    # show the event in registered monitors
    self.tier_manager.showEvent(event, self.name)
    if task is not None:
      layer = event.getLayer()
      perk = task.getPerk()
      task_data = event.getDataForTask()
      if not perk.preExecute(layer):
        return
      # execute that Task
      self._executeTask(task, None, layer, task_data, True)
      perk.postExecute(False)

  def manageNamedTask(self, por, name, layer, task_data):
    '''
    Executes a L{Task} registered under the given name in this L{Tier} by
    configuring it to run in the given layer and with the given keyword 
    argument data.
    
    @todo: PP: show doTask?
    
    @param name: Name of the L{Task} to execute
    @type name: string
    @param layer: Layer in which to execute the L{Task}, one of 
      L{AEEvent.LAYER_FOCUS}, L{AEEvent.LAYER_TIER}, 
      L{AEEvent.LAYER_BACKGROUND}
    @type layer: integer
    @param task_data: Keyword data to pass to the L{Task} on execution
    @type task_data: dictionary
    '''
    # look for a Perk with a Task registered to respond to this event
    task = self.getNamedTask(name)
    # show the event in registered monitors
    #self.tier_manager.showEvent(event, self.name)
    if task is not None:
      perk = task.getPerk()
      if not perk.preExecute(layer, por):
        return
      # execute that Task
      self._executeTask(task, por, layer, task_data, True)
      perk.postExecute(False)