'''
Defines the base class for all audio output devices.

@author: Brett Clippingdale
@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import time
import Base
import Word
from ThreadProxy import *

class Audio(Base.AEOutput):
  ''' 
  Defines the base class for all audio output devices. Provides default 
  implementations of L{sendStringSync} and L{parseString} specific to audio
  devices.
  '''
  def getProxy(self):
    '''
    Looks at the L{USE_THREAD} flag to see if the device implementing this 
    interface wants a thread proxy or not.
    
    @return: self or L{ThreadProxy.AudioThreadProxy}
    @rtype: L{AEOutput}
    '''
    if self.USE_THREAD == True:
      return AudioThreadProxy(self)
    elif self.USE_THREAD == False:
      return self
    else:
      raise NotImplementedError('USE_THREAD not specified')
  
  def sendStringSync(self, text, style, stop=True):
    '''
    Buffers a complete string to send to the device synchronously.

    This should B{not} be used in place of L{sendString} since this will not
    return until the string is finished being output. This is provided B{only}
    for the convenience of utility writers. Uses L{sendStop}, L{sendString}, 
    L{sendTalk}, and then sleeps until L{isActive} returns False.

    @param text: String to send to the device
    @type text: string
    @param style: Style on which this string should be output; None implies some
      reasonable default should be used
    @type style: L{AEOutput.Style}
    @param stop: Stop current output before doing this output?
    @type stop: boolean
    '''
    proxy = self.getProxy()
    if stop:
      proxy.sendStop()
    proxy.sendString(text, style)
    proxy.sendTalk()

    # wait until done speaking; this blocks the main thread -- be careful
    while proxy.isActive():
      time.sleep(0.1)
      
  def _parseMain(self, word, style, ch, part):
    '''    
    Adds support for the L{AEOutput.Style.AudioStyleDefault} settings of
    CapExpand and NumExpand. This method is notified during parsing of the main
    part of a L{Word}.
    
    @param word: Word currently parsing its source text
    @type word: L{Word}
    @param style: Style used to configure the parsing process
    @type style: L{AEOutput.Style.AudioStyleDefault}
    @param ch: Character in the word
    @type ch: string
    @param part: List of characters already processed in this part of the word
    @type part: list
    @return: Character(s) to be appended to the list
    @rtype: string
    '''
    if style.CapExpand and word.isCap(ch) and word.getMainLength():
      part.append(' ')
    elif style.NumExpand and word.isNumeric(ch) and word.getMainLength():
      part.append(' ')
    return ch

  def parseString(self, text, style, por):
    '''
    Provides a default implementation of parsing that formats words for audio
    devices supporting speech output. The base L{Word} class is used plus some
    additional processing for:
    
      - blank words
      - expanded caps
      - expanded numbers
    
    @todo: PP: add support for all other speech parsing features (see old
      LSRSettings module)
    
    @param text: Text to be parsed
    @type text: string
    @param style: Style object defining how the text should be parsed
    @type style: L{AEOutput.Style}
    @param por: Point of regard for the first character in the text, or None if
      the text is not associated with a POR
    @type por: L{POR}
    @return: Parsed words
    @rtype: 3-tuple of lists of string, L{POR}, L{AEOutput.Style}
    '''
    rv = []
    # parse words
    words = Word.buildWordsFromString(text, por, style, self._parseMain)
    for i, w in enumerate(words):
      if w.isAllBlank():
        if i == 0 and len(words) > 1:
          # ignore initial blank words when there is more than one word in the
          # group
          continue
        # fill in blank words
        text = style.Blank
      else:
        # leave non-blanks alone
        text = unicode(w)
      if por is None:
        # don't bother with PORs if we didn't start with one
        por = None
      else:
        por = w.getPOR()
      # ignore style changes for the time being
      rv.append((text, por, style))
    return rv