'''
Defines a GTK dialog for buffering L{AEEvent}s and how L{Perk}s and L{Task}s 
handle them.

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
from AEEvent import LAYER_NAMES
from LSRConstants import SEMANTIC_NAMES
from DeviceManager import InputEvent, OutputEvent, DeviceEvent
from GTKEventDialog import GTKEventDialog
from i18n import _

__uie__ = dict(kind='monitor', profiles=['developer'])

class IOMonitor(GTKEventDialog):
  '''
  Logs information about input and output to a GUI window.
  '''
  def init(self):
    '''
    Initialize the monitor instance variables.
    '''
    super(IOMonitor, self).init()
    self.last_dev = None
    self.last_kind = None

  def getName(self):
    '''
    Gets the localized name of this monitor.
    
    @return: Monitor name
    @rtype: string
    '''
    return _('I/O monitor')
  
  def getEventNames(self):
    '''
    Gets the event categories to be displayed in the View menu for filtering.
    
    @return: Event categories
    @rtype: list of string
    '''
    return InputEvent.getNames() + OutputEvent.getNames()
  
  def getEventDefaults(self):
    '''
    Gets the default event categories to check in the View menu.
    
    @return: Event categories
    @rtype: list of string
    '''
    return InputEvent.getDefaults() + OutputEvent.getDefaults()
  
  def getEventType(self):
    '''
    Gets the L{DeviceManager.DeviceEvent} base class to indicate the type 
    of events this monitor wants to buffer.
    
    @return: Base type of the event this monitor should buffer
    @rtype: L{DeviceManager.DeviceEvent} class
    '''
    return DeviceEvent
      
  def show(self, event, text=None, dev=None, sem=None, layer=None, 
           gesture=None):
    '''
    Buffers additional details about how the L{AEEvent} was handled by various
    L{Perk}s and L{Task}s. The event is only provided as a means of filtering 
    information.
    
    @param event: Event to filter on
    @type event: string
    @param text: Text or filename sent to the device
    @type text: string
    @param sem: Semantic constant indicating the kind of output
    @type sem: integer
    @param layer: Layer constant indicating from where the event came
    @type layer: integer
    @param dev: Device on which the event occurred
    @type dev: L{AEInput} or L{AEOutput}
    @param gesture: Gesture seen on the device
    @type gesture: L{AEInput.Gesture}
    @raise IOError: When the monitor is no longer accepting data to buffer 
    '''
    if not self.isInitialized():
      raise IOError
    elif not self._isShown(event):
      return
    
    if dev and dev.getName() != self.last_dev:
      # show the name of a device
      name = dev.getName()
      self.last_dev = name
      self._queueText('\n\n{device: %s}\n' % name)

    if event == InputEvent.GESTURE:
      self._queueText('{gesture: %s}\n' % str(gesture))
    if event == OutputEvent.TALK:
      self._queueText(' {talk: %s} ' % SEMANTIC_NAMES.get(sem))
    elif event == OutputEvent.STOP:
      self._queueText('{stop}\n')
    else:
      if event == OutputEvent.STRING:
        self._queueText(str(text))
      elif event == OutputEvent.FILENAME:
        self._queueText('{file %s}', str(text))

