'''
Contains the L{GSpeech} abstract base class which provides support for all
gnome-speech devices. Device definitions using gnome-speech should derive from
this class and override the L{GSpeech.USE_THREAD} and L{GSpeech.COMMAND_CHARS}
properties if desired. The subclass should also provide createDistinctStyles
and _applyStyle methods. This module should never be directly registered as
its own speech device with L{UIRegistrar} as the GSpeech class does not fully
implement the L{AEOutput.Base.AEOutput} interface.

@var MAX_BUFFER: Maximum number of characters to buffer before forcing a 
  L{GSpeech.sendTalk} call.
@type MAX_BUFFER: integer
@var SAY_WAIT_RETRY: Time to wait between calls to say if say fails
@type SAY_WAIT_RETRY: float

@author: Larry Weiss
@author: Peter Parente
@author: Brett Clippingdale
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import time
import AEOutput
from i18n import _

import ORBit, bonobo
# try to get the typelib
ORBit.load_typelib('GNOME_Speech')
import GNOME.Speech, GNOME__POA.Speech

# constants specific to GSpeech devices
SAY_WAIT_RETRY = 0.01
MAX_BUFFER = 80

class GSpeechStyle(AEOutput.StyleFlyweight):
  '''
  Overrides the base L{AEOutput.Style.StyleFlyweight} class to provide style
  settings relative to the default style for this device.
  '''
  ADDITIVE = ('Rate', 'Pitch', 'Volume')
  Rate = 0
  Volume = 0
  Pitch = 0
  
  def getSettings(self):
    '''
    Gets configurable relative settings affecting output from this device for
    particular types of information.
    
    @todo: PP: support in a later version
    
    @return: Group of all configurable settings per semantic information type
    @rtype: L{AEState.Setting.Group}
    '''
    raise NotImplementedError

class GSpeechStyleDefault(AEOutput.AudioStyleDefault):
  '''
  Empty shell for a default style class that will have its attributes 
  programmatically determined.
  '''
  def getSettings(self):
    '''
    Gets configurable absolute settings affecting all output from this device.
    Builds the panel based on settings supported by the configured gnome-speech
    device capabilities.
    
    @return: Group of all configurable settings
    @rtype: L{AEState.Setting.Group}
    '''
    g = self._newGroup()
    a = g.newGroup(_('Speech'))
    if self.Pitch is not None:
      a.newRange('Pitch', _('Pitch'), self.MinPitch, self.MaxPitch, 0,
                 _('Baseline voice pitch'))
    if self.Rate is not None:
      a.newRange('Rate', _('Rate'), self.MinRate, self.MaxRate, 0,
                 _('Speech rate'))
    if self.Volume is not None:
      a.newRange('Volume', _('Volume'), self.MinVolume, self.MaxVolume, 0,
                 _('Speech volume as a percentage'))
    # generate a group for standard word parsing settings
    self._newWordGroup(g)
    return g

class GSpeech(AEOutput.Audio):
  '''
  Defines an abstract base class to send output from LSR to a speech device via
  gnome-speech U{http://cvs.gnome.org/viewcvs/gnome-speech/}. A subclass must
  override the DEVICE_IID class variable to indicate which gnome-speech server
  should be instantiated. If the specific speech server does not have named
  parameters that map automatically to the style properties in 
  L{AEOutput.Style}, L{createDistinctStyles} and L{_applyStyle} should be 
  overridden. If the default style should be something other than an empty shell
  object to be dynamically populated in createDistinctStyles, the subclass 
  should also override L{getDefaultStyle} to return a pre-populated instance of
  L{GSpeechStyleDefault}.

  To be compliant with LSR requirements, this implements the interface defined 
  in the L{AEOutput.Audio} class.

  @ivar last_style: Last style object to be applied to output
  @type last_style: L{AEOutput.Style}
  @ivar driver: Reference to a speech engine server
  @type driver: GNOME.Speech.SynthesisDriver
  @ivar speaker: Speaker that will synthesize speech from buffered text
  @type speaker: GNOME.Speech.Speaker
  @ivar buffer: Buffer of text and styles to be sent to the device
  @type buffer: list
  @cvar DEVICE_IID: Interface identifier for the gnome-speech device. Defaults 
    to None and should be overridden in a subclass.
  @type DEVICE_IID: string
  ''' 
  DEVICE_IID = None
  FLYWEIGHT_STYLE = GSpeechStyle
  DEFAULT_STYLE = GSpeechStyleDefault
  
  def _applyStyle(self, style):
    '''
    Applies a given style to this output device. All output following this 
    call will be presented in the given style until this method is called again
    with a different style.

    @param style: Style to apply to future output
    @type style: L{AEOutput.Style}
    '''  
    sp = self.driver.createSpeaker(self.voices[style.Voice])
    #if self.speaker is not None:
      #self.speaker.unref()
    
    for param in sp.getSupportedParameters():
      cap_name = param.name.title()
      sp.setParameterValue(param.name, getattr(style, cap_name))
    self.speaker = sp
    
  def init(self):
    '''
    Initializes the speech driver through gnome-speech.

    @raise AEOutput.InitError: When the device can not be initialized
    '''
    self.driver = None
    self.speaker = None
    self.voices = []
    self.buffer = []
    self.last_style = None
    
    # initialize CORBA
    ORBit.CORBA.ORB_init()
    # try to activate Festival
    self.driver = bonobo.activation.activate_from_id(self.DEVICE_IID, 0, False)
    try:
      # check if already initialized
      bInit = self.driver.isInitialized()
    except AttributeError:
      # driver is None when the engine is not available
      raise AEOutput.InitError
    # could have been initialized by someone else
    if not bInit:
      if not self.driver.driverInit():
        # driverInit fails when engine is not in a "good state"
        raise AEOutput.InitError
    # store all voices for later reference
    self.voices = self.driver.getAllVoices()

  def _say(self, text):
    '''
    Loops until L{speaker}.say succeeds by returning a non-negative value. 

    @note: gnome-speech returns a negative value when there is a backlog of 
      events that must be processed by a GNOME.Speech.SpeechCallback before
      say will succeed
    @todo: PP: store the return value of say to correlate callback markers
    @param text: Text to say
    @type text: string
    '''
    while 1:
      try:
        text = text.encode('latin-1', 'ignore')
        sid = self.speaker.say(text)
      except (TypeError, UnicodeDecodeError), e:
        break
      if sid >= 0: break
      time.sleep(SAY_WAIT_RETRY)

  def close(self):
    '''
    Stops and closes the speech device.
    '''
    if self.speaker is not None:
      self.speaker.stop()
    del self.speaker
    del self.driver

  def sendStop(self, style=None):
    '''
    Stops speech immediately.
    
    @param style: Ignored
    @type style: L{AEOutput.Style}
    '''
    if self.speaker is not None:
      self.speaker.stop()
    self.buffer = []

  def sendTalk(self, style=None):
    '''
    Begins synthesis of the buffered text.
  
    @param style: Ignored
    @type style: L{AEOutput.Style}
    '''
    stream = []
    for text, style in self.buffer:
      if style.isDirty() or self.last_style != style:
        self._applyStyle(style)
        self._say(' '.join(stream))
        self.last_style = style
        stream = []
      if text is not None:
        stream.append(text)
    if stream:
      self._say(' '.join(stream))
    self.buffer = []

  def sendString(self, text, style):
    '''
    Adds the given string to the text to be synthesized. The string is buffered
    locally in this object since gnome-speech does not support buffering in
    the speech engine driver, even if the engine does support it.

    @param text: String to buffer on the device
    @type text: string
    @param style: Style with which this string should be output; None means no
      style change should be applied
    @type style: integer
    '''
    # just buffer the text since gnome-speech will try speaking it right away
    # actually send the text in sendTalk
    self.buffer.append((text, style.copy()))

  def isActive(self):
    '''
    Indicates whether the device is active meaning it is busy doing output or
    has text waiting to be output.

    @return: True when the speech device is synthesizing, False otherwise
    @rtype: boolean
    '''
    return self.speaker.isSpeaking()
   
  def sendIndex(self, style):
    '''
    Sends an index marker to the device driver. The driver should notify the
    device when the marker has been reached during output.
    
    @todo: PP: implement when index working

    @param style: Style indicating channel in which the marker will be appended
    @type style: L{AEOutput.Style}
    '''
    pass

  def createDistinctStyles(self, num_groups, num_layers):
    '''
    Creates distinct styles following the guidelines set forth in the base
    class. Only distinguishes groups in terms of voice, not layers. Assumes
    rate, pitch, and volume are exposed under those weakly specified, but
    typically used names by gnome-speech.
    
    @param num_groups: Number of sematic groups the requestor would like to
      represent using distinct styles
    @type num_groups: integer
    @param num_layers: Number of content origins (e.g. output originating from
      a background task versus the focus) the requestor would like to represent
      using distinct styles
    @type num_layers: integer   
    @return: Styles
    @rtype: list of L{AEOutput.Style}
    '''
    styles = []
    # bounds on the number of voices
    self.default_style.MaxVoice = len(self.voices)-1
    self.default_style.MinVoice = 0
    self.default_style.Voice = 0
    for i in range(0, num_groups*num_layers):
      # compute a valid voice number
      v_num = i % len(self.voices)
      # get that voice from the list of all voices
      v = self.voices[v_num]
      # create a speaker with that voice
      sp = self.driver.createSpeaker(v)
      # create a style object with relative rate, pitch, and volume
      s = GSpeechStyle(self.default_style)
      # store the current voice number
      s.Voice = v_num
      # iterate over the supported parameters and stick them on the style obj
      # as style attributes; this may give us more than our known default set
      # of styles, but it's OK because they'll just be ignored by LSR
      for param in sp.getSupportedParameters():
        cap_name = param.name.title()
        val = sp.getParameterValue(param.name)
        try:
          def_val = getattr(self.default_style, cap_name)
        except AttributeError:
          def_val = None
          
        if def_val is None:
          # put this value on the default if it does not already exist
          setattr(self.default_style, cap_name, val)
          val = 0
        elif val != def_val:
          # put value on the current style if it differs from the default
          if cap_name in GSpeechStyle.ADDITIVE:
            # make the value relative to the default
            val = val - def_val
        else:
          val = 0
        setattr(s, cap_name, val)

        # max and min should probably go in the default style; I guess they
        # could change per speaker, but that would be really weird
        setattr(self.default_style, 'Max%s' % cap_name, param.max) 
        setattr(self.default_style, 'Min%s' % cap_name, param.min)
      styles.append(s)
    return styles
