'''
Defines a class responsible for managing all L{Tier}s.

@author: Peter Parente
@author: Pete Brunet
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
import logging
import AEEvent, AEMonitor, LSRConstants
from Tier import Tier
from UIRegistrar import UIRegistrar, MONITOR, PERK
from LSRInterfaces import implements

log = logging.getLogger('Tier')

class TierManager(object):
  '''
  Creates L{Tier}s for all processes detectable by L{pyLinAcc}. Routes
  L{AEEvent}s to the active L{Tier} for handling by L{Task}s registered by
  L{Perk}s. Changes the active L{Tier} as the active top-level application 
  changes.
  
  @ivar tiers: All L{Tier}s created for running applications
  @type tiers: dictionary
  @ivar active_tier: L{Tier} for the application currently in focus
  @type active_tier: L{Tier}
  @ivar acc_eng: The Access Engine that started this manager
  @type acc_eng: L{AccessEngine}
  @ivar view_manager: Reference to the L{ViewManager} to be used to dynamically 
    change which events should be monitored based on the registered L{Task}s
  @type view_manager: L{ViewManager}
  '''
  def __init__(self, ae):
    '''
    Creates the empty dictionary for storing L{Tier}s and initializes the active
    tier to None. Stores a reference to L{AccessEngine}.
    
    @param ae: References to the L{AccessEngine} that created this manager
    @type ae: L{AccessEngine}
    '''
    self.tiers = {}
    self.active_tier = None
    self.acc_eng = ae
    self.view_manager = None
    self.monitors = AEMonitor.MonitorCollection()
    
  def init(self, view_man, **kwargs):
    '''
    Stores a references to the L{DeviceManager}. Called by L{AccessEngine} 
    at startup.
    
    @param view_man: Reference to the L{ViewManager} to be used to dynamically 
      change which events should be monitored based on the registered L{Task}s
    @type view_man: L{ViewManager}
    @param kwargs: References to managers 
    @type kwargs: dictionary
    '''
    self.view_manager = view_man
    # load all startup monitors
    reg = UIRegistrar()
    mons = reg.loadAssociated(MONITOR, self.acc_eng.getProfile())
    self.addMonitors(*mons)
    
  def getMonitors(self):
    '''
    Gets the collection of loaded L{AEMonitor}s.    
    
    @return: Collection of monitors
    @rtype: L{AEMonitor.MonitorCollection}
    '''
    return self.monitors
    
  def addMonitors(self, *monitors):
    '''
    Adds one or more L{AEMonitor.EventMonitor}s to the list of monitors to be 
    notified about events. 
    
    @param monitors: L{AEMonitor.EventMonitor}s to notify
    @type monitors: tuple of L{AEMonitor.EventMonitor}s
    '''
    self.monitors.add(AEMonitor.EventMonitor, AEEvent.Base.AccessEngineEvent,
                      monitors)
    
  def removeAllMonitors(self):
    '''    
    Removes all L{AEMonitor.EventMonitor}s from the list of monitors to be
    notified about events. Closes the monitors when removed.
    '''
    self.monitors.clear()
        
  def showTask(self, event, perk_name, task_name, propogate):
    '''    
    Informs L{AEMonitor}s added via L{addMonitors} of a L{Task} handling an 
    event. Calls show on the monitors collection as soon as the L{Task} has 
    finished executing.
    
    @param event: An L{AEEvent} object
    @type event: L{AEEvent}
    @param perk_name: Name of the Perk in which the Task is registered
    @type perk_name: string
    @param task_name: Name of the Task
    @type task_name: string
    @param propogate: Was the event consumed (False) or allowed to propogate 
      (True)?
    @type propogate: boolean
    '''
    self.monitors.show(event=event, perk_name=perk_name, 
                       task_name=task_name, propogate=propogate)
        
  def showEvent(self, event, tier_name):
    '''    
    Informs L{AEMonitor}s added via L{addMonitors} of an event. Calls
    show on the monitors collection as soon as an event is accepted for
    processing by a L{Tier}.
    
    @param event: An L{AEEvent} object
    @type event: L{AEEvent}
    '''
    self.monitors.show(event=event, tier_name=tier_name)
   
  def close(self):
    '''Does nothing.'''
    pass
  
  def createTier(self, name, aid):
    '''
    Creates a new L{Tier} using the application name and ID as a hash code.

    @param name: Name of the now active application
    @type name: string
    @param aid: Unique ID for the application associated with the L{Tier}
    @type aid: hashable
    @return: The new L{Tier} that was created
    @rtype: L{Tier}
    '''
    # create the Tier object and store it
    tier = Tier(self, name, aid)
    self.tiers[aid] = tier
    # create a UIRegistrar instance
    reg = UIRegistrar()
    # load all default and application specific Perks, push them onto the Tier
    one = reg.loadAssociated(PERK, self.acc_eng.getProfile(), name)
    any = reg.loadAssociated(PERK, self.acc_eng.getProfile())
    tier.pushPerk(self.acc_eng, *(any+one))
    return tier
    
  def removeTier(self, aid):
    '''
    Removes an existing L{Tier}.
    
    @param aid: Unique ID for the application associated with the L{Tier}
    @type aid: hashable
    '''
    del self.tiers[aid]
    #log.debug('removed Tier %s', aid)
    
  def switchTier(self, name, aid):
    '''
    Switches the active L{Tier} based on the current view's root accessible. The
    view's root accessible is queried for information that can uniquely identify
    the L{Tier} to use. 
    
    If a L{Tier} already exists for the given view, it is activated. If one
    does not exist, a L{Tier} is created using L{TierManager.createTier} and 
    made active.
    
    @param name: Name of the now active application
    @type name: string
    @param aid: Unique ID for the application associated with the L{Tier}
    @type aid: hashable
    '''
    try:
      tier = self.tiers[aid]
    except KeyError:
      # create a new tier for this application
      tier = self.createTier(name, aid)
    # make the tier active
    self.active_tier = tier
    
  def setEventInterest(self, kind, wants):
    '''
    Informs the L{ViewManager} of a change in interest in L{AEEvent}s by some
    L{Tier}. Used to optimize which system events are monitored in order to
    improve performance.
    
    @param kind: Kind of L{AEEvent} of interest to a L{Task}
    @type kind: L{AEEvent} class
    @param wants: Does a L{Perk} want an event (i.e. a L{Task} is registering 
      for it or no longer want an event (i.e. a L{Task} is unregistering for 
      it)?
    @type wants: boolean
    '''
    self.view_manager.setEventInterest(kind, wants)
    
  def manageEvent(self, event):
    '''
    Dispatches an L{AEEvent} to the L{Tier} to be handled by its L{Perk}s.
    Makes the final determination of on what layer the event occurred. Events
    from a focused source are already marked as L{AEEvent.FOCUS_LAYER}. Events
    from a unfocused source in the active L{Tier} are marked as
    L{AEEvent.TIER_LAYER}. All other events for which a L{Tier} exists to handle
    them are marked as L{AEEvent.BACKGROUND_LAYER}.
    
    @param event: Event to dispatch to the L{Tier}
    @type event: L{AEEvent.Base.AccessEngineEvent}
    '''
    # opt: check if the active tier wants this kind of event so we don't waste
    # time completing PORs when we request the event layer
    if self.active_tier.wantsEvent(event):
      # get the current task layer and completes the event POR
      task_layer = event.getLayer()
    else:
      # otherwise, assume background
      task_layer = AEEvent.BACKGROUND_LAYER
    
    if task_layer == AEEvent.FOCUS_LAYER:
      tier = self.active_tier  # if in the focus layer, use the active tier
    else: # event layer is bkgd:
      # this may throw a lookup error if the event source is dead, but we simply
      # ignore it at a higher level
      aid = event.getAppID()
      try:
        tier = self.tiers[aid]
      except KeyError:
        return
      if tier is self.active_tier:
        # the event came from an unfocused control in the active tier
        event.setLayer(AEEvent.TIER_LAYER)
    # let the tier manage the event
    tier.manageEvent(self.acc_eng, event) 
   
  def manageGesture(self, event):
    '''
    Dispatches an L{AEEvent} to the active L{Tier} so that it can determine 
    which registered L{Task}, if any, should be executed in response to some
    L{AEInput.Gesture}.

    @param event: Event to dispatch to the L{Tier}
    @type event: L{AEEvent.Base.AccessEngineEvent}
    '''
    # quit immediately if there is no active Tier
    if self.active_tier is not None:
      # let the Tier manage the event
      self.active_tier.manageGesture(self.acc_eng, event)
      
  def manageChooser(self, event):
    '''
    Dispatches an L{AEEvent.ChooserChange} to the active L{Tier} so that it can
    determine which registered L{Task}, if any, should be executed in response 
    to a change in the chooser such as its completion or its cancellation.

    @param event: Event to dispatch to the L{Tier}
    @type event: L{AEEvent.Base.AccessEngineEvent}
    '''
    # quit immediately if there is no active Tier
    if self.active_tier is not None:
      # let the Tier manage the event
      self.active_tier.manageGesture(self.acc_eng, event)
      
