'''
Defines a class representing a configurable setting in a L{AEState} object.

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''
class Group(list):
  '''  
  Named collection of related L{Setting}s. Provides the L{save} and L{restore}
  methods which act on all members in this group.
  
  @ivar context: Object whose properties are contained within this group
  @type context: L{AEState.AEState}
  @ivar name: Name of the group
  @type name: string
  '''
  def __init__(self, context, name=None):
    '''
    Stores the L{Group} context and its name.
    
    @param context: Object whose properties are contained within this group
    @type context: L{AEState.AEState}
    @param name: Name of the group
    @type name: string
    '''
    self.context = context    
    self.name = name
    
  def __str__(self):
    return 'Group %s: %s' % (self.name, list.__str__(self))
    
  def newGroup(self, name):
    '''
    Adds a subgroup to this group with the given name.
    
    @param name: Name of the new group
    @type name: string
    '''
    grp = Group(self.context, name)
    self.append(grp)
    return grp

  def newInt(self, variable, label):
    '''
    Adds a new L{IntSetting} to this group.
    
    @param variable: Name of the variable in the L{context} having the integer
      value
    @type variable: string
    @param label: Description of the new L{Setting}
    @type label: string
    '''
    self.append(IntSetting(self.context, variable, label))
  
  def newFloat(self, variable, label):
    '''
    Adds a new L{FloatSetting} to this group.
    
    @param variable: Name of the variable in the L{context} having the float
      value
    @type variable: string
    @param label: Description of the new L{Setting}
    @type label: string
    '''
    self.append(FloatSetting(self.context, variable, label))
    
  def newString(self, variable, label):
    '''
    Adds a new L{StringSetting} to this group.
    
    @param variable: Name of the variable in the L{context} having the string
      value
    @type variable: string
    @param label: Description of the new L{Setting}
    @type label: string
    '''
    self.append(StringSetting(self.context, variable, label))
  
  def newBool(self, variable, label):
    '''
    Adds a new L{BoolSetting} to this group.
    
    @param variable: Name of the variable in the L{context} having the boolean
      value
    @type variable: string
    @param label: Description of the new L{Setting}
    @type label: string
    '''
    self.append(BoolSetting(self.context, variable, label))

  def newRange(self, variable, label, min, max):
    '''
    Adds a new L{RangeSetting} to this group.
    
    @param variable: Name of the variable in the L{context} having the numeric
      value
    @type variable: string
    @param label: Description of the new L{Setting}
    @type label: string
    @param min: Minimum value in the range
    @type min: number
    @param max: Maximum value in the range
    @type max: number
    '''
    self.append(RangeSetting(self.context, variable, label, min, max))
  
  def newChoice(self, variable, label, choices):
    '''
    Adds a new L{ChoiceSetting} to this group.
    
    @param variable: Name of the variable in the L{context} having the value
    @type variable: string
    @param label: Description of the new L{Setting}
    @type label: string
    @param choices: Collection of keys or values that can be used for this 
      setting
    @type choices: list or dictionary
    '''
    self.append(ChoiceSetting(self.context, variable, label, choices))

  def newSplitList(self, variable, label, delim):
    '''
    Adds a new L{SplitListSetting} to this group.
    
    @param variable: Name of the variable in the L{context} having the value
    @type variable: string
    @param label: Description of the new L{Setting}
    @type label: string
    @param delim: Delimiter of list items
    @type delim: string
    '''
    self.append(ChoiceSetting(self.context, variable, label, delim))

  def save(self):
    '''Invokes L{Setting.save} on all settings in the group.'''
    for s in self: s.save()

  def restore(self):
    '''Invokes L{Setting.restore} on all settings in the group.'''
    for s in self: s.restore()

class Setting(object):
  '''
  A proxy for a value in a L{AEState.AEState} object. Provides a description
  of the value. Defines methods for caching the current value and restoring it
  later.
  
  @ivar context: Object in which the variable containing the setting resides
  @type context: L{AEState.AEState}
  @ivar variable: Name of the variable to proxy
  @type variable: string
  @ivar label: Description of the variable
  @type label: string
  @ivar cached: Cached value from the last L{save} invocation
  @type cached: object
  '''
  def __init__(self, context, variable, label):
    '''
    Initializes the instance variables.
    
    @param context: Object in which the variable containing the setting resides
    @type context: L{AEState.AEState}
    @param variable: Name of the variable to proxy
    @type variable: string
    @param label: Description of the variable
    @type label: string
    '''
    self.context = context
    self.variable = variable
    self.label = label
    self.cached = None
    
  def getValue(self):
    '''
    @return: Current value of the proxied variable
    @rtype: object
    '''
    return getattr(self.context, self.variable)
    
  def setValue(self, val):
    '''
    @param val: New value to store in the proxied variable
    @type val: object
    '''
    setattr(self.context, self.variable, val)
    
  value = property(getValue, setValue)
    
  def save(self):
    '''Saves the current variable value.'''
    self.cached = self.value
    
  def restore(self):
    '''Restores the previously saved variable value.'''
    if self.cached is not None:
      self.value = self.cached
    self.cached = None
    
class IntSetting(Setting):
  '''Represents an integer setting.'''
  def setValue(self, val):
    Setting.setValue(self, int(val))
    
  value = property(Setting.getValue, setValue)
    
class BoolSetting(Setting):
  '''Represents a boolean setting.'''
  def setValue(self, val):
    Setting.setValue(self, bool(val))
    
  value = property(Setting.getValue, setValue)

class StringSetting(Setting):
  '''Represents a string setting.'''
  def setValue(self, val):
    Setting.setValue(self, str(val))
    
  value = property(Setting.getValue, setValue)
    
class FloatSetting(Setting):
  '''Represents a floating point setting.'''
  def setValue(self, val):
    Setting.setValue(self, float(val))
    
  value = property(Setting.getValue, setValue)
    
class RangeSetting(Setting):
  '''
  Represents a numeric range setting.
  
  @ivar min: Minimum value in the range.
  @type min: number
  @ivar max: Maximum value in the range
  @type max: number
  '''
  def __init__(self, context, variable, label, min, max):
    Setting.__init__(self, context, variable, label)
    self.min = min
    self.max = max
  
  def setValue(self, val):
    if val < self.min or val > self.max:
      raise ValueError
    Setting.setValue(self, int(val))
    
  value = property(Setting.getValue, setValue)
    
class ChoiceSetting(Setting):
  '''
  Represents a one of many or many of many choice setting.
  
  @ivar choices: Collection of choice keys or values
  @type choices: list or dictionary
  '''
  def __init__(self, context, variable, label, choices):
    Setting.__init__(self, context, variable, label)
    self.choices = choices
    
  def setValue(self, val):
    if val not in self.choices:
      raise ValueError
    Setting.setValue(self, val)
    
  value = property(Setting.getValue, setValue)
  
class SplitListSetting(Setting):
  '''
  Represents a list items built by splitting a delimited string.
  
  @ivar delim: List item delimiter
  @type delim: string
  '''
  def __init__(self, context, variable, label, delim):
    Setting.__init__(self, context, variable, label)
    self.delim = delim
    
  def getValue(self):
    l = Setting.getValue(self)
    return self.delim.join(l)
  
  def setValue(self, val):
    s = val.split(self.delim)
    Setting.setValue(self, s)
  