'''
Defines a GTK dialog for buffering L{AEEvent}s and how L{Perk}s and L{Task}s 
handle them.

@author: Peter Parente
@organization: IBM Corporation
@copyright: Copyright (c) 2005, 2006 IBM Corporation
@license: Common Public License 1.0

All rights reserved. This program and the accompanying materials are made 
available under the terms of the Common Public License v1.0 which accompanies
this distribution, and is available at
U{http://www.opensource.org/licenses/cpl1.0.php}
'''

import AEEvent
from GTKEventDialog import GTKEventDialog
from i18n import _

class TaskMonitor(GTKEventDialog):
  '''
  Buffers text describing L{AEEvent}s and how L{Perk}s and L{Task}s handle them.
  '''
  def getName(self):
    '''
    Gets the localized name of this monitor.
    
    @return: Monitor name
    @rtype: string
    '''
    return _('Task monitor')
  
  def getEventNames(self):
    '''
    Gets the event categories to be displayed in the View menu for filtering.
    
    @return: Event categories
    @rtype: list of string
    '''
    return AEEvent.getNames()
  
  def getEventDefaults(self):
    '''
    Gets the default event categories to check in the View menu.
    
    @return: Event categories
    @rtype: list of string
    '''
    return AEEvent.getDefaults()
  
  def getEventType(self):
    '''
    Gets the L{AEEvent.Base.AccessEngineEvent} base class to indicate the type 
    of events this monitor wants to buffer.
    
    @return: Base type of the event this monitor should buffer
    @rtype: L{AEEvent.Base.AccessEngineEvent} class
    '''
    return AEEvent.Base.AccessEngineEvent    
      
  def show(self, event, tier_name='', perk_name='', task_name='', propogate='', 
           **kwargs):
    '''
    Buffers additional details about how the L{AEEvent} was handled by various
    L{Perk}s and L{Task}s. The event is only provided as a means of filtering 
    information.
    
    @param event: Event to filter on
    @type event: object
    @param tier_name: Name of the L{Tier} that handled the event
    @type tier_name: string    
    @param perk_name: Name of the L{Perk} that handled the event
    @type perk_name: string
    @param task_name: Name of the L{Task} that handled the event
    @type task_name: string
    @param propogate: Was the event allowed to propogate to other L{Perk}s?
    @type propogate: boolean
    @raise IOError: When the monitor is no longer accepting data to buffer 
    '''
    if not self.isInitialized():
      raise IOError
    # assume the class name is the one we used to register the event type
    elif self._isShown(event.__class__.__name__):
      if tier_name:
        self._queueText(str(event))
        self._queueText('\ttier: '+tier_name)
      else:
        self._queueText('\t%s.%s' % (perk_name, task_name))
        self._queueText('\t\tpropogate: ' + str(propogate))
