/*
 *  gnome-print-page-selector.c:
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Authors:
 *    Lutz Mueller <lutz@users.sourceforge.net>
 *
 *  Copyright (C) 2005 Lutz Mueller
 *
 */
#include <config.h>

#include <libgnomeprintui/gnome-print-dialog.h>
#include <libgnomeprintui/gnome-print-page-selector.h>
#include <libgnomeprintui/gnome-print-i18n.h>

#include <libgnomeprint/gnome-print-config.h>
#include <libgnomeprint/gnome-print-filter.h>

#include <libgnomeprint/private/gpa-node.h>
#include <libgnomeprint/private/gpa-utils.h>
#include <libgnomeprint/private/gnome-print-private.h>
#include <libgnomeprint/private/gnome-print-config-private.h>

#include <gtk/gtkframe.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtktable.h>
#include <gtk/gtkradiobutton.h>
#include <gtk/gtkspinbutton.h>
#include <gtk/gtkentry.h>
#include <gtk/gtkcheckbutton.h>

#include <gdk/gdkkeysyms.h>

#include <string.h>

struct _GnomePrintPageSelector {
	GtkFrame parent;

	GnomePrintConfig *config;
	guint current;

	GtkWidget *r_all, *r_from_to, *r_selection, *r_current;
	GtkWidget *c_even, *c_odd;
	GtkAdjustment *a_from, *a_to;
	GtkWidget *e_selection;

	gboolean loading, saving;
};

struct _GnomePrintPageSelectorClass {
	GtkFrameClass parent_class;
};

static GtkFrameClass *parent_class = NULL;

enum {
	PROP_0,
	PROP_CONFIG,
	PROP_CURRENT,
	PROP_TOTAL
};

static void
gnome_print_page_selector_load_config (GnomePrintPageSelector *ps)
{
	gchar *description;
	GnomePrintFilter *f;
	guint first, last, skip;
	gboolean collect;
	GValueArray *pages = NULL;

	g_return_if_fail (GNOME_IS_PRINT_PAGE_SELECTOR (ps));

	if (!ps->config || ps->loading || ps->saving)
		return;

	description = (gchar *) gnome_print_config_get (ps->config,
			 (const guchar *) "Settings.Output.Job.Filter");
	if (!description)
		return;

	f = gnome_print_filter_new_from_description (description, NULL);
	g_free (description);
	if (!f)
		return;

	while (strcmp ("GnomePrintFilterSelect", G_OBJECT_TYPE_NAME (G_OBJECT (f))) &&
			(gnome_print_filter_count_successors (f) == 1)) {
		f = gnome_print_filter_get_successor (f, 0);
		g_object_ref (G_OBJECT (f));
		g_object_unref (G_OBJECT (gnome_print_filter_get_predecessor (f, 0)));
	}

	if (strcmp ("GnomePrintFilterSelect", G_OBJECT_TYPE_NAME (G_OBJECT (f)))) {
		if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ps->r_all)))
			g_object_set (G_OBJECT (ps->r_all), "active", TRUE, NULL);
		g_object_unref (G_OBJECT (f));
		return;
	}

	g_object_get (G_OBJECT (f), "first", &first, "last", &last, "skip", &skip,
			"collect", &collect, NULL);
	if (collect || (skip > 1)) {
		if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ps->r_all)))
			g_object_set (G_OBJECT (ps->r_all), "active", TRUE, NULL);
		g_object_unref (G_OBJECT (f));
		return;
	}

	ps->loading = TRUE;
	g_object_get (G_OBJECT (f), "pages", &pages, NULL);
	if (pages) {
		guint i;
		gchar *s, *str = NULL;

		if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ps->r_selection)))
			g_object_set (G_OBJECT (ps->r_selection), "active", TRUE, NULL);
		for (i = 0; i < pages->n_values; i++) {
			gboolean p, n, c;

			p = i ? g_value_get_boolean (g_value_array_get_nth (pages, i - 1)) : FALSE;
			n = (i + 1 < pages->n_values) ? g_value_get_boolean (g_value_array_get_nth (pages, i + 1)) : FALSE;
			c = g_value_get_boolean (g_value_array_get_nth (pages, i));

			if ((p && c && n) || (p && !c && !n) ||
					(!p && !c && n) || (!p && !c && !n) || (p && !c && n))
				continue;
			if (p && c && !n) {
				s = g_strdup_printf ("%s-%i", str, i + 1);
				g_free (str);
				str = s;
				continue;
			}
			if (p && !c && n) {
				s = g_strdup_printf ("%s,", str);
				g_free (str);
				str = s;
				continue;
			}
			if ((!p && c && n) || (!p && c && !n)) {
				s = g_strdup_printf ("%s%s%i", str ? str : "", str ? "," : "", i + 1);
				g_free (str);
				str = s;
				continue;
			}
		}
		if (!str)
			gtk_entry_set_text (GTK_ENTRY (ps->e_selection), "");
		else {
			if (strcmp (str, gtk_entry_get_text (GTK_ENTRY (ps->e_selection))))
				gtk_entry_set_text (GTK_ENTRY (ps->e_selection), str);
			g_free (str);
		}
		g_value_array_free (pages);
	} else if ((first == 0) && (last == G_MAXUINT)) {
		if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ps->r_all)))
			g_object_set (G_OBJECT (ps->r_all), "active", TRUE, NULL);
	} else if ((ps->current > 0) &&
			(first == ps->current - 1) &&
			(last == ps->current - 1)) {
		if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ps->r_current)))
			g_object_set (G_OBJECT (ps->r_current), "active", TRUE, NULL);
	} else {
		if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ps->r_from_to)))
			g_object_set (G_OBJECT (ps->r_from_to), "active", TRUE, NULL);
		if ((guint) ps->a_from->value != first)
			g_object_set (G_OBJECT (ps->a_from), "value", (gdouble) first, NULL);
		if ((guint) ps->a_to->value != last)
			g_object_set (G_OBJECT (ps->a_to), "value", (gdouble) last, NULL);
	}
	if (skip) {
		g_object_set (G_OBJECT (ps->c_even), "active", (gboolean) (first & 1), NULL);
		g_object_set (G_OBJECT (ps->c_odd), "active", (gboolean) !(first & 1), NULL);
	}
	ps->loading = FALSE;
}

typedef enum {
	STATE_BEGIN,
	STATE_NUMBER_FROM,
	STATE_NUMBER_FROM_TO,
	STATE_NUMBER_BEGIN_TO,
	STATE_BEGIN_TO,
	STATE_FROM_TO
} ParseState;

static GArray *
gnome_print_page_selector_get_array (GnomePrintPageSelector *ps)
{
	const gchar *s;
	guint i, n_from = 0, n_to = 0, m;
	ParseState state = STATE_BEGIN;
	gboolean err = FALSE;
	GArray *a;

	g_return_val_if_fail (GNOME_IS_PRINT_PAGE_SELECTOR (ps), NULL);

	/* Total number of pages */
	m = (ps->a_to->upper < 1.) ? 1000 : (guint) ps->a_to->upper;

	a = g_array_new (FALSE, TRUE, sizeof (gboolean));
	s = gtk_editable_get_chars (GTK_EDITABLE (ps->e_selection), 0, -1);
	for (i = 0; !err && (i < strlen (s)); i++) {
		if ((s[i] == '0') || (s[i] == '1') || (s[i] == '2') ||
		    (s[i] == '3') || (s[i] == '4') || (s[i] == '5') ||
		    (s[i] == '6') || (s[i] == '7') || (s[i] == '8') ||
		    (s[i] == '9')) {
			switch (state) {
			case STATE_BEGIN:
				state = STATE_NUMBER_FROM;
				n_from = atoi (&s[i]);
				if (!n_from) err = TRUE;
				break;
			case STATE_NUMBER_FROM:
				n_from = n_from * 10 + atoi (&s[i]);
				if (n_from > G_MAXUINT16) err = TRUE;
				break;
			case STATE_NUMBER_BEGIN_TO:
			case STATE_NUMBER_FROM_TO:
				n_to = n_to * 10 + atoi (&s[i]);
				if (n_to > G_MAXUINT16) err = TRUE;
				break;
			case STATE_FROM_TO:
				state = STATE_NUMBER_FROM_TO;
				n_to = atoi (&s[i]);
				if (!n_to) err = TRUE;
				break;
			case STATE_BEGIN_TO:
				state = STATE_NUMBER_BEGIN_TO;
				n_to = atoi (&s[i]);
				if (!n_to) err = TRUE;
				break;
			}
		} else if (s[i] == '-') {
			switch (state) {
			case STATE_BEGIN:
				state = STATE_BEGIN_TO;
				break;
			case STATE_NUMBER_FROM_TO:
			case STATE_NUMBER_BEGIN_TO:
			case STATE_BEGIN_TO:
			case STATE_FROM_TO:
				err = TRUE;
				break;
			case STATE_NUMBER_FROM:
				state = STATE_FROM_TO;
				break;
			}
		} else if ((s[i] == ',')) {
			switch (state) {
			case STATE_BEGIN:
				break;
			case STATE_FROM_TO:
				if (a->len < m)
					g_array_set_size (a, m);
				for (i = n_from - 1; i < m; i++)
					g_array_index (a, gboolean, i) = TRUE;
				break;
			case STATE_BEGIN_TO:
				err = TRUE;
				break;
			case STATE_NUMBER_FROM:
				state = STATE_BEGIN;
				if (a->len < n_from)
					g_array_set_size (a, n_from);
				g_array_index (a, gboolean, n_from - 1) = TRUE;
				break;
			case STATE_NUMBER_FROM_TO:
				state = STATE_BEGIN;
				if (a->len < MAX (n_from, n_to))
					g_array_set_size (a, MAX (n_from, n_to));
				for (i = MIN (n_from, n_to) - 1; i < MAX (n_from, n_to); i++)
					g_array_index (a, gboolean, i) = TRUE;
				break;
			case STATE_NUMBER_BEGIN_TO:
				state = STATE_BEGIN;
				if (a->len < n_to)
					g_array_set_size (a, n_to);
				for (i = 0; i < n_to; i++)
					g_array_index (a, gboolean, i) = TRUE;
				break;
			}
		} else
			err = TRUE;
	}

	if (!err) switch (state) {
	case STATE_BEGIN:
		break;
	case STATE_BEGIN_TO:
		err = TRUE;
		break;
	case STATE_FROM_TO:
		if (a->len < m)
			g_array_set_size (a, m);
		for (i = n_from - 1; i < m; i++)
			g_array_index (a, gboolean, i) = TRUE;
		break;
	case STATE_NUMBER_FROM:
		if (a->len < n_from)
			g_array_set_size (a, n_from);
		g_array_index (a, gboolean, n_from - 1) = TRUE;
		break;
	case STATE_NUMBER_FROM_TO:
		if (a->len < MAX (n_from, n_to))
			g_array_set_size (a, MAX (n_from, n_to));
		for (i = MIN (n_from, n_to) - 1; i < MAX (n_from, n_to); i++)
			g_array_index (a, gboolean, i) = TRUE;
		break;
	case STATE_NUMBER_BEGIN_TO:
		if (a->len < n_to)
			g_array_set_size (a, n_to);
		for (i = 0; i < n_to; i++)
			g_array_index (a, gboolean, i) = TRUE;
		break;
	}

	if (err) {
		i = gtk_editable_get_position (GTK_EDITABLE (ps->e_selection));
		gtk_editable_delete_text (GTK_EDITABLE (ps->e_selection), i, i + 1);
	}

	return a;
}

static void
gnome_print_page_selector_get_before_and_after (GnomePrintPageSelector *ps,
		gchar **s_before, gchar **s_after)
{
	gchar *description;
	GnomePrintFilter *f, *before = NULL, *after = NULL;
	gboolean collect = FALSE;
	guint i, skip = 0;

	g_return_if_fail (GNOME_IS_PRINT_PAGE_SELECTOR (ps));
	g_return_if_fail (s_before && s_after);

	*s_before = NULL;
	*s_after = NULL;

	description = (gchar *) gnome_print_config_get (ps->config,
			(const guchar *) "Settings.Output.Job.Filter");
	if (!description)
		return;

	f = gnome_print_filter_new_from_description (description, NULL);
	g_free (description);
	if (!f)
		return;

	while (strcmp ("GnomePrintFilterSelect", G_OBJECT_TYPE_NAME (G_OBJECT (f))) &&
			(gnome_print_filter_count_successors (f) == 1)) {
		if (!before)
			before = f;
		f = gnome_print_filter_get_successor (f, 0);
	}
	if (before) {
		g_object_ref (G_OBJECT (f));
		gnome_print_filter_remove_predecessor (f,
				gnome_print_filter_get_predecessor (f, 0));
		*s_before = gnome_print_filter_description (before);
		g_object_unref (G_OBJECT (before));
	}

	if (strcmp (G_OBJECT_TYPE_NAME (G_OBJECT (f)), "GnomePrintFilterSelect")) {
		*s_after = gnome_print_filter_description (f);
		g_object_unref (G_OBJECT (f));
		return;
	}

	g_object_get (G_OBJECT (f), "collect", &collect, "skip", &skip, NULL);
	if (collect || skip > 1) {
		*s_after = gnome_print_filter_description (f);
		g_object_unref (G_OBJECT (f));
		return;
	}

	switch (gnome_print_filter_count_successors (f)) {
	case 0:
		return;
	case 1:
		*s_after = gnome_print_filter_description (
				gnome_print_filter_get_successor (f, 0));
		g_object_unref (G_OBJECT (f));
		return;
	default:
		after = g_object_new (GNOME_TYPE_PRINT_FILTER, NULL);
		for (i = 0; i < gnome_print_filter_count_successors (f); i++) {
			GnomePrintFilter *s = gnome_print_filter_get_successor (f, i);

			gnome_print_filter_add_filter (after, s);
			gnome_print_filter_remove_predecessor (s, f);
		}
		g_object_unref (G_OBJECT (f));
		*s_after = gnome_print_filter_description (after);
		g_object_unref (G_OBJECT (after));
		return;
	}
}

static void
gnome_print_page_selector_save_config (GnomePrintPageSelector *ps)
{
	gchar *description = NULL, *before = NULL, *after = NULL, *str;
	GnomePrintFilter *filter = NULL;

	g_return_if_fail (GNOME_IS_PRINT_PAGE_SELECTOR (ps));

	if (ps->loading || !ps->config || ps->loading)
		return;

	gnome_print_page_selector_get_before_and_after (ps, &before, &after);

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ps->r_all))) {

		/* All */
		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ps->c_even)))
			filter = gnome_print_filter_new_from_module_name ("select",
					"first", 1, "skip", 1, NULL);
		else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ps->c_odd)))
			filter = gnome_print_filter_new_from_module_name ("select",
					"first", 0, "skip", 1, NULL);
		else
			filter = NULL;
	} else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ps->r_current))) {

		/* Current */
		filter = gnome_print_filter_new_from_module_name ("select",
				"first", ps->current - 1, "last", ps->current - 1, NULL);
	} else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ps->r_selection))) {
		GArray *a;
		GValueArray *va = NULL;

		/* Selection */
		a = gnome_print_page_selector_get_array (ps);
		if (a) {
			GValue v = {0,};
			guint i;

			g_value_init (&v, G_TYPE_BOOLEAN);
			va = g_value_array_new (a->len);
			for (i = 0; i < a->len; i++) {
				g_value_set_boolean (&v, g_array_index (a, gboolean, i));
				g_value_array_append (va, &v);
			}
			g_array_free (a, TRUE);
			g_value_unset (&v);
		}
		filter = gnome_print_filter_new_from_module_name ("select",
				"pages", va, NULL);
	} else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ps->r_from_to))) {
		guint first = ps->a_from->value, last = ps->a_to->value, skip = 0;

		/* From ... to ... */
		if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ps->c_even))) {
			skip = 1;
			if (first & 1) first++;
		} else if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ps->c_odd))) {
			skip = 1;
			if (!(first & 1)) first++;
		}
		filter = gnome_print_filter_new_from_module_name ("select",
				"first", first, "last", last, "skip", skip, NULL);
	}

	if (filter) {
		description = gnome_print_filter_description (filter);
		g_object_unref (G_OBJECT (filter));
	}
	str = g_strdup_printf ("%s%s%s%s%s",
			before ? before : "", 
			(before && description) ? " ! " : "",
			description ? description : "",
			((before && after) || (description && after)) ? " ! " : "",
			after ? after : "");
	if (description)
		g_free (description);
	if (before)
		g_free (before);
	if (after)
		g_free (after);
	ps->saving = TRUE;
	gnome_print_config_set (ps->config,
			(const guchar *) "Settings.Output.Job.Filter",
			(const guchar *) str);
	ps->saving = FALSE;
	g_free (str);
}

static void
gnome_print_page_selector_get_property (GObject *object, guint n, GValue *v,
		GParamSpec *pspec)
{
	GnomePrintPageSelector *ps = GNOME_PRINT_PAGE_SELECTOR (object);

	switch (n) {
	case PROP_CONFIG:
		g_value_set_object (v, ps->config);
		break;
	case PROP_CURRENT:
		g_value_set_uint (v, ps->current);
		break;
	case PROP_TOTAL:
		g_value_set_uint (v, (guint) ps->a_to->upper);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, n, pspec);
	}
}

static void
on_filter_modified (GPANode *node, guint flags, GnomePrintPageSelector *ps)
{
	gnome_print_page_selector_load_config (ps);
}

static void
gnome_print_page_selector_set_property (GObject *object, guint n,
		const GValue *v, GParamSpec *pspec)
{
	GnomePrintPageSelector *ps = GNOME_PRINT_PAGE_SELECTOR (object);

	switch (n) {
	case PROP_CONFIG:
		if (ps->config)
			g_object_unref (G_OBJECT (ps->config));
		ps->config = g_value_get_object (v);
		if (ps->config) {
			GPANode *node;

			g_object_ref (G_OBJECT (ps->config));
			node = gpa_node_get_child_from_path (
					GNOME_PRINT_CONFIG_NODE (ps->config),
					(const guchar *) "Settings.Output.Job.Filter");
			if (node)
				g_signal_connect (G_OBJECT (node), "modified",
						G_CALLBACK (on_filter_modified), ps);
		}
		gnome_print_page_selector_load_config (ps);
		break;
	case PROP_CURRENT:
		ps->current = g_value_get_uint (v);
		gtk_widget_set_sensitive (ps->r_current, ps->current);
		break;
	case PROP_TOTAL:
		ps->a_to->upper = (gdouble) g_value_get_uint (v);
		ps->a_from->upper = (gdouble) g_value_get_uint (v);
		gtk_adjustment_changed (ps->a_to);
		gtk_adjustment_changed (ps->a_from);
		if (ps->a_to->value > ps->a_to->upper) {
			ps->a_to->value = ps->a_to->upper;
			gtk_adjustment_value_changed (ps->a_to);
		}
		if (ps->a_from->value > ps->a_from->upper) {
			ps->a_from->value = ps->a_from->upper;
			gtk_adjustment_value_changed (ps->a_from);
		}
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, n, pspec);
	}
}

static void
gnome_print_page_selector_finalize (GObject *object)
{
	GnomePrintPageSelector *ps = GNOME_PRINT_PAGE_SELECTOR (object);

	if (ps->config) {
		g_object_unref (G_OBJECT (ps->config));
		ps->config = NULL;
	}
}

static void
gnome_print_page_selector_class_init (GnomePrintPageSelectorClass *klass)
{
	GObjectClass *object_class = (GObjectClass *) klass;

	parent_class = g_type_class_peek_parent (klass);

	object_class->get_property = gnome_print_page_selector_get_property;
	object_class->set_property = gnome_print_page_selector_set_property;
	object_class->finalize     = gnome_print_page_selector_finalize;

	g_object_class_install_property (object_class, PROP_CONFIG,
			g_param_spec_object ("config", _("Configuration"), _("Configuration"),
				GNOME_TYPE_PRINT_CONFIG, G_PARAM_READWRITE));
	g_object_class_install_property (object_class, PROP_CURRENT,
			g_param_spec_uint ("current", _("Current page"), _("Current page"),
				0, G_MAXUINT, 0, G_PARAM_READWRITE));
	g_object_class_install_property (object_class, PROP_TOTAL,
			g_param_spec_uint ("total", _("Number of pages"), _("Number of pages"),
				0, G_MAXUINT, 0, G_PARAM_READWRITE));
}

static void
on_even_toggled (GtkToggleButton *b, GnomePrintPageSelector *ps)
{
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ps->c_even)) && 
			gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ps->c_odd)))
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (ps->c_odd), FALSE);
	gnome_print_page_selector_save_config (ps);
}

static void
on_odd_toggled (GtkToggleButton *b, GnomePrintPageSelector *ps)
{
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ps->c_even)) &&
			gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ps->c_odd)))
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (ps->c_even), FALSE);
	gnome_print_page_selector_save_config (ps);
}

static void
on_all_toggled (GtkToggleButton *b, GnomePrintPageSelector *ps)
{
	gtk_widget_set_sensitive (ps->c_even, TRUE);
	gtk_widget_set_sensitive (ps->c_odd, TRUE);
	gnome_print_page_selector_save_config (ps);
}

static void
on_current_toggled (GtkToggleButton *b, GnomePrintPageSelector *ps)
{
	gtk_widget_set_sensitive (ps->c_even, FALSE);
	gtk_widget_set_sensitive (ps->c_odd, FALSE);
	gnome_print_page_selector_save_config (ps);
}

static void
on_from_to_toggled (GtkToggleButton *b, GnomePrintPageSelector *ps)
{
	gtk_widget_set_sensitive (ps->c_even, TRUE);
	gtk_widget_set_sensitive (ps->c_odd, TRUE);
	gnome_print_page_selector_save_config (ps);
}

static void
on_selection_toggled (GtkToggleButton *b, GnomePrintPageSelector *ps)
{
	gtk_widget_set_sensitive (ps->c_even, FALSE);
	gtk_widget_set_sensitive (ps->c_odd, FALSE);
	gnome_print_page_selector_save_config (ps);
}

static void
on_to_value_changed (GtkAdjustment *a, GnomePrintPageSelector *ps)
{
	if (!GTK_TOGGLE_BUTTON (ps->r_from_to)->active)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (ps->r_from_to), TRUE);
	gnome_print_page_selector_save_config (ps);
}

static void
on_from_value_changed (GtkAdjustment *a, GnomePrintPageSelector *ps)
{
	if (!GTK_TOGGLE_BUTTON (ps->r_from_to)->active)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (ps->r_from_to), TRUE);
	g_object_set (G_OBJECT (ps->a_to), "lower", ps->a_from->value, NULL);
	g_object_set (G_OBJECT (ps->a_to), "value", MAX (ps->a_to->value, ps->a_from->value), NULL);
	gnome_print_page_selector_save_config (ps);
}

static void
on_selection_changed (GtkEditable *editable, GnomePrintPageSelector *ps)
{
	if (ps->saving) return;

	if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ps->r_selection)))
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (ps->r_selection), TRUE);
	gnome_print_page_selector_save_config (ps);
}

static void
on_from_to_focus_in_event (GtkWidget *widget, GdkEventFocus *event,
		GnomePrintPageSelector *ps)
{
	if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ps->r_from_to)))
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (ps->r_from_to), TRUE);
}

static void
on_selection_focus_in_event (GtkWidget *widget, GdkEventFocus *event,
		GnomePrintPageSelector *ps)
{
	if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (ps->r_selection)))
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (ps->r_selection), TRUE);
}

static void
gnome_print_page_selector_init (GnomePrintPageSelector *ps)
{
	GtkWidget *t, *hb, *w;
	gchar *s;

	s = g_strdup_printf ("<b>%s</b>", _("Print Range"));
	g_object_set (G_OBJECT (ps), "label", s,
			"shadow-type", GTK_SHADOW_NONE, NULL);
	g_free (s);
	g_object_set (G_OBJECT (GTK_FRAME (ps)->label_widget),
			"use-markup", TRUE, "use-underline", TRUE, NULL);

	t = g_object_new (GTK_TYPE_TABLE, "column-spacing", 5,
			"row-spacing", 5, "border-width", 5, NULL);
	gtk_widget_show (t);
	gtk_container_add (GTK_CONTAINER (ps), t);

	/* All pages */
	ps->r_all = gtk_radio_button_new_with_mnemonic (NULL, _("_All"));
	gtk_widget_show (ps->r_all);
	gtk_table_attach (GTK_TABLE (t), ps->r_all, 0, 1, 0, 1, GTK_FILL, 0, 0, 0);

	/* Current page */
	ps->r_current = gtk_radio_button_new_with_mnemonic_from_widget (
			GTK_RADIO_BUTTON (ps->r_all), _("_Current"));
	gtk_widget_show (ps->r_current);
	gtk_table_attach (GTK_TABLE (t), ps->r_current, 0, 1, 1, 2, GTK_FILL, 0, 0, 0);
	gtk_widget_set_sensitive (ps->r_current, FALSE);

	/* From - To */
	hb = g_object_new (GTK_TYPE_HBOX, NULL);
	gtk_widget_show (hb);
	gtk_table_attach (GTK_TABLE (t), hb, 0, 1, 2, 3, GTK_FILL, 0, 0, 0);
	ps->r_from_to = g_object_new (GTK_TYPE_RADIO_BUTTON, "group", ps->r_all, NULL);
	gtk_widget_show (ps->r_from_to);
	gtk_box_pack_start (GTK_BOX (hb), ps->r_from_to, FALSE, FALSE, 0);
	w = g_object_new (GTK_TYPE_LABEL, "label", _("_From page "),
			"use-markup", TRUE, "use-underline", TRUE,
			"mnemonic-widget", ps->r_from_to, NULL);
	gtk_widget_show (w);
	gtk_box_pack_start (GTK_BOX (hb), w, FALSE, FALSE, 0);
	ps->a_from = g_object_new (GTK_TYPE_ADJUSTMENT, "lower", 1., "upper", G_MAXDOUBLE,
			"value", 1., "step-increment", 1., "page-increment", 10., NULL);
	w = g_object_new (GTK_TYPE_SPIN_BUTTON, "adjustment", ps->a_from, NULL);
	gtk_widget_show (w);
	gtk_box_pack_start (GTK_BOX (hb), w, FALSE, FALSE, 0);
	g_signal_connect (G_OBJECT (w), "focus_in_event", 
			G_CALLBACK (on_from_to_focus_in_event), ps);
	w = g_object_new (GTK_TYPE_LABEL, "label", _(" to "), NULL);
	gtk_widget_show (w);
	gtk_box_pack_start (GTK_BOX (hb), w, FALSE, FALSE, 0);
	ps->a_to = g_object_new (GTK_TYPE_ADJUSTMENT, "lower", 1., "upper", G_MAXDOUBLE,
			"value", 1., "step-increment", 1., "page-increment", 10., NULL);
	w = g_object_new (GTK_TYPE_SPIN_BUTTON, "adjustment", ps->a_to, NULL);
	gtk_widget_show (w);
	gtk_box_pack_start (GTK_BOX (hb), w, FALSE, FALSE, 0);
	g_signal_connect (G_OBJECT (w), "focus_in_event",
			G_CALLBACK (on_from_to_focus_in_event), ps);

	/* Custom page selection */
	hb = g_object_new (GTK_TYPE_HBOX, NULL);
	gtk_widget_show (hb);
	gtk_table_attach (GTK_TABLE (t), hb, 0, 1, 3, 4, GTK_FILL, 0, 0, 0);
	ps->r_selection= g_object_new (GTK_TYPE_RADIO_BUTTON, "group", ps->r_all, NULL);
	gtk_widget_show (ps->r_selection);
	gtk_box_pack_start (GTK_BOX (hb), ps->r_selection, FALSE, FALSE, 0);
	w = g_object_new (GTK_TYPE_LABEL, "label", _("_Selection: "),
			"use-markup", TRUE, "use-underline", TRUE,
			"mnemonic-widget", ps->r_selection, NULL);
	gtk_widget_show (w);
	gtk_box_pack_start (GTK_BOX (hb), w, FALSE, FALSE, 0);
	ps->e_selection = g_object_new (GTK_TYPE_ENTRY, NULL);
	gtk_widget_show (ps->e_selection);
	gtk_box_pack_start (GTK_BOX (hb), ps->e_selection, FALSE, FALSE, 0);
	g_signal_connect (G_OBJECT (ps->e_selection), "focus_in_event",
			G_CALLBACK (on_selection_focus_in_event), ps);

	/* Print only... */
	w = g_object_new (GTK_TYPE_LABEL, "label", _("Print only..."), "xalign", 0., NULL),
	gtk_widget_show (w);
	gtk_table_attach (GTK_TABLE (t), w, 1, 2, 0, 1, GTK_FILL, 0, 0, 0);

	/* Even */
	ps->c_even = gtk_check_button_new_with_mnemonic (" _even pages");
	gtk_widget_show (ps->c_even);
	gtk_table_attach (GTK_TABLE (t), ps->c_even, 1, 2, 1, 2, GTK_FILL, 0, 0, 0);

	/* Odd */
	ps->c_odd = gtk_check_button_new_with_mnemonic (" _odd pages");
	gtk_widget_show (ps->c_odd);
	gtk_table_attach (GTK_TABLE (t), ps->c_odd, 1, 2, 2, 3, GTK_FILL, 0, 0, 0);

	/* Connect some signals */
	g_signal_connect (G_OBJECT (ps->r_all), "toggled", G_CALLBACK (on_all_toggled), ps);
	g_signal_connect (G_OBJECT (ps->r_current), "toggled", G_CALLBACK (on_current_toggled), ps);
	g_signal_connect (G_OBJECT (ps->r_from_to), "toggled", G_CALLBACK (on_from_to_toggled), ps);
	g_signal_connect (G_OBJECT (ps->r_selection), "toggled", G_CALLBACK (on_selection_toggled), ps);
	g_signal_connect (G_OBJECT (ps->c_even), "toggled", G_CALLBACK (on_even_toggled), ps);
	g_signal_connect (G_OBJECT (ps->c_odd), "toggled", G_CALLBACK (on_odd_toggled), ps);
	g_signal_connect (G_OBJECT (ps->a_to), "value_changed", G_CALLBACK (on_to_value_changed), ps);
	g_signal_connect (G_OBJECT (ps->a_from), "value_changed", G_CALLBACK (on_from_value_changed), ps);
	g_signal_connect (G_OBJECT (ps->e_selection), "changed", G_CALLBACK (on_selection_changed), ps);

	gnome_print_set_atk_relation (GTK_FRAME (ps)->label_widget, ps->r_all);
	gnome_print_set_atk_relation (GTK_FRAME (ps)->label_widget, ps->r_current);
	gnome_print_set_atk_relation (GTK_FRAME (ps)->label_widget, ps->r_from_to);
	gnome_print_set_atk_relation (GTK_FRAME (ps)->label_widget, ps->r_selection);
	gnome_print_set_atk_relation (GTK_FRAME (ps)->label_widget, ps->c_even);
	gnome_print_set_atk_relation (GTK_FRAME (ps)->label_widget, ps->c_odd);
}

GType
gnome_print_page_selector_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomePrintPageSelectorClass), NULL, NULL,
			(GClassInitFunc) gnome_print_page_selector_class_init,
			NULL, NULL, sizeof (GnomePrintPageSelector), 0,
			(GInstanceInitFunc) gnome_print_page_selector_init
		};
		type = g_type_register_static (GTK_TYPE_FRAME,
				"GnomePrintPageSelector", &info, 0);
	}
	return type;
}
