/* gnome-db-canvas.c
 *
 * Copyright (C) 2002 - 2006 Vivien Malerba
 * Copyright (C) 2002 Fernando Martins
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <gtk/gtk.h>
#include <libgnomedb/marshal.h>
#include "gnome-db-canvas.h"
#include "gnome-db-canvas-tip.h"
#include "gnome-db-canvas-cursor.h"
#include <libgda/libgda.h>
#ifdef HAVE_GRAPHVIZ
#include <stddef.h>
#include <gvc.h>
static GVC_t* gvc = NULL;
#endif

static void gnome_db_canvas_class_init (GnomeDbCanvasClass * class);
static void gnome_db_canvas_init       (GnomeDbCanvas * canvas);
static void gnome_db_canvas_dispose    (GObject   * object);
static void gnome_db_canvas_finalize   (GObject   * object);
static void gnome_db_canvas_post_init  (GnomeDbCanvas * canvas);
static void gnome_db_canvas_size_allocate (GtkWidget *widget, GtkAllocation *allocation);

static void gnome_db_canvas_set_property    (GObject *object,
				       guint param_id,
				       const GValue *value,
				       GParamSpec *pspec);
static void gnome_db_canvas_get_property    (GObject *object,
				       guint param_id,
				       GValue *value,
				       GParamSpec *pspec);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *parent_class = NULL;


struct _GnomeDbCanvasPrivate
{
	GdaGraph           *graph;
	GSList             *items; /* GnomeDbCanvasItem objects, non ordered */
};


enum
{
	DRAG_ACTION,
	LAST_SIGNAL
};

enum
{
	PROP_0,
	PROP_GRAPH
};

static gint canvas_signals[LAST_SIGNAL] = { 0 };

GType
gnome_db_canvas_get_type (void)
{
	static GType type = 0;

	if (G_UNLIKELY (type == 0)) {
		static const GTypeInfo info = {
			sizeof (GnomeDbCanvasClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_canvas_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbCanvas),
			0,
			(GInstanceInitFunc) gnome_db_canvas_init
		};		

		type = g_type_register_static (GNOME_TYPE_CANVAS, "GnomeDbCanvas", &info, 0);
	}
	return type;
}

static void
gnome_db_canvas_class_init (GnomeDbCanvasClass * class)
{
	GtkWidgetClass *widget_class;
	GObjectClass   *object_class = G_OBJECT_CLASS (class);
	parent_class = g_type_class_peek_parent (class);

	widget_class = (GtkWidgetClass *) class;

	canvas_signals[DRAG_ACTION] =
		g_signal_new ("drag_action",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_FIRST,
			      G_STRUCT_OFFSET (GnomeDbCanvasClass, drag_action),
			      NULL, NULL,
			      gnome_db_marshal_VOID__POINTER_POINTER, G_TYPE_NONE, 2,
			      G_TYPE_POINTER, G_TYPE_POINTER);

	class->drag_action = NULL;

	/* properties */
	object_class->set_property = gnome_db_canvas_set_property;
	object_class->get_property = gnome_db_canvas_get_property;

	g_object_class_install_property (object_class, PROP_GRAPH,
					 g_param_spec_pointer ("graph", NULL, NULL,
                                                               (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	
	/* virtual functions */
	class->create_canvas_items = NULL;
	class->clean_canvas_items = NULL;
	class->graph_item_added = NULL;
	class->graph_item_dropped = NULL;
	class->build_context_menu = NULL;
	class->set_pixels_per_unit = NULL;

	widget_class->size_allocate = gnome_db_canvas_size_allocate;
	object_class->dispose = gnome_db_canvas_dispose;
	object_class->finalize = gnome_db_canvas_finalize;
}

static int canvas_event (GnomeCanvas *gcanvas, GdkEvent *event, GnomeDbCanvas *canvas);
static void
gnome_db_canvas_init (GnomeDbCanvas * canvas)
{
	canvas->priv = g_new0 (GnomeDbCanvasPrivate, 1);
	canvas->priv->graph = NULL;
	canvas->priv->items = NULL;

	canvas->xmouse = 50.;
	canvas->ymouse = 50.;
	
	g_signal_connect (G_OBJECT (canvas), "event-after",
			  G_CALLBACK (canvas_event), canvas);
}

static void popup_zoom_in_cb (GtkMenuItem *mitem, GnomeDbCanvas *canvas);
static void popup_zoom_out_cb (GtkMenuItem *mitem, GnomeDbCanvas *canvas);
static void popup_zoom_fit_cb (GtkMenuItem *mitem, GnomeDbCanvas *canvas);
static int 
canvas_event (GnomeCanvas *gcanvas, GdkEvent *event, GnomeDbCanvas *canvas)
{
	gboolean done = TRUE;
	GnomeCanvasItem *item;
	gdouble wx, wy;
	GnomeDbCanvasClass *class = GNOME_DB_CANVAS_CLASS (G_OBJECT_GET_CLASS (canvas));

	if (g_object_get_data (G_OBJECT (gnome_canvas_root (GNOME_CANVAS (canvas))), "dragged_from"))
		/* Dragging cancelled */
		g_object_set_data (G_OBJECT (gnome_canvas_root (GNOME_CANVAS (canvas))), "dragged_from", NULL);

	switch (event->type) {
	case GDK_BUTTON_PRESS:
		gnome_canvas_window_to_world (GNOME_CANVAS (canvas), 
					      ((GdkEventButton *) event)->x, 
					      ((GdkEventButton *) event)->y,
					      &wx, &wy);
		item = gnome_canvas_get_item_at (GNOME_CANVAS (canvas), wx, wy);

		if (!item) {
			if ((((GdkEventButton *) event)->button == 3) && (class->build_context_menu)) {
				GtkWidget *menu, *mitem;
				
				canvas->xmouse = wx;
				canvas->ymouse = wy;

				/* extra menu items, if any */
				menu = (class->build_context_menu) (canvas);
				
				/* default menu items */
				if (!menu)
					menu = gtk_menu_new ();
				else {
					mitem = gtk_separator_menu_item_new ();
					gtk_widget_show (mitem);
					gtk_menu_append (menu, mitem);
				}
				mitem = gtk_image_menu_item_new_from_stock (GTK_STOCK_ZOOM_IN, NULL);
				gtk_widget_show (mitem);
				gtk_menu_append (menu, mitem);
				g_signal_connect (G_OBJECT (mitem), "activate", G_CALLBACK (popup_zoom_in_cb), canvas);
				mitem = gtk_image_menu_item_new_from_stock (GTK_STOCK_ZOOM_OUT, NULL);
				gtk_widget_show (mitem);
				gtk_menu_append (menu, mitem);
				g_signal_connect (G_OBJECT (mitem), "activate", G_CALLBACK (popup_zoom_out_cb), canvas);
				mitem = gtk_image_menu_item_new_from_stock (GTK_STOCK_ZOOM_FIT, NULL);
				gtk_widget_show (mitem);
				gtk_menu_append (menu, mitem);
				g_signal_connect (G_OBJECT (mitem), "activate", G_CALLBACK (popup_zoom_fit_cb), canvas);

				mitem = gtk_separator_menu_item_new ();
				gtk_widget_show (mitem);
				gtk_menu_append (menu, mitem);
				
				mitem = gtk_image_menu_item_new_from_stock (GTK_STOCK_PRINT, NULL);
				gtk_widget_show (mitem);
				gtk_widget_set_sensitive (mitem, FALSE);
				gtk_menu_append (menu, mitem);

				gtk_menu_popup (GTK_MENU (menu), NULL, NULL,
						NULL, NULL, ((GdkEventButton *)event)->button,
						((GdkEventButton *)event)->time);
				
				done = TRUE;
			}
		}
		break;
	default:
		done = FALSE;
		break;
	}
	return done;	
}

static void
popup_zoom_in_cb (GtkMenuItem *mitem, GnomeDbCanvas *canvas)
{
	gnome_db_canvas_set_zoom_factor (canvas, gnome_db_canvas_get_zoom_factor (canvas) + .05);
}

static void
popup_zoom_out_cb (GtkMenuItem *mitem, GnomeDbCanvas *canvas)
{
	gnome_db_canvas_set_zoom_factor (canvas, gnome_db_canvas_get_zoom_factor (canvas) - .05);
}

static void
popup_zoom_fit_cb (GtkMenuItem *mitem, GnomeDbCanvas *canvas)
{
	gnome_db_canvas_fit_zoom_factor (canvas);
}


static void graph_destroyed_cb (GdaGraph *graph, GnomeDbCanvas *canvas);
static void graph_item_added_cb (GdaGraph *graph, GdaGraphItem *item, GnomeDbCanvas *canvas);
static void graph_item_dropped_cb (GdaGraph *graph, GdaGraphItem *item, GnomeDbCanvas *canvas);
static void item_destroyed_cb (GnomeDbCanvasItem *item, GnomeDbCanvas *canvas);
static void item_moved_cb (GnomeDbCanvasItem *item, GnomeDbCanvas *canvas);
static void drag_action_cb (GnomeDbCanvasItem *citem, GnomeDbCanvasItem *drag_from, 
			    GnomeDbCanvasItem *drag_to, GnomeDbCanvas *canvas);
static void
gnome_db_canvas_dispose (GObject   * object)
{
	GnomeDbCanvas *canvas;

	g_return_if_fail (object != NULL);
	g_return_if_fail (GNOME_DB_IS_CANVAS (object));

	canvas = GNOME_DB_CANVAS (object);

	if (canvas->priv->graph) 
		graph_destroyed_cb (canvas->priv->graph, canvas);

	/* get rid of the GnomeCanvasItems */
	while (canvas->priv->items) 
		item_destroyed_cb (GNOME_DB_CANVAS_ITEM (canvas->priv->items->data), canvas);

	/* for the parent class */
	parent_class->dispose (object);
}


static void
gnome_db_canvas_finalize (GObject   * object)
{
	GnomeDbCanvas *canvas;

	g_return_if_fail (object != NULL);
	g_return_if_fail (GNOME_DB_IS_CANVAS (object));
	canvas = GNOME_DB_CANVAS (object);

	if (canvas->priv) {
		g_free (canvas->priv);
		canvas->priv = NULL;
	}

	/* for the parent class */
	parent_class->finalize (object);
}

static void canvas_scrollbar_adjust (GnomeDbCanvas *canvas);
static void
gnome_db_canvas_size_allocate (GtkWidget *widget, GtkAllocation *allocation)
{
	if (GTK_WIDGET_CLASS (parent_class)->size_allocate)
		(* GTK_WIDGET_CLASS (parent_class)->size_allocate) (widget, allocation);

	canvas_scrollbar_adjust (GNOME_DB_CANVAS (widget));
}

static void 
gnome_db_canvas_set_property    (GObject *object,
			   guint param_id,
			   const GValue *value,
			   GParamSpec *pspec)
{
	GnomeDbCanvas *canvas;
	gpointer ptr;
	
	canvas = GNOME_DB_CANVAS (object);

	switch (param_id) {
	case PROP_GRAPH:
		ptr = g_value_get_pointer (value);

		if (ptr == canvas->priv->graph)
			return;

		if (canvas->priv->graph) 
			graph_destroyed_cb (canvas->priv->graph, canvas);
		
		if (ptr) {
			g_return_if_fail (GDA_IS_GRAPH (ptr));
			canvas->priv->graph = GDA_GRAPH (ptr);
			g_object_ref (G_OBJECT (ptr));
			gda_object_connect_destroy (ptr, G_CALLBACK (graph_destroyed_cb), canvas);
			g_signal_connect (G_OBJECT (ptr), "item_added",
					  G_CALLBACK (graph_item_added_cb), canvas);
			g_signal_connect (G_OBJECT (ptr), "item_dropped",
					  G_CALLBACK (graph_item_dropped_cb), canvas);
		}

		break;
	}
	
	if (canvas->priv->graph) 
		gnome_db_canvas_post_init (canvas);
}

static void
gnome_db_canvas_get_property    (GObject *object,
			   guint param_id,
			   GValue *value,
			   GParamSpec *pspec)
{
	GnomeDbCanvas *canvas;
	
	canvas = GNOME_DB_CANVAS (object);

	switch (param_id) {
	case PROP_GRAPH:
		g_value_set_pointer (value, canvas->priv->graph);
		break;
	}
}

static void
graph_destroyed_cb (GdaGraph *graph, GnomeDbCanvas *canvas)
{
	g_return_if_fail (canvas->priv->graph == graph);

	g_signal_handlers_disconnect_by_func (G_OBJECT (graph),
					      G_CALLBACK (graph_destroyed_cb), canvas);
	g_signal_handlers_disconnect_by_func (G_OBJECT (graph),
					      G_CALLBACK (graph_item_added_cb), canvas);
	g_signal_handlers_disconnect_by_func (G_OBJECT (graph),
					      G_CALLBACK (graph_item_dropped_cb), canvas);
	canvas->priv->graph = NULL;
	/* some clean-up if there are already some canvas items */
	while (canvas->priv->items) 
		gtk_object_destroy (GTK_OBJECT (canvas->priv->items->data));
	g_object_unref (G_OBJECT (graph));
}

static void
graph_item_added_cb (GdaGraph *graph, GdaGraphItem *item, GnomeDbCanvas *canvas)
{
	GnomeDbCanvasClass *class = GNOME_DB_CANVAS_CLASS (G_OBJECT_GET_CLASS (canvas));
	if (class->graph_item_added)
		(class->graph_item_added) (canvas, item);
}

static void
graph_item_dropped_cb (GdaGraph *graph, GdaGraphItem *item, GnomeDbCanvas *canvas)
{
	GnomeDbCanvasClass *class = GNOME_DB_CANVAS_CLASS (G_OBJECT_GET_CLASS (canvas));
	if (class->graph_item_dropped)
		(class->graph_item_dropped) (canvas, item);
}

static void 
gnome_db_canvas_post_init  (GnomeDbCanvas *canvas)
{
	GnomeDbCanvasClass *class = GNOME_DB_CANVAS_CLASS (G_OBJECT_GET_CLASS (canvas));

	/* some clean-up if there are already some canvas items */
	if (class->clean_canvas_items)
		(class->clean_canvas_items) (canvas);

	/* reseting the zoom */
	gnome_canvas_set_pixels_per_unit (GNOME_CANVAS (canvas), 1.0);

	/* adding some new canvas items */
	if (class->create_canvas_items)
		(class->create_canvas_items) (canvas);

	canvas_scrollbar_adjust (canvas);
}

GdaGraph *
gnome_db_canvas_get_graph (GnomeDbCanvas *canvas)
{
	g_return_val_if_fail (GNOME_DB_IS_CANVAS (canvas), NULL);
	g_return_val_if_fail (canvas->priv, NULL);

	return canvas->priv->graph;
}

/**
 * gnome_db_canvas_declare_item
 * @canvas: a #GnomeDbCanvas widget
 * @item: a #GnomeDbCanvasItem object
 *
 * Declares @item to be listed by @canvas as one of its items.
 * This functions should be called after each #GnomeDbCanvasItem is added to
 * @canvas.
 *
 * If it was not called for one item, then that item won't be used
 * in @canvas's computations (no drag and drop, cleanup, etc).
 */
void
gnome_db_canvas_declare_item (GnomeDbCanvas *canvas, GnomeDbCanvasItem *item)
{
	g_return_if_fail (GNOME_DB_IS_CANVAS (canvas));
	g_return_if_fail (canvas->priv);
	g_return_if_fail (item && GNOME_DB_IS_CANVAS_ITEM (item));

	if (g_slist_find (canvas->priv->items, item))
		return;

	canvas->priv->items = g_slist_append (canvas->priv->items, item);
	g_signal_connect (G_OBJECT (item), "moved",
			  G_CALLBACK (item_moved_cb), canvas);
	g_signal_connect (G_OBJECT (item), "drag_action",
			  G_CALLBACK (drag_action_cb), canvas);
	g_signal_connect (G_OBJECT (item), "destroy",
			  G_CALLBACK (item_destroyed_cb), canvas);
}

static void
item_destroyed_cb (GnomeDbCanvasItem *item, GnomeDbCanvas *canvas)
{
	g_return_if_fail (g_slist_find (canvas->priv->items, item));
	g_signal_handlers_disconnect_by_func (G_OBJECT (item), G_CALLBACK (item_moved_cb), canvas);
	g_signal_handlers_disconnect_by_func (G_OBJECT (item), G_CALLBACK (drag_action_cb), canvas);
	g_signal_handlers_disconnect_by_func (G_OBJECT (item), G_CALLBACK (item_destroyed_cb), canvas);
	canvas->priv->items = g_slist_remove (canvas->priv->items, item);
}

static void 
item_moved_cb (GnomeDbCanvasItem *item, GnomeDbCanvas *canvas)
{
	canvas_scrollbar_adjust (canvas);
}

static void 
drag_action_cb (GnomeDbCanvasItem *citem, GnomeDbCanvasItem *drag_from, GnomeDbCanvasItem *drag_to, GnomeDbCanvas *canvas)
{
	g_print ("Canvas Drag action from %p to %p\n", drag_from, drag_to);
#ifdef debug_signal
        g_print (">> 'DRAG_ACTION' from %s, %s\n", __FILE__, __FUNCTION__);
#endif
        g_signal_emit (G_OBJECT (canvas), canvas_signals[DRAG_ACTION], 0, drag_from, drag_to);
#ifdef debug_signal
        g_print ("<< 'DRAG_ACTION' from %s, %s\n", __FILE__, __FUNCTION__);
#endif
}


/**
 * gnome_db_canvas_set_in_scrolled_window
 * @canvas: a #GnomeDbCanvas widget
 *
 * Creates a new #GtkScrolledWindow object and put @canvas
 * in it. @canvas can be retrieved using a "canvas" user property of
 * the new scrolled window.
 *
 * Returns: the new scrolled window.
 */
GtkWidget *
gnome_db_canvas_set_in_scrolled_window (GnomeDbCanvas *canvas)
{
	GtkWidget *sw;
	g_return_val_if_fail (GNOME_DB_IS_CANVAS (canvas), NULL);
	g_return_val_if_fail (canvas->priv, NULL);

	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw), 
					GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
        gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw), GTK_SHADOW_IN);
	gtk_container_add (GTK_CONTAINER (sw), GTK_WIDGET (canvas));
	gtk_widget_show (GTK_WIDGET (canvas));
	g_object_set_data (G_OBJECT (sw), "canvas", canvas);
	g_object_set_data (G_OBJECT (canvas), "scrolled-window", sw);

	gnome_canvas_set_scroll_region (GNOME_CANVAS (canvas), 0., 0., 5., 5.);
	gnome_canvas_scroll_to (GNOME_CANVAS (canvas), 0, 0);
	canvas_scrollbar_adjust (canvas);

	return sw;
}

static gboolean move_all_items (GnomeDbCanvas *canvas, gdouble wdx, gdouble wdy);
static gboolean
move_all_items (GnomeDbCanvas *canvas, gdouble wdx, gdouble wdy)
{
	GSList *list;
	gdouble px, py;
	gboolean moved = FALSE;

	list = canvas->priv->items;
	while (list) {
		GdaGraphItem *item = gnome_db_canvas_item_get_graph_item (GNOME_DB_CANVAS_ITEM (list->data));
		if (item) {
			gda_graph_item_get_position (item, &px, &py);
			px += wdx;
			py += wdy;
			gda_graph_item_set_position (item, px, py);
			moved = TRUE;
		}
		list = g_slist_next (list);
	}

	return (moved && (wdx || wdy));
}

/*
 * Adjusts the canvas when in a scrolled window
 */
static void
canvas_scrollbar_adjust (GnomeDbCanvas *canvas)
{
	GtkAdjustment *hadj, *vadj;
	gdouble wx, wy, wx1, wy1, wx2, wy2;

	if (!GTK_WIDGET_REALIZED (GTK_WIDGET (canvas)))
		return;

	/* in case there is not yet any item on the canvas */
	if (! GNOME_CANVAS (canvas)->root ||
	    ! GNOME_CANVAS_GROUP (GNOME_CANVAS (canvas)->root)->item_list) {
		gnome_canvas_set_scroll_region (GNOME_CANVAS (canvas), 0., 0., 5., 5.);
		gnome_canvas_scroll_to (GNOME_CANVAS (canvas), 0, 0);
		gnome_canvas_set_pixels_per_unit (GNOME_CANVAS (canvas), 1.0);

		return;
	}

	/* World Coordinates */
	gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (gnome_canvas_root (GNOME_CANVAS (canvas))),
				      &wx1, &wy1, &wx2, &wy2);
	/*g_print ("1Bounds: %.3f x %.3f --> %.3f x %.3f\n", wx1, wy1, wx2, wy2);*/
	wy1 -= 5.; wy2 += 5.;
	wx1 -= 5.; wx2 += 5.;

	if (move_all_items (canvas, - wx1,- wy1)) {
		gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (gnome_canvas_root (GNOME_CANVAS (canvas))),
					      &wx1, &wy1, &wx2, &wy2);
		wy1 -= 5.; wy2 += 5.;
		wx1 -= 5.; wx2 += 5.;
		/*g_print ("2Bounds: %.3f x %.3f --> %.3f x %.3f\n", wx1, wy1, wx2, wy2);*/
	}

	gnome_canvas_c2w (GNOME_CANVAS (canvas), GTK_WIDGET (canvas)->allocation.width, 
			  GTK_WIDGET (canvas)->allocation.height, &wx, &wy);
	if (wx >= wx2 - wx1) {
		wx1 -= (wx - wx2 + wx1) / 2.;
		wx2 += (wx - wx2 + wx1) / 2.;
	}
	if (wy >= wy2 - wy1) {
		wy1 -= (wy - wy2 + wy1) / 2.;
		wy2 += (wy - wy2 + wy1) / 2.;
	}
	
	wx1 = 0.; wy1 = .0;
	gnome_canvas_set_scroll_region (GNOME_CANVAS (canvas), wx1, wy1, wx2, wy2);
	/*g_print ("Scroll: %.3f x %.3f --> %.3f x %.3f\n", wx1, wy1, wx2, wy2);*/

	/* vertical adjustment */
	vadj = gtk_layout_get_vadjustment (GTK_LAYOUT (canvas));
	vadj->step_increment = 10.;
	gtk_adjustment_changed (vadj);

	/* horizontal adjustment */
	hadj = gtk_layout_get_hadjustment (GTK_LAYOUT (canvas));
	hadj->step_increment = 10.;
	gtk_adjustment_changed (hadj);
}

/**
 * gnome_db_canvas_get_item_for_object
 * @canvas: a #GnomeDbCanvas widget
 * @ref_obj:
 *
 * Get the #GnomeDbCanvasItem corresponding to @obj;
 *
 * Returns:
 */
GnomeDbCanvasItem *
gnome_db_canvas_get_item_for_object (GnomeDbCanvas *canvas, GdaObject *ref_obj)
{
	GnomeDbCanvasItem *item = NULL;
	GSList *list = canvas->priv->items;

	while (list && !item) {
		GdaGraphItem *gitem = gnome_db_canvas_item_get_graph_item (GNOME_DB_CANVAS_ITEM (list->data));
		if (gitem && (gda_graph_item_get_ref_object (gitem) == ref_obj))
			item = GNOME_DB_CANVAS_ITEM (list->data);

		list = g_slist_next (list);
	}

	return item;
}

/**
 * gnome_db_canvas_set_zoom_factor
 * @canvas: a #GnomeDbCanvas widget
 * @n: the zoom factor
 *
 * Sets the zooming factor of a canvas by specifying the number of pixels that correspond 
 * to one canvas unit. A zoom factor of 1.0 is the default value; greater than 1.0 makes a zoom in
 * and lower than 1.0 makes a zoom out.
 */
void
gnome_db_canvas_set_zoom_factor (GnomeDbCanvas *canvas, gdouble n)
{
	GnomeDbCanvasClass *class = GNOME_DB_CANVAS_CLASS (G_OBJECT_GET_CLASS (canvas));
	g_return_if_fail (GNOME_DB_IS_CANVAS (canvas));
	g_return_if_fail (canvas->priv);

	gnome_canvas_set_pixels_per_unit (GNOME_CANVAS (canvas), n);
	if (class->set_pixels_per_unit)
		(class->set_pixels_per_unit) (canvas, n);
}

/**
 * gnome_db_canvas_get_zoom_factor
 * @canvas: a #GnomeDbCanvas widget
 *
 * Get the current zooming factor of a canvas.
 *
 * Returns: the zooming factor.
 */
gdouble
gnome_db_canvas_get_zoom_factor (GnomeDbCanvas *canvas)
{
	g_return_val_if_fail (GNOME_DB_IS_CANVAS (canvas), 1.);
	g_return_val_if_fail (canvas->priv, 1.);

	return GNOME_CANVAS (canvas)->pixels_per_unit;
}

/**
 * gnome_db_canvas_fit_zoom_factor
 * @canvas: a #GnomeDbCanvas widget
 *
 * Compute and set the correct zoom factor so that all the items on @canvas can be displayed
 * at once.
 *
 * Returns: the new zooming factor.
 */
gdouble
gnome_db_canvas_fit_zoom_factor (GnomeDbCanvas *canvas)
{
	gdouble zoom, xall, yall;
	gdouble wx1, wy1, wx2, wy2;

	g_return_val_if_fail (GNOME_DB_IS_CANVAS (canvas), 1.);
	g_return_val_if_fail (canvas->priv, 1.);

	xall = GTK_WIDGET (canvas)->allocation.width;
	yall = GTK_WIDGET (canvas)->allocation.height;
	gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (gnome_canvas_root (GNOME_CANVAS (canvas))),
				      &wx1, &wy1, &wx2, &wy2);
	wy1 -= 5.; wy2 += 5.;
	wx1 -= 5.; wx2 += 5.;
	zoom = yall / (wy2 - wy1);
	if (xall / (wx2 - wx1) < zoom)
		zoom = xall / (wx2 - wx1);

	/* set a limit to the zoom */
	if (zoom > 1.0)
		zoom = 1.0;
	
	gnome_db_canvas_set_zoom_factor (GNOME_DB_CANVAS (canvas), zoom);
	return zoom;
}

/**
 * gnome_db_canvas_auto_layout_enabled
 * @canvas: a #GnomeDbCanvas widget
 *
 * Tells if @canvas has the possibility to automatically adjust its layout
 * using the GraphViz library.
 *
 * Returns: TRUE if @canvas can automatically adjust its layout
 */
gboolean
gnome_db_canvas_auto_layout_enabled (GnomeDbCanvas *canvas)
{
	g_return_val_if_fail (GNOME_DB_IS_CANVAS (canvas), FALSE);
	g_return_val_if_fail (canvas->priv, FALSE);

#ifdef HAVE_GRAPHVIZ
	return TRUE;
#else
	return FALSE;
#endif
}

#ifdef HAVE_GRAPHVIZ
typedef struct {
	GnomeDbCanvas *canvas;
	Agraph_t      *graph;
	GSList        *nodes_list; /* list of NodeLayout structures */
} GraphLayout;

typedef struct {
	GnomeDbCanvasItem *item; /* item to be moved */
	GdaGraphItem      *graph_item;
	Agnode_t          *node;
	gdouble            start_x;
	gdouble            start_y;
	gdouble            end_x;
	gdouble            end_y;
	gdouble            width;
	gdouble            height;
} NodeLayout;
static gboolean canvas_animate_to (GraphLayout *gl);
#endif

/**
 * gnome_db_canvas_auto_layout
 * @canvas: a #GnomeDbCanvas widget
 *
 * Re-organizes the layout of the @canvas' items using the GraphViz
 * layout engine.
 */
void
gnome_db_canvas_perform_auto_layout (GnomeDbCanvas *canvas, gboolean animate, GnomeDbCanvasLayoutAlgorithm algorithm)
{
	g_return_if_fail (GNOME_DB_IS_CANVAS (canvas));
	g_return_if_fail (canvas->priv);

#define GV_SCALE 72.

#ifndef HAVE_GRAPHVIZ
	g_message ("GraphViz library support not compiled, cannot do graph layout...\n");
	return FALSE;
#else
	GSList *list;
	Agraph_t *graph;
	GHashTable *nodes_hash; /* key = GnomeDbCanvasItem, value = Agnode_t *node */
	GSList *nodes_list = NULL; /* list of NodeLayout structures */

	if (!gvc)
		gvc = gvContext ();

	graph = agopen ("GnomedbCanvasLayout", AGDIGRAPHSTRICT);
        agnodeattr (graph, "shape", "box");
        agnodeattr (graph, "height", ".1");
        agnodeattr (graph, "width", ".1");
        agnodeattr (graph, "fixedsize", "true");
        agnodeattr (graph, "pack", "true");
	agnodeattr (graph, "packmode", "node");

	/* Graph nodes creation */
	nodes_hash = g_hash_table_new (NULL, NULL);
	for (list = canvas->priv->items; list; list = list->next) {
		GnomeDbCanvasItem *item = GNOME_DB_CANVAS_ITEM (list->data);

		if (gnome_db_canvas_item_get_graph_item (item)) {
			Agnode_t *node;
			gchar *tmp;
			double x1, y1, x2, y2, val;

			NodeLayout *nl;
			nl = g_new0 (NodeLayout, 1);
			nl->item = item;
			nl->graph_item = gnome_db_canvas_item_get_graph_item (item);
			gda_graph_item_get_position (nl->graph_item, &(nl->start_x), &(nl->start_y));
			nodes_list = g_slist_prepend (nodes_list, nl);
			
			tmp = g_strdup_printf ("%p", item);
			node = agnode (graph, tmp);
			nl->node = node;
			g_hash_table_insert (nodes_hash, item, node);
			
			tmp = g_strdup_printf ("%p", node);
			agset (node, "label", tmp);
			g_free (tmp);

			gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (item), &x1, &y1, &x2, &y2);
			nl->width = x2 - x1;
			nl->height = y2 - y1;
			val = (y2 - y1) / GV_SCALE;
			tmp = g_strdup_printf ("%.3f", val);
			agset (node, "height", tmp);
			g_free (tmp);
			val = (x2 - x1) / GV_SCALE;
			tmp = g_strdup_printf ("%.3f", val);
			agset (node, "width", tmp);
			g_free (tmp);

			/*g_print ("Before: Node %p: HxW: %.3f %.3f\n", node, (y2 - y1) / GV_SCALE, (x2 - x1) / GV_SCALE);*/
		}
	}
	/* Graph edges creation */
	for (list = canvas->priv->items; list; list = list->next) {
		GnomeDbCanvasItem *item = GNOME_DB_CANVAS_ITEM (list->data);
		GnomeDbCanvasItem *from, *to;
		gnome_db_canvas_item_get_edge_nodes (item, &from, &to);
		if (from && to) {
			Agnode_t *from_node, *to_node;
			from_node = (Agnode_t*) g_hash_table_lookup (nodes_hash, from);
			to_node = (Agnode_t*) g_hash_table_lookup (nodes_hash, to);
			g_assert (from_node);
			g_assert (to_node);
			agedge (graph, from_node, to_node);
		}
	}

	switch (algorithm) {
	default:
	case GNOME_DB_CANVAS_LAYOUT_DEFAULT:
		gvLayout (gvc, graph, "dot");
		break;
	case GNOME_DB_CANVAS_LAYOUT_RADIAL:
		gvLayout (gvc, graph, "circo");
		break;
	}
        gvRender (gvc, graph, "dot", NULL);
	/*gvRenderFilename (gvc, graph, "png", "out.png");*/
        /*gvRender (gvc, graph, "dot", stdout);*/

	for (list = nodes_list; list; list = list->next) {
		NodeLayout *nl = (NodeLayout*) list->data;
		nl->end_x = ND_coord_i (nl->node).x - (nl->width / 2.);
		nl->end_y = - ND_coord_i (nl->node).y - (nl->height / 2.);
		/*g_print ("After: Node %p: HxW: %.3f %.3f XxY = %d, %d\n", nl->node, 
			 ND_height (nl->node), ND_width (nl->node),
			 ND_coord_i (nl->node).x, - ND_coord_i (nl->node).y);*/
		if (!animate)
			gda_graph_item_set_position (nl->graph_item, nl->end_x, nl->end_y);
	}

	g_hash_table_destroy (nodes_hash);
	gvFreeLayout (gvc, graph);

	if (animate) {
		GraphLayout *gl;
		gl = g_new0 (GraphLayout, 1);
		gl->canvas = canvas;
		gl->graph = graph;
		gl->nodes_list = nodes_list;
		g_timeout_add_full (G_PRIORITY_HIGH_IDLE, 10, (GSourceFunc) canvas_animate_to, gl, NULL);
	}
	else {
		agclose (graph);
		g_slist_foreach (nodes_list, (GFunc) g_free, NULL);
		g_slist_free (nodes_list);	
	}
#endif
}

#ifdef HAVE_GRAPHVIZ
static gdouble
compute_animation_inc (float start, float stop, float current)
{
        gdouble inc;
#ifndef PI
#define PI 3.14159265
#endif

        if (stop == start)
                return 0.;

        inc = (cos ((current - start) / (stop - start) * 2. * PI - PI) + 1.) * (stop - start) / 50.;

        if (stop >= start) {
                if (inc < (stop - start) / 100.)
                        inc = (stop - start) / 100.;
        }
        else {
                if (inc > (stop - start) / 100.)
                        inc = (stop - start) / 100.;
        }

        return inc;
}

static gboolean
canvas_animate_to (GraphLayout *gl) 
{
	gboolean keep = FALSE;
	GSList *list;
	
#define EPSILON 10.
	for (list = gl->nodes_list; list; list = list->next) {
		NodeLayout *nl = (NodeLayout*) list->data;
		gdouble cur_x, cur_y, dx, dy;
		gda_graph_item_get_position (nl->graph_item, &cur_x, &cur_y);
		dx = compute_animation_inc (nl->start_x, nl->end_x, cur_x);
		dy = compute_animation_inc (nl->start_y, nl->end_y, cur_y);
		gda_graph_item_set_position (nl->graph_item, cur_x + dx, cur_y + dy);
		if (!keep &&
		    ((fabs ((cur_x + dx - nl->end_x)) > EPSILON) ||
		     (fabs ((cur_y + dy - nl->end_y)) > EPSILON)))
			keep = TRUE;
	}

	gnome_canvas_update_now (GNOME_CANVAS (gl->canvas));

	if (!keep) {
		agclose (gl->graph);
		g_slist_foreach (gl->nodes_list, (GFunc) g_free, NULL);
		g_slist_free (gl->nodes_list);
		g_free (gl);
	}
	return keep;
}
#endif
