/* gnome-db-canvas-text.c
 *
 * Copyright (C) 2002 - 2006 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <libgda/libgda.h>
#include "gnome-db-canvas.h"
#include "gnome-db-canvas-text.h"

static void gnome_db_canvas_text_class_init (GnomeDbCanvasTextClass * class);
static void gnome_db_canvas_text_init       (GnomeDbCanvasText * drag);
static void gnome_db_canvas_text_dispose    (GObject   * object);
static void gnome_db_canvas_text_finalize   (GObject   * object);

static void gnome_db_canvas_text_set_property    (GObject *object,
					     guint param_id,
					     const GValue *value,
					     GParamSpec *pspec);
static void gnome_db_canvas_text_get_property    (GObject *object,
					     guint param_id,
					     GValue *value,
					     GParamSpec *pspec);

enum
{
	PROP_0,
	PROP_GDA_OBJECT,
	PROP_WIDTH,
	PROP_HEIGHT,
	PROP_HIGHLIGHT_COLOR,
	PROP_UNDERLINE,
	PROP_BOLD,
	PROP_SCALE
};

struct _GnomeDbCanvasTextPrivate
{
	GdaObject            *base_obj;
	
	/* properties */
	PangoFontDescription *font_desc;
	gint                  init_font_size;
	gchar                *highlight_color;

	/* UI building information */
        GnomeCanvasItem      *bg;
        GnomeCanvasItem      *text;
        gdouble              outline_width;
        gdouble              outline_height;

	/* presentation parameters */
        gdouble              x_text_space;
        gdouble              y_text_space;
};

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass *text_parent_class = NULL;

GType
gnome_db_canvas_text_get_type (void)
{
	static GType type = 0;

        if (G_UNLIKELY (type == 0)) {
		static const GTypeInfo info = {
			sizeof (GnomeDbCanvasTextClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_canvas_text_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbCanvasText),
			0,
			(GInstanceInitFunc) gnome_db_canvas_text_init
		};		

		type = g_type_register_static (GNOME_DB_TYPE_CANVAS_ITEM, "GnomeDbCanvasText", &info, 0);
	}

	return type;
}	

static void
gnome_db_canvas_text_class_init (GnomeDbCanvasTextClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	text_parent_class = g_type_class_peek_parent (class);

	object_class->dispose = gnome_db_canvas_text_dispose;
	object_class->finalize = gnome_db_canvas_text_finalize;

	/* Properties */
	object_class->set_property = gnome_db_canvas_text_set_property;
	object_class->get_property = gnome_db_canvas_text_get_property;

	g_object_class_install_property
                (object_class, PROP_GDA_OBJECT,
                 g_param_spec_object("gda_object", NULL, NULL, GDA_TYPE_OBJECT, (G_PARAM_READABLE | G_PARAM_WRITABLE)));

	g_object_class_install_property 
		(object_class, PROP_WIDTH,
		 g_param_spec_double ("width", NULL, NULL, 0., G_MAXDOUBLE, 0., G_PARAM_WRITABLE));

	g_object_class_install_property 
		(object_class, PROP_HEIGHT,
		 g_param_spec_double ("height", NULL, NULL, 0., G_MAXDOUBLE, 0., G_PARAM_WRITABLE));

	g_object_class_install_property 
		(object_class, PROP_HIGHLIGHT_COLOR,
		 g_param_spec_string ("highlight_color", NULL, NULL, NULL, (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	g_object_class_install_property 
		(object_class, PROP_UNDERLINE,
		 g_param_spec_boolean ("text_underline", NULL, NULL, FALSE, (G_PARAM_READABLE | G_PARAM_WRITABLE)));
	g_object_class_install_property 
		(object_class, PROP_BOLD,
		 g_param_spec_boolean ("text_bold", NULL, NULL, FALSE, (G_PARAM_READABLE | G_PARAM_WRITABLE)));

	g_object_class_install_property 
		(object_class, PROP_SCALE,
		 g_param_spec_double ("scale", "Scale", "Text scaling factor", 0., G_MAXDOUBLE, 0., G_PARAM_WRITABLE));
}

static void
gnome_db_canvas_text_init (GnomeDbCanvasText *text)
{
	text->priv = g_new0 (GnomeDbCanvasTextPrivate, 1);
	text->priv->base_obj = NULL;
	text->priv->font_desc = NULL;
	text->priv->highlight_color = g_strdup (GNOME_DB_CANVAS_ENTITY_COLOR);
	text->priv->outline_width = 0.;
	text->priv->outline_height = 0.;

	text->priv->x_text_space = 3.;
	text->priv->y_text_space = 3.;
}

static void text_destroyed_cb (GdaObject *base, GnomeDbCanvasText *ct);
static void
gnome_db_canvas_text_dispose (GObject *object)
{
	GnomeDbCanvasText *ct;
	g_return_if_fail (object != NULL);
	g_return_if_fail (GNOME_DB_IS_CANVAS_TEXT (object));

	ct = GNOME_DB_CANVAS_TEXT (object);
	if (ct->priv->base_obj) {
		g_signal_handlers_disconnect_by_func (G_OBJECT (ct->priv->base_obj),
						      G_CALLBACK (text_destroyed_cb), ct);
		g_object_unref (ct->priv->base_obj);
		ct->priv->base_obj = NULL;
	}

	if (ct->priv->font_desc) {
		pango_font_description_free (ct->priv->font_desc);
		ct->priv->font_desc = NULL;
	}

	/* for the parent class */
	text_parent_class->dispose (object);
}


static void
gnome_db_canvas_text_finalize (GObject *object)
{
	GnomeDbCanvasText *ct;
	g_return_if_fail (object != NULL);
	g_return_if_fail (GNOME_DB_IS_CANVAS_TEXT (object));

	ct = GNOME_DB_CANVAS_TEXT (object);
	if (ct->priv) {
		if (ct->priv->highlight_color)
			g_free (ct->priv->highlight_color);

		g_free (ct->priv);
		ct->priv = NULL;
	}

	/* for the parent class */
	text_parent_class->finalize (object);
}


static void
text_destroyed_cb (GdaObject *base, GnomeDbCanvasText *ct)
{
	g_signal_handlers_disconnect_by_func (G_OBJECT (base),
					      G_CALLBACK (text_destroyed_cb), ct);
	g_object_unref (ct->priv->base_obj);
	ct->priv->base_obj = NULL;
	gtk_object_destroy (GTK_OBJECT (ct));
}

static void clean_items (GnomeDbCanvasText *ct);
static void create_items (GnomeDbCanvasText *ct);

static void 
gnome_db_canvas_text_set_property    (GObject *object,
				guint param_id,
				const GValue *value,
				GParamSpec *pspec)
{
	GnomeDbCanvasText *ct = NULL;
	GObject* propobject = NULL;
	const gchar *str = NULL;
	gdouble size = 0;
	gboolean bool = FALSE;

	ct = GNOME_DB_CANVAS_TEXT (object);

	switch (param_id) {
	case PROP_GDA_OBJECT:
		propobject = g_value_get_object (value);
		if (propobject == G_OBJECT (ct->priv->base_obj))
			return;

		if (ct->priv->base_obj) {
			g_signal_handlers_disconnect_by_func (G_OBJECT (ct->priv->base_obj),
							      G_CALLBACK (text_destroyed_cb), ct);
			g_object_unref (ct->priv->base_obj);
			ct->priv->base_obj = NULL;
			clean_items (ct);
		}
		if (propobject) {
			g_return_if_fail (GDA_IS_OBJECT (propobject));
			ct->priv->base_obj = GDA_OBJECT(propobject);
			g_object_ref (ct->priv->base_obj);
			
			gda_object_connect_destroy (ct->priv->base_obj,
						 G_CALLBACK (text_destroyed_cb), ct);
			/* Signals to keep the display up to date */
			/* g_signal_connect (G_OBJECT (ct->priv->base_obj), "text_added", */
/* 					  G_CALLBACK (texts_changed_cb), ct); */
/* 			g_signal_connect (G_OBJECT (ct->priv->base_obj), "text_removed", */
/* 					  G_CALLBACK (texts_changed_cb), ct); */

			create_items (ct);
		}
		break;
	case PROP_WIDTH:
		size = g_value_get_double (value);
		g_object_set (G_OBJECT (ct->priv->bg),
			      "x2", size - ct->priv->x_text_space,
			      NULL);
		break;
	case PROP_HEIGHT:
		size = g_value_get_double (value);
		g_object_set (G_OBJECT (ct->priv->bg),
			      "y2", size - ct->priv->y_text_space,
			      NULL);
		break;
	case PROP_HIGHLIGHT_COLOR:
		str = g_value_get_string (value);
		if (ct->priv->highlight_color) {
			g_free (ct->priv->highlight_color);
			ct->priv->highlight_color = NULL;
		}
		if (str) 
			ct->priv->highlight_color = g_strdup (str);
		else 
			ct->priv->highlight_color = g_strdup (GNOME_DB_CANVAS_ENTITY_COLOR);
		break;
	case PROP_UNDERLINE:
		bool = g_value_get_boolean (value);
		g_object_set (G_OBJECT (ct->priv->text), "underline", 
			      bool ? PANGO_UNDERLINE_SINGLE : PANGO_UNDERLINE_NONE,
			      "underline_set", TRUE, NULL);
	case PROP_BOLD:
		bool = g_value_get_boolean (value);
		g_object_set (G_OBJECT (ct->priv->text), "weight", 
			      bool ? PANGO_WEIGHT_BOLD : PANGO_WEIGHT_NORMAL,
			      "weight_set", TRUE, NULL);
		pango_font_description_set_weight (ct->priv->font_desc, 
						   bool ? PANGO_WEIGHT_BOLD : PANGO_WEIGHT_NORMAL);
		break;
	case PROP_SCALE:
		size = g_value_get_double (value);
		pango_font_description_set_size (ct->priv->font_desc, size * ct->priv->init_font_size);
		g_object_set (G_OBJECT (ct->priv->text), "font-desc", ct->priv->font_desc, NULL);
		break;
	}
}

static void 
gnome_db_canvas_text_get_property    (GObject *object,
				    guint param_id,
				    GValue *value,
				    GParamSpec *pspec)
{
	GnomeDbCanvasText *ct;

	ct = GNOME_DB_CANVAS_TEXT (object);

	switch (param_id) {
	case PROP_GDA_OBJECT:
		g_value_set_pointer (value, ct->priv->base_obj);
		break;
	default:
		g_warning ("No such property!");
		break;
	}
}

static gboolean text_item_event (GnomeCanvasItem *ci, GdkEvent *event, GnomeDbCanvasText *ct);

/* 
 * destroy any existing GnomeCanvasItem obejcts 
 */
static void 
clean_items (GnomeDbCanvasText *ct)
{
	if (ct->priv->bg) {
		gtk_object_destroy (GTK_OBJECT (ct->priv->bg));
		ct->priv->bg = NULL;
	}
	if (ct->priv->text) {
		gtk_object_destroy (GTK_OBJECT (ct->priv->text));
		ct->priv->text = NULL;
	}
	
	/* Signals */
	g_signal_handlers_disconnect_by_func (G_OBJECT (ct),
					      G_CALLBACK (text_item_event), ct);
}

/*
 * create new GnomeCanvasItem objects
 */
static void 
create_items (GnomeDbCanvasText *ct)
{
	GnomeCanvasItem *item, *text;
	const gchar *str = NULL;
	double x1, y1, x2, y2;
	GnomeCanvasPoints *points;
	/* WARNING: the text items are first drawn without taking care of the real zoom factor =>
	 * we must take that into account when using the text sizes */
	gdouble scale = GNOME_CANVAS_ITEM (ct)->canvas->pixels_per_unit;

	gnome_canvas_item_set (GNOME_CANVAS_ITEM (ct), 
			       "allow_move", FALSE,
			       "allow_drag", TRUE,
			       NULL);

	if (!ct->priv->font_desc) {
		ct->priv->font_desc = pango_font_description_copy
			(GTK_WIDGET (GNOME_CANVAS_ITEM (ct)->canvas)->style->font_desc);
		ct->priv->init_font_size = pango_font_description_get_size (ct->priv->font_desc);
		pango_font_description_set_size (ct->priv->font_desc, scale * ct->priv->init_font_size);
	}

	/* text: text's name */
	str = gda_object_get_name (GDA_OBJECT (ct->priv->base_obj));

	text = gnome_canvas_item_new (GNOME_CANVAS_GROUP (ct),
				      GNOME_TYPE_CANVAS_TEXT,
				      "x", ct->priv->x_text_space,
				      "y", ct->priv->y_text_space,
				      "font-desc", ct->priv->font_desc,
				      "text", str,
				      "fill_color", "black",
				      "justification", GTK_JUSTIFY_RIGHT, 
				      "anchor", GTK_ANCHOR_NORTH_WEST, 
				      "scale-set", TRUE,
				      "stretch-set", TRUE,
				      NULL);
	ct->priv->text = text;

	/* UI metrics */
	gnome_canvas_item_get_bounds (text, &x1, &y1, &x2, &y2);
	
	/* underline */
	points = gnome_canvas_points_new (2);
	
	/* background */
	item = gnome_canvas_item_new (GNOME_CANVAS_GROUP (ct),
				      GNOME_TYPE_CANVAS_RECT,
				      "x1", ct->priv->x_text_space,
				      "y1", 0.,
				      "x2", x2 - x1 + 2 * ct->priv->x_text_space,
				      "y2", y2 - y1 + ct->priv->y_text_space,
				      "fill_color", "white",
				      NULL);

	ct->priv->bg = item;
	gnome_canvas_item_lower_to_bottom (item);	


	/* Signals */
	g_signal_connect(G_OBJECT (ct), "event", 
 			 G_CALLBACK (text_item_event), ct); 
}

static gboolean
text_item_event(GnomeCanvasItem *ci, GdkEvent *event, GnomeDbCanvasText *ct)
{
	switch (event->type) {
	case GDK_ENTER_NOTIFY:
		gnome_db_canvas_text_set_highlight (ct, TRUE); 
		break;
	case GDK_LEAVE_NOTIFY:
		gnome_db_canvas_text_set_highlight (ct, FALSE); 
		break;
	default:
		break;
	}

	return FALSE;
}

/**
 * gnome_db_canvas_text_set_highlight
 * @ct: a #GnomeDbCanvasText object
 * @highlight:
 *
 * Turns ON or OFF the highlighting of @ct
 */
void 
gnome_db_canvas_text_set_highlight (GnomeDbCanvasText *ct, gboolean highlight)
{

	g_return_if_fail (ct && GNOME_DB_IS_CANVAS_TEXT (ct));
	g_return_if_fail (ct->priv);

	if (! ct->priv->base_obj)
		return;

	if (highlight) 
		gnome_canvas_item_set (ct->priv->bg, "fill_color", ct->priv->highlight_color, NULL);
	else 
		gnome_canvas_item_set (ct->priv->bg, "fill_color", "white", NULL);
}
