#include <glib/gi18n-lib.h>
#include <libgnomedb/libgnomedb.h>
#include <gtk/gtk.h>
#include <unistd.h>
#include <string.h>
#include <libgda/graph/gda-dict-reg-graphs.h>

#define SQL_QUERY_ROWS "SELECT c.name, o.creation_date FROM orders o INNER JOIN customers c ON (c.id=o.customer) ORDER BY c.name, o.creation_date"
#define SQL_QUERY_COLS "SELECT p.name, p.price FROM products p ORDER BY p.name, p.price"

/* options */
gint test_number = 0;
gboolean list_tests = FALSE;

static GOptionEntry entries[] = {
	{ "test-number", 't', 0, G_OPTION_ARG_INT, &test_number, "Test number to run", "test number"},
	{ "list", 'l', 0, G_OPTION_ARG_NONE, &list_tests, "List tests", NULL},
	{ NULL }
};

/* 
 * Global configuration 
 */ 
typedef struct {
	/* General section */
	GdaClient     *client;
	GdaDict       *dict;
	gchar         *filename;

	gchar         *dsn;
	gchar         *username;
	gchar         *password;

	GtkWidget     *mainwin;
	GtkWidget     *tests_nb;
	GtkWidget     *cnc_status;

	GSList        *tests; /* list of ATest structures */

	/* Menu */
	GtkWidget     *menu_save;
	GtkWidget     *menu_db_selectds;
	GtkWidget     *menu_db_action;

	GtkWidget     *selector;
	GtkWidget     *del_obj_button;
	GtkWidget     *del_test_button;
	GtkWidget     *nb;
} MainConfig;

gboolean use_local_db = FALSE;

/*
 * Structure for each test type
 */
typedef struct _ATest {
	gchar        *test_name;
	gchar        *test_descr;
	GtkWidget  *(*make_test_widget)     (MainConfig *config);
	void        (*clean_data)           (MainConfig *config, GtkWidget *test_widget);
	gboolean    (*signal_conn_changed)  (MainConfig *config, GtkWidget *test_widget);
} ATest;


/*
 * Functions for all tests
 */
static void       build_tests (MainConfig *dict);
static void       signal_tests(MainConfig *dict);

/*
 * Utility functions
 */
static GdaDataModel *get_data_model_from_query (GdaQuery *query, const gchar *ident);
static void          param_list_param_changed_cb (GdaParameterList *paramlist, GdaParameter *param, gchar *prefix);

/*
 * list of tests' functions
 */
static GtkWidget *rels_graph_test_make   (MainConfig *config);
static void       rels_graph_test_clean  (MainConfig *config, GtkWidget *test_widget);
static gboolean   rels_graph_test_signal (MainConfig *config, GtkWidget *test_widget);

static GtkWidget *form_grid_test_make_form (MainConfig *config);
static GtkWidget *form_grid_test_make_grid (MainConfig *config);
static void       form_grid_test_clean     (MainConfig *config, GtkWidget *test_widget);
static gboolean   form_grid_test_signal    (MainConfig *config, GtkWidget *test_widget);

static GtkWidget *cust_form_grid_test_make_form (MainConfig *config);
static GtkWidget *cust_form_grid_test_make_grid (MainConfig *config);
static gboolean   cust_form_grid_test_signal    (MainConfig *config, GtkWidget *test_widget);

static GtkWidget *selector_test_make   (MainConfig *config);
static void       selector_test_clean  (MainConfig *config, GtkWidget *test_widget);
static gboolean   selector_test_signal (MainConfig *config, GtkWidget *test_widget);

static GtkWidget *query_graph_test_make   (MainConfig *config);
static void       query_graph_test_clean  (MainConfig *config, GtkWidget *test_widget);
static gboolean   query_graph_test_signal (MainConfig *config, GtkWidget *test_widget);

static GtkWidget *dsn_selector_test_make   (MainConfig *config);

static GtkWidget *editor_test_make   (MainConfig *config);

static GtkWidget *combo_test_make (MainConfig *config);
static void       combo_test_clean (MainConfig *config, GtkWidget *test_widget);
static gboolean   combo_test_signal (MainConfig *config, GtkWidget *test_widget);

static GtkWidget *console_test_make   (MainConfig *config);

static GtkWidget *xml_spec_test_make   (MainConfig *config);
static void       xml_spec_test_clean  (MainConfig *config, GtkWidget *test_widget);

static GtkWidget *forms_grids_test_make (MainConfig *config);
static void       forms_grids_test_clean     (MainConfig *config, GtkWidget *test_widget);
static gboolean   forms_grids_test_signal    (MainConfig *config, GtkWidget *test_widget);

static GtkWidget *cust_forms_grids_test_make (MainConfig *config);
static gboolean   cust_forms_grids_test_signal    (MainConfig *config, GtkWidget *test_widget);

static GtkWidget *import_test_make   (MainConfig *config);

static GtkWidget *server_op_test_make  (MainConfig *config);
static void       server_op_test_clean (MainConfig *config, GtkWidget *test_widget);
static gboolean   server_op_test_signal (MainConfig *config, GtkWidget *test_widget);

/*
 * tests decalaration
 */
static void
build_tests (MainConfig *config)
{
	GSList *list = NULL;
	ATest *test;

	/* XML GdaParameterList test */
	test = g_new0 (ATest, 1);
	test->test_name = "Basic form & XML parameters spec.";
	test->test_descr = "";
	test->make_test_widget = xml_spec_test_make;
	test->clean_data = xml_spec_test_clean;
	test->signal_conn_changed = NULL;
	list = g_slist_append (list, test);

	/* GnomeDbCanvasDbRelations test */
	test = g_new0 (ATest, 1);
	test->test_name = "GnomeDbCanvasDbRelations";
	test->test_descr = "Requires a selected graph";
	test->make_test_widget = rels_graph_test_make;
	test->clean_data = rels_graph_test_clean;
	test->signal_conn_changed = rels_graph_test_signal;
	list = g_slist_append (list, test);

	/* GnomeDbSelector test */
	test = g_new0 (ATest, 1);
	test->test_name = "GnomeDbSelector";
	test->test_descr = "";
	test->make_test_widget = selector_test_make;
	test->clean_data = selector_test_clean;
	test->signal_conn_changed = selector_test_signal;
	list = g_slist_append (list, test);

	/* GnomeDbCanvasQueryStruct test */
	test = g_new0 (ATest, 1);
	test->test_name = "GnomeDbCanvasQueryStruct";
	test->test_descr = "Requires a selected query";
	test->make_test_widget = query_graph_test_make;
	test->clean_data = query_graph_test_clean;
	test->signal_conn_changed = query_graph_test_signal;
	list = g_slist_append (list, test);

	/* GnomeDbDataSourceSelector test */
	test = g_new0 (ATest, 1);
	test->test_name = "GnomeDbDataSourceSelector";
	test->test_descr = "";
	test->make_test_widget = dsn_selector_test_make;
	test->clean_data = NULL;
	test->signal_conn_changed = NULL;
	list = g_slist_append (list, test);

	/* GnomeDbEditor test */
	test = g_new0 (ATest, 1);
	test->test_name = "GnomeDbEditor";
	test->test_descr = "";
	test->make_test_widget = editor_test_make;
	test->clean_data = NULL;
	test->signal_conn_changed = NULL;
	list = g_slist_append (list, test);

	/* GnomeDbCombo test */
	test = g_new0 (ATest, 1);
	test->test_name = "Combo";
	test->test_descr = "Requires a selected table or SELECT query, and an opened connection";
	test->make_test_widget = combo_test_make;
	test->clean_data = combo_test_clean;
	test->signal_conn_changed = combo_test_signal;
	list = g_slist_append (list, test);

	/* GnomeDbForm test */
	test = g_new0 (ATest, 1);
	test->test_name = "Form";
	test->test_descr = "Requires a selected table or SELECT query, and an opened connection";
	test->make_test_widget = form_grid_test_make_form;
	test->clean_data = form_grid_test_clean;
	test->signal_conn_changed = form_grid_test_signal;
	list = g_slist_append (list, test);

	/* GnomeDbForm test with combo boxes */
	test = g_new0 (ATest, 1);
	test->test_name = "Form +";
	test->test_descr = "Requires a table 'customers' and 'salesrep', and an opened connection";
	test->make_test_widget = cust_form_grid_test_make_form;
	test->clean_data = NULL;
	test->signal_conn_changed = cust_form_grid_test_signal;
	list = g_slist_append (list, test);

	/* GnomeDbGrid test */
        test = g_new0 (ATest, 1);
        test->test_name = "Grid";
        test->test_descr = "Requires a selected table or SELECT query, and an opened connection";
        test->make_test_widget = form_grid_test_make_grid;
        test->clean_data = form_grid_test_clean;
        test->signal_conn_changed = form_grid_test_signal;
        list = g_slist_append (list, test);

	/* GnomeDbForm test with combo boxes */
	test = g_new0 (ATest, 1);
	test->test_name = "Grid +";
	test->test_descr = "Requires a table 'customers', and an opened connection";
	test->make_test_widget = cust_form_grid_test_make_grid;
	test->clean_data = NULL;
	test->signal_conn_changed = cust_form_grid_test_signal;
	list = g_slist_append (list, test);

	/* several GnomeDbDataWidget test */
        test = g_new0 (ATest, 1);
        test->test_name = "Several forms and grids";
        test->test_descr = "Requires a selected table or SELECT query, and an opened connection";
        test->make_test_widget = forms_grids_test_make;
        test->clean_data = forms_grids_test_clean;
        test->signal_conn_changed = forms_grids_test_signal;
        list = g_slist_append (list, test);

	/* several GnomeDbDataWidget test */
        test = g_new0 (ATest, 1);
        test->test_name = "Several forms and grids +";
        test->test_descr = "Requires a table 'customers' and 'salesrep', and an opened connection";
        test->make_test_widget = cust_forms_grids_test_make;
        test->clean_data = NULL;
        test->signal_conn_changed = cust_forms_grids_test_signal;
        list = g_slist_append (list, test);

	/* GnomeDbSqlConsole test */
	test = g_new0 (ATest, 1);
	test->test_name = "SQL Console";
	test->test_descr = "";
	test->make_test_widget = console_test_make;
	test->clean_data = NULL;
	test->signal_conn_changed = NULL;
	list = g_slist_append (list, test);

	/* GnomeDbDataImport test */
	test = g_new0 (ATest, 1);
	test->test_name = "Data import test";
	test->test_descr = "";
	test->make_test_widget = import_test_make;
	test->clean_data = NULL;
	test->signal_conn_changed = NULL;
	list = g_slist_append (list, test);

	/* GnomeDbServerOperation test */
	test = g_new0 (ATest, 1);
	test->test_name = "Server operation";
	test->test_descr = "";
	test->make_test_widget = server_op_test_make;
	test->clean_data = server_op_test_clean;
	test->signal_conn_changed = server_op_test_signal;
	list = g_slist_append (list, test);

	config->tests = list;
}

static void
signal_tests (MainConfig *config)
{
	gint nb_pages, i;

	nb_pages = gtk_notebook_get_n_pages (GTK_NOTEBOOK (config->tests_nb));
	for (i=0; i<nb_pages; i++) {
		GtkWidget *page = gtk_notebook_get_nth_page (GTK_NOTEBOOK (config->tests_nb), i);
		ATest *test = g_object_get_data (G_OBJECT (page), "atest");
		if (test->signal_conn_changed)
			(test->signal_conn_changed) (config, page);
	}
}


/*
 * MAIN
 */
static gboolean conn_open (MainConfig *config);
static void set_filename (MainConfig *config, const gchar *filename);
static void destroy (GtkWidget *widget, gpointer data);
static gboolean delete_event (GtkWidget *widget, GdkEvent  *event, gpointer data);
static GtkWidget *build_menu (MainConfig *config, GtkWidget *mainwin);
static GtkWidget *build_page (MainConfig *config);
static void       load_file (MainConfig *config);
int 
main (int argc, char **argv)
{
	GtkWidget *mainwin, *vbox, *menu, *page;
	GdaDict *dict;
	MainConfig *config;
	GError *error = NULL;	
	GOptionContext *context;
	
	/* Initialize i18n support */
	gtk_set_locale ();

	/* command line parsing */
	context = g_option_context_new ("[Dictionary file] - GnomeDb widgets testing");
	g_option_context_add_main_entries (context, entries, GETTEXT_PACKAGE);
	if (!g_option_context_parse (context, &argc, &argv, &error)) {
		g_warning ("Can't parse arguments: %s", error->message);
		exit (1);
	}
	g_option_context_free (context);
	
	/* Initialize the widget set */
	gnome_db_init ("test-dyn-widgets", "", argc, argv);

	/* Test Configuration */
	config = g_new0 (MainConfig, 1);
	dict = GDA_DICT (gda_dict_new ());

	config->dict = dict;
	build_tests (config);
	gda_dict_extend_with_functions (dict);

	if (list_tests) {
		GSList *list = config->tests;
		gint i = 0;

		while (list) {
			ATest *test = (ATest *)(list->data);
			g_print ("%02d - %s (%s)\n", i++, test->test_name, test->test_descr);
			list = g_slist_next (list);
		}
		exit (0);
	}
	
	/* Create the main window */
	mainwin = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_container_set_border_width (GTK_CONTAINER (mainwin), 0);
	g_signal_connect (G_OBJECT (mainwin), "delete_event",
			  G_CALLBACK (delete_event), NULL);
	g_signal_connect (G_OBJECT (mainwin), "destroy",
			  G_CALLBACK (destroy), NULL);
	config->mainwin = mainwin;

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (mainwin), vbox);
	gtk_widget_show (vbox);

	/* menu */
	menu = build_menu (config, mainwin);
	gtk_widget_show (menu);
	gtk_box_pack_start (GTK_BOX (vbox), menu, FALSE, FALSE, 0);

	/* tables and views page */
	page = build_page (config);
	gtk_box_pack_start (GTK_BOX (vbox), page, TRUE, TRUE, 0);

	/* Show the application window */
	gtk_widget_set_size_request (mainwin, 800, 600);
	gtk_widget_show_all (mainwin);

	/* Application init */
	config->client = gda_client_new ();
	if (argc > 1) {
		set_filename (config, argv[1]);
		use_local_db = FALSE;
		load_file (config);
	}
	else {
		gchar *path, *file;

		path = g_get_current_dir ();
		file = g_build_filename (path, "TEST_DICT_sqlite.xml", NULL);
		g_free (path);
		set_filename (config, file);
		g_free (file);

		load_file (config);
		use_local_db = TRUE;
		conn_open (config);
	}
	
	gtk_main ();
	g_object_unref (G_OBJECT (dict));

	return 0;
}



/*
 * Common callbacks
 */
static gboolean delete_event (GtkWidget *widget, GdkEvent  *event, gpointer data)
{
    g_print ("Leaving DB test...\n");

    return FALSE;
}

static void destroy (GtkWidget *widget, gpointer data)
{
    gtk_main_quit ();
}

static void
set_filename (MainConfig *config, const gchar *filename)
{
	if (config->filename) {
		g_free (config->filename);
		config->filename = NULL;
	}
	if (filename)
		config->filename = g_strdup (filename);

	if (filename)
		gtk_window_set_title (GTK_WINDOW (config->mainwin), filename);
	else
		gtk_window_set_title (GTK_WINDOW (config->mainwin), _("No File"));
}

static void
load_file (MainConfig *config)
{
	GError *error = NULL;
	GdaConnection *cnc;
	if (!config->filename)
		return;

	/* remove dict's connection if there is one */
	cnc = gda_dict_get_connection (config->dict);
	if (cnc) 
		gda_dict_set_connection (config->dict, NULL);

	/* actual file loading */
	if (!gda_dict_load_xml_file (config->dict, config->filename, &error)) {
		GtkWidget *msg;
		
		msg = gtk_message_dialog_new (GTK_WINDOW (config->mainwin), 
					      GTK_DIALOG_DESTROY_WITH_PARENT,
					      GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, 
					      _("Error loading file '%s':\n%s\n"), config->filename,
					      error ? error->message : _("unreported error"));
		
		gtk_dialog_run (GTK_DIALOG (msg));
		gtk_widget_destroy (msg);
		g_error_free (error);
		error = NULL;
	}
	else {
		g_free (config->dsn);
		g_free (config->username);
		g_object_get (config->dict, "dsn", &(config->dsn), 
			      "username", &(config->username), NULL);
	}
}


/*
 * Menu building
 */
static void add_test_cb (GtkWidget *wid, MainConfig *config);
static void open_file_cb (GtkWidget *wid, MainConfig *config);
static void save_file_cb (GtkWidget *wid, MainConfig *config);
static void save_as_file_cb (GtkWidget *wid, MainConfig *config);
static void select_ds_cb (GtkWidget *wid, MainConfig *config);
static void sync_action_cb (GtkWidget *wid, MainConfig *config);
static void open_conn_cb (GtkWidget *wid, MainConfig *config);
GtkWidget *
build_menu (MainConfig *config, GtkWidget *mainwin)
{
	GtkWidget *menubar1, *menuitem1, *menuitem1_menu, *entry;
	GtkAccelGroup *accel_group;
	GSList *list;

	accel_group = gtk_accel_group_new ();

	menubar1 = gtk_menu_bar_new ();
	gtk_widget_show (menubar1);

	/* File menu */
	menuitem1 = gtk_menu_item_new_with_mnemonic (_("_File"));
	gtk_widget_show (menuitem1);
	gtk_container_add (GTK_CONTAINER (menubar1), menuitem1);
	
	menuitem1_menu = gtk_menu_new ();
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (menuitem1), menuitem1_menu);
	
	entry = gtk_image_menu_item_new_from_stock (GTK_STOCK_OPEN, accel_group);
	gtk_widget_show (entry);
	gtk_container_add (GTK_CONTAINER (menuitem1_menu), entry);

	g_signal_connect (G_OBJECT (entry), "activate",
			  G_CALLBACK (open_file_cb), config);

	entry = gtk_image_menu_item_new_from_stock (GTK_STOCK_SAVE, accel_group);
	gtk_widget_show (entry);
	gtk_container_add (GTK_CONTAINER (menuitem1_menu), entry);

	g_signal_connect (G_OBJECT (entry), "activate",
			  G_CALLBACK (save_file_cb), config);

	entry = gtk_image_menu_item_new_from_stock (GTK_STOCK_SAVE_AS, accel_group);
	gtk_widget_show (entry);
	gtk_container_add (GTK_CONTAINER (menuitem1_menu), entry);

	g_signal_connect (G_OBJECT (entry), "activate",
			  G_CALLBACK (save_as_file_cb), config);

	entry = gtk_image_menu_item_new_from_stock (GTK_STOCK_QUIT, accel_group);
	gtk_widget_show (entry);
	gtk_container_add (GTK_CONTAINER (menuitem1_menu), entry);

	g_signal_connect (G_OBJECT (entry), "activate",
			  G_CALLBACK (destroy), config);

	/* Database menu */
	menuitem1 = gtk_menu_item_new_with_mnemonic (_("_Database"));
	gtk_widget_show (menuitem1);
	gtk_container_add (GTK_CONTAINER (menubar1), menuitem1);
	
	menuitem1_menu = gtk_menu_new ();
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (menuitem1), menuitem1_menu);
	
	entry = gtk_menu_item_new_with_mnemonic (_("Open connection"));
	gtk_widget_show (entry);
	gtk_container_add (GTK_CONTAINER (menuitem1_menu), entry);
	g_signal_connect (G_OBJECT (entry), "activate",
			  G_CALLBACK (open_conn_cb), config);

	entry = gtk_menu_item_new_with_mnemonic (_("Select datasource"));
	gtk_widget_show (entry);
	gtk_container_add (GTK_CONTAINER (menuitem1_menu), entry);

	g_signal_connect (G_OBJECT (entry), "activate",
			  G_CALLBACK (select_ds_cb), config);

	entry = gtk_menu_item_new_with_mnemonic (_("Synchronise metadata with DBMS"));
	gtk_widget_show (entry);
	gtk_container_add (GTK_CONTAINER (menuitem1_menu), entry);

	g_signal_connect (G_OBJECT (entry), "activate",
			  G_CALLBACK (sync_action_cb), config);

	/* Tests menu */
	menuitem1 = gtk_menu_item_new_with_mnemonic (_("_New Test"));
	gtk_widget_show (menuitem1);
	gtk_container_add (GTK_CONTAINER (menubar1), menuitem1);

	menuitem1_menu = gtk_menu_new ();
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (menuitem1), menuitem1_menu);

	list = config->tests;
	while (list) {
		ATest *test = (ATest*) (list->data);
		entry = gtk_menu_item_new_with_mnemonic (test->test_name);
		gtk_widget_show (entry);
		gtk_container_add (GTK_CONTAINER (menuitem1_menu), entry);
		g_object_set_data (G_OBJECT (entry), "atest", test);
		g_signal_connect (G_OBJECT (entry), "activate",
				  G_CALLBACK (add_test_cb), config);
		
		list = g_slist_next (list);
	}

	gtk_window_add_accel_group (GTK_WINDOW (mainwin), accel_group);

	return menubar1;
}

static void add_test_page (MainConfig *config, ATest *test);
static void
add_test_cb (GtkWidget *wid, MainConfig *config)
{
	ATest *test;

	test = g_object_get_data (G_OBJECT (wid), "atest");
	add_test_page (config, test);
}

static void
open_file_cb (GtkWidget *wid, MainConfig *config)
{
	GtkWidget *file_selector;
	gint response;

	/* Create the selector */
	file_selector = gtk_file_selection_new (_("Select a file to load"));
	response = gtk_dialog_run (GTK_DIALOG (file_selector));
	if (response == GTK_RESPONSE_OK) {
		set_filename (config, gtk_file_selection_get_filename (GTK_FILE_SELECTION (file_selector)));
		load_file (config);
	}
	gtk_widget_destroy (file_selector);
}

static void
save_file_cb (GtkWidget *wid, MainConfig *config)
{
	if (config->filename) {
		GError *error = NULL;

		if (!gda_dict_save_xml_file (config->dict, config->filename, &error)) {
			GtkWidget *msg;
			
			msg = gtk_message_dialog_new (GTK_WINDOW (config->mainwin), 
						      GTK_DIALOG_DESTROY_WITH_PARENT,
						      GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, 
						      _("Error saving file '%s':\n%s\n"), config->filename,
						      error->message);
			gtk_dialog_run (GTK_DIALOG (msg));
			gtk_widget_destroy (msg);
			g_error_free (error);
			error = NULL;
		}
	}
	else
		save_as_file_cb (NULL, config);	
}

static void
save_as_file_cb (GtkWidget *wid, MainConfig *config)
{
	GtkWidget *file_selector;
	gint response;

	/* Create the selector */
	file_selector = gtk_file_selection_new (_("Select a file to save to"));
	response = gtk_dialog_run (GTK_DIALOG (file_selector));
	if (response == GTK_RESPONSE_OK) {
		set_filename (config, gtk_file_selection_get_filename (GTK_FILE_SELECTION (file_selector)));
		save_file_cb (NULL, config);
	}
	gtk_widget_destroy (file_selector);	
}

static void stop_dbms_update_cb (GtkWidget *dlg, gint response, MainConfig *config);
static void
sync_action_cb (GtkWidget *wid, MainConfig *config)
{
	GtkWidget *dlg, *updview;
	GdaDictDatabase *db;
	GError *error = NULL;

	db = gda_dict_get_database (config->dict);
	if (!conn_open (config)) {
		g_warning ("Can't open connection!\n");
		return;
	}

	dlg = gtk_dialog_new_with_buttons (_("Metadata synchronisation"),
					   GTK_WINDOW (config->mainwin), 
					   GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR,
					   GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, NULL);
	g_signal_connect (G_OBJECT (dlg), "response",
			  G_CALLBACK (stop_dbms_update_cb), config);
	updview = gnome_db_dbms_update_viewer_new (config->dict);
	gtk_widget_show (updview);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG(dlg)->vbox), updview, TRUE, TRUE, 0);
	gtk_widget_show (dlg);
	/* data types, functions and aggregates */
	if (!gda_dict_update_dbms_data (config->dict, 0, NULL, &error)) {
		if (error->code != GDA_DICT_META_DATA_UPDATE_USER_STOPPED) {
			GtkWidget *msg;
			
			msg = gtk_message_dialog_new (GTK_WINDOW (config->mainwin), 
						      GTK_DIALOG_DESTROY_WITH_PARENT,
						      GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE, 
						      _("Error updating Server metadata:\n%s\n"), error->message);
			gtk_dialog_run (GTK_DIALOG (msg));
			gtk_widget_destroy (msg);
		}
		g_error_free (error);
		error = NULL;	
	}
	/* database */
	if (!gda_dict_database_update_dbms_data (db, 0, NULL, &error)) {
		if (error->code != GDA_DICT_DATABASE_META_DATA_UPDATE_USER_STOPPED) {
			GtkWidget *msg;
			
			msg = gtk_message_dialog_new (GTK_WINDOW (config->mainwin), 
						      GTK_DIALOG_DESTROY_WITH_PARENT,
						      GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE, 
						      _("Error updating Database metadata:\n%s\n"), 
						      error->message);
			gtk_dialog_run (GTK_DIALOG (msg));
			gtk_widget_destroy (msg);
		}
		g_error_free (error);
		error = NULL;	
	}

	signal_tests (config);
	gtk_widget_destroy (dlg);
}

static void
stop_dbms_update_cb (GtkWidget *dlg, gint response, MainConfig *config)
{
	gda_dict_stop_update_dbms_data (config->dict);
}

static void
select_ds_cb (GtkWidget *wid, MainConfig *config)
{
	GtkWidget *props;
                                                                                                                    
        props = gnome_db_login_dialog_new (_("Connection's configuration"));
        if (gnome_db_login_dialog_run (GNOME_DB_LOGIN_DIALOG (props))) {
		GdaConnection *cnc;
		
		cnc = gda_dict_get_connection (config->dict);
		gda_connection_set_dsn (cnc, gnome_db_login_dialog_get_dsn (GNOME_DB_LOGIN_DIALOG (props)));
		if (!gnome_db_login_dialog_get_username (GNOME_DB_LOGIN_DIALOG (props)))
			gda_connection_set_username (cnc, "");
		else
			gda_connection_set_username (cnc, gnome_db_login_dialog_get_username (GNOME_DB_LOGIN_DIALOG (props)));
		gda_connection_set_password (cnc, gnome_db_login_dialog_get_password (GNOME_DB_LOGIN_DIALOG (props)));
		gtk_widget_destroy (props);
		sync_action_cb (NULL, config);
	}
	else
		gtk_widget_destroy (props);
}

static void
conn_status_changed_cb (GdaConnection *cnc, MainConfig *config)
{
	gchar *str;
	if (gda_connection_is_opened (cnc))
		str = "<i>Connection is opened</i>";
	else
		str = "<i>Connection is closed</i>";
	gtk_label_set_markup (GTK_LABEL (config->cnc_status), str);
}

static gboolean 
conn_open (MainConfig *config)
{
	GdaConnection *cnc;
	GError *error = NULL;
	GtkWidget *msg;

	cnc = gda_dict_get_connection (config->dict);
	if (cnc) {
		if (gda_connection_open (cnc, &error)) {
			signal_tests (config);
			return TRUE;
		}
	}
	else {
		if (use_local_db) {
			/* use a local SQLite database */
			gchar *dir, *str;
			if ((dir = getcwd(NULL, 0)) == NULL)
				exit(1);
			str = g_build_filename (dir, "TEST_sqlite.db", NULL);
			free (dir);
			dir = g_strdup_printf ("URI=%s", str);
			g_free (str);
			str = dir;

			cnc = gda_client_open_connection_from_string (config->client, "SQLite",
								      NULL, NULL,
								      str, 0, &error);
			g_free (str);
			use_local_db = FALSE;
		}
		else
			cnc = gda_client_open_connection (config->client, config->dsn,
							  config->username, config->password, 0, &error);
		if (cnc) {
			g_print ("Dict %p has cnc %p\n", config->dict, cnc);
			gda_dict_set_connection (config->dict, cnc);
			g_object_unref (cnc);
			
			g_signal_connect (cnc, "conn_opened",
					  G_CALLBACK (conn_status_changed_cb), config);
			g_signal_connect (cnc, "conn_closed",
					  G_CALLBACK (conn_status_changed_cb), config);
			conn_status_changed_cb (cnc, config);
			signal_tests (config);
			return TRUE;
		}
	}

	/* there is an error */
	msg = gtk_message_dialog_new (GTK_WINDOW (config->mainwin), 
				      GTK_DIALOG_DESTROY_WITH_PARENT,
				      GTK_MESSAGE_ERROR, GTK_BUTTONS_OK, 
				      _("Error opening the connection '%s':\n%s\n"), 
				      config->filename,
				      error && error->message ?  error->message : _("Unknown error"));
	gtk_dialog_run (GTK_DIALOG (msg));
	gtk_widget_destroy (msg);
	g_error_free (error);

	signal_tests (config);

	return FALSE;
}

static void
open_conn_cb (GtkWidget *wid, MainConfig *config)
{
	if (!conn_open (config)) {
		g_warning ("Can't open connection!\n");
		return;
	}
}

/*
 * Main page building
 */
static void selection_changed_cb (GnomeDbSelector *mgsel, GObject *obj, MainConfig *config);
static void left_del_clicked_cb (GtkWidget *button, MainConfig *config);
static void right_del_clicked_cb (GtkWidget *button, MainConfig *config);
static GtkWidget *
build_page (MainConfig *config)
{
	GtkWidget *wid, *label, *paned, *vb, *bb, *button, *nb;
	gchar *str;
	ATest *test;

	paned = gtk_hpaned_new ();
	gtk_container_set_border_width (GTK_CONTAINER (paned), 5);

	/* left part */
	vb = gtk_vbox_new (FALSE, 5);
	gtk_container_set_border_width (GTK_CONTAINER (vb), 5);
	gtk_paned_add1 (GTK_PANED (paned), vb);

	label = gtk_label_new (NULL);
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
	str = g_strdup_printf ("<b>%s</b>", _("Available dictionary objects:"));
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	gtk_box_pack_start (GTK_BOX (vb), label, FALSE, TRUE, 0);

	label = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL (label), "<i>Connection is closed</i>");
	gtk_box_pack_start (GTK_BOX (vb), label, FALSE, TRUE, 0);
	config->cnc_status = label;

	wid = gnome_db_selector_new (config->dict, NULL, 
			       GNOME_DB_SELECTOR_TABLES | GNOME_DB_SELECTOR_FIELDS |
			       /* GNOME_DB_SELECTOR_FUNCTIONS | */
			       GNOME_DB_SELECTOR_QUERIES | GNOME_DB_SELECTOR_QVIS_FIELDS |
			       GNOME_DB_SELECTOR_GRAPHS |
			       GNOME_DB_SELECTOR_FORMS, 0);
	gnome_db_selector_set_headers_visible (GNOME_DB_SELECTOR (wid), FALSE);
	gtk_box_pack_start (GTK_BOX (vb), wid, TRUE, TRUE, 0);
	g_signal_connect (G_OBJECT (wid), "selection_changed",
			  G_CALLBACK (selection_changed_cb), config);
	gtk_widget_set_size_request (wid, 200, 300);
	config->selector = wid;

	bb = gtk_vbutton_box_new ();
	gtk_box_pack_start (GTK_BOX (vb), bb, FALSE, FALSE, 0);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (bb), GTK_BUTTONBOX_SPREAD);
	button = gtk_button_new_from_stock (GTK_STOCK_DELETE);
	gtk_container_add (GTK_CONTAINER (bb), button);
	config->del_obj_button = button;
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (left_del_clicked_cb), config);

	/* right part */
	vb = gtk_vbox_new (FALSE, 5);
	gtk_container_set_border_width (GTK_CONTAINER (vb), 5);
	gtk_paned_add2 (GTK_PANED (paned), vb);

	label = gtk_label_new (NULL);
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
	str = g_strdup_printf ("<b>%s</b>", _("Tested widget:"));
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	gtk_box_pack_start (GTK_BOX (vb), label, FALSE, FALSE, 0);

	nb = gtk_notebook_new ();
	config->tests_nb = nb;
	gtk_box_pack_start (GTK_BOX (vb), nb, TRUE, TRUE, 0);
	gtk_widget_show (nb);

	bb = gtk_hbutton_box_new ();
	gtk_box_pack_start (GTK_BOX (vb), bb, FALSE, FALSE, 0);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (bb), GTK_BUTTONBOX_SPREAD);
	button = gtk_button_new_from_stock (GTK_STOCK_DELETE);
	config->del_test_button = button;
	gtk_container_add (GTK_CONTAINER (bb), button);
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (right_del_clicked_cb), config);

	/* start with the first test */
	test = g_slist_nth_data (config->tests, test_number);
	if (!test)
		test = config->tests->data;
	add_test_page (config, test);

	return paned;
}

/*
 * utility function to provide a new GnomeDbResultSet
 *
 * Returns: NULL if an error occurred, or if the user did not enter any required parameter
 */
static GdaDataModel *
get_data_model_from_query (GdaQuery *query, const gchar *ident)
{
	GdaDataModel *model = NULL;

	if (GDA_IS_QUERY (query) && gda_query_is_select_query (GDA_QUERY (query))) {
		GdaParameterList *dset;
		
		model = gda_data_model_query_new (query);
		dset = gda_data_model_query_get_param_list (GDA_DATA_MODEL_QUERY (model));
		if (dset && !gda_parameter_list_is_valid (dset)) {
			GtkWidget *dlg;
			gint status;
			gchar *str;
			GdaDict *dict;

			dict = gda_object_get_dict (GDA_OBJECT (query));
			str = g_strdup_printf (_("%s: parameters required"), ident);
			dlg = gnome_db_basic_form_new_in_dialog (dset, NULL, str,
								 _("Some parameters are required to execute this query"));
			g_free (str);
			status = gtk_dialog_run (GTK_DIALOG (dlg));
			gtk_widget_destroy (dlg);
			if (status == GTK_RESPONSE_REJECT) {
				g_object_unref (model);
				model = NULL;
			}
		}
		if (model) {
			GError *error = NULL;
			if (!gda_data_model_query_refresh (GDA_DATA_MODEL_QUERY (model), &error)) {
				g_warning ("Can't refresh GdaDataModelQuery data model: %s\n",
					   error && error->message ? error->message : _("Unknown error"));
				g_error_free (error);
			}
		}
	}

	return model;
}


/* Add a new tested widget */
static void
add_test_page (MainConfig *config, ATest *test)
{
	GtkWidget *wid, *label;
	gint page;

	if (!test)
		return;

	wid = (test->make_test_widget) (config);
	g_object_set_data (G_OBJECT (wid), "atest", test);
	label = gtk_label_new (test->test_name);
	page = gtk_notebook_append_page (GTK_NOTEBOOK (config->tests_nb), wid, label);
	gtk_widget_show (wid);
	gtk_widget_show (label);
	gtk_widget_set_sensitive (config->del_test_button, TRUE);
	gtk_notebook_set_current_page (GTK_NOTEBOOK (config->tests_nb), page);
}

static void
selection_changed_cb (GnomeDbSelector *mgsel, GObject *obj, MainConfig *config)
{
	gtk_widget_set_sensitive (config->del_obj_button, obj ? TRUE : FALSE);
}

static void
left_del_clicked_cb (GtkWidget *button, MainConfig *config)
{
	GObject *obj = gnome_db_selector_get_selected_object (GNOME_DB_SELECTOR (config->selector));

	if (obj) 
		gda_object_destroy (GDA_OBJECT (obj));
}

static void
right_del_clicked_cb (GtkWidget *button, MainConfig *config)
{
	GtkWidget *page;
	ATest *test;
	gint pageno = gtk_notebook_get_current_page (GTK_NOTEBOOK (config->tests_nb));

	page = gtk_notebook_get_nth_page (GTK_NOTEBOOK (config->tests_nb), pageno);
	test = g_object_get_data (G_OBJECT (page), "atest");
	if (test->clean_data) 
		(test->clean_data) (config, page);		
	gtk_notebook_remove_page (GTK_NOTEBOOK (config->tests_nb), pageno);
	gtk_widget_set_sensitive (config->del_test_button,
				  gtk_notebook_get_n_pages (GTK_NOTEBOOK (config->tests_nb)) != 0);
}


static void
param_list_param_changed_cb (GdaParameterList *paramlist, GdaParameter *param, gchar *prefix)
{
	const GValue *value;
	gchar *str;

	value = gda_parameter_get_value (param);
	if (value)
		str = gda_value_stringify ((GValue *) value);
	else
		str = "_null_";
	g_print ("%s: parameter %p '%s' changed to %s%s\n", prefix,
		 param, gda_object_get_name (GDA_OBJECT (param)), str,
		 gda_parameter_is_valid (param) ? "" : " (invalid)");
	if (value)
		g_free (str);
}




















/*
 * GnomeDbCanvasDbRelations test
 */
typedef struct {
	GtkWidget  *vbox;
	GtkWidget  *canvas;
	gulong      signal_handler;
} TestData1;

static void rels_graph_test_graph_changed (GnomeDbSelector *mgsel, GObject *obj, TestData1 *data);
static GtkWidget *
rels_graph_test_make (MainConfig *config)
{
	GtkWidget *ret, *wid = NULL;
	TestData1 *data;
	
	ret = gtk_vbox_new (FALSE, 0);
	wid = gnome_db_canvas_db_relations_new (config->dict, NULL);
	gtk_box_pack_start (GTK_BOX (ret), wid, TRUE, TRUE, 0);
	gtk_widget_show (wid);

	/* private test data */
	data = g_new0 (TestData1, 1);
	data->vbox = ret;
	data->canvas = wid;
	data->signal_handler = g_signal_connect (G_OBJECT (config->selector), "selection_changed",
						 G_CALLBACK (rels_graph_test_graph_changed), data);
	g_object_set_data (G_OBJECT (ret), "private_data", data);

	return ret;
}

static void
rels_graph_test_graph_changed (GnomeDbSelector *mgsel, GObject *obj, TestData1 *data)
{
	if (GDA_IS_GRAPH (obj))
		g_object_set (G_OBJECT (data->canvas), "graph", obj, NULL);
}

static void
rels_graph_test_clean (MainConfig *config, GtkWidget *test_widget)
{
	TestData1 *data = g_object_get_data (G_OBJECT (test_widget), "private_data");

	if (data) {
		g_signal_handler_disconnect (G_OBJECT (config->selector), data->signal_handler);
		g_free (data);
	}
}

static gboolean
rels_graph_test_signal (MainConfig *config, GtkWidget *test_widget)
{
	GObject *obj;
	TestData1 *data = g_object_get_data (G_OBJECT (test_widget), "private_data");

	obj = gnome_db_selector_get_selected_object (GNOME_DB_SELECTOR (config->selector));
	if (GDA_IS_GRAPH (obj))
		g_object_set (G_OBJECT (data->canvas), "graph", obj, NULL);

	return TRUE;
}



/*
 * GnomeDbCanvasQueryStruct test
 */
typedef struct {
	GtkWidget  *vbox;
	GtkWidget  *canvas;
	gulong      signal_handler;
} TestData2;

static void query_graph_test_graph_changed (GnomeDbSelector *mgsel, GObject *obj, TestData2 *data);
static GtkWidget *
query_graph_test_make (MainConfig *config)
{
	GtkWidget *ret;
	TestData2 *data;
	
	ret = gtk_vbox_new (FALSE, 0);

	/* private test data */
	data = g_new0 (TestData2, 1);
	data->vbox = ret;
	data->canvas = NULL;
	data->signal_handler = g_signal_connect (G_OBJECT (config->selector), "selection_changed",
						 G_CALLBACK (query_graph_test_graph_changed), data);
	g_object_set_data (G_OBJECT (ret), "private_data", data);

	return ret;
}

static void
query_graph_test_graph_changed (GnomeDbSelector *mgsel, GObject *obj, TestData2 *data)
{
	if (GDA_IS_QUERY (obj)) {
		GtkWidget *wid;
		GdaDict *dict;
		GdaGraph *graph;

		dict = gda_object_get_dict (GDA_OBJECT (obj));
		graph = gda_graphs_get_graph_for_object (dict, obj);
		if (!graph) {
			/* the query did not have any graph: create one */
			graph = GDA_GRAPH (gda_graph_query_new (GDA_QUERY (obj)));
			gda_dict_assume_object (dict, (GdaObject *) graph);
			g_object_unref (graph);
		}
		if (data->canvas)
			gtk_widget_destroy (data->canvas);
		
		wid = gnome_db_canvas_query_struct_new (GDA_QUERY (obj), graph);
		gtk_box_pack_start (GTK_BOX (data->vbox), wid, TRUE, TRUE, 0);
		gtk_widget_show (wid);
		data->canvas = wid;
	}
}

static void
query_graph_test_clean (MainConfig *config, GtkWidget *test_widget)
{
	TestData2 *data = g_object_get_data (G_OBJECT (test_widget), "private_data");

	if (data) {
		g_signal_handler_disconnect (G_OBJECT (config->selector), data->signal_handler);
		g_free (data);
	}
}

static gboolean
query_graph_test_signal (MainConfig *config, GtkWidget *test_widget)
{
	GObject *obj;
	TestData2 *data = g_object_get_data (G_OBJECT (test_widget), "private_data");

	obj = gnome_db_selector_get_selected_object (GNOME_DB_SELECTOR (config->selector));
	query_graph_test_graph_changed (GNOME_DB_SELECTOR (config->selector), obj, data);

	return TRUE;
}


/*
 * GnomeDbForm and GnomeDbGrid tests
 */
typedef struct {
	GtkWidget  *vbox;
	GtkWidget  *nodata;
	GtkWidget  *wid;
	gulong      signal_handler;
	gboolean    is_grid;
} TestData3;

static void form_grid_selected_obj_changed (GnomeDbSelector *mgsel, GObject *obj, TestData3 *data);

static GtkWidget *
form_grid_test_make_form (MainConfig *config)
{
	GtkWidget *ret, *wid = NULL;
	TestData3 *data;
	
	ret = gtk_vbox_new (FALSE, 0);
	wid = gtk_label_new ("Select a table");
	gtk_box_pack_start (GTK_BOX (ret), wid, TRUE, TRUE, 0);
	gtk_widget_show (wid);

	/* private test data */
	data = g_new0 (TestData3, 1);
	data->vbox = ret;
	data->nodata = wid;
	data->wid = NULL;
	data->is_grid = FALSE;
	data->signal_handler = g_signal_connect (G_OBJECT (config->selector), "selection_changed",
						 G_CALLBACK (form_grid_selected_obj_changed), data);
	g_object_set_data (G_OBJECT (ret), "private_data", data);
	form_grid_test_signal (config, ret);

	return ret;
}
static GtkWidget *
form_grid_test_make_grid (MainConfig *config)
{
	GtkWidget *ret, *wid = NULL;
	TestData3 *data;
	
	ret = gtk_vbox_new (FALSE, 0);
	wid = gtk_label_new ("Select a table");
	gtk_box_pack_start (GTK_BOX (ret), wid, TRUE, TRUE, 0);
	gtk_widget_show (wid);

	/* private test data */
	data = g_new0 (TestData3, 1);
	data->vbox = ret;
	data->nodata = wid;
	data->wid = NULL;
	data->is_grid = TRUE;
	data->signal_handler = g_signal_connect (G_OBJECT (config->selector), "selection_changed",
						 G_CALLBACK (form_grid_selected_obj_changed), data);
	g_object_set_data (G_OBJECT (ret), "private_data", data);
	form_grid_test_signal (config, ret);

	return ret;	
}

static void
form_grid_selected_obj_changed (GnomeDbSelector *mgsel, GObject *obj, TestData3 *data)
{
	if (data->wid) {
		gtk_widget_destroy (data->wid);
		data->wid = NULL;
		gtk_widget_show (data->nodata);
	}

	if (GDA_IS_DICT_TABLE (obj)) {
		gchar *str;
		GdaQuery *query;
		GdaDict *dict = gda_object_get_dict (GDA_OBJECT (obj));

		str = g_strdup_printf ("SELECT * FROM %s", gda_object_get_name (GDA_OBJECT (obj)));
		query = GDA_QUERY (gda_query_new_from_sql (dict, str, NULL));
		g_free (str);
		g_assert (gda_query_get_query_type (query) == GDA_QUERY_TYPE_SELECT);

		{
			GdaDataModel *query_model;
			GdaDataModelIter *data_param = NULL;
			GtkWidget *wid;
			
			query_model = get_data_model_from_query (query, gda_object_get_name (GDA_OBJECT (obj)));
			if (data->is_grid) {
				GnomeDbRawGrid *raw;
				wid = gnome_db_grid_new (query_model);
				g_object_get (G_OBJECT (wid), "raw_grid", &raw, NULL);
				data_param = gnome_db_data_widget_get_current_data (GNOME_DB_DATA_WIDGET (raw));
			}
			else {
				GnomeDbRawForm *raw;
				wid = gnome_db_form_new (query_model);
				g_object_get (G_OBJECT (wid), "raw_form", &raw, NULL);
				data_param = gnome_db_data_widget_get_current_data (GNOME_DB_DATA_WIDGET (raw));
			}
			if (query_model)
				g_object_unref (G_OBJECT (query_model));
			
			g_signal_connect (data_param, "param_changed",
					  G_CALLBACK (param_list_param_changed_cb), 
					  data->is_grid ? "Grid test" : "Form test");

			gtk_box_pack_start (GTK_BOX (data->vbox), wid, TRUE, TRUE, 0);
			gtk_widget_show (wid);
			data->wid = wid;
			gtk_widget_hide (data->nodata);
		}
		g_object_unref (G_OBJECT (query));
	}
	
	if (GDA_IS_QUERY (obj) && gda_query_is_select_query (GDA_QUERY (obj))) {
		GdaDataModel *query_model;
		GSList *targets;
		GdaQueryTarget *target = NULL;
		GdaQuery *query = GDA_QUERY (obj);
		GtkWidget *wid;
		GdaDataModelIter *data_param = NULL;
		
		targets = gda_query_get_targets (query);
		if (targets) {
			target = GDA_QUERY_TARGET (targets->data);
			g_slist_free (targets);
		}

		query_model = get_data_model_from_query (query, gda_object_get_name (GDA_OBJECT (obj)));
		if (data->is_grid) {
			GnomeDbRawGrid *raw;
			wid = gnome_db_grid_new (query_model);
			g_object_get (G_OBJECT (wid), "raw_grid", &raw, NULL);
			data_param = gnome_db_data_widget_get_current_data (GNOME_DB_DATA_WIDGET (raw));
		}
		else {
			GnomeDbRawForm *raw;
			wid = gnome_db_form_new (query_model);
			g_object_get (G_OBJECT (wid), "raw_form", &raw, NULL);
			data_param = gnome_db_data_widget_get_current_data (GNOME_DB_DATA_WIDGET (raw));
		}
		
		if (query_model)
			g_object_unref (G_OBJECT (query_model));

		g_signal_connect (data_param, "param_changed",
				  G_CALLBACK (param_list_param_changed_cb), 
				  data->is_grid ? "Grid test" : "Form test");

		gtk_box_pack_start (GTK_BOX (data->vbox), wid, TRUE, TRUE, 0);
		gtk_widget_show (wid);
		data->wid = wid;
		gtk_widget_hide (data->nodata);
	}
}
static void
form_grid_test_clean (MainConfig *config, GtkWidget *test_widget)
{
	TestData3 *data = g_object_get_data (G_OBJECT (test_widget), "private_data");
	
	if (data) {
		g_signal_handler_disconnect (G_OBJECT (config->selector), data->signal_handler);
		g_free (data);
	}
}

static gboolean
form_grid_test_signal (MainConfig *config, GtkWidget *test_widget)
{
	GObject *obj;
	TestData3 *data = g_object_get_data (G_OBJECT (test_widget), "private_data");

	obj = gnome_db_selector_get_selected_object (GNOME_DB_SELECTOR (config->selector));
	form_grid_selected_obj_changed (NULL, obj, data);

	return TRUE;
}


/*
 * GnomeDbForm and GnomeDbGrid at the same time on the same data
 */
typedef struct {
	GtkWidget  *vbox;
	GtkWidget  *wid;
} TestData4;

static GtkWidget *cust_forms_grids_test_prepare (MainConfig *config, TestData4 *data);

static GtkWidget *
cust_forms_grids_test_make (MainConfig *config)
{
	GtkWidget *ret;
	TestData4 *data;
	
	ret = gtk_vbox_new (FALSE, 0);

	/* private test data */
	data = g_new0 (TestData4, 1);
	data->vbox = ret;
	data->wid = NULL;
	g_object_set_data (G_OBJECT (ret), "private_data", data);
	cust_forms_grids_test_signal (config, ret);

	return ret;	
}

static GtkWidget *
cust_forms_grids_test_prepare (MainConfig *config, TestData4 *data)
{
	GdaDataModel *cust_model, *sr_model;
	GtkWidget *wid, *vbox;
	GdaDataModelIter *iter;
	GdaQuery *query;
	GdaParameter *param;
	GError *error = NULL;
	GnomeDbRawGrid *raw_grid;
	GnomeDbRawForm *raw_form;
	GdaDataProxy *proxy;

	query = (GdaQuery *) gda_query_new_from_sql (config->dict, 
						     "SELECT c.id, c.name, c.default_served_by, s.name "
						     "FROM customers c "
						     "LEFT JOIN salesrep s ON (s.id=c.default_served_by)", NULL);
	cust_model = get_data_model_from_query (query, "Customers");
	g_object_unref (query);

	query = (GdaQuery *) gda_query_new_from_sql (config->dict, "SELECT id, name FROM salesrep", NULL);
	sr_model = get_data_model_from_query (query, "Sales people");
	g_object_unref (query);
	
	/* new container */
	vbox = gtk_vbox_new (FALSE, 0);

	/* tested widgets creation */
	wid = gtk_label_new ("<b>This test creates a GnomeDbForm and a GnomeDbGrid widgets\n"
			     "operating on the same data</b>");
	gtk_misc_set_alignment (GTK_MISC (wid), 0.5, -1);
	gtk_label_set_use_markup (GTK_LABEL (wid), TRUE);
	gtk_box_pack_start (GTK_BOX (vbox), wid, FALSE, TRUE, 0);
	gtk_widget_show (wid);

	/* grid */
	wid = gtk_label_new ("<b>1 - GnomeDbGrid</b>");
	gtk_label_set_use_markup (GTK_LABEL (wid), TRUE);
	gtk_box_pack_start (GTK_BOX (vbox), wid, FALSE, TRUE, 0);
	gtk_widget_show (wid);

	wid = gnome_db_grid_new (cust_model);
	gtk_box_pack_start (GTK_BOX (vbox), wid, TRUE, TRUE, 0);
	gtk_widget_show (wid);
	g_object_get (G_OBJECT (wid), "raw_grid", &raw_grid, NULL);

	iter = gnome_db_data_widget_get_current_data (GNOME_DB_DATA_WIDGET (raw_grid));
	param = g_slist_nth_data (GDA_PARAMETER_LIST (iter)->parameters, 2);
	if (!gda_parameter_restrict_values (param, sr_model, 0, &error)) {
		g_print ("Cant' restrict parameter: %s\n", error->message);
		g_error_free (error);
	}
	g_signal_connect (iter, "param_changed",
			  G_CALLBACK (param_list_param_changed_cb), "Grids & forms + 1 test");
	gnome_db_data_widget_column_hide (GNOME_DB_DATA_WIDGET (raw_grid), 3);

	proxy = gnome_db_data_widget_get_proxy (GNOME_DB_DATA_WIDGET (raw_grid));
	/*gda_data_proxy_assign_model_col (proxy, sr_model, 3, 1);*/

	/* form */
	wid = gtk_label_new ("<b>2 - GnomeDbForm</b>");
	gtk_label_set_use_markup (GTK_LABEL (wid), TRUE);
	gtk_box_pack_start (GTK_BOX (vbox), wid, FALSE, TRUE, 0);
	gtk_widget_show (wid);

	wid = gnome_db_form_new (GDA_DATA_MODEL (proxy));
	gtk_box_pack_start (GTK_BOX (vbox), wid, TRUE, TRUE, 0);
	gtk_widget_show (wid);
	g_object_get (G_OBJECT (wid), "raw_form", &raw_form, NULL);

	iter = gnome_db_data_widget_get_current_data (GNOME_DB_DATA_WIDGET (raw_form));
	param = g_slist_nth_data (GDA_PARAMETER_LIST (iter)->parameters, 2);
	if (!gda_parameter_restrict_values (param, sr_model, 0, &error)) {
		g_print ("Cant' restrict parameter: %s\n", error->message);
		g_error_free (error);
	}
	g_signal_connect (iter, "param_changed",
			  G_CALLBACK (param_list_param_changed_cb), "Grids & forms + 2 test");
	gnome_db_data_widget_column_hide (GNOME_DB_DATA_WIDGET (raw_form), 3);

	g_object_unref (cust_model);
	g_object_unref (sr_model);

	return vbox;
}

static gboolean
cust_forms_grids_test_signal (MainConfig *config, GtkWidget *test_widget)
{
	TestData4 *data = g_object_get_data (G_OBJECT (test_widget), "private_data");
	GdaConnection *cnc;

	if (data->wid) {
		gtk_widget_destroy (data->wid);
		data->wid = NULL;
	}

	cnc = gda_dict_get_connection (config->dict);
	if (cnc && gda_connection_is_opened (cnc)) {
		data->wid = cust_forms_grids_test_prepare (config, data);
		gtk_box_pack_start (GTK_BOX (data->vbox), data->wid, TRUE, TRUE, 0);
		gtk_widget_show (data->wid);
	}
	else {
		data->wid = gtk_label_new ("A connection is needed");
		gtk_box_pack_start (GTK_BOX (data->vbox), data->wid, TRUE, TRUE, 0);
		gtk_widget_show (data->wid);
	}

	return TRUE;
}


/*
 * GnomeDbForm and GnomeDbGrid tests with combo boxes
 */
typedef struct {
	GtkWidget  *vbox;
	GtkWidget  *wid;
	gboolean    is_grid;
} TestData5;

static GtkWidget *cust_form_grid_test_prepare (MainConfig *config, TestData5 *data);
static GtkWidget *
cust_form_grid_test_make_form (MainConfig *config)
{
	TestData5 *data;
	GtkWidget *vbox;
	
	vbox = gtk_vbox_new (FALSE, 0);

	/* private test data */
	data = g_new0 (TestData5, 1);
	data->vbox = vbox;
	data->is_grid = FALSE;
	data->wid = NULL;
	g_object_set_data (G_OBJECT (vbox), "private_data", data);
	cust_form_grid_test_signal (config, vbox);

	return vbox;
}
static GtkWidget *
cust_form_grid_test_make_grid (MainConfig *config)
{
	TestData5 *data;
	GtkWidget *vbox;
	
	vbox = gtk_vbox_new (FALSE, 0);

	/* private test data */
	data = g_new0 (TestData5, 1);
	data->vbox = vbox;
	data->is_grid = TRUE;
	data->wid = NULL;
	g_object_set_data (G_OBJECT (vbox), "private_data", data);
	cust_form_grid_test_signal (config, vbox);

	return vbox;
}

static GtkWidget *
cust_form_grid_test_prepare (MainConfig *config, TestData5 *data)
{
	GdaDataModel *cust_model, *sr_model;
	GtkWidget *wid;
	GdaDataModelIter *iter;
	GdaQuery *query;
	GdaParameter *param;
	GError *error = NULL;
	GdaDataProxy *proxy;
	GnomeDbDataWidget *data_widget;

	query = (GdaQuery *) gda_query_new_from_sql (config->dict, 
						     "SELECT c.id, c.name, c.default_served_by, s.name "
						     "FROM customers c "
						     "INNER JOIN salesrep s ON (s.id=c.default_served_by)", NULL);
	cust_model = get_data_model_from_query (query, "Customers");
	g_object_unref (query);

	query = (GdaQuery *) gda_query_new_from_sql (config->dict, "SELECT id, name FROM salesrep", NULL);
	sr_model = get_data_model_from_query (query, "Sales people");
	g_object_unref (query);
	
	if (data->is_grid) {
		wid = gnome_db_grid_new (cust_model);
		g_object_get (G_OBJECT (wid), "raw_grid", &data_widget, NULL);
	}
	else {
		wid = gnome_db_form_new (cust_model);
		g_object_get (G_OBJECT (wid), "raw_form", &data_widget, NULL);
	}

	iter = gnome_db_data_widget_get_current_data (data_widget);
	proxy = gnome_db_data_widget_get_proxy (data_widget);
	gda_data_proxy_assign_model_col (proxy, sr_model, 3, 1);

	param = g_slist_nth_data (GDA_PARAMETER_LIST (iter)->parameters, 2);
	if (!gda_parameter_restrict_values (param, sr_model, 0, &error)) {
		g_print ("Cant' restrict parameter: %s\n", error->message);
		g_error_free (error);
	}
	g_signal_connect (iter, "param_changed",
			  G_CALLBACK (param_list_param_changed_cb), 
			  data->is_grid ? "Grid + test" : "Form + test");
	gnome_db_data_widget_column_hide (data_widget, 3);

	g_object_unref (cust_model);
	g_object_unref (sr_model);

	return wid;
}

static gboolean
cust_form_grid_test_signal (MainConfig *config, GtkWidget *test_widget)
{
	TestData5 *data = g_object_get_data (G_OBJECT (test_widget), "private_data");
	GdaConnection *cnc;

	if (data->wid) {
		gtk_widget_destroy (data->wid);
		data->wid = NULL;
	}

	cnc = gda_dict_get_connection (config->dict);
	if (cnc && gda_connection_is_opened (cnc)) {
		data->wid = cust_form_grid_test_prepare (config, data);
		gtk_box_pack_start (GTK_BOX (data->vbox), data->wid, TRUE, TRUE, 0);
		gtk_widget_show (data->wid);
	}
	else {
		data->wid = gtk_label_new ("A connection is needed");
		gtk_box_pack_start (GTK_BOX (data->vbox), data->wid, TRUE, TRUE, 0);
		gtk_widget_show (data->wid);
	}

	return TRUE;
}


/*
 * GnomeDbForm and GnomeDbGrid at the same time on the same data
 */
typedef struct {
	GtkWidget  *vbox;
	GtkWidget  *nodata;
	GtkWidget  *wid;
	gulong      signal_handler;
} TestData6;

static void forms_grids_selected_obj_changed (GnomeDbSelector *mgsel, GObject *obj, TestData6 *data);

static GtkWidget *
forms_grids_test_make (MainConfig *config)
{
	GtkWidget *ret, *wid = NULL;
	TestData6 *data;
	
	ret = gtk_vbox_new (FALSE, 0);
	wid = gtk_label_new ("Select a table");
	gtk_box_pack_start (GTK_BOX (ret), wid, TRUE, TRUE, 0);
	gtk_widget_show (wid);

	/* private test data */
	data = g_new0 (TestData6, 1);
	data->vbox = ret;
	data->nodata = wid;
	data->wid = NULL;
	data->signal_handler = g_signal_connect (G_OBJECT (config->selector), "selection_changed",
						 G_CALLBACK (forms_grids_selected_obj_changed), data);
	g_object_set_data (G_OBJECT (ret), "private_data", data);
	forms_grids_test_signal (config, ret);

	return ret;	
}

static void
forms_grids_selected_obj_changed (GnomeDbSelector *mgsel, GObject *obj, TestData6 *data)
{
	GdaDataModel *model = NULL;
	GtkWidget *table, *wid;
	GdaDataModelIter *data_param;
	GnomeDbRawGrid *raw_grid;
	GnomeDbRawForm *raw_form;
	GdaDataProxy *proxy;

	if (data->wid) {
		gtk_widget_destroy (data->wid);
		data->wid = NULL;
		gtk_widget_show (data->nodata);
	}

	if (GDA_IS_DICT_TABLE (obj)) {
		gchar *str;
		GdaQuery *query;
		GdaDict *dict = gda_object_get_dict (GDA_OBJECT (obj));

		str = g_strdup_printf ("SELECT * FROM %s", gda_object_get_name (GDA_OBJECT (obj)));
		query = GDA_QUERY (gda_query_new_from_sql (dict, str, NULL));
		g_free (str);
		g_assert (gda_query_get_query_type (query) == GDA_QUERY_TYPE_SELECT);

		model = get_data_model_from_query (query, gda_object_get_name (GDA_OBJECT (obj)));
		g_object_unref (G_OBJECT (query));
	}

	if (GDA_IS_QUERY (obj) && gda_query_is_select_query (GDA_QUERY (obj))) {
		GdaQuery *query = GDA_QUERY (obj);
		model = get_data_model_from_query (query, gda_object_get_name (GDA_OBJECT (obj)));
	}

	if (!model)
		return;
	
	/* table to store widgets */
	table = gtk_table_new (5, 2, FALSE);
	gtk_box_pack_start (GTK_BOX (data->vbox), table, TRUE, TRUE, 0);	
	gtk_table_set_row_spacings (GTK_TABLE (table), 5);
	gtk_table_set_col_spacings (GTK_TABLE (table), 5);
	gtk_widget_show (table);
	data->wid = table;
	gtk_widget_hide (data->nodata);

	/* tested widgets creation */
	wid = gtk_label_new ("<b>This test creates 2 GnomeDbForm and 2 GnomeDbGrid widgets\n"
			     "operating on the same data</b>");
	gtk_misc_set_alignment (GTK_MISC (wid), 0.5, -1);
	gtk_label_set_use_markup (GTK_LABEL (wid), TRUE);
	gtk_table_attach (GTK_TABLE (table), wid, 0, 2, 0, 1, 
			  GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0, 0);
	gtk_widget_show (wid);

	wid = gtk_label_new ("<b>1 - GnomeDbGrid</b>");
	gtk_label_set_use_markup (GTK_LABEL (wid), TRUE);
	gtk_table_attach (GTK_TABLE (table), wid, 0, 1, 1, 2, 
			  GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0, 0);
	gtk_widget_show (wid);

	wid = gnome_db_grid_new (model);
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 0, 1, 2, 3);
	gtk_widget_show (wid);
	g_object_get (G_OBJECT (wid), "raw_grid", &raw_grid, NULL);
	data_param = gnome_db_data_widget_get_current_data (GNOME_DB_DATA_WIDGET (raw_grid));
	g_signal_connect (data_param, "param_changed",
			  G_CALLBACK (param_list_param_changed_cb), "Grids & forms 1 test");
	proxy = gnome_db_data_widget_get_proxy (GNOME_DB_DATA_WIDGET (raw_grid));

	wid = gtk_label_new ("<b>4 - GnomeDbGrid</b>");
	gtk_label_set_use_markup (GTK_LABEL (wid), TRUE);
	gtk_table_attach (GTK_TABLE (table), wid, 1, 2, 3, 4, 
			  GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0, 0);
	gtk_widget_show (wid);

	wid = gnome_db_grid_new (GDA_DATA_MODEL (proxy));
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 1, 2, 4, 5);
	gtk_widget_show (wid);
	g_object_get (G_OBJECT (wid), "raw_grid", &raw_grid, NULL);
	data_param = gnome_db_data_widget_get_current_data (GNOME_DB_DATA_WIDGET (raw_grid));
	g_signal_connect (data_param, "param_changed",
			  G_CALLBACK (param_list_param_changed_cb), "Grids & forms 4 test");

	wid = gtk_label_new ("<b>3 - GnomeDbForm</b>");
	gtk_label_set_use_markup (GTK_LABEL (wid), TRUE);
	gtk_table_attach (GTK_TABLE (table), wid, 0, 1, 3, 4, 
			  GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0, 0);
	gtk_widget_show (wid);

	wid = gnome_db_form_new (GDA_DATA_MODEL (proxy));
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 0, 1, 4, 5);
	gtk_widget_show (wid);
	g_object_get (G_OBJECT (wid), "raw_form", &raw_form, NULL);
	data_param = gnome_db_data_widget_get_current_data (GNOME_DB_DATA_WIDGET (raw_form));
	g_signal_connect (data_param, "param_changed",
			  G_CALLBACK (param_list_param_changed_cb), "Grids & forms 3 test");

	wid = gtk_label_new ("<b>2 - GnomeDbForm</b>");
	gtk_label_set_use_markup (GTK_LABEL (wid), TRUE);
	gtk_table_attach (GTK_TABLE (table), wid, 1, 2, 1, 2, 
			  GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0, 0);
	gtk_widget_show (wid);

	wid = gnome_db_form_new (GDA_DATA_MODEL (proxy));
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 1, 2, 2, 3);
	gtk_widget_show (wid);
	g_object_get (G_OBJECT (wid), "raw_form", &raw_form, NULL);
	data_param = gnome_db_data_widget_get_current_data (GNOME_DB_DATA_WIDGET (raw_form));
	g_signal_connect (data_param, "param_changed",
			  G_CALLBACK (param_list_param_changed_cb), "Grids & forms 2 test");

	g_object_unref (model);
}
static void
forms_grids_test_clean (MainConfig *config, GtkWidget *test_widget)
{
	TestData6 *data = g_object_get_data (G_OBJECT (test_widget), "private_data");
	
	if (data) {
		g_signal_handler_disconnect (G_OBJECT (config->selector), data->signal_handler);
		g_free (data);
	}
}

static gboolean
forms_grids_test_signal (MainConfig *config, GtkWidget *test_widget)
{
	GObject *obj;
	TestData6 *data = g_object_get_data (G_OBJECT (test_widget), "private_data");

	obj = gnome_db_selector_get_selected_object (GNOME_DB_SELECTOR (config->selector));
	forms_grids_selected_obj_changed (NULL, obj, data);

	return TRUE;
}



/*
 * GnomeDbSelector test
 */
typedef struct {
	MainConfig *config;
	GtkWidget  *vbox;
	GtkWidget  *selector;
	gulong      signal_handler;
	guint       sel_mode;
	guint       sel_columns;
} TestData7;

static void selector_test_mode_toggled_cb (GtkCheckButton *cb, TestData7 *data);
static void selector_test_changed (GnomeDbSelector *mgsel, GObject *obj, TestData7 *data);
static GtkWidget *
selector_test_make (MainConfig *config)
{
	GtkWidget *ret, *vb, *cb, *label, *hbox;
	TestData7 *data;
	
	/* private test data */
	data = g_new0 (TestData7, 1);

	ret = gtk_table_new (10, 2, FALSE);

	/* toggle buttons for selector mode */
	label = gtk_label_new ("<b>Mode(s) for the\nGnomeDbSelector widget:</b>");
	gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
	gtk_table_attach (GTK_TABLE (ret), label, 0, 1, 0, 1, 0, 0, 0, 0);
	gtk_widget_show (label);

	vb = gtk_vbox_new (FALSE, 0);
	gtk_table_attach (GTK_TABLE (ret), vb, 0, 1, 1, 2, 0, GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
	gtk_widget_show (vb);

	cb = gtk_check_button_new_with_label ("Data types");
	g_object_set_data (G_OBJECT (cb), "mode", GUINT_TO_POINTER (GNOME_DB_SELECTOR_DATA_TYPES));
	gtk_box_pack_start (GTK_BOX (vb), cb, FALSE, FALSE, 0);
	gtk_widget_show (cb);
	g_signal_connect (G_OBJECT (cb), "toggled", 
			  G_CALLBACK (selector_test_mode_toggled_cb), data);


	cb = gtk_check_button_new_with_label ("Functions");
	g_object_set_data (G_OBJECT (cb), "mode", GUINT_TO_POINTER (GNOME_DB_SELECTOR_FUNCTIONS));
	gtk_box_pack_start (GTK_BOX (vb), cb, FALSE, FALSE, 0);
	gtk_widget_show (cb);
	g_signal_connect (G_OBJECT (cb), "toggled", 
			  G_CALLBACK (selector_test_mode_toggled_cb), data);

	cb = gtk_check_button_new_with_label ("Aggregates");
	g_object_set_data (G_OBJECT (cb), "mode", GUINT_TO_POINTER (GNOME_DB_SELECTOR_AGGREGATES));
	gtk_box_pack_start (GTK_BOX (vb), cb, FALSE, FALSE, 0);
	gtk_widget_show (cb);
	g_signal_connect (G_OBJECT (cb), "toggled", 
			  G_CALLBACK (selector_test_mode_toggled_cb), data);


	cb = gtk_check_button_new_with_label ("Tables");
	g_object_set_data (G_OBJECT (cb), "mode", GUINT_TO_POINTER (GNOME_DB_SELECTOR_TABLES));
	gtk_box_pack_start (GTK_BOX (vb), cb, FALSE, FALSE, 0);
	gtk_widget_show (cb);
	g_signal_connect (G_OBJECT (cb), "toggled", 
			  G_CALLBACK (selector_test_mode_toggled_cb), data);

	hbox = gtk_hbox_new (FALSE, 0); /* Offset */
	gtk_box_pack_start (GTK_BOX (vb), hbox, FALSE, FALSE, 0);
	gtk_widget_show (hbox);
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);

	cb = gtk_check_button_new_with_label ("Fields");
	g_object_set_data (G_OBJECT (cb), "mode", GUINT_TO_POINTER (GNOME_DB_SELECTOR_FIELDS));
	gtk_box_pack_start (GTK_BOX (hbox), cb, FALSE, FALSE, 0);
	gtk_widget_show (cb);
	g_signal_connect (G_OBJECT (cb), "toggled", 
			  G_CALLBACK (selector_test_mode_toggled_cb), data);


	cb = gtk_check_button_new_with_label ("Queries");
	g_object_set_data (G_OBJECT (cb), "mode", GUINT_TO_POINTER (GNOME_DB_SELECTOR_QUERIES));
	gtk_box_pack_start (GTK_BOX (vb), cb, FALSE, FALSE, 0);
	gtk_widget_show (cb);
	g_signal_connect (G_OBJECT (cb), "toggled", 
			  G_CALLBACK (selector_test_mode_toggled_cb), data);

	hbox = gtk_hbox_new (FALSE, 0); /* Offset */
	gtk_box_pack_start (GTK_BOX (vb), hbox, FALSE, FALSE, 0);
	gtk_widget_show (hbox);
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);

	vb = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), vb, FALSE, FALSE, 0);
	gtk_widget_show (vb);

	cb = gtk_check_button_new_with_label ("Targets");
	g_object_set_data (G_OBJECT (cb), "mode", GUINT_TO_POINTER (GNOME_DB_SELECTOR_TARGETS));
	gtk_box_pack_start (GTK_BOX (vb), cb, FALSE, FALSE, 0);
	gtk_widget_show (cb);
	g_signal_connect (G_OBJECT (cb), "toggled", 
			  G_CALLBACK (selector_test_mode_toggled_cb), data);

	cb = gtk_check_button_new_with_label ("Joins");
	g_object_set_data (G_OBJECT (cb), "mode", GUINT_TO_POINTER (GNOME_DB_SELECTOR_JOINS));
	gtk_box_pack_start (GTK_BOX (vb), cb, FALSE, FALSE, 0);
	gtk_widget_show (cb);
	g_signal_connect (G_OBJECT (cb), "toggled", 
			  G_CALLBACK (selector_test_mode_toggled_cb), data);

	cb = gtk_check_button_new_with_label ("Visible fields");
	g_object_set_data (G_OBJECT (cb), "mode", GUINT_TO_POINTER (GNOME_DB_SELECTOR_QVIS_FIELDS));
	gtk_box_pack_start (GTK_BOX (vb), cb, FALSE, FALSE, 0);
	gtk_widget_show (cb);
	g_signal_connect (G_OBJECT (cb), "toggled", 
			  G_CALLBACK (selector_test_mode_toggled_cb), data);

	cb = gtk_check_button_new_with_label ("All fields");
	g_object_set_data (G_OBJECT (cb), "mode", GUINT_TO_POINTER (GNOME_DB_SELECTOR_QALL_FIELDS));
	gtk_box_pack_start (GTK_BOX (vb), cb, FALSE, FALSE, 0);
	gtk_widget_show (cb);
	g_signal_connect (G_OBJECT (cb), "toggled", 
			  G_CALLBACK (selector_test_mode_toggled_cb), data);

	cb = gtk_check_button_new_with_label ("Sub queries");
	g_object_set_data (G_OBJECT (cb), "mode", GUINT_TO_POINTER (GNOME_DB_SELECTOR_SUB_QUERIES));
	gtk_box_pack_start (GTK_BOX (vb), cb, FALSE, FALSE, 0);
	gtk_widget_show (cb);
	g_signal_connect (G_OBJECT (cb), "toggled", 
			  G_CALLBACK (selector_test_mode_toggled_cb), data);


	/* vbox to hold the actual GnomeDbSelector widget */
	label = gtk_label_new ("<b>GnomeDbSelector widget:</b>");
	gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
	gtk_table_attach (GTK_TABLE (ret), label, 1, 2, 0, 1, 0, 0, 0, 0);
	gtk_widget_show (label);

	vb = gtk_vbox_new (FALSE, 0);
	gtk_table_attach_defaults (GTK_TABLE (ret), vb, 1, 2, 1, 2);
	gtk_widget_show (vb);

	/* private test data */
	data->config = config;
	data->vbox = vb;
	data->selector = NULL;
	data->signal_handler = g_signal_connect (G_OBJECT (config->selector), "selection_changed",
						 G_CALLBACK (selector_test_changed), data);
	data->sel_mode = 0;
	data->sel_columns = GNOME_DB_SELECTOR_COLUMN_OWNER |
		GNOME_DB_SELECTOR_COLUMN_COMMENTS |
		GNOME_DB_SELECTOR_COLUMN_TYPE |
		GNOME_DB_SELECTOR_COLUMN_FIELD_LENGTH |
		GNOME_DB_SELECTOR_COLUMN_FIELD_NNUL |
		GNOME_DB_SELECTOR_COLUMN_FIELD_DEFAULT |
		GNOME_DB_SELECTOR_COLUMN_QFIELD_VALUE |
		GNOME_DB_SELECTOR_COLUMN_QFIELD_TYPE;
	g_object_set_data (G_OBJECT (ret), "private_data", data);
	selector_test_signal (config, ret);

	return ret;
}

static void
selector_test_mode_toggled_cb (GtkCheckButton *cb, TestData7 *data)
{
	guint cb_mode = GPOINTER_TO_UINT (g_object_get_data (G_OBJECT (cb), "mode"));
	GObject *obj = NULL;

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (cb))) 
		data->sel_mode = data->sel_mode | cb_mode;
	else
		data->sel_mode = data->sel_mode & ~cb_mode;

	if (data->selector)
		obj = gnome_db_selector_get_selected_object (GNOME_DB_SELECTOR (data->config->selector));
	selector_test_changed (NULL, obj, data);
}

static void
selector_test_changed (GnomeDbSelector *mgsel, GObject *obj, TestData7 *data)
{
	GtkWidget *wid;

	/* don't care if the mode does not require the selection of an object */
	if (mgsel && 
	    ((data->sel_mode & GNOME_DB_SELECTOR_TABLES) ||
	     (data->sel_mode & GNOME_DB_SELECTOR_QUERIES) ||
	     (data->sel_mode & GNOME_DB_SELECTOR_DATA_TYPES) ||
	     (data->sel_mode & GNOME_DB_SELECTOR_FUNCTIONS) ||
	     (data->sel_mode & GNOME_DB_SELECTOR_AGGREGATES)))
	    return;

	if (data->selector) {
		gtk_widget_destroy (data->selector);
		data->selector = NULL;
	}

	wid = gnome_db_selector_new (data->config->dict, obj, data->sel_mode, data->sel_columns);
	gtk_box_pack_start (GTK_BOX (data->vbox), wid, TRUE, TRUE, 0);
	gtk_widget_show (wid);
	data->selector = wid;
}

static void
selector_test_clean (MainConfig *config, GtkWidget *test_widget)
{
	TestData7 *data = g_object_get_data (G_OBJECT (test_widget), "private_data");
	
	if (data) {
		g_signal_handler_disconnect (G_OBJECT (config->selector), data->signal_handler);
		g_free (data);
	}
}

static gboolean
selector_test_signal (MainConfig *config, GtkWidget *test_widget)
{
	GObject *obj;
	TestData7 *data = g_object_get_data (G_OBJECT (test_widget), "private_data");

	obj = gnome_db_selector_get_selected_object (GNOME_DB_SELECTOR (config->selector));
	selector_test_changed (NULL, obj, data);

	return TRUE;	
}



/*
 * GnomeDbDataSourceSelector test
 */
static GtkWidget *
dsn_selector_test_make (MainConfig *config)
{
	GtkWidget *ret, *wid;
	
	ret = gtk_table_new (3, 3, FALSE);
	wid = gnome_db_data_source_selector_new (NULL);
	gtk_table_attach (GTK_TABLE (ret), wid, 1, 2, 1, 2, 0, 0, 0, 0);
	gtk_widget_show (wid);

	return ret;
}


/*
 * GnomeDbEditor test
 */
static GtkWidget *
editor_test_make (MainConfig *config)
{
	GtkWidget *ret;
	
	ret = gnome_db_editor_new ();
	gnome_db_editor_set_highlight (GNOME_DB_EDITOR (ret), TRUE);

	return ret;
}

/*
 * GnomeDbSqlConsole test
 */
static GtkWidget *
console_test_make (MainConfig *config)
{
	GtkWidget *sw, *console;
	
	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw), GTK_SHADOW_IN);	
	
	console = gnome_db_sql_console_new (config->dict, _("SQL Console (type \\? + ENTER for help)\n"));
	gtk_container_add (GTK_CONTAINER (sw), console);
	gtk_widget_show (console);

	return sw;
}

/*
 * GnomeDbDataImport test
 */
static GtkWidget *
import_test_make (MainConfig *config)
{
	GtkWidget *import;
	
	import = gnome_db_data_import_new (config->dict);
	return import;
}

/*
 * GnomeDbCombo test
 */
typedef struct {
	GtkWidget  *combo_vbox;
	GtkWidget  *combo;
	gulong      signal_handler;
} TestData8;

static void combo_test_combo_changed (GnomeDbSelector *mgsel, GObject *obj, TestData8 *data);
static GtkWidget *
combo_test_make (MainConfig *config)
{
	GtkWidget *ret, *wid;
	TestData8 *data;
	
	ret = gtk_vbox_new (FALSE, 0);

	/* private test data */
	data = g_new0 (TestData8, 1);
	data->combo = NULL;
	data->signal_handler = g_signal_connect (G_OBJECT (config->selector), "selection_changed",
						 G_CALLBACK (combo_test_combo_changed), data);
	g_object_set_data (G_OBJECT (ret), "private_data", data);

	wid = gtk_label_new ("Combo box test, select a SELECT query");
	gtk_box_pack_start (GTK_BOX (ret), wid, FALSE, FALSE, 10);
	gtk_widget_show (wid);

	wid = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (ret), wid, FALSE, FALSE, 10);
	gtk_widget_show (wid);
	data->combo_vbox = wid;
	
	return ret;
}

static void
combo_test_combo_changed (GnomeDbSelector *mgsel, GObject *obj, TestData8 *data)
{
	GdaDataModel *model = NULL;
	GtkWidget *wid;

	gint i, n_cols;
	gint *cols_index;

	if (GDA_IS_DICT_TABLE (obj)) {
		gchar *str;
		GdaQuery *query;
		GdaDict *dict = gda_object_get_dict (GDA_OBJECT (obj));

		str = g_strdup_printf ("SELECT * FROM %s", gda_object_get_name (GDA_OBJECT (obj)));
		query = GDA_QUERY (gda_query_new_from_sql (dict, str, NULL));
		g_free (str);
		g_assert (gda_query_get_query_type (query) == GDA_QUERY_TYPE_SELECT);

		model = get_data_model_from_query (query, gda_object_get_name (GDA_OBJECT (obj)));
		g_object_unref (G_OBJECT (query));
	}

	if (GDA_IS_QUERY (obj) && gda_query_is_select_query (GDA_QUERY (obj))) {
		GdaQuery *query = GDA_QUERY (obj);
		model = get_data_model_from_query (query, gda_object_get_name (GDA_OBJECT (obj)));
	}

	if (!model)
		return;

	if (data->combo) {
		gtk_widget_destroy (data->combo);
		data->combo = NULL;
	}

	n_cols = gda_data_model_get_n_columns (model);
	cols_index = g_new0 (gint, n_cols);
	for (i=0; i<n_cols; i++)
		cols_index [i] = i;
	wid = gnome_db_combo_new_with_model (model, n_cols, cols_index);
	g_free (cols_index);

	gtk_box_pack_start (GTK_BOX (data->combo_vbox), wid, FALSE, FALSE, 0);
	gtk_widget_show (wid);
	data->combo = wid;
	
	g_object_unref (model);
}

static void
combo_test_clean (MainConfig *config, GtkWidget *test_widget)
{
	TestData8 *data = g_object_get_data (G_OBJECT (test_widget), "private_data");

	if (data) {
		g_signal_handler_disconnect (G_OBJECT (config->selector), data->signal_handler);
		g_free (data);
	}
}

static gboolean
combo_test_signal (MainConfig *config, GtkWidget *test_widget)
{
	GObject *obj;
	TestData8 *data = g_object_get_data (G_OBJECT (test_widget), "private_data");

	obj = gnome_db_selector_get_selected_object (GNOME_DB_SELECTOR (config->selector));
	combo_test_combo_changed (GNOME_DB_SELECTOR (config->selector), obj, data);

	return TRUE;
}


/*
 * XML GdaParameterList spec test
 */
typedef struct {
	GtkTextBuffer    *buffer;
	GtkWidget        *paned;

	GdaParameterList *paramlist;
	GtkWidget        *form_vbox;
	GtkWidget        *form1;
	GtkWidget        *form2;

	GtkWidget        *glade_file;
	GtkWidget        *root_element;
	GtkWidget        *form_prefix;
} TestData9;

static void xml_spec_button_clicked_cb (GtkWidget *button, MainConfig *config);
static GtkWidget *
xml_spec_test_make (MainConfig *config)
{
	GtkTextBuffer *buffer;
	GtkWidget *vbox, *view, *sw, *wid, *paned, *label, *table;
	GtkTextIter iter;
	TestData9 *data;
	gchar *test_file;
	gsize length;
	GError *error = NULL;
	GtkFileFilter *filter;

	/* private test data */
	data = g_new0 (TestData9, 1);
	data->paramlist = NULL;

	/* paned widget */
	paned = gtk_vpaned_new ();
	data->paned = paned;

	/* a box for text and button */
	vbox = gtk_vbox_new (FALSE, 5);
	g_object_set_data (G_OBJECT (vbox), "private_data", data);
	gtk_paned_add1 (GTK_PANED (paned), vbox);
	gtk_widget_show (vbox);

	/* Text at the top */
	label = gtk_label_new ("<b>Tested XML and Glade file specifications:</b>");
	gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, TRUE, 2);
	gtk_misc_set_alignment (GTK_MISC (label), 0, -1);
	g_object_set (label, "use-markup", TRUE, NULL);
	gtk_widget_show (label);

	buffer = gtk_text_buffer_new (NULL);
	data->buffer = buffer;
	gtk_text_buffer_get_start_iter (buffer, &iter);
	if (g_file_get_contents ("TEST_xml_spec.xml", &test_file, &length, &error)) {
		gtk_text_buffer_insert (buffer, &iter, test_file, length);
		g_free (test_file);
	}
	else {
		g_print ("%s\n", error->message);
		g_error_free (error);
		gtk_text_buffer_insert (buffer, &iter,
					"<?xml version=\"1.0\"?>\n"
					"<data-set-spec>\n"
					"  <parameter name=\"Name\" descr=\"A description\" gda-type=\"string\"/>\n"
					"</data-set-spec>", -1);
	}
	view = gtk_text_view_new_with_buffer (buffer);
	g_object_unref (buffer);
	
	sw = gtk_scrolled_window_new (NULL, NULL);
	gtk_box_pack_start (GTK_BOX (vbox), sw, TRUE, TRUE, 2);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (sw),
					GTK_POLICY_AUTOMATIC,
					GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (sw), GTK_SHADOW_IN);	
	
	gtk_container_add (GTK_CONTAINER (sw), view);
	gtk_widget_show (view);
	gtk_widget_show (sw);

	/* table to hold optional stuff andccommand button */
	table = gtk_table_new (4, 2, FALSE);
	gtk_box_pack_start (GTK_BOX (vbox), table, FALSE, TRUE, 5);
	gtk_widget_show (table);

	/* file chooser button for glade XML file */
	label = gtk_label_new ("Glade file:");
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1, 0, 0, 0, 0);
	gtk_widget_show (label);

	wid = gtk_file_chooser_button_new ("Choose an Glade file", GTK_FILE_CHOOSER_ACTION_OPEN);
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 1, 2, 0, 1);
	gtk_widget_show (wid);
	data->glade_file = wid;

	filter = gtk_file_filter_new ();
	gtk_file_filter_add_pattern (filter, "*.glade");
	gtk_file_chooser_add_filter (GTK_FILE_CHOOSER (wid), filter);

	label = gtk_label_new ("Root element:");
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2, 0, 0, 0, 0);
	gtk_widget_show (label);

	wid = gtk_entry_new ();
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 1, 2, 1, 2);
	gtk_widget_show (wid);
	data->root_element = wid;

	label = gtk_label_new ("Form prefix:");
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 2, 3, 0, 0, 0, 0);
	gtk_widget_show (label);

	wid = gtk_entry_new ();
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 1, 2, 2, 3);
	gtk_widget_show (wid);
	data->form_prefix = wid;

	/* button */
	wid = gtk_button_new_with_label ("Test the XML spec above");
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 0, 2, 3, 4);	
	gtk_widget_show (wid);
	g_signal_connect (G_OBJECT (wid), "clicked",
			  G_CALLBACK (xml_spec_button_clicked_cb), config);
	g_object_set_data (G_OBJECT (wid), "private_data", data);

	/* form */
	vbox = gtk_vbox_new (FALSE, 5);
	gtk_paned_add2 (GTK_PANED (data->paned), vbox);
	data->form_vbox = vbox;
	gtk_widget_show (vbox);

	label = gtk_label_new ("<b>GnomeDbBasicForm widgets:</b>\n<small>Two widgets are created to demonstrate"
			       "that they do operate on the same GdaParameterList object, they should always be"
			       "in sync.</small>");
	gtk_box_pack_start (GTK_BOX (vbox), label, FALSE, TRUE, 2);
	gtk_misc_set_alignment (GTK_MISC (label), 0, -1);
	g_object_set (label, "use-markup", TRUE, NULL);
	gtk_widget_show (label);

	data->form1 = NULL;
	data->form2 = NULL;

	return paned;
}

static void
xml_spec_button_clicked_cb (GtkWidget *button, MainConfig *config)
{
	TestData9 *data;
	GdaParameterList *dset;
	GtkTextIter start, end;
	gchar *str;
	GError *error = NULL;

	data = g_object_get_data (G_OBJECT (button), "private_data");
	if (data->form1) {
		gtk_widget_destroy (data->form1);
		data->form1 = NULL;
	}
	if (data->form2) {
		gtk_widget_destroy (data->form2);
		data->form2 = NULL;
	}

	if (data->paramlist) {
		g_signal_handlers_disconnect_by_func (data->paramlist,
						      G_CALLBACK (param_list_param_changed_cb), data);

		g_object_unref (data->paramlist);
		data->paramlist = NULL;
	}

	gtk_text_buffer_get_bounds (data->buffer, &start, &end);
	str = gtk_text_buffer_get_text (data->buffer, &start, &end, FALSE);
	dset = GDA_PARAMETER_LIST (gda_parameter_list_new_from_spec_string (config->dict, str, &error));
	g_free (str);
	if (dset) {
		GtkWidget *form;

		/* form 1 */
		form = gnome_db_basic_form_new (dset);
		gtk_box_pack_start (GTK_BOX (data->form_vbox), form, TRUE, TRUE, 5);
		gtk_widget_show (form);
		data->form1 = form;
		
		/* form 2 */
		str = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (data->glade_file));
		if (str) {
			form = gnome_db_basic_form_new_custom (dset, str,
							       gtk_entry_get_text (GTK_ENTRY (data->root_element)),
							       gtk_entry_get_text (GTK_ENTRY (data->form_prefix)));
			g_free (str);
		}
		else
			form = gnome_db_basic_form_new (dset);
		gtk_box_pack_start (GTK_BOX (data->form_vbox), form, TRUE, TRUE, 5);
		gtk_widget_show (form);
		data->form2 = form;

		g_signal_connect (dset, "param_changed",
				  G_CALLBACK (param_list_param_changed_cb), "XML Spec test");
		data->paramlist = dset;
	}
	else {
		GtkWidget *label;

		str = g_strdup_printf ("XML spec is invalid:\n%s", error ? error->message : "???");
		label = gtk_label_new (str);
		g_free (str);
		gtk_box_pack_start (GTK_BOX (data->form_vbox), label, TRUE, TRUE, 5);
		gtk_widget_show (label);
		data->form1 = label;
	}
}

static void
xml_spec_test_clean (MainConfig *config, GtkWidget *test_widget)
{
	TestData9 *data = g_object_get_data (G_OBJECT (test_widget), "private_data");

	if (data) 
		g_free (data);
}

/*
 *
 * ServerOperation test
 *
 */
typedef struct {
	MainConfig              *config;
	GtkWidget               *form_vbox;
	GdaServerOperation      *op;
	GtkWidget               *op_form;
	GnomeDbProviderSelector *prov_sel;
	GtkWidget               *filesel;
	GtkWidget               *op_combo;
	gulong                   signal_handler;
	GdaServerProvider       *prov;
} TestData10;

static void server_op_test_do_test (GtkButton *button, TestData10 *data);
static void server_op_test_do_dump (GtkButton *button, TestData10 *data);
static void server_op_test_do_load (GtkButton *button, TestData10 *data);
static void server_op_test_do_sql (GtkButton *button, TestData10 *data);
static void server_op_test_do_perform (GtkButton *button, TestData10 *data);
static void server_op_test_update_model (TestData10 *data);
static void server_op_test_changed (GnomeDbSelector *mgsel, GObject *obj, TestData10 *data);
static GdaServerProvider *server_op_test_get_provider_obj (TestData10 *data);
static void server_op_test_provider_changed_cb (GnomeDbProviderSelector *prov_sel, TestData10 *data);


static GtkWidget *
server_op_test_make (MainConfig *config)
{
	GtkWidget *wid;
	TestData10 *data;
	GtkWidget *table;

	/* private test data */
	data = g_new0 (TestData10, 1);
	data->op_form = NULL;
	data->config = config;

	data->form_vbox = gtk_vbox_new (FALSE, 0);
	g_object_set_data (G_OBJECT (data->form_vbox), "private_data", data);

	/* a box for text and button */
	table = gtk_table_new (3, 5, FALSE);
	gtk_box_pack_start (GTK_BOX (data->form_vbox), table, FALSE, TRUE, 0);
	gtk_widget_show (table);

	/* provider selection */
	wid = gtk_label_new ("Provider:");
	gtk_table_attach (GTK_TABLE (table), wid, 0, 1, 0, 1, 0, 0, 0, 0);
	gtk_widget_show (wid);

	wid = gnome_db_provider_selector_new ();
	gnome_db_provider_selector_set_selected_provider (GNOME_DB_PROVIDER_SELECTOR (wid),
							  "PostgreSQL");
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 1, 2, 0, 1);
	gtk_widget_show (wid);
	data->prov_sel = GNOME_DB_PROVIDER_SELECTOR (wid);
	g_signal_connect (G_OBJECT (wid), "changed", 
			  G_CALLBACK (server_op_test_provider_changed_cb), data);

	/* operation selector */
	wid = gtk_label_new ("Operation:");
	gtk_table_attach (GTK_TABLE (table), wid, 2, 3, 0, 1, 0, 0, 0, 0);
	gtk_widget_show (wid);

	wid = gnome_db_combo_new ();
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 3, 4, 0, 1);
	gtk_widget_show (wid);
	data->op_combo = wid;

	/* test! */
	wid = gtk_button_new_with_label ("Test!");
	gtk_table_attach (GTK_TABLE (table), wid, 0, 1, 1, 2, 0, 0, 0, 0);
	g_signal_connect (G_OBJECT (wid), "clicked",
			  G_CALLBACK (server_op_test_do_test), data);
	gtk_widget_show (wid);

	/* dump */
	wid = gtk_button_new_with_label ("Dump");
	gtk_table_attach (GTK_TABLE (table), wid, 1, 2, 1, 2, 0, 0, 0, 0);
	g_signal_connect (G_OBJECT (wid), "clicked",
			  G_CALLBACK (server_op_test_do_dump), data);
	gtk_widget_show (wid);

	/* render SQL */
	wid = gtk_button_new_with_label ("SQL");
	gtk_table_attach (GTK_TABLE (table), wid, 2, 3, 1, 2, 0, 0, 0, 0);
	g_signal_connect (G_OBJECT (wid), "clicked",
			  G_CALLBACK (server_op_test_do_sql), data);
	gtk_widget_show (wid);

	/* Perform operation */
	wid = gtk_button_new_with_label ("Perform!");
	gtk_table_attach (GTK_TABLE (table), wid, 3, 4, 1, 2, 0, 0, 0, 0);
	g_signal_connect (G_OBJECT (wid), "clicked",
			  G_CALLBACK (server_op_test_do_perform), data);
	gtk_widget_show (wid);

	/* load data */
	wid = gtk_label_new ("Load data:");
	gtk_table_attach (GTK_TABLE (table), wid, 0, 1, 2, 3, 0, 0, 0, 0);
	gtk_widget_show (wid);

	wid = gtk_file_chooser_button_new ("File to load into the GdaServerOperation object", 
					   GTK_FILE_CHOOSER_ACTION_OPEN);
	data->filesel = wid;
	gtk_table_attach_defaults (GTK_TABLE (table), wid, 1, 2, 2, 3);
	gtk_widget_show (wid);

	wid = gtk_button_new_with_label ("Load");
	gtk_table_attach (GTK_TABLE (table), wid, 2, 3, 2, 3, 0, 0, 0, 0);
	g_signal_connect (G_OBJECT (wid), "clicked",
			  G_CALLBACK (server_op_test_do_load), data);
	gtk_widget_show (wid);

	data->signal_handler = g_signal_connect (G_OBJECT (config->selector), "selection_changed",
						 G_CALLBACK (server_op_test_changed), data);
	gtk_widget_show (data->form_vbox);

	return data->form_vbox;

}

static void
server_op_test_changed (GnomeDbSelector *mgsel, GObject *obj, TestData10 *data)
{
	server_op_test_update_model (data);
}

static void
server_op_test_provider_changed_cb (GnomeDbProviderSelector *prov_sel, TestData10 *data)
{
	if (data->prov) {
		g_object_unref (data->prov);
		data->prov = NULL;
	}
	server_op_test_update_model (data);
}

static void
server_op_test_update_model (TestData10 *data)
{
	GdaParameter *param = NULL;
	GdaParameterList *options = NULL;
	GdaServerOperationType type;
	GdaDataModel *model;
	GObject *obj;
	
	obj = gnome_db_selector_get_selected_object (GNOME_DB_SELECTOR (data->config->selector));
	if (GDA_IS_DICT_TABLE (obj)) {
		if (gda_dict_table_is_view (GDA_DICT_TABLE (obj)))
			param = gda_parameter_new_string ("VIEW", gda_object_get_name (GDA_OBJECT (obj)));
		else
			param = gda_parameter_new_string ("TABLE", gda_object_get_name (GDA_OBJECT (obj)));
	}
		
	if (param) {
		options = gda_parameter_list_new (NULL);
		gda_parameter_list_add_param (options, param);
		g_object_unref (param);
	}

	model = gnome_db_combo_get_model (GNOME_DB_COMBO (data->op_combo));
	if (!model) {
		gint columns[] = {1};
		model = gda_data_model_array_new_with_types (2, G_TYPE_INT, G_TYPE_STRING);
		
		gnome_db_combo_set_model (GNOME_DB_COMBO (data->op_combo), model, 1, columns);
	}
	else
		gda_data_model_array_clear (GDA_DATA_MODEL_ARRAY (model));
	for (type = 0; type < GDA_SERVER_OPERATION_NB; type ++)
		if (gda_server_provider_supports_operation (server_op_test_get_provider_obj (data),
							    gda_dict_get_connection (data->config->dict), 
							    type, options)) {
			gint row;

			row = gda_data_model_append_row (model, NULL);
			if (row < 0)
				g_error ("Cant' append data to a GdaDataModelArray");
			else {
				GValue value;

				memset (&value, 0, sizeof (GValue));
				g_value_init (&value, G_TYPE_INT);
				g_value_set_int (&value, type);
				gda_data_model_set_value_at (model, 0, row, &value, NULL);

				memset (&value, 0, sizeof (GValue));
				g_value_init (&value, G_TYPE_STRING);
				g_value_set_string (&value, gda_server_operation_op_type_to_string (type));
				gda_data_model_set_value_at (model, 1, row, &value, NULL);
			}
		}
	if (options)
		g_object_unref (options);
}

static GdaServerProvider *
server_op_test_get_provider_obj (TestData10 *data)
{
	GdaConnection *cnc;
	GdaServerProvider *prov = NULL;

	cnc = gda_dict_get_connection (data->config->dict);
	if (cnc && gda_connection_is_opened (cnc))
		return gda_connection_get_provider_obj (cnc);

	if (data->prov)
		prov = data->prov;
	else {
		/* create the GdaServerProvider object */
		data->prov = gnome_db_provider_selector_get_selected_provider_obj (data->prov_sel);
		prov = data->prov;
	}

	return prov;
}

static void
server_op_test_do_test (GtkButton *button, TestData10 *data)
{
	GdaServerProvider *prov = NULL;
	GdaServerOperationType type;
	GError *error = NULL;
	GSList *combo_values;
	gint columns[] = {0};

	if (data->op) {
		g_object_unref (data->op);
		data->op = NULL;
	}
	if (data->op_form) 
		gtk_widget_destroy (data->op_form);

	combo_values = gnome_db_combo_get_values_ext (GNOME_DB_COMBO (data->op_combo), 1, columns);
	if (!combo_values || !gda_value_isa ((GValue *) (combo_values->data), G_TYPE_INT)) {
		GtkWidget *label;

		label = gtk_label_new ("Select an operation to perform");
		gtk_box_pack_start (GTK_BOX (data->form_vbox), label, TRUE, TRUE, 10);
		data->op_form = label;
		return;
	}
	type = g_value_get_int ((GValue *) (combo_values->data));
	g_slist_free (combo_values);
	
	prov = server_op_test_get_provider_obj (data);
	if (prov)
		data->op = gda_server_provider_create_operation (prov, gda_dict_get_connection (data->config->dict), type, NULL, &error);

	if (!data->op) {
		GtkWidget *label;
		gchar *str;

		str = g_strdup_printf ("Can't create GdaServerOperation widget: %s",
				       error && error->message ? error->message : "No detail");
		label = gtk_label_new (str);
		g_free (str);
		gtk_box_pack_start (GTK_BOX (data->form_vbox), label, TRUE, TRUE, 10);
		data->op_form = label;
	} 
	else {
		data->op_form = gnome_db_server_operation_new (data->op);
		gtk_box_pack_start (GTK_BOX (data->form_vbox), data->op_form, TRUE, TRUE, 0);
	}
	gtk_widget_show (data->op_form);
}

static void
server_op_test_do_dump (GtkButton *button, TestData10 *data)
{
	xmlNodePtr node;

	if (!data->op || !data->op_form || ! IS_GNOME_DB_SERVER_OPERATION (data->op_form)) {
		g_print ("No form!\n");
		return;
	}

	node = gda_server_operation_save_data_to_xml (data->op, NULL);
	if (node) {
		xmlDocPtr doc;
		xmlChar *buffer;
		
		doc = xmlNewDoc ("1.0");
		xmlDocSetRootElement (doc, node);
		xmlIndentTreeOutput = 1;
		xmlKeepBlanksDefault (0);
		xmlDocDumpFormatMemory (doc, &buffer, NULL, 1);
		g_print ("%s\n", buffer);
		xmlFree (buffer);
		xmlFreeDoc (doc);
	}
	else 
		g_warning ("Saving to XML failed!");
}

static void
server_op_test_do_load (GtkButton *button, TestData10 *data)
{
	xmlNodePtr node;
	xmlDocPtr doc;
	GError *error = NULL;
	gchar *filename;

	if (!data->op || !data->op_form || ! IS_GNOME_DB_SERVER_OPERATION (data->op_form)) {
		g_print ("No form!\n");
		return;
	}
	
	filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (data->filesel));
	if (!filename) {
		g_print ("No file selected to load\n");
		return;
	}

	doc = xmlParseFile (filename);
	if (!doc) {
		g_print ("Can't load XML file '%s'\n", filename);
		g_free (filename);
		return;
	}
	g_free (filename);

	node = xmlDocGetRootElement (doc);
	if (!gda_server_operation_load_data_from_xml (data->op, node, &error)) {
		g_print ("Can't load data: %s\n", error && error->message ? error->message : "No detail");
		if (error)
			g_error_free (error);
	}

	xmlFreeDoc (doc);
}

static void
server_op_test_do_sql (GtkButton *button, TestData10 *data)
{
	GdaServerProvider *prov;

	if (!data->op) {
		g_print ("No GdaServerOperation!\n");
		return;
	}
	prov = server_op_test_get_provider_obj (data);
	if (prov) {
		gchar *sql;
		GError *error = NULL;

		sql = gda_server_provider_render_operation (prov, gda_dict_get_connection (data->config->dict), data->op, &error);
		if (!sql) {
			g_print ("Can't render operation: %s\n", error && error->message ? error->message : "No detail");
			if (error)
				g_error_free (error);
		}
		else {
			g_print ("SQL: %s\n", sql);
			g_free (sql);
		}
	}
	else 
		g_warning ("Could not get provider object");
}

static void
server_op_test_do_perform (GtkButton *button, TestData10 *data)
{
	GdaServerProvider *prov;

	if (!data->op) {
		g_print ("No GdaServerOperation!\n");
		return;
	}
	prov = server_op_test_get_provider_obj (data);
	if (prov) {
		GError *error = NULL;

		if (!gda_server_provider_perform_operation (prov, 
							    gda_dict_get_connection (data->config->dict), 
							    data->op, &error)) {
			g_print ("Can't perform operation: %s\n", error && error->message ? error->message : "No detail");
			if (error)
				g_error_free (error);
		}
		else 
			g_print ("Operation done!\n");
	}
	else 
		g_warning ("Could not get provider object");
}

static void
server_op_test_clean (MainConfig *config, GtkWidget *test_widget)
{
	TestData10 *data = g_object_get_data (G_OBJECT (test_widget), "private_data");

	if (data) {
		g_signal_handler_disconnect (G_OBJECT (config->selector), data->signal_handler);
		
		if (data->op)
			g_object_unref (data->op);
		g_free (data);
	}
}

static gboolean
server_op_test_signal (MainConfig *config, GtkWidget *test_widget)
{
	TestData10 *data = g_object_get_data (G_OBJECT (test_widget), "private_data");
	GdaConnection *cnc;

	cnc = gda_dict_get_connection (config->dict);
	gtk_widget_set_sensitive (GTK_WIDGET (data->prov_sel), !cnc || !gda_connection_is_opened (cnc));
	if (cnc && gda_connection_is_opened (cnc)) {
		GdaServerProvider *prov;
		GdaServerProviderInfo* pinfo;

		prov = gda_connection_get_provider_obj (cnc);
		pinfo = gda_server_provider_get_info (prov, cnc);
		gnome_db_provider_selector_set_selected_provider (GNOME_DB_PROVIDER_SELECTOR (data->prov_sel),
								  pinfo->provider_name);
	}

	return TRUE;
}
