/* GNOME DB library
 * Copyright (C) 1999-2002 The GNOME Foundation.
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtkbbox.h>
#include <gtk/gtkentry.h>
#include <gtk/gtkenums.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkstock.h>
#include <gtk/gtktable.h>
#include <gtk/gtktogglebutton.h>
#include <libgnomedb/gnome-db-form.h>
#include <libgnomedb/gnome-db-grid.h>
#include <libgnomedb/gnome-db-util.h>
#include "gnome-db-i18n.h"

#define PARENT_TYPE GTK_TYPE_VBOX

struct _GnomeDbFormPrivate {
	GdaDataModel *data_model;
	GnomeDbFormType type;

	/* widgets */
	GtkWidget *form_container;

	GList *navigator_widgets; /* navigator widget */
	GtkWidget *navigator_label;
	GtkWidget *navigator_first;
	GtkWidget *navigator_back;
	GtkWidget *navigator_fwd;
	GtkWidget *navigator_last;
	gint navigator_current_row;
};

static void gnome_db_form_class_init   (GnomeDbFormClass *klass);
static void gnome_db_form_init         (GnomeDbForm *form, GnomeDbFormClass *klass);
static void gnome_db_form_set_property (GObject *object,
					guint paramid,
					const GValue *value,
					GParamSpec *pspec);
static void gnome_db_form_get_property (GObject *object,
					guint param_id,
					GValue *value,
					GParamSpec *pspec);
static void gnome_db_form_finalize     (GObject *object);

static void show_navigator_row (GnomeDbForm *form);

enum {
	MODEL_CHANGED,
	LAST_SIGNAL
};

static gint form_signals[LAST_SIGNAL];
static GObjectClass *parent_class = NULL;

/*
 * Callbacks
 */

static void
navigator_first_cb (GtkButton *button, gpointer user_data)
{
	GnomeDbForm *form = (GnomeDbForm *) user_data;

	g_return_if_fail (GNOME_DB_IS_FORM (form));

	form->priv->navigator_current_row = 0;
	show_navigator_row (form);
}

static void
navigator_back_cb (GtkButton *button, gpointer user_data)
{
	GnomeDbForm *form = (GnomeDbForm *) user_data;

	g_return_if_fail (GNOME_DB_IS_FORM (form));

	if (form->priv->navigator_current_row > 0) {
		form->priv->navigator_current_row--;
		show_navigator_row (form);
	}
}

static void
navigator_forward_cb (GtkButton *button, gpointer user_data)
{
	gint max_rows;
	GnomeDbForm *form = (GnomeDbForm *) user_data;

	g_return_if_fail (GNOME_DB_IS_FORM (form));

	max_rows = gda_data_model_get_n_rows (form->priv->data_model) - 1;

	if (form->priv->navigator_current_row < max_rows) {
		form->priv->navigator_current_row++;
		show_navigator_row (form);
	}
}

static void
navigator_last_cb (GtkButton *button, gpointer user_data)
{
	GnomeDbForm *form = (GnomeDbForm *) user_data;

	g_return_if_fail (GNOME_DB_IS_FORM (form));

	form->priv->navigator_current_row = gda_data_model_get_n_rows (form->priv->data_model) - 1;
	show_navigator_row (form);
}

/*
 * Private functions
 */

static GtkWidget *
create_navigator (GnomeDbForm *form)
{
	GtkWidget *navigator;
	GtkWidget *bbox;
	GtkWidget *sep;
	gint cols;
	gint col;

	g_return_val_if_fail (GNOME_DB_IS_FORM (form), NULL);
	g_return_val_if_fail (GDA_IS_DATA_MODEL (form->priv->data_model), NULL);

	cols = gda_data_model_get_n_columns (form->priv->data_model);
	navigator = gnome_db_new_table_widget (cols + 3, 3, FALSE);
	form->priv->navigator_current_row = 0;

	/* create the entry widgets for each field */
	for (col = 0; col < cols; col++) {
		GtkWidget *label;
		GtkWidget *entry;
		GdaFieldAttributes *fa;

		fa = gda_data_model_describe_column (form->priv->data_model, col);

		label = gnome_db_new_label_widget (gda_field_attributes_get_name (fa));
		gtk_table_attach (GTK_TABLE (navigator), label, 0, 1, col, col + 1,
				  GTK_FILL, GTK_FILL, 2, 2);

		switch (gda_field_attributes_get_gdatype (fa)) {
		case GDA_VALUE_TYPE_BOOLEAN :
			entry = gnome_db_new_check_button_widget (NULL);
			gtk_widget_set_sensitive (entry, FALSE);
			break;
		default :
			entry = gnome_db_new_entry_widget (0, FALSE);
			break;
		}

		gtk_table_attach (GTK_TABLE (navigator), entry, 1, 2, col, col + 1,
				  GTK_FILL, GTK_FILL, 2, 2);
		form->priv->navigator_widgets = g_list_append (
			form->priv->navigator_widgets,
			entry);

		gda_field_attributes_free (fa);
	}

	sep = gnome_db_new_vseparator_widget ();
	gtk_table_attach (GTK_TABLE (navigator), sep, 0, 3, cols, cols + 1,
			  GTK_FILL, GTK_FILL | GTK_SHRINK | GTK_EXPAND, 2, 2);

	form->priv->navigator_label = gnome_db_new_label_widget (_("No records"));
	gtk_misc_set_alignment (GTK_MISC (form->priv->navigator_label), 0.5, 0.5);
	gtk_table_attach (GTK_TABLE (navigator), form->priv->navigator_label, 0, 3,
			  cols + 1, cols + 2,
			  GTK_FILL, GTK_FILL | GTK_EXPAND | GTK_SHRINK, 2, 2);

	/* create navigation buttons */
	bbox = gnome_db_new_hbutton_box_widget ();
	gtk_button_box_set_layout (GTK_BUTTON_BOX (bbox), GTK_BUTTONBOX_DEFAULT_STYLE);
	gtk_table_attach (GTK_TABLE (navigator), bbox, 0, 3, cols + 2, cols + 3,
			  GTK_FILL, GTK_FILL | GTK_EXPAND | GTK_SHRINK, 2, 2);

	form->priv->navigator_first = gnome_db_new_button_widget_from_stock (GTK_STOCK_GOTO_FIRST);
	g_signal_connect (G_OBJECT (form->priv->navigator_first), "clicked",
			  G_CALLBACK (navigator_first_cb), form);
	gtk_box_pack_start (GTK_BOX (bbox), form->priv->navigator_first, FALSE, FALSE, 0);

	form->priv->navigator_back = gnome_db_new_button_widget_from_stock (GTK_STOCK_GO_BACK);
	g_signal_connect (G_OBJECT (form->priv->navigator_back), "clicked",
			  G_CALLBACK (navigator_back_cb), form);
	gtk_box_pack_start (GTK_BOX (bbox), form->priv->navigator_back, FALSE, FALSE, 0);

	form->priv->navigator_fwd = gnome_db_new_button_widget_from_stock (GTK_STOCK_GO_FORWARD);
	g_signal_connect (G_OBJECT (form->priv->navigator_fwd), "clicked",
			  G_CALLBACK (navigator_forward_cb), form);
	gtk_box_pack_start (GTK_BOX (bbox), form->priv->navigator_fwd, FALSE, FALSE, 0);

	form->priv->navigator_last = gnome_db_new_button_widget_from_stock (GTK_STOCK_GOTO_LAST);
	g_signal_connect (G_OBJECT (form->priv->navigator_last), "clicked",
			  G_CALLBACK (navigator_last_cb), form);
	gtk_box_pack_start (GTK_BOX (bbox), form->priv->navigator_last, FALSE, FALSE, 0);

	show_navigator_row (form);

	return navigator;
}

static GtkWidget *
create_list (GnomeDbForm *form)
{
	g_return_val_if_fail (GNOME_DB_IS_FORM (form), NULL);
	g_return_val_if_fail (GDA_IS_DATA_MODEL (form->priv->data_model), NULL);

	return gnome_db_new_grid_widget (form->priv->data_model);
}

static void
setup_widgets_for_model (GnomeDbForm *form)
{
	g_return_if_fail (GNOME_DB_IS_FORM (form));

	/* destroy previous widgets */
	if (form->priv->form_container)
		gtk_widget_destroy (form->priv->form_container);
	if (form->priv->navigator_widgets) {
		g_list_free (form->priv->navigator_widgets);
		form->priv->navigator_widgets = NULL;
	}

	/* create the new form depending on the type */
	switch (form->priv->type) {
	case GNOME_DB_FORM_TYPE_NAVIGATOR :
		form->priv->form_container = create_navigator (form);
		break;
	case GNOME_DB_FORM_TYPE_LIST :
		form->priv->form_container = create_list (form);
		break;
	default :
		form->priv->form_container = NULL;
		break;
	}

	if (!form->priv->form_container)
		return;

	gtk_widget_show (form->priv->form_container);
	gtk_box_pack_start (GTK_BOX (form), form->priv->form_container, TRUE, TRUE, 0);
}

static void
show_navigator_row (GnomeDbForm *form)
{
	gint i;
	GList *l;
	gint total_rows;
	gchar *tmp;

	g_return_if_fail (GNOME_DB_IS_FORM (form));

	if (!GDA_IS_DATA_MODEL (form->priv->data_model)) {
		gtk_label_set_text (GTK_LABEL (form->priv->navigator_label), _("No records"));
		return;
	}

	total_rows = gda_data_model_get_n_rows (form->priv->data_model);

	for (i = 0, l = form->priv->navigator_widgets; l != NULL; i++, l = l->next) {
		const GdaValue *value;

		value = gda_data_model_get_value_at (form->priv->data_model, i,
						     form->priv->navigator_current_row);
		if (value) {
			gchar *str;

			if (gda_value_isa ((GdaValue *) value, GDA_VALUE_TYPE_BOOLEAN)) {
				gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (l->data),
							      gda_value_get_boolean ((GdaValue *) value));
			}
			else {
				str = gda_value_stringify ((GdaValue *) value);
				gtk_entry_set_text (GTK_ENTRY (l->data), str);
				g_free (str);
			}
		}
		else
			gtk_entry_set_text (GTK_ENTRY (l->data), "");
	}

	if (total_rows <= 0) {
		gtk_widget_set_sensitive (form->priv->navigator_first, FALSE);
		gtk_widget_set_sensitive (form->priv->navigator_back, FALSE);
		gtk_widget_set_sensitive (form->priv->navigator_fwd, FALSE);
		gtk_widget_set_sensitive (form->priv->navigator_last, FALSE);
		gtk_label_set_text (GTK_LABEL (form->priv->navigator_label), _("No records"));
	}
	else {
		if (form->priv->navigator_current_row == 0) {
			gtk_widget_set_sensitive (form->priv->navigator_first, FALSE);
			gtk_widget_set_sensitive (form->priv->navigator_back, FALSE);
		}
		else {
			gtk_widget_set_sensitive (form->priv->navigator_first, TRUE);
			gtk_widget_set_sensitive (form->priv->navigator_back, TRUE);
		}

		if (form->priv->navigator_current_row == total_rows - 1) {
			gtk_widget_set_sensitive (form->priv->navigator_fwd, FALSE);
			gtk_widget_set_sensitive (form->priv->navigator_last, FALSE);
		}
		else {
			gtk_widget_set_sensitive (form->priv->navigator_fwd, TRUE);
			gtk_widget_set_sensitive (form->priv->navigator_last, TRUE);
		}

		tmp = g_strdup_printf (_("Record %d of %d"),
				       form->priv->navigator_current_row + 1,
				       total_rows);
		gtk_label_set_text (GTK_LABEL (form->priv->navigator_label), tmp);
		g_free (tmp);
	}
}

/*
 * GnomeDbForm class implementation
 */

static void
gnome_db_form_class_init (GnomeDbFormClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->set_property = gnome_db_form_set_property;
	object_class->get_property = gnome_db_form_get_property;
	object_class->finalize = gnome_db_form_finalize;

	/* add class signals */
	form_signals[MODEL_CHANGED] =
		g_signal_new ("model_changed",
			      G_TYPE_FROM_CLASS (object_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (GnomeDbFormClass, model_changed),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);
}

static void
gnome_db_form_init (GnomeDbForm *form, GnomeDbFormClass *klass)
{
	g_return_if_fail (GNOME_DB_IS_FORM (form));

	/* allocate private structure */
	form->priv = g_new0 (GnomeDbFormPrivate, 1);
	form->priv->data_model = NULL;
	form->priv->type = GNOME_DB_FORM_TYPE_NAVIGATOR;
	form->priv->form_container = NULL;
	form->priv->navigator_widgets = NULL;
	form->priv->navigator_label = NULL;
}

static void
gnome_db_form_set_property (GObject *object,
			    guint param_id,
			    const GValue *value,
			    GParamSpec *pspec)
{
	GnomeDbForm *form = (GnomeDbForm *) object;

	g_return_if_fail (GNOME_DB_IS_FORM (form));

	switch (param_id) {
	default :
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}

static void
gnome_db_form_get_property (GObject *object,
			    guint param_id,
			    GValue *value,
			    GParamSpec *pspec)
{
	GnomeDbForm *form = (GnomeDbForm *) object;

	g_return_if_fail (GNOME_DB_IS_FORM (form));

	switch (param_id) {
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, param_id, pspec);
		break;
	}
}

static void
gnome_db_form_finalize (GObject *object)
{
	GnomeDbForm *form = (GnomeDbForm *) object;

	g_return_if_fail (GNOME_DB_IS_FORM (form));

	/* free memory */
	if (GDA_IS_DATA_MODEL (form->priv->data_model)) {
		g_object_unref (G_OBJECT (form->priv->data_model));
		form->priv->data_model = NULL;
	}

	if (form->priv->navigator_widgets) {
		g_list_free (form->priv->navigator_widgets);
		form->priv->navigator_widgets = NULL;
	}

	g_free (form->priv);
	form->priv = NULL;

	/* chain to parent class */
	parent_class->finalize (object);
}

GType
gnome_db_form_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbFormClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_form_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbForm),
			0,
			(GInstanceInitFunc) gnome_db_form_init
		};
		type = g_type_register_static (PARENT_TYPE, "GnomeDbForm", &info, 0);
	}
	return type;
}

/**
 * gnome_db_form_new
 *
 * Create a new #GnomeDbForm widget, which is a high-level widget which
 * lets you display recordsets in many different ways, such as a list,
 * record-by-record, etc.
 *
 * Returns: the newly created widget.
 */
GtkWidget *
gnome_db_form_new (void)
{
	GnomeDbForm *form;

	form = g_object_new (GNOME_DB_TYPE_FORM, NULL);
	return GTK_WIDGET (form);
}

/**
 * gnome_db_form_get_form_type
 */
GnomeDbFormType
gnome_db_form_get_form_type (GnomeDbForm *form)
{
	g_return_val_if_fail (GNOME_DB_IS_FORM (form), -1);
	return form->priv->type;
}

/**
 * gnome_db_form_set_form_type
 */
void
gnome_db_form_set_form_type (GnomeDbForm *form, GnomeDbFormType type)
{
	g_return_if_fail (GNOME_DB_IS_FORM (form));

	form->priv->type = type;
	setup_widgets_for_model (form);
}

/**
 * gnome_db_form_get_model
 * @form: a #GnomeDbForm widget.
 *
 * Return the #GdaDataModel being used by the given widget.
 */
GdaDataModel *
gnome_db_form_get_model (GnomeDbForm *form)
{
	g_return_val_if_fail (GNOME_DB_IS_FORM (form), NULL);
	return form->priv->data_model;
}

/**
 * gnome_db_form_set_model
 * @form: a #GnomeDbForm widget.
 * @model: the data model to associate the widget with.
 */
void
gnome_db_form_set_model (GnomeDbForm *form, GdaDataModel *model)
{
	g_return_if_fail (GNOME_DB_IS_FORM (form));

	if (GDA_IS_DATA_MODEL (form->priv->data_model))
		g_object_unref (G_OBJECT (form->priv->data_model));

	g_object_ref (G_OBJECT (model));
	form->priv->data_model = model;

	setup_widgets_for_model (form);
	g_signal_emit (G_OBJECT (form), form_signals[MODEL_CHANGED], 0);
}

/**
 * gnome_db_form_get_current_row
 * @form: a #GnomeDbForm widget.
 *
 * Get the row currently being shown (navigator form) or currently selected
 * (list form) by the given form widget.
 *
 * Returns: the current row.
 */
gint
gnome_db_form_get_current_row (GnomeDbForm *form)
{
	GList *selection;
	gint ret;

	g_return_val_if_fail (GNOME_DB_IS_FORM (form), -1);

	switch (form->priv->type) {
	case GNOME_DB_FORM_TYPE_NAVIGATOR :
		return form->priv->navigator_current_row;
	case GNOME_DB_FORM_TYPE_LIST :
		g_return_val_if_fail (GNOME_DB_IS_GRID (form->priv->form_container), -1);

		selection = gnome_db_grid_get_selection (GNOME_DB_GRID (form->priv->form_container));
		if (selection) {
			ret = GPOINTER_TO_INT (selection->data);
			g_list_free (selection);
			return ret;
		}
		break;
	}

	return -1;
}

/**
 * gnome_db_form_set_current_row
 * @form: a #GnomeDbForm widget.
 * @row: the new row to display.
 *
 * Sets the current row on the given form widget.
 */
void
gnome_db_form_set_current_row (GnomeDbForm *form, gint row)
{
	g_return_if_fail (GNOME_DB_IS_FORM (form));
	g_return_if_fail (row >= 0);
	g_return_if_fail (row < gda_data_model_get_n_rows (form->priv->data_model));

	switch (form->priv->type) {
	case GNOME_DB_FORM_TYPE_NAVIGATOR :
		form->priv->navigator_current_row = row;
		show_navigator_row (form);
		break;
	case GNOME_DB_FORM_TYPE_LIST : ;
	}
}
